/******************************************************************************
    Atari 400/800

    MESS Driver

    Juergen Buchmueller, June 1998

    2009-05 FP changes:
     Factored out MESS specific code from MAME
     Added skeleton support for other XL/XE machines (VERY preliminary):
     - a600xl based on maxaflex emulation in MAME
     - a1200xl sharing a800xl code without BASIC
     - a65xe, a65xea, a130xe, a800xe, xegs sharing a800xl code (and this is wrong
      at least for xegs)
     Added proper dumps and labels, thanks to Freddy Offenga researches (a few
     are still marked BAD_DUMP while waiting for crc confirmation, since they
     have been obtained by splitting whole dumps)

     To Do:
     - Find out why a600xl and a800xl don't work (xe machines should then follow)
     - Investigate supported RAM sizes and OS versions in different models
     - Implement differences between various models (currently most of the
      XL/XE are exactly an a800xl, but this will change as soon as emulation
      starts to work)
     - Fix various keyboard differences
     - Freddy emulation for 800XLF?
     - Add support for proto boards and expansions (a1400xl, C/PM board, etc.)
     - Clean up the whole driver + cart + floppy structure

    2013-11-06 Robert Tuccitto:
    Updated Palette per 'CGIA D020577' and 'GTIA C014805', including
    normalized grayscale with proper color gradient.  Added Phase Shift
    values 24.7 thru 27.7 degrees in 0.5 degree increments.  Enabled
    Phase Shift 26.2 degrees as default.

    2013-11-23 Robert Tuccitto:
    Added palette notes

******************************************************************************/

#include "emu.h"
#include "cpu/m6502/m6502.h"
#include "includes/atari.h"
#include "machine/ataridev.h"
#include "imagedev/cartslot.h"
#include "sound/pokey.h"
#include "machine/6821pia.h"
#include "video/gtia.h"
#include "sound/dac.h"
#include "machine/ram.h"
#include "hashfile.h"
#include "drivlgcy.h"
#include "scrlegcy.h"


/******************************************************************************
    Atari 800 memory map (preliminary)

    ***************** read access *******************
    range     short   description
    0000-9FFF RAM     main memory
    A000-BFFF RAM/ROM RAM or (banked) ROM cartridges
    C000-CFFF ROM     unused or monitor ROM

    ********* GTIA    ********************************

    D000      m0pf    missile 0 playfield collisions
    D001      m1pf    missile 1 playfield collisions
    D002      m2pf    missile 2 playfield collisions
    D003      m3pf    missile 3 playfield collisions
    D004      p0pf    player 0 playfield collisions
    D005      p1pf    player 1 playfield collisions
    D006      p2pf    player 2 playfield collisions
    D007      p3pf    player 3 playfield collisions
    D008      m0pl    missile 0 player collisions
    D009      m1pl    missile 1 player collisions
    D00A      m2pl    missile 2 player collisions
    D00B      m3pl    missile 3 player collisions
    D00C      p0pl    player 0 player collisions
    D00D      p1pl    player 1 player collisions
    D00E      p2pl    player 2 player collisions
    D00F      p3pl    player 3 player collisions
    D010      but0    button stick 0
    D011      but1    button stick 1
    D012      but2    button stick 2
    D013      but3    button stick 3
    D014      xff     unused
    D015      xff     unused
    D016      xff     unused
    D017      xff     unused
    D018      xff     unused
    D019      xff     unused
    D01A      xff     unused
    D01B      xff     unused
    D01C      xff     unused
    D01D      xff     unused
    D01E      xff     unused
    D01F      cons    console keys
    D020-D0FF repeated 7 times

    D100-D1FF xff

    ********* POKEY   ********************************
    D200      pot0    paddle 0
    D201      pot1    paddle 1
    D202      pot2    paddle 2
    D203      pot3    paddle 3
    D204      pot4    paddle 4
    D205      pot5    paddle 5
    D206      pot6    paddle 6
    D207      pot7    paddle 7
    D208      potb    all paddles
    D209      kbcode  keyboard scan code
    D20A      random  random number generator
    D20B      xff     unused
    D20C      xff     unused
    D20D      serin   serial input
    D20E      irqst   IRQ status
    D20F      skstat  sk status
    D210-D2FF repeated 15 times

    ********* PIO     ********************************
    D300      porta   read pio port A
    D301      portb   read pio port B
    D302      pactl   read pio port A control
    D303      pbctl   read pio port B control
    D304-D3FF repeated 63 times

    ********* ANTIC   ********************************
    D400      xff     unused
    D401      xff     unused
    D402      xff     unused
    D403      xff     unused
    D404      xff     unused
    D405      xff     unused
    D406      xff     unused
    D407      xff     unused
    D408      xff     unused
    D409      xff     unused
    D40A      xff     unused
    D40B      vcount  vertical (scanline) counter
    D40C      penh    light pen horizontal pos
    D40D      penv    light pen vertical pos
    D40E      xff     unused
    D40F      nmist   NMI status

    D500-D7FF xff     unused memory

    D800-DFFF ROM     floating point ROM
    E000-FFFF ROM     bios ROM

    ***************** write access *******************
    range     short   description
    0000-9FFF RAM     main memory
    A000-BFFF RAM/ROM RAM or (banked) ROM
    C000-CFFF ROM     unused or monitor ROM

    ********* GTIA    ********************************
    D000      hposp0  player 0 horz position
    D001      hposp1  player 1 horz position
    D002      hposp2  player 2 horz position
    D003      hposp3  player 3 horz position
    D004      hposm0  missile 0 horz position
    D005      hposm1  missile 0 horz position
    D006      hposm2  missile 0 horz position
    D007      hposm3  missile 0 horz position
    D008      sizep0  size player 0
    D009      sizep1  size player 0
    D00A      sizep2  size player 0
    D00B      sizep3  size player 0
    D00C      sizem   size missiles
    D00D      grafp0  graphics data for player 0
    D00E      grafp1  graphics data for player 1
    D00F      grafp2  graphics data for player 2
    D010      grafp3  graphics data for player 3
    D011      grafm   graphics data for missiles
    D012      colpm0  color for player/missile 0
    D013      colpm1  color for player/missile 1
    D014      colpm2  color for player/missile 2
    D015      colpm3  color for player/missile 3
    D016      colpf0  color 0 playfield
    D017      colpf1  color 1 playfield
    D018      colpf2  color 2 playfield
    D019      colpf3  color 3 playfield
    D01A      colbk   background playfield
    D01B      prior   priority select
    D01C      vdelay  delay until vertical retrace
    D01D      gractl  graphics control
    D01E      hitclr  clear collisions
    D01F      wcons   write console (speaker)
    D020-D0FF repeated 7 times

    D100-D1FF xff     unused

    ********* POKEY   ********************************
    D200      audf1   frequency audio chan #1
    D201      audc1   control audio chan #1
    D202      audf2   frequency audio chan #2
    D203      audc2   control audio chan #2
    D204      audf3   frequency audio chan #3
    D205      audc3   control audio chan #3
    D206      audf4   frequency audio chan #4
    D207      audc4   control audio chan #4
    D208      audctl  audio control
    D209      stimer  start timer
    D20A      skres   sk reset
    D20B      potgo   start pot AD conversion
    D20C      xff     unused
    D20D      serout  serial output
    D20E      irqen   IRQ enable
    D20F      skctl   sk control
    D210-D2FF repeated 15 times

    ********* PIO     ********************************
    D300      porta   write pio port A (output or mask)
    D301      portb   write pio port B (output or mask)
    D302      pactl   write pio port A control
    D303      pbctl   write pio port B control
    D304-D3FF         repeated

    ********* ANTIC   ********************************
    D400      dmactl  write DMA control
    D401      chactl  write character control
    D402      dlistl  write display list lo
    D403      dlisth  write display list hi
    D404      hscrol  write horz scroll
    D405      vscrol  write vert scroll
    D406      xff     unused
    D407      pmbash  player/missile base addr hi
    D408      xff     unused
    D409      chbash  character generator base addr hi
    D40A      wsync   wait for hsync
    D40B      xff     unused
    D40C      xff     unused
    D40D      xff     unused
    D40E      nmien   NMI enable
    D40F      nmires  NMI reset

    D500-D7FF xff     unused memory

    D800-DFFF ROM     floating point ROM
    E000-FFFF ROM     BIOS ROM
******************************************************************************/

#define LEFT_CARTSLOT_MOUNTED  1
#define RIGHT_CARTSLOT_MOUNTED 2

/* PCB */
enum
{
	A800_UNKNOWN = 0,
	A800_4K, A800_8K, A800_12K, A800_16K,
	A800_RIGHT_4K, A800_RIGHT_8K,
	OSS_034M, OSS_M091, PHOENIX_8K, XEGS_32K,
	BBSB, DIAMOND_64K, WILLIAMS_64K, EXPRESS_64,
	SPARTADOS_X
};


class a400_state : public driver_device
{
public:
	a400_state(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag),
		m_maincpu(*this, "maincpu"),
		m_ram(*this, RAM_TAG),
		m_pia(*this, "pia"),
		m_region_maincpu(*this, "maincpu"),
		m_region_lslot(*this, "lslot"),
		m_region_rslot(*this, "rslot"),
		m_region_user1(*this, "user1"),
		m_a000(*this, "a000"),
		m_b000(*this, "b000"),
		m_0000(*this, "0000"),
		m_8000(*this, "8000"),
		m_9000(*this, "9000"),
		m_bank0(*this, "bank0"),
		m_bank1(*this, "bank1"),
		m_bank2(*this, "bank2"),
		m_bank3(*this, "bank3"),
		m_bank4(*this, "bank4"),
		m_a800_cart_loaded(0),
		m_atari(0),
		m_a800_cart_type(A800_UNKNOWN),
		m_xegs_banks(0),
		m_xegs_cart(0) { }

	DECLARE_DRIVER_INIT(xegs);
	DECLARE_DRIVER_INIT(a800xl);
	DECLARE_DRIVER_INIT(a600xl);
	DECLARE_MACHINE_START(xegs);
	DECLARE_MACHINE_START(a400);
	DECLARE_MACHINE_START(a800);
	DECLARE_MACHINE_START(a800xl);
	DECLARE_MACHINE_START(a5200);
	virtual void palette_init();
	DECLARE_WRITE8_MEMBER(a1200xl_pia_pb_w);
	DECLARE_WRITE8_MEMBER(a800xl_pia_pb_w);
	DECLARE_WRITE8_MEMBER(xegs_pia_pb_w);
	DECLARE_WRITE8_MEMBER(x32_bank_w);
	DECLARE_WRITE8_MEMBER(w64_bank_w);
	DECLARE_WRITE8_MEMBER(ex64_bank_w);
	DECLARE_WRITE8_MEMBER(bbsb_bankl_w);
	DECLARE_WRITE8_MEMBER(bbsb_bankh_w);
	DECLARE_WRITE8_MEMBER(oss_034m_w);
	DECLARE_WRITE8_MEMBER(oss_m091_w);
	DECLARE_WRITE8_MEMBER(xegs_bankswitch);

	DECLARE_DEVICE_IMAGE_LOAD_MEMBER( a800_cart );
	DECLARE_DEVICE_IMAGE_UNLOAD_MEMBER( a800_cart );

	DECLARE_DEVICE_IMAGE_LOAD_MEMBER( a800_cart_right );
	DECLARE_DEVICE_IMAGE_UNLOAD_MEMBER( a800_cart_right );

	DECLARE_DEVICE_IMAGE_LOAD_MEMBER( a5200_cart );
	DECLARE_DEVICE_IMAGE_UNLOAD_MEMBER( a5200_cart );

	DECLARE_DEVICE_IMAGE_LOAD_MEMBER( xegs_cart );
	DECLARE_DEVICE_IMAGE_UNLOAD_MEMBER( xegs_cart );

	void ms_atari_machine_start(int type, int has_cart);
	void ms_atari800xl_machine_start(int type, int has_cart);

protected:
	required_device<cpu_device> m_maincpu;
	required_device<ram_device> m_ram;
	required_device<pia6821_device> m_pia;
	required_memory_region m_region_maincpu;
	optional_memory_region m_region_lslot;
	optional_memory_region m_region_rslot;
	optional_memory_region m_region_user1;
	optional_memory_bank m_a000;
	optional_memory_bank m_b000;
	optional_memory_bank m_0000;
	optional_memory_bank m_8000;
	optional_memory_bank m_9000;
	optional_memory_bank m_bank0;
	optional_memory_bank m_bank1;
	optional_memory_bank m_bank2;
	optional_memory_bank m_bank3;
	optional_memory_bank m_bank4;

	int m_a800_cart_loaded;
	int m_atari;
	int m_a800_cart_type;
	UINT8 m_xegs_banks;
	UINT8 m_xegs_cart;

	void a800_setbank(int cart_mounted);
	void a800xl_mmu(UINT8 new_mmu);
	void a1200xl_mmu(UINT8 new_mmu);
	void xegs_mmu(UINT8 new_mmu);
	void a800_setup_mappers(int type);
	int a800_get_pcb_id(const char *pcb);
	int a800_get_type(device_image_interface &image);
	int a800_check_cart_type(device_image_interface &image);
};

/**************************************************************
 *
 * Memory maps
 *
 **************************************************************/


static ADDRESS_MAP_START(a400_mem, AS_PROGRAM, 8, a400_state)
	AM_RANGE(0x0000, 0x9fff) AM_NOP /* RAM installed at runtime */
	AM_RANGE(0xa000, 0xbfff) AM_RAMBANK("a000")
	AM_RANGE(0xc000, 0xcfff) AM_ROM
	AM_RANGE(0xd000, 0xd0ff) AM_READWRITE_LEGACY(atari_gtia_r, atari_gtia_w)
	AM_RANGE(0xd100, 0xd1ff) AM_NOP
	AM_RANGE(0xd200, 0xd2ff) AM_DEVREADWRITE("pokey", pokey_device, read, write)
	AM_RANGE(0xd300, 0xd3ff) AM_DEVREADWRITE("pia", pia6821_device, read_alt, write_alt)
	AM_RANGE(0xd400, 0xd4ff) AM_READWRITE_LEGACY(atari_antic_r, atari_antic_w)
	AM_RANGE(0xd500, 0xd7ff) AM_NOP
	AM_RANGE(0xd800, 0xffff) AM_ROM
ADDRESS_MAP_END


static ADDRESS_MAP_START(a800_mem, AS_PROGRAM, 8, a400_state)
	AM_RANGE(0x0000, 0x7fff) AM_RAMBANK("0000")
	AM_RANGE(0x8000, 0x9fff) AM_RAMBANK("8000")
	AM_RANGE(0xa000, 0xbfff) AM_RAMBANK("a000")
	AM_RANGE(0xc000, 0xcfff) AM_ROM
	AM_RANGE(0xd000, 0xd0ff) AM_READWRITE_LEGACY(atari_gtia_r, atari_gtia_w)
	AM_RANGE(0xd100, 0xd1ff) AM_NOP
	AM_RANGE(0xd200, 0xd2ff) AM_DEVREADWRITE("pokey", pokey_device, read, write)
	AM_RANGE(0xd300, 0xd3ff) AM_DEVREADWRITE("pia", pia6821_device, read_alt, write_alt)
	AM_RANGE(0xd400, 0xd4ff) AM_READWRITE_LEGACY(atari_antic_r, atari_antic_w)
	AM_RANGE(0xd500, 0xd7ff) AM_NOP
	AM_RANGE(0xd800, 0xffff) AM_ROM
ADDRESS_MAP_END


static ADDRESS_MAP_START(a600xl_mem, AS_PROGRAM, 8, a400_state)
	AM_RANGE(0x0000, 0x3fff) AM_RAM
	AM_RANGE(0x5000, 0x57ff) AM_ROM AM_REGION("maincpu", 0x5000)    /* self test */
	AM_RANGE(0xa000, 0xbfff) AM_ROM /* BASIC */
	AM_RANGE(0xc000, 0xcfff) AM_ROM /* OS */
	AM_RANGE(0xd000, 0xd0ff) AM_READWRITE_LEGACY(atari_gtia_r, atari_gtia_w)
	AM_RANGE(0xd100, 0xd1ff) AM_NOP
	AM_RANGE(0xd200, 0xd2ff) AM_DEVREADWRITE("pokey", pokey_device, read, write)
	AM_RANGE(0xd300, 0xd3ff) AM_DEVREADWRITE("pia", pia6821_device, read_alt, write_alt)
	AM_RANGE(0xd400, 0xd4ff) AM_READWRITE_LEGACY(atari_antic_r, atari_antic_w)
	AM_RANGE(0xd500, 0xd7ff) AM_NOP
	AM_RANGE(0xd800, 0xffff) AM_ROM /* OS */
ADDRESS_MAP_END


static ADDRESS_MAP_START(a800xl_mem, AS_PROGRAM, 8, a400_state)
	AM_RANGE(0x0000, 0x4fff) AM_RAM
	AM_RANGE(0x5000, 0x57ff) AM_RAMBANK("bank2")
	AM_RANGE(0x5800, 0x9fff) AM_RAM
	AM_RANGE(0xa000, 0xbfff) AM_RAMBANK("bank1")
	AM_RANGE(0xc000, 0xcfff) AM_RAMBANK("bank3")
	AM_RANGE(0xd000, 0xd0ff) AM_READWRITE_LEGACY(atari_gtia_r, atari_gtia_w)
	AM_RANGE(0xd100, 0xd1ff) AM_NOP
	AM_RANGE(0xd200, 0xd2ff) AM_DEVREADWRITE("pokey", pokey_device, read, write)
	AM_RANGE(0xd300, 0xd3ff) AM_DEVREADWRITE("pia", pia6821_device, read_alt, write_alt)
	AM_RANGE(0xd400, 0xd4ff) AM_READWRITE_LEGACY(atari_antic_r, atari_antic_w)
	AM_RANGE(0xd500, 0xd7ff) AM_NOP
	AM_RANGE(0xd800, 0xffff) AM_RAMBANK("bank4")
ADDRESS_MAP_END

static ADDRESS_MAP_START(xegs_mem, AS_PROGRAM, 8, a400_state)
	AM_RANGE(0x0000, 0x4fff) AM_RAM
	AM_RANGE(0x5000, 0x57ff) AM_RAMBANK("bank2")
	AM_RANGE(0x5800, 0x7fff) AM_RAM
	AM_RANGE(0x8000, 0x9fff) AM_ROMBANK("bank0")
	AM_RANGE(0xa000, 0xbfff) AM_ROMBANK("bank1")
	AM_RANGE(0xc000, 0xcfff) AM_RAMBANK("bank3")
	AM_RANGE(0xd000, 0xd0ff) AM_READWRITE_LEGACY(atari_gtia_r, atari_gtia_w)
	AM_RANGE(0xd100, 0xd1ff) AM_NOP
	AM_RANGE(0xd200, 0xd2ff) AM_DEVREADWRITE("pokey", pokey_device, read, write)
	AM_RANGE(0xd300, 0xd3ff) AM_DEVREADWRITE("pia", pia6821_device, read_alt, write_alt)
	AM_RANGE(0xd400, 0xd4ff) AM_READWRITE_LEGACY(atari_antic_r, atari_antic_w)
	AM_RANGE(0xd500, 0xd7ff) AM_NOP
	AM_RANGE(0xd800, 0xffff) AM_RAMBANK("bank4")
ADDRESS_MAP_END


static ADDRESS_MAP_START(a5200_mem, AS_PROGRAM, 8, a400_state)
	AM_RANGE(0x0000, 0x3fff) AM_RAM
	AM_RANGE(0x4000, 0xbfff) AM_ROM
	AM_RANGE(0xc000, 0xc0ff) AM_READWRITE_LEGACY(atari_gtia_r, atari_gtia_w)
	AM_RANGE(0xd400, 0xd5ff) AM_READWRITE_LEGACY(atari_antic_r, atari_antic_w)
	AM_RANGE(0xe800, 0xe8ff) AM_DEVREADWRITE("pokey", pokey_device, read, write)
	AM_RANGE(0xf800, 0xffff) AM_ROM
ADDRESS_MAP_END



/**************************************************************
 *
 * Input ports
 *
 **************************************************************/


#define JOYSTICK_DELTA          10
#define JOYSTICK_SENSITIVITY    200

static INPUT_PORTS_START( atari_artifacting )
	PORT_START("artifacts")
	PORT_CONFNAME(0x40, 0x00, "Television Artifacts" )
	PORT_CONFSETTING(0x00, DEF_STR( Off ) )
	PORT_CONFSETTING(0x40, DEF_STR( On ) )
INPUT_PORTS_END



static INPUT_PORTS_START( atari_console )
	PORT_START("console")
	PORT_BIT(0x04, 0x04, IPT_KEYBOARD) PORT_NAME("CONS.2: Option") PORT_CODE(KEYCODE_F3)
	PORT_BIT(0x02, 0x02, IPT_KEYBOARD) PORT_NAME("CONS.1: Select") PORT_CODE(KEYCODE_F2)
	PORT_BIT(0x01, 0x01, IPT_KEYBOARD) PORT_NAME("CONS.0: Start") PORT_CODE(KEYCODE_F1)
INPUT_PORTS_END



static INPUT_PORTS_START( atari_digital_joystick2 )
	PORT_START("djoy_0_1")  /* IN1 digital joystick #1 + #2 (PIA port A) */
	PORT_BIT(0x01, 0x01, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x04, 0x04, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x08, 0x08, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(1)
	PORT_BIT(0x10, 0x10, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(2)
	PORT_BIT(0x20, 0x20, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x80, 0x80, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(2)

	PORT_START("djoy_2_3")  /* IN2 digital joystick #3 + #4 (PIA port B) */
	PORT_BIT(0x01, 0x01, IPT_UNUSED)
	PORT_BIT(0x02, 0x02, IPT_UNUSED)
	PORT_BIT(0x04, 0x04, IPT_UNUSED)
	PORT_BIT(0x08, 0x08, IPT_UNUSED)
	PORT_BIT(0x10, 0x10, IPT_UNUSED)
	PORT_BIT(0x20, 0x20, IPT_UNUSED)
	PORT_BIT(0x40, 0x40, IPT_UNUSED)
	PORT_BIT(0x80, 0x80, IPT_UNUSED)

	PORT_START("djoy_b")    /* IN3 digital joystick buttons (GTIA button bits) */
	PORT_BIT(0x01, 0x01, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x04, 0x04, IPT_UNUSED)
	PORT_BIT(0x08, 0x08, IPT_UNUSED)
	PORT_BIT(0x10, 0x10, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x20, 0x20, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_UNUSED)
	PORT_BIT(0x80, 0x80, IPT_UNUSED)
INPUT_PORTS_END



static INPUT_PORTS_START( atari_digital_joystick4 )
	PORT_START("djoy_0_1")  /* digital joystick #1 + #2 (PIA port A) */
	PORT_BIT(0x01, 0x01, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x04, 0x04, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(1)
	PORT_BIT(0x08, 0x08, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(1)
	PORT_BIT(0x10, 0x10, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(2)
	PORT_BIT(0x20, 0x20, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(2)
	PORT_BIT(0x80, 0x80, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(2)

	PORT_START("djoy_2_3")  /* digital joystick #3 + #4 (PIA port B) */
	PORT_BIT(0x01, 0x01, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(3)
	PORT_BIT(0x02, 0x02, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(3)
	PORT_BIT(0x04, 0x04, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(3)
	PORT_BIT(0x08, 0x08, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(3)
	PORT_BIT(0x10, 0x10, IPT_JOYSTICK_UP)    PORT_CODE(KEYCODE_8_PAD) PORT_CODE(JOYCODE_Y_UP_SWITCH)    PORT_PLAYER(4)
	PORT_BIT(0x20, 0x20, IPT_JOYSTICK_DOWN)  PORT_CODE(KEYCODE_2_PAD) PORT_CODE(JOYCODE_Y_DOWN_SWITCH)  PORT_PLAYER(4)
	PORT_BIT(0x40, 0x40, IPT_JOYSTICK_LEFT)  PORT_CODE(KEYCODE_4_PAD) PORT_CODE(JOYCODE_X_LEFT_SWITCH)  PORT_PLAYER(4)
	PORT_BIT(0x80, 0x80, IPT_JOYSTICK_RIGHT) PORT_CODE(KEYCODE_6_PAD) PORT_CODE(JOYCODE_X_RIGHT_SWITCH) PORT_PLAYER(4)

	PORT_START("djoy_b")    /* digital joystick buttons (GTIA button bits) */
	PORT_BIT(0x01, 0x01, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x04, 0x04, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON3) PORT_PLAYER(3)
	PORT_BIT(0x08, 0x08, IPT_BUTTON1) PORT_CODE(KEYCODE_0_PAD) PORT_CODE(JOYCODE_BUTTON4) PORT_PLAYER(4)
	PORT_BIT(0x10, 0x10, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x20, 0x20, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON3) PORT_PLAYER(3)
	PORT_BIT(0x80, 0x80, IPT_BUTTON2) PORT_CODE(KEYCODE_DEL_PAD) PORT_CODE(JOYCODE_BUTTON4) PORT_PLAYER(4)
INPUT_PORTS_END



/* 2009-04 FP:
Small note about natural keyboard support: currently,
- "Break" is mapped to 'F1'
- "Clear" is mapped to 'F2'
- "Atari" is mapped to 'F3'                         */

static INPUT_PORTS_START( atari_keyboard )
	PORT_START("keyboard_0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_L) PORT_CHAR('l') PORT_CHAR('L')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_J) PORT_CHAR('j') PORT_CHAR('J')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_COLON) PORT_CHAR(';') PORT_CHAR(':')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Break") PORT_CODE(KEYCODE_BACKSPACE) PORT_CHAR(UCHAR_MAMEKEY(F1))
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_K) PORT_CHAR('k') PORT_CHAR('K')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_QUOTE) PORT_CHAR('+') PORT_CHAR('\\')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_BACKSLASH) PORT_CHAR('*') PORT_CHAR('^')

	PORT_START("keyboard_1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_O) PORT_CHAR('o') PORT_CHAR('O')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) // None!
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_P) PORT_CHAR('p') PORT_CHAR('P')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_U) PORT_CHAR('u') PORT_CHAR('U')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Return") PORT_CODE(KEYCODE_ENTER) PORT_CHAR(13)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_I) PORT_CHAR('i') PORT_CHAR('I')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_OPENBRACE) PORT_CHAR('-') PORT_CHAR('_')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_CLOSEBRACE) PORT_CHAR('=') PORT_CHAR('|')

	PORT_START("keyboard_2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_V) PORT_CHAR('v') PORT_CHAR('V')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_C) PORT_CHAR('c') PORT_CHAR('C')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_B) PORT_CHAR('b') PORT_CHAR('B')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_X) PORT_CHAR('x') PORT_CHAR('X')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Z) PORT_CHAR('z') PORT_CHAR('Z')

	PORT_START("keyboard_3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_4) PORT_CHAR('4') PORT_CHAR('$')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_3) PORT_CHAR('3') PORT_CHAR('#')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_6) PORT_CHAR('6') PORT_CHAR('&')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Escape") PORT_CODE(KEYCODE_ESC) PORT_CHAR(27)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_5) PORT_CHAR('5') PORT_CHAR('%')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_2) PORT_CHAR('2') PORT_CHAR('\"')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_1) PORT_CHAR('1') PORT_CHAR('!')

	PORT_START("keyboard_4")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_COMMA) PORT_CHAR(',') PORT_CHAR('[')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_SPACE) PORT_CHAR(' ')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_STOP) PORT_CHAR('.') PORT_CHAR(']')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_N) PORT_CHAR('n') PORT_CHAR('N')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_M) PORT_CHAR('m') PORT_CHAR('M')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_SLASH) PORT_CHAR('/') PORT_CHAR('?')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Atari") PORT_CODE(KEYCODE_RCONTROL) PORT_CHAR(UCHAR_MAMEKEY(F3))

	PORT_START("keyboard_5")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_R) PORT_CHAR('r') PORT_CHAR('R')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_E) PORT_CHAR('e') PORT_CHAR('E')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Y) PORT_CHAR('y') PORT_CHAR('Y')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Tab") PORT_CODE(KEYCODE_TAB) PORT_CHAR(UCHAR_MAMEKEY(TAB))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_T) PORT_CHAR('t') PORT_CHAR('T')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_W) PORT_CHAR('w') PORT_CHAR('W')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Q) PORT_CHAR('q') PORT_CHAR('Q')

	PORT_START("keyboard_6")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_9) PORT_CHAR('9') PORT_CHAR('(')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_0) PORT_CHAR('0') PORT_CHAR(')')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_7) PORT_CHAR('7') PORT_CHAR('\'')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("BackS  Delete") PORT_CODE(KEYCODE_BACKSLASH2) PORT_CHAR(8) PORT_CHAR(UCHAR_MAMEKEY(DEL))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_8) PORT_CHAR('8') PORT_CHAR('@')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("<  Clear") PORT_CODE(KEYCODE_MINUS) PORT_CHAR('<') PORT_CHAR(UCHAR_MAMEKEY(F2))
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME(">  Insert") PORT_CODE(KEYCODE_EQUALS) PORT_CHAR('>') PORT_CHAR(UCHAR_MAMEKEY(INSERT))

	PORT_START("keyboard_7")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_F) PORT_CHAR('f') PORT_CHAR('F')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_H) PORT_CHAR('h') PORT_CHAR('H')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_D) PORT_CHAR('d') PORT_CHAR('D')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Caps") PORT_CODE(KEYCODE_CAPSLOCK) PORT_TOGGLE PORT_CHAR(UCHAR_MAMEKEY(CAPSLOCK))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_G) PORT_CHAR('g') PORT_CHAR('G')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_S) PORT_CHAR('s') PORT_CHAR('S')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_A) PORT_CHAR('a') PORT_CHAR('A')

	PORT_START("fake")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Shift") PORT_CODE(KEYCODE_LSHIFT) PORT_CODE(KEYCODE_RSHIFT) PORT_CHAR(UCHAR_SHIFT_1)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Ctrl") PORT_CODE(KEYCODE_LCONTROL) PORT_CHAR(UCHAR_SHIFT_2)
INPUT_PORTS_END


static INPUT_PORTS_START( atari_analog_paddles )
	PORT_START("analog_0") /* IN8 analog in #1 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1) PORT_REVERSE

	PORT_START("analog_1") /* IN9 analog in #2 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2) PORT_REVERSE

	PORT_START("analog_2") /* IN10 analog in #3 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3) PORT_REVERSE

	PORT_START("analog_3") /* IN11 analog in #4 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4) PORT_REVERSE

	PORT_START("analog_4") /* IN12 analog in #5 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(5) */

	PORT_START("analog_5") /* IN13 analog in #6 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(6) */

	PORT_START("analog_6") /* IN14 analog in #7 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(7) */

	PORT_START("analog_7") /* IN15 analog in #8 */
	PORT_BIT(0xff, 0x74, IPT_PADDLE) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_REVERSE /* PORT_PLAYER(8) */
INPUT_PORTS_END



static INPUT_PORTS_START( a800 )
	PORT_INCLUDE( atari_artifacting )
	PORT_INCLUDE( atari_console )
	PORT_INCLUDE( atari_digital_joystick4 )
	PORT_INCLUDE( atari_keyboard )
	PORT_INCLUDE( atari_analog_paddles )
INPUT_PORTS_END



static INPUT_PORTS_START( a800xl )
	PORT_INCLUDE( atari_artifacting )
	PORT_INCLUDE( atari_console )
	PORT_INCLUDE( atari_digital_joystick2 )
	PORT_INCLUDE( atari_keyboard )
	PORT_INCLUDE( atari_analog_paddles )
INPUT_PORTS_END



static INPUT_PORTS_START( a5200 )
	PORT_INCLUDE( atari_artifacting )

	PORT_START("djoy_b")    /* lower/upper buttons */
	PORT_BIT(0x01, 0x01, IPT_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_BUTTON1) PORT_PLAYER(2)
	PORT_BIT(0x04, 0x04, IPT_BUTTON1) PORT_PLAYER(3)
	PORT_BIT(0x08, 0x08, IPT_BUTTON1) PORT_PLAYER(4)
	PORT_BIT(0x10, 0x10, IPT_BUTTON2) PORT_PLAYER(1)
	PORT_BIT(0x20, 0x20, IPT_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_BUTTON2) PORT_PLAYER(3)
	PORT_BIT(0x80, 0x80, IPT_BUTTON2) PORT_PLAYER(4)

	PORT_START("keypad_0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("(Break)") PORT_CODE(KEYCODE_PAUSE)    // is this correct?
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[#]") PORT_CODE(KEYCODE_ENTER_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[0]") PORT_CODE(KEYCODE_0_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[*]") PORT_CODE(KEYCODE_PLUS_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad_1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("Reset") PORT_CODE(KEYCODE_F3)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[9]") PORT_CODE(KEYCODE_9_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[8]") PORT_CODE(KEYCODE_8_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[7]") PORT_CODE(KEYCODE_7_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad_2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME(DEF_STR(Pause)) PORT_CODE(KEYCODE_F2)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[6]") PORT_CODE(KEYCODE_6_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[5]") PORT_CODE(KEYCODE_5_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[4]") PORT_CODE(KEYCODE_4_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad_3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_START)    PORT_NAME("Start")
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[3]") PORT_CODE(KEYCODE_3_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[2]") PORT_CODE(KEYCODE_2_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[1]") PORT_CODE(KEYCODE_1_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("analog_0")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1)

	PORT_START("analog_1")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1)

	PORT_START("analog_2")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2)

	PORT_START("analog_3")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2)

	PORT_START("analog_4")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3)

	PORT_START("analog_5")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3)

	PORT_START("analog_6")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4)

	PORT_START("analog_7")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4)

INPUT_PORTS_END
/***************************************************************
Atari 5200 Palette Notes:

Palette on a modern flat panel display (LCD, LED, Plasma, etc.)
appears different from a traditional CRT. The most outstanding
difference is Hue 1x, the hue begin point. Hue 1x looks very
'green' (~-60 to -45 degrees - depending on how poor or well it
handles the signal conversion and its calibration) on a modern
flat panel display, as opposed to 'gold' (~-33 degrees) on a
CRT.  The official technical document, "GTIA C014805 NTSC"
stipulates Hue 1x as gold.

The "Atari 5200 Field Service Manual" provides two different
sets of instructions in harmony with utilizing the "PAM
Diagnostic SALT Cartridge v1.1".  In one account it states the
color just below and above the reference bar to be within one
shade of each other.

Under the same reference document, directions are given for it
to be the same color.  Phase Shift 25.7 degrees matches Hue 1x,
15x and the color below the reference bar.

However, if the system is adjusted within the first several
minutes of running, the warm up, consistent system run time,
causes Hue 15x (F$) to become stronger/darker gold (More brown
then ultimately red-brown); as well as leans Hue 14x (E$) more
brown than green.  Once achieving a phase shift of 27.7,
Hue 14x (E$) and Hue 15x (F$) near-exact match Hue 1x and 2x
respectively.

Accounting for system 'warm-up', phase shifting, as well as the
instructions for it to be within one shade of each other, would
make Phase Shift 26.2 degrees or 26.7 degrees a realistic
logical choice.

It also collaborates with the official "GTIA C014805 NTSC"
document for color order: Hue 1x = Gold, Hue 2x = Orange,
Hue 15x (F$) = Light-Orange; Phase Shift 26.2 places
Hue 15x (F$) between Hue 1x, Gold and Hue 2x, Orange;
a Light Orange in color.  Color descriptions are best measured
in the middle of the brightness scale.

It should be mentioned that Green-Yellow is referenced at
Hue 13x (D$), nowhere near Hue 1x.  A Green-Yellow Hue 1x is
how the palette is manipulated and modified (in part) under
a modern flat panel display.

Note though, even a properly calibrated console, at power on,
the phase shift appears as low as ~23 degrees and after a
considerable consistent runtime, can be as high as ~28 degrees.
In general, the low end of ~23 degrees lasts for maybe several
seconds, whereas higher values such as ~25-27 degrees is the
most dominant during system run time.

Additionally, the blue to red (And consequently blue to green)
ratio proportions may appear different on a modern flat panel
display than a CRT in some instances for the Atari 5200 system.
Furthermore, you may have some variation of proportions even
within the same display type.

One side effect of this on the console's palette is that some
values of red may appear too pinkish - Too much blue to red.
This is not the same as a traditional tint-hue control
adjustment; rather, can be demonstrated by changing the blue
ratio values via MESS HLSL settings.

Lastly, the Atari 2600 & 7800 NTSC color palettes hold the same
hue structure order and have similar appearance differences
dependent upon display type.
***************************************************************/
/**************************************************************
 *
 * Palette - Phase Shift 26.2
 *
 **************************************************************/

static const UINT8 atari_palette[256*3] =
{
	/* Grey */
	0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
	0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
	0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
	0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,
	/* Gold */
	0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
	0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
	0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
	0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,
	/* Orange */
	0x31,0x00,0x00, 0x42,0x06,0x00, 0x53,0x17,0x00, 0x64,0x28,0x00,
	0x75,0x39,0x00, 0x86,0X4A,0x00, 0x97,0x5B,0x0B, 0xA8,0x6C,0x1C,
	0xB9,0x7D,0x2D, 0xCA,0x8E,0x3E, 0xDB,0x9F,0x4F, 0xEC,0xB0,0x60,
	0xFD,0xC1,0x71, 0xFF,0xD2,0x86, 0xFF,0xE3,0x9D, 0xFF,0xF4,0xB3,
	/* Red-Orange */
	0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x08,0x00, 0x71,0x19,0x00,
	0x82,0x2A,0x0F, 0x93,0x3B,0x20, 0xA4,0x4C,0x31, 0xB5,0x5D,0x42,
	0xC6,0x6E,0x53, 0xD7,0x7F,0x64, 0xE8,0x90,0x75, 0xF9,0xA1,0x86,
	0xFF,0xB2,0x9A, 0xFF,0xC3,0xB0, 0xFF,0xD4,0xC6, 0xFF,0xE5,0xDC,
	/* Pink */
	0x3E,0x00,0x06, 0x4F,0x00,0x12, 0x60,0x00,0x1E, 0x71,0x0E,0x2E,
	0x82,0x1F,0x3F, 0x93,0x30,0x50, 0xA4,0x41,0x61, 0xB5,0x52,0x72,
	0xC6,0x63,0x83, 0xD7,0x74,0x94, 0xE8,0x85,0xA5, 0xF9,0x96,0xB6,
	0xFF,0xA7,0xCB, 0xFF,0xB8,0xE1, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,
	/* Purple */
	0x32,0x00,0x38, 0x43,0x00,0x44, 0x54,0x00,0x50, 0x65,0x0C,0x5F,
	0x76,0x1D,0x70, 0x87,0x2E,0x81, 0x98,0x3F,0x92, 0xA9,0x50,0xA3,
	0xBA,0x61,0xB4, 0xCB,0x72,0xC5, 0xDC,0x83,0xD6, 0xED,0x94,0xE4,
	0xFE,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,
	/* Purple-Blue */
	0x1B,0x00,0x5F, 0x2C,0x00,0x6B, 0x3D,0x00,0x77, 0x4E,0x11,0x88,
	0x5F,0x22,0x99, 0x70,0x33,0xAA, 0x81,0x44,0xBB, 0x92,0x55,0xCC,
	0xA3,0x66,0xDD, 0xB4,0x77,0xED, 0xC5,0x88,0xED, 0xD6,0x99,0xED,
	0xE7,0xAA,0xED, 0xF8,0xBB,0xED, 0xFF,0xCC,0xF0, 0xFF,0xDD,0xF5,
	/* Blue 1 */
	0x00,0x00,0x72, 0x10,0x00,0x7E, 0x21,0x0D,0x8E, 0x32,0x1E,0x9F,
	0x43,0x2F,0xB0, 0x54,0x40,0xC1, 0x65,0x51,0xD2, 0x76,0x62,0xE3,
	0x87,0x73,0xF4, 0x98,0x84,0xF9, 0xA9,0x95,0xF9, 0xBA,0xA6,0xF9,
	0xCB,0xB7,0xF9, 0xDC,0xC8,0xF9, 0xED,0xD9,0xF9, 0xFE,0xEA,0xF9,
	/* Blue 2 */
	0x00,0x00,0x65, 0x00,0x0C,0x7A, 0x05,0x1D,0x8E, 0x16,0x2E,0x9F,
	0x27,0x3F,0xB0, 0x38,0x50,0xC1, 0x49,0x61,0xD2, 0x5A,0x72,0xE3,
	0x6B,0x83,0xF4, 0x7C,0x94,0xFF, 0x8D,0xA5,0xFF, 0x9E,0xB6,0xFF,
	0xAF,0xC7,0xFF, 0xC0,0xD8,0xFF, 0xD1,0xE9,0xFF, 0xE2,0xFA,0xFF,
	/* Light-Blue */
	0x00,0x0D,0x48, 0x00,0x1E,0x5E, 0x00,0x2F,0x74, 0x00,0x40,0x8A,
	0x11,0x51,0x9B, 0x22,0x62,0xAC, 0x33,0x73,0xBD, 0x44,0x84,0xCE,
	0x55,0x95,0xDF, 0x66,0xA6,0xF0, 0x77,0xB7,0xFF, 0x88,0xC8,0xFF,
	0x99,0xD9,0xFF, 0xAA,0xEA,0xFF, 0xBB,0xFB,0xFF, 0xCC,0xFF,0xFF,
	/* Turquoise */
	0x00,0x1C,0x1C, 0x00,0x2D,0x32, 0x00,0x3E,0x49, 0x00,0x4F,0x5F,
	0x05,0x60,0x73, 0x16,0x71,0x84, 0x27,0x82,0x95, 0x38,0x93,0xA6,
	0x49,0xA4,0xB7, 0x5A,0xB5,0xC8, 0x6B,0xC6,0xD9, 0x7C,0xD7,0xEA,
	0x8D,0xE8,0xFB, 0x9E,0xF9,0xFF, 0xAF,0xFF,0xFF, 0xC0,0xFF,0xFF,
	/* Green-Blue */
	0x00,0x25,0x0B, 0x00,0x36,0x10, 0x00,0x47,0x18, 0x00,0x58,0x2E,
	0x07,0x69,0x42, 0x18,0x7A,0x53, 0x29,0x8B,0x64, 0x3A,0x9C,0x75,
	0x4B,0xAD,0x86, 0x5C,0xBE,0x97, 0x6D,0xCF,0xA8, 0x7E,0xE0,0xB9,
	0x8F,0xF1,0xCA, 0xA0,0xFF,0xDA, 0xB1,0xFF,0xE6, 0xC2,0xFF,0xF2,
	/* Green */
	0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x04,0x5A,0x1A,
	0x15,0x6B,0x1A, 0x26,0x7C,0x22, 0x37,0x8D,0x33, 0x48,0x9E,0x44,
	0x59,0xAF,0x55, 0x6A,0xC0,0x66, 0x7B,0xD1,0x77, 0x8C,0xE2,0x88,
	0x9D,0xF3,0x99, 0xAE,0xFF,0xA8, 0xBF,0xFF,0xB4, 0xD0,0xFF,0xC0,
	/* Yellow-Green */
	0x00,0x21,0x0A, 0x00,0x32,0x0F, 0x0A,0x43,0x11, 0x1B,0x54,0x11,
	0x2C,0x65,0x11, 0x3D,0x76,0x11, 0x4E,0x87,0x11, 0x5F,0x98,0x1E,
	0x70,0xA9,0x2F, 0x81,0xBA,0x40, 0x92,0xCB,0x51, 0xA3,0xDC,0x62,
	0xB4,0xED,0x73, 0xC5,0xFE,0x84, 0xD6,0xFF,0x90, 0xE7,0xFF,0x9C,
	/* Orange-Green */
	0x05,0x13,0x04, 0x16,0x24,0x04, 0x27,0x35,0x04, 0x38,0x46,0x04,
	0x49,0x57,0x04, 0x5A,0x68,0x04, 0x6B,0x79,0x04, 0x7C,0x8A,0x09,
	0x8D,0x9B,0x1A, 0x9E,0xAC,0x2B, 0xAF,0xBD,0x3C, 0xC0,0xCE,0x4D,
	0xD1,0xDF,0x5E, 0xE2,0xF0,0x6F, 0xF3,0xFF,0x80, 0xFF,0xFF,0x8D,
	/* Light-Orange */
	0x21,0x02,0x00, 0x32,0x13,0x00, 0x43,0x24,0x00, 0x54,0x35,0x00,
	0x65,0x46,0x00, 0x76,0x57,0x00, 0x87,0x68,0x00, 0x98,0x79,0x0C,
	0xA9,0x8A,0x1D, 0xBA,0x9B,0x2E, 0xCB,0xAC,0x3F, 0xDC,0xBD,0x50,
	0xED,0xCE,0x61, 0xFE,0xDF,0x72, 0xFF,0xF0,0x87, 0xFF,0xFF,0x9D
};


/* Initialise the palette */
void a400_state::palette_init()
{
	int i;

	for ( i = 0; i < sizeof(atari_palette) / 3; i++ )
	{
		palette_set_color_rgb(machine(), i, atari_palette[i*3], atari_palette[i*3+1], atari_palette[i*3+2]);
	}
}
/******************************************************************
    PALETTE - PHASE 24.7 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x30,0x00,0x00, 0x41,0x07,0x00, 0x52,0x18,0x00, 0x63,0x29,0x00,
    0x74,0x3A,0x00, 0x85,0x4B,0x00, 0x96,0x5C,0x0A, 0xA7,0x6D,0x1B,
    0xB8,0x7E,0x2C, 0xC9,0x8F,0x3D, 0xDA,0xA0,0x4E, 0xEB,0xB1,0x5F,
    0xFC,0xC2,0x70, 0xFF,0xD3,0x85, 0xFF,0xE4,0x9B, 0xFF,0xF5,0xB1,

RED-ORANGE
    0x3D,0x00,0x00, 0x4E,0x00,0x00, 0x5F,0x09,0x00, 0x70,0x1A,0x00,
    0x81,0x2B,0x09, 0x92,0x3C,0x1A, 0xA3,0x4D,0x2B, 0xB4,0x5E,0x3C,
    0xC5,0x6F,0x4D, 0xD6,0x80,0x5E, 0xE7,0x91,0x6F, 0xF8,0xA2,0x80,
    0xFF,0xB3,0x94, 0xFF,0xC4,0xAA, 0xFF,0xD5,0xC0, 0xFF,0xE6,0xD6,

PINK
    0x3F,0x00,0x00, 0x50,0x00,0x09, 0x61,0x00,0x15, 0x72,0x10,0x26,
    0x83,0x21,0x37, 0x94,0x32,0x48, 0xA5,0x43,0x59, 0xB6,0x54,0x6A,
    0xC7,0x65,0x7B, 0xD8,0x76,0x8C, 0xE9,0x87,0x9D, 0xFA,0x98,0xAE,
    0xFF,0xA9,0xC2, 0xFF,0xBA,0xD8, 0xFF,0xCB,0xEE, 0xFF,0xDC,0xF4,

PURPLE
    0x36,0x00,0x2E, 0x47,0x00,0x3A, 0x58,0x00,0x46, 0x69,0x0C,0x55,
    0x7A,0x1D,0x66, 0x8B,0x2E,0x77, 0x9C,0x3F,0x88, 0xAD,0x50,0x99,
    0xBE,0x61,0xAA, 0xCF,0x72,0xBB, 0xE0,0x83,0xCC, 0xF1,0x94,0xDD,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x23,0x00,0x55, 0x34,0x00,0x61, 0x45,0x00,0x6D, 0x56,0x0F,0x7E,
    0x67,0x20,0x8F, 0x78,0x31,0xA0, 0x89,0x42,0xB1, 0x9A,0x53,0xC2,
    0xAB,0x64,0xD3, 0xBC,0x75,0xE4, 0xCD,0x86,0xEA, 0xDE,0x97,0xEA,
    0xEF,0xA8,0xEA, 0xFF,0xB9,0xEA, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

BLUE1
    0x09,0x00,0x6E, 0x1A,0x00,0x7A, 0x2B,0x08,0x88, 0x3C,0x19,0x99,
    0x4D,0x2A,0xAA, 0x5E,0x3B,0xBB, 0x6F,0x4C,0xCC, 0x80,0x5D,0xDD,
    0x91,0x6E,0xEE, 0xA2,0x7F,0xF4, 0xB3,0x90,0xF4, 0xC4,0xA1,0xF4,
    0xD5,0xB2,0xF4, 0xE6,0xC3,0xF4, 0xF7,0xD4,0xF4, 0xFF,0xE5,0xF7,

BLUE2
    0x00,0x00,0x6D, 0x00,0x05,0x80, 0x10,0x16,0x91, 0x21,0x27,0xA2,
    0x32,0x38,0xB3, 0x43,0x49,0xC4, 0x54,0x5A,0xD5, 0x65,0x6B,0xE6,
    0x76,0x7C,0xF7, 0x87,0x8D,0xFF, 0x98,0x9E,0xFF, 0xA9,0xAF,0xFF,
    0xBA,0xC0,0xFF, 0xCB,0xD1,0xFF, 0xDC,0xE2,0xFF, 0xED,0xF3,0xFF

LIGHT-BLUE
    0x00,0x05,0x57, 0x00,0x16,0x6E, 0x00,0x27,0x84, 0x09,0x38,0x97,
    0x1A,0x49,0xA8, 0x2B,0x5A,0xB9, 0x3C,0x6B,0xCA, 0x4D,0x7C,0xDB,
    0x5E,0x8D,0xEC, 0x6F,0x9E,0xFD, 0x80,0xAF,0xFF, 0x91,0xC0,0xFF,
    0xA2,0xD1,0xFF, 0xB3,0xE2,0xFF, 0xC4,0xF3,0xFF, 0xD5,0xFF,0xFF,

TURQUOISE
    0x00,0x15,0x34, 0x00,0x26,0x4A, 0x00,0x37,0x60, 0x00,0x48,0x77,
    0x0A,0x59,0x8A, 0x1B,0x6A,0x9B, 0x2C,0x7B,0xAC, 0x3D,0x8C,0xBD,
    0x4E,0x9D,0xCE, 0x5F,0xAE,0xDF, 0x70,0xBF,0xF0, 0x81,0xD0,0xFF,
    0x92,0xE1,0xFF, 0xA3,0xF2,0xFF, 0xB4,0xFF,0xFF, 0xC5,0xFF,0xFF

GREEN-BLUE
    0x00,0x21,0x0A, 0x00,0x32,0x1F, 0x00,0x43,0x35, 0x00,0x54,0x4B,
    0x04,0x65,0x60, 0x15,0x76,0x71, 0x26,0x87,0x82, 0x37,0x98,0x93,
    0x48,0xA9,0xA4, 0x59,0xBA,0xB5, 0x6A,0xCB,0xC6, 0x7B,0xDC,0xD7,
    0x8C,0xED,0xE8, 0x9D,0xFE,0xF9, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1D,
    0x0A,0x6B,0x30, 0x1B,0x7C,0x41, 0x2C,0x8D,0x52, 0x3D,0x9E,0x63,
    0x4E,0xAF,0x74, 0x5F,0xC0,0x85, 0x70,0xD1,0x96, 0x81,0xE2,0xA7,
    0x92,0xF3,0xB8, 0xA3,0xFF,0xC8, 0xB4,0xFF,0xD3, 0xC5,0xFF,0xDF,

YELLOW-GREEN
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x0A,0x59,0x18,
    0x1B,0x6A,0x18, 0x2C,0x7B,0x18, 0x3D,0x8C,0x27, 0x4E,0x9D,0x38,
    0x5F,0xAE,0x49, 0x70,0xBF,0x5A, 0x81,0xD0,0x6B, 0x92,0xE1,0x7C,
    0xA3,0xF2,0x8D, 0xB4,0xFF,0x9C, 0xC5,0xFF,0xA8, 0xD6,0xFF,0xB4,

ORANGE-GREEN
    0x00,0x1E,0x09, 0x00,0x2F,0x0E, 0x11,0x40,0x0E, 0x22,0x51,0x0E,
    0x33,0x62,0x0E, 0x44,0x73,0x0E, 0x55,0x84,0x0E, 0x66,0x95,0x17,
    0x77,0xA6,0x28, 0x88,0xB7,0x39, 0x99,0xC8,0x4A, 0xAA,0xD9,0x5B,
    0xBB,0xEA,0x6C, 0xCC,0xFB,0x7D, 0xDD,0xFF,0x8A, 0xEE,0xFF,0x96,

LIGHT-ORANGE
    0x0A,0x11,0x02, 0x1B,0x22,0x02, 0x2C,0x33,0x02, 0x3D,0x44,0x02,
    0x4E,0x55,0x02, 0x5F,0x66,0x02, 0x70,0x77,0x02, 0x81,0x88,0x09,
    0x92,0x99,0x1A, 0xA3,0xAA,0x2B, 0xB4,0xBB,0x3C, 0xC5,0xCC,0x4D,
    0xD6,0xDD,0x5E, 0xE7,0xEE,0x6F, 0xF8,0xFF,0x80, 0xFF,0xFF,0x8F,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 25.2 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x30,0x00,0x00, 0x41,0x07,0x00, 0x52,0x18,0x00, 0x63,0x29,0x00,
    0x74,0x3A,0x00, 0x85,0x4B,0x00, 0x96,0x5C,0x0A, 0xA7,0x6D,0x1B,
    0xB8,0x7E,0x2C, 0xC9,0x8F,0x3D, 0xDA,0xA0,0x4E, 0xEB,0xB1,0x5F,
    0xFC,0xC2,0x70, 0xFF,0xD3,0x85, 0xFF,0xE4,0x9B, 0xFF,0xF5,0xB1,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x09,0x00, 0x71,0x1A,0x00,
    0x82,0x2B,0x0B, 0x93,0x3C,0x1C, 0xA4,0x4D,0x2D, 0xB5,0x5E,0x3E,
    0xC6,0x6F,0x4F, 0xD7,0x80,0x60, 0xE8,0x91,0x71, 0xF9,0xA2,0x82,
    0xFF,0xB3,0x96, 0xFF,0xC4,0xAC, 0xFF,0xD5,0xC2, 0xFF,0xE6,0xD8,

PINK
    0x3F,0x00,0x00, 0x50,0x00,0x0C, 0x61,0x00,0x18, 0x72,0x0F,0x28,
    0x83,0x20,0x39, 0x94,0x31,0x4A, 0xA5,0x42,0x5B, 0xB6,0x53,0x6C,
    0xC7,0x64,0x7D, 0xD8,0x75,0x8E, 0xE9,0x86,0x9F, 0xFA,0x97,0xB0,
    0xFF,0xA8,0xC5, 0xFF,0xB9,0xDB, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

PURPLE
    0x35,0x00,0x31, 0x46,0x00,0x3D, 0x57,0x00,0x49, 0x68,0x0C,0x58,
    0x79,0x1D,0x69, 0x8A,0x2E,0x7A, 0x9B,0x3F,0x8B, 0xAC,0x50,0x9C,
    0xBD,0x61,0xAD, 0xCE,0x72,0xBE, 0xDF,0x83,0xCF, 0xF0,0x94,0xE0,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x20,0x00,0x59, 0x31,0x00,0x65, 0x42,0x00,0x71, 0x53,0x10,0x82,
    0x64,0x21,0x93, 0x75,0x32,0xA4, 0x86,0x43,0xB5, 0x97,0x54,0xC6,
    0xA8,0x65,0xD7, 0xB9,0x76,0xE8, 0xCA,0x87,0xEB, 0xDB,0x98,0xEB,
    0xEC,0xA9,0xEB, 0xFD,0xBA,0xEB, 0xFF,0xCB,0xEF, 0xFF,0xDC,0xF4,

BLUE1
    0x05,0x00,0x70, 0x16,0x00,0x7C, 0x27,0x09,0x8B, 0x38,0x1A,0x9C,
    0x49,0x2B,0xAD, 0x5A,0x3C,0xBE, 0x6B,0x4D,0xCF, 0x7C,0x5E,0xE0,
    0X8D,0x6F,0xF1, 0x9E,0x80,0xF6, 0xAF,0x91,0xF6, 0xC0,0xA2,0xF6,
    0xD1,0xB3,0xF6, 0xE2,0xC4,0xF6, 0xF3,0xD5,0xF6, 0xFF,0xE6,0xF7,

BLUE2
    0x00,0x00,0x6B, 0x00,0x08,0x7E, 0x0C,0x19,0x91, 0x1D,0x2A,0xA2,
    0x2E,0x3B,0xB3, 0x3F,0x4C,0xC4, 0x50,0x5D,0xD5, 0x61,0x6E,0xE6,
    0x72,0x7F,0xF7, 0x83,0x90,0xFF, 0x94,0xA1,0xFF, 0xA5,0xB2,0xFF,
    0xB6,0xC3,0xFF, 0xC7,0xD4,0xFF, 0xD8,0xE5,0xFF, 0xE9,0xF6,0xFF,

LIGHT-BLUE
    0x00,0x08,0x52, 0x00,0x19,0x68, 0x00,0x2A,0x7F, 0x05,0x3B,0x93,
    0x16,0x4C,0xA4, 0x27,0x5D,0xB5, 0x38,0x6E,0xC6, 0x49,0x7F,0xD7,
    0x5A,0x90,0xE8, 0x6B,0xA1,0xF9, 0x7C,0xB2,0xFF, 0x8D,0xC3,0xFF,
    0x9E,0xD4,0xFF, 0xAF,0xE5,0xFF, 0xC0,0xF6,0xFF, 0xD1,0xFF,0xFF,

TURQUOISE
    0x00,0x17,0x2D, 0x00,0x28,0x43, 0x00,0x39,0x59, 0x00,0x4A,0x6F,
    0x08,0x5B,0x83, 0x19,0x6C,0x94, 0x2A,0x7D,0xA5, 0x3B,0x8E,0xB6,
    0x4C,0x9F,0xC7, 0x5D,0xB0,0xD8, 0x6E,0xC1,0xE9, 0x7F,0xD2,0xFA,
    0x90,0xE3,0xFF, 0xA1,0xF4,0xFF, 0xB2,0xFF,0xFF, 0xC3,0xFF,0xFF,

GREEN-BLUE
    0x00,0x23,0x0A, 0x00,0x34,0x15, 0x00,0x45,0x2B, 0x00,0x56,0x41,
    0x04,0x67,0x56, 0x15,0x78,0x67, 0x26,0x89,0x78, 0x37,0x9A,0x89,
    0x48,0xAB,0x9A, 0x59,0xBC,0xAB, 0x6A,0xCD,0xBC, 0x7B,0xDE,0xCD,
    0x8C,0xEF,0xDE, 0x9D,0xFF,0xEE, 0xAE,0xFF,0xFA, 0xBF,0xFF,0xFF,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x0D,0x6B,0x25, 0x1E,0x7C,0x36, 0x2F,0x8D,0x47, 0x40,0x9E,0x58,
    0x51,0xAF,0x69, 0x62,0xC0,0x7A, 0x73,0xD1,0x8B, 0x84,0xE2,0x9C,
    0x95,0xF3,0xAD, 0xA6,0xFF,0xBD, 0xB7,0xFF,0xC9, 0xC8,0xFF,0xD4,

YELLOW-GREEN
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x00,0x46,0x15, 0x10,0x57,0x15,
    0x21,0x68,0x15, 0x32,0x79,0x15, 0x43,0x8A,0x1C, 0x54,0x9B,0x2D,
    0x65,0xAC,0x3E, 0x76,0xBD,0x4F, 0x87,0xCE,0x60, 0x98,0xDF,0x71,
    0xA9,0xF0,0x82, 0xBA,0xFF,0x93, 0xCB,0xFF,0x9F, 0xDC,0xFF,0xAA,

ORANGE-GREEN
    0x00,0x1B,0x08, 0x08,0x2C,0x0B, 0x19,0x3D,0x0B, 0x2A,0x4E,0x0B,
    0x3B,0x5F,0x0B, 0x4C,0x70,0x0B, 0x5D,0x81,0x0B, 0x6E,0x92,0x11,
    0x7F,0xA3,0x22, 0x90,0xB4,0x33, 0xA1,0xC5,0x44, 0xB2,0xD6,0x55,
    0xC3,0xE7,0x66, 0xD4,0xF8,0x77, 0xE5,0xFF,0x85, 0xF6,0xFF,0x91,

LIGHT-ORANGE
    0x12,0x0C,0x00, 0x23,0x1D,0x00, 0x34,0x2E,0x00, 0x45,0x3F,0x00,
    0x56,0x50,0x00, 0x67,0x61,0x00, 0x78,0x72,0x00, 0x89,0x83,0x08,
    0x9A,0x94,0x19, 0xAB,0xA5,0x2A, 0xBC,0xB6,0x3B, 0xCD,0xC7,0x4C,
    0xDE,0xD8,0x5D, 0xEF,0xE9,0x6E, 0xFF,0xFA,0x80, 0xFF,0xFF,0x92,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 25.7 SHIFT
GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x31,0x00,0x00, 0x42,0x06,0x00, 0x53,0x17,0x00, 0x64,0x28,0x00,
    0x75,0x39,0x00, 0x86,0X4A,0x00, 0x97,0x5B,0x0A, 0xA8,0x6C,0x1B,
    0xB9,0x7D,0x2C, 0xCA,0x8E,0x3D, 0xDB,0x9F,0x4E, 0xEC,0xB0,0x5F,
    0xFD,0xC1,0x70, 0xFF,0xD2,0x85, 0xFF,0xE3,0x9C, 0xFF,0xF4,0xB2,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x08,0x00, 0x71,0x19,0x00,
    0x82,0x2A,0x0D, 0x93,0x3B,0x1E, 0xA4,0x4C,0x2F, 0xB5,0x5D,0x40,
    0xC6,0x6E,0x51, 0xD7,0x7F,0x62, 0xE8,0x90,0x73, 0xF9,0xA1,0x83,
    0xFF,0xB2,0x98, 0xFF,0xC3,0xAE, 0xFF,0xD4,0xC4, 0xFF,0xE5,0xDA,

PINK
    0x3F,0x00,0x03, 0x50,0x00,0x0F, 0x61,0x00,0x1B, 0x72,0x0F,0x2B,
    0x83,0x20,0x3C, 0x94,0x31,0x4D, 0xA5,0x42,0x5E, 0xB6,0x53,0x6F,
    0xC7,0x64,0x80, 0xD8,0x75,0x91, 0xE9,0x86,0xA2, 0xFA,0x97,0xB3,
    0xFF,0xA8,0xC8, 0xFF,0xB9,0xDE, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

PURPLE
    0x33,0x00,0x35, 0x44,0x00,0x41, 0x55,0x00,0x4C, 0x66,0x0C,0x5C,
    0x77,0x1D,0x6D, 0x88,0x2E,0x7E, 0x99,0x3F,0x8F, 0xAA,0x50,0xA0,
    0xBB,0x61,0xB1, 0xCC,0x72,0xC2, 0xDD,0x83,0xD3, 0xEE,0x94,0xE4,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x1D,0x00,0x5C, 0x2E,0x00,0x68, 0x40,0x00,0x74, 0x51,0x10,0x84,
    0x62,0x21,0x95, 0x73,0x32,0xA6, 0x84,0x43,0xB7, 0x95,0x54,0xC8,
    0xA6,0x65,0xD9, 0xB7,0x76,0xEA, 0xC8,0x87,0xEB, 0xD9,0x98,0xEB,
    0xE9,0xA9,0xEC, 0xFB,0xBA,0xEB, 0xFF,0xCB,0xEF, 0xFF,0xDC,0xF4,

BLUE1
    0x02,0x00,0x71, 0x13,0x00,0x7D, 0x24,0x0B,0x8C, 0x35,0x1C,0x9D,
    0x46,0x2D,0xAE, 0x57,0x3E,0xBF, 0x68,0x4F,0xD0, 0x79,0x60,0xE1,
    0x8A,0x71,0xF2, 0x9B,0x82,0xF7, 0xAC,0x93,0xF7, 0xBD,0xA4,0xF7,
    0xCE,0xB5,0xF7, 0xDF,0xC6,0xF7, 0xF0,0xD7,0xF7, 0xFF,0xE8,0xF8,

BLUE2
    0x00,0x00,0x68, 0x00,0x0A,0x7C, 0x08,0x1B,0x90, 0x19,0x2C,0xA1,
    0x2A,0x3D,0xB2, 0x3B,0x4E,0xC3, 0x4C,0x5F,0xD4, 0x5D,0x70,0xE5,
    0x6E,0x81,0xF6, 0x7F,0x92,0xFF, 0x90,0xA3,0xFF, 0xA1,0xB4,0xFF,
    0xB2,0xC5,0xFF, 0xC3,0xD6,0xFF, 0xD4,0xE7,0xFF, 0xE5,0xF8,0xFF,

LIGHT-BLUE
    0x00,0x0A,0x4D, 0x00,0x1B,0x63, 0x00,0x2C,0x79, 0x02,0x3D,0x8F,
    0x13,0x4E,0xA0, 0x24,0x5F,0xB1, 0x35,0x70,0xC2, 0x46,0x81,0xD3,
    0x57,0x92,0xE4, 0x68,0xA3,0xF5, 0x79,0xB4,0xFF, 0x8A,0xC5,0xFF,
    0x9B,0xD6,0xFF, 0xAC,0xE7,0xFF, 0xBD,0xF8,0xFF, 0xCE,0xFF,0xFF,

TURQUOISE
    0x00,0x1A,0x26, 0x00,0x2B,0x3C, 0x00,0x3C,0x52, 0x00,0x4D,0x68,
    0x06,0x5E,0x7C, 0x17,0x6F,0x8D, 0x28,0x80,0x9E, 0x39,0x91,0xAF,
    0x4A,0xA2,0xC0, 0x5B,0xB3,0xD1, 0x6C,0xC4,0xE2, 0x7D,0xD5,0xF3,
    0x8E,0xE6,0xFF, 0x9F,0xF7,0xFF, 0xB0,0xFF,0xFF, 0xC1,0xFF,0xFF,

GREEN-BLUE
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x00,0x46,0x22, 0x00,0x57,0x38,
    0x05,0x68,0x4D, 0x16,0x79,0x5E, 0x27,0x8A,0x6F, 0x38,0x9B,0x80,
    0x49,0xAC,0x91, 0x5A,0xBD,0xA2, 0x6B,0xCE,0xB3, 0x7C,0xDF,0xC4,
    0x8D,0xF0,0xD5, 0x9E,0xFF,0xE5, 0xAF,0xFF,0xF1, 0xC0,0xFF,0xFD,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x10,0x6B,0x1B, 0x21,0x7C,0x2C, 0x32,0x8D,0x3D, 0x43,0x9E,0x4E,
    0x54,0xAF,0x5F, 0x65,0xC0,0x70, 0x76,0xD1,0x81, 0x87,0xE2,0x92,
    0x98,0xF3,0xA3, 0xA9,0xFF,0xB3, 0xBA,0xFF,0xBF, 0xCB,0xFF,0xCB,

YELLOW-GREEN
    0x00,0x23,0x0A, 0x00,0x34,0x10, 0x04,0x45,0x13, 0x15,0x56,0x13,
    0x26,0x67,0x13, 0x37,0x78,0x13, 0x48,0x89,0x14, 0x59,0x9A,0x25,
    0x6A,0xAB,0x36, 0x7B,0xBC,0x47, 0x8C,0xCD,0x58, 0x9D,0xDE,0x69,
    0xAE,0xEF,0x7A, 0xBF,0xFF,0x8B, 0xD0,0xFF,0x97, 0xE1,0xFF,0xA3,

ORANGE-GREEN
    0x00,0x17,0x07, 0x0E,0x28,0x08, 0x1F,0x39,0x08, 0x30,0x4A,0x08,
    0x41,0x5B,0x08, 0x52,0x6C,0x08, 0x63,0x7D,0x08, 0x74,0x8E,0x0D,
    0x85,0x9F,0x1E, 0x96,0xB0,0x2F, 0xA7,0xC1,0x40, 0xB8,0xD2,0x51,
    0xC9,0xE3,0x62, 0xDA,0xF4,0x73, 0xEB,0xFF,0x82, 0xFC,0xFF,0x8E,

LIGHT-ORANGE
    0x19,0x07,0x00, 0x2A,0x18,0x00, 0x3B,0x29,0x00, 0x4C,0x3A,0x00,
    0x5D,0x4B,0x00, 0x6E,0x5C,0x00, 0x7F,0x6D,0x00, 0x90,0x7E,0x09,
    0xA1,0x8F,0x1A, 0xB2,0xA0,0x2B, 0xC3,0xB1,0x3C, 0xD4,0xC2,0x4D,
    0xE5,0xD3,0x5E, 0xF6,0xE4,0x6F, 0xFF,0xF5,0x82, 0xFF,0xFF,0x96,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 26.7 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x06,0x00, 0x54,0x17,0x00, 0x65,0x28,0x00,
    0x79,0x39,0x00, 0x87,0x4A,0x00, 0x98,0x5B,0x0C, 0xA9,0x6C,0x1D,
    0xBA,0x7D,0x2E, 0xCB,0x8E,0x3F, 0xDC,0x9F,0x50, 0xED,0xB0,0x61,
    0xFE,0xC1,0x72, 0xFF,0xD2,0x87, 0xFF,0xE3,0x9E, 0xFF,0xF4,0xB4,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x07,0x00, 0x71,0x18,0x00,
    0x82,0x29,0x10, 0x93,0x3A,0x21, 0xA4,0x4B,0x32, 0xB5,0x5C,0x43,
    0xC6,0x6D,0x54, 0xD7,0x7E,0x65, 0xE8,0x8F,0x76, 0xF9,0xA0,0x87,
    0xFF,0xB1,0x9C, 0xFF,0xC2,0xB2, 0xFF,0xD3,0xC8, 0xFF,0xE4,0xDE,

PINK
    0x3E,0x00,0x09, 0x4F,0x00,0x15, 0x60,0x00,0x21, 0x71,0x0E,0x31,
    0x82,0x1F,0x42, 0x93,0x30,0x53, 0xA4,0x41,0x64, 0xB5,0x52,0x75,
    0xC6,0x63,0x86, 0xD7,0x74,0x97, 0xE8,0x85,0xA8, 0xF9,0x96,0xB9,
    0xFF,0xA7,0xCE, 0xFF,0xB8,0xE4, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,

PURPLE
    0x30,0x00,0x3D, 0x41,0x00,0x48, 0x52,0x00,0x54, 0x63,0x0C,0x64,
    0x74,0x1D,0x75, 0x85,0x2E,0x86, 0x96,0x3F,0x97, 0xA7,0x50,0xA8,
    0xB8,0x61,0xB9, 0xC9,0x72,0xCA, 0xDA,0x83,0xDB, 0xEB,0x94,0xE5,
    0xFC,0xA5,0xE5, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x18,0x00,0x62, 0x29,0x00,0x6E, 0x3A,0x01,0x7A, 0x4B,0x12,0x8B,
    0x5C,0x23,0x9C, 0x6D,0x34,0xAD, 0x7E,0x45,0xBE, 0x8F,0x56,0xCF,
    0xA0,0x67,0xE0, 0xB1,0x78,0xEE, 0xC2,0x89,0xEE, 0xD3,0x9A,0xEE,
    0xE4,0xAB,0xEE, 0xF5,0xBC,0xEE, 0xFF,0xCD,0xE0, 0xFF,0xDE,0xF5,

BLUE1
    0x00,0x00,0x72, 0x0C,0x00,0x7F, 0x1D,0x0E,0x8F, 0x2E,0x1F,0xA0,
    0x3F,0x30,0xB1, 0x50,0x41,0xC2, 0x61,0x52,0xD3, 0x72,0x63,0xE4,
    0x83,0x74,0xF5, 0x94,0x85,0xFA, 0xA5,0x96,0xFA, 0xB6,0xA7,0xFA,
    0xC7,0xB8,0xFA, 0xD8,0xC9,0xFA, 0xE9,0xDA,0xFA, 0xFA,0xE8,0xFA,

BLUE2
    0x00,0x00,0x62, 0x00,0x0F,0x77, 0x01,0x20,0x8D, 0x12,0x31,0x9E,
    0x23,0x42,0xAF, 0x34,0x53,0xC0, 0x45,0x64,0xD1, 0x56,0x75,0xE2,
    0x67,0x86,0xF3, 0x78,0x97,0xFF, 0x89,0xA8,0xFF, 0x9A,0xB9,0xFF,
    0xAB,0xCA,0xFF, 0xBC,0xDB,0xFF, 0xCD,0xEC,0xFF, 0xDE,0xFD,0xFF,

LIGHT-BLUE
    0x00,0x10,0x42, 0x00,0x21,0x58, 0x00,0x32,0x6E, 0x00,0x43,0x84,
    0x0E,0x54,0x96, 0x1F,0x65,0xA7, 0x30,0x76,0xB8, 0x41,0x87,0xC9,
    0x52,0x98,0xDA, 0x63,0xA9,0xEB, 0x74,0xBA,0xFC, 0x85,0xCB,0xFF,
    0x96,0xDC,0xFF, 0xA7,0xED,0xFF, 0xB8,0xFE,0xFF, 0xC9,0xFF,0xFF,

TURQUOISE
    0x00,0x1E,0x14, 0x00,0x2F,0x2A, 0x00,0x40,0x40, 0x00,0x51,0x56,
    0x04,0x62,0x6B, 0x15,0x73,0x7C, 0x26,0x84,0x8D, 0x37,0x95,0x9E,
    0x48,0xA6,0xAF, 0x59,0xB7,0xC0, 0x6A,0xC8,0xD1, 0x7B,0xD9,0xE2,
    0x8C,0xEA,0xF3, 0x9D,0xFB,0xFF, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x00,0x59,0x25,
    0x08,0x6A,0x38, 0x19,0x7B,0x49, 0x2A,0x8C,0x5A, 0x3B,0x9D,0x6B,
    0x4C,0xAE,0x7C, 0x5D,0xBF,0x8D, 0x6E,0xD0,0x9E, 0x7F,0xE1,0xAF,
    0x90,0xF2,0xC0, 0xA1,0xFF,0xD0, 0xB2,0xFF,0xDC, 0xC3,0xFF,0xE8,

GREEN
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x08,0x59,0x18,
    0x19,0x6A,0x18, 0x2A,0x7B,0x18, 0x3B,0x8C,0x29, 0x4C,0x9D,0x3A,
    0x5D,0xAE,0x4B, 0x6E,0xBF,0x5C, 0x7F,0xD0,0x6D, 0x90,0xE1,0x7E,
    0xA1,0xF2,0x8F, 0xB2,0xFF,0x9F, 0xC3,0xFF,0xAB, 0xD4,0xFF,0xB7,

YELLOW-GREEN
    0x00,0x1E,0x09, 0x00,0x2F,0x0E, 0x11,0x40,0x0E, 0x22,0x51,0x0E,
    0x33,0x62,0x0E, 0x44,0x73,0x0E, 0x55,0x84,0x0E, 0x66,0x95,0x17,
    0x77,0xA6,0x28, 0x88,0xB7,0x39, 0x99,0xC8,0x4A, 0xAA,0xD9,0x5B,
    0xBB,0xEA,0x6C, 0xCC,0xFB,0x7D, 0xDD,0xFF,0x8A, 0xEE,0xFF,0x96,

ORANGE-GREEN
    0x0D,0x0F,0x01, 0x1E,0x20,0x01, 0x2F,0x31,0x01, 0x40,0x42,0x01,
    0x51,0x53,0x01, 0x62,0x64,0x01, 0x73,0x75,0x01, 0x84,0x86,0x08,
    0x95,0x97,0x19, 0xA6,0xA8,0x2A, 0xB7,0xB9,0x3B, 0xC8,0xCA,0x4C,
    0xD9,0xDB,0x5D, 0xEA,0xEC,0x6E, 0xFB,0xFD,0x7F, 0xFF,0xFF,0x8F,

LIGHT-ORANGE
    0x28,0x00,0x00, 0x39,0x0E,0x00, 0x4A,0x1F,0x00, 0x5B,0x30,0x00,
    0x6C,0x41,0x00, 0x7D,0x52,0x00, 0x8E,0x63,0x00, 0x9F,0x74,0x10,
    0xB0,0x85,0x21, 0xC1,0x96,0x32, 0xD2,0xA7,0x43, 0xE3,0xB8,0x54,
    0xF4,0xC9,0x65, 0xFF,0xDA,0x78, 0xFF,0xEB,0x8E, 0xFF,0xFC,0xA4,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 27.2 SHIFT


GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x05,0x00, 0x54,0x16,0x00, 0x65,0x27,0x00,
    0x76,0x38,0x00, 0x87,0X49,0x00, 0x98,0x5A,0x0C, 0xA9,0x6B,0x1D,
    0xBA,0x7C,0x2E, 0xCB,0x8D,0x3F, 0xDC,0x9E,0x50, 0xED,0xAF,0x61,
    0xFE,0xC0,0x72, 0xFF,0xD1,0x88, 0xFF,0xE2,0x9E, 0xFF,0xF3,0xB4,

RED-ORANGE
    0x3F,0x00,0x00, 0x50,0x00,0x00, 0x61,0x07,0x00, 0x72,0x18,0x01,
    0x83,0x29,0x12, 0x94,0x3A,0x23, 0xA5,0x4B,0x34, 0xB6,0x5C,0x45,
    0xC7,0x6D,0x56, 0xD8,0x7E,0x67, 0xE9,0x8F,0x78, 0xFA,0xA0,0x89,
    0xFF,0xB1,0x9E, 0xFF,0xC2,0xB4, 0xFF,0xD3,0xCA, 0xFF,0xE4,0xE0,

PINK
    0x3E,0x00,0x0C, 0x4F,0x00,0x18, 0x60,0x00,0x24, 0x71,0x0E,0x34,
    0x82,0x1F,0x45, 0x93,0x30,0x56, 0xA4,0x41,0x67, 0xB5,0x52,0x78,
    0xC6,0x63,0x89, 0xD7,0x74,0x9A, 0xE8,0x85,0xAB, 0xF9,0x96,0xB6,
    0xFF,0xA7,0xD1, 0xFF,0xB8,0xE7, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,

PURPLE
    0x2F,0x00,0x3F, 0x40,0x00,0x4B, 0x51,0x00,0x57, 0x62,0x0C,0x66,
    0x73,0x1D,0x77, 0x84,0x2E,0x88, 0x95,0x3F,0x99, 0xA6,0x50,0xAA,
    0xB7,0x61,0xBB, 0xC8,0x72,0xCC, 0xD9,0x83,0xDD, 0xEA,0x94,0xE5,
    0xFB,0xA5,0xE5, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x16,0x00,0x64, 0x27,0x00,0x70, 0x38,0x02,0x7D, 0x49,0x13,0x8E,
    0x5A,0x24,0x9F, 0x6B,0x35,0xB0, 0x7C,0x46,0xC1, 0x8D,0x57,0xD2,
    0x9E,0x68,0xE3, 0xAF,0x79,0xEF, 0xC0,0x8A,0xEF, 0xD1,0x9D,0xEF,
    0xE2,0xAC,0xEF, 0xF3,0xBD,0xEF, 0xFF,0xCE,0xF0, 0xFF,0xDF,0xF5,

BLUE1
    0x00,0x00,0x71, 0x09,0x00,0x7F, 0x1A,0x10,0x90, 0x2B,0x21,0xA1,
    0x3C,0x32,0xB2, 0x4D,0x43,0xC3, 0x5E,0x54,0xD4, 0x6F,0x65,0xE5,
    0x80,0x76,0xF6, 0x91,0x87,0xFC, 0xA2,0x98,0xFC, 0xB3,0xA9,0xFC,
    0xC4,0xBA,0xFC, 0xD5,0xCB,0xFC, 0xE6,0xDC,0xFC, 0xF7,0xED,0xFC,

BLUE2
    0x00,0x00,0x5E, 0x00,0x11,0x74, 0x00,0x22,0x8A, 0x0F,0x33,0x9C,
    0x20,0x44,0xAD, 0x31,0x55,0xBE, 0x42,0x66,0xCF, 0x53,0x77,0xE0,
    0x64,0x88,0xF1, 0x75,0x99,0xFF, 0x86,0xAA,0xFF, 0x97,0xBB,0xFF,
    0xA8,0xCC,0xFF, 0xB9,0xDD,0xFF, 0xCA,0xEE,0xFF, 0xDB,0xFF,0xFF,

LIGHT-BLUE
    0x00,0x12,0x3B, 0x00,0x23,0x51, 0x00,0x34,0x68, 0x00,0x45,0x7E,
    0x0C,0x56,0x90, 0x1D,0x67,0xA1, 0x2E,0x78,0xB2, 0x3F,0x89,0xC3,
    0x50,0x9A,0xD4, 0x61,0xAB,0xE5, 0x72,0xBC,0xF6, 0x83,0xCD,0xFF,
    0x94,0xDE,0xFF, 0xA5,0xEF,0xFF, 0xB6,0xFF,0xFF, 0xC7,0xFF,0xFF,

TURQUOISE
    0x00,0x20,0x0C, 0x00,0x31,0x22, 0x00,0x42,0x38, 0x00,0x53,0x4E,
    0x04,0x64,0x63, 0x15,0x75,0x74, 0x26,0x86,0x85, 0x37,0x97,0x96,
    0x48,0xA8,0xA7, 0x59,0xB9,0xB8, 0x6A,0xCA,0xC9, 0x7B,0xDB,0xDA,
    0x8C,0xEC,0xEB, 0x9D,0xFD,0xFC, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1C,
    0x0B,0x6B,0x2F, 0x1C,0x7C,0x40, 0x2D,0x8D,0x51, 0x3E,0x9E,0x62,
    0x4F,0xAF,0x73, 0x60,0xC0,0x84, 0x71,0xD1,0x95, 0x82,0xE2,0xA6,
    0x93,0xF3,0xB7, 0xA4,0xFF,0xC6, 0xB5,0xFF,0xD2, 0xC6,0xFF,0xDE,

GREEN
    0x00,0x25,0x0B, 0x00,0x36,0x10, 0x00,0x47,0x15, 0x0D,0x58,0x16,
    0x1E,0x69,0x16, 0x2F,0x7A,0x16, 0x40,0x8B,0x21, 0x51,0x9C,0x32,
    0x62,0xAD,0x43, 0x73,0xBE,0x54, 0x84,0xCF,0x65, 0x95,0xE0,0x76,
    0xA6,0xF1,0x87, 0xB7,0xFF,0x98, 0xC8,0xFF,0xA3, 0xD9,0xFF,0xAF,

YELLOW-GREEN
    0x00,0x1B,0x08, 0x06,0x2C,0x0B, 0x17,0x3D,0x0B, 0x28,0x4E,0x0B,
    0x39,0x5F,0x0B, 0x4A,0x70,0x0B, 0x5B,0x81,0x0B, 0x6C,0x92,0x12,
    0x7D,0xA3,0x23, 0x8E,0xB4,0x34, 0x9F,0xC5,0x45, 0xB0,0xD6,0x56,
    0xC1,0xE7,0x67, 0xD2,0xF8,0x78, 0xE3,0xFF,0x86, 0xF4,0xFF,0x92,

ORANGE-GREEN
    0x13,0x0B,0x00, 0x24,0x1C,0x00, 0x35,0x2D,0x00, 0x46,0x3E,0x00,
    0x57,0x4F,0x00, 0x68,0x60,0x00, 0x79,0x71,0x00, 0x8A,0x82,0x08,
    0x9B,0x93,0x19, 0xAC,0xA4,0x2A, 0xBD,0xB5,0x3B, 0xCE,0xC6,0x4C,
    0xDF,0xD7,0x5D, 0xF0,0xE8,0x6E, 0xFF,0xF9,0x7F, 0xFF,0xFF,0x92,

LIGHT-ORANGE
    0x2D,0x00,0x00, 0x3E,0x0A,0x00, 0x4F,0x1B,0x00, 0x60,0x2C,0x00,
    0x71,0x3D,0x00, 0x82,0x4E,0x00, 0x93,0x5F,0x05, 0xA4,0x70,0x16,
    0xB5,0x81,0x27, 0xC4,0x90,0x37, 0xD7,0xA3,0x49, 0xE8,0xB4,0x5A,
    0xF9,0xC5,0x6B, 0xFF,0xD6,0x80, 0xFF,0xE7,0x96, 0xFF,0xF8,0xAC,
*******************************************************************

*******************************************************************
    PALETTE - 27.7 PHASE SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x05,0x00, 0x54,0x16,0x00, 0x65,0x27,0x00,
    0x76,0x38,0x00, 0x87,0x49,0x00, 0x98,0x5A,0x0C, 0xA9,0x6B,0x1D,
    0xBA,0x7C,0x2E, 0xCB,0x8D,0x3F, 0xDC,0x9E,0x50, 0xED,0xAF,0x61,
    0xFE,0xC0,0x72, 0xFF,0xD1,0x88, 0xFF,0xE2,0x9E, 0xFF,0xF3,0xB4,

RED-ORANGE
    0x3F,0x00,0x00, 0x50,0x00,0x00, 0x61,0x06,0x00, 0x72,0x17,0x03,
    0x83,0x28,0x14, 0x94,0x39,0x25, 0xA5,0x4A,0x36, 0xB6,0x5B,0x47,
    0xC7,0x6C,0x58, 0xD8,0x7D,0x69, 0xE9,0x8E,0x7A, 0xFA,0x9F,0x8B,
    0xFF,0xB0,0x9F, 0xFF,0xC1,0xB5, 0xFF,0xD2,0xCB, 0xFF,0xE3,0xE1,

PINK
    0x3D,0x00,0x10, 0x4E,0x00,0x1C, 0x5F,0x00,0x27, 0x70,0x0D,0x37,
    0x81,0x1E,0x48, 0x92,0x2F,0x59, 0xA3,0x40,0x6A, 0xB4,0x51,0x7B,
    0xC5,0x62,0x8C, 0xD6,0x73,0x9D, 0xE7,0x84,0xAE, 0xF8,0x95,0xBF,
    0xFF,0xA6,0xD3, 0xFF,0xB7,0xE9, 0xFF,0xC8,0xEE, 0xFF,0xD9,0xF4,

PURPLE
    0x2D,0x00,0x42, 0x3E,0x00,0x4E, 0x4F,0x00,0x5A, 0x60,0x0C,0x6A,
    0x71,0x1D,0x7B, 0x82,0x2E,0x8C, 0x93,0x3F,0x9D, 0xA4,0x50,0xAE,
    0xB5,0x61,0xBF, 0xC6,0x72,0xD0, 0xD7,0x83,0xE1, 0xE8,0x94,0xE6,
    0xF9,0xA5,0xE6, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x13,0x00,0x67, 0x24,0x00,0x73, 0x35,0x03,0x80, 0x46,0x14,0x91,
    0x57,0x25,0xA2, 0x68,0x36,0xB3, 0x79,0x47,0xC4, 0x8A,0x58,0xD5,
    0x9B,0x69,0xE6, 0xAC,0x7A,0xF0, 0xBD,0x8B,0xF0, 0xCE,0x9C,0xF0,
    0xDF,0xAD,0xF0, 0xF0,0xBE,0xF0, 0xFF,0xCF,0xF1, 0xFF,0xE0,0xF6,

BLUE1
    0x00,0x00,0x70, 0x05,0x01,0x80, 0x16,0x12,0x91, 0x27,0x23,0xA2,
    0x38,0x34,0xB3, 0x49,0x45,0xC4, 0x5A,0x56,0xD5, 0x6B,0x67,0xE6,
    0x7C,0x78,0xF7, 0x8D,0x89,0xFE, 0x9E,0x9A,0xFE, 0xAF,0xAB,0xFE,
    0xC0,0xBC,0xFE, 0xD1,0xCD,0xFE, 0xE2,0xDE,0xFE, 0xF3,0xEF,0xFE,

BLUE2
    0x00,0x03,0x5B, 0x00,0x14,0x71, 0x00,0x25,0x87, 0x0C,0x36,0x9A,
    0x1D,0x47,0xAB, 0x2E,0x58,0xBC, 0x3F,0x69,0xCD, 0x50,0x7A,0xDE,
    0x61,0x8B,0xEF, 0x72,0x9C,0xFF, 0x83,0xAD,0xFF, 0x94,0xBE,0xFF,
    0xA5,0xCF,0xFF, 0xB6,0xE0,0xFF, 0xC7,0xF1,0xFF, 0xD8,0xFF,0xFF,

LIGHT-BLUE
    0x00,0x15,0x35, 0x00,0x26,0x4B, 0x00,0x37,0x61, 0x00,0x48,0x78,
    0x0A,0x59,0x8B, 0x1B,0x6A,0x9C, 0x2C,0x7B,0xAD, 0x3D,0x8C,0xBE,
    0x4E,0x9D,0xCF, 0x5F,0xAE,0xE0, 0x70,0xBF,0xF1, 0x81,0xD0,0xFF,
    0x92,0xE1,0xFF, 0xA3,0xF2,0xFF, 0xB4,0xFF,0xFF, 0xC5,0xFF,0xFF,

TURQUOISE
    0x00,0x22,0x0A, 0x00,0x33,0x19, 0x00,0x44,0x2F, 0x00,0x55,0x45,
    0x04,0x66,0x5A, 0x15,0x77,0x6B, 0x26,0x88,0x7C, 0x37,0x99,0x8D,
    0x48,0xAA,0x9E, 0x59,0xBB,0xAF, 0x6A,0xCC,0xC0, 0x7B,0xDD,0xD1,
    0x8C,0xEE,0xE2, 0x9D,0xFF,0xF3, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x0D,0x6B,0x25, 0x1E,0x7C,0x36, 0x2F,0x8D,0x47, 0x40,0x9E,0x58,
    0x51,0xAF,0x69, 0x62,0xC0,0x7A, 0x73,0xD1,0x8B, 0x84,0xE2,0x9C,
    0x95,0xF3,0xAD, 0xA6,0xFF,0xBD, 0xB7,0xFF,0xC9, 0xC8,0xFF,0xD4,

GREEN
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x01,0x46,0x15, 0x12,0x57,0x15,
    0x23,0x68,0x15, 0x34,0x79,0x15, 0x45,0x8A,0x19, 0x56,0x9B,0x2A,
    0x67,0xAC,0x3B, 0x78,0xBD,0x4C, 0x89,0xCE,0x5D, 0x9A,0xDF,0x6E,
    0xAB,0xF0,0x7F, 0xBC,0xFF,0x8F, 0xCD,0xFF,0x9B, 0xDE,0xFF,0xA7,

YELLOW-GREEN
    0x00,0x18,0x07, 0x00,0x29,0x0C, 0x1E,0x3A,0x08, 0x2F,0x4B,0x08,
    0x40,0x5C,0x08, 0x51,0x6D,0x08, 0x62,0x7E,0x08, 0x73,0x8F,0x0D,
    0x84,0xA0,0x1E, 0x95,0xB1,0x2F, 0xA6,0xC2,0x40, 0xB7,0xD3,0x51,
    0xC8,0xE4,0x62, 0xD9,0xF5,0x73, 0xEA,0xFF,0x82, 0xFB,0xFF,0x8E,

ORANGE-GREEN
    0x1B,0x07,0x00, 0x2C,0x18,0x00, 0x3D,0x29,0x00, 0x4E,0x3A,0x00,
    0x5F,0x4B,0x00, 0x70,0x5C,0x00, 0x81,0x6D,0x00, 0x92,0x7E,0x09,
    0xA3,0x8F,0x1A, 0xB4,0xA0,0x2B, 0xC5,0xB1,0x3C, 0xD6,0xC2,0x4D,
    0xE7,0xD3,0x5E, 0xF8,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

LIGHT-ORANGE
    0x33,0x00,0x00, 0x44,0x05,0x00, 0x55,0x16,0x00, 0x66,0x27,0x00,
    0x77,0x38,0x00, 0x88,0x49,0x00, 0x99,0x5A,0x0D, 0xAA,0x6B,0x1E,
    0xBB,0x7C,0x2F, 0xCC,0x8D,0x40, 0xDD,0x9E,0x51, 0xEE,0xAF,0x62,
    0xFF,0xC0,0x73, 0xFF,0xD1,0x89, 0xFF,0xE2,0x9F, 0xFF,0xF3,0xB5
*******************************************************************/

/**************************************************************
 *
 * Memory banking
 *
 **************************************************************/

void a400_state::a800xl_mmu(UINT8 new_mmu)
{
	UINT8 *base = m_region_maincpu->base();
	UINT8 *base1, *base2, *base3, *base4;

	/* check if memory C000-FFFF changed */
	if( new_mmu & 0x01 )
	{
		logerror("%s MMU BIOS ROM\n", machine().system().name);
		base3 = base + 0x14000;  /* 8K lo BIOS */
		base4 = base + 0x15800;  /* 4K FP ROM + 8K hi BIOS */
		m_maincpu->space(AS_PROGRAM).install_read_bank(0xc000, 0xcfff, "bank3");
		m_maincpu->space(AS_PROGRAM).unmap_write(0xc000, 0xcfff);
		m_maincpu->space(AS_PROGRAM).install_read_bank(0xd800, 0xffff, "bank4");
		m_maincpu->space(AS_PROGRAM).unmap_write(0xd800, 0xffff);
	}
	else
	{
		logerror("%s MMU BIOS RAM\n", machine().system().name);
		base3 = base + 0x0c000;  /* 8K RAM */
		base4 = base + 0x0d800;  /* 4K RAM + 8K RAM */
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xc000, 0xcfff, "bank3");
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xd800, 0xffff, "bank4");
	}
	m_bank3->set_base(base3);
	m_bank4->set_base(base4);

	/* check if BASIC changed */
	if( new_mmu & 0x02 )
	{
		logerror("%s MMU BASIC RAM\n", machine().system().name);
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xa000, 0xbfff, "bank1");
		base1 = base + 0x0a000;  /* 8K RAM */
	}
	else
	{
		logerror("%s MMU BASIC ROM\n", machine().system().name);
		m_maincpu->space(AS_PROGRAM).install_read_bank(0xa000, 0xbfff, "bank1");
		m_maincpu->space(AS_PROGRAM).nop_write(0xa000, 0xbfff);
		base1 = base + 0x10000;  /* 8K BASIC */
	}

	m_bank1->set_base(base1);

	/* check if self-test ROM changed */
	if( new_mmu & 0x80 )
	{
		logerror("%s MMU SELFTEST RAM\n", machine().system().name);
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0x5000, 0x57ff, "bank2");
		base2 = base + 0x05000;  /* 0x0800 bytes */
	}
	else
	{
		logerror("%s MMU SELFTEST ROM\n", machine().system().name);
		m_maincpu->space(AS_PROGRAM).install_read_bank(0x5000, 0x57ff, "bank2");
		m_maincpu->space(AS_PROGRAM).nop_write(0x5000, 0x57ff);
		base2 = base + 0x15000;  /* 0x0800 bytes */
	}
	m_bank2->set_base(base2);
}

/* BASIC was available in a separate cart, so we don't test it */
void a400_state::a1200xl_mmu(UINT8 new_mmu)
{
	UINT8 *base = m_region_maincpu->base();
	UINT8 *base2, *base3, *base4;

	/* check if memory C000-FFFF changed */
	if( new_mmu & 0x01 )
	{
		logerror("%s MMU BIOS ROM\n", machine().system().name);
		base3 = base + 0x14000;  /* 8K lo BIOS */
		base4 = base + 0x15800;  /* 4K FP ROM + 8K hi BIOS */
		m_maincpu->space(AS_PROGRAM).install_read_bank(0xc000, 0xcfff, "bank3");
		m_maincpu->space(AS_PROGRAM).unmap_write(0xc000, 0xcfff);
		m_maincpu->space(AS_PROGRAM).install_read_bank(0xd800, 0xffff, "bank4");
		m_maincpu->space(AS_PROGRAM).unmap_write(0xd800, 0xffff);
	}
	else
	{
		logerror("%s MMU BIOS RAM\n", machine().system().name);
		base3 = base + 0x0c000;  /* 8K RAM */
		base4 = base + 0x0d800;  /* 4K RAM + 8K RAM */
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xc000, 0xcfff, "bank3");
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xd800, 0xffff, "bank4");
	}
	m_bank3->set_base(base3);
	m_bank4->set_base(base4);

	/* check if self-test ROM changed */
	if( new_mmu & 0x80 )
	{
		logerror("%s MMU SELFTEST RAM\n", machine().system().name);
		base2 = base + 0x05000;  /* 0x0800 bytes */
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0x5000, 0x57ff, "bank2");
	}
	else
	{
		logerror("%s MMU SELFTEST ROM\n", machine().system().name);
		base2 = base + 0x15000;  /* 0x0800 bytes */
		m_maincpu->space(AS_PROGRAM).install_read_bank(0x5000, 0x57ff, "bank2");
		m_maincpu->space(AS_PROGRAM).unmap_write(0x5000, 0x57ff);
	}
	m_bank2->set_base(base2);
}

void a400_state::xegs_mmu(UINT8 new_mmu)
{
	UINT8 *base = m_region_maincpu->base();
	UINT8 *base2, *base3, *base4;

	/* check if memory C000-FFFF changed */
	if( new_mmu & 0x01 )
	{
		logerror("%s MMU BIOS ROM\n", machine().system().name);
		base3 = base + 0x14000;  /* 8K lo BIOS */
		base4 = base + 0x15800;  /* 4K FP ROM + 8K hi BIOS */
		m_maincpu->space(AS_PROGRAM).install_read_bank(0xc000, 0xcfff, "bank3");
		m_maincpu->space(AS_PROGRAM).unmap_write(0xc000, 0xcfff);
		m_maincpu->space(AS_PROGRAM).install_read_bank(0xd800, 0xffff, "bank4");
		m_maincpu->space(AS_PROGRAM).unmap_write(0xd800, 0xffff);
	}
	else
	{
		logerror("%s MMU BIOS RAM\n", machine().system().name);
		base3 = base + 0x0c000;  /* 8K RAM */
		base4 = base + 0x0d800;  /* 4K RAM + 8K RAM */
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xc000, 0xcfff, "bank3");
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0xd800, 0xffff, "bank4");
	}
	m_bank3->set_base(base3);
	m_bank4->set_base(base4);


	/* check if self-test ROM changed */
	if( new_mmu & 0x80 )
	{
		logerror("%s MMU SELFTEST RAM\n", machine().system().name);
		m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0x5000, 0x57ff, "bank2");
		base2 = base + 0x05000;  /* 0x0800 bytes */
	}
	else
	{
		logerror("%s MMU SELFTEST ROM\n", machine().system().name);
		m_maincpu->space(AS_PROGRAM).install_read_bank(0x5000, 0x57ff, "bank2");
		m_maincpu->space(AS_PROGRAM).nop_write(0x5000, 0x57ff);
		base2 = base + 0x15000;  /* 0x0800 bytes */
	}
	m_bank2->set_base(base2);
}


// Currently, the drivers have fixed 40k RAM, however the function here is ready for different sizes too
void a400_state::a800_setbank(int cart_mounted)
{
	offs_t ram_top;
	// take care of 0x0000-0x7fff: RAM or NOP
	ram_top = MIN(m_ram->size(), 0x8000) - 1;
	m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0x0000, ram_top, "0000");
	if ( m_0000 == NULL )
	{
		m_0000.findit();
	}
	m_0000->set_base(m_ram->pointer());

	// take care of 0x8000-0x9fff: A800 -> either right slot or RAM or NOP, others -> RAM or NOP
	// is there anything in the right slot?
	if (cart_mounted & RIGHT_CARTSLOT_MOUNTED)
	{
		m_maincpu->space(AS_PROGRAM).install_read_bank(0x8000, 0x9fff, "8000");
		if ( m_8000 == NULL )
		{
			m_8000.findit();
		}
		m_8000->set_base(m_region_rslot->base());
		m_maincpu->space(AS_PROGRAM).unmap_write(0x8000, 0x9fff);
	}
	else if (m_a800_cart_type != BBSB)
	{
		ram_top = MIN(m_ram->size(), 0xa000) - 1;
		if (ram_top > 0x8000)
		{
			m_maincpu->space(AS_PROGRAM).install_readwrite_bank(0x8000, ram_top, "8000");
			if ( m_8000 == NULL )
			{
				m_8000.findit();
			}
			m_8000->set_base(m_ram->pointer() + 0x8000);
		}
	}

	// take care of 0xa000-0xbfff: is there anything in the left slot?
	if (cart_mounted & LEFT_CARTSLOT_MOUNTED)
	{
		// FIXME: this is an hack to keep XL working until we clean up its memory map as well!
		if (m_atari == ATARI_800XL)
		{
			if (m_a800_cart_type == A800_16K)
			{
				m_maincpu->space(AS_PROGRAM).install_read_bank(0x8000, 0x9fff, "8000");
				if ( m_8000 == NULL )
				{
					m_8000.findit();
				}
				m_8000->set_base(m_region_lslot->base());
				m_maincpu->space(AS_PROGRAM).unmap_write(0x8000, 0x9fff);

				memcpy(m_region_maincpu->base() + 0x10000, m_region_lslot->base() + 0x2000, 0x2000);
			}
			else if (m_a800_cart_type == A800_8K)
				memcpy(m_region_maincpu->base() + 0x10000, m_region_lslot->base(), 0x2000);
			else
				fatalerror("This type of cart is not supported yet in this driver. Please use a400 or a800.\n");
		}
		else if (m_a800_cart_type == A800_16K)
		{
			m_8000->set_base(m_region_lslot->base());
			m_a000->set_base(m_region_lslot->base() + 0x2000);
			m_maincpu->space(AS_PROGRAM).unmap_write(0x8000, 0xbfff);
		}
		else if (m_a800_cart_type == BBSB)
		{
			// this requires separate banking in 0x8000 & 0x9000!
			m_maincpu->space(AS_PROGRAM).install_read_bank(0x8000, 0x8fff, "8000");
			if ( m_8000 == NULL )
			{
				m_8000.findit();
			}
			m_maincpu->space(AS_PROGRAM).install_read_bank(0x9000, 0x9fff, "9000");
			if ( m_9000 == NULL )
			{
				m_9000.findit();
			}
			m_8000->set_base(m_region_lslot->base() + 0x0000);
			m_9000->set_base(m_region_lslot->base() + 0x4000);
			m_a000->set_base(m_region_lslot->base() + 0x8000);
			m_maincpu->space(AS_PROGRAM).unmap_write(0xa000, 0xbfff);
		}
		else if (m_a800_cart_type == OSS_034M)
		{
			// this requires separate banking in 0xa000 & 0xb000!
			m_maincpu->space(AS_PROGRAM).install_read_bank(0xa000, 0xafff, "a000");
			if ( m_a000 == NULL )
			{
				m_a000.findit();
			}
			m_maincpu->space(AS_PROGRAM).install_read_bank(0xb000, 0xbfff, "b000");
			if ( m_b000 == NULL )
			{
				m_b000.findit();
			}
			m_b000->set_base(m_region_lslot->base() + 0x3000);
			m_maincpu->space(AS_PROGRAM).unmap_write(0xa000, 0xbfff);
		}
		else if (m_a800_cart_type == OSS_M091)
		{
			// this requires separate banking in 0xa000 & 0xb000!
			m_maincpu->space(AS_PROGRAM).install_read_bank(0xa000, 0xafff, "a000");
			if ( m_a000 == NULL )
			{
				m_a000.findit();
			}
			m_maincpu->space(AS_PROGRAM).install_read_bank(0xb000, 0xbfff, "b000");
			if ( m_b000 == NULL )
			{
				m_b000.findit();
			}
			m_b000->set_base(m_region_lslot->base());
			m_maincpu->space(AS_PROGRAM).unmap_write(0xa000, 0xbfff);
		}
		else if (m_a800_cart_type == XEGS_32K)
		{
			m_8000->set_base(m_region_lslot->base());
			m_a000->set_base(m_region_lslot->base() + 0x6000);
			m_maincpu->space(AS_PROGRAM).unmap_write(0x8000, 0xbfff);
		}
		else
		{
			m_a000->set_base(m_region_lslot->base());
			m_maincpu->space(AS_PROGRAM).unmap_write(0xa000, 0xbfff);
		}
	}
}


/* MESS specific parts that have to be started */
void a400_state::ms_atari_machine_start(int type, int has_cart)
{
	/* set atari type (temporarily not used) */
	m_atari = type;
	a800_setbank(m_a800_cart_loaded);
}

void a400_state::ms_atari800xl_machine_start(int type, int has_cart)
{
	/* set atari type (temporarily not used) */
	m_atari = type;
	a800_setbank(m_a800_cart_loaded);
}


struct a800_pcb
{
	const char              *pcb_name;
	int                     pcb_id;
};

// Here, we take the feature attribute from .xml (i.e. the PCB name) and we assign a unique ID to it
// WARNING: most of these are still unsupported by the driver
static const a800_pcb pcb_list[] =
{
	{"standard 4k", A800_8K},
	{"standard 8k", A800_8K},
	{"standard 12k", A800_16K},
	{"standard 16k", A800_16K},
	{"right slot 4k", A800_RIGHT_4K},
	{"right slot 8k", A800_RIGHT_8K},

	{"oss 034m", OSS_034M},
	{"oss m091", OSS_M091},
	{"phoenix 8k", PHOENIX_8K},
	{"xegs 32k", XEGS_32K},
	{"bbsb", BBSB},
	{"diamond 64k", DIAMOND_64K},
	{"williams 64k", WILLIAMS_64K},
	{"express 64", EXPRESS_64},
	{"spartados x", SPARTADOS_X},
	{"N/A", A800_UNKNOWN}
};

int a400_state::a800_get_pcb_id(const char *pcb)
{
	int i;

	for (i = 0; i < ARRAY_LENGTH(pcb_list); i++)
	{
		if (!mame_stricmp(pcb_list[i].pcb_name, pcb))
			return pcb_list[i].pcb_id;
	}

	return A800_UNKNOWN;
}


WRITE8_MEMBER( a400_state::x32_bank_w )
{
	//  printf("written %x\n", data);
	int bank = data & 0x03;
	m_8000->set_base(m_region_lslot->base() + bank * 0x2000);
}


WRITE8_MEMBER( a400_state::w64_bank_w )
{
//  printf("write to %x\n", offset);

	if (offset < 8)
		m_a000->set_base(m_region_lslot->base() + offset * 0x2000);
	else
		m_a000->set_base(m_region_maincpu->base());
	// FIXME: writes to 0x8-0xf should disable the cart
}


// this covers Express 64, Diamond 64 and SpartaDOS (same bankswitch, but at different addresses)
WRITE8_MEMBER( a400_state::ex64_bank_w )
{
//  printf("write to %x\n", offset);

	if (offset < 8)
		m_a000->set_base(m_region_lslot->base() + (7 - offset) * 0x2000);
	else
		m_a000->set_base(m_region_maincpu->base());
	// FIXME: writes to 0x8-0xf should disable the cart
}


WRITE8_MEMBER( a400_state::bbsb_bankl_w )
{
//  printf("write to %x\n", 0x8000 + offset);
	if (offset >= 0xff6 && offset <= 0xff9)
		m_8000->set_base(m_region_lslot->base() + 0x0000 + (offset - 0xff6) * 0x1000);
}


WRITE8_MEMBER( a400_state::bbsb_bankh_w )
{
//  printf("write to %x\n", 0x9000 + offset);
	if (offset >= 0xff6 && offset <= 0xff9)
		m_9000->set_base(m_region_lslot->base() + 0x4000 + (offset - 0xff6) * 0x1000);
}


WRITE8_MEMBER( a400_state::oss_034m_w )
{
	switch (offset & 0x0f)
	{
		case 0:
		case 1:
			m_a000->set_base(m_region_lslot->base());
			m_b000->set_base(m_region_lslot->base() + 0x3000);
			break;
		case 2:
		case 6:
			// docs says this should put 0xff in the 0xa000 bank -> let's point to the end of the cart
			m_a000->set_base(m_region_lslot->base() + 0x4000);
			m_b000->set_base(m_region_lslot->base() + 0x3000);
			break;
		case 3:
		case 7:
			m_a000->set_base(m_region_lslot->base() + 0x1000);
			m_b000->set_base(m_region_lslot->base() + 0x3000);
			break;
		case 4:
		case 5:
			m_a000->set_base(m_region_lslot->base() + 0x2000);
			m_b000->set_base(m_region_lslot->base() + 0x3000);
			break;
		default:
			m_a000->set_base(m_region_maincpu->base() + 0xa000);
			m_b000->set_base(m_region_maincpu->base() + 0xb000);
			break;
	}
}


WRITE8_MEMBER( a400_state::oss_m091_w )
{
	switch (offset & 0x09)
	{
		case 0:
			m_a000->set_base(m_region_lslot->base() + 0x1000);
			m_b000->set_base(m_region_lslot->base());
			break;
		case 1:
			m_a000->set_base(m_region_lslot->base() + 0x3000);
			m_b000->set_base(m_region_lslot->base());
			break;
		case 8:
			m_a000->set_base(m_region_maincpu->base() + 0xa000);
			m_b000->set_base(m_region_maincpu->base() + 0xb000);
			break;
		case 9:
			m_a000->set_base(m_region_lslot->base() + 0x2000);
			m_b000->set_base(m_region_lslot->base());
			break;
	}
}


WRITE8_MEMBER( a400_state::xegs_bankswitch )
{
	UINT8 *cart = m_region_user1->base();
	data &= m_xegs_banks - 1;
	m_bank0->set_base(cart + data * 0x2000);
}

MACHINE_START_MEMBER( a400_state, xegs )
{
	address_space &space = m_maincpu->space(AS_PROGRAM);
	UINT8 *cart = m_region_user1->base();
	UINT8 *cpu  = m_region_maincpu->base();

	atari_machine_start(machine());
	space.install_write_handler(0xd500, 0xd5ff, write8_delegate(FUNC(a400_state::xegs_bankswitch),this));

	if (m_xegs_cart)
	{
		m_bank0->set_base(cart);
		m_bank1->set_base(cart + (m_xegs_banks - 1) * 0x2000);
	}
	else
	{
		// point to built-in Missile Command (this does not work well, though... FIXME!!)
		m_bank0->set_base(cpu + 0x10000);
		m_bank1->set_base(cpu + 0x10000);
	}
}


// currently this does nothing, but it will eventually install the memory handlers required by the mappers
void a400_state::a800_setup_mappers(int type)
{
	switch (type)
	{
		case A800_4K:
		case A800_RIGHT_4K:
		case A800_12K:
		case A800_8K:
		case A800_16K:
		case A800_RIGHT_8K:
		case PHOENIX_8K:    // as normal 8k cart, but it can be disabled by writing to 0xd500-0xdfff
			break;
		case XEGS_32K:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd5ff, write8_delegate(FUNC(a400_state::x32_bank_w),this));
			break;
		case OSS_034M:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd5ff, write8_delegate(FUNC(a400_state::oss_034m_w),this));
			break;
		case OSS_M091:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd5ff, write8_delegate(FUNC(a400_state::oss_m091_w),this));
			break;
		case BBSB:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0x8000, 0x8fff, write8_delegate(FUNC(a400_state::bbsb_bankl_w),this));
			m_maincpu->space(AS_PROGRAM).install_write_handler(0x9000, 0x9fff, write8_delegate(FUNC(a400_state::bbsb_bankh_w),this));
			break;
		case WILLIAMS_64K:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd500, 0xd50f, write8_delegate(FUNC(a400_state::w64_bank_w),this));
			break;
		case DIAMOND_64K:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd5d0, 0xd5df, write8_delegate(FUNC(a400_state::ex64_bank_w),this));
			break;
		case EXPRESS_64:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd570, 0xd57f, write8_delegate(FUNC(a400_state::ex64_bank_w),this));
			break;
		case SPARTADOS_X:
			m_maincpu->space(AS_PROGRAM).install_write_handler(0xd5e0, 0xd5ef, write8_delegate(FUNC(a400_state::ex64_bank_w),this));
			break;
		default:
			break;
	}
}


int a400_state::a800_get_type(device_image_interface &image)
{
	UINT8 header[16];
	image.fread(header, 0x10);
	int hdr_type, cart_type = A800_UNKNOWN;

	// add check of CART format
	if (strncmp((const char *)header, "CART", 4))
		fatalerror("Invalid header detected!\n");

	hdr_type = (header[4] << 24) + (header[5] << 16) +  (header[6] << 8) + (header[7] << 0);
	switch (hdr_type)
	{
		case 1:
			cart_type = A800_8K;
			break;
		case 2:
			cart_type = A800_16K;
			break;
		case 3:
			cart_type = OSS_034M;
			break;
		case 8:
			cart_type = WILLIAMS_64K;
			break;
		case 9:
			cart_type = DIAMOND_64K;
			break;
		case 10:
			cart_type = EXPRESS_64;
			break;
		case 11:
			cart_type = SPARTADOS_X;
			break;
		case 12:
			cart_type = XEGS_32K;
			break;
		case 15:
			cart_type = OSS_M091;
			break;
		case 18:
			cart_type = BBSB;
			break;
		case 21:
			cart_type = A800_RIGHT_8K;
			break;
		case 39:
			cart_type = PHOENIX_8K;
			break;
		case 4:
		case 6:
		case 7:
		case 16:
		case 19:
		case 20:
			fatalerror("Cart type \"%d\" means this is an Atari 5200 cart.\n", hdr_type);
			break;
		default:
			mame_printf_info("Cart type \"%d\" is currently unsupported.\n", hdr_type);
			break;
	}
	return cart_type;
}


int a400_state::a800_check_cart_type(device_image_interface &image)
{
	const char  *pcb_name;
	int type = A800_UNKNOWN;

	if (image.software_entry() == NULL)
	{
		UINT32 size = image.length();

		// check if there is an header, if so extract cart_type from it, otherwise
		// try to guess the cart_type from the file size (notice that after the
		// a800_get_type call, we point at the start of the data)
		if ((size % 0x1000) == 0x10)
			type = a800_get_type(image);
		else if (size == 0x4000)
			type = A800_16K;
		else if (size == 0x2000)
		{
			if (strcmp(image.device().tag(),":cart2") == 0)
				type = A800_RIGHT_8K;
			else
				type = A800_8K;
		}
	}
	else
	{
		if ((pcb_name = image.get_feature("cart_type")) != NULL)
			type = a800_get_pcb_id(pcb_name);

		switch (type)
		{
			case A800_UNKNOWN:
			case A800_4K:
			case A800_RIGHT_4K:
			case A800_12K:
			case A800_8K:
			case A800_16K:
			case A800_RIGHT_8K:
				break;
			default:
				mame_printf_info("Cart type \"%s\" currently unsupported.\n", pcb_name);
				break;
		}
	}

	if ((strcmp(image.device().tag(),":cart2") == 0) && (type != A800_RIGHT_8K))
		fatalerror("You cannot load this image '%s' in the right slot\n", image.filename());

	return type;
}


DEVICE_IMAGE_LOAD_MEMBER( a400_state, a800_cart )
{
	UINT32 size, start = 0;

	m_a800_cart_loaded = m_a800_cart_loaded & ~LEFT_CARTSLOT_MOUNTED;
	m_a800_cart_type = a800_check_cart_type(image);

	a800_setup_mappers(m_a800_cart_type);

	if (image.software_entry() == NULL)
	{
		size = image.length();
		// if there is an header, skip it
		if ((size % 0x1000) == 0x10)
		{
			size -= 0x10;
			start = 0x10;
		}
		image.fread(m_region_lslot->base(), size - start);
	}
	else
	{
		size = image.get_software_region_length("rom");
		memcpy(m_region_lslot->base(), image.get_software_region("rom"), size);
	}

	m_a800_cart_loaded |= (size > 0x0000) ? 1 : 0;

	logerror("%s loaded left cartridge '%s' size %dK\n", machine().system().name, image.filename(), size/1024);
	return IMAGE_INIT_PASS;
}


DEVICE_IMAGE_LOAD_MEMBER( a400_state, a800_cart_right )
{
	UINT32 size, start = 0;

	m_a800_cart_loaded = m_a800_cart_loaded & ~RIGHT_CARTSLOT_MOUNTED;
	m_a800_cart_type = a800_check_cart_type(image);

	a800_setup_mappers(m_a800_cart_type);

	if (image.software_entry() == NULL)
	{
		size = image.length();
		// if there is an header, skip it
		if ((size % 0x1000) == 0x10)
		{
			size -= 0x10;
			start = 0x10;
		}
		image.fread(m_region_rslot->base(), size - start);
	}
	else
	{
		size = image.get_software_region_length("rom");
		memcpy(m_region_rslot->base(), image.get_software_region("rom"), size);
	}

	m_a800_cart_loaded |= (size > 0x0000) ? 2 : 0;

	logerror("%s loaded right cartridge '%s' size 8K\n", machine().system().name, image.filename());
	return IMAGE_INIT_PASS;
}


DEVICE_IMAGE_UNLOAD_MEMBER( a400_state, a800_cart )
{
	m_a800_cart_loaded = m_a800_cart_loaded & ~LEFT_CARTSLOT_MOUNTED;
	m_a800_cart_type = A800_UNKNOWN;
	a800_setbank(m_a800_cart_loaded);
}


DEVICE_IMAGE_UNLOAD_MEMBER( a400_state, a800_cart_right )
{
	m_a800_cart_loaded = m_a800_cart_loaded & ~RIGHT_CARTSLOT_MOUNTED;
	m_a800_cart_type = A800_UNKNOWN;
	a800_setbank(m_a800_cart_loaded);
}


DEVICE_IMAGE_LOAD_MEMBER( a400_state, a5200_cart )
{
	UINT8 *mem = m_region_maincpu->base();
	UINT32 size;
	bool A13_mirr = FALSE;

	if (image.software_entry() == NULL)
	{
		/* load an optional (dual) cartidge */
		size = image.fread(&mem[0x4000], 0x8000);
		const char *info = hashfile_extrainfo(image);
		if (info && !strcmp(info, "A13MIRRORING"))
			A13_mirr = TRUE;
	}
	else
	{
		size = image.get_software_region_length("rom");
		memcpy(mem + 0x4000, image.get_software_region("rom"), size);
		const char *pcb_name = image.get_feature("cart_type");
		if (pcb_name && !strcmp(pcb_name, "A13MIRRORING"))
			A13_mirr = TRUE;
	}

	if (size<0x8000) memmove(mem+0x4000+0x8000-size, mem+0x4000, size);
	// mirroring of smaller cartridges
	if (size <= 0x1000) memcpy(mem+0xa000, mem+0xb000, 0x1000);
	if (size <= 0x2000) memcpy(mem+0x8000, mem+0xa000, 0x2000);
	if (size <= 0x4000)
	{
		memcpy(&mem[0x4000], &mem[0x8000], 0x4000);
		if (A13_mirr)
		{
			memcpy(&mem[0x8000], &mem[0xa000], 0x2000);
			memcpy(&mem[0x6000], &mem[0x4000], 0x2000);
		}
	}
	logerror("A5200 loaded cartridge '%s' size %dK\n", image.filename() , size/1024);
	return IMAGE_INIT_PASS;
}


DEVICE_IMAGE_UNLOAD_MEMBER( a400_state, a5200_cart )
{
	UINT8 *mem = m_region_maincpu->base();
	/* zap the cartridge memory (again) */
	memset(&mem[0x4000], 0x00, 0x8000);
}


DEVICE_IMAGE_LOAD_MEMBER( a400_state, xegs_cart )
{
	UINT32 size;
	UINT8 *ptr = m_region_user1->base();

	if (image.software_entry() == NULL)
	{
		// skip the header
		image.fseek(0x10, SEEK_SET);
		size = image.length() - 0x10;
		if (image.fread(ptr, size) != size)
			return IMAGE_INIT_FAIL;
	}
	else
	{
		size = image.get_software_region_length("rom");
		memcpy(ptr, image.get_software_region("rom"), size);
	}

	m_xegs_banks = size / 0x2000;
	m_xegs_cart = 1;

	return IMAGE_INIT_PASS;
}


DEVICE_IMAGE_UNLOAD_MEMBER( a400_state, xegs_cart )
{
	m_xegs_cart = 0;
	m_xegs_banks = 0;
}


MACHINE_START_MEMBER( a400_state, a400 )
{
	atari_machine_start(machine());
	ms_atari_machine_start(ATARI_400, TRUE);
}


MACHINE_START_MEMBER( a400_state, a800 )
{
	atari_machine_start(machine());
	ms_atari_machine_start(ATARI_800, TRUE);
}


MACHINE_START_MEMBER( a400_state, a800xl )
{
	atari_machine_start(machine());
	ms_atari800xl_machine_start(ATARI_800XL, TRUE);
}


MACHINE_START_MEMBER( a400_state, a5200 )
{
	atari_machine_start(machine());
	ms_atari_machine_start(ATARI_800XL, TRUE);
}



/**************************************************************
 *
 * PIA interface
 *
 **************************************************************/

WRITE8_MEMBER(a400_state::a1200xl_pia_pb_w){ a1200xl_mmu(data); }
WRITE8_MEMBER(a400_state::a800xl_pia_pb_w)
{
	if (m_pia->port_b_z_mask() != 0xff)
		a800xl_mmu(data);
}

WRITE8_MEMBER(a400_state::xegs_pia_pb_w)
{
	if (m_pia->port_b_z_mask() != 0xff)
		xegs_mmu(data);
}

static const pokey_interface atari_pokey_interface =
{
	{
		DEVCB_INPUT_PORT("analog_0"),
		DEVCB_INPUT_PORT("analog_1"),
		DEVCB_INPUT_PORT("analog_2"),
		DEVCB_INPUT_PORT("analog_3"),
		DEVCB_INPUT_PORT("analog_4"),
		DEVCB_INPUT_PORT("analog_5"),
		DEVCB_INPUT_PORT("analog_6"),
		DEVCB_INPUT_PORT("analog_7")
	},
	DEVCB_NULL,
	DEVCB_DEVICE_HANDLER("fdc", atari_serin_r),
	DEVCB_DEVICE_HANDLER("fdc", atari_serout_w)
};


static const pia6821_interface atari_pia_interface =
{
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pa_r),        /* port A in */
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pb_r),    /* port B in */
	DEVCB_NULL,     /* line CA1 in */
	DEVCB_NULL,     /* line CB1 in */
	DEVCB_NULL,     /* line CA2 in */
	DEVCB_NULL,     /* line CB2 in */
	DEVCB_NULL,     /* port A out */
	DEVCB_NULL,     /* port B out */
	DEVCB_NULL,     /* line CA2 out */
	DEVCB_DEVICE_LINE("fdc",atarifdc_pia_cb2_w),        /* port CB2 out */
	DEVCB_NULL,     /* IRQA */
	DEVCB_NULL      /* IRQB */
};

static const pia6821_interface a600xl_pia_interface =
{
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pa_r),        /* port A in */
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pb_r),    /* port B in */
	DEVCB_NULL,     /* line CA1 in */
	DEVCB_NULL,     /* line CB1 in */
	DEVCB_NULL,     /* line CA2 in */
	DEVCB_NULL,     /* line CB2 in */
	DEVCB_NULL,     /* port A out */
	DEVCB_DEVICE_HANDLER("pia", a600xl_pia_pb_w),       /* port B out */
	DEVCB_NULL,     /* line CA2 out */
	DEVCB_DEVICE_LINE("fdc",atarifdc_pia_cb2_w),        /* port CB2 out */
	DEVCB_NULL,     /* IRQA */
	DEVCB_NULL      /* IRQB */
};

static const pia6821_interface a1200xl_pia_interface =
{
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pa_r),        /* port A in */
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pb_r),    /* port B in */
	DEVCB_NULL,     /* line CA1 in */
	DEVCB_NULL,     /* line CB1 in */
	DEVCB_NULL,     /* line CA2 in */
	DEVCB_NULL,     /* line CB2 in */
	DEVCB_NULL,     /* port A out */
	DEVCB_DRIVER_MEMBER(a400_state, a1200xl_pia_pb_w),      /* port B out */
	DEVCB_NULL,     /* line CA2 out */
	DEVCB_DEVICE_LINE("fdc",atarifdc_pia_cb2_w),        /* port CB2 out */
	DEVCB_NULL,     /* IRQA */
	DEVCB_NULL      /* IRQB */
};

static const pia6821_interface a800xl_pia_interface =
{
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pa_r),        /* port A in */
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pb_r),    /* port B in */
	DEVCB_NULL,     /* line CA1 in */
	DEVCB_NULL,     /* line CB1 in */
	DEVCB_NULL,     /* line CA2 in */
	DEVCB_NULL,     /* line CB2 in */
	DEVCB_NULL,     /* port A out */
	DEVCB_DRIVER_MEMBER(a400_state,a800xl_pia_pb_w),        /* port B out */
	DEVCB_NULL,     /* line CA2 out */
	DEVCB_DEVICE_LINE("fdc",atarifdc_pia_cb2_w),        /* port CB2 out */
	DEVCB_NULL,     /* IRQA */
	DEVCB_NULL      /* IRQB */
};

static const pia6821_interface xegs_pia_interface =
{
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pa_r),        /* port A in */
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pb_r),    /* port B in */
	DEVCB_NULL,     /* line CA1 in */
	DEVCB_NULL,     /* line CB1 in */
	DEVCB_NULL,     /* line CA2 in */
	DEVCB_NULL,     /* line CB2 in */
	DEVCB_NULL,     /* port A out */
	DEVCB_DRIVER_MEMBER(a400_state,xegs_pia_pb_w),      /* port B out */
	DEVCB_NULL,     /* line CA2 out */
	DEVCB_DEVICE_LINE("fdc",atarifdc_pia_cb2_w),        /* port CB2 out */
	DEVCB_NULL,     /* IRQA */
	DEVCB_NULL      /* IRQB */
};

// FIXME: should there be anything connected where other system have the fdc?
static const pokey_interface a5200_pokey_interface =
{
	{
		DEVCB_INPUT_PORT("analog_0"),
		DEVCB_INPUT_PORT("analog_1"),
		DEVCB_INPUT_PORT("analog_2"),
		DEVCB_INPUT_PORT("analog_3"),
		DEVCB_INPUT_PORT("analog_4"),
		DEVCB_INPUT_PORT("analog_5"),
		DEVCB_INPUT_PORT("analog_6"),
		DEVCB_INPUT_PORT("analog_7")
	},
	DEVCB_NULL,
	DEVCB_NULL, // FIXME: is there anything connected here?
	DEVCB_NULL  // FIXME: is there anything connected here?
};

static const pia6821_interface a5200_pia_interface =
{
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pa_r),        /* port A in */
	DEVCB_DEVICE_HANDLER("pia", atari_pia_pb_r),    /* port B in */
	DEVCB_NULL,     /* line CA1 in */
	DEVCB_NULL,     /* line CB1 in */
	DEVCB_NULL,     /* line CA2 in */
	DEVCB_NULL,     /* line CB2 in */
	DEVCB_NULL,     /* port A out */
	DEVCB_NULL,     /* port B out */
	DEVCB_NULL,     /* line CA2 out */
	DEVCB_NULL,     /* port CB2 out */  // FIXME: is there anything connected here
	DEVCB_NULL,     /* IRQA */
	DEVCB_NULL      /* IRQB */
};


/**************************************************************
 *
 * Machine drivers
 *
 **************************************************************/

static MACHINE_CONFIG_FRAGMENT( a400_cartslot )
	MCFG_CARTSLOT_ADD("cart1")
	MCFG_CARTSLOT_EXTENSION_LIST("rom,bin")
	MCFG_CARTSLOT_NOT_MANDATORY
	MCFG_CARTSLOT_LOAD(a400_state,a800_cart)
	MCFG_CARTSLOT_UNLOAD(a400_state,a800_cart)
	MCFG_CARTSLOT_INTERFACE("a800_cart")
MACHINE_CONFIG_END

static MACHINE_CONFIG_FRAGMENT( a800_cartslot )
	MCFG_CARTSLOT_ADD("cart1")
	MCFG_CARTSLOT_EXTENSION_LIST("rom,bin")
	MCFG_CARTSLOT_NOT_MANDATORY
	MCFG_CARTSLOT_LOAD(a400_state,a800_cart)
	MCFG_CARTSLOT_UNLOAD(a400_state,a800_cart)
	MCFG_CARTSLOT_INTERFACE("a800_cart")

	MCFG_CARTSLOT_ADD("cart2")
	MCFG_CARTSLOT_EXTENSION_LIST("rom,bin")
	MCFG_CARTSLOT_NOT_MANDATORY
	MCFG_CARTSLOT_LOAD(a400_state,a800_cart_right)
	MCFG_CARTSLOT_UNLOAD(a400_state,a800_cart_right)
	MCFG_CARTSLOT_INTERFACE("a800_cart")
MACHINE_CONFIG_END

static MACHINE_CONFIG_START( atari_common_nodac, a400_state )
	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", M6502, FREQ_17_EXACT)

	/* video hardware */
	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC(1))
	MCFG_SCREEN_VISIBLE_AREA(MIN_X, MAX_X, MIN_Y, MAX_Y)
	MCFG_PALETTE_LENGTH(sizeof(atari_palette) / 3)
	MCFG_SCREEN_UPDATE_STATIC(atari)

	MCFG_VIDEO_START(atari)

	MCFG_PIA6821_ADD( "pia", atari_pia_interface )

	/* sound hardware */
	MCFG_SPEAKER_STANDARD_MONO("mono")
	MCFG_POKEY_ADD("pokey", FREQ_17_EXACT)
	MCFG_POKEY_CONFIG(atari_pokey_interface)
	MCFG_POKEY_KEYBOARD_HANDLER(atari_a800_keyboard)
	MCFG_POKEY_INTERRUPT_HANDLER(atari_interrupt_cb)

	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 1.00)

	/* internal ram */
	MCFG_RAM_ADD(RAM_TAG)
	MCFG_RAM_DEFAULT_SIZE("40K")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( atari_common, atari_common_nodac )
	MCFG_SOUND_ADD("dac", DAC, 0)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.50)

	MCFG_ATARI_FDC_ADD("fdc")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a400, atari_common )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(a400_mem)
	MCFG_TIMER_ADD_SCANLINE("scantimer", a400_interrupt, "screen", 0, 1)

	MCFG_MACHINE_START_OVERRIDE( a400_state, a400 )

	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_60HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_60HZ)

	MCFG_FRAGMENT_ADD(a400_cartslot)

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a800")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a400pal, atari_common )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(a400_mem)
	MCFG_TIMER_ADD_SCANLINE("scantimer", a400_interrupt, "screen", 0, 1)

	MCFG_MACHINE_START_OVERRIDE( a400_state, a400 )

	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_50HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_50HZ)

	MCFG_FRAGMENT_ADD(a400_cartslot)

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a800")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a800, atari_common )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(a800_mem)
	MCFG_TIMER_ADD_SCANLINE("scantimer", a800_interrupt, "screen", 0, 1)

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800 )

	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_60HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_60HZ)

	MCFG_FRAGMENT_ADD(a800_cartslot)

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a800")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a800pal, atari_common )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(a800_mem)
	MCFG_TIMER_ADD_SCANLINE("scantimer", a800_interrupt, "screen", 0, 1)

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800 )

	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_50HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_50HZ)

	MCFG_FRAGMENT_ADD(a800_cartslot)

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a800")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a600xl, atari_common )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(a600xl_mem)    // FIXME?
	MCFG_TIMER_ADD_SCANLINE("scantimer", a800xl_interrupt, "screen", 0, 1)

	MCFG_PIA6821_MODIFY( "pia", a600xl_pia_interface )

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800xl )    // FIXME?

	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_60HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_60HZ)

	MCFG_FRAGMENT_ADD(a400_cartslot)

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a800")

	/* internal ram */
	MCFG_RAM_MODIFY(RAM_TAG)
	MCFG_RAM_DEFAULT_SIZE("16K")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a800xl, atari_common )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(a800xl_mem)
	MCFG_TIMER_ADD_SCANLINE("scantimer", a800xl_interrupt, "screen", 0, 1)

	MCFG_PIA6821_MODIFY( "pia", a800xl_pia_interface )

	MCFG_MACHINE_START_OVERRIDE( a400_state, a800xl )

	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_60HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_60HZ)

	MCFG_FRAGMENT_ADD(a400_cartslot)

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a800")
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( a800xlpal, a800xl )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_CLOCK( 1773000 )

	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_50HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_50HZ)

	MCFG_SOUND_MODIFY("pokey")
	MCFG_SOUND_CLOCK(1773000)
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( a1200xl, a800xl )

	MCFG_PIA6821_MODIFY( "pia", a1200xl_pia_interface )
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( xegs, a800xl )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(xegs_mem)

	MCFG_MACHINE_START_OVERRIDE( a400_state, xegs )

	MCFG_PIA6821_MODIFY( "pia", xegs_pia_interface )

	MCFG_DEVICE_REMOVE("cart1")
	MCFG_DEVICE_REMOVE("cart_list")

	MCFG_CARTSLOT_ADD("cart1")
	MCFG_CARTSLOT_EXTENSION_LIST("rom,bin")
	MCFG_CARTSLOT_NOT_MANDATORY
	MCFG_CARTSLOT_LOAD(a400_state,xegs_cart)
	MCFG_CARTSLOT_UNLOAD(a400_state,xegs_cart)
	MCFG_CARTSLOT_INTERFACE("xegs_cart")

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","xegs")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a5200, atari_common_nodac )

	MCFG_CPU_MODIFY( "maincpu" )
	MCFG_CPU_PROGRAM_MAP(a5200_mem)
	MCFG_TIMER_ADD_SCANLINE("scantimer", a5200_interrupt, "screen", 0, 1)

	MCFG_DEVICE_REMOVE("pokey")
	MCFG_POKEY_ADD("pokey", FREQ_17_EXACT)
	MCFG_POKEY_CONFIG(a5200_pokey_interface)
	MCFG_POKEY_KEYBOARD_HANDLER(atari_a5200_keypads)
	MCFG_POKEY_INTERRUPT_HANDLER(atari_interrupt_cb)

	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 1.00)

	MCFG_PIA6821_MODIFY( "pia", a5200_pia_interface )

	MCFG_MACHINE_START_OVERRIDE( a400_state, a5200 )

	MCFG_SCREEN_MODIFY( "screen" )
	MCFG_SCREEN_REFRESH_RATE(FRAME_RATE_60HZ)
	MCFG_SCREEN_SIZE(HWIDTH*8, TOTAL_LINES_60HZ)

	MCFG_CARTSLOT_ADD("cart")
	MCFG_CARTSLOT_EXTENSION_LIST("rom,bin,a52")
	MCFG_CARTSLOT_NOT_MANDATORY
	MCFG_CARTSLOT_LOAD(a400_state,a5200_cart)
	MCFG_CARTSLOT_UNLOAD(a400_state,a5200_cart)
	MCFG_CARTSLOT_INTERFACE("a5200_cart")

	/* Software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a5200")

	/* internal ram */
	MCFG_RAM_MODIFY(RAM_TAG)
	MCFG_RAM_DEFAULT_SIZE("16K")
MACHINE_CONFIG_END


/**************************************************************
 *
 * ROM loading
 *
 **************************************************************/

ROM_START(a400)
	ROM_REGION(0x14000, "maincpu", 0) /* 64K for the CPU + 2 * 8K for cartridges */
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. B")
	ROMX_LOAD( "co12499b.rom",  0xe000, 0x1000, BAD_DUMP CRC(d818f3e8) SHA1(bcdec2188f6a6a5bfc1df4e383bd828d34b5c4ac), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599b.rom",  0xf000, 0x1000, BAD_DUMP CRC(c1690a9b) SHA1(c5248e8565574fd39ae1c3f4f356aa4cac07df95), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "reva", "OS Rev. A")
	ROMX_LOAD( "co12499a.rom",  0xe000, 0x1000, BAD_DUMP CRC(29f64e17) SHA1(abf7ec488c6b600f1b7f30bdc7f8a2bf6a727675), ROM_BIOS(2) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599a.rom",  0xf000, 0x1000, BAD_DUMP CRC(bc533f0c) SHA1(e217148495fa747fe5488132d8d22533e68c7e58), ROM_BIOS(2) )    // CRC and label waiting for confirmation

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a400pal)
	ROM_REGION(0x14000, "maincpu", 0) /* 64K for the CPU + 2 * 8K for cartridges */
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_LOAD( "co15199.rom", 0xe000, 0x1000, BAD_DUMP CRC(8e547f56) SHA1(1bd746ea798b723bfb18495a7facca113183d713) )    // Rev. A - CRC and label waiting for confirmation
	ROM_LOAD( "co15299.rom", 0xf000, 0x1000, BAD_DUMP CRC(be55b413) SHA1(d88afae49b08e75943d0258cb580e5d34756414a) )    // Rev. A - CRC and label waiting for confirmation

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a800)
	ROM_REGION(0x14000, "maincpu", 0) /* 64K for the CPU + 2 * 8K for cartridges */
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. B")
	ROMX_LOAD( "co12499b.rom",  0xe000, 0x1000, BAD_DUMP CRC(d818f3e8) SHA1(bcdec2188f6a6a5bfc1df4e383bd828d34b5c4ac), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599b.rom",  0xf000, 0x1000, BAD_DUMP CRC(c1690a9b) SHA1(c5248e8565574fd39ae1c3f4f356aa4cac07df95), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "reva", "OS Rev. A")
	ROMX_LOAD( "co12499a.rom",  0xe000, 0x1000, BAD_DUMP CRC(29f64e17) SHA1(abf7ec488c6b600f1b7f30bdc7f8a2bf6a727675), ROM_BIOS(2) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599a.rom",  0xf000, 0x1000, BAD_DUMP CRC(bc533f0c) SHA1(e217148495fa747fe5488132d8d22533e68c7e58), ROM_BIOS(2) )    // CRC and label waiting for confirmation

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)

	ROM_REGION(0x2000, "rslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a800pal)
	ROM_REGION(0x14000, "maincpu", 0) /* 64K for the CPU + 2 * 8K for cartridges */
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_LOAD( "co15199.rom", 0xe000, 0x1000, BAD_DUMP CRC(8e547f56) SHA1(1bd746ea798b723bfb18495a7facca113183d713) )    // Rev. A - CRC and label waiting for confirmation
	ROM_LOAD( "co15299.rom", 0xf000, 0x1000, BAD_DUMP CRC(be55b413) SHA1(d88afae49b08e75943d0258cb580e5d34756414a) )    // Rev. A - CRC and label waiting for confirmation

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)

	ROM_REGION(0x2000, "rslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a1200xl)
	ROM_REGION(0x18000, "maincpu", 0)
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. 11")
	ROMX_LOAD( "co60616b.rom", 0x14000, 0x2000, BAD_DUMP CRC(6e29ec8d) SHA1(3f9c06d6b4d261f3d5bf4354e3cff0c17b9347b9), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co60617b.rom", 0x16000, 0x2000, BAD_DUMP CRC(d73ce29a) SHA1(64790242d902643fe0c40dd842749f1fe461831b), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "rev10", "OS Rev. 10")
	ROMX_LOAD( "co60616a.rom", 0x14000, 0x2000, BAD_DUMP CRC(0391386b) SHA1(7c176657c88b89b8a69bf021fa8e0939efc0dff2), ROM_BIOS(2) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co60617a.rom", 0x16000, 0x2000, BAD_DUMP CRC(b502f1e7) SHA1(6688db57d97fa570aef5c15cef3e5fb2688879c2), ROM_BIOS(2) )    // CRC and label waiting for confirmation

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a600xl)
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co60302a.rom", 0xa000, 0x2000, CRC(f0202fb3) SHA1(7ad88dd99ff4a6ee66f6d162074db6f8bef7a9b6) )    // Rev. B
	ROM_LOAD( "co62024.rom",  0xc000, 0x4000, CRC(643bcc98) SHA1(881d030656b40bbe48f15a696b28f22c0b752ab0) )    // Rev. 1

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a800xl)
	ROM_REGION(0x18000, "maincpu", 0)
	ROM_FILL( 0, 0x10000, 0x00 )
	ROM_LOAD( "co60302a.rom", 0x10000, 0x2000, CRC(f0202fb3) SHA1(7ad88dd99ff4a6ee66f6d162074db6f8bef7a9b6) )   // Rev. B
	ROM_LOAD( "co61598b.rom", 0x14000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

#define rom_a800xlp rom_a800xl

ROM_START(a65xe)
	ROM_REGION(0x18000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0x10000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "co61598b.rom", 0x14000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a65xea)
	ROM_REGION(0x18000, "maincpu", 0)
	ROM_LOAD( "basic_ar.rom", 0x10000, 0x2000, CRC(c899f4d6) SHA1(043df191d1fe402e792266a108e147ffcda35130) )   // is this correct? or shall we use Rev. C?
//  ROM_LOAD( "c101700.rom",  0x14000, 0x4000, CRC(7f9a76c8) SHA1(57eb6d87850a763f11767f53d4eaede186f831a2) )   // this was from Savetz and has wrong bits!
	ROM_LOAD( "c101700.rom",  0x14000, 0x4000, CRC(45f47988) SHA1(a36b8b20f657580f172749bb0625c08706ed824c) )   // Rev. 3B ?

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a130xe)
	ROM_REGION(0x18000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0x10000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "co61598b.rom", 0x14000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(a800xe)
	ROM_REGION(0x18000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0x10000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "c300717.rom",  0x14000, 0x4000, CRC(29f133f7) SHA1(f03b9b93000ee84abb9cf8d6367241006f172182) )   // Rev. 3

	ROM_REGION(0x10000, "lslot", ROMREGION_ERASEFF)
ROM_END

ROM_START(xegs)
	ROM_REGION(0x1a000, "maincpu", 0)
	ROM_LOAD( "c101687.rom", 0x10000, 0x8000, CRC(d50260d1) SHA1(0e0625ab2473f8431640df3ac8af61925760b9b9) )    // Rev. C + Rev. 4 + Missile Command

	ROM_REGION(0x20000, "user1", ROMREGION_ERASE00)
ROM_END


ROM_START(a5200)
	ROM_REGION(0x14000, "maincpu", 0) /* 64K for the CPU + 16K for cartridges */
	ROM_SYSTEM_BIOS(0, "default", "a5200")
	ROMX_LOAD( "5200.rom",  0xf800, 0x0800, CRC(4248d3e3) SHA1(6ad7a1e8c9fad486fbec9498cb48bf5bc3adc530), ROM_BIOS(1) )
	ROM_SYSTEM_BIOS(1, "alt", "a5200 (alt)")
	ROMX_LOAD( "5200a.rom", 0xf800, 0x0800, CRC(c2ba2613) SHA1(1d2a3f00109d75d2d79fecb565775eb95b7d04d5), ROM_BIOS(2) )
ROM_END

/**************************************************************
 *
 * Driver initializations
 *
 **************************************************************/

DRIVER_INIT_MEMBER(a400_state,a800xl)
{
	a800xl_mmu(0xff);
}

DRIVER_INIT_MEMBER(a400_state,xegs)
{
	xegs_mmu(0xff);
}

DRIVER_INIT_MEMBER(a400_state,a600xl)
{
	UINT8 *rom = m_region_maincpu->base();
	memcpy( rom + 0x5000, rom + 0xd000, 0x800 );
}

/**************************************************************
 *
 * Game driver(s)
 *
 **************************************************************/

/*     YEAR  NAME      PARENT    COMPAT MACHINE     INPUT    INIT      COMPANY    FULLNAME */
COMP ( 1979, a400,     0,        0,     a400,       a800, driver_device,    0,      "Atari",   "Atari 400 (NTSC)", 0)
COMP ( 1979, a400pal,  a400,     0,     a400pal,    a800, driver_device,    0,      "Atari",   "Atari 400 (PAL)",  0)
COMP ( 1979, a800,     0,        0,     a800,       a800, driver_device,    0,      "Atari",   "Atari 800 (NTSC)", 0)
COMP ( 1979, a800pal,  a800,     0,     a800pal,    a800, driver_device,    0,      "Atari",   "Atari 800 (PAL)",  0)
COMP ( 1982, a1200xl,  a800,     0,     a1200xl,    a800xl, a400_state,  a800xl, "Atari",   "Atari 1200XL",     GAME_NOT_WORKING )      // 64k RAM
COMP ( 1983, a600xl,   a800xl,   0,     a600xl,     a800xl, a400_state,  a600xl, "Atari",   "Atari 600XL",      GAME_NOT_WORKING )      // 16k RAM
COMP ( 1983, a800xl,   0,        0,     a800xl,     a800xl, a400_state,  a800xl, "Atari",   "Atari 800XL (NTSC)",GAME_IMPERFECT_GRAPHICS )      // 64k RAM
COMP ( 1983, a800xlp,  a800xl,   0,     a800xlpal,  a800xl, a400_state,  a800xl, "Atari",   "Atari 800XL (PAL)", GAME_IMPERFECT_GRAPHICS )      // 64k RAM
COMP ( 1986, a65xe,    a800xl,   0,     a800xl,     a800xl, a400_state,  a800xl, "Atari",   "Atari 65XE",       GAME_NOT_WORKING )      // 64k RAM
COMP ( 1986, a65xea,   a800xl,   0,     a800xl,     a800xl, a400_state,  a800xl, "Atari",   "Atari 65XE (Arabic)", GAME_NOT_WORKING )
COMP ( 1986, a130xe,   a800xl,   0,     a800xl,     a800xl, a400_state,  a800xl, "Atari",   "Atari 130XE",      GAME_NOT_WORKING )      // 128k RAM
COMP ( 1986, a800xe,   a800xl,   0,     a800xl,     a800xl, a400_state,  a800xl, "Atari",   "Atari 800XE",      GAME_NOT_WORKING )      // 64k RAM
COMP ( 1987, xegs,     0,        0,     xegs,       a800xl, a400_state,  xegs,   "Atari",   "Atari XE Game System", GAME_NOT_WORKING )  // 64k RAM

CONS ( 1982, a5200,    0,        0,     a5200,      a5200, driver_device,   0,      "Atari",   "Atari 5200",       0)
