// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __ClangTransformInfo_h__
#define __ClangTransformInfo_h__

#include "ACModel/Elements.h"
#include "ACToken.h"
#include "ACFileID.h"
#include "ThisJoinPoint.h"
#include "PointCutExpr.h"
#include "SyntacticContext.h"
#include "CFlow.h"
#include "WeaverBase.h"

#include "clang/Basic/Version.h"
#include "clang/Basic/SourceLocation.h"
#include "clang/AST/Decl.h"
#include "clang/AST/DeclTemplate.h"
#include "clang/AST/Type.h"
#if !(CLANG_VERSION_MAJOR == 3 && CLANG_VERSION_MINOR == 4 && !defined(CLANG_VERSION_PATCHLEVEL))
#include "clang/AST/ExprCXX.h"
#endif
#include "llvm/Support/raw_ostream.h"

class TI_CodeAdvice : public ModelTransformInfo {
  const Condition *_condition;
  AdviceInfo *_advice_info;
public:
  void set_condition (const Condition *c) { _condition = c; }
  const Condition *get_condition () const { return _condition; }
  void set_advice_info (AdviceInfo *ai) { _advice_info = ai; }
  AdviceInfo *get_advice_info () const { return _advice_info; }

  static TI_CodeAdvice *of (const ACM_CodeAdvice &loc) {
    return static_cast<TI_CodeAdvice*>(loc.transform_info ());
  }
};

class TransformInfo : public ModelTransformInfo {
public:
  virtual ModelNode &jpl () = 0;
  virtual clang::Decl *decl () const = 0;

  static inline const TransformInfo *of (const ModelNode &loc);
  static inline clang::Decl *decl (const ModelNode &loc);
  static inline Puma::Location location (const ModelNode &loc);

  static const WeavePos &get_pos_after_token (clang::SourceLocation loc,
      WeaverBase &wb, WeavePos::Pos pos = WeavePos::WP_AFTER) {
    return wb.get_pos_after_loc(loc, pos);
  }

  // parameter 'is_id': if true, the searched string is only replaced if the
  //                    character before and after the matching substring is
  //                    not in an identifier.
  //                    (example: don't replace 'shortcut' by 'short intcut')
  static void replace_in_string(std::string& subject, const std::string& search,
      const std::string& replace, bool is_id = false) {
    size_t pos = 0;
    while ((pos = subject.find(search, pos)) != std::string::npos) {
      bool id_before = (pos > 0 && (subject[pos - 1] == '_' || std::isalpha(subject[pos - 1])));
      bool id_after  = ((pos + search.length () < subject.length()) &&
          (subject[pos + search.length()] == '_' || std::isalnum(subject[pos + search.length()])));
      if (!is_id || (!id_before && !id_after)) {
        subject.replace(pos, search.length(), replace);
        pos += replace.length();
      }
      else
        pos += search.length();
    }
  }

  static string fix_types_in_signature (const string &sig) {
    string result = sig;
    replace_in_string(result, "long", "long int", true);
    replace_in_string(result, "long int long int", "long long int", true);
    replace_in_string(result, "long int double", "long double", true);
    replace_in_string(result, "short", "short int", true);
    replace_in_string(result, "<anonymous namespace>::", "");
    replace_in_string(result, "*restrict", "*");
    replace_in_string(result, " restrict", " ");
    replace_in_string(result, "class ", "");
    replace_in_string(result, "struct ", "");
    replace_in_string(result, "union ", "");
    replace_in_string(result, "enum ", "");
    return result;
  }

  static bool needs_this (clang::FunctionDecl *func) {
    if (clang::CXXMethodDecl *m = llvm::dyn_cast<clang::CXXMethodDecl>(func))
      if (!m->isStatic())
        return true;

    return false;
  }

  // TODO: duplicate of function in ClangModelBuilder.cc
  template <typename T>
  static bool isTemplateInstantiation(T node) {
    return (node->getTemplateSpecializationKind() ==
                clang::TSK_ImplicitInstantiation ||
            node->getTemplateSpecializationKind() ==
                clang::TSK_ExplicitInstantiationDefinition);
  }

  // TODO: duplicate of function in ClangModelBuilder.cc
  static bool inside_template_instance (clang::DeclContext *scope) {
    if (llvm::isa<clang::TranslationUnitDecl>(scope))
      return false;

    if (clang::FunctionDecl *FD = llvm::dyn_cast<clang::FunctionDecl>(scope))
      if (isTemplateInstantiation(FD))
        return true;

    if (clang::VarDecl *VD = llvm::dyn_cast<clang::VarDecl>(scope))
      if (isTemplateInstantiation(VD))
        return true;

    if (clang::CXXRecordDecl *RD = llvm::dyn_cast<clang::CXXRecordDecl>(scope))
      if (isTemplateInstantiation(RD))
        return true;

    return inside_template_instance(scope->getParent());
  }

};

inline const TransformInfo *TransformInfo::of (const ModelNode &loc) {
  return loc.transform_info () ? (TransformInfo*)loc.transform_info () : 0;
}
//inline CTree *TransformInfo::tree (const ModelNode &loc) {
//  return loc.transform_info () ?
//    ((TransformInfo*)loc.transform_info ())->tree () : 0;
//}
//inline Unit *TransformInfo::unit (const ModelNode &loc) {
//  return loc.transform_info () ?
//    ((TransformInfo*)loc.transform_info ())->unit () : 0;
//}
inline clang::Decl *TransformInfo::decl (const ModelNode &loc) {
  return loc.transform_info () ?
    ((TransformInfo*)loc.transform_info ())->decl () : 0;
}
inline Puma::Location TransformInfo::location (const ModelNode &loc) {
  return Puma::Location (); // FIXME: implement for Clang
}

class TI_Namespace : public TransformInfo {
  // pointer to the Clang namespace object (for transformation)
  clang::NamespaceDecl *_decl;
public:
  void decl (clang::NamespaceDecl *n) { _decl = n; }
  virtual clang::NamespaceDecl *decl () const { return _decl; }
};

class TI_Class : public TransformInfo {
  clang::RecordDecl *_decl;

  static void get_member_contexts (const clang::RecordDecl *decl,
      list<ClangSyntacticContext> &member_contexts) {

    for (clang::RecordDecl::field_iterator i = decl->field_begin ();
        i != decl->field_end (); ++i) {

      // Only certain members are delivered.
      // If this is not the right choice for all use case, add filter flags to
      // the argument list of this function
      clang::FieldDecl *attr = *i;
      if (attr->getNameAsString ().empty ()) {
        const clang::RecordType *UT = attr->getType ()->getAsUnionType ();
        if (UT) { // members of anonymous unions in a record are also record members
          get_member_contexts (UT->getDecl (), member_contexts);
        }
        continue;
      }
//      if (attr->isStatic () || attr->isAnonymous () || attr->EnumeratorInfo ())
//        continue;

      member_contexts.push_back (ClangSyntacticContext (attr));
    }
  }

public:
  TI_Class () : _decl (0) {}

  bool valid () const { return _decl != 0; }

  void decl (clang::RecordDecl *c) { _decl = c; }
  virtual clang::RecordDecl *decl () const { return _decl; }

  SyntacticContext get_def_context () const { return SyntacticContext (_decl); }

  void get_member_contexts (list<ClangSyntacticContext> &member_contexts) const {
    get_member_contexts (_decl, member_contexts);
  }

  enum SMKind { CONSTRUCTOR, COPY_CONSTRUCTOR, DESTRUCTOR };
  bool may_have_implicit (SMKind kind) {
    return may_have_implicit (kind, _decl);
  }

  bool may_have_implicit (SMKind kind, const clang::RecordDecl *decl) {
    const clang::CXXRecordDecl *d = llvm::cast<clang::CXXRecordDecl>(decl);
    if (!d)
      return false;
    if (kind == CONSTRUCTOR || kind == COPY_CONSTRUCTOR) {
      for (clang::CXXRecordDecl::ctor_iterator i = d->ctor_begin ();
          i != d->ctor_end (); ++i) {
        clang::CXXConstructorDecl *cd = *i;
        if (kind == CONSTRUCTOR && cd->isDefaultConstructor () &&
            cd->getAccess () == clang::AS_private)
          return false;
        if (kind == COPY_CONSTRUCTOR && cd->isCopyConstructor () &&
            cd->getAccess () == clang::AS_private)
          return false;
      }
    }
    else { // destructor
      const clang::CXXDestructorDecl *dd = d->getDestructor ();
      if (dd && kind == DESTRUCTOR && dd->getAccess () == clang::AS_private)
        return false;
    }

    for (clang::CXXRecordDecl::base_class_const_iterator i = d->bases_begin ();
        i != d->bases_end (); ++i) {
      const clang::CXXRecordDecl *bd = (*i).getType ()->getAsCXXRecordDecl ();
      if (bd && !may_have_implicit (kind, bd))
        return false;
    }

    for (clang::CXXRecordDecl::field_iterator i = d->field_begin ();
        i != d->field_end (); ++i) {
      const clang::FieldDecl *attr = *i;
      const clang::CXXRecordDecl *rd = attr->getType ()->getAsCXXRecordDecl ();
      if (rd && !may_have_implicit (kind, rd))
        return false;
    }
    return true;
  }

  // return the position behind the opening bracket of the class body
  const WeavePos &body_start_pos (WeaverBase &wb) const {
    // TODO: iterating over all decls and finding the one with the smallest
    // location is a terribly comlicated solution. However, I haven't found
    // a way to get the location of the opening bracket.
    clang::SourceLocation min_loc;
    for (clang::DeclContext::decl_iterator i = _decl->decls_begin ();
        i != _decl->decls_end (); ++i) {
      clang::SourceLocation cur_loc = (*i)->getLocStart();
      if ((*i)->isImplicit () || !cur_loc.isValid ())
        continue;
      if (!min_loc.isValid () || (cur_loc < min_loc))
          min_loc = (*i)->getLocStart();
    }
    if (min_loc.isValid ())
      return wb.weave_pos(min_loc, WeavePos::WP_BEFORE);
    else
      return wb.weave_pos(_decl->getRBraceLoc(), WeavePos::WP_BEFORE);
  }

  // return the position in front of the closing bracket of the class body
  const WeavePos &body_end_pos (WeaverBase &wb) const {
    return wb.weave_pos(_decl->getRBraceLoc(), WeavePos::WP_BEFORE);
  }

  // return the position of the first token of the class definition
  const WeavePos &objdecl_start_pos (WeaverBase &wb) const {
    return wb.weave_pos (_decl->getLocStart(), WeavePos::WP_BEFORE);
  }

  // return the position after the ";" of the class definition
  const WeavePos &objdecl_end_pos (WeaverBase &wb) const {
    // FIXME: This relies on the lack of spaces between the closing '}' and ';'.
    return wb.weave_pos (_decl->getLocEnd().getLocWithOffset(2), WeavePos::WP_AFTER);
  }

  // check whether this is a class and not a struct
  bool is_class () const { return _decl->isClass(); }

  // check whether this is a struct (more restrictive than 'is_class')
  bool is_struct () const { return _decl->isStruct(); }

  // check whether the class is defined (=has a body) and not only declared
  bool is_defined () const { return _decl->isCompleteDefinition (); }

  // checks whether this class is a template *instance*
  bool is_template_instance () const {
    const clang::CXXRecordDecl *d = llvm::cast<clang::CXXRecordDecl>(_decl);
    return d && isTemplateInstantiation (d);
  }

  // check whether the class is defined in a extern "C" block
  bool is_extern_c () const {
    return is_extern_c (_decl);
  }

  static bool is_extern_c (clang::RecordDecl *d) {
    clang::DeclContext *dc = d;
    while (dc->getDeclKind() != clang::Decl::TranslationUnit) {
      if (dc->getDeclKind() == clang::Decl::LinkageSpec)
        return clang::cast<clang::LinkageSpecDecl>(dc)->getLanguage() ==
            clang::LinkageSpecDecl::lang_c;
      dc = dc->getParent();
    }
    return false;
    // Future clang versions will support this: return _decl->isExternCContext ();
    // TODO: current version support now: return d->isExternCContext();
  }

  // checks whther the class is defined within a template instance
  bool is_in_template_instance () const { return inside_template_instance(_decl); }

  static string name(clang::RecordDecl *ci) {
    string name_with_template_args;
    llvm::raw_string_ostream os(name_with_template_args);
    ci->getNameForDiagnostic(os, ci->getASTContext().getPrintingPolicy(), false);
    string result = os.str ();
    replace_in_string(result, ", ", ",");
    return fix_types_in_signature(result);
  }

  static TI_Class *of (const ACM_Class &loc) {
    return static_cast<TI_Class*>(loc.transform_info ());
  }
};

class TI_Aspect : public TI_Class {
public:

  clang::FunctionDecl *aspectof () const {

    clang::CXXRecordDecl *d = llvm::cast<clang::CXXRecordDecl>(decl());
    for (clang::CXXRecordDecl::decl_iterator di = d->decls_begin(),
                                             de = d->decls_end();
         di != de; ++di) {
      clang::NamedDecl *nd = llvm::dyn_cast<clang::NamedDecl> (*di);
      if (!nd)
        continue;
      std::string name = nd->getNameAsString();
      if (name != "aspectof" && name != "aspectOf")
        continue;
      if (nd->getKind () == clang::Decl::FunctionTemplate)
        return llvm::dyn_cast<clang::FunctionTemplateDecl>(nd)->getTemplatedDecl ();
      else if (nd->getKind () == clang::Decl::CXXMethod)
        return llvm::dyn_cast<clang::CXXMethodDecl>(nd);
    }
    return 0;
  }

  static const TI_Aspect *of (const ACM_Aspect &loc) {
    return static_cast<TI_Aspect*>(loc.transform_info ());
  }

};

class TI_Function : public TransformInfo {
  mutable vector<ClangSyntacticContext> _contexts;
  clang::FunctionDecl *_decl;
public:

  const vector<ClangSyntacticContext> &syntactic_contexts () const {
    if (_contexts.size() == 0) {
      // Store a pointer to each declaration and the definition (if one exists)
      for (clang::FunctionDecl::redecl_iterator ri = _decl->redecls_begin(),
                                                re = _decl->redecls_end();
           ri != re; ++ri)
        _contexts.push_back(ClangSyntacticContext(*ri));
    }
    return _contexts;
  }

  void decl (clang::FunctionDecl *c) { _decl = c; }
  virtual clang::FunctionDecl *decl () const { return _decl; }
  void add_decl (clang::FunctionDecl *c) {
    _contexts.push_back(ClangSyntacticContext(c));
  }

  static string name(clang::FunctionDecl *func_info) {
    string out = func_info->getNameAsString();
    // Add a space after "operator" for Puma compatibility.
    if (out.size() > 8 && llvm::StringRef(out).startswith("operator") &&
        out[8] != ' ' && !isalnum(out[8]))
      out.insert(out.begin() + 8, ' ');
    clang::FunctionTemplateSpecializationInfo *ftsi =
        func_info->getTemplateSpecializationInfo ();
    if (ftsi) {
      clang::ASTContext &ctx = func_info->getASTContext();
      out += "<";
      for (unsigned a = 0; a < ftsi->TemplateArguments->size (); a++) {
        if (a > 0) out += ", ";
        const clang::TemplateArgument &ta = ftsi->TemplateArguments->get (a);
        std::string buf;
        llvm::raw_string_ostream arg(buf);
        ta.print(ctx.getPrintingPolicy(), arg);
        out += arg.str ();
      }
      if (out[out.size () -1] == '>')
        out += " ";
      out += ">";
    }
    return fix_types_in_signature(out);
  }

  static string signature(clang::FunctionDecl *func_info) {
    std::string str;
    llvm::raw_string_ostream out(str);
    clang::ASTContext &ctx = func_info->getASTContext();

    // FIXME: templates

    out << '(';
    for (clang::FunctionDecl::param_iterator i = func_info->param_begin(),
                                             e = func_info->param_end();
         i != e; ++i) {
      if (i != func_info->param_begin())
        out << ',';
      clang::QualType ty = ctx.getSignatureParameterType((*i)->getType());
      ty.getCanonicalType ().print(out, ctx.getPrintingPolicy());
//      (*i)->getType().getCanonicalType ().print(out, ctx.getPrintingPolicy());
    }
    out << ')';

    // add CV qualifiers
    if (clang::CXXMethodDecl *m =
            llvm::dyn_cast<clang::CXXMethodDecl>(func_info)) {
      if (m->isConst())
        out << " const";
      if (m->isVolatile())
        out << " volatile";
    }

    return name(func_info) + fix_types_in_signature(out.str ());
  }

  bool is_const () const {
    clang::CXXMethodDecl *m = llvm::dyn_cast<clang::CXXMethodDecl>(_decl);
    return m && m->isConst ();
  }

  static const TI_Function *of (const ACM_Function &loc) {
    return static_cast<TI_Function*>(loc.transform_info ());
  }
};

class TI_Variable : public TransformInfo {
  clang::DeclaratorDecl *_decl;
public:
  TI_Variable () : _decl (0) {}

  void decl (clang::DeclaratorDecl *oi) { _decl = oi; }
  virtual clang::DeclaratorDecl *decl () const { return _decl; }
public:
  static const TI_Variable *of (const ACM_Variable &loc) {
    return static_cast<TI_Variable*>(loc.transform_info ());
  }
};

class TI_Type : public TransformInfo {
  clang::QualType _type;
public:
  void type (clang::QualType ti) { _type = ti; }
  clang::QualType type () const { return _type; }

  virtual clang::Decl *decl () const { return 0; }

  bool is_const () const { return _type.isConstQualified (); }
  bool is_reference () const { return _type.getTypePtr ()->isReferenceType (); }

  static string name (clang::ASTContext &ctx, clang::QualType type_info) {
    // Get the canonical type as a string, looking through template parameters
    // and typedefs.
    std::string type = "?";
    type_info.getCanonicalType().getAsStringInternal(type,
                                                     ctx.getPrintingPolicy());

    // Replace <anonymous> with <unnamed>, for PUMA compatibility.
    size_t loc = 0;
#if CLANG_VERSION_MAJOR == 3 && CLANG_VERSION_MINOR == 4 && !defined(CLANG_VERSION_PATCHLEVEL)
    while ((loc = type.find("<anonymous>"), loc) != std::string::npos)
      type.replace(loc, sizeof("<anonymous>") - 1, "<unnamed>");
#else // C++ 11 interface
    while ((loc = type.find("(anonymous namespace)"), loc) != std::string::npos)
      type.replace(loc, sizeof("(anonymous namespace)") - 1, "<unnamed>");
#endif

    int last = type.length() - 1;
    if (type[last] == '?') last--;
    while(type[last] == ' ') last--;
    return fix_types_in_signature(type.substr (0, last + 1));
  }
  
  /** Print the textual representation of a type on the given stream.
   *  \param type The type to print
   *  \param ctx The ASTContext
   *  \param os The output stream. 
   *  \param name Optional name of the entity to print. 
   *  \param abs Print qualified names with root qualifier. 
   *  \param tdef Print the name of a typedef instead of the underlying type.
   *  \param elaborated_type_spec Print elaborated type specifier before 
   *                              class, union, and enumeration types.
   *  \param unnamed Print unnamed namespaces as '<unnamed>' */
  static void TypeText (clang::QualType type, clang::ASTContext &ctx,
                 llvm::raw_string_ostream &os, const char *name = (const char*)0,
                 bool abs = false, bool tdef = false, 
                 bool elaborated_type_spec = false,
                 bool unnamed = false) {
    //FIXME TODO:
    // the 'tdef && abs' and 'unnamed' parameters are ignored currently

    clang::PrintingPolicy pp(ctx.getPrintingPolicy());
    pp.SuppressUnwrittenScope = true; // Suppress <anonymous> in generated types
    
    if (tdef && !abs) {
      // clang's print functionality yields already non-absolute typedefs
      type.print(os, pp, name);
      return;
    }

    std::string str;
    llvm::raw_string_ostream stream(str);
    type.getCanonicalType().print(stream, pp, name); // FIXME: tdef=true?
    
    std::string elaborated = "";
    if (elaborated_type_spec && !llvm::isa<clang::TypedefType>(type)) {
      // elaborated type specifiers must not refer to a typedef
      if(type->isEnumeralType()) {
        elaborated.append("enum ");
      }
      else if(type->isStructureType()) {
        elaborated.append("struct ");
      }
      else if(type->isClassType()) {
        elaborated.append("class ");
      }
      else if(type->isUnionType()) {
        elaborated.append("union ");
      }
    }
    
    std::string root_qualifier = "";
    if (abs) {
      // append root namespace scope if possible
      clang::QualType base_type = type.getCanonicalType();
      if (base_type->isArrayType()) {
        // get the type of the array's elements
        base_type = base_type->getBaseElementTypeUnsafe()->getCanonicalTypeInternal();
      }
      else if (base_type->isAnyPointerType() || base_type->isReferenceType()) {
        base_type = base_type->getPointeeType();
      }
      if (base_type->isStructureOrClassType() ||
        base_type->isEnumeralType() ||
        base_type->isUnionType()) {
        // only append "::" for user-defined types
        root_qualifier = "::";
      }
    }
    
    // insert 'elaborated' and 'root_qualifier' after leading qualifiers
    // FIXME: find a better solution ...
    std::string leading_qualifiers;
    const std::string const_str("const ");
    const std::string volatile_str("volatile ");
    const std::string restrict_str("restrict ");
    while (true) {
      if (stream.str().find(const_str) == 0) {
        leading_qualifiers.append(const_str);
        stream.str().replace(0, const_str.length(), "");
      }
      else if (stream.str().find(volatile_str) == 0) {
        leading_qualifiers.append(volatile_str);
        stream.str().replace(0, volatile_str.length(), "");
      }
      else if (stream.str().find(restrict_str) == 0) {
        leading_qualifiers.append(restrict_str);
        stream.str().replace(0, restrict_str.length(), "");
      }
      else {
        break;
      }
    }
    os << leading_qualifiers << elaborated << root_qualifier << stream.str();
  }

  static const TI_Type *of (const ACM_Type &loc) {
    return static_cast<TI_Type*>(loc.transform_info ());
  }
};

class TI_Arg : public TransformInfo {
  clang::QualType _type;
public:
  void type (clang::QualType ti) { _type = ti; }
  clang::QualType type () const { return _type; }

  virtual clang::Decl *decl () const { return 0; }

  static const TI_Arg *of (const ACM_Arg &loc) {
    return static_cast<TI_Arg*>(loc.transform_info ());
  }
};

class TI_Code : public TransformInfo {
  bool _has_implicit;
  CFlowList _triggers;
public:
  TI_Code() : _has_implicit( false ) {};

  // remember implicit joinpoints needing to be considered
  void remember_implicit() { _has_implicit = true; }
  bool has_implicit_joinpoints() { return _has_implicit; }

  // consider a necessary cflow trigger at this join point
  bool consider (const CFlow &cflow) {
    _triggers.push_back (cflow);
    return true;
  }

  // return the list of cflows that must be entered/left here
  const CFlowList &cflows () const { return _triggers; }

  // that types (for the JoinPoint-API)
  virtual std::string that_type_string () const { return "void"; }

  // target type (for the JoinPoint-API)
  virtual std::string target_type_string () const { return "void"; }

  // argument type (for the JoinPoint-API)
  virtual std::string arg_type_string (unsigned no) const { return "void"; }

  // entity type (for the JoinPoint-API)
  virtual std::string entity_type_string() const {
    return format_type( decl() );
  }

  // helper functions for derived classes
  static std::string get_type_string (clang::Decl *obj) {
    clang::PrintingPolicy pp (obj->getASTContext().getPrintingPolicy());
    pp.SuppressUnwrittenScope = true; // Suppress <anonymous>

    // if the 'obj' refers to a record or member function, we have to print a record 'r'
    clang::CXXRecordDecl *r = llvm::dyn_cast<clang::CXXRecordDecl>(obj);
    clang::CXXMethodDecl *m = llvm::dyn_cast<clang::CXXMethodDecl>(obj);
    if (m)
      r = llvm::dyn_cast<clang::CXXRecordDecl>(m->getParent ());

    if (!r)
      if (clang::ParmVarDecl *pd = llvm::dyn_cast<clang::ParmVarDecl>(obj))
        r = pd->getType()->getAsCXXRecordDecl();

    if (r) {
      std::string buf;
      llvm::raw_string_ostream out (buf);
      if (m) {
        // preserve const volatile qualifiers of member functions
        // we could also print m->getThisType(obj->getASTContext())
        if(m->isConst())
          out << "const ";
        if (m->isVolatile())
          out << "volatile ";
      }

      std::string QualName;
      llvm::raw_string_ostream OS(QualName);
      r->printQualifiedName(OS, pp);

//      out << "::" << r->getQualifiedNameAsString ();
      out << "::" << OS.str ();

      // if the record is a template instance, print the arguments too
      const clang::ClassTemplateSpecializationDecl *spec =
                clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(r);
      if (spec) {
        const clang::TemplateArgumentList &TemplateArgs = spec->getTemplateArgs();
        clang::TemplateSpecializationType::PrintTemplateArgumentList(out,
            TemplateArgs.data(), TemplateArgs.size(), pp);
      }
      return fix_types_in_signature (out.str ());
    }
    else if (clang::ParmVarDecl *pd = llvm::dyn_cast<clang::ParmVarDecl>(obj))
      // TODO: how to make the generated type names absolute?
      return fix_types_in_signature (pd->getOriginalType ().getCanonicalType ().getAsString (pp));
    else
      return "void";
  }

  static std::string format_type( clang::Decl *obj ) {
    clang::ValueDecl *typed_obj = llvm::dyn_cast_or_null<clang::ValueDecl>( obj );
    if( ! typed_obj )
      return "void";

    clang::PrintingPolicy pp (typed_obj->getASTContext().getPrintingPolicy());
    pp.SuppressUnwrittenScope = true; // Suppress <anonymous>

    // pp.PolishForDeclaration = true; // should remove __attribute__, but is
                                       // ignored by clang 3.4
    // re-implement clang's 'getAsString' for functions to remove __attribute__
    if( clang::FunctionDecl* func_info = llvm::dyn_cast<clang::FunctionDecl>(obj) ) {
      std::string str;
      llvm::raw_string_ostream out(str);
      
      // arguments string
      out << '(';
      for (clang::FunctionDecl::param_iterator i = func_info->param_begin(),
	e = func_info->param_end();
      i != e; ++i) {
	if (i != func_info->param_begin())
	  out << ',';
	(*i)->getType().getCanonicalType ().print(out, pp);
      }
      out << ')';

#if CLANG_VERSION_MAJOR == 3 && CLANG_VERSION_MINOR == 4 && !defined(CLANG_VERSION_PATCHLEVEL)
      func_info->getResultType().getCanonicalType().getAsStringInternal( out.str(), pp );
#else // C++ 11 interface
      func_info->getReturnType().getCanonicalType().getAsStringInternal( out.str(), pp );
#endif

      // add CV qualifiers
      if (clang::CXXMethodDecl *m =
	llvm::dyn_cast<clang::CXXMethodDecl>(func_info)) {
	if (m->isConst())
	  out << " const";
	if (m->isVolatile())
	  out << " volatile";
      }
      return fix_types_in_signature( out.str() );
    }
    
    return fix_types_in_signature( typed_obj->getType().getCanonicalType().getAsString( pp ) );
  }

  static TI_Code *of (const ACM_Code &loc) {
    return static_cast<TI_Code*>(loc.transform_info ());
  }
};

class TI_Method : public TI_Code {
  clang::FunctionDecl *_decl;

public:
  TI_Method () : _decl (0) {}

  void decl (clang::FunctionDecl *f) { _decl = f; }
  virtual clang::Decl *decl () const { return _decl; }

  // that type (for the JoinPoint-API)
  virtual std::string that_type_string() const {
    return get_type_string(_decl);
  }

  // target type (for the JoinPoint-API)
  virtual std::string target_type_string() const {
    return get_type_string(_decl);
  }

  virtual std::string arg_type_string (unsigned no) const {
    return get_type_string (_decl->getParamDecl (no));
  }
};

class TI_Access : public TI_Code {
private:
  clang::DeclaratorDecl *_entity;

  clang::Expr *_node;
  clang::Expr *_ref_node;

  clang::Decl *_origin;

  clang::Expr *_target_expr;
public:
  TI_Access() : _entity(0), _node(0), _ref_node(0), _origin(0), _target_expr(0) {}

protected: // make setter protected as derived class might need one with different signature
  void entity( clang::DeclaratorDecl *ent ) { _entity = ent; }
public:
  clang::DeclaratorDecl *entity() const { return _entity; }
  virtual clang::Decl *decl() const { return _entity; } // defined in TransformInfo

  bool entity_is_const() const { return false; }

protected: // make setter protected as derived class might need one with different signature
  void tree_node( clang::Expr *n ) { _node = n; };
public:
  clang::Expr *tree_node() const { return _node; };

  void ref_node( clang::Expr *ref ) {
    _ref_node = ref;

    _target_expr = find_target_expr(); // init caching var
  };
  clang::Expr *ref_node() const { return _ref_node; };

  // checks if the original access uses a qualified target entity name
  bool is_qualified () const {
    clang::MemberExpr *me = clang::dyn_cast_or_null<clang::MemberExpr>( _ref_node );
    return me && me->hasQualifier();
  }

  void origin( clang::Decl *o ) { _origin = o; }
  clang::Decl *origin() const { return _origin; }

  bool target_is_ptr() const { return _target_expr && _target_expr->getType().getTypePtr()->isPointerType(); }
  bool target_is_implicit() const { return _target_expr && _target_expr->isImplicitCXXThis(); }
  bool target_is_const() const { return _target_expr && _target_expr->getType ().isConstQualified(); }
  bool has_target_expr() const { return _target_expr && ! target_is_implicit(); }
  clang::Expr *target_expr() const {
    if( ! target_is_implicit() ) // by convention only explicit expr are returned
      return _target_expr;
    else
      return 0;
  }
  // no setter as it is done internally
  virtual clang::Expr *find_target_expr() const {
    // analog to clang::CXXMemberCallExpr::getImplicitObjectArgument()
    if( const clang::MemberExpr *member = llvm::dyn_cast<clang::MemberExpr>( _ref_node ) )
      return member->getBase();
    else if( const clang::BinaryOperator *op = llvm::dyn_cast<clang::BinaryOperator>( _ref_node ) )
      if( op->getOpcode() == clang::BO_PtrMemD || op->getOpcode() == clang::BO_PtrMemI )
        return op->getLHS();

    return 0;
  }

  // target type (for the JoinPoint-API)
  virtual const clang::RecordDecl *target_class () const = 0;

  static TI_Access *of( const ACM_Access &loc ) {
    return static_cast<TI_Access *>( loc.transform_info() );
  }

  const SyntacticContext access_context () const { return SyntacticContext( _origin ); }
  const SyntacticContext entity_context () const { return SyntacticContext( _entity ); }

  // that type (for the JoinPoint-API)
  virtual std::string that_type_string() const {
    if( clang::CXXMethodDecl *m = llvm::dyn_cast<clang::CXXMethodDecl>( _origin ) )
      return get_type_string( m->getParent() );
    else if( clang::VarDecl *vd = llvm::dyn_cast<clang::VarDecl>( _origin ) )
      if( clang::CXXRecordDecl *r = llvm::dyn_cast<clang::CXXRecordDecl>( vd->getDeclContext() ) )
        return get_type_string( r );
    return "void";
  }

  // target type (for the JoinPoint-API)
  virtual std::string target_type_string() const {
    clang::Decl *tc = const_cast<clang::RecordDecl *>( target_class() ); // FIXME cast should be unnecessary
    return ( tc ? get_type_string( tc ) : string( "void" ));
  }

  virtual bool has_result() const = 0;
  virtual clang::QualType result_type() const = 0;

  const WeavePos &before_pos (WeaverBase &wb) {
    return wb.weave_pos(_node->getLocStart (), WeavePos::WP_BEFORE);
  }
  const WeavePos &after_pos (WeaverBase &wb) {
    return get_pos_after_token(_node->getLocEnd (), wb);
  }

  struct PH : public clang::PrinterHelper {
    virtual bool  handledStmt (clang::Stmt *node, llvm::raw_ostream &os) {
      clang::CXXOperatorCallExpr *ce = clang::dyn_cast<clang::CXXOperatorCallExpr> (node);
      clang::FunctionDecl *fd = (ce ? ce->getDirectCallee () : 0);
      if (ce && ce->getNumArgs () == 1 &&
          fd->getNameAsString () == "operator->") {
        clang::ASTContext &ctx = fd->getASTContext();
        os << "(";
        ce->getArg (0)->printPretty(os, this, ctx.getPrintingPolicy(), 0);
        os << ").operator->()";
        return true;
      }
      // WORKAROUND for Clang 3.4 problem:
      // Implicit calls to conversion operators are printed explicitly, but
      // the object argument is not put into brackets. For example:
      // "a & b" might become "a & b.operator int()"
      // This fix generates the brackets: "(a & b).operator int()"
      clang::CXXMemberCallExpr *mce = clang::dyn_cast<clang::CXXMemberCallExpr> (node);
      fd = (mce ? mce->getDirectCallee () : 0);
      if (fd && clang::dyn_cast<clang::CXXConversionDecl>(fd)) {
        clang::MemberExpr *me = clang::dyn_cast<clang::MemberExpr>(mce->getCallee ());
        if (me) {
          os << "(";
          clang::ASTContext &ctx = fd->getASTContext();
          me->getBase ()->printPretty(os, this, ctx.getPrintingPolicy(), 0);
          os << ")" << (me->isArrow () ? "->" : ".");
          os << fd->getNameAsString () << "()";
          return true;
        }
      }
      return false;
    }
  };

  string code () const {
    PH ph;
    clang::ASTContext &ctx = _origin->getASTContext();
    std::string buf;
    llvm::raw_string_ostream out (buf);
    _node->printPretty(out, &ph, ctx.getPrintingPolicy(), 0);
    return buf;
  }

  // returns true if the access needs special rights
  bool needs_rights () const {
    if( ! target_class() )
      return false;
    if( _entity->getAccess () == clang::AS_public )
      return false;
    // TODO: check real accessiblity from target_class()
    return true;
#if 0
    // get the target object type
    CTypeInfo *type = target_type ()->UnqualType ();

    // no member function => no accessibility problem
    if (type->isVoid ())
      return false;

    // static member => no problem only if public
    if (called_func->isStaticMethod ())
      return (called_func->Protection () != CProtection::PROT_PUBLIC);

    // normal member function => look up the accessibility
    if (type->ClassInfo () &&
      type->ClassInfo ()->Accessibility (called_func) == CProtection::PROT_PUBLIC)
      return false;

    return true;
#endif
  }
};

class TI_MethodCall : public TI_Access {
  clang::FunctionDecl *_called_func;
  clang::Decl *_caller_obj; // FIXME: can be removed in the future, origin hold the same information
public:
  TI_MethodCall () : _called_func (0), _caller_obj (0) {}

  void called (clang::FunctionDecl *f) { _called_func = f; entity( f ); }
  clang::FunctionDecl *called () const { return _called_func; }
  void caller (clang::Decl *c) { _caller_obj = c; origin( c ); }
  clang::Decl *caller () { return _caller_obj; }
  void tree_node( clang::Expr *n ) {
    TI_Access::tree_node( n );
    if( clang::CallExpr *call = llvm::dyn_cast_or_null<clang::CallExpr>( n ) )
      ref_node( call->getCallee()->IgnoreParenImpCasts() );
  }
  
  static TI_MethodCall *of (const ACM_Call &loc) {
    return static_cast<TI_MethodCall*>(loc.transform_info ());
  }

  bool is_builtin_operator() const {
    clang::Expr *node = TI_Access::tree_node();
    if( llvm::isa<clang::UnaryOperator>( node )
        || llvm::isa<clang::BinaryOperator>( node )
        || llvm::isa<clang::ArraySubscriptExpr>( node )
        || llvm::isa<clang::ConditionalOperator>( node ) )
      return true;
    else if( llvm::isa<clang::CallExpr>( node ) )
      return false;
    else {
      assert( false && "Unknow expr type in TI_MethodCall" );
      return false;
    }
  }

  static string built_in_operator_kind_as_string( clang::Expr *node ) {
    if( clang::UnaryOperator *uo = llvm::dyn_cast<clang::UnaryOperator>( node ) )
      return clang::UnaryOperator::getOpcodeStr( uo->getOpcode() ).str();
    else if( clang::BinaryOperator * bo = llvm::dyn_cast<clang::BinaryOperator>( node ) )
      return clang::BinaryOperator::getOpcodeStr( bo->getOpcode() ).str();
    else if( llvm::isa<clang::ArraySubscriptExpr>( node ) )
      return "[]";
    else if( llvm::isa<clang::ConditionalOperator>( node ) )
      return "?:";

    assert( false && "Unknown expr type" );
    return "<?>"; // dummy
  }

  string built_in_operator_kind_as_string() {
    assert( is_builtin_operator() );
    return( built_in_operator_kind_as_string( TI_Access::tree_node() ) );
  }

  static clang::QualType argType( const clang::Expr *node,  clang::ASTContext &ctx, unsigned no ) {
    if( const clang::UnaryOperator * uo = llvm::dyn_cast<clang::UnaryOperator>( node ) ) {
      if( uo->isPostfix() && ( no == 1 ) )
        return ctx.IntTy; // Dummy argument of postfix ops
    }

    clang::QualType type;
    const clang::Expr *arg = get_argument( node, no );
    type = arg->getType();
    if( ! llvm::isa<clang::CallExpr>( node ) ) {
      if( arg->isLValue() )
        type = ctx.getLValueReferenceType( type );
      else if( arg->isXValue() )
        type = ctx.getRValueReferenceType( type );
    }

    return type;
  }

  static std::string arg_type_string( const clang::Expr *node, clang::ASTContext &ctx, unsigned no ) {
    clang::PrintingPolicy pp( ctx.getPrintingPolicy() );
    pp.SuppressUnwrittenScope = true; // Suppress <anonymous>

    return fix_types_in_signature( argType( node, ctx, no ).getCanonicalType().getAsString(pp) );
  }

  virtual std::string arg_type_string (unsigned no) const {
    if( ! is_builtin_operator() && ( no < _called_func->getNumParams () ) )
      return get_type_string (_called_func->getParamDecl (no));

    clang::ASTContext &ctx = origin()->getASTContext();
    return arg_type_string( TI_Access::tree_node(), ctx, no );
  }

  bool is_ternary_expr() const {
    return llvm::isa<clang::ConditionalOperator>( TI_Access::tree_node() );
  }

  bool is_binary_expr () const {
    if( llvm::isa<clang::BinaryOperator>( TI_Access::tree_node() ) )
      return true; //return true for built-in binary operator

    clang::CXXOperatorCallExpr *ce = clang::dyn_cast<clang::CXXOperatorCallExpr> (TI_Access::tree_node());
    return (ce && ce->getNumArgs () == 2 && !is_index_expr () && !is_postfix_expr ());
  }

  bool is_compound_assignment() const {
    if( llvm::isa<clang::CompoundAssignOperator>( TI_Access::tree_node() ) )
      return true;
    else if( clang::UnaryOperator *UO = llvm::dyn_cast<clang::UnaryOperator>( TI_Access::tree_node() ) )
      return UO->isIncrementDecrementOp();
    else
      return false;
  }

  bool is_unary_expr () const {
    if( llvm::isa<clang::UnaryOperator>( TI_Access::tree_node() ) && ! is_postfix_expr() )
      return true; //return true for built-in unary operator

    clang::CXXOperatorCallExpr *ce = clang::dyn_cast<clang::CXXOperatorCallExpr> (TI_Access::tree_node());
    return (ce && ce->getNumArgs () == 1);
  }

  bool is_postfix_expr () const {
    if( clang::UnaryOperator *op = llvm::dyn_cast<clang::UnaryOperator>( TI_Access::tree_node() ) )
      if( op->isPostfix() )
        return true; // return true for built-in postfix operator

    clang::CXXOperatorCallExpr *ce = clang::dyn_cast<clang::CXXOperatorCallExpr> (TI_Access::tree_node());
    return (ce && ce->getNumArgs () == 2 &&
        (_called_func->getNameAsString () == "operator++" ||
            _called_func->getNameAsString () == "operator--"));
  }

  bool is_implicit_conversion () const {
    if (!llvm::isa<clang::CXXConversionDecl> (_called_func))
      return false;
    clang::CXXMemberCallExpr *mce = clang::dyn_cast<clang::CXXMemberCallExpr>( TI_Access::tree_node() );
    if (!mce)
      return false;
    // TODO: is there a better way to distinguish 'c' from 'c.operator int*()'?
    return (mce->getCallee ()->getLocEnd () == TI_Access::tree_node()->getLocEnd ());
  }

  // This method returns the clang::SourceLocation of the operator according to the operator-type.
  clang::SourceLocation get_operator_location() const {
    clang::Expr *node = TI_Access::tree_node();
    if( clang::CXXOperatorCallExpr *ce = clang::dyn_cast<clang::CXXOperatorCallExpr> ( node ) )
      return ce->getOperatorLoc ();
    else if( clang::UnaryOperator *uo = clang::dyn_cast<clang::UnaryOperator>( node ) ) {
      return uo->getOperatorLoc();
    }
    else if( clang::BinaryOperator *bo = clang::dyn_cast<clang::BinaryOperator>( node ) ) {
      return bo->getOperatorLoc();
    }

    assert( false && "The expr does not have THE ONE operator-location." );
    return node->getLocStart(); // dummy
  }

  const WeavePos &op_before_pos (WeaverBase &wb) {
    return wb.weave_pos(get_operator_location(), WeavePos::WP_BEFORE);
  }

  const WeavePos &op_after_pos (WeaverBase &wb) {
    return get_pos_after_token(get_operator_location(), wb);
  }

  // This method takes an argument-index as unsigned int and returns the
  // correspondent argument-clang::Expr-pointer.
  static unsigned int get_argument_count( const clang::Expr *node ) {
    if( const clang::CallExpr *ce = clang::dyn_cast<clang::CallExpr>( node ) )
      return ce->getNumArgs();
    else {
      if( const clang::UnaryOperator *uo = llvm::dyn_cast<clang::UnaryOperator>( node ) ) {
        if( uo->isPostfix() )
          return 2;
        else
          return 1;
      }
      else if( llvm::isa<clang::BinaryOperator>( node ) )
        return 2;
      else if( llvm::isa<clang::ArraySubscriptExpr>( node ) )
        return 2;
      else if( llvm::isa<clang::ConditionalOperator>( node ) )
        return 3;

      assert( false && "Unknown expr type" );
      return 0; // dummy
    }
  }

  unsigned int get_argument_count() const {
    return get_argument_count( TI_Access::tree_node() );
  }

  // This method takes an argument-index as unsigned int and returns the correspondent argument-clang::Expr-pointer.
  static const clang::Expr* get_argument( const clang::Expr *node, const unsigned int index ) {
    if( const clang::CallExpr *ce = clang::dyn_cast<clang::CallExpr>( node ) )
      return ce->getArg(index);
    else {
      // Valid index?
      assert( index < get_argument_count( node ) );
      if( const clang::UnaryOperator *uo = llvm::dyn_cast<clang::UnaryOperator>( node ) ) {
        if( uo->isPostfix() && index == 1 )
          return 0;
        else
          return uo->getSubExpr();
      }
      else if( const clang::BinaryOperator * bo = llvm::dyn_cast<clang::BinaryOperator>( node ) )
        return index == 0 ? bo->getLHS() : bo->getRHS();
      else if( const clang::ArraySubscriptExpr *ase = llvm::dyn_cast<clang::ArraySubscriptExpr>( node ) )
        return ( index == 0 ) ? ase->getLHS() : ase->getRHS();
      else if( const clang::ConditionalOperator *co = llvm::dyn_cast<clang::ConditionalOperator>( node ) )
        return ( index == 0 ) ? co->getCond() : ( ( index == 1 ) ? co->getTrueExpr() : co->getFalseExpr() );

      assert( false && "Unknown expr type" );
      return 0; // dummy
    }
  }

  // This method takes an argument-index as unsigned int and returns the correspondent argument-clang::Expr-pointer.
  const clang::Expr* get_argument(const unsigned int index) const {
    return get_argument( TI_Access::tree_node(), index );
  }

  bool is_index_expr () const {
    if( llvm::isa<clang::ArraySubscriptExpr>( TI_Access::tree_node() ) )
      return true; //return true for Array-Subscript

    clang::CXXOperatorCallExpr *ce = clang::dyn_cast<clang::CXXOperatorCallExpr> (TI_Access::tree_node());
    return (ce && ce->getNumArgs () == 2 && _called_func->getNameAsString () == "operator[]");
  }

  const WeavePos &index_open_before_pos (WeaverBase &wb) {
    assert (is_index_expr ());
    return get_pos_after_token(get_argument(0)->getLocEnd(), wb, WeavePos::WP_BEFORE);
  }

  const WeavePos &index_open_after_pos (WeaverBase &wb) {
    assert (is_index_expr ());
    return wb.weave_pos(get_argument(1)->getLocStart (), WeavePos::WP_AFTER);
  }

  const WeavePos &index_close_before_pos (WeaverBase &wb) {
    assert (is_index_expr ());
    return get_pos_after_token(get_argument(1)->getLocEnd (), wb, WeavePos::WP_BEFORE);
  }

  const WeavePos &index_close_after_pos (WeaverBase &wb) {
    assert (is_index_expr ());
    return get_pos_after_token(TI_Access::tree_node()->getLocEnd (), wb, WeavePos::WP_AFTER);
  }

  // The following methods return the corresponding weave-positions of ternary operators
  const WeavePos& ternary_op_first_delim_before_pos(WeaverBase &wb) {
    assert (is_ternary_expr());
    return get_pos_after_token(get_argument(0)->getLocEnd(), wb, WeavePos::WP_BEFORE);
  }
  const WeavePos& ternary_op_first_delim_after_pos(WeaverBase &wb) {
    assert (is_ternary_expr());
    return wb.weave_pos(get_argument(1)->getLocStart(), WeavePos::WP_AFTER);
  }
  const WeavePos& ternary_op_second_delim_before_pos(WeaverBase &wb) {
    assert (is_ternary_expr());
    return get_pos_after_token(get_argument(1)->getLocEnd(), wb, WeavePos::WP_BEFORE);
  }
  const WeavePos& ternary_op_second_delim_after_pos(WeaverBase &wb) {
    assert (is_ternary_expr());
    return wb.weave_pos(get_argument(2)->getLocStart(), WeavePos::WP_AFTER);
  }

  const WeavePos &args_open_before_pos (WeaverBase &wb) {
    assert( ! is_builtin_operator() );
    if (is_call_op ())
      return wb.weave_pos (static_cast<clang::CallExpr*>(TI_Access::tree_node())->getCallee ()->getLocStart (), WeavePos::WP_BEFORE);
    else
      return get_pos_after_token(static_cast<clang::CallExpr*>(TI_Access::tree_node())->getCallee ()->getLocEnd (), wb, WeavePos::WP_BEFORE);
  }

  const WeavePos &args_open_after_pos (WeaverBase &wb) {
    assert( ! is_builtin_operator() );
    if (is_call_op ()) {
      if (call_args () > 1)
        return wb.weave_pos (get_argument (1)->getLocStart (), WeavePos::WP_AFTER);
      else
        return wb.weave_pos (TI_Access::tree_node()->getLocEnd (), WeavePos::WP_AFTER);
    }
    else {
      if (call_args () > 0)
        return wb.weave_pos (get_argument (0)->getLocStart (), WeavePos::WP_AFTER);
      else
        return wb.weave_pos (TI_Access::tree_node()->getLocEnd (), WeavePos::WP_AFTER);
    }
  }

  const WeavePos &args_close_before_pos (WeaverBase &wb) {
    return wb.weave_pos(TI_Access::tree_node()->getLocEnd (), WeavePos::WP_BEFORE);
  }

  const WeavePos &args_close_after_pos (WeaverBase &wb) {
    return get_pos_after_token(TI_Access::tree_node()->getLocEnd (), wb, WeavePos::WP_AFTER);
  }

  const WeavePos &callee_before_pos (WeaverBase &wb) {
    assert( ! is_builtin_operator() );
    if (has_target_expr()) {
      clang::Expr *callee = static_cast<clang::CallExpr*>(TI_Access::tree_node())->getCallee ();
      while (clang::dyn_cast<clang::ImplicitCastExpr> (callee))
        callee = clang::dyn_cast<clang::ImplicitCastExpr> (callee)->getSubExpr ();
      while (clang::dyn_cast<clang::ParenExpr> (callee))
        callee = clang::dyn_cast<clang::ParenExpr> (callee)->getSubExpr ();
      clang::MemberExpr *me = clang::dyn_cast<clang::MemberExpr> (callee);
      assert (me);
      return get_pos_after_token(me->getBase ()->getLocEnd (), wb, WeavePos::WP_BEFORE);
    }
    else {
      if (is_call_op ())
        return wb.weave_pos (TI_Access::tree_node()->getLocStart (), WeavePos::WP_BEFORE);
      else
        return wb.weave_pos (static_cast<clang::CallExpr*>(TI_Access::tree_node())->getCallee ()->getLocStart (), WeavePos::WP_BEFORE);
    }
  }

  const WeavePos &callee_after_pos (WeaverBase &wb) {
    assert( ! is_builtin_operator() );
    if (is_call_op ())
      return wb.weave_pos (static_cast<clang::CallExpr*>(TI_Access::tree_node())->getCallee ()->getLocStart (), WeavePos::WP_AFTER);
    else {
      clang::Expr *callee = static_cast<clang::CallExpr*>(TI_Access::tree_node())->getCallee ();
      while (clang::dyn_cast<clang::ParenExpr> (callee))
        callee = clang::dyn_cast<clang::ParenExpr> (callee)->getSubExpr ();
      return get_pos_after_token(callee->getLocEnd (), wb, WeavePos::WP_AFTER);
    }
  }

  // return the number of arguments, not including the object in case of
  // member function calls and not including implicitly passed default arguments
  unsigned call_args () const {
    assert( ! is_builtin_operator() && "internal error: method \"call_args\" is not implemented for built-in operators." );
    unsigned args = 0;
    clang::CallExpr* call_node = static_cast<clang::CallExpr*>( TI_Access::tree_node() );
    while (args < call_node->getNumArgs ()) {
      if( clang::dyn_cast<clang::CXXDefaultArgExpr>( call_node->getArg( args ) ) )
        break;
      args++;
    }
    return args;
  }

  virtual bool has_result () const {
    if( is_builtin_operator() ) {
      return true; // As per C++-standard chapter 13.6 all built-in operators have a result.
    }
#if CLANG_VERSION_MAJOR == 3 && CLANG_VERSION_MINOR == 4 && !defined(CLANG_VERSION_PATCHLEVEL)
    return !_called_func->getResultType()->isVoidType ();
#else // C++ 11 interface
    return !_called_func->getReturnType()->isVoidType ();
#endif
  }

  virtual clang::QualType result_type() const {
    if( is_builtin_operator() ) {
      clang::Expr *node = TI_Access::tree_node();
      clang::ASTContext& ctx = origin()->getASTContext();
      if( node->isLValue() )
        return ctx.getLValueReferenceType( node->getType() );
      else if( node->isXValue() )
        return ctx.getRValueReferenceType( node->getType() );
      else
        return node->getType();
    }
    else
#if CLANG_VERSION_MAJOR == 3 && CLANG_VERSION_MINOR == 4 && !defined(CLANG_VERSION_PATCHLEVEL)
      return _called_func->getResultType ();
#else // C++ 11 interface
    return _called_func->getReturnType ();
#endif
  }

  // target type (for the JoinPoint-API)
  virtual const clang::RecordDecl *target_class () const {
    const clang::RecordDecl *result = 0;

    const clang::CXXMethodDecl *md = clang::dyn_cast_or_null<clang::CXXMethodDecl> (_called_func);
    if( has_target_expr() ) {
      const clang::Type *type = target_expr()->getType ().getTypePtr ();
      result = type->getPointeeCXXRecordDecl();
      if (!result)
        result = type->getAsCXXRecordDecl ();
    }
    else if (md) {
      result = md->getParent();
      if (clang::CXXMethodDecl *caller = llvm::dyn_cast<clang::CXXMethodDecl>(_caller_obj)) {
        if (!md->isStatic())
          result = caller->getParent ();
      }
    }
    return result;
  }

  bool is_call_op () const {
    return (clang::dyn_cast<clang::CXXOperatorCallExpr>(TI_Access::tree_node()) &&
        _called_func->getNameAsString () == "operator()");
  }

  // the target object of the call or NULL
  virtual clang::Expr *find_target_expr() const {

    // check if this call has a target object
    clang::CXXMethodDecl *md = clang::dyn_cast_or_null<clang::CXXMethodDecl> (_called_func);
    if (!md /* || md->isStatic ()*/)
      return 0;

    clang::Expr *result = 0;

    // an ordinary member function call, e.g. foo->bar()
    clang::CXXMemberCallExpr *mce =
        clang::dyn_cast<clang::CXXMemberCallExpr> (TI_Access::tree_node());
    // .. or an operator call, e.g. !foo or foo+bar
    clang::CXXOperatorCallExpr *oce =
        clang::dyn_cast<clang::CXXOperatorCallExpr> (TI_Access::tree_node());
    if (mce)
      result = mce->getImplicitObjectArgument ();
    else if (oce && md->getParent ())
      result = oce->getArg (0);
    else {
      // it might still be a static member function call with unused target expr,
      // e.g. foo->static_bar()
      const clang::Expr *callee = static_cast<clang::CallExpr*>(TI_Access::tree_node())->getCallee();
      if (clang::dyn_cast<clang::ImplicitCastExpr> (callee)) {
        callee = clang::dyn_cast<clang::ImplicitCastExpr> (callee)->getSubExpr ();
        if (clang::dyn_cast<clang::MemberExpr> (callee)) {
          result = clang::dyn_cast<clang::MemberExpr> (callee)->getBase();
        }
      }
    }

    // TODO: check if implicit calls are handled correctly here
    return result;
  }
  
  // checks whether the call uses explicit template parameters
  bool has_explicit_template_params () const {
    if( clang::DeclRefExpr *dre = llvm::dyn_cast<clang::DeclRefExpr>( ref_node() ) )
      return dre->hasExplicitTemplateArgs();
    else if( clang::MemberExpr *me = llvm::dyn_cast<clang::MemberExpr>( ref_node() ) )
      return me->hasExplicitTemplateArgs();
    else
      return false;
  }

  const clang::TemplateArgumentLoc *get_explicit_template_params() const {
    if( clang::DeclRefExpr *dre = llvm::dyn_cast<clang::DeclRefExpr>( ref_node() ) )
      return dre->getTemplateArgs();
    else if( clang::MemberExpr *me = llvm::dyn_cast<clang::MemberExpr>( ref_node() ) )
      return me->getTemplateArgs();
    else
      return 0;
  }

  unsigned int num_explicit_template_params() const {
    if( clang::DeclRefExpr *dre = llvm::dyn_cast<clang::DeclRefExpr>( ref_node() ) )
      return dre->getNumTemplateArgs();
    else if( clang::MemberExpr *me = llvm::dyn_cast<clang::MemberExpr>( ref_node() ) )
      return me->getNumTemplateArgs();
    else
      return 0;
  }
};

class TI_VariableAccess : public TI_Access {
public:
  TI_VariableAccess() {}

  // unprotect function by forwarding (we need no special sig)
  void entity( clang::DeclaratorDecl *v ) { TI_Access::entity( v ); }
  void variable( clang::DeclaratorDecl *v ) { entity( v ); }
  clang::DeclaratorDecl *variable() const { return TI_Access::entity(); }

  bool entity_is_const() const {
    clang::DeclaratorDecl *var = variable();
    return var && var->getType().isConstQualified();
  }

  // unprotect function by forwarding (we need no special sig)
  void tree_node( clang::Expr* n ) { TI_Access::tree_node( n ); }

  // target type (for the JoinPoint-API)
  virtual const clang::RecordDecl *target_class () const {
    const clang::RecordDecl *result = 0;

    if( target_is_implicit() ) {
      clang::CXXMethodDecl *origin = llvm::dyn_cast<clang::CXXMethodDecl>( this->origin() );
      assert( origin );

      result = origin->getParent();
    }
    else if( has_target_expr() ) {
      const clang::Type *type = target_expr()->getType ().getTypePtr();
      result = type->getPointeeCXXRecordDecl();
      if( !result )
        result = type->getAsCXXRecordDecl();
    }
    /*if( const clang::FieldDecl *fd = llvm::dyn_cast<clang::FieldDecl>( variable() ) )
      result = fd->getParent();*/
    else if( const clang::VarDecl *vd = llvm::dyn_cast<clang::VarDecl>( variable() ) ) {
      if( vd->isStaticDataMember() ) {
        const clang::DeclContext *dc = vd->getDeclContext();
	assert( dc->isRecord() );

        result = llvm::dyn_cast<clang::RecordDecl>( dc );
      }
      else
        result = 0;
    }

    return result;
  }
};

class TI_Get : public TI_VariableAccess {
public:
  TI_Get() {}

  static const TI_Get *of( const ACM_Get &loc ) {
    return static_cast<TI_Get *>(loc.transform_info());
  }

  virtual bool has_result() const {
    return true;
  }

  virtual clang::QualType result_type() const {
    return TI_Access::entity()->getType();
  }
};

class TI_Set : public TI_VariableAccess {
public:
  TI_Set() {}

  static const TI_Set *of( const ACM_Set &loc ) {
    return static_cast<TI_Set *>(loc.transform_info());
  }

  virtual std::string arg_type_string (unsigned no) const {
    assert( no == 0 );

    clang::ASTContext& ctx = origin()->getASTContext();
    clang::PrintingPolicy pp( ctx.getPrintingPolicy() );
    pp.SuppressUnwrittenScope = true; // Suppress <anonymous>

    return fix_types_in_signature( TI_Access::entity()->getType().getCanonicalType().getAsString(pp) );
  }

  virtual bool has_result() const {
    return false;
  }

  virtual clang::QualType result_type() const {
    clang::ASTContext& ctx = origin()->getASTContext();
    return ctx.VoidTy;
  }
};

class TI_Ref : public TI_VariableAccess {
public:
  TI_Ref() {}

  static const TI_Ref *of( const ACM_Ref &loc ) {
    return static_cast<TI_Ref *>(loc.transform_info());
  }

  virtual bool has_result() const {
    return true;
  }

  virtual clang::QualType result_type() const {
    clang::ASTContext& ctx = origin()->getASTContext();

    clang::QualType result = TI_Access::tree_node()->getType();
    if( TI_Access::tree_node()->isLValue() )
      result = ctx.getLValueReferenceType( result );
    return result;
  }

  bool result_is_ptr() const {
    return result_type().getTypePtr()->isPointerType();
  }

  const bool is_explicit_operator() const {
    if( clang::UnaryOperator *uo = clang::dyn_cast<clang::UnaryOperator>( TI_Access::tree_node() ) )
      if( uo->getOpcode() == clang::UO_AddrOf )
        return true;

    return false;
  }

  const WeavePos &op_before_pos( WeaverBase &wb ) const {
    assert( clang::isa<clang::UnaryOperator>( TI_Access::tree_node() ) );
    return wb.weave_pos( clang::dyn_cast<clang::UnaryOperator>( TI_Access::tree_node() )->getOperatorLoc(), WeavePos::WP_BEFORE );
  }

  const WeavePos &op_after_pos( WeaverBase &wb ) const {
    assert( clang::isa<clang::UnaryOperator>( TI_Access::tree_node() ) );
    return get_pos_after_token( clang::dyn_cast<clang::UnaryOperator>( TI_Access::tree_node() )->getOperatorLoc(), wb );
  }
};

class TI_RefAccess : public TI_Access {
public:
  TI_RefAccess() {}

  bool entity_is_const() const {
    return entity_type().isConstQualified();
  }

  // unprotect function by forwarding (we need no special sig)
  void tree_node( clang::Expr* n ) { TI_Access::tree_node( n ); }

  // target type (for the JoinPoint-API)
  virtual const clang::RecordDecl *target_class () const {
    return 0; // refs dont have a target type currently, it has to be recovered from runtime info
  }

  // common function to determine entity type
  clang::QualType entity_type() const {
    return TI_Access::ref_node()->getType().getNonReferenceType();
  }

  // entity type (for the JoinPoint-API)
  virtual std::string entity_type_string() const {
    clang::ASTContext& ctx = origin()->getASTContext();
    clang::PrintingPolicy pp( ctx.getPrintingPolicy() );
    pp.SuppressUnwrittenScope = true; // Suppress <anonymous>

    return fix_types_in_signature( entity_type().getAsString( pp ) );
  }
};

class TI_GetRef : public TI_RefAccess {
public:
  TI_GetRef() {}

  static const TI_GetRef *of( const ACM_GetRef &loc ) {
    return static_cast<TI_GetRef *>(loc.transform_info());
  }

  virtual bool has_result() const {
    return true;
  }

  virtual clang::QualType result_type() const {
    return entity_type();
  }
};

class TI_SetRef : public TI_RefAccess {
public:
  TI_SetRef() {}

  static const TI_SetRef *of( const ACM_SetRef &loc ) {
    return static_cast<TI_SetRef *>(loc.transform_info());
  }

  virtual std::string arg_type_string (unsigned no) const {
    assert( no == 0 );

    clang::ASTContext& ctx = origin()->getASTContext();
    clang::PrintingPolicy pp( ctx.getPrintingPolicy() );
    pp.SuppressUnwrittenScope = true; // Suppress <anonymous>

    return fix_types_in_signature( entity_type().getAsString(pp) );
  }

  virtual bool has_result() const {
    return false;
  }

  virtual clang::QualType result_type() const {
    clang::ASTContext& ctx = origin()->getASTContext();
    return ctx.VoidTy;
  }
};

class TI_Construction : public TI_Code {
  clang::FunctionDecl *_decl;
  clang::CXXRecordDecl *_that_decl;

public:
  TI_Construction () : _decl (0), _that_decl (0) {}

  void decl (clang::FunctionDecl *f) { _decl = f; }
  virtual clang::Decl *decl () const { return _that_decl; }
  void that_decl (clang::CXXRecordDecl *r) { _that_decl = r; }
  virtual clang::CXXRecordDecl *that_decl () const { return _that_decl; }
  
  // that type (for the JoinPoint-API)
  virtual std::string that_type_string() const {
    return get_type_string(_that_decl);
  }

  // target type (for the JoinPoint-API)
  virtual std::string target_type_string() const {
    return get_type_string(_that_decl);
  }

  virtual std::string arg_type_string (unsigned no) const {
    if (_decl) { // user-defined constructor
      return get_type_string (_decl->getParamDecl (no));
    }
    else { // built-in constructor
      assert (no == 0); // may have at most one argument
      string result;
      if (_that_decl->hasCopyConstructorWithConstParam ())
        result += "const ";
      result += get_type_string(_that_decl);
      result += "&";
      return result;
    }
  }

  // entity type (for the JoinPoint-API)
  virtual std::string entity_type_string() const {
    return format_type( _decl );
  }
};

class TI_Destruction : public TI_Code {
  clang::FunctionDecl *_decl;
  clang::CXXRecordDecl *_that_decl;

public:
  TI_Destruction () : _decl (0), _that_decl (0) {}

  void decl (clang::FunctionDecl *f) { _decl = f; }
  virtual clang::Decl *decl () const { return _that_decl; }
  void that_decl (clang::CXXRecordDecl *r) { _that_decl = r; }
  virtual clang::CXXRecordDecl *that_decl () const { return _that_decl; }

  // that type (for the JoinPoint-API)
  virtual std::string that_type_string() const {
    return get_type_string(_that_decl);
  }

  // target type (for the JoinPoint-API)
  virtual std::string target_type_string() const {
    return get_type_string(_that_decl);
  }

  // entity type (for the JoinPoint-API)
  virtual std::string entity_type_string() const {
    return format_type( _decl );
  }
};

class TI_AdviceCode : public TransformInfo {
  clang::FunctionDecl *_decl;
  ThisJoinPoint _this_join_point;
  
public:
  TI_AdviceCode () : _decl (0) {}
  
  void decl(clang::FunctionDecl *f) { _decl = f; }
  virtual clang::FunctionDecl *decl () const { return _decl; }

  clang::DeclContext *Scope () const {
    return _decl ? _decl->getParent() : 0;
  }
  string name () const {
    return _decl ? _decl->getNameAsString() : "";
  }
  string qual_name () {
    return _decl ? _decl->getQualifiedNameAsString() : "";
  }
  
  ThisJoinPoint &this_join_point () { return _this_join_point; }
  const ThisJoinPoint &this_join_point () const { return _this_join_point; }

  static TI_AdviceCode *of (const ACM_AdviceCode &loc) {
    return static_cast<TI_AdviceCode*>(loc.transform_info ());
  }
};

class TI_Introduction : public TransformInfo {
public:
  virtual clang::Decl *decl () const { return 0; }

  static TI_Introduction *of (const ACM_Introduction &loc) {
    return static_cast<TI_Introduction*>(loc.transform_info ());
  }
};

class TI_Order : public TransformInfo {
public:
  virtual clang::Decl *decl () const { return 0; }

  static TI_Order *of (const ACM_Order &loc) {
    return static_cast<TI_Order*>(loc.transform_info ());
  }
};

class TI_Pointcut : public TransformInfo {
  clang::FunctionDecl *_decl;
  int _phase;
  PointCutExpr *_pce;
  clang::SourceLocation _loc;
public:
  TI_Pointcut () : _decl(0), _phase (0), _pce (0) {}
  ~TI_Pointcut () { PointCutExpr::destroy(_pce); }

  void decl (clang::FunctionDecl *c) { _decl = c; }
  virtual clang::Decl *decl () const { return _decl; }

  void phase (int p) { _phase = p; }
  int phase () const { return _phase; }
  void set_pce (PointCutExpr *pce) { _pce = pce; }
  PointCutExpr *get_pce () const { return _pce; }
  void set_location (clang::SourceLocation loc) { _loc = loc; }
  clang::SourceLocation get_location () const { return _loc; }

  static TI_Pointcut *of (const ACM_Pointcut &loc) {
    return static_cast<TI_Pointcut*>(loc.transform_info ());
  }
};

class TI_ClassSlice : public TransformInfo {
public:
  struct SliceBody {
    enum InsertType {
      TARGET_NAME,
      TARGET_QUAL_NAME,
      JP_NAME
    };
    std::string text;
    std::vector<std::pair<size_t, InsertType> > positions;
  };

private:
  ACFileID _slice_unit;

  // new phase 1 implementation:
  SliceBody _tokens; // class slice body
  std::string _base_intro;
  bool _has_base_intro, _has_member_intro;
  std::list<SliceBody> _non_inline_members; // members defined outside of body
  std::vector<ACFileID> _non_inline_member_units; // corresponding source units

public:

  TI_ClassSlice () : _slice_unit (0), _has_base_intro (false),
                     _has_member_intro (false) {}

  // new phase 1 implementation:
  void set_tokens (const SliceBody &body, const std::string &base_intro,
                   bool has_base_intro, bool has_member_intro) {
    _tokens = body;
    _base_intro = base_intro;
    _has_base_intro = has_base_intro;
    _has_member_intro = has_member_intro;
  }
  const SliceBody &get_tokens () const { return _tokens; }
  std::list<SliceBody> &non_inline_members () { return _non_inline_members; }
  std::vector<ACFileID> &non_inline_member_units () { return _non_inline_member_units; }
  void analyze_tokens (bool &has_base_intro, bool &has_member_intro) {
    has_base_intro = _has_base_intro;
    has_member_intro = _has_member_intro;
  }
  const std::string &base_intro () const {
    return _base_intro;
  }
  // end - new phase 1 implementation

  virtual clang::Decl *decl () const { return 0; }
  void slice_unit (ACFileID su) { _slice_unit = su; }
  ACFileID slice_unit () const { return _slice_unit; }

  static TI_ClassSlice *of (const ACM_ClassSlice &loc) {
    return static_cast<TI_ClassSlice*>(loc.transform_info ());
  }
};

#endif // __ClangTransformInfo_h__
