/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "globals.h"
#include "din.h"
#include "command.h"
#include "console.h"
#include "math.h"
#include "delay.h"
#include "scalelist.h"
#include "font.h"
#include "glyph.h"
#include "chrono.h"
#include "utils.h"
#include "ui_list.h"
#include "curve_editor.h"
#include "bot.h"
#include "tokenizer.h"
#include "tcl_interp.h"
#include "chrono.h"
#include "sine_mixer.h"
#include "scale_info.h"
#include "keyboard_keyboard.h"
#include "main.h"
#include "fft.h"

#include <string>
using namespace std;


#define buf cmdlst.buf
#define result cmdlst.result

extern console cons;
extern cmdlist cmdlst;
extern tcl_interp interpreter;
extern keyboard_keyboard keybd2;
extern ui_list uis;
extern din din0;
extern ui_list uis;

extern string SCALE;
extern string TUNING;

extern float DELTA_BPM;
extern int DRONE_HANDLE_SIZE;
extern int TRAIL_LENGTH;
extern int NUM_OCTAVES;

extern float WAVE_VOLUME;

extern float NOTE_VOLUME;
extern float ATTACK_TIME;

extern float DECAY_TIME;

extern float DELTA_TIME;

extern string INSTRUMENT;

extern float PITCH_BEND;

extern int NUM_NOTES;
extern char* WESTERN_FLAT [];
extern char* WESTERN_SHARP [];
extern float WIKIPEDIA_KEY_FREQUENCIES [];


template <class T> void get_data (const string& s, vector<T>& data) {

  tokenizer tz (s);
  data.clear ();
  stringstream ss;
  string t;
  T tt;
  while (1) {
    ss.clear (); tz >> t;
    if (t == "") break;
    ss << t; ss >> tt; data.push_back (tt);
  }
}

void cmdlist::add (command* cmd) {
  cmds.push_back (cmd);
  ncmds = cmds.size ();
}

int set_var::set_scroll (string& varn, tokenizer& tz) {
  while (1) {
    tz >> varn;
    if (varn == "r" || varn == "rate") {
      tz >> d->dinfo.scroll.rate;
      d->dinfo.scroll.calc_repeat_time ();
    } else if (varn == "x") {
      tz >> d->dinfo.scroll.dx;
    } else if (varn == "y") {
      tz >> d->dinfo.scroll.dy;
    } else {
      if (varn != "") {
        result = varn + ": bad scroll param";
        return 0;
      } else return 1;
    }
  }
}

int set_var::set_zoom_pan (int& rate, float& amount, string& varn, tokenizer& tz) {
  while (1) {
    tz >> varn;
    if (varn == "r" || varn == "rate") {
      tz >> rate;
      window::calc_repeats ();
    } else if (varn == "a" || varn == "amount") {
      tz >> amount;
      window::calc_repeats ();
      extern ui_list uis;
      if (uis.crved) uis.crved->win.calc_panxy ();
    } else {
      if (varn != "") {
        result = varn + ": bad zoom/pan param";
        return 0;
      } else return 1;
    }
  }
}

bool set_var::operator() (tokenizer& tz) {

  string varn;
  int extract = 1;
  while (1) {
    if (extract) {
      tz >> varn;
      result = "";
    } else extract = 1;
    if (varn == "fmd" || varn == "fm_depth") {
      tz >> d->fm_depth;
      hz2step (d->fm_depth, d->fm_step);
    } else if (varn == "amd" || varn == "am_depth") {
      tz >> d->am_depth;
    } else if (varn == "pb" || varn == "pitch_bend") {
      tz >> PITCH_BEND;
      uis.lfs_pitch_bend.set_val (PITCH_BEND);
    } else if (varn == "dmv" || varn == "drone_master_volume") {
      float f; tz >> f;
      d->update_drone_master_volume (f - d->drone_master_volume);
    } else if (varn == "at" || varn == "attack_time") {
      float dt; tz >> dt;
      if (dt <= 0) {
        cons << console::red << "bad attack time" << eol;
        return false;
      } else ATTACK_TIME = dt;
      uis.lfs_attack_time.set_val (ATTACK_TIME);
    } else if (varn == "dt" || varn == "decay_time") {
      float dt; tz >> dt;
      if (dt <= 0) {
        cons << console::red << "bad decay time" << eol;
        return false;
      } else DECAY_TIME = dt;
      uis.lfs_decay_time.set_val (DECAY_TIME);
    } else if (varn == "det" || varn == "delta_time") {
      float dt; tz >> dt;
      if (dt == 0) {
        cons << console::red << "bad delta time" << eol;
        return false;
      } else DELTA_TIME = dt;
    } else if (varn == "su" || varn == "sustain") {
      float g; tz >> g; _gotog.set (g);
    } else if (varn == "wv" || varn == "wave_volume") {
      tz >> WAVE_VOLUME;
    } else if (varn == "nv" || varn == "note_volume") {
      float nv; tz >> nv;
      if (nv > 0) NOTE_VOLUME = nv; else cons << console::red << nv << ": bad note volume" << eol;
      uis.lfs_note_volume.set_val (NOTE_VOLUME);
      keybd2.calc_visual_params ();
    } else if (varn == "dbpm" || varn == "delta_bpm") {
      tz >> DELTA_BPM;
      DELTA_BPM = fabs (DELTA_BPM);
    } else if (varn == "ins" || varn == "instrument") {
      tz >> INSTRUMENT;
    } else if (varn == "hz") {
      curve_editor* crved = uis.crved;
      if (crved) {
        tz >> crved->hz;
        crved->cs.set (crved->hz);
        crved->cs.render (crved->curveinfo[0].curve);
      }
    } else if (varn == "num_sine_samples" || varn == "nss") {
      int nss; tz >> nss;
      sine_mixer::set_sine_samples (nss);
    } else if (varn == "tu" || varn == "tuning") {
      extern load_globals lg;
      string name; tz >> name;
      string fname = name + ".tuning";
      if (lg.load_intervals (fname)) {
        TUNING = name;
        result = name;
        d->restore_all_notes ();
        keybd2.setup_notes ();
        keybd2.setup_midi_notes ();
        keybd2.calc_visual_params ();

        cons ("list2var [get-intervals]");
      } else {
        result = "bad tuning";
        return false;
      }
    } else if (varn == "snap_drones" || varn == "sd") {
      tz >> d->snap_drones;
    } else if (varn == "sc" || varn == "scroll") {
      extract = set_scroll (varn, tz);
    } else if (varn == "zoom" || varn == "z") {
      extract = set_zoom_pan (window::ZOOM_RATE, window::ZOOM_AMOUNT, varn, tz);
    } else if (varn == "pan" || varn == "p") {
      extract = set_zoom_pan (window::PAN_RATE, window::PAN_AMOUNT, varn, tz);
    } else if (varn == "fps") {
      extern int FPS; tz >> FPS; if (FPS <= 0) FPS = 100;
      extern double FPS_T; FPS_T = 1.0 / FPS;
    } else if (varn == "usleep" || varn == "us") {
      extern int USLEEP; tz >> USLEEP; if (USLEEP < 0) USLEEP = 0;
    } else if (varn == "show_cursor_info" || varn == "sci") {
      int v; tz >> v;
      d->show_cursor_info = v;
    } else if (varn == "jog" || varn == "j") {
      tz >> phrasor::JOG;
    } else if (varn == "ps1" || varn == "prompt") {
      tz >> cons.ps1;
      cons.calc_visual_params ();
    } else if (varn == "dhs" || varn == "drone_handle_size") {
      tz >> DRONE_HANDLE_SIZE;
      d->update_drone_x (0, d->last_range);
    } else if (varn == "tl" || varn == "trail_length") {
      tz >> TRAIL_LENGTH;
      if (TRAIL_LENGTH < 1) TRAIL_LENGTH = 1;
    } else if (varn == "auto_connect_outputs" || varn == "aco") {
      tz >> audio_out::AUTO_CONNECT_OUTPUTS;
    } else {
      if (varn != "") {
        result = varn + ": not found.";
        return false;
      } else break;
    }
  }
  return true;
}

bool get_var::operator() (tokenizer& tz) {
  stringstream ss;
  string varn;
  while (1) {
    tz >> varn;
    if (varn == "fmd" || varn == "fm_depth") {
      ss << d->fm_depth << ' ';
    } else if (varn == "amd" || varn == "am_depth") {
      ss << d->am_depth << ' ';
    } else if (varn == "pb" || varn == "pitch_bend") {
      ss << PITCH_BEND << ' ';
    } else if (varn == "wv" || varn == "wave_volume") {
      ss << WAVE_VOLUME << ' ';
    } else if (varn == "nv" || varn == "note_volume") {
      ss << NOTE_VOLUME << ' ';
    } else if (varn == "at" || varn == "attack_time") {
      ss << ATTACK_TIME << ' ';
    } else if (varn == "dt" || varn == "decay_time") {
      ss << DECAY_TIME << ' ';
    } else if (varn == "det" || varn == "delta_time") {
      ss << DELTA_TIME << ' ';
    } else if (varn == "su" || varn == "sustain") {
      ss << _gotog.g << ' ';
    } else if (varn == "dbpm" || varn == "delta_bpm") {
      ss << DELTA_BPM << ' ';
    } else if (varn == "num_octaves" || varn == "no") {
      ss << NUM_OCTAVES << ' ';
    } else if (varn == "dmv" || varn == "drone_master_volume") {
      ss << d->drone_master_volume << ' ';
    } else if (varn == "hz") {
      curve_editor* crved = uis.crved;
      if (crved) {
        ss << crved->hz << ' ';
      } else ss << -1 << ' ';
    } else if (varn == "ins" || varn == "instrument") {
      ss << INSTRUMENT << ' ';
    } else if (varn == "sd" || varn == "snap_drones") {
      sprintf (buf, "%d", d->snap_drones);
      ss << d->snap_drones << ' ';
    } else if (varn == "tu" || varn == "tuning") {
      ss << TUNING << ' ';
    } else if (varn == "scale") {
      ss << SCALE << ' ';
    } else if (varn == "sc" || varn == "scroll") {
      sprintf (buf, "rate %d x %d y %d ", d->dinfo.scroll.rate, d->dinfo.scroll.dx, d->dinfo.scroll.dy);
      ss << buf;
    } else if (varn == "zoom") {
      ss << "rate " << window::ZOOM_RATE << " amount " << window::ZOOM_AMOUNT << ' ';
    } else if (varn == "pan") {
      ss << "rate " << window::PAN_RATE << " amount " << window::PAN_AMOUNT << ' ';
    } else if (varn == "fps") {
      extern int FPS; ss << FPS << ' ';
    } else if (varn == "usleep" || varn == "us") {
      extern int USLEEP; ss << USLEEP << ' ';
    } else if (varn == "num_sine_samples" || varn == "nss") {
      ss << sine_mixer::NUM_SINE_SAMPLES << ' ';
    } else if (varn == "show_cursor_info" || varn == "sci") {
      ss << d->show_cursor_info << ' ';
    } else if (varn == "jog" || varn == "j") {
      ss << phrasor::JOG << ' ';
    } else if (varn == "dhs" || varn == "drone_handle_size") {
      ss << DRONE_HANDLE_SIZE << ' ';
    } else if (varn == "tl" || varn == "trail_length") {
      ss << TRAIL_LENGTH << ' ';
    } else if (varn == "auto_connect_outputs" || varn == "aco") {
      ss << audio_out::AUTO_CONNECT_OUTPUTS << ' ';
    } else {
      if (varn != "") {
        result = varn + ": not found.";
        return false;
      } else break;
    }
  }

  result = ss.str().substr (0, ss.str().length() - 1);
  return true;

}

bool set_delay::operator() (tokenizer& tz) {

  string name;
  float value;

  delay* d[2] = {left, right};
  delay** pd = d;
  int n = -1;

  tz >> name >> value;

  if (name == "a" || name == "all") n = 2;
  else if (name == "l" || name == "left") n = 1;
  else if (name == "r" || name == "right") {
    pd++;
    n = 1;
  } else {
    result = "bad delay name";
    return false;
  }

  for (int i = 0; i < n; ++i, pd++) {
    (*pd)->set (value);
  }

  return true;

}

bool get_delay::operator() (tokenizer& tz) {

  delay* d;
  string s; tz >> s;
  if (s == "left" || s == "l") d = left;
  else if (s == "right" || s == "r") d = right;
  else {
    result = "bad delay name";
    return false;
  }

  float t; d->get (t);
  sprintf (buf, "%0.2f", t);
  result = buf;

  return true;

}

int bpm_com::getid (const string& name) {

  for (int i = 0; i < NUM; ++i) {
    if (name == str[i]) {
      return i;
    }
  }
  return -1;
}

bool ls_bpm::operator() (tokenizer& tz) {
  static const char spc = ' ';
  for (int i = 0;i < NUM; ++i) {
    result = result + str[i] + spc;
  }
  return true;
}

bool set_bpm::operator() (tokenizer& tz) {

  vector<string> names;
  vector<float> bpms;
  string s;
  tz >> s; get_data (s, names);
  tz >> s; get_data (s, bpms);

  for (int i = 0, j = names.size (), k = bpms.size (), l = k - 1; i < j; ++i) {
    int id = getid (names[i]);
    if (id != -1) {
      if (i < k) bv[id]->set_bpm (bpms[i]); else if (l > -1) bv[id]->set_bpm (bpms[l]);
    }
  }

  return true;

}

bool get_bpm::operator() (tokenizer& tz) {

  vector<string> names;
  string s; tz >> s; get_data (s, names);
  stringstream ss;
  for (int i = 0, j = names.size (); i < j; ++i) {
    int id = getid (names[i]);
    if (id != -1) ss << bv[id]->get_bpm () << ' ';
  }
  result = ss.str ();
  return true;

}

bool set_beat::operator() (tokenizer& tz) {

  vector<string> names;
  vector<float> nows;
  string s;
  tz >> s; get_data (s, names);
  tz >> s; get_data (s, nows);

  for (int i = 0, j = names.size (), k = nows.size (), l = k - 1; i < j; ++i) {
    int id = getid (names[i]);
    if (id != -1) {
      if (i < k) bv[id]->now = nows[i]; else if (l > -1) bv[id]->now = nows[l];
    }
  }

  return true;

}

bool get_beat::operator() (tokenizer& tz) {

  vector<string> names;
  stringstream ss;
  string s; tz >> s; get_data (s, names);
  string what; tz >> what;

  for (int i = 0, j = names.size (); i < j; ++i) {
    int id = getid (names[i]);
    if (id != -1) {
      if (what == "") ss << bv[id]->now << ' ';
      else if (what == "first") ss << bv[id]->sol.firstx << ' ';
      else if (what == "last") ss << bv[id]->sol.lastx << ' ';
      else if (what == "all") ss << '{' << bv[id]->now << ' ' << bv[id]->sol.firstx << ' ' << bv[id]->sol.lastx << "} ";
    }
  }
  result = ss.str ();
  return true;
}

bool set_style::operator() (tokenizer& tz) {

  vector<string> names;
  vector<string> styles;
  string s;
  tz >> s; get_data (s, names);
  tz >> s; get_data (s, styles);

  for (int i = 0, j = names.size (), k = styles.size (), l = k - 1; i < j; ++i) {
    int id = getid (names[i]);
    if (id != -1) {
      string style;
      if (i < k) style = styles[i]; else if (l > -1) style = styles[l];
      xhandler *xmin = 0, *xmax = 0;
      if (style == "loop") {
        xmin = &_loopmin;
        xmax = &_loopmax;
      } else if (style == "pong") {
        xmin = &_pongmin;
        xmax = &_pongmax;
      } else {
        result = "bad style: " + style;
        return false;
      }
      beat2value* b = bv[id];
      b->xmin = xmin;
      b->xmax = xmax;
      b->style = style;
    }
  }
  return true;
}

bool get_style::operator() (tokenizer& tz) {

  vector<string> names;
  string s; tz >> s; get_data (s, names);
  stringstream ss;
  for (int i = 0, j = names.size (); i < j; ++i) {
    int id = getid (names[i]);
    if (id != -1) ss << bv[id]->style << ' ';
  }
  result = ss.str ();
  return true;

}

bool add_scale::operator() (tokenizer& tz) {
  extern scalelist scalelst;
  scalelst.add (tz);
  return true;
}

bool ls_scales::operator() (tokenizer& tz) {
  extern scalelist scalelst;
  scalelst.ls ();
  return true;
}

bool remove_scale::operator() (tokenizer& tz) {
  string nam; tz >> nam;
  if (nam != "") {
    extern scalelist scalelst;
    scalelst.remove (nam);
    return true;
  }
  result = "bad scale name";
  return false;
}

bool load_scale::operator() (tokenizer& tz) {

  string name; tz >> name;

  if (name != "") {

    extern scalelist scalelst;
    scale r;
    if (scalelst.get (name, r)) {

      if (d->drones.size ()) {
        if (d->delete_all_drones()) {
          d->save_scale ();
          d->on_delete = "load-scale " + name;
          return true;
        } else return false;
      }

      extern scale_info scaleinfo;
      scaleinfo.set_scale (name);

      d->load_scale ();

      keybd2.setup_notes ();
      keybd2.setup_midi_notes ();
      keybd2.calc_visual_params ();

      SCALE = name;

    } else {
      result = name + ": not found";
      return false;
    }

  } else {
    result = name + ": bad scale name";
    return false;
  }

  result = "scale is " + name;
  return true;
}

bool ls_notes::operator() (tokenizer& tz) {
  string name; tz >> name;
  extern scalelist scalelst;
  scale r;
  if (scalelst.get (name, r)) {
    result = r.notes;
    return true;
  } else {
    result = name + ": bad scale";
    return false;
  }
}

bool set_kern::operator() (tokenizer& tz) {

  extern font fnt;

  char a, b;
  int k;

  tz >> a >> b >> k;

  if (a == '.') {
    map<char, glyph>& m = fnt.characters;
    for (map<char, glyph>::iterator i = m.begin(), j = m.end(); i != j; ++i) fnt.kern[(*i).first][b] = k * fnt.xsize;
    fnt.modified = 1;
  } else if (b == '.') {
    map<char, glyph>& m = fnt.characters;
    for (map<char, glyph>::iterator i = m.begin(), j = m.end(); i != j; ++i) fnt.kern[a][(*i).first] = k * fnt.xsize;
    fnt.modified = 1;
  } else {
    fnt.kern[a][b] = k * fnt.xsize;
    fnt.modified = 1;
  }

  return true;

}

bool get_kern::operator() (tokenizer& tz) {

  extern font fnt;

  char a, b;
  int k;

  tz >> a >> b >> k;

  if (a == '.') {
    map<char, glyph>& m = fnt.characters;
    for (map<char, glyph>::iterator i = m.begin(), j = m.end(); i != j; ++i) {
      a = (*i).first;
      cons << a << ' ' << b << ' ' << fnt.kern[a][b] / fnt.xsize << eol;
    }
  } else if (b == '.') {
    map<char, glyph>& m = fnt.characters;
    for (map<char, glyph>::iterator i = m.begin(), j = m.end(); i != j; ++i) {
      b = (*i).first;
      cons << a << ' ' << b << ' ' << fnt.kern[a][b] / fnt.xsize << eol;
    }
  } else {
    cons << a << ' ' << b << ' ' << fnt.kern[a][b] / fnt.xsize << eol;
  }

  return true;

}

bool set_font_size::operator() (tokenizer& tz) {

  extern font fnt;

  int xsize, ysize, charspc, headroom;
  tz >> xsize >> ysize >> charspc >> headroom;

  if (xsize < 1) xsize = fnt.xsize;
  if (ysize < 1) ysize = fnt.ysize;
  if (charspc < 1) charspc = fnt.charspc;
  if (headroom < 1) headroom = fnt.headroom;

  fnt.xsize = xsize;
  fnt.ysize = ysize;
  fnt.load (fnt.fname);

  fnt.charspc = charspc;
  fnt.headroom = headroom;
  fnt.wordspc = 2 * fnt.charspc;

  cons.calc_visual_params ();

  din0.calc_range_label ();
  uis.update_widgets ();

  keybd2.calc_visual_params ();

  return true;

}

bool get_font_size::operator() (tokenizer& tz) {
  sprintf (buf, "%d %d %d %d", fnt.xsize, fnt.ysize, fnt.charspc, fnt.headroom);
  result = buf;
  return true;
}

int get_note_num (const string& s) {
  for (int i = 0; i < NUM_NOTES; ++i) {
    if (s == WESTERN_SHARP [i] || s == WESTERN_FLAT[i]) return i;
  }
  return -1;
}

bool note_distance::operator() (tokenizer& tz) {

  string note1, note2, verbose;
  tz >> note1 >> note2 >> verbose;
  if (note1.length() && note2.length()) {
    note1[0] = toupper (note1[0]);
    note2[0] = toupper (note2[0]);
  } else return false;

  static const char* dist1 [] = {
    "1", "2b", "2", "3b", "3", "4", "4#", "5", "6b", "6", "7b", "7", "8"
  };
  static const char* dist2 [] = {
    "same", "minor_second", "second", "minor_third", "third", "perfect_fourth", "diminished_fifth", "perfect_fifth", "minor_sixth", "sixth", "minor_seventh", "seventh", "octave"
  };

  int i = get_note_num (note1), j = get_note_num (note2);
  if (i == -1 || j == -1) return false; else {
    if (j <= i) j += 12;
    int ji = j - i;
    if (verbose.length()) {
      sprintf (buf, "%s %s", dist1[ji], dist2[ji]);
      result = buf;
      return true;
    } else {
      result = dist1[ji];
      return true;
    }
  }

}

bool chord::operator() (tokenizer& tz) {

  string root, type;
  tz >> root >> type;

  if (root.length()) root[0] = toupper (root[0]); else return false;
  int nn = get_note_num (root);
  if (nn == -1) {
    result = root + ": bad note";
    return false;
  }else {

    const int ntypes = 20;
    static const char* types[] = {
      "", // major
      "m", // minor
      "maj7", // major 7
      "m7", // minor 7
      "sus4", // suspended 4th
      "7sus4", // dominant 7th, suspended 4th
      "6", // major 6th
      "m6", // minor 6th
      "7", // dominant 7th
      "9", // dominant 9th
      "5", // power chord
      "69", // maj 6th, add 9
      "11", // dominant 11th
      "13", // dominant 13th
      "add9", // major add 9th
      "m9", // minor 9th
      "maj9", // major 9th
      "+", // augmented
      "07", // diminished 7th
      "0" // diminished triad
    };

    static const int steps[][7] = {
      {3, 0, 4, 7}, // num_notes, root, note1, note2....
      {3, 0, 3, 7},
      {4, 0, 4, 7, 11},
      {4, 0, 3, 7, 10},
      {3, 0, 5, 7},
      {4, 0, 5, 7, 10},
      {4, 0, 4, 7, 9},
      {4, 0, 3, 7, 9},
      {4, 0, 4, 7, 10},
      {5, 0, 4, 7, 10, 14},
      {2, 0, 7},
      {5, 0, 4, 7, 9, 14},
      {6, 0, 4, 7, 10, 14, 17},
      {6, 0, 4, 7, 10, 14, 21},
      {4, 0, 4, 7, 14},
      {5, 0, 3, 7, 10, 14},
      {5, 0, 4, 7, 11, 14},
      {3, 0, 4, 8},
      {4, 0, 3, 6, 10},
      {3, 0, 3, 6}
    };

    int t = -1;
    for (int i = 0; i < ntypes; ++i) {
      if (type == types[i]) {
        t = i; break;
      }
    }

    if (t == -1) return false; else {
      const int* ch = &steps[t][0];
      stringstream ss1, ss2;
      for (int i = 0, j = ch[0], k = 1; i < j; ++i, ++k) {
        int p = (nn + ch[k]) % 12;
        ss1 << WESTERN_SHARP [p] << ' ';
        ss2 << WESTERN_FLAT [p] << ' ';
      }
      result = ss2.str () + " or " + ss1.str ();
      return true;
    }


  }
}

void key::find_nearest_note (string& note, float& frequency, float& dist) {

  float left = WIKIPEDIA_KEY_FREQUENCIES [0] / 2048, right = 2 * left;
  float tonic = d->get_tonic ();
  if (tonic < left) return;

  while (1) {
    if ((left <= tonic) && (right > tonic)) break;
    else {
      left*=2;
      right*=2;
    }
  }

  float oct = left / WIKIPEDIA_KEY_FREQUENCIES[0];
  dist = tonic - left;
  int id = 0;
  float tone = 0;
  for (int i = 0; i < 13; ++i) {
    tone = WIKIPEDIA_KEY_FREQUENCIES[i] * oct;
    float newdist = tone - tonic; if (newdist < 0) newdist = -newdist;
    if (newdist < dist) {
      dist = newdist;
      id = i;
    }
  }

  string nn;
  if (WESTERN_FLAT[id] == WESTERN_SHARP[id]) nn = WESTERN_SHARP[id]; else nn = (string)WESTERN_FLAT[id] + "/" + (string)WESTERN_SHARP[id];
  note = nn;
  frequency = WIKIPEDIA_KEY_FREQUENCIES[id] * oct;
  dist = tonic - frequency;
  d->set_key_id (id);

}

bool key::operator() (tokenizer& tz) {

  string note ("invalid-note");
  float frequency, dist;

  string s; tz >> s;

  if (s == "") {
    // no args, just print key
    find_nearest_note (note, frequency, dist);
    sprintf (buf, "key %s @ %0.3f Hz, tonic@ %0.3f Hz & distance %0.3f Hz", note.c_str(), frequency, d->get_tonic(), dist);
    result = buf;
    return true;
  } else if (s == "value" || s == "v") {
    sprintf (buf, "%0.3f", d->get_tonic ());
    result = buf;
    return true;
  } else if (s == "note" || s == "n") {
    find_nearest_note (note, frequency, dist);
    result = note;
    return true;
  } else {

    float freq = 0;
    int oct = 0;

    float val = atof (s.c_str());
    if (val <= 0) { // a note
      if (s[0] != '.') {
        s[0] = toupper(s[0]);
        for (int i = 0; i < 12; ++i) {
          if ((s == WESTERN_SHARP[i]) || (s == WESTERN_FLAT[i])) {
            freq = WIKIPEDIA_KEY_FREQUENCIES[i];
            break;
          }
        }
      } else freq = d->get_tonic ();
    } else { // a value
      freq = val;
    }

    // octave
    tz >> s;
    oct = atoi (s.c_str());

    freq = freq * pow (2, oct);

    set_tonic (freq);
    find_nearest_note (note, freq, dist);

  }

  return false;

}

bool notation::operator () (tokenizer& tz) {

  string s; tz >> s;

  if (s == "numeric" || s == "n") d->set_notation ("numeric"); else
  if (s == "w" || s == "west" || s == "western") d->set_notation ("western");
  else result = d->notation;
  return true;

}

bool echo::operator() (tokenizer& tz) {
  into_lines (tz.cur2end().c_str(), cons);
  return true;
}

multi_curve* get_curve (const string& name) {

  extern vector<multi_curve *> curve_list;
  multi_curve* crv = 0;
  for (int i = 0, j = curve_list.size (); i < j; ++i) {
    multi_curve* mc = curve_list[i];
    if (mc->name == name) crv = mc;
  }
  return crv;

}

bool curve_value::operator() (tokenizer& tz) {

  string name; tz >> name;

  multi_curve* crv = get_curve (name);

  if (crv) {

    string what;
    int id;

    string sx, sy;
    tz >> what >> id >> sx >> sy;
    float x, y, xs = atof (sx.c_str()), ys = atof (sy.c_str());

    if (what == "v") {
      int carry_tangents = 1;
      crv->get_vertex (id, x, y);
      if (sx == ".") crv->set_vertex (id, x, ys, carry_tangents); else
      if (sy == ".") crv->set_vertex (id, xs, y, carry_tangents); else
      crv->set_vertex (id, xs, ys, carry_tangents);
      crv->evaluate ();
      return true;
    } else if (what == "lt") {
      crv->get_left_tangent (id, x, y);
      if (sx == ".") crv->set_left_tangent (id, x, ys); else
      if (sy == ".") crv->set_left_tangent (id, xs, y); else
      crv->set_left_tangent (id, xs, ys);
      crv->evaluate ();
      return true;
    } else if (what == "rt") {
      crv->get_right_tangent (id, x, y);
      if (sx == ".") crv->set_right_tangent (id, x, ys); else
      if (sy == ".") crv->set_right_tangent (id, xs, y); else
      crv->set_right_tangent (id, xs, ys);
      crv->evaluate ();
      return true;
    } else {
      result = "bad target: must be v lt OR rt";
      return false;
    }

  }
  result = "curve " + name + " not found";
  return false;
}

bool curve_name::operator() (tokenizer& tz) {
  curve_editor* ced = uis.crved;
  if (ced && ced->pik()) {
    multi_curve* crv = ced->curveinfo[ced->pik.crv_id].curve;
    string name; tz >> name;
    if (name != "") crv->name = name;
  }
  return true;
}

bool curve__library::operator() (tokenizer& tz) {
  curve_editor* ced = uis.crved;
  if (!ced) {
    result = "not a curve editor";
    return false;
  }
  curve_library* cl = ced->library;
  if (cl) {
    string cmd; tz >> cmd;
    if (cmd == "a" || cmd == "add") ced->add_curve (); else
    if (cmd == "d" || cmd == "delete") cl->del (); else
    if (cmd == "u" || cmd == "update") ced->replace_curve (); else
    if (cmd == "i" || cmd == "insert") ced->insert_curve (); else
    if (cmd == "+") {int n; tz >> n; cl->move (n);}
    else if (cmd == "-") {int n; tz >> n; cl->move (-n);} else {
      result = "bad sub-command. see help curve-library";
      return false;
    }
  } else {
    result = "no curve library";
    return false;
  }
  return true;
}

bool set_curve_editor::operator() (tokenizer& tz) {
  string name;
  int screen;
  tz >> name >> screen;
  if (uis.set_editor (name, screen)) {
    stringstream ss;
    ss << "update-editor " << name << ' ' << screen;
    interpreter (ss.str());
  }
  return true;
}

bool set_scope::operator() (tokenizer& tz) {
  string name, color; tz >> name >> color;
  float r, g, b; stringstream ss (color); ss >> r >> g >> b;
  if (name == "left" || name == "l") {
    d->osc.lr = r;
    d->osc.lg = g;
    d->osc.lb = b;
    return true;
  } else if (name == "right" || name == "r") {
    d->osc.rr = r;
    d->osc.rg = g;
    d->osc.rb = b;
    return true;
  } else return false;
}

bool get_scope::operator() (tokenizer& tz) {
  string name; tz >> name;
  if (name == "left" || name == "l") {
    sprintf (buf, "%f %f %f", d->osc.lr, d->osc.lg, d->osc.lb);
    result = buf;
    return true;
  } else if (name == "right" || name == "r") {
    sprintf (buf, "%f %f %f", d->osc.rr, d->osc.rg, d->osc.rb);
    result = buf;
    return true;
  } else return false;
}

bool set_drone::operator() (tokenizer& tz) {

  string param; tz >> param;
  if (param == "volume" || param == "v") {
    vector<int> ids;
    vector<float> vols;
    string s;
    tz >> s; if (s != "") get_data (s, ids); else return false;
    tz >> s; if (s != "") get_data (s, vols); else return false;
    float v = 0;
    for (int i = 0, j = ids.size(), m = vols.size (); i < j; ++i) {
      int k = ids[i];
      if (i < m) v = vols[i];
      d.set_drone_volume (k, v);
    }
    return true;
  } else {
    result = "bad param";
    return false;
  }

}

bool get_drone::operator() (tokenizer& tz) {
  int num_drones = d.drones.size ();
  string param; tz >> param;
  if (param == "n" || param == "num_drones") {
    sprintf (buf, "%d", num_drones);
    result = buf;
    return true;
  } else if (param == "volume" || param == "v") {
    stringstream ss;
    string s; tz >> s;
    if (s != "") {
      vector<int> ids; get_data (s, ids);
      for (int i = 0, j = ids.size (); i < j; ++i) {
        int k = ids[i];
        if (k > -1 && k < num_drones) ss << d.drones[k].vol << ' ';
      }
      result = ss.str ();
      return true;
    } else return false;
  } else if (param == "selected" || param == "s") {
    result = d.get_selected_drones ();
    return true;
  }
  return false;
}

bool set_text_color::operator() (tokenizer& tz) {
  tz >> cons.clr.r >> cons.clr.g >> cons.clr.b;
  return true;
}

bool load_tuning::operator() (tokenizer& tz) {
    extern load_globals lg;
    string name; tz >> name;
    string fname = name + ".tuning";
    if (lg.load_intervals (fname)) {
      result = "tuning: " + name;
      d->restore_all_notes ();
      keybd2.setup_notes ();
      keybd2.setup_midi_notes ();
      return true;
    } else {
      result = "bad tuning file: " + fname + " or invalid tuning.";
      return false;
    }
}

bool paste_gater::operator() (tokenizer& tz) {

  if (curve_editor::copy.num_vertices ()) {
    extern din din0;
    float r, g, b;
    din0.gatr.crv.get_color (r, g, b);
    din0.gatr.crv = curve_editor::copy;
    din0.gatr.crv.set_color (r, g, b);
    din0.gatr.sol.update ();
  }
  return true;
}

bool get_selection::operator () (tokenizer& tz) {
  extern ui_list uis;
  if (uis.crved) result = uis.crved -> selection ();
  return true;
}

bool get_intervals::operator() (tokenizer& tz) {

  string what; tz >> what;
  stringstream ss;
  if (what == "piano") {
    for (int i = 0, j = NUM_NOTES - 1; i < j; ++i) {
      ss << WESTERN_FLAT[i] << ' ' << WIKIPEDIA_KEY_FREQUENCIES[i] << ' ';
    }
    result = ss.str ();
  } else {
    extern map <string, float> INTERVALS;
    for (map<string, float>::iterator i = INTERVALS.begin (), j = INTERVALS.end (); i != j; ++i) {
      const pair<string, float>& p = *i;
      ss << p.first << ' ' << p.second << ' ';
    }
  }
  result = ss.str ();
  return true;

}

bool num_octaves::operator() (tokenizer& tz) {
  tz >> NUM_OCTAVES;
  if (NUM_OCTAVES < 1) NUM_OCTAVES = 1;
  d.setup_ranges (0); // 0 => dont load from disk
  d.update_drone_ranges ();
  d.update_drone_tone ();
  return true;
}

bool quit::operator() (tokenizer& tz) {
  extern int quit_now;
  quit_now = 1;
  return true;
}
