package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.ItemEvent;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;

import latexDraw.figures.Figure;
import latexDraw.figures.LaTeXDrawPolygon;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.DrawPanel;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;


/** 
 * This class defines the frame which allows to change the parameters of polygons.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParametersPolygonJoinedLinesFrame extends AbstractParametersFrame
{
	private static final long serialVersionUID = 1L;

	/** Allows to select the point to change */
	protected JComboBox pointChoice;
	
	/** The X-coordinates of the polygon */
	protected double xs[];
	
	/** The Y-coordinates of the polygon */
	protected double ys[];
	
	/** The number of points of the polygon */
	protected int nbPoints;
	
	/** Allows to set the X-coordinate of a point */
	protected JSpinner xCoord;
	
	/** Allows to set the Y-coordinate of a point */
	protected JSpinner yCoord;
	
	public static final String LABEL_CHOICE_PTS = "ptChoice"; //$NON-NLS-1$
	
	
	
	/**
	 * The constructor by default
	 */
	public ParametersPolygonJoinedLinesFrame(Frame parent, DrawPanel p, boolean withoutArrow, boolean attachToMainFrame)
	{
		super(parent, p, attachToMainFrame);
		
		try
		{
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
	  		JTabbedPane tabbedPane = new JTabbedPane();
	  		JPanel pButton  = createButtonsPanel(this);
	  		JPanel pGeneral = new JPanel();
	  		pGeneral.setLayout(new BoxLayout(pGeneral, BoxLayout.Y_AXIS));
	  		
	     	pGeneral.add(createPositionPanel());
	     	pGeneral.add(createBordersPanel(false));
	     	pGeneral.add(createFillPanel());
	     	pGeneral.add(createHatchingPanel());
	     	
	     	JPanel pOther = new JPanel();
	     	pOther.setLayout(new BoxLayout(pOther, BoxLayout.Y_AXIS));
	     	pOther.add(createShadowPanel());
	     	pOther.add(createGradientPanel());
	     	pOther.add(createDoubleLinePanel());
	     	
	     	tabbedPane.addTab(TITLE_TABBED_PANE_GENERAL, pGeneral);
	     	
	     	if(!withoutArrow)
	     	{
		     	tabbedPane.addTab(NAME_TABBEDPANE_ARROW, createArrowPanel(true, false));
		     	updateEnableParamsArrow();
	     	}
			
	     	tabbedPane.addTab(TITLE_TABBED_PANE_OTHERS, pOther);
	     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
	     	getContentPane().add(glimpsePanelZoom);
	     	getContentPane().add(tabbedPane);
			getContentPane().add(pButton);
	  		pack();
	  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);
		}
		catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	/**
	 * @return A panel allowing to change the coordinates of the points of the polygon.
	 * @since 1.9
	 */
	protected JPanel createPositionPanel()
	{
		JPanel coordP = new JPanel(new GridBagLayout());
  		GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;
     	constraint.gridy = 0;
     	constraint.gridwidth = 1;
     	constraint.gridheight = 1;
     	constraint.weightx = 0.1;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.CENTER;
     	coordP.add(new JLabel(LaTeXDrawLang.getDialogFrameString("ParametersPolygonFrame.1")) //$NON-NLS-1$
     					, constraint);
  		
     	pointChoice = new JComboBox();
     	pointChoice.addItemListener(this);
     	pointChoice.setName(LABEL_CHOICE_PTS);
     	constraint.gridx++;	
     	coordP.add(pointChoice, constraint);
  		
    	constraint.gridx=0; constraint.gridy++;
     	coordP.add(new JLabel(LABEL_X), constraint);
  		
     	SpinnerModel model = new SpinnerNumberModel(0,0,10000,0.1);
     	xCoord = new JSpinner(model);
     	xCoord.setEditor(new JSpinner.NumberEditor(xCoord, "0.0"));//$NON-NLS-1$
		xCoord.addChangeListener(this);
     	xCoord.setName(LABEL_X);
     	constraint.gridx++;
     	coordP.add(xCoord, constraint);
     	
    	constraint.gridx++;	
     	coordP.add(new JLabel(LABEL_Y), constraint);
  		
     	model = new SpinnerNumberModel(0,0,10000,0.1);
     	yCoord = new JSpinner(model);
     	yCoord.setEditor(new JSpinner.NumberEditor(yCoord, "0.0"));//$NON-NLS-1$
     	yCoord.addChangeListener(this);
     	yCoord.setName(LABEL_Y);
     	constraint.gridx++;	
     	coordP.add(yCoord, constraint);
     	
     	constraint.gridx = 0;	constraint.gridy++;
     	coordP.add(new JLabel(LABEL_ROTATION), constraint);
     	constraint.gridx++;
     	coordP.add(rotationField, constraint);
     	
     	coordP.setBorder(new CompoundBorder(new TitledBorder(null, LABEL_POSITION_DIMENSIONS, 
						  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	
     	return coordP;
	}

	
	

	@Override
	public void setFigureFrameField()
	{
		LaTeXDrawPolygon p = ((LaTeXDrawPolygon)figure);
		
		nbPoints = p.getNbPoints();
		xs = new double[nbPoints];
		ys = new double[nbPoints];
		LaTeXDrawPoint2D pt;
		pointChoice.removeItemListener(this);
		pointChoice.removeAllItems();
		pointChoice.addItemListener(this);
		
		// We copy all points
		for(int i=0; i<nbPoints; i++)
		{
			pt = p.getPoint(i);
			xs[i] = pt.x;
			ys[i] = pt.y;
			pointChoice.addItem(String.valueOf(i+1));		
		}	
		
		pointChoice.setSelectedIndex(0);
		pointChoice.updateUI();
		xCoord.setValue(xs[0]);
		yCoord.setValue(ys[0]);
	}
	
	
	protected void updatePointsValue(LaTeXDrawPolygon pol) {
		nbPoints = pol.getNbPoints();
		LaTeXDrawPoint2D pt;
		
		for(int i=0; i<nbPoints; i++) {
			pt = pol.getPoint(i);
			xs[i] = pt.x;
			ys[i] = pt.y;
		}	
		
		xCoord.setValue(xs[pointChoice.getSelectedIndex()]);
		yCoord.setValue(ys[pointChoice.getSelectedIndex()]);
	}

	

	@Override
	public void saveParameters()
	{
		super.saveParameters();
		
		LaTeXDrawPolygon p = ((LaTeXDrawPolygon)figure);
		
		// We save all the points
		for(int i=0; i<nbPoints; i++)
			p.setPoint(xs[i], ys[i], i);
	}
	
	
	

	@Override
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox) 
	{
		try
		{
			if(!(selected instanceof LaTeXDrawPolygon))
				throw new ClassCastException();
			
			super.setVisible(visible, selected, deleteOnCancel, isFramedBox);
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}
	
	
	

	@Override
	public void itemStateChanged(ItemEvent e) 
	{
		Object o = e.getSource();
		
		if(o instanceof JComboBox)
		{
			if(((JComboBox)o).getName().equals(LABEL_CHOICE_PTS))
			{
				// We must change the fields concerning the point 
				// when the choice change
				int id = Integer.valueOf(pointChoice.getSelectedItem().toString()).intValue()-1;
				xCoord.setValue(xs[id]);
				yCoord.setValue(ys[id]);
				setModified(true);
				return ;
			}
		}
		
		super.itemStateChanged(e);
	}

	
	
	
	@Override
	public void stateChanged(ChangeEvent e) 
	{
		Object o = e.getSource();
		
		if(o instanceof JSpinner)
		{
			String name = ((JSpinner)o).getName();
			LaTeXDrawPolygon p = (LaTeXDrawPolygon) glimpsePanel.getGlimpseFigure();
			
			if(name.equals(LABEL_X))
			{
				int id = Integer.valueOf(pointChoice.getSelectedItem().toString()).intValue()-1;				
				float x = Float.valueOf(xCoord.getValue().toString()).floatValue();
				xs[id] = x;
				p.setPoint(x, p.getPoint(id).y, id);
				glimpsePanel.repaint();
				setModified(true);
				return ;
			}
			
			if(name.equals(LABEL_Y))
			{
				int id = Integer.valueOf(pointChoice.getSelectedItem().toString()).intValue()-1;
				float y = Float.valueOf(yCoord.getValue().toString()).floatValue();
				ys[id] = y;
				p.setPoint(p.getPoint(id).x, y, id);
				glimpsePanel.repaint();
				setModified(true);
				return ;
			}
			
			if(name.equals(LaTeXDrawResources.LABEL_ROTATE))
			{
				setModified(true);
				p.setRotationAngle(Math.toRadians( Double.valueOf(((JSpinner)o).getValue().toString()).doubleValue()));
				updatePointsValue(p);
				glimpsePanel.repaint();
				return;
			}
		}
		super.stateChanged(e);
	}
}
