/***********************************************************************************

    Copyright (C) 2007-2024 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_FILTERING_HEADER
#define LIFEOGRAPH_FILTERING_HEADER


#include <cassert>

#include "diarydata.hpp"
#include "entry.hpp"


namespace LIFEO
{

// FILTERED OBJECT CLASSES
namespace FOC
{
    static const int NOTHING        = 0x00000;
    static const int ENTRIES        = 0x00001;
    static const int PARAGRAPHS     = 0x00002;
    static const int DIARYELEMS     = 0x0000F;
    static const int NUMBERS        = 0x00010;
    static const int STRINGS        = 0x00020;
    static const int DATES          = 0x00040;
    static const int BASIC_DATA     = 0x0FFF0;
    static const int ALL_REAL       = 0x0FFFF;
    static const int SUBGROUPS      = 0x10000;
    static const int EVERYTHING     = 0xFFFFF;
    static const int FILTER_PANEL   = ENTRIES|SUBGROUPS;
}

// FORWARD DECLARATIONS
class FiltererContainer;
class Filter;

class Filterer
{
    public:
        Filterer( Diary* diary, FiltererContainer* p2container )
        : m_p2container( p2container ), m_p2diary( diary ) {}
        virtual ~Filterer() {}

        virtual bool            is_container() const    { return false; }
        bool                    is_not() const          { return m_F_not; }
        void                    set_not( bool F_not )   { m_F_not = F_not; }
        void                    toggle_not()            { m_F_not = !m_F_not; }

        virtual bool            filter( const Entry* ) const { return true; }
        virtual bool            filter( const Paragraph* ) const { return true; }
        virtual bool            filter( const double ) const { return true; }
        virtual int             get_obj_classes() const = 0;
        virtual void            get_as_string( Ustring& ) const = 0;
        virtual Ustring         get_as_human_readable_str() const = 0;

        FiltererContainer*      m_p2container;

    protected:
        bool                    m_F_not     { false };

        Diary*                  m_p2diary   { nullptr };
};

class FiltererStatus : public Filterer
{
    public:
        static const ElemStatus STATUS_DEFAULT
                                { ES::SHOW_NOT_TODO|ES::SHOW_TODO|ES::SHOW_PROGRESSED };

        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererStatus( Diary* d, FiltererContainer* ctr, ElemStatus es = STATUS_DEFAULT )
        : Filterer( d, ctr ), m_included_statuses( es ) {}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        ElemStatus              m_included_statuses;
};

class FiltererSize : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererSize( Diary* d, FiltererContainer* ctr, char type = VT::SO::CHAR_COUNT::C,
                      double range_b = -1, bool F_incl_b = false,
                      double range_e = -1, bool F_incl_e = false )
        : Filterer( d, ctr ), m_type( type ),
          m_range_b( range_b ), m_F_incl_b( F_incl_b ),
          m_range_e( range_e ), m_F_incl_e( F_incl_e ){}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        bool                    filter_v( int ) const;

        char                    m_type;
        int                     m_range_b;
        bool                    m_F_incl_b; // ( or [
        int                     m_range_e;
        bool                    m_F_incl_e; // ) or ]
};

class FiltererFavorite : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererFavorite( Diary* d, FiltererContainer* ctr )
        : Filterer( d, ctr ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
};

class FiltererTrashed : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererTrashed( Diary* d, FiltererContainer* ctr )
        : Filterer( d, ctr ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
};

class FiltererUnit : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererUnit( Diary* d, FiltererContainer* ctr, const Ustring& unit )
        : Filterer( d, ctr ), m_unit( unit ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
        Ustring                 m_unit;
};

class FiltererIs : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererIs( Diary* d, FiltererContainer* ctr, DEID id )
        : Filterer( d, ctr ), m_id( id ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
        DEID                    m_id          { DEID_UNSET };
};

class FiltererDescendantOf : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererDescendantOf( Diary*, FiltererContainer*, DEID, bool );

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
        Entry*                  m_entry_p     { nullptr };
        bool                    m_F_or_itself { false };
};

class FiltererTaggedBy : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererTaggedBy( Diary* d, FiltererContainer* ctr, Entry* tag )
        : Filterer( d, ctr ), m_tag( tag ) {}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        Entry*                  m_tag         { nullptr };
};

class FiltererSubtaggedBy : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererSubtaggedBy( Diary* d, FiltererContainer* ctr,
                             Entry* tag_parent, Entry* tag_child,
                             char rel = '=', int type = VT::LAST )
        : Filterer( d, ctr ), m_tag_parent( tag_parent ), m_tag_child( tag_child ),
                              m_relation( rel ), m_type( type ) {}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        bool                    filter_common( Entry* ) const;

        Entry*                  m_tag_parent  { nullptr };
        Entry*                  m_tag_child   { nullptr };
        char                    m_relation;
        int                     m_type;
};

class FiltererTagValue : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererTagValue( Diary* d, FiltererContainer* ctr, Entry* tag, int type = VT::REALIZED,
                          double range_b = Constants::INFINITY_MNS, bool F_incl_b = false,
                          double range_e = Constants::INFINITY_PLS, bool F_incl_e = false )
        : Filterer( d, ctr ), m_tag( tag ), m_type( type ),
          m_range_b( range_b ), m_F_incl_b( F_incl_b ),
          m_range_e( range_e ), m_F_incl_e( F_incl_e ){}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        bool                    filter_v( double ) const;

        Entry*                  m_tag         { nullptr };
        int                     m_type;
        double                  m_range_b;
        bool                    m_F_incl_b; // ( or [
        double                  m_range_e;
        bool                    m_F_incl_e; // ) or ]
};

class FiltererTheme : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererTheme( Diary* d, FiltererContainer* ctr, Theme* theme )
        : Filterer( d, ctr ), m_theme( theme ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
        Theme*                  m_theme       { nullptr };
};

class FiltererBetweenDates : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererBetweenDates( Diary* d, FiltererContainer* ctr,
                              DateV date_b = Date::NOT_SET, bool f_incl_b = false,
                              DateV date_e = Date::NOT_SET, bool f_incl_e = false )
        : Filterer( d, ctr ), m_date_b( std::min( date_b, date_e ) ), m_F_incl_b( f_incl_b ),
                              m_date_e( std::max( date_b, date_e ) ), m_F_incl_e( f_incl_e ) {}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS | FOC::DATES };

    protected:
        bool                    filter_v( DateV ) const;

        DateV                   m_date_b;
        bool                    m_F_incl_b; // ( or [
        DateV                   m_date_e;
        bool                    m_F_incl_e; // ) or ]
};

class FiltererBetweenEntries : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererBetweenEntries( Diary* d, FiltererContainer* ctr,
                                Entry* entry_b = nullptr, bool f_incl_b = false,
                                Entry* entry_e = nullptr, bool f_incl_e = false )
        : Filterer( d, ctr ), m_entry_b( entry_b ), m_F_incl_b( f_incl_b ),
                              m_entry_e( entry_e ), m_F_incl_e( f_incl_e ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
        Entry*                  m_entry_b;
        bool                    m_F_incl_b; // ( or [
        Entry*                  m_entry_e;
        bool                    m_F_incl_e; // ) or ]
};

class FiltererCompletion : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererCompletion( Diary* d, FiltererContainer* ctr,
                            double compl_b = 0.0, double compl_e = 100.0 )
        : Filterer( d, ctr ), m_compl_b( std::min( compl_b, compl_e ) ),
                              m_compl_e( std::max( compl_b, compl_e ) ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
        double                  m_compl_b;
        double                  m_compl_e;
};

class FiltererContainsText : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererContainsText( Diary* d, FiltererContainer* ctr,
                              const Ustring& text, bool case_sensitive, bool name_only )
        : Filterer( d, ctr ), m_text( text ), m_case_sensitive( case_sensitive ),
          m_name_only( name_only ) {}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        Ustring                 m_text;
        bool                    m_case_sensitive;
        bool                    m_name_only;
};

class FiltererIsCurrentEntry : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererIsCurrentEntry( Diary* d, FiltererContainer* ctr, bool F_include_descendants )
        : Filterer( d, ctr ), m_F_include_descendants( F_include_descendants ) {}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        bool                    m_F_include_descendants;
};

class FiltererChildFilter : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererChildFilter( Diary*, FiltererContainer*, Filter* );

        ~FiltererChildFilter();

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
        Filter*                 m_p2filter;
        FiltererContainer*      m_FC_stack    { nullptr };
};

class FiltererIsImage : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererIsImage( Diary* d, FiltererContainer* ctr )
        : Filterer( d, ctr ) {}

        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::PARAGRAPHS };

    protected:
};

class FiltererHasCoords : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererHasCoords( Diary* d, FiltererContainer* ctr )
        : Filterer( d, ctr ) {}

        bool                    filter( const Entry* ) const override;
        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES | FOC::PARAGRAPHS };

    protected:
};

class FiltererTitleStyle : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererTitleStyle( Diary* d, FiltererContainer* ctr, char style )
        : Filterer( d, ctr ), m_title_style( style ) {}

        bool                    filter( const Entry* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::ENTRIES };

    protected:
        char                    m_title_style;
};

class FiltererHeadingLevel : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererHeadingLevel( Diary* d, FiltererContainer* ctr, int H_levels )
        : Filterer( d, ctr ), m_H_levels( H_levels ) {}

        bool                    filter( const Paragraph* ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::PARAGRAPHS };

    protected:
        int                     m_H_levels;
};

// BASIC FILTERERS =================================================================================
class FiltererEquals : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }

        FiltererEquals( Diary* d, FiltererContainer* ctr, char relation, double value )
        : Filterer( d, ctr ), m_relation( relation ), m_value( value ) {}

        bool                    filter( const double ) const override;
        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;

        const static int        m_obj_classes { FOC::NUMBERS };

    protected:
        char                    m_relation;
        double                  m_value;
};

typedef std::vector< Filterer* > VecFilterers;

// FILTERER CONTAINER ==============================================================================
class FiltererContainer : public Filterer
{
    public:
        int                     get_obj_classes() const override { return m_obj_classes; }
        // this is used by parent dialogs:
        int                     calculate_obj_classes() const
        {
            int obj_classes { 0 };

            for( auto& filterer : m_pipeline )
                obj_classes |= filterer->get_obj_classes();

            // return obj_classes;
            return( obj_classes & FOC::ALL_REAL );
        }

        FiltererContainer( Diary* diary, FiltererContainer* ctr, bool flag_or = false )
        : Filterer( diary, ctr ), m_F_or( flag_or ) {}

        ~FiltererContainer()
        { clear_pipeline(); }

        virtual void            clear_pipeline();

        virtual void            update_state() {}

        bool                    filter( const Entry* e ) const override
        { return filter_internal( e ); }
        bool                    filter( const Paragraph* p ) const override
        { return filter_internal( p ); }
        bool                    filter( const double d ) const override
        { return filter_internal( d ); }

        bool                    is_container() const override { return true; }
        bool                    is_or() const
        { return m_F_or; }

        void                    get_as_string( Ustring& ) const override;
        Ustring                 get_as_human_readable_str() const override;
        virtual void            set_from_string( const Ustring& );

        virtual void            toggle_logic() {}
        virtual void            update_logic_label() {}

        virtual void            add_filterer_status( ElemStatus es )
        { m_pipeline.push_back( new FiltererStatus( m_p2diary, this, es ) ); }
        virtual void            add_filterer_size( int type,
                                                   double range_b, bool f_incl_b,
                                                   double range_e, bool f_incl_e )
        { m_pipeline.push_back( new FiltererSize( m_p2diary, this, type, range_b, f_incl_b,
                                                                         range_e, f_incl_e ) ); }
        virtual void            add_filterer_favorite()
        { m_pipeline.push_back( new FiltererFavorite( m_p2diary, this ) ); }
        virtual void            add_filterer_trashed()
        { m_pipeline.push_back( new FiltererTrashed( m_p2diary, this ) ); }
        virtual void            add_filterer_unit( const Ustring& unit )
        { m_pipeline.push_back( new FiltererUnit( m_p2diary, this, unit ) ); }
        virtual void            add_filterer_is( DEID id )
        { m_pipeline.push_back( new FiltererIs( m_p2diary, this, id ) ); }
        virtual void            add_filterer_descendant_of( DEID id, bool F_itself )
        { m_pipeline.push_back( new FiltererDescendantOf( m_p2diary, this, id, F_itself ) ); }
        virtual void            add_filterer_tagged_by( Entry* tag )
        { m_pipeline.push_back( new FiltererTaggedBy( m_p2diary, this, tag ) ); }
        virtual void            add_filterer_subtagged_by( Entry* tag_p, Entry* tag_c,
                                                           char rel, int type )
        { m_pipeline.push_back( new FiltererSubtaggedBy( m_p2diary, this,
                                                         tag_p, tag_c, rel, type ) ); }
        virtual void            add_filterer_tag_value( Entry* tag, int type,
                                                        double range_b, bool f_incl_b,
                                                        double range_e, bool f_incl_e )
        { m_pipeline.push_back( new FiltererTagValue( m_p2diary, this, tag, type,
                                                      range_b, f_incl_b,
                                                      range_e, f_incl_e ) ); }
        virtual void            add_filterer_theme( Theme* theme )
        { m_pipeline.push_back( new FiltererTheme( m_p2diary, this, theme ) ); }
        virtual void            add_filterer_between_dates( DateV date_b, bool f_incl_b,
                                                            DateV date_e, bool f_incl_e )
        { m_pipeline.push_back( new FiltererBetweenDates(
                m_p2diary, this, date_b, f_incl_b, date_e, f_incl_e ) ); }
        virtual void            add_filterer_between_entries( Entry* entry_b, bool f_incl_b,
                                                              Entry* entry_e, bool f_incl_e )
        { m_pipeline.push_back( new FiltererBetweenEntries(
                m_p2diary, this, entry_b, f_incl_b, entry_e, f_incl_e ) ); }
        virtual void            add_filterer_completion( double compl_b, double compl_e )
        { m_pipeline.push_back( new FiltererCompletion( m_p2diary, this, compl_b, compl_e ) ); }
        virtual void            add_filterer_contains_text( const Ustring& t, bool cs, bool no )
        { m_pipeline.push_back( new FiltererContainsText( m_p2diary, this, t, cs, no ) ); }
        void                    add_filterer_current_entry( bool F_include_descs )
        { m_pipeline.push_back( new FiltererIsCurrentEntry( m_p2diary, this, F_include_descs ) ); }
        virtual void            add_filterer_child_filter( Filter* filter )
        { m_pipeline.push_back( new FiltererChildFilter( m_p2diary, this, filter ) ); }
        virtual void            add_filterer_is_image()
        { m_pipeline.push_back( new FiltererIsImage( m_p2diary, this ) ); }

        virtual void            add_filterer_has_coords()
        { m_pipeline.push_back( new FiltererHasCoords( m_p2diary, this ) ); }

        virtual void            add_filterer_title_style( char title_style )
        { m_pipeline.push_back( new FiltererTitleStyle( m_p2diary, this, title_style ) ); }

        virtual void            add_filterer_heading_level( int h_levels )
        { m_pipeline.push_back( new FiltererHeadingLevel( m_p2diary, this, h_levels ) ); }

        virtual void            add_filterer_equals( char relation, double value )
        { m_pipeline.push_back( new FiltererEquals( m_p2diary, this, relation, value ) ); }

        virtual FiltererContainer*
                                add_filterer_subgroup()
        {
            FiltererContainer* container{ new FiltererContainer( m_p2diary, this, !m_F_or ) };
            m_pipeline.push_back( container );
            return container;
        }
        virtual void            remove_filterer( Filterer* );
        void                    toggle_not( Filterer* );

        virtual void            set_last_filterer_not()
        { m_pipeline.back()->set_not( true ); }

        const static int        m_obj_classes { FOC::EVERYTHING };

    protected:
        template< typename T >
        bool                    filter_internal( const T item ) const
        {
            if( m_pipeline.empty() )
                return true;

            for( auto& filterer : m_pipeline )
            {
                if( filterer->filter( item ) != filterer->is_not() )
                {
                    if( m_F_or )
                        return true;
                }
                else
                if( ! m_F_or )
                    return false;
            }

            return( ! m_F_or );
        }

        VecFilterers            m_pipeline;

        bool                    m_F_or;

    friend class Filter;
};

class Filter : public StringDefElem
{
    public:
        static const Ustring    DEFINITION_DEFAULT;
        static const Ustring    DEFINITION_MINIMAL;
        static const Ustring    DEFINITION_CUR_ENTRY;
        static const Ustring    DEFINITION_TRASHED;

        Filter( Diary* const diary, const Ustring& name, const Ustring& definition )
        : StringDefElem( diary, name, definition ) {}

        DiaryElement::Type      get_type() const override
        { return ET_FILTER; }

        const R2Pixbuf&         get_icon() const override;

        FiltererContainer*      get_filterer_stack() const;

        bool                    can_filter_class( int ) const;

        SKVVec                  get_as_skvvec() const override;

        int                     m_num_users{ 0 };
};

typedef std::map< Ustring, Filter*, FuncCmpStrings > MapUstringFilter;

} // end of namespace LIFEO

#endif
