/*
 * MIT License
 *
 * Copyright (c) 2002-2021 Mikko Tommila
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.apfloat;

import junit.framework.TestSuite;

/**
 * @version 1.10.1
 * @author Mikko Tommila
 */

public class ApcomplexMathTest
    extends ApfloatTestCase
{
    public ApcomplexMathTest(String methodName)
    {
        super(methodName);
    }

    public static void main(String[] args)
    {
        junit.textui.TestRunner.run(suite());
    }

    public static TestSuite suite()
    {
        TestSuite suite = new TestSuite();

        suite.addTest(new ApcomplexMathTest("testNegate"));
        suite.addTest(new ApcomplexMathTest("testAbs"));
        suite.addTest(new ApcomplexMathTest("testNorm"));
        suite.addTest(new ApcomplexMathTest("testScale"));
        suite.addTest(new ApcomplexMathTest("testIntegerPow"));
        suite.addTest(new ApcomplexMathTest("testInverseRoot"));
        suite.addTest(new ApcomplexMathTest("testRoot"));
        suite.addTest(new ApcomplexMathTest("testAllRoots"));
        suite.addTest(new ApcomplexMathTest("testAgm"));
        suite.addTest(new ApcomplexMathTest("testLog"));
        suite.addTest(new ApcomplexMathTest("testLogBase"));
        suite.addTest(new ApcomplexMathTest("testExp"));
        suite.addTest(new ApcomplexMathTest("testPow"));
        suite.addTest(new ApcomplexMathTest("testArg"));
        suite.addTest(new ApcomplexMathTest("testAcosh"));
        suite.addTest(new ApcomplexMathTest("testAsinh"));
        suite.addTest(new ApcomplexMathTest("testAtanh"));
        suite.addTest(new ApcomplexMathTest("testCosh"));
        suite.addTest(new ApcomplexMathTest("testSinh"));
        suite.addTest(new ApcomplexMathTest("testTanh"));
        suite.addTest(new ApcomplexMathTest("testAcos"));
        suite.addTest(new ApcomplexMathTest("testAsin"));
        suite.addTest(new ApcomplexMathTest("testAtan"));
        suite.addTest(new ApcomplexMathTest("testCos"));
        suite.addTest(new ApcomplexMathTest("testSin"));
        suite.addTest(new ApcomplexMathTest("testTan"));
        suite.addTest(new ApcomplexMathTest("testProduct"));
        suite.addTest(new ApcomplexMathTest("testSum"));
        suite.addTest(new ApcomplexMathTest("testGamma"));
        suite.addTest(new ApcomplexMathTest("testGammaIncomplete"));
        suite.addTest(new ApcomplexMathTest("testGammaIncompleteGeneralized"));
        suite.addTest(new ApcomplexMathTest("testUlp"));

        return suite;
    }

    @SuppressWarnings("deprecation")
    public static void testNegate()
    {
        Apcomplex z = new Apcomplex("(2,3)");
        assertEquals("(2,3)", new Apcomplex("(-2,-3)"), ApcomplexMath.negate(z));

        z = new Apcomplex("0");
        assertEquals("0", new Apcomplex("0"), ApcomplexMath.negate(z));
    }

    public static void testAbs()
    {
        Apcomplex z = new Apcomplex("(3,4)");
        assertEquals("(3,4)", new Apfloat(5), ApcomplexMath.abs(z));

        z = new Apcomplex("-4");
        assertEquals("-4", new Apfloat(4), ApcomplexMath.abs(z));

        z = new Apcomplex("(0,-4)");
        assertEquals("(0,-4)", new Apfloat(4), ApcomplexMath.abs(z));

        z = new Apcomplex("0");
        assertEquals("0", new Apfloat(0), ApcomplexMath.abs(z));
    }

    public static void testNorm()
    {
        Apcomplex z = new Apcomplex(new Apfloat(2), new Apfloat(3));
        assertEquals("(2,3)", new Apfloat(13), ApcomplexMath.norm(z));

        z = new Apcomplex("0");
        assertEquals("0", new Apfloat(0), ApcomplexMath.norm(z));
    }

    public static void testScale()
    {
        Apcomplex z = new Apcomplex("(2,3)");
        assertEquals("(2,3)", new Apcomplex("(20,30)"), ApcomplexMath.scale(z, 1));

        z = new Apcomplex("0");
        assertEquals("0", new Apfloat(0), ApcomplexMath.scale(z, 1));
    }

    public static void testIntegerPow()
    {
        Apcomplex z = new Apcomplex(new Apfloat(2, 1000), new Apfloat(3, 1000));
        assertEquals("(2,3)^3", new Apcomplex("(-46,9)"), ApcomplexMath.pow(z, 3));
        z = new Apcomplex(new Apfloat(3, 1000), new Apfloat(4, 1000));
        assertEquals("(3,4)^-1", new Apcomplex("(0.12,-0.16)"), ApcomplexMath.pow(z, -1));
        assertEquals("(3,4)^-2", new Apcomplex("(-0.0112,-0.0384)"), ApcomplexMath.pow(z, -2));
        assertEquals("(3,4)^0", new Apcomplex("1"), ApcomplexMath.pow(z, 0));
        z = new Apcomplex(Apfloat.ZERO, new Apfloat(4, 1000));
        assertEquals("(0,4)^0", new Apcomplex("1"), ApcomplexMath.pow(z, 0));

        assertEquals("(2,3)^1000000000", ApcomplexMath.pow(z, 1000000000).precision(10), ApcomplexMath.pow(z.precision(11), 1000000000).precision(10));

        z = new Apcomplex("(0,1.0000000000000000000000000000000000000000100000000)");
        assertEquals("1.00000000000000000000000000000000000000001^0x7FFFFFFFFFFFFFFF", new Apcomplex("(0,-1.0000000000000000000000922337203685477580700042535)"), ApcomplexMath.pow(z, 0x7FFFFFFFFFFFFFFFL), new Apfloat(1e-48));
        assertEquals("1.00000000000000000000000000000000000000001^0x8000000000000000", new Apfloat("0.99999999999999999999990776627963145224192000425352"), ApcomplexMath.pow(z, 0x8000000000000000L), new Apfloat(1e-48));

        try
        {
            ApcomplexMath.pow(new Apcomplex("0"), 0);
            fail("0^0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be undefined
        }
    }

    public static void testInverseRoot()
    {
        Apcomplex z = new Apcomplex(new Apfloat(0), new Apfloat(1, 30));
        assertEquals("(0,1)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(0,1)^-1/2, value", new Apcomplex("(0.707106781186547524400844362105, -0.707106781186547524400844362105)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-30"));

        z = new Apcomplex(new Apfloat(0), new Apfloat(1, 20));
        assertEquals("(0,1)^-1/2, precision", 20, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(0,1)^-1/2, value", new Apcomplex("(0.70710678118654752440, -0.70710678118654752440)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-20"));

        z = new Apcomplex(new Apfloat(0), new Apfloat(1, 10));
        assertEquals("(0,1)^-1/2, precision", 10, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(0,1)^-1/2, value", new Apcomplex("(0.7071067812, -0.7071067812)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-10"));

        z = new Apcomplex(new Apfloat(0), new Apfloat(1, 5));
        assertEquals("(0,1)^-1/2, precision", 5, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(0,1)^-1/2, value", new Apcomplex("(0.70711, -0.70711)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-5"));

        z = new Apcomplex(new Apfloat(0), new Apfloat(-1, 30));
        assertEquals("(0,-1)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(0,-1)^-1/2, value", new Apcomplex("(0.707106781186547524400844362105, 0.707106781186547524400844362105)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-30"));

        z = new Apcomplex(new Apfloat(-1, 30), new Apfloat(0));
        assertEquals("(-1,0)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(-1,0)^-1/2, value", new Apcomplex("(0, -1)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-30"));

        z = new Apcomplex(new Apfloat(-1, 30), new Apfloat(0));
        assertEquals("(-1,0)^-1/3, precision", 30, ApcomplexMath.inverseRoot(z, 3).precision());
        assertEquals("(-1,0)^-1/3, value", new Apcomplex("(0.5, -0.866025403784438646763723170753)"), ApcomplexMath.inverseRoot(z, 3), new Apfloat("5e-30"));

        z = new Apcomplex(new Apfloat(-1, 30), new Apfloat(0));
        assertEquals("(-1,0)^-1/4, precision", 30, ApcomplexMath.inverseRoot(z, 4).precision());
        assertEquals("(-1,0)^-1/4, value", new Apcomplex("(0.707106781186547524400844362105, -0.707106781186547524400844362105)"), ApcomplexMath.inverseRoot(z, 4), new Apfloat("5e-30"));

        z = new Apcomplex("(-0.707106781186547524400844362105, 0.707106781186547524400844362105)");
        assertEquals("(-1/sqrt(2),1/sqrt(2))^-1/3, precision", 30, ApcomplexMath.inverseRoot(z, 3).precision());
        assertEquals("(-1/sqrt(2),1/sqrt(2))^-1/3, value", new Apcomplex("(0.707106781186547524400844362105, -0.707106781186547524400844362105)"), ApcomplexMath.inverseRoot(z, 3), new Apfloat("5e-30"));

        z = new Apcomplex("(0.707106781186547524400844362105, 0.707106781186547524400844362105)");
        assertEquals("(1/sqrt(2),1/sqrt(2))^-1/3, precision", 30, ApcomplexMath.inverseRoot(z, 3).precision());
        assertEquals("(1/sqrt(2),1/sqrt(2))^-1/3, value", new Apcomplex("(0.965925826289068286749743199729, -0.258819045102520762348898837624)"), ApcomplexMath.inverseRoot(z, 3), new Apfloat("5e-30"));

        z = new Apcomplex(new Apfloat(2, 30), new Apfloat(0));
        assertEquals("2^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("2^-1/2, value", new Apcomplex("0.707106781186547524400844362105"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-30"));

        z = new Apcomplex(new Apfloat("-1e20", 30), new Apfloat(1));
        assertEquals("(-1e20,1)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(-1e20,1)^-1/2, value", new Apcomplex(new Apfloat("5e-31", 30), new Apfloat("-1e-10", 30)), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-40"));

        z = new Apcomplex(new Apfloat("-1e20", 30), new Apfloat(-1));
        assertEquals("(-1e20,-1)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(-1e20,-1)^-1/2, value", new Apcomplex(new Apfloat("5e-31", 30), new Apfloat("1e-10", 30)), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-40"));

        z = new Apcomplex(new Apfloat(0), new Apfloat("1e2000", 30));
        assertEquals("(0,1e2000)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(0,1e2000)^-1/2, value", new Apcomplex("(0.707106781186547524400844362105e-1000, -0.707106781186547524400844362105e-1000)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex(new Apfloat(1), new Apfloat("1e2000", 30));
        assertEquals("(1,1e2000)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(1,1e2000)^-1/2, value", new Apcomplex("(0.707106781186547524400844362105e-1000, -0.707106781186547524400844362105e-1000)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex(new Apfloat(0), new Apfloat("-1e2000", 30));
        assertEquals("(0,-1e2000)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(0,-1e2000)^-1/2, value", new Apcomplex("(0.707106781186547524400844362105e-1000, 0.707106781186547524400844362105e-1000)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex(new Apfloat(1), new Apfloat("-1e2000", 30));
        assertEquals("(1,-1e2000)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(1,-1e2000)^-1/2, value", new Apcomplex("(0.707106781186547524400844362105e-1000, 0.707106781186547524400844362105e-1000)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex(new Apfloat("1e2000", 30), new Apfloat(0));
        assertEquals("(1e2000,0)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(1e2000,0)^-1/2, value", new Apcomplex("1e-1000"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex(new Apfloat("1e2000", 30), new Apfloat(1));
        assertEquals("(1e2000,1)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(1e2000,1)^-1/2, value", new Apcomplex("1e-1000"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex(new Apfloat("-1e2000", 30), new Apfloat(0));
        assertEquals("(-1e2000,0)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(-1e2000,0)^-1/2, value", new Apcomplex("(0, -1e-1000)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex(new Apfloat("-1e2000", 30), new Apfloat(1));
        assertEquals("(-1e2000,1)^-1/2, precision", 30, ApcomplexMath.inverseRoot(z, 2).precision());
        assertEquals("(-1e2000,1)^-1/2, value", new Apcomplex("(0, -1e-1000)"), ApcomplexMath.inverseRoot(z, 2), new Apfloat("5e-1030"));

        z = new Apcomplex("(-0.707106781186547524400844362105e3000, 0.707106781186547524400844362105e3000)");
        assertEquals("(-1e3000/sqrt(2),1e3000/sqrt(2))^-1/3, precision", 30, ApcomplexMath.inverseRoot(z, 3).precision());
        assertEquals("(-1e3000/sqrt(2),1e3000/sqrt(2))^-1/3, value", new Apcomplex("(0.707106781186547524400844362105e-1000, -0.707106781186547524400844362105e-1000)"), ApcomplexMath.inverseRoot(z, 3), new Apfloat("5e-1030"));

        assertEquals("1", new Apfloat(1), ApcomplexMath.inverseRoot(Apcomplex.ONE, 5));
        assertEquals("i^1/2", new Apcomplex("(0.70710678,0.70710678)"), ApcomplexMath.inverseRoot(new Apcomplex("(0,1.0000000)"), -2));

        z = new Apcomplex(new Apfloat("3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127372458700660631558817488152092096282925409171536436789259036001133053054882046652138414695194151160943305727036575959195309218611738193261179310511854807446237996274956735188575272489122793818301194912983367336244065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960917363717872146844090122495343014654958537105079227968925892354201995611212902196086403441815981362977477130996051870721134999999837297804995105973173281609631859502445945534690830264252230825334468503526193118817101000313783875288658753320838142061717766914730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586327886593615338182796823030195203530185296899577362259941389124972177528347913151557485724245415069595082953311686172785588907509838175463746493931925506040092770167113900984882401285836160356370766010471018194295559619894676783744944825537977472684710404753464620804668425906949129331367702898915210475216205696602405803815019351125338243003558764024749647326391419927260426992279678235478163600934172164121992458631503028618297455570674983850549458858692699569092721079750930295532116534498720275596023648066549911988183479775356636980742654252786255181841757467289097777279380008164706001614524919217321721477235014144197356854816136115735255213347574184946843852332390739414333454776241686251898356948556209921922218427255025425688767179049460165346680498862723279178608578438382796797668145410095388378636095068006422512520511739298489608412848862694560424196528502221066118630674427862203919494504712371378696095636437191728746776465757396241389086583264599581339047802759009946576407895126946839835259570982582262052248940772671947826848260147699090264013639443745530506820349625245174939965143142980919065925093722169646151570985838741059788595977297549893016175392846813826868386894277415599185592524595395943104997252468084598727364469584865383673622262609912460805124388439045124413654976278079771569143599770012961608944169486855584840635342207222582848864815845602850601684273945226746767889525213852254995466672782398645659611635488623057745649803559363456817432411251507606947945109659609402522887971089314566913686722874894056010150330861792868092087476091782493858900971490967598526136554978189312978482168299894872265880485756401427047755513237964145152374623436454285844479526586782105114135473573952311342716610213596953623144295248493718711014576540359027993440374200731057853906219838744780847848968332144571386875194350643021845319104848100537061468067491927819119793995206141966342875444064374512371819217999839101591956181467514269123974894090718649423196156794520809514655022523160388193014209376213785595663893778708303906979207734672218256259966150142150306803844773454920260541466592520149744285073251866600213243408819071048633173464965145390579626856100550810665879699816357473638405257145910289706414011097120628043903975951567715770042033786993600723055876317635942187312514712053292819182618612586732157919841484882916447060957527069572209175671167229109816909152801735067127485832228718352093539657251210835791513698820914442100675103346711031412671113699086585163983150197016515116851714376576183515565088490998985998238734552833163550764791853589322618548963213293308985706420467525907091548141654985946163718027098199430992448895757128289059232332609729971208443357326548938239119325974636673058360414281388303203824903758985243744170291327656180937734440307074692112019130203303801976211011004492932151608424448596376698389522868478312355265821314495768572624334418930396864262434107732269780280731891544110104468232527162010526522721116603966655730925471105578537634668206531098965269186205647693125705863566201855810072936065987648611791045334885034611365768675324944166803962657978771855608455296541266540853061434443185867697514566140680070023787765913440171274947042056223053899456131407112700040785473326993908145466464588079727082668306343285878569830523580893306575740679545716377525420211495576158140025012622859413021647155097925923099079654737612551765675135751782966645477917450112996148903046399471329621073404375189573596145890193897131117904297828564750320319869151402870808599048010941214722131794764777262241425485454033215718530614228813758504306332175182979866223717215916077166925474873898665494945011465406284336639379003976926567214638530673609657120918076383271664162748888007869256029022847210403172118608204190004229661711963779213375751149595015660496318629472654736425230817703675159067350235072835405670403867435136222247715891504953098444893330963408780769325993978054193414473774418426312986080998886874132604721569516239658645730216315981931951673538129741677294786724229246543668009806769282382806899640048243540370141631496589794092432378969070697794223625082216889573837986230015937764716512289357860158816175578297352334460428151262720373431465319777741603199066554187639792933441952154134189948544473456738316249934191318148092777710386387734317720754565453220777092120190516609628049092636019759882816133231666365286193266863360627356763035447762803504507772355471058595487027908143562401451718062464362679456127531813407833033625423278394497538243720583531147711992606381334677687969597030983391307710987040859133746414428227726346594704745878477872019277152807317679077071572134447306057007334924369311383504931631284042512192565179806941135280131470130478164378851852909285452011658393419656213491434159562586586557055269049652098580338507224264829397285847831630577775606888764462482468579260395352773480304802900587607582510474709164396136267604492562742042083208566119062545433721315359584506877246029016187667952406163425225771954291629919306455377991403734043287526288896399587947572917464263574552540790914513571113694109119393251910760208252026187985318877058429725916778131496990090192116971737278476847268608490033770242429165130050051683233643503895170298939223345172201381280696501178440874519601212285993716231301711444846409038906449544400619869075485160263275052983491874078668088183385102283345085048608250393021332197155184306354550076682829493041377655279397517546139539846833936383047461199665385815384205685338621867252334028308711232827892125077126294632295639898989358211674562701021835646220134967151881909730381198004973407239610368540664319395097901906996395524530054505806855019567302292191393391856803449039820595510022635353619204199474553859381023439554495977837790237421617271117236434354394782218185286240851400666044332588856986705431547069657474585503323233421073015459405165537906866273337995851156257843229882737231989875714159578111963583300594087306812160287649628674460477464915995054973742562690104903778198683593814657412680492564879855614537234786733039046883834363465537949864192705638729317487233208376011230299113679386270894387993620162951541337142489283072201269014754668476535761647737946752004907571555278196536213239264061601363581559074220202031872776052772190055614842555187925303435139844253223415762336106425063904975008656271095359194658975141310348227693062474353632569160781547818115284366795706110861533150445212747392454494542368288606134084148637767009612071512491404302725386076482363414334623518975766452164137679690314950191085759844239198629164219399490723623464684411739403265918404437805133389452574239950829659122850855582157250310712570126683024029295252201187267675622041542051618416348475651699981161410100299607838690929160302884002691041407928862150784245167090870006992821206604183718065355672525325675328612910424877618258297651579598470356222629348600341587229805349896502262917487882027342092222453398562647669149055628425039127577102840279980663658254889264880254566101729670266407655904290994568150652653053718294127033693137851786090407086671149655834343476933857817113864558736781230145876871266034891390956200993936103102916161528813843790990423174733639480457593149314052976347574811935670911013775172100803155902485309066920376719220332290943346768514221447737939375170344366199104033751117354719185504644902636551281622882446257591633303910722538374218214088350865739177150968288747826569959957449066175834413752239709683408005355984917541738188399944697486762655165827658483588453142775687900290951702835297163445621296404352311760066510124120065975585127617858382920419748442360800719304576189323492292796501987518721272675079812554709589045563579212210333466974992356302549478024901141952123828153091140790738602515227429958180724716259166854513331239480494707911915326734302824418604142636395480004480026704962482017928964766975831832713142517029692348896276684403232609275249603579964692565049368183609003238092934595889706953653494060340216654437558900456328822505452556405644824651518754711962184439658253375438856909411303150952617937800297412076651479394259029896959469955657612186561967337862362561252163208628692221032748892186543648022967807057656151446320469279068212073883778142335628236089632080682224680122482611771858963814091839036736722208883215137556003727983940041529700287830766709444745601345564172543709069793961225714298946715435784687886144458123145935719849225284716050492212424701412147805734551050080190869960330276347870810817545011930714122339086639383395294257869050764310063835198343893415961318543475464955697810382930971646514384070070736041123735998434522516105070270562352660127648483084076118301305279320542746286540360367453286510570658748822569815793678976697422057505968344086973502014102067235850200724522563265134105592401902742162484391403599895353945909440704691209140938700126456001623742880210927645793106579229552498872758461012648369998922569596881592056001016552563756785667227966198857827948488558343975187445455129656344348039664205579829368043522027709842942325330225763418070394769941597915945300697521482933665556615678736400536665641654732170439035213295435291694145990416087532018683793702348886894791510716378529023452924407736594956305100742108714261349745956151384987137570471017879573104229690666702144986374645952808243694457897723300487647652413390759204340196340391147320233807150952220106825634274716460243354400515212669324934196739770415956837535551667302739007497297363549645332888698440611964961627734495182736955882207573551766515898551909866653935494810688732068599075407923424023009259007017319603622547564789406475483466477604114632339056513433068449539790709030234604614709616968868850140834704054607429586991382966824681857103188790652870366508324319744047718556789348230894310682870272280973624809399627060747264553992539944280811373694338872940630792615959954626246297070625948455690347119729964090894180595343932512362355081349490043642785271383159125689892951964272875739469142725343669415323610045373048819855170659412173524625895487301676002988659257866285612496655235338294287854253404830833070165372285635591525347844598183134112900199920598135220511733658564078264849427644113763938669248031183644536985891754426473998822846218449008777697763127957226726555625962825427653183001340709223343657791601280931794017185985999338492354956400570995585611349802524990669842330173503580440811685526531170995708994273287092584878944364600504108922669178352587078595129834417295351953788553457374260859029081765155780390594640873506123226112009373108048548526357228257682034160504846627750450031262008007998049254853469414697751649327095049346393824322271885159740547021482897111777923761225788734771881968254629812686858170507402725502633290449762778944236216741191862694396506715157795867564823993917604260176338704549901761436412046921823707648878341968968611815581587360629386038101712158552726683008238340465647588040513808016336388742163714064354955618689641122821407533026551004241048967835285882902436709048871181909094945331442182876618103100735477054981596807720094746961343609286148494178501718077930681085469000944589952794243981392135055864221964834915126390128038320010977386806628779239718014613432445726400973742570073592100315415089367930081699805365202760072774967458400283624053460372634165542590276018348403068113818551059797056640075094260878857357960373245141467867036880988060971642584975951380693094494015154222219432913021739125383559150310033303251117491569691745027149433151558854039221640972291011290355218157628232831823425483261119128009282525619020526301639114772473314857391077758744253876117465786711694147764214411112635835538713610110232679877564102468240322648346417663698066378576813492045302240819727856471983963087815432211669122464159117767322532643356861461865452226812688726844596844241610785401676814208088502800541436131462308210259417375623899420757136275167457318918945628352570441335437585753426986994725470316566139919996826282472706413362221789239031760854289437339356188916512504244040089527198378738648058472689546243882343751788520143956005710481194988423906061369573423155907967034614914344788636041031823507365027785908975782727313050488939890099239135033732508559826558670892426124294736701939077271307068691709264625484232407485503660801360466895118400936686095463250021458529309500009071510582362672932645373821049387249966993394246855164832611341461106802674466373343753407642940266829738652209357016263846485285149036293201991996882851718395366913452224447080459239660281715655156566611135982311225062890585491450971575539002439315351909021071194573002438801766150352708626025378817975194780610137150044899172100222013350131060163915415895780371177927752259787428919179155224171895853616805947412341933984202187456492564434623925319531351033114763949119950728584306583619353693296992898379149419394060857248639688369032655643642166442576079147108699843157337496488352927693282207629472823815374099615455987982598910937171262182830258481123890119682214294576675807186538065064870261338928229949725745303328389638184394477077940228435988341003583854238973542439564755568409522484455413923941000162076936368467764130178196593799715574685419463348937484391297423914336593604100352343777065888677811394986164787471407932638587386247328896456435987746676384794665040741118256583788784548581489629612739984134427260860618724554523606431537101127468097787044640947582803487697589483282412392929605829486191966709189580898332012103184303401284951162035342801441276172858302435598300320420245120728725355811958401491809692533950757784000674655260314461670508276827722235341911026341631571474061238504258459884199076112872580591139356896014316682831763235673254170734208173322304629879928049085140947903688786878949305469557030726190095020764334933591060245450864536289354568629585313153371838682656178622736371697577418302398600659148161640494496501173213138957470620884748023653710311508984279927544268532779743113951435741722197597993596852522857452637962896126915723579866205734083757668738842664059909935050008133754324546359675048442352848747014435454195762584735642161981340734685411176688311865448937769795665172796623267148103386439137518659467300244345005449953997423723287124948347060440634716063258306498297955101095418362350303094530973358344628394763047756450150085075789495489313939448992161255255977014368589435858775263796255970816776438001254365023714127834679261019955852247172201777237004178084194239487254068015560359983905489857235467456423905858502167190313952629445543913166313453089390620467843877850542393905247313620129476918749751910114723152893267725339181466073000890277689631148109022097245207591672970078505807171863810549679731001678708506942070922329080703832634534520380278609905569001341371823683709919495164896007550493412678764367463849020639640197666855923356546391383631857456981471962108410809618846054560390384553437291414465134749407848844237721751543342603066988317683310011331086904219390310801437843341513709243530136776310849135161564226984750743032971674696406665315270353254671126675224605511995818319637637076179919192035795820075956053023462677579439363074630569010801149427141009391369138107258137813578940055995001835425118417213605572752210352680373572652792241737360575112788721819084490061780138897107708229310027976659358387589093956881485602632243937265624727760378908144588378550197028437793624078250527048758164703245812908783952324532378960298416692254896497156069811921865849267704039564812781021799132174163058105545988013004845629976511212415363745150056350701278159267142413421033015661653560247338078430286552572227530499988370153487930080626018096238151613669033411113865385109193673938352293458883225508870645075394739520439680790670868064450969865488016828743437861264538158342807530618454859037982179945996811544197425363443996029025100158882721647450068207041937615845471231834600726293395505482395571372568402322682130124767945226448209102356477527230820810635188991526928891084555711266039650343978962782500161101532351605196559042118449499077899920073294769058685778787209829013529566139788848605097860859570177312981553149516814671769597609942100361835591387778176984587581044662839988060061622984861693533738657877359833616133841338536842119789389001852956919678045544828584837011709672125353387586215823101331038776682721157269495181795897546939926421979155233857662316762754757035469941489290413018638611943919628388705436777432242768091323654494853667680000010652624854730558615989991401707698385483188750142938908995068545307651168033373222651756622075269517914422528081651716677667279303548515420402381746089232839170327542575086765511785939500279338959205766827896776445318404041855401043513483895312013263783692835808271937831265496174599705674507183320650345566440344904536275600112501843356073612227659492783937064784264567633881880756561216896050416113903906396016202215368494109260538768871483798955999911209916464644119185682770045742434340216722764455893301277815868695250694993646101756850601671453543158148010545886056455013320375864548584032402987170934809105562116715468484778039447569798042631809917564228098739987669732376957370158080682290459921236616890259627304306793165311494017647376938735140933618332161428021497633991898354848756252987524238730775595559554651963944018218409984124898262367377146722606163364329640633572810707887581640438148501884114318859882769449011932129682715888413386943468285900666408063140777577257056307294004929403024204984165654797367054855804458657202276378404668233798528271057843197535417950113472736257740802134768260450228515797957976474670228409995616015691089038458245026792659420555039587922981852648007068376504183656209455543461351341525700659748819163413595567196496540321872716026485930490397874895890661272507948282769389535217536218507962977851461884327192232238101587444505286652380225328438913752738458923844225354726530981715784478342158223270206902872323300538621634798850946954720047952311201504329322662827276321779088400878614802214753765781058197022263097174950721272484794781695729614236585957820908307332335603484653187302930266596450137183754288975579714499246540386817992138934692447419850973346267933210726868707680626399193619650440995421676278409146698569257150743157407938053239252394775574415918458215625181921552337096074833292349210345146264374498055961033079941453477845746999921285999993996122816152193148887693880222810830019860165494165426169685867883726095877456761825072759929508931805218729246108676399589161458550583972742098090978172932393010676638682404011130402470073508578287246271349463685318154696904669686939254725194139929146524238577625500474852954768147954670070503479995888676950161249722820403039954632788306959762493615101024365553522306906129493885990157346610237122354789112925476961760050479749280607212680392269110277722610254414922157650450812067717357120271802429681062037765788371669091094180744878140490755178203856539099104775941413215432844062503018027571696508209642734841469572639788425600845312140659358090412711359200419759851362547961606322887361813673732445060792441176399759746193835845749159880976674470930065463424234606342374746660804317012600520559284936959414340814685298150539471789004518357551541252235905906872648786357525419112888773717663748602766063496035367947026923229718683277173932361920077745221262475186983349515101986426988784717193966497690708252174233656627259284406204302141137199227852699846988477023238238400556555178890876613601304770984386116870523105531491625172837327286760072481729876375698163354150746088386636406934704372066886512756882661497307886570156850169186474885416791545965072342877306998537139043002665307839877638503238182155355973235306860430106757608389086270498418885951380910304235957824951439885901131858358406674723702971497850841458530857813391562707603563907639473114554958322669457024941398316343323789759556808568362972538679132750555425244919435891284050452269538121791319145135009938463117740179715122837854601160359554028644059024964669307077690554810288502080858008781157738171917417760173307385547580060560143377432990127286772530431825197579167929699650414607066457125888346979796429316229655201687973000356463045793088403274807718115553309098870255052076804630346086581653948769519600440848206596737947316808641564565053004988161649057883115434548505266006982309315777650037807046612647060214575057932709620478256152471459189652236083966456241051955105223572397395128818164059785914279148165426328920042816091369377737222999833270820829699557377273756676155271139225880552018988762011416800546873655806334716037342917039079863965229613128017826797172898229360702880690877686605932527463784053976918480820410219447197138692560841624511239806201131845412447820501107987607171556831540788654390412108730324020106853419472304766667217498698685470767812051247367924791931508564447753798537997322344561227858432968466475133365736923872014647236794278700425032555899268843495928761240075587569464137056251400117971331662071537154360068764773186755871487839890810742953094106059694431584775397009439883949144323536685392099468796450665339857388878661476294434140104988899316005120767810358861166020296119363968213496075011164983278563531614516845769568710900299976984126326650234771672865737857908574664607722834154031144152941880478254387617707904300015669867767957609099669360755949651527363498118964130433116627747123388174060373174397054067031096767657486953587896700319258662594105105335843846560233917967492678447637084749783336555790073841914731988627135259546251816043422537299628632674968240580602964211463864368642247248872834341704415734824818333016405669596688667695634914163284264149745333499994800026699875888159350735781519588990053951208535103572613736403436753471410483601754648830040784641674521673719048310967671134434948192626811107399482506073949507350316901973185211955263563258433909982249862406703107683184466072912487475403161796994113973877658998685541703188477886759290260700432126661791922352093822787888098863359911608192353555704646349113208591897961327913197564909760001399623444553501434642686046449586247690943470482932941404111465409239883444351591332010773944111840741076849810663472410482393582740194493566516108846312567852977697346843030614624180358529331597345830384554103370109167677637427621021370135485445092630719011473184857492331816720721372793556795284439254815609137281284063330393735624200160456645574145881660521666087387480472433912129558777639069690370788285277538940524607584962315743691711317613478388271941686066257210368513215664780014767523103935786068961112599602818393095487090590738613519145918195102973278755710497290114871718971800469616977700179139196137914171627070189584692143436967629274591099400600849835684252019155937037010110497473394938778859894174330317853487076032219829705797511914405109942358830345463534923498268836240433272674155403016195056806541809394099820206099941402168909007082133072308966211977553066591881411915778362729274615618571037217247100952142369648308641025928874579993223749551912219519034244523075351338068568073544649951272031744871954039761073080602699062580760202927314552520780799141842906388443734996814582733720726639176702011830046481900024130835088465841521489912761065137415394356572113903285749187690944137020905170314877734616528798482353382972601361109845148418238081205409961252745808810994869722161285248974255555160763716750548961730168096138038119143611439921063800508321409876045993093248510251682944672606661381517457125597549535802399831469822036133808284993567055755247129027453977621404931820146580080215665360677655087838043041343105918046068008345911366408348874080057412725867047922583191274157390809143831384564241509408491339180968402511639919368532255573389669537490266209232613188558915808324555719484538756287861288590041060060737465014026278240273469625282171749415823317492396835301361786536737606421667781377399510065895288774276626368418306801908046098498094697636673356622829151323527888061577682781595886691802389403330764419124034120223163685778603572769415417788264352381319050280870185750470463129333537572853866058889045831114507739429352019943219711716422350056440429798920815943071670198574692738486538334361457946341759225738985880016980147574205429958012429581054565108310462972829375841611625325625165724980784920998979906200359365099347215829651741357984910471116607915874369865412223483418877229294463351786538567319625598520260729476740726167671455736498121056777168934849176607717052771876011999081441130586455779105256843048114402619384023224709392498029335507318458903553971330884461741079591625117148648744686112476054286734367090466784686702740918810142497111496578177242793470702166882956108777944050484375284433751088282647719785400065097040330218625561473321177711744133502816088403517814525419643203095760186946490886815452856213469883554445602495566684366029221951248309106053772019802183101032704178386654471812603971906884623708575180800353270471856594994761242481109992886791589690495639476246084240659309486215076903149870206735338483495508363660178487710608098042692471324100094640143736032656451845667924566695510015022983307984960799498824970617236744936122622296179081431141466094123415935930958540791390872083227335495720807571651718765994498569379562387555161757543809178052802946420044721539628074636021132942559160025707356281263873310600589106524570802447493754318414940148211999627645310680066311838237616396631809314446712986155275982014514102756006892975024630401735148919457636078935285550531733141645705049964438909363084387448478396168405184527328840323452024705685164657164771393237755172947951261323982296023945485797545865174587877133181387529598094121742273003522965080891777050682592488223221549380483714547816472139768209633205083056479204820859204754998573203888763916019952409189389455767687497308569559580106595265030362661597506622250840674288982659075106375635699682115109496697445805472886936310203678232501823237084597901115484720876182124778132663304120762165873129708112307581598212486398072124078688781145016558251361789030708608701989758898074566439551574153631931919810705753366337380382721527988493503974800158905194208797113080512339332219034662499171691509485414018710603546037946433790058909577211808044657439628061867178610171567409676620802957665770512912099079443046328929473061595104309022214393718495606340561893425130572682914657832933405246350289291754708725648426003496296116541382300773133272983050016025672401418515204189070115428857992081219844931569990591820118197335001261877280368124819958770702075324063612593134385955425477819611429351635612234966615226147353996740515849986035529533292457523888101362023476246690558164389678630976273655047243486430712184943734853006063876445662721866617012381277156213797461498613287441177145524447089971445228856629424402301847912054784985745216346964489738920624019435183100882834802492490854030778638751659113028739587870981007727182718745290139728366148421428717055317965430765045343246005363614726181809699769334862640774351999286863238350887566835950972655748154319401955768504372480010204137498318722596773871549583997184449072791419658459300839426370208756353982169620553248032122674989114026785285996734052420310917978999057188219493913207534317079800237365909853755202389116434671855829068537118979526262344924833924963424497146568465912489185566295893299090352392333336474352037077010108438800329075983421701855422838616172104176030116459187805393674474720599850235828918336929223373239994804371084196594731626548257480994825099918330069765693671596893644933488647442135008407006608835972350395323401795825570360169369909886711321097988970705172807558551912699306730992507040702455685077867906947661262980822516331363995211709845280926303759224267425755998928927837047444521893632034894155210445972618838003006776179313813991620580627016510244588692476492468919246121253102757313908404700071435613623169923716948481325542009145304103713545329662063921054798243921251725401323149027405858920632175894943454890684639931375709103463327141531622328055229729795380188016285907357295541627886764982741861642187898857410716490691918511628152854867941736389066538857642291583425006736124538491606741373401735727799563410433268835695078149313780073623541800706191802673285511919426760912210359874692411728374931261633950012395992405084543756985079570462226646190001035004901830341535458428337643781119885563187777925372011667185395418359844383052037628194407615941068207169703022851522505731260930468984234331527321313612165828080752126315477306044237747535059522871744026663891488171730864361113890694202790881431194487994171540421034121908470940802540239329429454938786402305129271190975135360009219711054120966831115163287054230284700731206580326264171161659576132723515666625366727189985341998952368848309993027574199164638414270779887088742292770538912271724863220288984251252872178260305009945108247835729056919885554678860794628053712270424665431921452817607414824038278358297193010178883456741678113989547504483393146896307633966572267270433932167454218245570625247972199786685427989779923395790575818906225254735822052364248507834071101449804787266919901864388229323053823185597328697809222535295910173414073348847610055640182423921926950620831838145469839236646136398910121021770959767049083050818547041946643713122996923588953849301363565761861060622287055994233716310212784574464639897381885667462608794820186474876727272220626764653380998019668836809941590757768526398651462533363124505364026105696055131838131742611844201890888531963569869627950367384243130113317533053298020166888174813429886815855778103432317530647849832106297184251843855344276201282345707169885305183261796411785796088881503296022907056144762209150947390359466469162353968092013945781758910889319921122600739281491694816152738427362642980982340632002440244958944561291670495082358124873917996486411334803247577752197089327722623494860150466526814398770516153170266969297049283162855042128981467061953319702695072143782304768752802873541261663917082459251700107141808548006369232594620190022780874098597719218051585321473926532515590354102092846659252999143537918253145452905984158176370589279069098969111643811878094353715213322614436253144901274547726957393934815469163116249288735747188240715039950094467319543161938554852076657388251396391635767231510055560372633948672082078086537349424401157996675073607111593513319591971209489647175530245313647709420946356969822266737752099451684506436238242118535348879893956731878066061078854400055082765703055874485418057788917192078814233511386629296671796434687600770479995378833878703487180218424373421122739402557176908196030920182401884270570460926225641783752652633583242406612533115294234579655695025068100183109004112453790153329661569705223792103257069370510908307894799990049993953221536227484766036136776979785673865846709366795885837887956259464648913766521995882869338018360119323685785585581955560421562508836502033220245137621582046181067051953306530606065010548871672453779428313388716313955969058320834168984760656071183471362181232462272588419902861420872849568796393254642853430753011052857138296437099903569488852851904029560473461311382638788975517885604249987483163828040468486189381895905420398898726506976202019955484126500053944282039301274816381585303964399254702016727593285743666616441109625663373054092195196751483287348089574777752783442210910731113518280460363471981856555729571447476825528578633493428584231187494400032296906977583159038580393535213588600796003420975473922967333106493956018122378128545843176055617338611267347807458506760630482294096530411183066710818930311088717281675195796753471885372293096161432040063813224658411111577583585811350185690478153689381377184728147519983505047812977185990847076219746058874232569958288925350419379582606162118423687685114183160683158679946016520577405294230536017803133572632670547903384012573059123396018801378254219270947673371919872873852480574212489211834708766296672072723256505651293331260595057777275424712416483128329820723617505746738701282095755443059683955556868611883971355220844528526400812520276655576774959696266126045652456840861392382657685833846984997787267065551918544686984694784957346226062942196245570853712727765230989554501930377321666491825781546772920052126671434632096378918523232150189761260343736840671941930377468809992968775824410478781232662531818459604538535438391144967753128642609252115376732588667226040425234910870269580996475958057946639734190640100363619040420331135793365424263035614570090112448008900208014780566037101541223288914657223931450760716706435568274377439657890679726874384730763464516775621030986040927170909512808630902973850445271828927496892121066700816485833955377359191369501531620189088874842107987068991148046692706509407620465027725286507289053285485614331608126930056937854178610969692025388650345771831766868859236814884752764984688219497397297077371871884004143231276365048145311228509900207424092558592529261030210673681543470152523487863516439762358604191941296976904052648323470099111542426012734380220893310966863678986949779940012601642276092608234930411806438291383473546797253992623387915829984864592717340592256207491053085315371829116816372193951887009577881815868504645076993439409874335144316263303172477474868979182092394808331439708406730840795893581089665647758599055637695252326536144247802308268118310377358870892406130313364773710116282146146616794040905186152603600925219472188909181073358719641421444786548995285823439470500798303885388608310357193060027711945580219119428999227223534587075662469261776631788551443502182870266856106650035310502163182060176092179846849368631612937279518730789726373537171502563787335797718081848784588665043358243770041477104149349274384575871071597315594394264125702709651251081155482479394035976811881172824721582501094960966253933953809221955919181885526780621499231727631632183398969380756168559117529984501320671293924041445938623988093812404521914848316462101473891825101090967738690664041589736104764365000680771056567184862814963711188321924456639458144914861655004956769826903089111856879869294705135248160917432430153836847072928989828460222373014526556798986277679680914697983782687643115988321090437156112997665215396354644208691975673700057387649784376862876817924974694384274652563163230055513041742273416464551278127845777724575203865437542828256714128858345444351325620544642410110379554641905811686230596447695870540721419852121067343324107567675758184569906930460475227701670056845439692340417110898889934163505851578873534308155208117720718803791040469830695786854739376564336319797868036718730796939242363214484503547763156702553900654231179201534649779290662415083288583952905426376876689688050333172278001858850697362324038947004718976193473443084374437599250341788079722358591342458131440498477017323616947197657153531977549971627856631190469126091825912498903676541769799036237552865263757337635269693443544004730671988689019681474287677908669796885225016369498567302175231325292653758964151714795595387842784998664563028788319620998304945198743963690706827626574858104391122326187940599415540632701319898957037611053236062986748037791537675115830432084987209202809297526498125691634250005229088726469252846661046653921714820801305022980526378364269597337070539227891535105688839381132497570713310295044303467159894487868471164383280506925077662745001220035262037094660234146489983902525888301486781621967751945831677187627572005054397944124599007711520515461993050983869825428464072555409274031325716326407929341833421470904125425335232480219322770753555467958716383587501815933871742360615511710131235256334858203651461418700492057043720182617331947157008675785393360786227395581857975872587441025420771054753612940474601000940954449596628814869159038990718659805636171376922272907641977551777201042764969496110562205925024202177042696221549587264539892276976603105249808557594716310758701332088614632664125911486338812202844406941694882615295776253250198703598706743804698219420563812558334364219492322759372212890564209430823525440841108645453694049692714940033197828613181861888111184082578659287574263844500599442295685864604810330153889114994869354360302218109434667640000223625505736312946262960961987605642599639461386923308371962659547392346241345977957485246478379807956931986508159776753505539189911513352522987361127791827485420086895396583594219633315028695611920122988898870060799927954111882690230789131076036176347794894320321027733594169086500719328040171638406449878717537567811853213284082165711075495282949749362146082155832056872321855740651610962748743750980922302116099826330339154694946444910045152809250897450748967603240907689836529406579201983152654106581368237919840906457124689484702093577611931399802468134052003947819498662026240089021501661638135383815150377350229660746279529103840686855690701575166241929872444827194293310048548244545807188976330032325258215812803274679620028147624318286221710543528983482082734516801861317195933247110746622285087106661177034653528395776259977446721857158161264111432717943478859908928084866949141390977167369002777585026866465405659503948678411107901161040085727445629384254941675946054871172359464291058509099502149587931121961359083158826206823321561530868337308381732793281969838750870834838804638847844188400318471269745437093732983624028751979208023218787448828728437273780178270080587824107493575148899789117397461293203510814327032514090304874622629423443275712600866425083331876886507564292716055252895449215376517514921963671810494353178583834538652556566406572513635750643532365089367904317025978781771903148679638408288102094614900797151377170990619549696400708676671023300486726314755105372317571143223174114116806228642063889062101923552235467116621374996932693217370431059872250394565749246169782609702533594750209138366737728944386964000281103440260847128990007468077648440887113413525033678773167977093727786821661178653442317322646378476978751443320953400016506921305464768909850502030150448808342618452087305309731894929164253229336124315143065782640702838984098416029503092418971209716016492656134134334222988279099217860426798124572853458013382609958771781131021673402565627440072968340661984806766158050216918337236803990279316064204368120799003162644491461902194582296909921227885539487835383056468648816555622943156731282743908264506116289428035016613366978240517701552196265227254558507386405852998303791803504328767038092521679075712040612375963276856748450791511473134400018325703449209097124358094479004624943134550289006806487042935340374360326258205357901183956490893543451013429696175452495739606214902887289327925206965353863964432253883275224996059869747598823299162635459733244451637553343774929289905811757863555556269374269109471170021654117182197505198317871371060510637955585889055688528879890847509157646390746936198815078146852621332524738376511929901561091897779220087057933964638274906806987691681974923656242260871541761004306089043779766785196618914041449252704808819714988015420577870065215940092897776013307568479669929554336561398477380603943688958876460549838714789684828053847017308711177611596635050399793438693391197898871091565417091330826076474063057114110988393880954814378284745288383680794188843426662220704387228874139478010177213922819119923654055163958934742639538248296090369002883593277458550608013179884071624465639979482757836501955142215513392819782269842786383916797150912624105487257009240700454884856929504481107380879965474815689139353809434745569721289198271770207666136024895814681191336141212587838955773571949863172108443989014239484966592517313881716026632619310653665350414730708044149391693632623737677770958503132559900957627319573086480424677012123270205337426670531424482081681303063973787366424836725398374876909806021827857862165127385635132901489035098832706172589325753639939790557291751600976154590447716922658063151110280384360173747421524760851520990161585823125715907334217365762671423904782795872815050956330928026684589376496497702329736413190609827406335310897924642421345837409011693919642504591288134034988106354008875968200544083643865166178805576089568967275315380819420773325979172784376256611843198910250074918290864751497940031607038455494653859460274524474668123146879434416109933389089926384118474252570445725174593257389895651857165759614812660203107976282541655905060424791140169579003383565748692528007430256234194982864679144763227740055294609039401775363356554719310001754300475047191448998410400158679461792416100164547165513370740739502604427695385538343975505488710997852054011751697475813449260794336895437832211724506873442319898788441285420647428097356258070669831069799352606933921356858813912148073547284632277849080870024677763036055512323866562951788537196730346347012229395816067925091532174890308408865160611190114984434123501246469280288059961342835118847154497712784733617662850621697787177438243625657117794500644777183702219991066950216567576440449979407650379999548450027106659878136038023141268369057831904607927652972776940436130230517870805465115424693952651271010529270703066730244471259739399505146284047674313637399782591845411764133279064606365841529270190302760173394748669603486949765417524293060407270050590395031485229213925755948450788679779252539317651564161971684435243697944473559642606333910551268260615957262170366985064732812667245219890605498802807828814297963366967441248059821921463395657457221022986775997467381260693670691340815594120161159601902377535255563006062479832612498812881929373434768626892192397778339107331065882568137771723283153290825250927330478507249771394483338925520811756084529665905539409655685417060011798572938139982583192936791003918440992865756059935989100029698644609747147184701015312837626311467742091455740418159088000649432378558393085308283054760767995243573916312218860575496738322431956506554608528812019023636447127037486344217272578795034284863129449163184753475314350413920961087960577309872013524840750576371992536504709085825139368634638633680428917671076021111598288755399401200760139470336617937153963061398636554922137415979051190835882900976566473007338793146789131814651093167615758213514248604422924453041131606527009743300884990346754055186406773426035834096086055337473627609356588531097609942383473822220872924644976845605795625167655740884103217313456277358560523582363895320385340248422733716391239732159954408284216666360232965456947035771848734420342277066538373875061692127680157661810954200977083636043611105924091178895403380214265239489296864398089261146354145715351943428507213534530183158756282757338982688985235577992957276452293915674775666760510878876484534936360682780505646228135988858792599409464460417052044700463151379754317371877560398159626475014109066588661621800382669899619655805872086397211769952194667898570117983324406018115756580742841829106151939176300591943144346051540477105700543390001824531177337189558576036071828605063564799790041397618089553636696031621931132502238517916720551806592635180362512145759262383693482226658955769946604919381124866090997981285718234940066155521961122072030922776462009993152442735894887105766238946938894464950939603304543408421024624010487233287500817491798755438793873814398942380117627008371960530943839400637561164585609431295175977139353960743227924892212670458081833137641658182695621058728924477400359470092686626596514220506300785920024882918608397437323538490839643261470005324235406470420894992102504047267810590836440074663800208701266642094571817029467522785400745085523777208905816839184465928294170182882330149715542352359117748186285929676050482038643431087795628929254056389466219482687110428281638939757117577869154301650586029652174595819888786804081103284327398671986213062055598552660364050462821523061545944744899088390819997387474529698107762014871340001225355222466954093152131153379157980269795557105085074738747507580687653764457825244326380461430428892359348529610582693821034980004052484070844035611678171705128133788057056434506161193304244407982603779511985486945591520519600930412710072778493015550388953603382619293437970818743209499141595933963681106275572952780042548630600545238391510689989135788200194117865356821491185282078521301255185184937115034221595422445119002073935396274002081104655302079328672547405436527175958935007163360763216147258154076420530200453401835723382926619153083540951202263291650544261236191970516138393573266937601569144299449437448568097756963031295887191611292946818849363386473927476012269641588489009657170861605981472044674286642087653347998582220906198021732116142304194777549907387385679411898246609130916917722742072333676350326783405863019301932429963972044451792881228544782119535308989101253429755247276357302262813820918074397486714535907786335301608215599113141442050914472935350222308171936635093468658586563148555758624478186201087118897606529698992693281787055764351433820601410773292610634315253371822433852635202177354407152818981376987551575745469397271504884697936195004777209705617939138289898453274262272886471088832701737232588182446584362495805925603381052156062061557132991560848920643403033952622634514542836786982880742514225674518061841495646861116354049718976821542277224794740335715274368194098920501136534001238467142965518673441537416150425632567134302476551252192180357801692403266995417460875924092070046693403965101781348578356944407604702325407555577647284507518268904182939661133101601311190773986324627782190236506603740416067249624901374332172464540974129955705291424382080760983648234659738866913499197840131080155813439791948528304367390124820824448141280954437738983200598649091595053228579145768849625786658859991798675205545580990045564611787552493701245532171701942828846174027366499784755082942280202329012216301023097721515694464279098021908266898688342630716092079140851976952355534886577434252775311972474308730436195113961190800302558783876442060850447306312992778889427291897271698905759252446796601897074829609491906487646937027507738664323919190422542902353189233772931667360869962280325571853089192844038050710300647768478632431910002239297852553723755662136447400967605394398382357646069924652600890906241059042154539279044115295803453345002562441010063595300395988644661695956263518780606885137234627079973272331346939714562855426154676506324656766202792452085813477176085216913409465203076733918411475041401689241213198268815686645614853802875393311602322925556189410429953356400957864953409351152664540244187759493169305604486864208627572011723195264050230997745676478384889734643172159806267876718380052476968840849891850861490034324034767426862459523958903585821350064509981782446360873177543788596776729195261112138591947254514003011805034378752776644027626189410175768726804281766238606804778852428874302591452470739505465251353394595987896197789110418902929438185672050709646062635417329446495766126519534957018600154126239622864138977967333290705673769621564981845068422636903678495559700260798679962610190393312637685569687670292953711625280055431007864087289392257145124811357786276649024251619902774710903359333093049483805978566288447874414698414990671237647895822632949046798120899848571635710878311918486302545016209298058292083348136384054217200561219893536693713367333924644161252231969434712064173754912163570085736943973059797097197266666422674311177621764030686813103518991122713397240368870009968629225464650063852886203938005047782769128356033725482557939129852515068299691077542576474883253414121328006267170940090982235296579579978030182824284902214707481111240186076134151503875698309186527806588966823625239378452726345304204188025084423631903833183845505223679923577529291069250432614469501098610888999146585518818735825281643025209392852580779697376208456374821144339881627100317031513344023095263519295886806908213558536801610002137408511544849126858412686958991741491338205784928006982551957402018181056412972508360703568510553317878408290000415525118657794539633175385320921497205266078312602819611648580986845875251299974040927976831766399146553861089375879522149717317281315179329044311218158710235187407572221001237687219447472093493123241070650806185623725267325407333248757544829675734500193219021991199607979893733836732425761039389853492787774739805080800155447640610535222023254094435677187945654304067358964910176107759483645408234861302547184764851895758366743997915085128580206078205544629917232020282229148869593997299742974711553718589242384938558585954074381048826246487880533042714630119415898963287926783273224561038521970111304665871005000832851773117764897352309266612345888731028835156264460236719966445547276083101187883891511493409393447500730258558147561908813987523578123313422798665035227253671712307568610450045489703600795698276263923441071465848957802414081584052295369374997106655948944592462866199635563506526234053394391421112718106910522900246574236041"), new Apfloat("3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127372458700660631558817488152092096282925409171536436789259036001133053054882046652138414695194151160943305727036575959195309218611738193261179310511854807446237996274956735188575272489122793818301194912983367336244065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960917363717872146844090122495343014654958537105079227968925892354201995611212902196086403441815981362977477130996051870721134999999837297804995105973173281609631859502445945534690830264252230825334468503526193118817101000313783875288658753320838142061717766914730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586327886593615338182796823030195203530185296899577362259941389124972177528347913151557485724245415069595082953311686172785588907509838175463746493931925506040092770167113900984882401285836160356370766010471018194295559619894676783744944825537977472684710404753464620804668425906949129331367702898915210475216205696602405803815019351125338243003558764024749647326391419927260426992279678235478163600934172164121992458631503028618297455570674983850549458858692699569092721079750930295532116534498720275596023648066549911988183479775356636980742654252786255181841757467289097777279380008164706001614524919217321721477235014144197356854816136115735255213347574184946843852332390739414333454776241686251898356948556209921922218427255025425688767179049460165346680498862723279178608578438382796797668145410095388378636095068006422512520511739298489608412848862694560424196528502221066118630674427862203919494504712371378696095636437191728746776465757396241389086583264599581339047802759009946576407895126946839835259570982582262052248940772671947826848260147699090264013639443745530506820349625245174939965143142980919065925093722169646151570985838741059788595977297549893016175392846813826868386894277415599185592524595395943104997252468084598727364469584865383673622262609912460805124388439045124413654976278079771569143599770012961608944169486855584840635342207222582848864815845602850601684273945226746767889525213852254995466672782398645659611635488623057745649803559363456817432411251507606947945109659609402522887971089314566913686722874894056010150330861792868092087476091782493858900971490967598526136554978189312978482168299894872265880485756401427047755513237964145152374623436454285844479526586782105114135473573952311342716610213596953623144295248493718711014576540359027993440374200731057853906219838744780847848968332144571386875194350643021845319104848100537061468067491927819119793995206141966342875444064374512371819217999839101591956181467514269123974894090718649423196156794520809514655022523160388193014209376213785595663893778708303906979207734672218256259966150142150306803844773454920260541466592520149744285073251866600213243408819071048633173464965145390579626856100550810665879699816357473638405257145910289706414011097120628043903975951567715770042033786993600723055876317635942187312514712053292819182618612586732157919841484882916447060957527069572209175671167229109816909152801735067127485832228718352093539657251210835791513698820914442100675103346711031412671113699086585163983150197016515116851714376576183515565088490998985998238734552833163550764791853589322618548963213293308985706420467525907091548141654985946163718027098199430992448895757128289059232332609729971208443357326548938239119325974636673058360414281388303203824903758985243744170291327656180937734440307074692112019130203303801976211011004492932151608424448596376698389522868478312355265821314495768572624334418930396864262434107732269780280731891544110104468232527162010526522721116603966655730925471105578537634668206531098965269186205647693125705863566201855810072936065987648611791045334885034611365768675324944166803962657978771855608455296541266540853061434443185867697514566140680070023787765913440171274947042056223053899456131407112700040785473326993908145466464588079727082668306343285878569830523580893306575740679545716377525420211495576158140025012622859413021647155097925923099079654737612551765675135751782966645477917450112996148903046399471329621073404375189573596145890193897131117904297828564750320319869151402870808599048010941214722131794764777262241425485454033215718530614228813758504306332175182979866223717215916077166925474873898665494945011465406284336639379003976926567214638530673609657120918076383271664162748888007869256029022847210403172118608204190004229661711963779213375751149595015660496318629472654736425230817703675159067350235072835405670403867435136222247715891504953098444893330963408780769325993978054193414473774418426312986080998886874132604721569516239658645730216315981931951673538129741677294786724229246543668009806769282382806899640048243540370141631496589794092432378969070697794223625082216889573837986230015937764716512289357860158816175578297352334460428151262720373431465319777741603199066554187639792933441952154134189948544473456738316249934191318148092777710386387734317720754565453220777092120190516609628049092636019759882816133231666365286193266863360627356763035447762803504507772355471058595487027908143562401451718062464362679456127531813407833033625423278394497538243720583531147711992606381334677687969597030983391307710987040859133746414428227726346594704745878477872019277152807317679077071572134447306057007334924369311383504931631284042512192565179806941135280131470130478164378851852909285452011658393419656213491434159562586586557055269049652098580338507224264829397285847831630577775606888764462482468579260395352773480304802900587607582510474709164396136267604492562742042083208566119062545433721315359584506877246029016187667952406163425225771954291629919306455377991403734043287526288896399587947572917464263574552540790914513571113694109119393251910760208252026187985318877058429725916778131496990090192116971737278476847268608490033770242429165130050051683233643503895170298939223345172201381280696501178440874519601212285993716231301711444846409038906449544400619869075485160263275052983491874078668088183385102283345085048608250393021332197155184306354550076682829493041377655279397517546139539846833936383047461199665385815384205685338621867252334028308711232827892125077126294632295639898989358211674562701021835646220134967151881909730381198004973407239610368540664319395097901906996395524530054505806855019567302292191393391856803449039820595510022635353619204199474553859381023439554495977837790237421617271117236434354394782218185286240851400666044332588856986705431547069657474585503323233421073015459405165537906866273337995851156257843229882737231989875714159578111963583300594087306812160287649628674460477464915995054973742562690104903778198683593814657412680492564879855614537234786733039046883834363465537949864192705638729317487233208376011230299113679386270894387993620162951541337142489283072201269014754668476535761647737946752004907571555278196536213239264061601363581559074220202031872776052772190055614842555187925303435139844253223415762336106425063904975008656271095359194658975141310348227693062474353632569160781547818115284366795706110861533150445212747392454494542368288606134084148637767009612071512491404302725386076482363414334623518975766452164137679690314950191085759844239198629164219399490723623464684411739403265918404437805133389452574239950829659122850855582157250310712570126683024029295252201187267675622041542051618416348475651699981161410100299607838690929160302884002691041407928862150784245167090870006992821206604183718065355672525325675328612910424877618258297651579598470356222629348600341587229805349896502262917487882027342092222453398562647669149055628425039127577102840279980663658254889264880254566101729670266407655904290994568150652653053718294127033693137851786090407086671149655834343476933857817113864558736781230145876871266034891390956200993936103102916161528813843790990423174733639480457593149314052976347574811935670911013775172100803155902485309066920376719220332290943346768514221447737939375170344366199104033751117354719185504644902636551281622882446257591633303910722538374218214088350865739177150968288747826569959957449066175834413752239709683408005355984917541738188399944697486762655165827658483588453142775687900290951702835297163445621296404352311760066510124120065975585127617858382920419748442360800719304576189323492292796501987518721272675079812554709589045563579212210333466974992356302549478024901141952123828153091140790738602515227429958180724716259166854513331239480494707911915326734302824418604142636395480004480026704962482017928964766975831832713142517029692348896276684403232609275249603579964692565049368183609003238092934595889706953653494060340216654437558900456328822505452556405644824651518754711962184439658253375438856909411303150952617937800297412076651479394259029896959469955657612186561967337862362561252163208628692221032748892186543648022967807057656151446320469279068212073883778142335628236089632080682224680122482611771858963814091839036736722208883215137556003727983940041529700287830766709444745601345564172543709069793961225714298946715435784687886144458123145935719849225284716050492212424701412147805734551050080190869960330276347870810817545011930714122339086639383395294257869050764310063835198343893415961318543475464955697810382930971646514384070070736041123735998434522516105070270562352660127648483084076118301305279320542746286540360367453286510570658748822569815793678976697422057505968344086973502014102067235850200724522563265134105592401902742162484391403599895353945909440704691209140938700126456001623742880210927645793106579229552498872758461012648369998922569596881592056001016552563756785667227966198857827948488558343975187445455129656344348039664205579829368043522027709842942325330225763418070394769941597915945300697521482933665556615678736400536665641654732170439035213295435291694145990416087532018683793702348886894791510716378529023452924407736594956305100742108714261349745956151384987137570471017879573104229690666702144986374645952808243694457897723300487647652413390759204340196340391147320233807150952220106825634274716460243354400515212669324934196739770415956837535551667302739007497297363549645332888698440611964961627734495182736955882207573551766515898551909866653935494810688732068599075407923424023009259007017319603622547564789406475483466477604114632339056513433068449539790709030234604614709616968868850140834704054607429586991382966824681857103188790652870366508324319744047718556789348230894310682870272280973624809399627060747264553992539944280811373694338872940630792615959954626246297070625948455690347119729964090894180595343932512362355081349490043642785271383159125689892951964272875739469142725343669415323610045373048819855170659412173524625895487301676002988659257866285612496655235338294287854253404830833070165372285635591525347844598183134112900199920598135220511733658564078264849427644113763938669248031183644536985891754426473998822846218449008777697763127957226726555625962825427653183001340709223343657791601280931794017185985999338492354956400570995585611349802524990669842330173503580440811685526531170995708994273287092584878944364600504108922669178352587078595129834417295351953788553457374260859029081765155780390594640873506123226112009373108048548526357228257682034160504846627750450031262008007998049254853469414697751649327095049346393824322271885159740547021482897111777923761225788734771881968254629812686858170507402725502633290449762778944236216741191862694396506715157795867564823993917604260176338704549901761436412046921823707648878341968968611815581587360629386038101712158552726683008238340465647588040513808016336388742163714064354955618689641122821407533026551004241048967835285882902436709048871181909094945331442182876618103100735477054981596807720094746961343609286148494178501718077930681085469000944589952794243981392135055864221964834915126390128038320010977386806628779239718014613432445726400973742570073592100315415089367930081699805365202760072774967458400283624053460372634165542590276018348403068113818551059797056640075094260878857357960373245141467867036880988060971642584975951380693094494015154222219432913021739125383559150310033303251117491569691745027149433151558854039221640972291011290355218157628232831823425483261119128009282525619020526301639114772473314857391077758744253876117465786711694147764214411112635835538713610110232679877564102468240322648346417663698066378576813492045302240819727856471983963087815432211669122464159117767322532643356861461865452226812688726844596844241610785401676814208088502800541436131462308210259417375623899420757136275167457318918945628352570441335437585753426986994725470316566139919996826282472706413362221789239031760854289437339356188916512504244040089527198378738648058472689546243882343751788520143956005710481194988423906061369573423155907967034614914344788636041031823507365027785908975782727313050488939890099239135033732508559826558670892426124294736701939077271307068691709264625484232407485503660801360466895118400936686095463250021458529309500009071510582362672932645373821049387249966993394246855164832611341461106802674466373343753407642940266829738652209357016263846485285149036293201991996882851718395366913452224447080459239660281715655156566611135982311225062890585491450971575539002439315351909021071194573002438801766150352708626025378817975194780610137150044899172100222013350131060163915415895780371177927752259787428919179155224171895853616805947412341933984202187456492564434623925319531351033114763949119950728584306583619353693296992898379149419394060857248639688369032655643642166442576079147108699843157337496488352927693282207629472823815374099615455987982598910937171262182830258481123890119682214294576675807186538065064870261338928229949725745303328389638184394477077940228435988341003583854238973542439564755568409522484455413923941000162076936368467764130178196593799715574685419463348937484391297423914336593604100352343777065888677811394986164787471407932638587386247328896456435987746676384794665040741118256583788784548581489629612739984134427260860618724554523606431537101127468097787044640947582803487697589483282412392929605829486191966709189580898332012103184303401284951162035342801441276172858302435598300320420245120728725355811958401491809692533950757784000674655260314461670508276827722235341911026341631571474061238504258459884199076112872580591139356896014316682831763235673254170734208173322304629879928049085140947903688786878949305469557030726190095020764334933591060245450864536289354568629585313153371838682656178622736371697577418302398600659148161640494496501173213138957470620884748023653710311508984279927544268532779743113951435741722197597993596852522857452637962896126915723579866205734083757668738842664059909935050008133754324546359675048442352848747014435454195762584735642161981340734685411176688311865448937769795665172796623267148103386439137518659467300244345005449953997423723287124948347060440634716063258306498297955101095418362350303094530973358344628394763047756450150085075789495489313939448992161255255977014368589435858775263796255970816776438001254365023714127834679261019955852247172201777237004178084194239487254068015560359983905489857235467456423905858502167190313952629445543913166313453089390620467843877850542393905247313620129476918749751910114723152893267725339181466073000890277689631148109022097245207591672970078505807171863810549679731001678708506942070922329080703832634534520380278609905569001341371823683709919495164896007550493412678764367463849020639640197666855923356546391383631857456981471962108410809618846054560390384553437291414465134749407848844237721751543342603066988317683310011331086904219390310801437843341513709243530136776310849135161564226984750743032971674696406665315270353254671126675224605511995818319637637076179919192035795820075956053023462677579439363074630569010801149427141009391369138107258137813578940055995001835425118417213605572752210352680373572652792241737360575112788721819084490061780138897107708229310027976659358387589093956881485602632243937265624727760378908144588378550197028437793624078250527048758164703245812908783952324532378960298416692254896497156069811921865849267704039564812781021799132174163058105545988013004845629976511212415363745150056350701278159267142413421033015661653560247338078430286552572227530499988370153487930080626018096238151613669033411113865385109193673938352293458883225508870645075394739520439680790670868064450969865488016828743437861264538158342807530618454859037982179945996811544197425363443996029025100158882721647450068207041937615845471231834600726293395505482395571372568402322682130124767945226448209102356477527230820810635188991526928891084555711266039650343978962782500161101532351605196559042118449499077899920073294769058685778787209829013529566139788848605097860859570177312981553149516814671769597609942100361835591387778176984587581044662839988060061622984861693533738657877359833616133841338536842119789389001852956919678045544828584837011709672125353387586215823101331038776682721157269495181795897546939926421979155233857662316762754757035469941489290413018638611943919628388705436777432242768091323654494853667680000010652624854730558615989991401707698385483188750142938908995068545307651168033373222651756622075269517914422528081651716677667279303548515420402381746089232839170327542575086765511785939500279338959205766827896776445318404041855401043513483895312013263783692835808271937831265496174599705674507183320650345566440344904536275600112501843356073612227659492783937064784264567633881880756561216896050416113903906396016202215368494109260538768871483798955999911209916464644119185682770045742434340216722764455893301277815868695250694993646101756850601671453543158148010545886056455013320375864548584032402987170934809105562116715468484778039447569798042631809917564228098739987669732376957370158080682290459921236616890259627304306793165311494017647376938735140933618332161428021497633991898354848756252987524238730775595559554651963944018218409984124898262367377146722606163364329640633572810707887581640438148501884114318859882769449011932129682715888413386943468285900666408063140777577257056307294004929403024204984165654797367054855804458657202276378404668233798528271057843197535417950113472736257740802134768260450228515797957976474670228409995616015691089038458245026792659420555039587922981852648007068376504183656209455543461351341525700659748819163413595567196496540321872716026485930490397874895890661272507948282769389535217536218507962977851461884327192232238101587444505286652380225328438913752738458923844225354726530981715784478342158223270206902872323300538621634798850946954720047952311201504329322662827276321779088400878614802214753765781058197022263097174950721272484794781695729614236585957820908307332335603484653187302930266596450137183754288975579714499246540386817992138934692447419850973346267933210726868707680626399193619650440995421676278409146698569257150743157407938053239252394775574415918458215625181921552337096074833292349210345146264374498055961033079941453477845746999921285999993996122816152193148887693880222810830019860165494165426169685867883726095877456761825072759929508931805218729246108676399589161458550583972742098090978172932393010676638682404011130402470073508578287246271349463685318154696904669686939254725194139929146524238577625500474852954768147954670070503479995888676950161249722820403039954632788306959762493615101024365553522306906129493885990157346610237122354789112925476961760050479749280607212680392269110277722610254414922157650450812067717357120271802429681062037765788371669091094180744878140490755178203856539099104775941413215432844062503018027571696508209642734841469572639788425600845312140659358090412711359200419759851362547961606322887361813673732445060792441176399759746193835845749159880976674470930065463424234606342374746660804317012600520559284936959414340814685298150539471789004518357551541252235905906872648786357525419112888773717663748602766063496035367947026923229718683277173932361920077745221262475186983349515101986426988784717193966497690708252174233656627259284406204302141137199227852699846988477023238238400556555178890876613601304770984386116870523105531491625172837327286760072481729876375698163354150746088386636406934704372066886512756882661497307886570156850169186474885416791545965072342877306998537139043002665307839877638503238182155355973235306860430106757608389086270498418885951380910304235957824951439885901131858358406674723702971497850841458530857813391562707603563907639473114554958322669457024941398316343323789759556808568362972538679132750555425244919435891284050452269538121791319145135009938463117740179715122837854601160359554028644059024964669307077690554810288502080858008781157738171917417760173307385547580060560143377432990127286772530431825197579167929699650414607066457125888346979796429316229655201687973000356463045793088403274807718115553309098870255052076804630346086581653948769519600440848206596737947316808641564565053004988161649057883115434548505266006982309315777650037807046612647060214575057932709620478256152471459189652236083966456241051955105223572397395128818164059785914279148165426328920042816091369377737222999833270820829699557377273756676155271139225880552018988762011416800546873655806334716037342917039079863965229613128017826797172898229360702880690877686605932527463784053976918480820410219447197138692560841624511239806201131845412447820501107987607171556831540788654390412108730324020106853419472304766667217498698685470767812051247367924791931508564447753798537997322344561227858432968466475133365736923872014647236794278700425032555899268843495928761240075587569464137056251400117971331662071537154360068764773186755871487839890810742953094106059694431584775397009439883949144323536685392099468796450665339857388878661476294434140104988899316005120767810358861166020296119363968213496075011164983278563531614516845769568710900299976984126326650234771672865737857908574664607722834154031144152941880478254387617707904300015669867767957609099669360755949651527363498118964130433116627747123388174060373174397054067031096767657486953587896700319258662594105105335843846560233917967492678447637084749783336555790073841914731988627135259546251816043422537299628632674968240580602964211463864368642247248872834341704415734824818333016405669596688667695634914163284264149745333499994800026699875888159350735781519588990053951208535103572613736403436753471410483601754648830040784641674521673719048310967671134434948192626811107399482506073949507350316901973185211955263563258433909982249862406703107683184466072912487475403161796994113973877658998685541703188477886759290260700432126661791922352093822787888098863359911608192353555704646349113208591897961327913197564909760001399623444553501434642686046449586247690943470482932941404111465409239883444351591332010773944111840741076849810663472410482393582740194493566516108846312567852977697346843030614624180358529331597345830384554103370109167677637427621021370135485445092630719011473184857492331816720721372793556795284439254815609137281284063330393735624200160456645574145881660521666087387480472433912129558777639069690370788285277538940524607584962315743691711317613478388271941686066257210368513215664780014767523103935786068961112599602818393095487090590738613519145918195102973278755710497290114871718971800469616977700179139196137914171627070189584692143436967629274591099400600849835684252019155937037010110497473394938778859894174330317853487076032219829705797511914405109942358830345463534923498268836240433272674155403016195056806541809394099820206099941402168909007082133072308966211977553066591881411915778362729274615618571037217247100952142369648308641025928874579993223749551912219519034244523075351338068568073544649951272031744871954039761073080602699062580760202927314552520780799141842906388443734996814582733720726639176702011830046481900024130835088465841521489912761065137415394356572113903285749187690944137020905170314877734616528798482353382972601361109845148418238081205409961252745808810994869722161285248974255555160763716750548961730168096138038119143611439921063800508321409876045993093248510251682944672606661381517457125597549535802399831469822036133808284993567055755247129027453977621404931820146580080215665360677655087838043041343105918046068008345911366408348874080057412725867047922583191274157390809143831384564241509408491339180968402511639919368532255573389669537490266209232613188558915808324555719484538756287861288590041060060737465014026278240273469625282171749415823317492396835301361786536737606421667781377399510065895288774276626368418306801908046098498094697636673356622829151323527888061577682781595886691802389403330764419124034120223163685778603572769415417788264352381319050280870185750470463129333537572853866058889045831114507739429352019943219711716422350056440429798920815943071670198574692738486538334361457946341759225738985880016980147574205429958012429581054565108310462972829375841611625325625165724980784920998979906200359365099347215829651741357984910471116607915874369865412223483418877229294463351786538567319625598520260729476740726167671455736498121056777168934849176607717052771876011999081441130586455779105256843048114402619384023224709392498029335507318458903553971330884461741079591625117148648744686112476054286734367090466784686702740918810142497111496578177242793470702166882956108777944050484375284433751088282647719785400065097040330218625561473321177711744133502816088403517814525419643203095760186946490886815452856213469883554445602495566684366029221951248309106053772019802183101032704178386654471812603971906884623708575180800353270471856594994761242481109992886791589690495639476246084240659309486215076903149870206735338483495508363660178487710608098042692471324100094640143736032656451845667924566695510015022983307984960799498824970617236744936122622296179081431141466094123415935930958540791390872083227335495720807571651718765994498569379562387555161757543809178052802946420044721539628074636021132942559160025707356281263873310600589106524570802447493754318414940148211999627645310680066311838237616396631809314446712986155275982014514102756006892975024630401735148919457636078935285550531733141645705049964438909363084387448478396168405184527328840323452024705685164657164771393237755172947951261323982296023945485797545865174587877133181387529598094121742273003522965080891777050682592488223221549380483714547816472139768209633205083056479204820859204754998573203888763916019952409189389455767687497308569559580106595265030362661597506622250840674288982659075106375635699682115109496697445805472886936310203678232501823237084597901115484720876182124778132663304120762165873129708112307581598212486398072124078688781145016558251361789030708608701989758898074566439551574153631931919810705753366337380382721527988493503974800158905194208797113080512339332219034662499171691509485414018710603546037946433790058909577211808044657439628061867178610171567409676620802957665770512912099079443046328929473061595104309022214393718495606340561893425130572682914657832933405246350289291754708725648426003496296116541382300773133272983050016025672401418515204189070115428857992081219844931569990591820118197335001261877280368124819958770702075324063612593134385955425477819611429351635612234966615226147353996740515849986035529533292457523888101362023476246690558164389678630976273655047243486430712184943734853006063876445662721866617012381277156213797461498613287441177145524447089971445228856629424402301847912054784985745216346964489738920624019435183100882834802492490854030778638751659113028739587870981007727182718745290139728366148421428717055317965430765045343246005363614726181809699769334862640774351999286863238350887566835950972655748154319401955768504372480010204137498318722596773871549583997184449072791419658459300839426370208756353982169620553248032122674989114026785285996734052420310917978999057188219493913207534317079800237365909853755202389116434671855829068537118979526262344924833924963424497146568465912489185566295893299090352392333336474352037077010108438800329075983421701855422838616172104176030116459187805393674474720599850235828918336929223373239994804371084196594731626548257480994825099918330069765693671596893644933488647442135008407006608835972350395323401795825570360169369909886711321097988970705172807558551912699306730992507040702455685077867906947661262980822516331363995211709845280926303759224267425755998928927837047444521893632034894155210445972618838003006776179313813991620580627016510244588692476492468919246121253102757313908404700071435613623169923716948481325542009145304103713545329662063921054798243921251725401323149027405858920632175894943454890684639931375709103463327141531622328055229729795380188016285907357295541627886764982741861642187898857410716490691918511628152854867941736389066538857642291583425006736124538491606741373401735727799563410433268835695078149313780073623541800706191802673285511919426760912210359874692411728374931261633950012395992405084543756985079570462226646190001035004901830341535458428337643781119885563187777925372011667185395418359844383052037628194407615941068207169703022851522505731260930468984234331527321313612165828080752126315477306044237747535059522871744026663891488171730864361113890694202790881431194487994171540421034121908470940802540239329429454938786402305129271190975135360009219711054120966831115163287054230284700731206580326264171161659576132723515666625366727189985341998952368848309993027574199164638414270779887088742292770538912271724863220288984251252872178260305009945108247835729056919885554678860794628053712270424665431921452817607414824038278358297193010178883456741678113989547504483393146896307633966572267270433932167454218245570625247972199786685427989779923395790575818906225254735822052364248507834071101449804787266919901864388229323053823185597328697809222535295910173414073348847610055640182423921926950620831838145469839236646136398910121021770959767049083050818547041946643713122996923588953849301363565761861060622287055994233716310212784574464639897381885667462608794820186474876727272220626764653380998019668836809941590757768526398651462533363124505364026105696055131838131742611844201890888531963569869627950367384243130113317533053298020166888174813429886815855778103432317530647849832106297184251843855344276201282345707169885305183261796411785796088881503296022907056144762209150947390359466469162353968092013945781758910889319921122600739281491694816152738427362642980982340632002440244958944561291670495082358124873917996486411334803247577752197089327722623494860150466526814398770516153170266969297049283162855042128981467061953319702695072143782304768752802873541261663917082459251700107141808548006369232594620190022780874098597719218051585321473926532515590354102092846659252999143537918253145452905984158176370589279069098969111643811878094353715213322614436253144901274547726957393934815469163116249288735747188240715039950094467319543161938554852076657388251396391635767231510055560372633948672082078086537349424401157996675073607111593513319591971209489647175530245313647709420946356969822266737752099451684506436238242118535348879893956731878066061078854400055082765703055874485418057788917192078814233511386629296671796434687600770479995378833878703487180218424373421122739402557176908196030920182401884270570460926225641783752652633583242406612533115294234579655695025068100183109004112453790153329661569705223792103257069370510908307894799990049993953221536227484766036136776979785673865846709366795885837887956259464648913766521995882869338018360119323685785585581955560421562508836502033220245137621582046181067051953306530606065010548871672453779428313388716313955969058320834168984760656071183471362181232462272588419902861420872849568796393254642853430753011052857138296437099903569488852851904029560473461311382638788975517885604249987483163828040468486189381895905420398898726506976202019955484126500053944282039301274816381585303964399254702016727593285743666616441109625663373054092195196751483287348089574777752783442210910731113518280460363471981856555729571447476825528578633493428584231187494400032296906977583159038580393535213588600796003420975473922967333106493956018122378128545843176055617338611267347807458506760630482294096530411183066710818930311088717281675195796753471885372293096161432040063813224658411111577583585811350185690478153689381377184728147519983505047812977185990847076219746058874232569958288925350419379582606162118423687685114183160683158679946016520577405294230536017803133572632670547903384012573059123396018801378254219270947673371919872873852480574212489211834708766296672072723256505651293331260595057777275424712416483128329820723617505746738701282095755443059683955556868611883971355220844528526400812520276655576774959696266126045652456840861392382657685833846984997787267065551918544686984694784957346226062942196245570853712727765230989554501930377321666491825781546772920052126671434632096378918523232150189761260343736840671941930377468809992968775824410478781232662531818459604538535438391144967753128642609252115376732588667226040425234910870269580996475958057946639734190640100363619040420331135793365424263035614570090112448008900208014780566037101541223288914657223931450760716706435568274377439657890679726874384730763464516775621030986040927170909512808630902973850445271828927496892121066700816485833955377359191369501531620189088874842107987068991148046692706509407620465027725286507289053285485614331608126930056937854178610969692025388650345771831766868859236814884752764984688219497397297077371871884004143231276365048145311228509900207424092558592529261030210673681543470152523487863516439762358604191941296976904052648323470099111542426012734380220893310966863678986949779940012601642276092608234930411806438291383473546797253992623387915829984864592717340592256207491053085315371829116816372193951887009577881815868504645076993439409874335144316263303172477474868979182092394808331439708406730840795893581089665647758599055637695252326536144247802308268118310377358870892406130313364773710116282146146616794040905186152603600925219472188909181073358719641421444786548995285823439470500798303885388608310357193060027711945580219119428999227223534587075662469261776631788551443502182870266856106650035310502163182060176092179846849368631612937279518730789726373537171502563787335797718081848784588665043358243770041477104149349274384575871071597315594394264125702709651251081155482479394035976811881172824721582501094960966253933953809221955919181885526780621499231727631632183398969380756168559117529984501320671293924041445938623988093812404521914848316462101473891825101090967738690664041589736104764365000680771056567184862814963711188321924456639458144914861655004956769826903089111856879869294705135248160917432430153836847072928989828460222373014526556798986277679680914697983782687643115988321090437156112997665215396354644208691975673700057387649784376862876817924974694384274652563163230055513041742273416464551278127845777724575203865437542828256714128858345444351325620544642410110379554641905811686230596447695870540721419852121067343324107567675758184569906930460475227701670056845439692340417110898889934163505851578873534308155208117720718803791040469830695786854739376564336319797868036718730796939242363214484503547763156702553900654231179201534649779290662415083288583952905426376876689688050333172278001858850697362324038947004718976193473443084374437599250341788079722358591342458131440498477017323616947197657153531977549971627856631190469126091825912498903676541769799036237552865263757337635269693443544004730671988689019681474287677908669796885225016369498567302175231325292653758964151714795595387842784998664563028788319620998304945198743963690706827626574858104391122326187940599415540632701319898957037611053236062986748037791537675115830432084987209202809297526498125691634250005229088726469252846661046653921714820801305022980526378364269597337070539227891535105688839381132497570713310295044303467159894487868471164383280506925077662745001220035262037094660234146489983902525888301486781621967751945831677187627572005054397944124599007711520515461993050983869825428464072555409274031325716326407929341833421470904125425335232480219322770753555467958716383587501815933871742360615511710131235256334858203651461418700492057043720182617331947157008675785393360786227395581857975872587441025420771054753612940474601000940954449596628814869159038990718659805636171376922272907641977551777201042764969496110562205925024202177042696221549587264539892276976603105249808557594716310758701332088614632664125911486338812202844406941694882615295776253250198703598706743804698219420563812558334364219492322759372212890564209430823525440841108645453694049692714940033197828613181861888111184082578659287574263844500599442295685864604810330153889114994869354360302218109434667640000223625505736312946262960961987605642599639461386923308371962659547392346241345977957485246478379807956931986508159776753505539189911513352522987361127791827485420086895396583594219633315028695611920122988898870060799927954111882690230789131076036176347794894320321027733594169086500719328040171638406449878717537567811853213284082165711075495282949749362146082155832056872321855740651610962748743750980922302116099826330339154694946444910045152809250897450748967603240907689836529406579201983152654106581368237919840906457124689484702093577611931399802468134052003947819498662026240089021501661638135383815150377350229660746279529103840686855690701575166241929872444827194293310048548244545807188976330032325258215812803274679620028147624318286221710543528983482082734516801861317195933247110746622285087106661177034653528395776259977446721857158161264111432717943478859908928084866949141390977167369002777585026866465405659503948678411107901161040085727445629384254941675946054871172359464291058509099502149587931121961359083158826206823321561530868337308381732793281969838750870834838804638847844188400318471269745437093732983624028751979208023218787448828728437273780178270080587824107493575148899789117397461293203510814327032514090304874622629423443275712600866425083331876886507564292716055252895449215376517514921963671810494353178583834538652556566406572513635750643532365089367904317025978781771903148679638408288102094614900797151377170990619549696400708676671023300486726314755105372317571143223174114116806228642063889062101923552235467116621374996932693217370431059872250394565749246169782609702533594750209138366737728944386964000281103440260847128990007468077648440887113413525033678773167977093727786821661178653442317322646378476978751443320953400016506921305464768909850502030150448808342618452087305309731894929164253229336124315143065782640702838984098416029503092418971209716016492656134134334222988279099217860426798124572853458013382609958771781131021673402565627440072968340661984806766158050216918337236803990279316064204368120799003162644491461902194582296909921227885539487835383056468648816555622943156731282743908264506116289428035016613366978240517701552196265227254558507386405852998303791803504328767038092521679075712040612375963276856748450791511473134400018325703449209097124358094479004624943134550289006806487042935340374360326258205357901183956490893543451013429696175452495739606214902887289327925206965353863964432253883275224996059869747598823299162635459733244451637553343774929289905811757863555556269374269109471170021654117182197505198317871371060510637955585889055688528879890847509157646390746936198815078146852621332524738376511929901561091897779220087057933964638274906806987691681974923656242260871541761004306089043779766785196618914041449252704808819714988015420577870065215940092897776013307568479669929554336561398477380603943688958876460549838714789684828053847017308711177611596635050399793438693391197898871091565417091330826076474063057114110988393880954814378284745288383680794188843426662220704387228874139478010177213922819119923654055163958934742639538248296090369002883593277458550608013179884071624465639979482757836501955142215513392819782269842786383916797150912624105487257009240700454884856929504481107380879965474815689139353809434745569721289198271770207666136024895814681191336141212587838955773571949863172108443989014239484966592517313881716026632619310653665350414730708044149391693632623737677770958503132559900957627319573086480424677012123270205337426670531424482081681303063973787366424836725398374876909806021827857862165127385635132901489035098832706172589325753639939790557291751600976154590447716922658063151110280384360173747421524760851520990161585823125715907334217365762671423904782795872815050956330928026684589376496497702329736413190609827406335310897924642421345837409011693919642504591288134034988106354008875968200544083643865166178805576089568967275315380819420773325979172784376256611843198910250074918290864751497940031607038455494653859460274524474668123146879434416109933389089926384118474252570445725174593257389895651857165759614812660203107976282541655905060424791140169579003383565748692528007430256234194982864679144763227740055294609039401775363356554719310001754300475047191448998410400158679461792416100164547165513370740739502604427695385538343975505488710997852054011751697475813449260794336895437832211724506873442319898788441285420647428097356258070669831069799352606933921356858813912148073547284632277849080870024677763036055512323866562951788537196730346347012229395816067925091532174890308408865160611190114984434123501246469280288059961342835118847154497712784733617662850621697787177438243625657117794500644777183702219991066950216567576440449979407650379999548450027106659878136038023141268369057831904607927652972776940436130230517870805465115424693952651271010529270703066730244471259739399505146284047674313637399782591845411764133279064606365841529270190302760173394748669603486949765417524293060407270050590395031485229213925755948450788679779252539317651564161971684435243697944473559642606333910551268260615957262170366985064732812667245219890605498802807828814297963366967441248059821921463395657457221022986775997467381260693670691340815594120161159601902377535255563006062479832612498812881929373434768626892192397778339107331065882568137771723283153290825250927330478507249771394483338925520811756084529665905539409655685417060011798572938139982583192936791003918440992865756059935989100029698644609747147184701015312837626311467742091455740418159088000649432378558393085308283054760767995243573916312218860575496738322431956506554608528812019023636447127037486344217272578795034284863129449163184753475314350413920961087960577309872013524840750576371992536504709085825139368634638633680428917671076021111598288755399401200760139470336617937153963061398636554922137415979051190835882900976566473007338793146789131814651093167615758213514248604422924453041131606527009743300884990346754055186406773426035834096086055337473627609356588531097609942383473822220872924644976845605795625167655740884103217313456277358560523582363895320385340248422733716391239732159954408284216666360232965456947035771848734420342277066538373875061692127680157661810954200977083636043611105924091178895403380214265239489296864398089261146354145715351943428507213534530183158756282757338982688985235577992957276452293915674775666760510878876484534936360682780505646228135988858792599409464460417052044700463151379754317371877560398159626475014109066588661621800382669899619655805872086397211769952194667898570117983324406018115756580742841829106151939176300591943144346051540477105700543390001824531177337189558576036071828605063564799790041397618089553636696031621931132502238517916720551806592635180362512145759262383693482226658955769946604919381124866090997981285718234940066155521961122072030922776462009993152442735894887105766238946938894464950939603304543408421024624010487233287500817491798755438793873814398942380117627008371960530943839400637561164585609431295175977139353960743227924892212670458081833137641658182695621058728924477400359470092686626596514220506300785920024882918608397437323538490839643261470005324235406470420894992102504047267810590836440074663800208701266642094571817029467522785400745085523777208905816839184465928294170182882330149715542352359117748186285929676050482038643431087795628929254056389466219482687110428281638939757117577869154301650586029652174595819888786804081103284327398671986213062055598552660364050462821523061545944744899088390819997387474529698107762014871340001225355222466954093152131153379157980269795557105085074738747507580687653764457825244326380461430428892359348529610582693821034980004052484070844035611678171705128133788057056434506161193304244407982603779511985486945591520519600930412710072778493015550388953603382619293437970818743209499141595933963681106275572952780042548630600545238391510689989135788200194117865356821491185282078521301255185184937115034221595422445119002073935396274002081104655302079328672547405436527175958935007163360763216147258154076420530200453401835723382926619153083540951202263291650544261236191970516138393573266937601569144299449437448568097756963031295887191611292946818849363386473927476012269641588489009657170861605981472044674286642087653347998582220906198021732116142304194777549907387385679411898246609130916917722742072333676350326783405863019301932429963972044451792881228544782119535308989101253429755247276357302262813820918074397486714535907786335301608215599113141442050914472935350222308171936635093468658586563148555758624478186201087118897606529698992693281787055764351433820601410773292610634315253371822433852635202177354407152818981376987551575745469397271504884697936195004777209705617939138289898453274262272886471088832701737232588182446584362495805925603381052156062061557132991560848920643403033952622634514542836786982880742514225674518061841495646861116354049718976821542277224794740335715274368194098920501136534001238467142965518673441537416150425632567134302476551252192180357801692403266995417460875924092070046693403965101781348578356944407604702325407555577647284507518268904182939661133101601311190773986324627782190236506603740416067249624901374332172464540974129955705291424382080760983648234659738866913499197840131080155813439791948528304367390124820824448141280954437738983200598649091595053228579145768849625786658859991798675205545580990045564611787552493701245532171701942828846174027366499784755082942280202329012216301023097721515694464279098021908266898688342630716092079140851976952355534886577434252775311972474308730436195113961190800302558783876442060850447306312992778889427291897271698905759252446796601897074829609491906487646937027507738664323919190422542902353189233772931667360869962280325571853089192844038050710300647768478632431910002239297852553723755662136447400967605394398382357646069924652600890906241059042154539279044115295803453345002562441010063595300395988644661695956263518780606885137234627079973272331346939714562855426154676506324656766202792452085813477176085216913409465203076733918411475041401689241213198268815686645614853802875393311602322925556189410429953356400957864953409351152664540244187759493169305604486864208627572011723195264050230997745676478384889734643172159806267876718380052476968840849891850861490034324034767426862459523958903585821350064509981782446360873177543788596776729195261112138591947254514003011805034378752776644027626189410175768726804281766238606804778852428874302591452470739505465251353394595987896197789110418902929438185672050709646062635417329446495766126519534957018600154126239622864138977967333290705673769621564981845068422636903678495559700260798679962610190393312637685569687670292953711625280055431007864087289392257145124811357786276649024251619902774710903359333093049483805978566288447874414698414990671237647895822632949046798120899848571635710878311918486302545016209298058292083348136384054217200561219893536693713367333924644161252231969434712064173754912163570085736943973059797097197266666422674311177621764030686813103518991122713397240368870009968629225464650063852886203938005047782769128356033725482557939129852515068299691077542576474883253414121328006267170940090982235296579579978030182824284902214707481111240186076134151503875698309186527806588966823625239378452726345304204188025084423631903833183845505223679923577529291069250432614469501098610888999146585518818735825281643025209392852580779697376208456374821144339881627100317031513344023095263519295886806908213558536801610002137408511544849126858412686958991741491338205784928006982551957402018181056412972508360703568510553317878408290000415525118657794539633175385320921497205266078312602819611648580986845875251299974040927976831766399146553861089375879522149717317281315179329044311218158710235187407572221001237687219447472093493123241070650806185623725267325407333248757544829675734500193219021991199607979893733836732425761039389853492787774739805080800155447640610535222023254094435677187945654304067358964910176107759483645408234861302547184764851895758366743997915085128580206078205544629917232020282229148869593997299742974711553718589242384938558585954074381048826246487880533042714630119415898963287926783273224561038521970111304665871005000832851773117764897352309266612345888731028835156264460236719966445547276083101187883891511493409393447500730258558147561908813987523578123313422798665035227253671712307568610450045489703600795698276263923441071465848957802414081584052295369374997106655948944592462866199635563506526234053394391421112718106910522900246574236041"));
        Apcomplex invZ = new Apcomplex(new Apfloat("0.15915494309189533576888376337251436203445964574045644874766734405889679763422653509011380276625308595607284272675795803689291184611457865287796741073169983922923996693740907757307774639692530768871739289621739766169336239024172362901183238011422269975571594046189008690267395612048941093693784408552872309994644340024867234773945961089832309678307490616698646280469944865218788157478656696424103899587413934860998386809919996244287558517117885843111751876716054654753698800973946036475933376805930249449663530532715677550322032477781639716602294674811959816584060601680303599813391198749883278665443527975507001624067756438884957131088012219937614768137776473789063306804645797848176131242731406996077502450029775985708905690279678513152521001631774602092481160624056145620314648408924845919143521157540755620087152660680221715914075747458272259774628539987515532939081398177240935825479707332871904069997590765770784934703935898280871734256403668951166254570594332763126865002612271797115321125995043866794503762556083631711695259758128224941623334314510612353687856311363669216714206974696012925057833605311960859450983955671870995474651043162381551758083944297997099950525438756612944588330684605078529151514104048929885063881607761969930734103899957869189059809373777206187543222718930136625526123878038753888110681406765434082827852693342679955607079038606035273899624512599574927629702359409558430116482964118557771240575444945702178976979240949032729477021664960356531815354400384068987471769158876319096650696440477697068768365677810477979545035339575830188183868793776612481495305996558021908359875103512712904323158049871968687775946566346221034204440855497850379273869429353661937782928735937843470323023714583792355711863634192946018318229196416500878307933135349779099745864929026745060989368909458830503370305380547312321580943197676032283131418980974982243833517435698984750103950068388003978672359960802400273901087495485478792356826113994890326899742708349611492082890377678474303550456845608367147930845672332703548539255620208683932409956221175331839402097079357077496549880868606636096866196703747454210283121925184622483499116114956655603796967613993128299607760827799010078303600233827298790854023876155744543092601191005433799838904654921248295160707285300522721023601752331317317975931105032815510937391363964530579260718008361795487672464598047397729244810920093712578691833289588628399043586866663975673445140950363732719174311388066383072592302759734506054821277803706533778303217098773496656849080032698850674179146468350828161685331433616073099514985311981973375844420984165595415225064339431286444038388356150879771645017064706751877456059160871685785793922623475633171113299865594159689071985068874423005751919770569003821839256220338742353625680835415651729710881172179593683256488518749974870855311659830610139214454460161488452770251141107024852173974510386673640387286009967489317356181207117404788993688865569230784850230570571440636386320236852010741005748592281115721968003978247595300166958522123034641877365043546764645656597190112308476709930970859128364666919177693879143331556650669813216415210089571172862384260706784517601113450800699476842235698962488051577598095339708085475059753626564903439445420581788643568304200031509559474343925254485067491429086475144230332133245695116349456775393942403609054383355282924342203494843661514663228602477666660495314065734357553014090827988091478669343492273760263499782995701816196432123314047576289748408289117409747826378991816999394874977151989818726662946018305395832752092363506853889228468247259972528300766856937583659722919824429747406163818311395830674434851692859738323739266240243450199780994040218961348342736136764499138271541660634248293637418506122610861321199863346284709941839942742955915628333990480382117501161211667205191257930355292924113440311613411249531838592695849044384680784909739828088552970451530539914009886988408836548366522246686240872540140400911787421220452307533473972538149403884190586842311594632274433906612516239310628319532388339213153455638151175203510874595582011237543597681553401874073943403633978038817210045316918295194879591767395417787924352761740724605939160273228287946819364912894971495343255272359165929807247998580612690073321884452679433504558019524925663062048766161343653399202875452085553441440990512982727454659118132223284051166615650709837557433729548631204112171638091560616116573200008330611460618128032625869595160246321661385766148047199327077713164412015949601106328305207595834850305079095584982982186740289838551383239570208076397550429225984764707101642697438450430916586452836032493360435465723755791613663241204578099697156634022158805457943132827800552461320889018742121092448910410052154968097113720754005710963406643135745439915976943578892079342561778302223701148642492523924872871313202176673607566455982726095741566023437874362913210974858971507130739104072643541417970572226547980381512759579124002534468048220261734229900102048306246303379647467819050181183037515380287952343341955021356897709129056143178787920862057449992578975690184921032420647138519113881475640209760554895793785141404145305151583964282326540602060331189158657027208625026991639375152788736060811455694842103224077727274216513642343669927163403094053074806526850930165892136921414312937134106157153714062039784761842650297807860626696996080918422347633504774671901745045144616638284620824086735951023713029044437794085350344544263341306263074595138303102293146934466832851766328241515210179422644395718121717021756492196444939653222218765848824451190940134050443213985862862108317939396084438980191478738977233102863101314869552126205182780634945711866277825659883100535155231665984394090221806314454521212978973447148874125826822386023602710998119152056882347239835801336606837863288679286197323672536066852168563201194897807339584191906659583867852941241871821727987506103946064819585745620060892122841639437384654958993202848123643346611970732430954585907336187862906318501651062675768512163575886963074519992200107766768309469814975622682434793671310841210219520899481912444048751171059184413990788945577518462161904153093454380280893862807323757861526779711433232419698578056376301808843866406071753683213626296712242609428540110963218262765120117022552929289655594608204938409069076069200395464619164002156733601790963187289199863434108690320057966371031286123569888176403642525408370981081483519031213186247228181050845123690190646632235938872454630737272808789830041018948591367374258941812405672919123800330634499821963158038638105424578934500845532803135118843410073730605956544373624887712926289807423539074061786905784443105274262641767830058221486462289361929669299203304669332843815805356486407318444059954968935377318367266131301086235880212880432893445621404797894542337360585063270439981932635916687341943656783901281912202816229500333012236091858755920195908122415367949909544888109975891989081158116353889163394029237220498483752242362091008340975667917100841679570223317897107102928884897013099533995424415335060625843921452433864640343244065731747755340540448100617761256908474646143297654390000838265211452101623664311197987319027511914412136169620456936026336102355962140467029012156796418735746835873172331004745963339773247704491888513441536376009153756426743845016622139371930674870628815954648197751922077102367432890626907091179194127762122451172354677115640433357720616661564674474627305622913332030953340551384171819460532150142632800087955181329675497284670188365742534250169942310691563431066260434122052138315879711150754540632906570248488648697402872037259869281149360627403842332874942332178578775073557185704378737969340233690291144696144864976971943452746744296030894371925405266588907106620625755099303799766583679361128137451104971506153783743579555867972129358764463093757203221320246056566112997131027586911284604325184343269155292845857349597150425653993021121849472321323805165498029099196768151180224831925127372199792134331067642187484426215985121676396779352982985195854539210695788058685312327754543322916198905318905372539158222292325972781334278182560648823337607196810144814531983362379107671255017528826351836492103572587410356573894694875444694018175923060937082814650185742532496921276462424783221076547375056819883456410354580272612522855031543250395918489189826304987591154063210354263890012837426155187877318375862355175378506956599570028011584125887015003017025916746302084241244912839238052577251473714123102301725639683055535832628403836381576868284643304568059940187001071952092970177990583216417579868116586547147748964716547948831214043183607984431405573117934967776373989893022776560705853040837477526409474350703952145247016838840709087061471944372256502823145872995869738316897126851939042297110721350756978037262545814109503827038898736451628482018046828820582913533901383564914430040157065098879267154174507066868887834380555835011967458623408059532724727843829259395771584036885940989939255241688378793572796795165407667392703125641876096219024304699348598919906001297774692145329704216778172615178506530085525599979402099694554315452745856704403686680428648404512881182309793496962721836492935516202987246958329948193297833580345902322705261254211443708435958494433836383883177518411608817112512792333745772193398208190054063292937775306906607415304997682647124407768817248673421685881509913342207593094717385515934080895712441063472089319491288078357631158294005497089180233665960770709275990105270281508688978285494340372642729262103487013992868853550062061514343078665396085995005871493914165206530207008526562407470366073660533380526376675720188394972770472221536338511354834636246198554259938719333674820422097449956672702505446423243957506869591330193746919142980999342423055017266521209241455962596055442759095199682431308427969371132070210498232381957471759855195018646309402975943631944500919150616049228764323192129703446093584259267276386814363309856853278602433214105233076065884149585871819707124299595922678117279644388537967631392743142279531145000649221265001332686230215508373001756887779405293922189029424143891860902523020125072081914513559252448175874191530285421032688755719724975463913213416650130089984858004988184539712256278637424126038402626928051361641332991194822906711506766502171984061981065252133878916262149742374847011302186990595582230140544519769218504088109866949722922088405432386591444077613593583977043162030198483687580783576429997384823060264097533891101537850288850208875452232316008470588468561438287055426578280588833996325463499127274201470388306349081714134544089291185028874099319952275905464542169235856036455129086436256830037214489582463463010911271935780422308203367573822202173963037384430357318747050709862946814186132911304352890724468689732842917280819233086617509641888502351666154124836167509421386532027958735884076218465024752214216006343124636083236846164408779345151046756110838674267901803569874943470363501020585861081379083328000450979761530404125029486318117174817838352800558398999904445335622512187782107356141197355485008078033722598929661207534586689194387026678701288235883661632206543667404027766354617520034284643729617064606010906465199030828152320178953367763499425648462967104953266723853904704937155459977223244359796704827248971177536576290495873282263979918608190185511935160957384245746193381018864540398341443436533794465262922870048336387630526474342434263492652939493937651558624005989912216084504237590530711424650102655299280994514986083383571564960853124164235244436418839471473944496583057057034126813678308616528510619213892914048179549332402960640814505103060473822729466747251091470938852715189704510903960069811099490561638179486071758425024112534307738806074627942710452584860382311418063616144670592786423022671878159420103387312236631641711565532384256016200055689990584850129614652403821718135404439939767455202147332201283897294683888991005191291681454544002802167882207080005379644281146076436811762252658697119084492232425967729532174944633069141466335696546210141637869295979380854984963288125832241613124908276514019822229902646896039733957070812119028686326137009638968436489077956313426920368199567596428576484886965373813120547380940758060791114396835798478889574202010394322644725932906550201938457089314396776199419230289067663807121779719696901986559084891516381324373872236574715900584088615980357198551814210081812709096071968326771380197511471144465824919283431439056546492279930152267942565150539544200689702523323921822542038882844715943684624104551827218334305559448316047742263873886047591632964026986643545610882926765585970956603378896581991747781168854913408532900306153854396889935716898788575384050149102375541735824374537629428754826710372969543274397792392280461782749854973868350413491458156904982330874385219664224959847715814815062706953375070797976022372569401691461374860095711922469228633128182687160159530822351748137454909809152290645895627021926526821573140360451847516620129376490520083105472967144371464788001888492767297732235305345095500099492568227839118072495134061931255331151369899255407534880858224444053927989329248877022137179052502751258103252877702152267073091243326679210110418381501739688041273383675434041752960485760040017205524549427482219234334245813743685502854868344594789826372474263384509459695000655718942975450917713445196880141258838983675590521239174962863542990657827429051833901384906970752823142116372645404586445634982957782376851603582706319559030459884878893335522399761398761675651669293819494189702204104055268172558117235459023653463904099051025828748982910473603873010140762927526864852738425659885920252228789612974764937984795128415446342496152015624584115723787430082351815151255483594577612834618334622458482425398960372059218207360467258453432348303024442922663551805927135907926458375065236610158705254420454427522399655918830598735800263484619358743583256039455770309665487608388839384373260593852766040391273789411845700468215083948377918584844252865857287541308158746962694464141982897858294679010335037765149290471990782228087484538463335994139875981935635595426094920021730282219291324610755034159723829784846735527316719499323774546101622154870667279654356889082750750385194364946461236421512237418239271076098118331165295952658857274986983349418174089463129164070470548239619996798378868374539679750722686025782433366536376048673771131619789162239321366833518612554248017772967038503289610060749515117568533833292101202911263748484763456287411828132173343055844411560985666312156750326304287262340215248778463994174203213186650730369807172498498530681398098707319409035845739814534039663873374980703942427975613598554206180580884925369866302690447390838349456334676718973333924898623831892805218091285187433667529017454054252600093320411156172367368088284607647496901787959927706320192284084421978708945298041959046916619176371836587466193465333597502489729784022696453686109615430801440877469875616727793287519251679638698687900105304899754152761928390384470591250502517721519181962635604911315371330390311544271312342509883739263727815007299780456497001552069286120427670813732628087406296079812707254081876582464008952529687990334347900909203036043681831055435175824977675284303811721391490221569035451503210741745997033257588836022746502930319742498660601623225569817090589136874195761344240977427674139865890480005809616523467250728010089811009065805578250348897369804636718251689565258448927370134133982044113868614568727887224903992294702605854343762487477342279111689592419760893193225114890176946635066301166883795731595667330500768497190567470905412957443010936499375408355670789570337778370321074482375374328699166793180310272410493276087581996767531352745364535800721367067365699411257212702227042069567226340208296011979565359615884428002882707898499538280053959919917494069303743540452523475352859835214270471396676956007137824367641047605020174081896422787957053812579637376541631047638980190601858743867858812089653378519146180507022618780263604368519294452416168652942199116952875926940308706060386289333855000555122745730526677540485512473707921273149232186276677411300097049533840799029876219798525798368377095034636446448389409460950156394827234182855341941024839124776836861446268995517276017210563092724653274594593884451709649405235168641272028775255035407215490698363949392451498685722939605724396108464408554158559355034963463681123907851539352267710090903118879564105734898433771385785234851422149083852636035419102663667724205976410551797630607255427800647213717346802149366018351580009547218890244396064884455195726319747214318281976387675907783428604693721805882299074821345901392973245807157950995813597977424904993670795527508308733445446573034723401896349618212253522518707157169130095463231915682070542750113877671754930532451488793919063143540607236796063137084898659860266496345162140297100248022017828040214350228801504060370924148729967351077410391455589236515892113625962948793578570907950007751252346265239812104220432084979737199714688045451271985666268027554027139241939527770864407486197233608417286516751218338078288087760289791191900286340895738514132564429686183854830825454532705485640040023648389127800685482685995520717716168699652852961031372516480825576231951140724399817560344873246859759275545425633444756081243688267604099015388058364149140163145863939802447021142224377191387130673624313426131898140272148472881743314261346515565441170763698868857118202441272935554272910205267078504466957682463383931434000081624082786850079095974704954643691338365214611148983844618968759988161668298955681662038733610407062684266219716746123536760620181116114551066460054572158976689657257370009722049072247517470037480577647805014161875741284165509450172506752928904791710069615272817894480556262662929275351089070605997166657654519676273178286475339174722918507638462280281054636832384189980484324160842479978258553509844381304520574865613364790045667924045491191063455973729115937854067338918313155567509004447443405376827759035968354045722720840983290794074426480423852283340803333850527847864064351934635094412845862413784114356434520367994162151354681697249887433953129154475298072500159730879828949048386112126446048152797582262056499653208918037513401702823112060569239119425381155906602935749616545385592402902986508397949806546208843483800441154054815129449860966310789040980284024073229933141743202883478824307422201251926515428429654317180984917510753492103216229559546746189234448922707900136955622396563734449724290094610156353436536934322450615018252887381345830125159268242135019171242880098145823317099938161239062744165096463476639916668845258498282782933200831213226714219626043508562449500071733176344021264524887023055935990918328153426643910132149357586946210184843569567858815666441935284534702812745040933746856393062399159701935259683850480109744498112892029721046247122858851766785444468414690449927428507272926169460812263104349599680544672819493773082628797165444600320373512445373855641619556699634425532428794658853425688754301734140966410944348170917237011835445449938700910454196460870206678224060546855803736809971496665086721113965568318370053617453043774552676303432075652930078336237475888141059765387876335569657825810200485685667569840114955194674334480448424604326304613670627563587307331729172362126988603910899665413682818986955949457226452651801278442185531160371246500299426901571437702099165812157653439242537518589554735252811332195410596355910394762558574318708926759446156265855181733243081877410054844715888651743344229449504348737486087615699262029157072507740245952280267849972132416551111222482951675008148607902496676190427824042669429212087099421047053381576519296465649542068499257850353555262124980813257530719612931570822324054189599577963566003944845865826569320531238715098369723969667018744611548016810952905099664784490509541809986372822352389351398549070547399213768857530630551048135924837849341951162499016279757756444141694891216457024746830939472472558318298511336170455221048438314967260772830546096163718731492000723680765376845351550447398868298811833750514600244822640747969692950890572799458552301748447354727097366674834706360880354558055975159758681731451309724176078572216704487460714953268048254391300801342304850524495028714792255878707079855682470190570724315342405086481460863153219862168296875578761821173883151334076536080159717801720639067537594803642916047563317560424255157433633267016725429821100072767613486912253163513445740579690032225474336244373590076985643184619635538794060627347036751655973937702459851979620650211329438683403631255380826502227542954198862804695005027355198130905023718790539564068835671704811171713971818230794022222949682042097031703067746719182745393042248731952224218434673334681428213356005757567324992732670335201562147303962184677067060119549903702856585116169224129153317412931365022469645385832975346425179639488091603463742381677543266093151038735960096430800471303980666061811309321734228473748305372396746905078121691192948286707750332675765623877762106922959181992713914016365508744309165143096216104384981232284167897516967760174337493405437012038441433462911968048552405254210309472994067752144205198561793143425039277635448562516010286370725635169713439323165420117843466174251385542867592341587231623519139729934734600228091859321390885287987019762290607124862469540231628345627871105578952331805156768008110721206775971325862127617737662222649290244555883290644997992422449648585204061333688959255376258732227120904878638290896418205044870583080625276569012677470548730386042260283980234926338232098076426664291945681577938985721106079390310214778134510488476502280465284996513080029880372455200462264916190269178971081589375890148194222530466513865690433479753830740538708744584748474170230678295524740736696701358581850783912460407438803851246314812156925515998485235329242665402644887381247387173712320986205744549629273853493023556003990941627836049062092375013697355177759868150779860027636650142014069126811799797053655371205479749164709272079542259884559654327198216359928183932753080550927824736772125546018868719716646270247968719866116356257626289835921612556381433462103191579188179895281391124532092652674291214106054192301007859712355622083814030687437156356241630428390870582209810294150736120855862752370631399081647899267337288424998647380959175174541696404766458748506256536038743339929973226966698197987524427429308641638670490060513840613845790885524410920394410862706288136259086575926369358322558943973099376048975314348688259966542698303243139184571676607111639029176637536280373728618859790576512250267004233095263441666959315847610684427207819305320997552104927412149769190259956637321446415948934207024837995806648608513373269490303109445035641956364938778615763710948580960412223637089767620119109382011012376518094635320015904919404624982362101814416641771785342081935125066829731629919658521196527504757982147423632513427390487223108789398780286299599565587113602427931236056186394327904271642442219329178210219377938494700575438674954316766348205991537579017664937439661851963299565557195910756574392370283938952637970039189118814492432345838612966708830454850885904101504465894375639230531370415421275531749855295347885782182098802217338561171152886971675999774825587348686317189102244115861602888498603655315759283660884665242192241406882393051559877682075272759667838988129353998432249960412477828986598669259796483309298869473919357076118149362088771908395861619282498471925898153186794809535642227129813609624003227959528573392368018301829622800630214644812312148456560195984082387047742743352904114911724891171997336166506821399422921448074552499923843632605225112777410439072668862078721462551484263232470476233075241709810857114107419261435657559191043270729312928344523585732115996240831623766393970383483316622480110947567398963567346035793755348877362288517380930458021038700098637422397513709404944639442950091254885614154234489775154098222565687150452162348604196421067111545727644616093010337591264051520064888867680894200306147909618019482690086747289872697813628466054185366697685409969547854328744187941723495604372839844375587797380795219717058755952759990369219172140689053315888619175279362845871523679162379980376717704795255012451574691501714237044922632797724273869333720589379491670574255518181183525512678365749236104804577876611572624101910900110211423898759868989017422869169872376924733762900817930883407240567921739244384617661949368060452415450168286615456023317327355252667688380068322056414786712804894975114796998372867390845377918647882913242891563207911998370666677743156244699718141236145578991521692783881635283809768411153731157014725591895771063616148136967310521813808835535874348368723389221603034667397227372418947523619639280281894445742976672558648383173108431880484814854692490088282519055914354110879747484442248122036360539357621613279040115278974055098067283411300003634133936060729918236164783519098630180255888154885797626773076012729787228261350713193411341025607930525951577347424638361223734952881255482513077390970381451955925948544297944709516421180704977426079339395522806795754199135202708990931667722773717018402541916405485607871933782190706149528185397429439710491793081436098030860657971369543992820550819856043716152378574113270643337602300187576785338059934724938388358908392082970269800162070470136797690009154485764619981282917508171229339537379941839512485340905187785860236060106779805737894646439877656511183409571876094327470800684586601296808735981879168588290407004908457254503127282163452920169435658008227603047530377852153512078506518275707987343736266150005184781412872698625827755944131188137245250931470944692273901613975953181940505585066462600673221872282420934872582822524464117145017339324983217829898766470675953873110925091149208264765484781677123642068532676257146090995177368221222076333514891942607361976614988509297497247047901771692117338673558630674416600265770977274425840769935846468868623720391374629890798082074556234539152238944652256748272395314888306443249487981366405346169068655941814843686097921997299683123571163930317642935169843866129564124705044002861993255503913284130487744659344793157017332955535633732311078461707545828973778617115999271848064677929093529956484338182564513742084644901161640928540373076599618499834351750522249726905116488487507284431867515931391742306480138108723856836056538952147597309680121195188245726745552009833863782509983785780450579980240843441864259643480088888211215272244518475825041804925249387548134194365560589385728049311612638655177929317129275948658370395014837593331320293902638042133738859358988190639849372698803817717396077918704335492051130595110891512766619076067148050861542082149913076341009646057645149675213049825823593771927370906460302033981035472344292867359895076436181942005147915034953277296746590023056642028444188463212877931480317003578769660994826546840582765564930769688592971134126121753684184516176342587321012078123534622914877693250364292656478267884126295589053109931435221802972027127306759262347849710214394198289895976478916837324143979267037086783827916231350069378897392087206505936191099644554769975862315545272051813846646581016664708546561992831104177479045039580403920112711890506543197058247900578930676429385049046720546301501470831992839740901638060955586421506287026323091624161155784927562542076028595617272523369044099138823065110909753140287494105573660519899043744007444076855540410658926474353808668364500255631681598282741611492860722275956921709175936680979239650757684499587586905168980844072605053548208905447940665643973229931920243293625403421445623160714784852883302078351627520913127792909618830568170296476431721954372988442999993385835723900176277137338203965727546006652827398444408593411487066752245358907979159149942836946507110928665768963247230129603765720141589357911081959028892341336053175567527681186345036075810463571011017758747759221847835248913877438080271341695962027010466354196130614111039672587950160102020348137498903790171282965513521741883356790759846284309129084593993003680295829871793367832562284486727215200262989667784070386267553775209612475012779180665649058595518432862400471938803244459150811187358913001631858333531380562127493821384016892187131526021989251416554383949706372475102423846782561740007514815266976467891865309220796186609256477828391733424965639848485018890966111411531272723960466038732481298109007454249872445714964309121504268923583694207779258602937220195518262312593353838292964119259273942104708865544255941886561896388314163251277980293081625387297113565270875029812892791796713806215079541420584155081653621272281045578835010987988769510341390441589472235564980292143469576197750229264425655224649696708095839719640568163824302770623517122119064991226847735788706281901059826922459215872920661395098177392094460667560790631267611024459636539836241433424539094354464335531453548867207391487846891781368648054049393027691279927489369529857480795393459585734183664500570139183732751532861856226905933231404109531707586521724926302397917275664364119535513154181705516443869670945649537166893375386990187677318022007172080382860202180065435902647423187698508267941223168537934452889094078626079386451141473212895739053435515810049596570049120013887044136319710574030517331915898636785546360830036600755414094101832626329670817973095735835633875731016843506175922085825625567790918419196328362422751668867569671009756715183730881832194647471420409697644515541431879225764539021180084718062422744951422437049671934010045749930022382500228400895117881728732043209203040419843245376695085916560654250029978997073164125519452733143962391871838908857152038647955564269054804862391681717329747921734072843743880084476468358661616865585709541969265293110005668786962677830083433200118503810366916033183222976709303885579956590828377382473517757644282050908528372707287477184315532786908952537518168956384557607764923261603976347502906622333704433368614042979349288367279085645195906876492646269959598811228696236644844214768047500613106524962133217169233675544322733015764452803459079242638080628971856843766418853391290461729435282208909064553396428821021686447862592606607067310065583204544641909168417678149435570220325159688206332396208679776291992282528704385119920914242275420381472815753727992979315843405969418277994253242158604164160963162837545548631121008137485889076818716739201234696062299521622590284326900631742352655591884981331800031475025079256993449360464556986940744976622404044704004586401404533692552329437194198895151332310748863926042415118526953426230377200886801840201125649629252004034173503112189307766641273998015798409511273908239678337727431927408226980327946091912801883737612054047711781237159771571422953529956181507461731931727818677916823243079126407041772641406777188432609929892141510572407771669949664931583344755836063133784626996183911438430036550856828419349748142086850095406328032345056890936176975580462571962263372810018067201792415720107972424517778935722042122067535757005764105492899250475080031184315668208127359798285138674508500342539200305576041937466284384630387106528601958722227701401679448464347858804116725005528604931393922296165003633044540169867915311913902084772450923235810885646628549111400946201887046483583199826593566576787179538537524087442481785687229192404375960783644328851206702150712226596023090750364638965140663128568735852840250643807415166994143888036267744459502985137777161655674667571114163762952290811296871371856001505427451732818030671600549039870613624142169702351432749114964296772385729654454385400023362334683068656486518970938191890459394883324936806110855317817072236840009759118395231993832171262021316914682020021152720481750082602152793204899640748035335680796441768352145148858624449187188492935013540912992664324790826906945809472578323157096216778765200723362891550688278185253214038720404220309564049510697230722057493834344347345612991767010146114010241410024484777915353433948473517108024054394262536228683912166125034979660058697687371216495036631028471741230388377908266707197018937269697414856922152586798184512969598864679814264184332658674764507971691154564676991163811421719586970822758516383758807858576174533727276586267737914983510978360019994290144109039273268580624145532100481164300090299871279844783449599210700589250100479953687561212571737812151396770013337959461141753167074272427242205091559704241858424575278352280772557697739091284439919653520454884182642532038039529141231714407184096557780459349945256612318611962581939187616629915577509343250968581849293597680272385976431075335150284489722205750866611508021826538278160253875164984284150073107838296929080962162142401322272527002813395854726935943051786865856278031952037846440535962813831454471803660730222053585552995441328481911290572228002761624429436941560649885212260007337843074312759443142492210170156685398079086663722797976035180974757869840254117844882139576325403705584231403062117216228161186783097343803423943298407515529031487222949869840985765178647313636799737791982247478402964695999946002252216638813528648883798886097835599203320105657187708439235413660350508387678822446494220229229090738234261857776590981448736667759127163754454657506510665443956295306961123676489505180826522996779317802324958762305471603172449923572985710834684327664393850141980313440209278405499954218540148009534210050875299645162637173224208472075290850379663595633053153272996345241182611028936178721426021938376350316678359889670959338033121974325760030300520333234963383867107817485673119846024251969017036398980210644689770000605078527429898489573151467461897393808614224451760169362713478084109758951723977054835978984410168767249994594624654019809412010905822812663277894753884927262295012844074450296732248982120762126828218365259311530520538045991659567349259390608414749642315542503420632040347232507310817256797541339552683857894336713552283859875901674974168938058908999329069639880906337526036570117199894953562599389498307367803242710424905723413354569340628481807299874261262081084448911354384465902096974664584001525476337789892858440363668798194526260795835945583263375791502537160473368792475698855240003510484909855356659678932456024609112592123860270073621632153302605827128887403419943918950064253028133104886616096915091487577261692553152701437736395815890930396883022781288168474069748684514204110305910648989307101741491540600781258360478763282487602462405705334328685955154299530568753742381819449718482914298788381312100806152393221706222301564415382381847157634357684001759515757436539763794218441130456807742493582112210627908174131123462234100063281658196908991203702368832358952096969686524377022773223364201437278961857235160197999308143541663029676034283825385167600161326785203133410029528783189528747976748894000694989877227702108543591486205546650478895510897726640945918106141157298720325863376660880267518545172821988803555183494388674652999206509725767810183458734734605211895034335056042244412076926639895046503509751847835402568395847493326151084851119493297713729483071120202170389205033053075194977785234207315472456405938518038947124316652502513967901291269462577870734131847871026156402721744636822988872773694391725831662716281659890049866144300774752573946141642429750582619859814882107196132194367194255426291355416564281180743333094378470301575523390008190131447851658022080984937185796902606092489780986487686349075120549179404600295724016608012167802238180725240832742995823398383979004045433978185908465315230363725199537687915329102657709856865935706280157650732128812584264792554264252809702103125141947118436266531674845056294043219148756291594327502141470212187436138517849871717035153813338591160907168703199165531437845763376799836951551872264504931266940284831908230638422045278313750765721061919361825109850265753724808341297151330159772441143105894213717225305754134940269059818897146807906501195267394438759814690110933640516119928236755033389350475933526571483084511746906508203191710065319395774540892266484859689343784744878936800491144418848068548546405907872074512690725303468095256743900942072943600508630010851122995453551596367340199524720697806972165130694494313399787297500627266375769552915032206495804445341056672586854978495792892906046353810026579201046889548791448210587242342985714006586603601087039976063135031757053209175050555397708393609742737513392572067736554938298131437895865377952698899771552636597191542696577821767219470879561427531536178933334027846070362932039156542547741891705788781712078965823139159721491067805343902687323819916534014274634544417941205434328049562758421326807858439845202350291376757007612053982663589486607013519081532395784224656899814369160847662485080655936469034371903753583987015657943838077605991457900250737286514980949786281850215248111532353082225458828970802749393691869793537672972065277717847739628131728725067566379958648452423903939169664944392622878909177651127858021581413363773495556728864201091640785574217563989971436293624278292495778364404771781811971341582611670324785987043156931635210707362668194857416154264391022183411943236478195521811966416662830683989676691528399555777355244178669231813809141401705723762762548289728399486281827662822304382625219718650920659147397235188163060052940763468278680701160937918632283532728195958976608277942047924056754441956914702074589053980493989188343822339118472781312793242742863455064382547866676617993850852636875455894033134176548274627394062554275640430362593871188018363663134071170193597340796053063345659262050567499337862979165784566519407322986005107222768365593766243641623417928872079924620216959395684151706533337739052982382918198668314550034882358898387578476052407549119882325728436920554531295442074390324991531190976294954576241839856948857831040703680188595099398579777675791353272515026479935060977725211711814671271100064952097996194344421433110350941783117988224268733074907659157967486580773348293838799112749948512365129879587397047068980476814920522411534991627225793606735799096462965511687510020011179098776974335764464266948212131919495296197154429109417518010308363867680676347196245403580270347678770334093277938723810830489612150010961708674774675122613597474956457071599898406571058603854867213151864037020242344879081895881228255249165058308361998124735847296529834751385670488873767100677450646901653313926412254789044042218643433222051687316293350780034360104971899262901765904832524941283907790423617466168827754936443571534784485223208418739918961324581422421259967731691470748517187020658059198743749547105859425140589104435479734780007467744469643790298666557306345693129125199292658867134802230890707370569851451745882376437317145269653761045073020341933389592194831115441888671560011922412645239422166532658001070184905653106063152916638705151214733569932330856586315475663909369439875068786199249192204537386753306346771344887767930890675773388875392976263846501891855612173173163388910110478243750901865177696882584708491490974713380782901152636584274648339843443114857936204719639832751855191237669745004384035107606280390504688409526294183326451210859981390675691454370189022599531958613039169736017850949128004594980473701864510706721938809276073025115741976484355829289447809652394774011967652378527464951610723318265727733506235006034930681292093571389004407121321282369532270130383717165527828980349394087895366207179014066830522664458654290856996560340131370830000893287893042522472948921971308044440205849096327480717691091951345160380600191131068463116216849307199988519593377700692606885063917517523800316491000234858373682715785627380273832938120398080866571288299053227063913315387648160268369682047438106695172304072914285641113575978026841477421665734516402043670408961095786786447188413864262556220715213871852190156217408017764818807110018262936444053051763274571562538550894797940268249478406259301766851743494770226993416872265580395198700304636441851017485230932281579833179472133117778325710799333565440949304790379260646727093020172653397716730831137624284031805143175575691891503525060361079542250480460108435280665578023254658679017618946247247573131651209542413100132008369047364165319853790514948965875010728174837086755365109294661610857916142739140164261109405738215571278005554036960427821981916149734868453306194177482148454599841800377065136280549404117579034050975063370262270197403754496106011930855484579497919575285463608949198978309057256178599859653093214550107833122342776155824490964555482125947833477860877629308113013805369250762068406132233294052761029094218787141281575616366346952793716157164694262408966023104658944120991664949062508856519774071638580061762505247868775545376429648617181620882117957225354855754672736290091400251467130172685429937186091083607915482678807005161282482183834573778700962752504253564944044304881200624645053574146250462316146137343930624589695166816900009012743806568877316380974607483832448471476224592628031548305893405009155501146669900238617084584290567237053687726812602786545533267863120057890118514356961541308514820717672379841260908720537794002798311061075034886504079407392366517116651712625316095278743906128983657011968854416787975954326562928044715676243274294477187219552724071509266777909672489051004985987425196816497625858160407295727733990991995661138092090368803342809821715773444330055490382289873524773308315104408819473785738436466844772087323254732837200499870310796138033173891711195706973441642537318132986420209247333130132502515018285489995264683725856369549494782040706757380009610998073507688115469057123609703310391552677470932921591217895802553355762464966432575924382782044176869653348659634993914986261139847908566974539265919931267161340701775971400837796757121393628277035626392271114704095056536322204531002687018076444462481060768844860491995135167265160899227179263059250636720839576873053317083307332577045155219182703892518702835164125385466614504022625456066553081102028238562620957989978814573708438186198150847323476681890256264428554538073387204865520531136806698381709848409647186073494714725361487135139270928349644492149338058943048941911812991168965602514271349991995891786149203609111559850894125658238027280279218627291965954442780796947870130521706067136229616141501529360732361877226388085546579500268822821353498569806810984543933373113891781101897413857244574677914158161642430512886960182760375156107023658532697332607063085096659294586650521595203875500086136774572119599487916066259875387398283805517679588532479331468421493614254852598847878680353487851235667458467535255299098221976445689726239033409412619313350061001389274457217438842646827715417984701679524700634807058795181028082341840352948073174801245657209392962316619114734331084671575413435991809837974437434264442988139786620617409270506793101860875979513302885932755545584444964396479482817339620032385678447201706394798910834831026851987446651363878915982646966447765679986130910382853197229018802161165197669737060704932567951532906076272939086925802787884417127155288957790775094623773951189345354675711606700617530961080585498426749681990167297374915659380085041974651195996489331911261823734411731731024097816700916973230986389351015789572555088777410794240717344815892680483857274193267895708502102287120728364301381969010336227892886553972107958222479877046323795468244003832149093952420031449916255849026186781606029603653130353549470839018598373913240912881598071797603384440236383544022909385280109259344942015290249368575323804307636751023829935792131419503452607609778290947352545772986193217052774985351638712074965100705160259932099787688505987520701073124634758014732783899182384302491689000622438581286294342850711882402238564480253693247805282364608938240402599610637123849259931196574383169060653995690606823988655691709578747059269653149569501720913638508738797156249932452814632042123543280934897893663122157181080254013209676189550205604925659240240657655555538852291227145021880428393392114070674817906296372721655486720296113957026402928501995561953407894661051802928180355733708904988300101471096225463097420340853209299367790473611488742647829825696528658851674786096041986579060857836773395536366059923987750382930377286261107412194875349050559108555236933470374478198959105052354953157442664445631459865854991665190516347987515656739667921481825490121320609871419911977469111683337121991522679608127698125914982665157102027387120699487802931415331685233050866882021473242575556249878092849014852673278135321730818403105060482196980019673579973898168392330754318433474394700633696386080077466906249105071524836205035172122744100452721353204069013509984357639198356778770374876669559836409147539245526371573235421387697786589431819872282474021432045916423505729594587156096452238282248509828978324579174407720565034636214732761459488582780651785828715701538287778471287969938738787092464887090821699717799823503990451237317101380061365320988556415264355889968537255037829767960446249020048793969635136380864080861770731225669242697613534653964520589541429738979101542140454504926147693231851246631078753449031960637072063890481632744706980233253887210241035923101942384333200352295068416655009199730502323510449883320093182152201924860349402967552873201782116280121239287626286489005626863942970453167195664048534088631941705570535038604624112075899209977486491787632937004402164640641356448174698618716608130949171807004991530441014774625652155952448155022458953863954554752732355676064687625138579581158663686593821786514517430373461657532898134097604051674742934336915793035314463892445209987849763760576296818980478791460456808396880526096626479262240184942531330951586605311364522694918929709230095346041626465198412087782311512150100791809314142295257752424989657659790099472067253470413617259747646213998708886791057984854235727661323381876329366402745291679634650670891612339293852509779520212317515273813161198670718415873193775226951879966567561925206955442094308400474842163825943172843522171275280151054228044429609356426701609127369783113727117086823624592757444467903935420461571322508846432431385504217477834421698043508582295927436495923749450542034567066927578930924788017506989027336648955614098289904049901892331132032579446716643873731874774112223401828929357727231648218545896723125396133573191161751111181996097583387395938010424745028062902370422050478519719083131318875991802431679717321635682426592498732571109232877944945249655260405291073188576736402311656335543297059421221915134425790576246887613474047445490879158347536571047931193713727595397505790093525448864870096623611930528041879270374709624579284886221048087937420010531448517897698874260852707733874761588124765239408116523387530870080613281384033560134305941525547905300496824104202708328859570095007525269624312215416386138443538334631786002590100953562218177873256775039458588741267498970305790224611359729813760255467877639973695886273460732928874925361778452637531664245508445457984557971932615040628294950069599895024885381030696943286336947121976090748411685507031740746819213121513132787490555761211819259623720033365812504349431231978296386447544645580559961361987940158977389625060099386921582429182762084272213717720366742955569470519163842031702271457516243413268873369979633440697207493301299192296087911852579769656466753526118330141492415800323463209196324827064208697211880588164054508748500461580291049120123296807537561744942430175727604024922889627425837438783515058926513621780768188574004881750283600870069849111826857455732095764600676146418910502372448957121364905159625187037491705621269664755117417353397087881065807278840878336668674950138249579869599329308103265226412878813706023903251401981245680900685451235027421021067002058043092046945112415014832219138162294496618075970741281653820921783708919411397756961102859357551303927638995428756561099262920573793367638241253753342603942637461599016036138144383347794909220198433020261170501012438738700118546987890835783028089795038724083882486417315903625282608830135410104827175934438477870364160821986699209282929104785006061058137147159423087320087098769130146608407270107291220518602024207525543717147489733902753420317634852342566354712499228055731210920173780009245512363914903645134121944351948478764498502893039169035729868212205965205692115138795297037544347041920099929302781042828072823724778542890192061073939696156759305072649503151491797730534265819777106794995565803953061152615357913948525937615935500598768401278533659476586758378532047974146726204124318147599653362651196397260685046028276871994633468211187660981215237944355537198689704512890732018519383692608737708058492314644371493154799056094997111406738550409973082824776640294881185828135397240491292772133453215592526506806894225496478822607334666621730030386474187021173331605841047912546268930115374212701575024742909195596313881695795898350175891131613530071804418059162704841639136712506159720598419883107458549506317220820883610130647028638304587818190682803895787588119485751655638238879128351211017971321337335587741878946239972168146862130084949356853319727895019787918954277413680657013550432121080747469331472325912039829666338045363711465402046641545257152456331842369796115970946322734748743127205753305486516451260022898898875640529724700269793962751475302943631296960473267795252829303109886937615597759212644401739887398755621272932755385195078572917353356952459115657688903753760962395027506109726925509814825646648465725411900119852767566145761832868928208095557476487020496797905991117504937316994313786365436081804581153699116072564578341157819143247562212245442732602804460708201762279505213947658196573272495909431680842970002627448899897019479185024957803257253905996963312944045377810985431972211637859263599453940448874068140959254239942425704834375933192732905950968498661492071549845472787663849668497523970298323404"), new Apfloat("-0.15915494309189533576888376337251436203445964574045644874766734405889679763422653509011380276625308595607284272675795803689291184611457865287796741073169983922923996693740907757307774639692530768871739289621739766169336239024172362901183238011422269975571594046189008690267395612048941093693784408552872309994644340024867234773945961089832309678307490616698646280469944865218788157478656696424103899587413934860998386809919996244287558517117885843111751876716054654753698800973946036475933376805930249449663530532715677550322032477781639716602294674811959816584060601680303599813391198749883278665443527975507001624067756438884957131088012219937614768137776473789063306804645797848176131242731406996077502450029775985708905690279678513152521001631774602092481160624056145620314648408924845919143521157540755620087152660680221715914075747458272259774628539987515532939081398177240935825479707332871904069997590765770784934703935898280871734256403668951166254570594332763126865002612271797115321125995043866794503762556083631711695259758128224941623334314510612353687856311363669216714206974696012925057833605311960859450983955671870995474651043162381551758083944297997099950525438756612944588330684605078529151514104048929885063881607761969930734103899957869189059809373777206187543222718930136625526123878038753888110681406765434082827852693342679955607079038606035273899624512599574927629702359409558430116482964118557771240575444945702178976979240949032729477021664960356531815354400384068987471769158876319096650696440477697068768365677810477979545035339575830188183868793776612481495305996558021908359875103512712904323158049871968687775946566346221034204440855497850379273869429353661937782928735937843470323023714583792355711863634192946018318229196416500878307933135349779099745864929026745060989368909458830503370305380547312321580943197676032283131418980974982243833517435698984750103950068388003978672359960802400273901087495485478792356826113994890326899742708349611492082890377678474303550456845608367147930845672332703548539255620208683932409956221175331839402097079357077496549880868606636096866196703747454210283121925184622483499116114956655603796967613993128299607760827799010078303600233827298790854023876155744543092601191005433799838904654921248295160707285300522721023601752331317317975931105032815510937391363964530579260718008361795487672464598047397729244810920093712578691833289588628399043586866663975673445140950363732719174311388066383072592302759734506054821277803706533778303217098773496656849080032698850674179146468350828161685331433616073099514985311981973375844420984165595415225064339431286444038388356150879771645017064706751877456059160871685785793922623475633171113299865594159689071985068874423005751919770569003821839256220338742353625680835415651729710881172179593683256488518749974870855311659830610139214454460161488452770251141107024852173974510386673640387286009967489317356181207117404788993688865569230784850230570571440636386320236852010741005748592281115721968003978247595300166958522123034641877365043546764645656597190112308476709930970859128364666919177693879143331556650669813216415210089571172862384260706784517601113450800699476842235698962488051577598095339708085475059753626564903439445420581788643568304200031509559474343925254485067491429086475144230332133245695116349456775393942403609054383355282924342203494843661514663228602477666660495314065734357553014090827988091478669343492273760263499782995701816196432123314047576289748408289117409747826378991816999394874977151989818726662946018305395832752092363506853889228468247259972528300766856937583659722919824429747406163818311395830674434851692859738323739266240243450199780994040218961348342736136764499138271541660634248293637418506122610861321199863346284709941839942742955915628333990480382117501161211667205191257930355292924113440311613411249531838592695849044384680784909739828088552970451530539914009886988408836548366522246686240872540140400911787421220452307533473972538149403884190586842311594632274433906612516239310628319532388339213153455638151175203510874595582011237543597681553401874073943403633978038817210045316918295194879591767395417787924352761740724605939160273228287946819364912894971495343255272359165929807247998580612690073321884452679433504558019524925663062048766161343653399202875452085553441440990512982727454659118132223284051166615650709837557433729548631204112171638091560616116573200008330611460618128032625869595160246321661385766148047199327077713164412015949601106328305207595834850305079095584982982186740289838551383239570208076397550429225984764707101642697438450430916586452836032493360435465723755791613663241204578099697156634022158805457943132827800552461320889018742121092448910410052154968097113720754005710963406643135745439915976943578892079342561778302223701148642492523924872871313202176673607566455982726095741566023437874362913210974858971507130739104072643541417970572226547980381512759579124002534468048220261734229900102048306246303379647467819050181183037515380287952343341955021356897709129056143178787920862057449992578975690184921032420647138519113881475640209760554895793785141404145305151583964282326540602060331189158657027208625026991639375152788736060811455694842103224077727274216513642343669927163403094053074806526850930165892136921414312937134106157153714062039784761842650297807860626696996080918422347633504774671901745045144616638284620824086735951023713029044437794085350344544263341306263074595138303102293146934466832851766328241515210179422644395718121717021756492196444939653222218765848824451190940134050443213985862862108317939396084438980191478738977233102863101314869552126205182780634945711866277825659883100535155231665984394090221806314454521212978973447148874125826822386023602710998119152056882347239835801336606837863288679286197323672536066852168563201194897807339584191906659583867852941241871821727987506103946064819585745620060892122841639437384654958993202848123643346611970732430954585907336187862906318501651062675768512163575886963074519992200107766768309469814975622682434793671310841210219520899481912444048751171059184413990788945577518462161904153093454380280893862807323757861526779711433232419698578056376301808843866406071753683213626296712242609428540110963218262765120117022552929289655594608204938409069076069200395464619164002156733601790963187289199863434108690320057966371031286123569888176403642525408370981081483519031213186247228181050845123690190646632235938872454630737272808789830041018948591367374258941812405672919123800330634499821963158038638105424578934500845532803135118843410073730605956544373624887712926289807423539074061786905784443105274262641767830058221486462289361929669299203304669332843815805356486407318444059954968935377318367266131301086235880212880432893445621404797894542337360585063270439981932635916687341943656783901281912202816229500333012236091858755920195908122415367949909544888109975891989081158116353889163394029237220498483752242362091008340975667917100841679570223317897107102928884897013099533995424415335060625843921452433864640343244065731747755340540448100617761256908474646143297654390000838265211452101623664311197987319027511914412136169620456936026336102355962140467029012156796418735746835873172331004745963339773247704491888513441536376009153756426743845016622139371930674870628815954648197751922077102367432890626907091179194127762122451172354677115640433357720616661564674474627305622913332030953340551384171819460532150142632800087955181329675497284670188365742534250169942310691563431066260434122052138315879711150754540632906570248488648697402872037259869281149360627403842332874942332178578775073557185704378737969340233690291144696144864976971943452746744296030894371925405266588907106620625755099303799766583679361128137451104971506153783743579555867972129358764463093757203221320246056566112997131027586911284604325184343269155292845857349597150425653993021121849472321323805165498029099196768151180224831925127372199792134331067642187484426215985121676396779352982985195854539210695788058685312327754543322916198905318905372539158222292325972781334278182560648823337607196810144814531983362379107671255017528826351836492103572587410356573894694875444694018175923060937082814650185742532496921276462424783221076547375056819883456410354580272612522855031543250395918489189826304987591154063210354263890012837426155187877318375862355175378506956599570028011584125887015003017025916746302084241244912839238052577251473714123102301725639683055535832628403836381576868284643304568059940187001071952092970177990583216417579868116586547147748964716547948831214043183607984431405573117934967776373989893022776560705853040837477526409474350703952145247016838840709087061471944372256502823145872995869738316897126851939042297110721350756978037262545814109503827038898736451628482018046828820582913533901383564914430040157065098879267154174507066868887834380555835011967458623408059532724727843829259395771584036885940989939255241688378793572796795165407667392703125641876096219024304699348598919906001297774692145329704216778172615178506530085525599979402099694554315452745856704403686680428648404512881182309793496962721836492935516202987246958329948193297833580345902322705261254211443708435958494433836383883177518411608817112512792333745772193398208190054063292937775306906607415304997682647124407768817248673421685881509913342207593094717385515934080895712441063472089319491288078357631158294005497089180233665960770709275990105270281508688978285494340372642729262103487013992868853550062061514343078665396085995005871493914165206530207008526562407470366073660533380526376675720188394972770472221536338511354834636246198554259938719333674820422097449956672702505446423243957506869591330193746919142980999342423055017266521209241455962596055442759095199682431308427969371132070210498232381957471759855195018646309402975943631944500919150616049228764323192129703446093584259267276386814363309856853278602433214105233076065884149585871819707124299595922678117279644388537967631392743142279531145000649221265001332686230215508373001756887779405293922189029424143891860902523020125072081914513559252448175874191530285421032688755719724975463913213416650130089984858004988184539712256278637424126038402626928051361641332991194822906711506766502171984061981065252133878916262149742374847011302186990595582230140544519769218504088109866949722922088405432386591444077613593583977043162030198483687580783576429997384823060264097533891101537850288850208875452232316008470588468561438287055426578280588833996325463499127274201470388306349081714134544089291185028874099319952275905464542169235856036455129086436256830037214489582463463010911271935780422308203367573822202173963037384430357318747050709862946814186132911304352890724468689732842917280819233086617509641888502351666154124836167509421386532027958735884076218465024752214216006343124636083236846164408779345151046756110838674267901803569874943470363501020585861081379083328000450979761530404125029486318117174817838352800558398999904445335622512187782107356141197355485008078033722598929661207534586689194387026678701288235883661632206543667404027766354617520034284643729617064606010906465199030828152320178953367763499425648462967104953266723853904704937155459977223244359796704827248971177536576290495873282263979918608190185511935160957384245746193381018864540398341443436533794465262922870048336387630526474342434263492652939493937651558624005989912216084504237590530711424650102655299280994514986083383571564960853124164235244436418839471473944496583057057034126813678308616528510619213892914048179549332402960640814505103060473822729466747251091470938852715189704510903960069811099490561638179486071758425024112534307738806074627942710452584860382311418063616144670592786423022671878159420103387312236631641711565532384256016200055689990584850129614652403821718135404439939767455202147332201283897294683888991005191291681454544002802167882207080005379644281146076436811762252658697119084492232425967729532174944633069141466335696546210141637869295979380854984963288125832241613124908276514019822229902646896039733957070812119028686326137009638968436489077956313426920368199567596428576484886965373813120547380940758060791114396835798478889574202010394322644725932906550201938457089314396776199419230289067663807121779719696901986559084891516381324373872236574715900584088615980357198551814210081812709096071968326771380197511471144465824919283431439056546492279930152267942565150539544200689702523323921822542038882844715943684624104551827218334305559448316047742263873886047591632964026986643545610882926765585970956603378896581991747781168854913408532900306153854396889935716898788575384050149102375541735824374537629428754826710372969543274397792392280461782749854973868350413491458156904982330874385219664224959847715814815062706953375070797976022372569401691461374860095711922469228633128182687160159530822351748137454909809152290645895627021926526821573140360451847516620129376490520083105472967144371464788001888492767297732235305345095500099492568227839118072495134061931255331151369899255407534880858224444053927989329248877022137179052502751258103252877702152267073091243326679210110418381501739688041273383675434041752960485760040017205524549427482219234334245813743685502854868344594789826372474263384509459695000655718942975450917713445196880141258838983675590521239174962863542990657827429051833901384906970752823142116372645404586445634982957782376851603582706319559030459884878893335522399761398761675651669293819494189702204104055268172558117235459023653463904099051025828748982910473603873010140762927526864852738425659885920252228789612974764937984795128415446342496152015624584115723787430082351815151255483594577612834618334622458482425398960372059218207360467258453432348303024442922663551805927135907926458375065236610158705254420454427522399655918830598735800263484619358743583256039455770309665487608388839384373260593852766040391273789411845700468215083948377918584844252865857287541308158746962694464141982897858294679010335037765149290471990782228087484538463335994139875981935635595426094920021730282219291324610755034159723829784846735527316719499323774546101622154870667279654356889082750750385194364946461236421512237418239271076098118331165295952658857274986983349418174089463129164070470548239619996798378868374539679750722686025782433366536376048673771131619789162239321366833518612554248017772967038503289610060749515117568533833292101202911263748484763456287411828132173343055844411560985666312156750326304287262340215248778463994174203213186650730369807172498498530681398098707319409035845739814534039663873374980703942427975613598554206180580884925369866302690447390838349456334676718973333924898623831892805218091285187433667529017454054252600093320411156172367368088284607647496901787959927706320192284084421978708945298041959046916619176371836587466193465333597502489729784022696453686109615430801440877469875616727793287519251679638698687900105304899754152761928390384470591250502517721519181962635604911315371330390311544271312342509883739263727815007299780456497001552069286120427670813732628087406296079812707254081876582464008952529687990334347900909203036043681831055435175824977675284303811721391490221569035451503210741745997033257588836022746502930319742498660601623225569817090589136874195761344240977427674139865890480005809616523467250728010089811009065805578250348897369804636718251689565258448927370134133982044113868614568727887224903992294702605854343762487477342279111689592419760893193225114890176946635066301166883795731595667330500768497190567470905412957443010936499375408355670789570337778370321074482375374328699166793180310272410493276087581996767531352745364535800721367067365699411257212702227042069567226340208296011979565359615884428002882707898499538280053959919917494069303743540452523475352859835214270471396676956007137824367641047605020174081896422787957053812579637376541631047638980190601858743867858812089653378519146180507022618780263604368519294452416168652942199116952875926940308706060386289333855000555122745730526677540485512473707921273149232186276677411300097049533840799029876219798525798368377095034636446448389409460950156394827234182855341941024839124776836861446268995517276017210563092724653274594593884451709649405235168641272028775255035407215490698363949392451498685722939605724396108464408554158559355034963463681123907851539352267710090903118879564105734898433771385785234851422149083852636035419102663667724205976410551797630607255427800647213717346802149366018351580009547218890244396064884455195726319747214318281976387675907783428604693721805882299074821345901392973245807157950995813597977424904993670795527508308733445446573034723401896349618212253522518707157169130095463231915682070542750113877671754930532451488793919063143540607236796063137084898659860266496345162140297100248022017828040214350228801504060370924148729967351077410391455589236515892113625962948793578570907950007751252346265239812104220432084979737199714688045451271985666268027554027139241939527770864407486197233608417286516751218338078288087760289791191900286340895738514132564429686183854830825454532705485640040023648389127800685482685995520717716168699652852961031372516480825576231951140724399817560344873246859759275545425633444756081243688267604099015388058364149140163145863939802447021142224377191387130673624313426131898140272148472881743314261346515565441170763698868857118202441272935554272910205267078504466957682463383931434000081624082786850079095974704954643691338365214611148983844618968759988161668298955681662038733610407062684266219716746123536760620181116114551066460054572158976689657257370009722049072247517470037480577647805014161875741284165509450172506752928904791710069615272817894480556262662929275351089070605997166657654519676273178286475339174722918507638462280281054636832384189980484324160842479978258553509844381304520574865613364790045667924045491191063455973729115937854067338918313155567509004447443405376827759035968354045722720840983290794074426480423852283340803333850527847864064351934635094412845862413784114356434520367994162151354681697249887433953129154475298072500159730879828949048386112126446048152797582262056499653208918037513401702823112060569239119425381155906602935749616545385592402902986508397949806546208843483800441154054815129449860966310789040980284024073229933141743202883478824307422201251926515428429654317180984917510753492103216229559546746189234448922707900136955622396563734449724290094610156353436536934322450615018252887381345830125159268242135019171242880098145823317099938161239062744165096463476639916668845258498282782933200831213226714219626043508562449500071733176344021264524887023055935990918328153426643910132149357586946210184843569567858815666441935284534702812745040933746856393062399159701935259683850480109744498112892029721046247122858851766785444468414690449927428507272926169460812263104349599680544672819493773082628797165444600320373512445373855641619556699634425532428794658853425688754301734140966410944348170917237011835445449938700910454196460870206678224060546855803736809971496665086721113965568318370053617453043774552676303432075652930078336237475888141059765387876335569657825810200485685667569840114955194674334480448424604326304613670627563587307331729172362126988603910899665413682818986955949457226452651801278442185531160371246500299426901571437702099165812157653439242537518589554735252811332195410596355910394762558574318708926759446156265855181733243081877410054844715888651743344229449504348737486087615699262029157072507740245952280267849972132416551111222482951675008148607902496676190427824042669429212087099421047053381576519296465649542068499257850353555262124980813257530719612931570822324054189599577963566003944845865826569320531238715098369723969667018744611548016810952905099664784490509541809986372822352389351398549070547399213768857530630551048135924837849341951162499016279757756444141694891216457024746830939472472558318298511336170455221048438314967260772830546096163718731492000723680765376845351550447398868298811833750514600244822640747969692950890572799458552301748447354727097366674834706360880354558055975159758681731451309724176078572216704487460714953268048254391300801342304850524495028714792255878707079855682470190570724315342405086481460863153219862168296875578761821173883151334076536080159717801720639067537594803642916047563317560424255157433633267016725429821100072767613486912253163513445740579690032225474336244373590076985643184619635538794060627347036751655973937702459851979620650211329438683403631255380826502227542954198862804695005027355198130905023718790539564068835671704811171713971818230794022222949682042097031703067746719182745393042248731952224218434673334681428213356005757567324992732670335201562147303962184677067060119549903702856585116169224129153317412931365022469645385832975346425179639488091603463742381677543266093151038735960096430800471303980666061811309321734228473748305372396746905078121691192948286707750332675765623877762106922959181992713914016365508744309165143096216104384981232284167897516967760174337493405437012038441433462911968048552405254210309472994067752144205198561793143425039277635448562516010286370725635169713439323165420117843466174251385542867592341587231623519139729934734600228091859321390885287987019762290607124862469540231628345627871105578952331805156768008110721206775971325862127617737662222649290244555883290644997992422449648585204061333688959255376258732227120904878638290896418205044870583080625276569012677470548730386042260283980234926338232098076426664291945681577938985721106079390310214778134510488476502280465284996513080029880372455200462264916190269178971081589375890148194222530466513865690433479753830740538708744584748474170230678295524740736696701358581850783912460407438803851246314812156925515998485235329242665402644887381247387173712320986205744549629273853493023556003990941627836049062092375013697355177759868150779860027636650142014069126811799797053655371205479749164709272079542259884559654327198216359928183932753080550927824736772125546018868719716646270247968719866116356257626289835921612556381433462103191579188179895281391124532092652674291214106054192301007859712355622083814030687437156356241630428390870582209810294150736120855862752370631399081647899267337288424998647380959175174541696404766458748506256536038743339929973226966698197987524427429308641638670490060513840613845790885524410920394410862706288136259086575926369358322558943973099376048975314348688259966542698303243139184571676607111639029176637536280373728618859790576512250267004233095263441666959315847610684427207819305320997552104927412149769190259956637321446415948934207024837995806648608513373269490303109445035641956364938778615763710948580960412223637089767620119109382011012376518094635320015904919404624982362101814416641771785342081935125066829731629919658521196527504757982147423632513427390487223108789398780286299599565587113602427931236056186394327904271642442219329178210219377938494700575438674954316766348205991537579017664937439661851963299565557195910756574392370283938952637970039189118814492432345838612966708830454850885904101504465894375639230531370415421275531749855295347885782182098802217338561171152886971675999774825587348686317189102244115861602888498603655315759283660884665242192241406882393051559877682075272759667838988129353998432249960412477828986598669259796483309298869473919357076118149362088771908395861619282498471925898153186794809535642227129813609624003227959528573392368018301829622800630214644812312148456560195984082387047742743352904114911724891171997336166506821399422921448074552499923843632605225112777410439072668862078721462551484263232470476233075241709810857114107419261435657559191043270729312928344523585732115996240831623766393970383483316622480110947567398963567346035793755348877362288517380930458021038700098637422397513709404944639442950091254885614154234489775154098222565687150452162348604196421067111545727644616093010337591264051520064888867680894200306147909618019482690086747289872697813628466054185366697685409969547854328744187941723495604372839844375587797380795219717058755952759990369219172140689053315888619175279362845871523679162379980376717704795255012451574691501714237044922632797724273869333720589379491670574255518181183525512678365749236104804577876611572624101910900110211423898759868989017422869169872376924733762900817930883407240567921739244384617661949368060452415450168286615456023317327355252667688380068322056414786712804894975114796998372867390845377918647882913242891563207911998370666677743156244699718141236145578991521692783881635283809768411153731157014725591895771063616148136967310521813808835535874348368723389221603034667397227372418947523619639280281894445742976672558648383173108431880484814854692490088282519055914354110879747484442248122036360539357621613279040115278974055098067283411300003634133936060729918236164783519098630180255888154885797626773076012729787228261350713193411341025607930525951577347424638361223734952881255482513077390970381451955925948544297944709516421180704977426079339395522806795754199135202708990931667722773717018402541916405485607871933782190706149528185397429439710491793081436098030860657971369543992820550819856043716152378574113270643337602300187576785338059934724938388358908392082970269800162070470136797690009154485764619981282917508171229339537379941839512485340905187785860236060106779805737894646439877656511183409571876094327470800684586601296808735981879168588290407004908457254503127282163452920169435658008227603047530377852153512078506518275707987343736266150005184781412872698625827755944131188137245250931470944692273901613975953181940505585066462600673221872282420934872582822524464117145017339324983217829898766470675953873110925091149208264765484781677123642068532676257146090995177368221222076333514891942607361976614988509297497247047901771692117338673558630674416600265770977274425840769935846468868623720391374629890798082074556234539152238944652256748272395314888306443249487981366405346169068655941814843686097921997299683123571163930317642935169843866129564124705044002861993255503913284130487744659344793157017332955535633732311078461707545828973778617115999271848064677929093529956484338182564513742084644901161640928540373076599618499834351750522249726905116488487507284431867515931391742306480138108723856836056538952147597309680121195188245726745552009833863782509983785780450579980240843441864259643480088888211215272244518475825041804925249387548134194365560589385728049311612638655177929317129275948658370395014837593331320293902638042133738859358988190639849372698803817717396077918704335492051130595110891512766619076067148050861542082149913076341009646057645149675213049825823593771927370906460302033981035472344292867359895076436181942005147915034953277296746590023056642028444188463212877931480317003578769660994826546840582765564930769688592971134126121753684184516176342587321012078123534622914877693250364292656478267884126295589053109931435221802972027127306759262347849710214394198289895976478916837324143979267037086783827916231350069378897392087206505936191099644554769975862315545272051813846646581016664708546561992831104177479045039580403920112711890506543197058247900578930676429385049046720546301501470831992839740901638060955586421506287026323091624161155784927562542076028595617272523369044099138823065110909753140287494105573660519899043744007444076855540410658926474353808668364500255631681598282741611492860722275956921709175936680979239650757684499587586905168980844072605053548208905447940665643973229931920243293625403421445623160714784852883302078351627520913127792909618830568170296476431721954372988442999993385835723900176277137338203965727546006652827398444408593411487066752245358907979159149942836946507110928665768963247230129603765720141589357911081959028892341336053175567527681186345036075810463571011017758747759221847835248913877438080271341695962027010466354196130614111039672587950160102020348137498903790171282965513521741883356790759846284309129084593993003680295829871793367832562284486727215200262989667784070386267553775209612475012779180665649058595518432862400471938803244459150811187358913001631858333531380562127493821384016892187131526021989251416554383949706372475102423846782561740007514815266976467891865309220796186609256477828391733424965639848485018890966111411531272723960466038732481298109007454249872445714964309121504268923583694207779258602937220195518262312593353838292964119259273942104708865544255941886561896388314163251277980293081625387297113565270875029812892791796713806215079541420584155081653621272281045578835010987988769510341390441589472235564980292143469576197750229264425655224649696708095839719640568163824302770623517122119064991226847735788706281901059826922459215872920661395098177392094460667560790631267611024459636539836241433424539094354464335531453548867207391487846891781368648054049393027691279927489369529857480795393459585734183664500570139183732751532861856226905933231404109531707586521724926302397917275664364119535513154181705516443869670945649537166893375386990187677318022007172080382860202180065435902647423187698508267941223168537934452889094078626079386451141473212895739053435515810049596570049120013887044136319710574030517331915898636785546360830036600755414094101832626329670817973095735835633875731016843506175922085825625567790918419196328362422751668867569671009756715183730881832194647471420409697644515541431879225764539021180084718062422744951422437049671934010045749930022382500228400895117881728732043209203040419843245376695085916560654250029978997073164125519452733143962391871838908857152038647955564269054804862391681717329747921734072843743880084476468358661616865585709541969265293110005668786962677830083433200118503810366916033183222976709303885579956590828377382473517757644282050908528372707287477184315532786908952537518168956384557607764923261603976347502906622333704433368614042979349288367279085645195906876492646269959598811228696236644844214768047500613106524962133217169233675544322733015764452803459079242638080628971856843766418853391290461729435282208909064553396428821021686447862592606607067310065583204544641909168417678149435570220325159688206332396208679776291992282528704385119920914242275420381472815753727992979315843405969418277994253242158604164160963162837545548631121008137485889076818716739201234696062299521622590284326900631742352655591884981331800031475025079256993449360464556986940744976622404044704004586401404533692552329437194198895151332310748863926042415118526953426230377200886801840201125649629252004034173503112189307766641273998015798409511273908239678337727431927408226980327946091912801883737612054047711781237159771571422953529956181507461731931727818677916823243079126407041772641406777188432609929892141510572407771669949664931583344755836063133784626996183911438430036550856828419349748142086850095406328032345056890936176975580462571962263372810018067201792415720107972424517778935722042122067535757005764105492899250475080031184315668208127359798285138674508500342539200305576041937466284384630387106528601958722227701401679448464347858804116725005528604931393922296165003633044540169867915311913902084772450923235810885646628549111400946201887046483583199826593566576787179538537524087442481785687229192404375960783644328851206702150712226596023090750364638965140663128568735852840250643807415166994143888036267744459502985137777161655674667571114163762952290811296871371856001505427451732818030671600549039870613624142169702351432749114964296772385729654454385400023362334683068656486518970938191890459394883324936806110855317817072236840009759118395231993832171262021316914682020021152720481750082602152793204899640748035335680796441768352145148858624449187188492935013540912992664324790826906945809472578323157096216778765200723362891550688278185253214038720404220309564049510697230722057493834344347345612991767010146114010241410024484777915353433948473517108024054394262536228683912166125034979660058697687371216495036631028471741230388377908266707197018937269697414856922152586798184512969598864679814264184332658674764507971691154564676991163811421719586970822758516383758807858576174533727276586267737914983510978360019994290144109039273268580624145532100481164300090299871279844783449599210700589250100479953687561212571737812151396770013337959461141753167074272427242205091559704241858424575278352280772557697739091284439919653520454884182642532038039529141231714407184096557780459349945256612318611962581939187616629915577509343250968581849293597680272385976431075335150284489722205750866611508021826538278160253875164984284150073107838296929080962162142401322272527002813395854726935943051786865856278031952037846440535962813831454471803660730222053585552995441328481911290572228002761624429436941560649885212260007337843074312759443142492210170156685398079086663722797976035180974757869840254117844882139576325403705584231403062117216228161186783097343803423943298407515529031487222949869840985765178647313636799737791982247478402964695999946002252216638813528648883798886097835599203320105657187708439235413660350508387678822446494220229229090738234261857776590981448736667759127163754454657506510665443956295306961123676489505180826522996779317802324958762305471603172449923572985710834684327664393850141980313440209278405499954218540148009534210050875299645162637173224208472075290850379663595633053153272996345241182611028936178721426021938376350316678359889670959338033121974325760030300520333234963383867107817485673119846024251969017036398980210644689770000605078527429898489573151467461897393808614224451760169362713478084109758951723977054835978984410168767249994594624654019809412010905822812663277894753884927262295012844074450296732248982120762126828218365259311530520538045991659567349259390608414749642315542503420632040347232507310817256797541339552683857894336713552283859875901674974168938058908999329069639880906337526036570117199894953562599389498307367803242710424905723413354569340628481807299874261262081084448911354384465902096974664584001525476337789892858440363668798194526260795835945583263375791502537160473368792475698855240003510484909855356659678932456024609112592123860270073621632153302605827128887403419943918950064253028133104886616096915091487577261692553152701437736395815890930396883022781288168474069748684514204110305910648989307101741491540600781258360478763282487602462405705334328685955154299530568753742381819449718482914298788381312100806152393221706222301564415382381847157634357684001759515757436539763794218441130456807742493582112210627908174131123462234100063281658196908991203702368832358952096969686524377022773223364201437278961857235160197999308143541663029676034283825385167600161326785203133410029528783189528747976748894000694989877227702108543591486205546650478895510897726640945918106141157298720325863376660880267518545172821988803555183494388674652999206509725767810183458734734605211895034335056042244412076926639895046503509751847835402568395847493326151084851119493297713729483071120202170389205033053075194977785234207315472456405938518038947124316652502513967901291269462577870734131847871026156402721744636822988872773694391725831662716281659890049866144300774752573946141642429750582619859814882107196132194367194255426291355416564281180743333094378470301575523390008190131447851658022080984937185796902606092489780986487686349075120549179404600295724016608012167802238180725240832742995823398383979004045433978185908465315230363725199537687915329102657709856865935706280157650732128812584264792554264252809702103125141947118436266531674845056294043219148756291594327502141470212187436138517849871717035153813338591160907168703199165531437845763376799836951551872264504931266940284831908230638422045278313750765721061919361825109850265753724808341297151330159772441143105894213717225305754134940269059818897146807906501195267394438759814690110933640516119928236755033389350475933526571483084511746906508203191710065319395774540892266484859689343784744878936800491144418848068548546405907872074512690725303468095256743900942072943600508630010851122995453551596367340199524720697806972165130694494313399787297500627266375769552915032206495804445341056672586854978495792892906046353810026579201046889548791448210587242342985714006586603601087039976063135031757053209175050555397708393609742737513392572067736554938298131437895865377952698899771552636597191542696577821767219470879561427531536178933334027846070362932039156542547741891705788781712078965823139159721491067805343902687323819916534014274634544417941205434328049562758421326807858439845202350291376757007612053982663589486607013519081532395784224656899814369160847662485080655936469034371903753583987015657943838077605991457900250737286514980949786281850215248111532353082225458828970802749393691869793537672972065277717847739628131728725067566379958648452423903939169664944392622878909177651127858021581413363773495556728864201091640785574217563989971436293624278292495778364404771781811971341582611670324785987043156931635210707362668194857416154264391022183411943236478195521811966416662830683989676691528399555777355244178669231813809141401705723762762548289728399486281827662822304382625219718650920659147397235188163060052940763468278680701160937918632283532728195958976608277942047924056754441956914702074589053980493989188343822339118472781312793242742863455064382547866676617993850852636875455894033134176548274627394062554275640430362593871188018363663134071170193597340796053063345659262050567499337862979165784566519407322986005107222768365593766243641623417928872079924620216959395684151706533337739052982382918198668314550034882358898387578476052407549119882325728436920554531295442074390324991531190976294954576241839856948857831040703680188595099398579777675791353272515026479935060977725211711814671271100064952097996194344421433110350941783117988224268733074907659157967486580773348293838799112749948512365129879587397047068980476814920522411534991627225793606735799096462965511687510020011179098776974335764464266948212131919495296197154429109417518010308363867680676347196245403580270347678770334093277938723810830489612150010961708674774675122613597474956457071599898406571058603854867213151864037020242344879081895881228255249165058308361998124735847296529834751385670488873767100677450646901653313926412254789044042218643433222051687316293350780034360104971899262901765904832524941283907790423617466168827754936443571534784485223208418739918961324581422421259967731691470748517187020658059198743749547105859425140589104435479734780007467744469643790298666557306345693129125199292658867134802230890707370569851451745882376437317145269653761045073020341933389592194831115441888671560011922412645239422166532658001070184905653106063152916638705151214733569932330856586315475663909369439875068786199249192204537386753306346771344887767930890675773388875392976263846501891855612173173163388910110478243750901865177696882584708491490974713380782901152636584274648339843443114857936204719639832751855191237669745004384035107606280390504688409526294183326451210859981390675691454370189022599531958613039169736017850949128004594980473701864510706721938809276073025115741976484355829289447809652394774011967652378527464951610723318265727733506235006034930681292093571389004407121321282369532270130383717165527828980349394087895366207179014066830522664458654290856996560340131370830000893287893042522472948921971308044440205849096327480717691091951345160380600191131068463116216849307199988519593377700692606885063917517523800316491000234858373682715785627380273832938120398080866571288299053227063913315387648160268369682047438106695172304072914285641113575978026841477421665734516402043670408961095786786447188413864262556220715213871852190156217408017764818807110018262936444053051763274571562538550894797940268249478406259301766851743494770226993416872265580395198700304636441851017485230932281579833179472133117778325710799333565440949304790379260646727093020172653397716730831137624284031805143175575691891503525060361079542250480460108435280665578023254658679017618946247247573131651209542413100132008369047364165319853790514948965875010728174837086755365109294661610857916142739140164261109405738215571278005554036960427821981916149734868453306194177482148454599841800377065136280549404117579034050975063370262270197403754496106011930855484579497919575285463608949198978309057256178599859653093214550107833122342776155824490964555482125947833477860877629308113013805369250762068406132233294052761029094218787141281575616366346952793716157164694262408966023104658944120991664949062508856519774071638580061762505247868775545376429648617181620882117957225354855754672736290091400251467130172685429937186091083607915482678807005161282482183834573778700962752504253564944044304881200624645053574146250462316146137343930624589695166816900009012743806568877316380974607483832448471476224592628031548305893405009155501146669900238617084584290567237053687726812602786545533267863120057890118514356961541308514820717672379841260908720537794002798311061075034886504079407392366517116651712625316095278743906128983657011968854416787975954326562928044715676243274294477187219552724071509266777909672489051004985987425196816497625858160407295727733990991995661138092090368803342809821715773444330055490382289873524773308315104408819473785738436466844772087323254732837200499870310796138033173891711195706973441642537318132986420209247333130132502515018285489995264683725856369549494782040706757380009610998073507688115469057123609703310391552677470932921591217895802553355762464966432575924382782044176869653348659634993914986261139847908566974539265919931267161340701775971400837796757121393628277035626392271114704095056536322204531002687018076444462481060768844860491995135167265160899227179263059250636720839576873053317083307332577045155219182703892518702835164125385466614504022625456066553081102028238562620957989978814573708438186198150847323476681890256264428554538073387204865520531136806698381709848409647186073494714725361487135139270928349644492149338058943048941911812991168965602514271349991995891786149203609111559850894125658238027280279218627291965954442780796947870130521706067136229616141501529360732361877226388085546579500268822821353498569806810984543933373113891781101897413857244574677914158161642430512886960182760375156107023658532697332607063085096659294586650521595203875500086136774572119599487916066259875387398283805517679588532479331468421493614254852598847878680353487851235667458467535255299098221976445689726239033409412619313350061001389274457217438842646827715417984701679524700634807058795181028082341840352948073174801245657209392962316619114734331084671575413435991809837974437434264442988139786620617409270506793101860875979513302885932755545584444964396479482817339620032385678447201706394798910834831026851987446651363878915982646966447765679986130910382853197229018802161165197669737060704932567951532906076272939086925802787884417127155288957790775094623773951189345354675711606700617530961080585498426749681990167297374915659380085041974651195996489331911261823734411731731024097816700916973230986389351015789572555088777410794240717344815892680483857274193267895708502102287120728364301381969010336227892886553972107958222479877046323795468244003832149093952420031449916255849026186781606029603653130353549470839018598373913240912881598071797603384440236383544022909385280109259344942015290249368575323804307636751023829935792131419503452607609778290947352545772986193217052774985351638712074965100705160259932099787688505987520701073124634758014732783899182384302491689000622438581286294342850711882402238564480253693247805282364608938240402599610637123849259931196574383169060653995690606823988655691709578747059269653149569501720913638508738797156249932452814632042123543280934897893663122157181080254013209676189550205604925659240240657655555538852291227145021880428393392114070674817906296372721655486720296113957026402928501995561953407894661051802928180355733708904988300101471096225463097420340853209299367790473611488742647829825696528658851674786096041986579060857836773395536366059923987750382930377286261107412194875349050559108555236933470374478198959105052354953157442664445631459865854991665190516347987515656739667921481825490121320609871419911977469111683337121991522679608127698125914982665157102027387120699487802931415331685233050866882021473242575556249878092849014852673278135321730818403105060482196980019673579973898168392330754318433474394700633696386080077466906249105071524836205035172122744100452721353204069013509984357639198356778770374876669559836409147539245526371573235421387697786589431819872282474021432045916423505729594587156096452238282248509828978324579174407720565034636214732761459488582780651785828715701538287778471287969938738787092464887090821699717799823503990451237317101380061365320988556415264355889968537255037829767960446249020048793969635136380864080861770731225669242697613534653964520589541429738979101542140454504926147693231851246631078753449031960637072063890481632744706980233253887210241035923101942384333200352295068416655009199730502323510449883320093182152201924860349402967552873201782116280121239287626286489005626863942970453167195664048534088631941705570535038604624112075899209977486491787632937004402164640641356448174698618716608130949171807004991530441014774625652155952448155022458953863954554752732355676064687625138579581158663686593821786514517430373461657532898134097604051674742934336915793035314463892445209987849763760576296818980478791460456808396880526096626479262240184942531330951586605311364522694918929709230095346041626465198412087782311512150100791809314142295257752424989657659790099472067253470413617259747646213998708886791057984854235727661323381876329366402745291679634650670891612339293852509779520212317515273813161198670718415873193775226951879966567561925206955442094308400474842163825943172843522171275280151054228044429609356426701609127369783113727117086823624592757444467903935420461571322508846432431385504217477834421698043508582295927436495923749450542034567066927578930924788017506989027336648955614098289904049901892331132032579446716643873731874774112223401828929357727231648218545896723125396133573191161751111181996097583387395938010424745028062902370422050478519719083131318875991802431679717321635682426592498732571109232877944945249655260405291073188576736402311656335543297059421221915134425790576246887613474047445490879158347536571047931193713727595397505790093525448864870096623611930528041879270374709624579284886221048087937420010531448517897698874260852707733874761588124765239408116523387530870080613281384033560134305941525547905300496824104202708328859570095007525269624312215416386138443538334631786002590100953562218177873256775039458588741267498970305790224611359729813760255467877639973695886273460732928874925361778452637531664245508445457984557971932615040628294950069599895024885381030696943286336947121976090748411685507031740746819213121513132787490555761211819259623720033365812504349431231978296386447544645580559961361987940158977389625060099386921582429182762084272213717720366742955569470519163842031702271457516243413268873369979633440697207493301299192296087911852579769656466753526118330141492415800323463209196324827064208697211880588164054508748500461580291049120123296807537561744942430175727604024922889627425837438783515058926513621780768188574004881750283600870069849111826857455732095764600676146418910502372448957121364905159625187037491705621269664755117417353397087881065807278840878336668674950138249579869599329308103265226412878813706023903251401981245680900685451235027421021067002058043092046945112415014832219138162294496618075970741281653820921783708919411397756961102859357551303927638995428756561099262920573793367638241253753342603942637461599016036138144383347794909220198433020261170501012438738700118546987890835783028089795038724083882486417315903625282608830135410104827175934438477870364160821986699209282929104785006061058137147159423087320087098769130146608407270107291220518602024207525543717147489733902753420317634852342566354712499228055731210920173780009245512363914903645134121944351948478764498502893039169035729868212205965205692115138795297037544347041920099929302781042828072823724778542890192061073939696156759305072649503151491797730534265819777106794995565803953061152615357913948525937615935500598768401278533659476586758378532047974146726204124318147599653362651196397260685046028276871994633468211187660981215237944355537198689704512890732018519383692608737708058492314644371493154799056094997111406738550409973082824776640294881185828135397240491292772133453215592526506806894225496478822607334666621730030386474187021173331605841047912546268930115374212701575024742909195596313881695795898350175891131613530071804418059162704841639136712506159720598419883107458549506317220820883610130647028638304587818190682803895787588119485751655638238879128351211017971321337335587741878946239972168146862130084949356853319727895019787918954277413680657013550432121080747469331472325912039829666338045363711465402046641545257152456331842369796115970946322734748743127205753305486516451260022898898875640529724700269793962751475302943631296960473267795252829303109886937615597759212644401739887398755621272932755385195078572917353356952459115657688903753760962395027506109726925509814825646648465725411900119852767566145761832868928208095557476487020496797905991117504937316994313786365436081804581153699116072564578341157819143247562212245442732602804460708201762279505213947658196573272495909431680842970002627448899897019479185024957803257253905996963312944045377810985431972211637859263599453940448874068140959254239942425704834375933192732905950968498661492071549845472787663849668497523970298323404")),
                  invSqrtZ = new Apcomplex(new Apfloat("0.43831154566767452033809937113724517918329797653879412078581592646466080448431552635817577776183752602140390120485511565996678561570417671460338414022447635348292881303321592116885117977991089906328816769108576652843641724078240740756330975654117597697860252078371643026148568319643017721235202767127271567977230498816951428740242825881300946335387922796476382481919087748230195527906845189297121011239814379097652925429697423695064745186914632634282936290207393484170136730797785283485138799463474853279113716617864951042534870533131769946790464244641397572446340701892225335365509694462646376741127526639598684566767739631852943419183324459311709988810339267555796922783520108593085399113129960554191286390194139776179179090480534351586148512970698326695772581093016744263589241527033204910534116802590312751892987202035313724429606502630963993169957390824710131749537737897194153370174604819937797047095172386230150057592275565241840374237527300335173501487991093902079984318185613739139182083935672230815249514797949446736522441626064496004185907321063000297695670347944466607747584991518427873595322396063266095768894223703239348999714048603738668574161529893362300091247525701589531718866460048500890675066357049661517394708143649732778433991647001451719454055615954417821771773167609542389527137677172386862156197498695990303859546807406391737261361879893238443236025947952915401454555892580695878629961187998700810172520213405851693109969322818516364104489021155564940317414567578908168815615723387056545110669137353631325852123554091891547119191706738068525338748567310984208004965911499159962346217934600269269577059667647869472021508158784396911370764751820053692715602286155756544552585654075900466534380245892003587563345257988719532643655470069913002722026685350551432589450585979723476811344911403900174446327395390075359699050596256432240639159850390659526881966372081843575342575213473499489423703254552830908191025011574595487178800250989237737086429701580753626633762649798341606046916010776483559822806490613248057836779271308802764423891143225009312691300360979752432004109221726811873739633115202712884839522670131260233533943250833201496255922201423035077079576101240670226591766997822385374378141564683689860716230399290851396062275890246291621542669672438457782128177433057753250494418064797134577801970414713798715736353018732360488783139871253750258768445369342080873433874475624241079434259358682475957005249217995987931458794051340708988446252726387508535210985718666570158693461844344537212025568572013410098153614956092947896074070295884960244487791054918316568359020237426909441630851559051064468521426629786792711072931764859599508287626335812374622718992500505291884322178613975115196511762324315757012379367029407204376321791558526847759877243463471099220872538114500533784151460001933002994528651184489453816790758609189499509953406912388544594607055964433615292544706770412494929252505848127971236261344952388149399905961911694732487849579771739757299662672820236636068249431933169926580181693034784610208124366324301014966361460653504626917197924871996890871720505312861396666731321916988363461909770075944450373811606111953766384442709714288134559590674600431323734465821372658066913860353327855393897874489382777281797121669169324702523395905998490456347043934969502821644638195407260482392638213923057971578904670151618363075092955986433420543204238644443296271702541875105938439192133630012746558748982443824588117799419567736364229573004844879114980022632076411142899182007245260236075878784769742124781994302673634484999456078701971040628612138274414759345218985348915644644155671071741786003037908492681502801112172537172403620447757191581334375438208343557190463071272522569356513899506016096221333099821933855236733420052366774090997223152379321537222956504615365289189602574027558067206627440834892592001540541185389342467720728858097484634522715957454540578536298270685510532025565479260025225058296621035197881078831706652029454656375786101832022121719464934733878967435270433735246129793661820776480026805319571035546240254104582391384797610519950542536497607400886579743259776398238845729335881797595334885296158879108536690293133351491840595670169832920293407826547907570649061513306611950166198090238451753947143358545639096750119101083224612154038958503033730831300022361891878981198054175297278615425236525429067094111694353729409258533621730511621253577971662142402310143591944197661743196798125339279953596538124065179735239549374664256712396757195070543124463497463984689731698530546387369708597331987692619815880161344249894611829193603559318499893083399481438901493423401907335862418830562549423440126785205643437257297976332802193147624236158091138899634093141863108493072856979478291251659294305977661831552063865026565321769203111023650075212272439907693961328111698835113845735014362562643135085362225893267912682366554730120394559381863451868142844083904686274861670891297971834887553953281779930787290842706469039592882360111143567515568558080446763021986380295621154759959076997697916896976310770419642315110437363865573364789816797606432569985159047933992366730881955695741345460892995257660015894276782351869734989720604229890926375638345063189645831367269299838784242720824776585733511117474575189140953387936266942453741709262061073186841349614147764772836166225811789222320646698495828038938094012321456410289479707102861758861202344805246461991425721902242822691549297145288744762372960763729970889288401539554124377600995553087641561846455032759995727656599847062810912013810584510272739954263572594049648512887028637867188769347437195844027342654954776680549439833703895672290926035992302072061789481062822725650064823768027195808424005247611496975372283497144180119255345210850955573415206229323244115956298948532773151713230615912742843514202361609010068747243033545979969243890020428485067594103138635848373589755925533236691954483744530883848820163367720408560714099548793254274588815773359489852637180859703875632926780857624446079162062497933949981423493718920050777513815034743020624710805062984663346811718637093599416784764157851417379001905748758497611208007410427445069027931470103903443456115485635674595108167512465239559951962962526889314627486975530501669827552255077735020616914605416712507269568579984513773949186078051661845817481781444805107204262301265963719612687697506995705630481940119468553962020792263346718201160941717777525511616498984391294475428505099808437436667751939070158660669354563072155116681271969753960055457242461197888797081576521281040685076853411754310459788199767415515961152459130992176802409945989738383901332717760041799243017136476063473169357663686072936851889668712766906749956552605323564451682381219509757899487099959404982769104952573760578674311480654134308346705991778318474399520129249815317780395951115765556370662875024121070093771281253900972784297197363143964169659271634244705595236688766502610188451497591341885499907004644527216050390912837431655194821203693996265603568520719194131781773736277081594756869997648433237566169053762601596371481981229884525115990557852629523872624475957988765036238149031924306313062403312093696158897612925978521804836060179709447551984755094449037849145699730886722892979135037235395977895053517906034425480133168633998173201123735107624996193579757989650246160053029484259020718682961296861626399990740497683895445678444356914451709601238394536965354838234684816009238594043454806127142801613983866119595177520667518392103786334844341284555315608596016572423250361300637973168096711228521475916075644934235111369093304640716669900525924169633982406787253724626975123179596108521138421200318440536407595705433799639314488044454887626342291768972602228255558098209448634969240892365757961646508927833422888696380160900766602768157663289862691834972385218875174288270685930177298336684062972970541898180263193230097183995770727950288538941426290519076922825764265905481362920087936192002035676322208960614420356442480581269313708865779293045415742159603657854148621335906269187462861759442493933217824542298337705692842314676227713142477975245116148894303504756844381384743435700756459411702997153291099596679753135223394724750219109416634222597004697421928302612654694004987653843310198566832892473511268714490057409994699556905985437519334823257600401871555154414864205813483208130683060335265061259917063562719382937756405775728402459380407866710922317846460285330638143418923978022729923712864495950882195910179981247672974085838890283736021046222238099829972144708259819535802972478247441012261571562559810910160362814591063267754486775361393316097662779629564447039946780160107523944668236577725017496454601866877270890733928105479829258561467622518176026912009698065930497759977465724210373805890965498099529369202970469873750493652504036810908521979856037301889144931310795656671387979811041112261476515536454924590784133501282637494990403977805491103089855301496194865618092592030294118700731456847976201965808650244069728149781791510952359226284058420489956431853456100630780195446282582560129394624567811599526125269382858320049446943783528920872433863364244169018351444415122067898078755024903957125010630564669086693808378024611732372537046944542853709789696099988027526696679350593569079866104948562960959462476961696960478534440369427340163134304983188676073726564103698246174273747798844056762754484369288011143926349326129646279511761504187976476853843488042462992439144488927576643076097685436214958270988466948457177859970044806363149322266801562517672329820671502073023162402294132590383832908409180469510697090305925140806546979320873356877700945189293244054006492028122397150826381989670232220643876028409895440399738147024992524603682000813230543789450632650953527145898011085508803829117453712806794326988990682767483670970421518473044162255108966891045321053969547011996469360628228973262888571502581007957330860232034618113996920148804859224929233512140530968394777941525374906404477386768236485408027741964992866833839418482635398593315243663855409600373072941706596253401952569515194618391930579800065627451303610510257521416070631681252031484319440518193636216734321873079468802000449902996533961953525501983859201430445629946343004972921652867340097404536321409172082930718164636992702919989830198061380101014140671056774565636690007577121967232679040619084558444244480112330539212387889539157049403297187406010008143004790080330799816612758529280458482114761646223388516521951079786457485132260079509759667301274875895762493494852979050845521819578131001809685611516137296262241698405839445524504826336253752624720400956097652191923738462272845458142675440962186327675858153849386070017569889845357495653630826572114528855640455503204467110134678092902581993271198000355587780639616511390711614776754706066390657962022417558927698209667178950627135984063012878090613532850961686376665372551760029626454677690264118265718354289063608596464536564528257150002991247007281414748772536551905013356837040342261339128533883054220726672778298825788510007498864867556936893095559348578109247969926860611309225035393023981724448920190704020928843975206393687039753857651113961547910484215650865396761469985556047728263150253136993011869719695713170277596230155797129636271367164535745716257138055845970233240417841238784009491291879381392380830472192937572678228621083779718723611953902833794015614011009557929687525370612213353234958718409687779694821958373946508134881918860184484650369439388053153051268383478083939929582411558305457388684834667981483317666268571711913584980615581910369067455862772334522842899864776159884811870809285727470882568006923498598914008897175886937413752096103582961001844444127980540353977799835299191474375438662977668982833450418283677997360239335846736009872126770173539320799650438898573120629975544150496047914134733859922187384762107637889232165989782311462265669437027255783349897631619340904530798660795846249846780049478064806658577469647793272408201003665109246338600415231917424187232816949873572747425977719253433184818676984537938883069311266214907736362216107643048680977117435152716237536426600976572495272369147584452724222443619258472952651385864088408577082264048713707172568831788346069339021091151320099734325971796790371616877821711470258264989582624273089885051784488610562937390137496186247744144754065261670301958183676749467002716939115928207783980930635822788657306441222147274919388058257520483529132691286140427323251821222330508922688370284317617320652178600193178482879603801698577213479783559038899279957016183441597537979034039555810201540536390080684842180166699697835947857463230775469263990335774529087246041686183680977406498850735316489140348618294039714515488585696257215080874536373762656054868265112297748865274692155459951207617341160558206732618948883984028588605727103370654370781412707306739470886264606009629958245379729893218516761277519906186825389038829195404913598072574881089701648051982630812429996020776295056277929793520899659853981030078045825473679270256715213600001634531590427759396172956111347135115166712396035643705325005650764541865669799173601255263443478391696993185185199448510170431239680569016708771943443098090865402824391647684403203186691064740879000670612203951871650147731408297749510018208638095745326855988661538609618692257320779429123330800244320105136547453445503145384908484989524765836990709686376628650694463937691158658550950004090776143965090789897173178091602923092424996659952502292583301557222837004536536506602217996193541664454438190977068911576318681969374151437682199169576807472788219392285684380916806393205689726247150698581882081990516188616682748326751625856126710329753570545421878225902905029214744698161447080278377373422025749299844880401965528115549093950030082862044705276619580355871826563331958456164563268683289941209916705735848884642739123896726909222519414315007315895415358266988340658192088776121249356276457445375035411456033505971173642321138068781892893727661531756847098094708066720099069435179476439607562164734856584070390364652831812796903183517105152375487940282351286658403333926995450062867956343273480592901044586565377288167733093445044431398884307349558814747920300411640079322084779400740841990494657827517635402733376762962369256420606146459001541031166441678796051174909914401964504892237086462603630248805648247484963599251605084742558705605574776625202076108848945283957495640817455282294166926971519344625627788777009546244997653786635797160309218059085327581271208292861146650363678109721197860913843296292563309601322718239288972606375469700544656037013097679843538993772111492058978745580811345819051002346623940861551831799786529922408406370434288017714160023618504084846051049966060973919331280038211749470340811681921334876873668076868041533514874462650391921726770442384976130205854838347877432637414751063370393855276688978852521072716829438037978261330420056982061658824660210070223997873475980700321714441962317272930533130745420811872326701467114270884354495678382455694118660189026561983070129466252330771209296975913657845763312465658424452980482843117522904106734636970928733283532716352805640278840883605335579574064224443924462242056222138744899395566467261316516345393722020141055669616789530059586046677289240242553072826138643988385098361682642053716541758268735266725451377773519700604680497085104363979655039534966376178305728768303682123297978546002516858081949160173892407364554722125939187458546157581096225447226405216984542723171441858191971368607331539647783133183121127073881212545279978896322176634354153417945613520457945608156548579291340467061378577407161282190851363750131633119875460711496243175311584785383782960534964297948755733171062100082667842680493202207628704116159440343251438709793681702896208612886816219867916684076213418394074830052790960926976099496583356916470725385720789276448422129501738152700112108840557141665346613072216226648147160328747889888313410282624899705163590948331673122350734756426577815800952656163703989170744397221617169665632729327310762151523544828471692749955212410504191779444493728424363763675901198803914967844847575728858027665480558572226743699624080262651118423113192672525207390765565215009894437630695255565533854539344645260791248350399388483069438549855137205166230037908792100933218319087437379861486326997851376134390891231590274087041488758928997592400195714210610944430299266543305307269927218984482045221769784688483382949547472804472181144237969192904171569443215016247060036997723184336907854216361381410198337840198698630001350392642725440879638237881474118301352161441400921000041665315848556448116840516352882271082656493027513647034834354503315339440463078828813005268551943682532840305661739039440088923741399497945001219856517738202419039877174868788793002701313836541598711377078987189301362852147001919361689514939761819445522443205617677564611626211044056010660737353883982712741746016456602891754332543548240656403006067031544540489536609283382572711462114944993146682337285944388210233424487235477359188015171613947389762577962548301794355547745462447839992428382451748515267827157557783382427342752425645765658386423820543626627933624292982080472529731079553726648564225807412814766143976589414489954763311924960736536764607122221355240059962549287492968487280844303874151105728674406751259890243652236856834485968018792060377247697662245656639984698938292343016984162488575446717191589718229321689478064387132402097867475440858033373679258791760377883880920659552837281078327045973747765530311633389837718928212044287144427910536810818548463601827167606712651057845458198909581911458766985102737311008263516138516933291064609961947142804724194618577158798682042188763783394247279363414419116960623785687819154961461693845253886659677080860393922539831737971809058614482766308675663814327359598750493550119540393097159814512957224426478989297708002711432885879711251342859140873795063926197359758318417478980332759986050087855212731483585955118514555140014695338620073641043786195872563778342299795171668881722638167264521351356876732063807200022077460773884027979404368976606534475551319551636632213504711312233113138370691399641617729829090581321962793209017514227703343015773641434837889293393482935428498762877731201767346761049257821133014296505723125677151529233026924104018373377063136247505074905733660787024891048295812260739607352621536806413364708882852759908919710620034364437536256044811968577474628694976969884693342524551124929838223175054638592222802329805549880161627382370399150261286484675201323708466619392304841407783095936043394223290635124930672030150601185767686939172433281711181880754422892414514844891332875870538708777306045119702885451002860073466128814879831100979634628763744994639228729533765157485987784850617356761704146828997018071968404887662891524597624142601049211671838205948497035435052414681582266657642358315840787163880229247366437030548188401094732396053589090660667964350093657767459074565009718821515707350877118244388630326766405211604473864467447728011874530850056077806743077971639014898297273213371702645666024905229367139706477406329581307765415621635798254835179457673670697467626176120768021645379984472072514892366235647576749808523407715197903273305451274423079516159318746341156258348851427644661899557134461210749336726863976138561291743940571545624894315354387090024833434264642019053501379707558736016004930799111331023355028406243653294419167258857732592389817598945010360278213574290721334073156929333550785763292568221628024330973482885884308709824257910416515324618964536582830418046126665211143282867398053966629122615953248486909004858829396182417690314339773117606170313159076709990169947676744151335609045979325471397042901735288812883812141548626266631519406018047887034988280643619938330715735236421748322721897751263360444925079516237387487712463568629905033770004785212181667560648268604726791662331754753140213391280473774656512414839147973129162950683557747717927808531004451603697738135176705686704668353550739349874950008640029807681247757591212006000325640548213841040453813346960645241157283818844015944585622286191225122870745775129060541325368745482481284013094257657401353319012013899181286756202474659299375247738359985810716030156211007946299048350576678330326044314018763074348313709116552539285323485982443673965116936252006293005826302716629745333339268979592439476349065649128166938878496710168365955677767184211589143972471467022374028363773168485265452072132720735463391360487030457569750441353210031374935543400761007472884088509730821821656034480600057584851933480111407287982570574914017636281761484700944402422938444895667827488542066621881223910426228889726772472389264264424862395283469217183958567934757371366879439947617000656152952280055326914655953610189653569044744489539227580403595897929330602579707322451163153369458895656149361001958347391841290439435836577964930325630876814127265943930965836664342544061086653412530437272870093018355700930849367003491765590216557065579054595886101839181786852035534885753312831294015382672007478261284730920531431234917448270854895523179520158631256128274531687388363964554409518368260964268576112034436249682744522233806375015998398735102443912189380760372706971816576069463585106282035792480776658606234714736537320668765225046588263030911894596002514449808736375331172535675484671778403939326624504027631090874676498248549034731202295116215886037097042002806078198296011810142028370281755609668680271775906590068088377783662146283545190327493577090409334491679337070370885602925165076744618844367775053243481118264738404314308080319675322792194088324773589695426732953425899295400349792723341442183833460411041600119282631255884293661122426444783992799332499364536044170841499690698034679190595043399802381648598951266963248306191599869584899001986041693867684419410043011287574785383235309603136004096797880075682091737454124964533008719280311596325744348046922877853196034559192773762235312636545432404303366682301541180728527410331265625967377428225936171468369726581825140306881560040263466094516057037466724287081096906237985765709625857165864611731974397444552912178520978840502780358194688456086686168841918052083445578793649552288827363972699093694978035912186545886297431431126761386691338481538167639213227793834064298864415436718630511631467117006355859702524452982411498986793819643801854709264430886422435623791676561882119621122581228328255552446247204130242891833300900914205816084773516361869631812992047296090773466063336397482025027509569015207957763321374686027542777327280801970195084047207923110511885688480809874417381458626034800132889670967519939098144472026604560557767375726602969497311403765347161564771335507425116606790099511591972045291741899948990463614087651973550868264233549418052011556049832280979185192891992043514045970178281135092298846903736060232930901023939130014706413013951918959322138851942675032277285884556386909757219069172379080791898010912254759841772799676796291493610012277812179493775454318940830675592935484853547984278414271097451840539379190047286267532487319184401101964701008267404661896595490491110412881019604951673635496337588363088597457950848904422978243482130662042184193915475647734617839093825191507091612334777516872759608213824450534217704540087297906933079306454920656082441968421781521764283449552539362983320425484005259864384472297893246942257736078719716477200395597399321545631757816681783566176238089396849877880116770027025007960252428306475088981902519926830893849584396455090611735378167945508507644043679679201230576891277452685998081341196357107460736280550132927641306219597600171130836602493546593095297690551521019327587492624938577798669663136504282628365426861661544309863663500017135040018816612060970934384318908519401170496322786617032129346772662620552197405831880237079466793337441972224223408218237954679021559076256303124106223708633757559891025187389298563082830464939487766453127374286089068898701646155174859746172943073294663324911202535403635648689827849245216742567895616331469728112661274654439941059494410782940566879493170398258271216927906158230984742626380674728793659420357634317187461255750998659996562004004863749747761224342425899348436215680587267171240539255358373039040721014417480296880036443229150063511886593628184353879284769831499712592524463062140428431853957809201591629125247356748672973267939961423896730169003900601320640747183936530790112498002572863335980507315439403994826260494892134005857904836658373469590298359526516012819594080408567613933618844535621584569103891277614963876951861301193066622380094245460274877989551595177436862266099035491220700457760401389519873457232591706205425719681762671670666241788473372299636745779288378747055955166023480810699584274814498097384739376607489557147113208819198304116817818355395813376186097481676229445488193193412808713584339865509774184184361012877909627874445080876602139558065720987708343009908309448227546305767376805381319511119825286114613461046442817295004272676229217470603640785836243823168964183015355490082840039788136626162515158407052041105394717072244596822550972974413765850615630242354369461044235842409395569698746143861594315913272113285422215607043456125156230215853170026883526228200641764777001834205738242343617673849962988948417588312059049081345331010311961442613143929941874020276110881052987137965246203989962053436192737815812187546813230523915561540429367284767973146263706554821484832832538185630410521394380083739737762914539411658265231915758515218392780780855726391670222605791025262690412976905998389282769074324448862499046053572730149003519839693367668214411072190189938763105542905418933752406431054308028121616850569542467811879478288549943151458349926617516939840429776436038131350352212946498245003858973667871321816734349180444300910783531956480393158747720449431174218787038626737701999425821957276857987872448537094110160512909029794020075314759109739326489042917523131329031621476431919159834680824936042721733237666118011961534993927600711300179946900652915847214396426393097401659750408655089118857601340745400005921242111594580850526965521612497576666222073775694198559376060042209253076682674569902399639335568424885487784444489219719151050194436587512911619520917137514538881849496013556244282927229095338446760561149061267037739965331070255486347575072134835250039606603707044810929676706456141074884036295238992742174443601955900026251728683384124525945519594767497413520215888912437478627004274672119013087987335000348963780348519817017221317137853362512190022044044929572453810162750306590197256499516828129245072420336785979798040452650061646903792195414626007391047140790043529253147495090315666332936226380439625547172367815124273899985209194015604263660696941281220464587763986831816330329073953888999823011869913962414470825159872717203461802057611813409043703852488444735991689007351340376491249867530169091774681556364644685155325329501911686815855958732686522359203343922080034762818271287113646735943434897257089848457326609157264195880403008365660665877199889095013807523628441554216307891389696652528997920102824993644194329896393733706807451760597946690722507422904556864414691363556017832621728694389139097652647263075465536532610077672855284376067887333956323804699179369403876409345599714498691010887562457503369241245814049926755689650358212031517422210762657604421635503763023641460988529285701558197734218421769579914209870756547126082885510921907702136369659503679323160103481948565629413422764252333255867745659296949951030378268514531324709486332843955640639199080961446284348490799126949602496352027460806324929988058144010069704423485048977263954967852707644647477180421710129192377647805170999201078313587248700614018903117818192875007878856880599082377873633567631463858615375471470895385294789254581687003919659686372016097335283271800592318535258328988801077096387558478084750034250444631883840041339017158095394281042572997871837328157056690317801341701590140178116084676778003893687814208726242342871906753790404153981290401899534090996246387196577091685762995491611529490135711348777526781294617542601312066414506063978452717787040341630116253906695544929647393936394569528518160545985715135884025743042749559403697019247642793850649211256165191514566939062856932626514506966806850632550415721694884720610268314115273254222448999046129784658539255955929369046001584365714581853488573011616858332452180934595875977963500915224004226148206642863826469646318481872276944091553009275880269121837914925391596570144127206650050258723059973862076974320685139055629482334163696066522214349002405679434179445354128190226912210320260025743373306033642051134599514756916860763868825046513141341984677940231172418423602864030555428904887996780418722790922096827818420540795522342867989236262202091183628488641683928229480890102011419650250227888063466734127247036737603063736461546907095829283571854562140609537148259424942190762053882092591776712257271052083767500312629152050151222577998998823849688531175709171826270354963583503524517777214825200865340294306762306408578160200465474015651603301456531789473330419242936173877330301812671916470724651781100802947381403463873170108723305636634991149650271363654714854870951477532526951742392206919680471374199304867153973210619191500624567633016754779285362851139541330356055463497043289125421977788598568738497027535683284429374328270575407446863773326584287279094142658588340462977391696770522090950893664563666469205941098519447320823606466663267637974650286636167283611321102132472485295142022804447378329387912754089290771246527137216929203856150750693092231715256959212836038230049192412266549260327559729741373050573993406641173948172009197580969460562524567268599074387574727581019549995237166564511046249667117249126964541121004758752513879069454546724664569980607200337928070252079703644082325906336445774201309715432812382675108700236251521726350653701080364251302334860439156388629038420328093015412481474241086685365735478888310957166714593528642600932764262834560966217453809356066169220493840286314929888904016784877054559675411135999421145780572338456292738355488052430831258571094872447484305330599801312194391457048227571199031790212348070190499119844035581434381766066419173431348453680176988533803310914610788122886511400788398003468803710555697450243358377088704372002302966138176068226185198541066330738487543131521392200068589411022250692545300427332941807353220997890727387737106287036045248346640483008423263377202197731574705647547893787429983539806739618329751443719031222686867536430794438232196716094565186463167420026155760614806424684858655762267931569179556522565762202336778940569884552667487750184122533194828584275271992779081581150186851734915662685621808181581766212088837155503500873716295315183781988407055281152058285779406516372361929325876955640528653199210531221857931729296058957052025811656140254658246764644433104010366471365279422203202197386512552839167584576822808944120492217366051435599343512016977342268345148728366564895217978391923362096422590615531688403231457080588087674297974074778745999313157743637128110331024238025969603914801388735988915563833320735633714802842716882006417467845692937700135029577430014166178520475853486247604745620952313221519691122831052907769995744131699448114593546216836029988817090248101244279330760608329052643829359282694068628444868901714221037176502417834362055977272751930427549872979349480609919391856425037007890335970399850591292472928561300638602705527742143286574437356084578934902247501505271989877432871084121420140117675002120355092989131908192300324737008914074225249754776361486967376917666201779331549436768856192830931943885532434586674606319101868658106146485574364649678038296171926507828229197960828070305486447729535224167876206914753765527995041456126196186111077747568211019433367842234499980362303246964811937359423195981677645855286313455503215161072522975814351595613351664304373288690329288400735694288236826591484643429841060740040227037122427051367874312752286507534306627865645806336176728983821741358124831586457226207716902480649912260014000872552760364716764721474275576559383463116103657903749429525088587308246131643776233882727874077157100194163782116958605787330524257151131562430925994214974172977646453559964583006365175331769667977788631887290585284766728705576422247407741909148755261903351136031546728500838300084927435713567241738510727442924280870309249579765864305247969655886487935015787588852908792579243536124280325072115838805307565967668405510335514388183207357615297916390061817620852091388273925331871918608330634333931462334994456596535516721438816445609371833714530629805013779103786034984655628536794203620958194694691872777312852334717303790705744946004501578512034013444733989222023833215780097124413355513032694383780786629668829145137935182320350977809569776996565129168338339973753992414849235789513075413599624385029084044879263413002968017163063799278479289653088063945926725183684012343527110300656466491996254403124333577650674090937543636782906868485263465396579160228416803419171763306105195847335720010239611967595134198765630167889822007793137722539919556220800689172944267761468746533277072544520337840447705387802852183085890213151267376861128499756247311613652855604861802248700966721526664915947215997635952046940177748924562199469807415617899184205837386322997284921137822611412068888888716763083929332903738277999696889169815119147816176807781697911475380831495662048750588546543145097294450816104356280488469455349422759529157580898576678507909204010973389683633948780166164711079720968451803808418243863457001378436970639142101752688819373460882804567897679417028021785456329767838197774419415084468196163086196918541518448096832990948184420072750496863694776505375456011530814702484338223821458848545800755003212213815502035660243168239226199448039979860980106122640226799000707856557761163310272901285462662333361123263549026923917906935277436204820866432359539469016282643848038264178885742633975083191917943352527546591023743481312572201295741433030737684816940459181252269656205692335527343302965733306236973386477058233242754903295027302942738261473924214043063645760173671606557336115134963392700000154109882130097941927423046546853156669814810493802719965339885931325458236285215204197322671032833423519172387160970475861035594436231019863858422698591466413533652688633945760484997871174414013694053386777733648550859957600614127595174113163889814100528145339736054689248610828529954194441292676173957708397715649171618284758718680900758724375554636753601381403076772494179255763040856299858875782547164999396820817908368648399753467299448540614222843778697258338406537156462619900248624526563099514161426874414878557293178078237644606467180993976246086213731026759708653876846103835807331259678363820530918267781919558680091477234954262076757839016207925001340082129477775397008871361662688719845405042447518690838237027133296395336660071808464014063984238399285918582970709313251133886907492539459039026193118604024277778942274778733765702095253083394094408979673227923035130812795500803593520425033905219322516564066032250166521991456848124429281214584680016111953422230594924801456112804674635273218136866483000149218799362010640038777172455200052357269945208833306918845122755432571487529917367170140503554056134372194928432809844087877614984958328410066003464409814063387972886339548142888182845278028152588582581567186335929997863945342950430850510935542151909213602647014364904922096301811318959850574989045887254903037426159645723973759873320186577537041828995138326743351288051581071547987683865829456912675479933745140768880347490236858257260055422931261057509752065538846301446847855316377552979917397485485368900334714668892927524158551243902528921458625061578210325769942587570829568240293497446883146414654304338867023189964720874406788936431298225561475938606126177047205334898205476608793116706534763463487785566162379258500124008597229518580476809529639279420261402571398291352770882881945522200019420482331602950389153544350794340923578701287104228093467144395320554537924668499047526916859605308738562810225706715825684866158324451656060288312224150732364580352307952875104764471656006342698061259454490111712893859656200199946992421674624548020388519067820612944858298825596006278656988760185601768046106724886818851838390773270251948836927250384004733584390167219727650183328641169244815591497995567811248917128975334483234296236731056806711484398988479550663862678182774909905275349109124971809617984863737764047270821324853037499040724437243030350344392303340882898099974232133203656296107171415110733817243956847910638113366212263276836185819067837031158033302692382353233101985204036214708440979771789598312065519018886275221202207547878597160749720074280652876539110073977481276293190365020267471577271689687007666550955002674971933760625923791321744178034625357123872214390755315093092628020875627874612306779092652668688854027689401701652272948769784388198867943648385976548202873901959380381121720750763061193523905669023700520944825403252789479333592701389905148983610507856496561156103418433808510479447932782262150544871001870568696079887233019696421554342369739166925574451434531334792678934440436907786198503944039873513641849361098460434069699891434291334920873784717856095208387410612665463257879578405306217260781346136142737144952572301215172241294864230783390603614568385146678763605627790417245139445504786840460937979041011649572277131214064656094227750687611814119831008576026785675962496624351179196212093880891236038919379845845094133429345748619612354173396693459497226214156652946494659263575288998922234963011339130321409457512288102518866117984807287558487931122948333095524171645421348647459161352290357190385748211735391405727961881336722569136344567680344853980303904801693607144093349407442745701030220862375102990906981553930642993063787088255902863895104095274667456616199054713698577182079429792237925953268356503325104484354259864643370434445958747680853010688971400540782753555904501358489157666628779812917401895210737229806560630691495054833683973821634409229697291963076555423451049678277943061118937678194575003997099947303002671098530526968792974164247861655891669493616066596651691575521423199465973040022285609253002851199867209444148511331539963281161685425438431405940133920372040429395791749411352728272192357650525236490380669748530709348664070921300139643026323473050093487026886128533453860451327094742229855894193714239765855529490558316492781567099845597943346534668949311904908189038184883339635818295506773892861844148101816504664369577461047342205095439419751180689386744524929888765408096941866113647571884613826472285589953585846963738522021547327712964341579590789879104617477966678302295365081043990092389655164392909690131698946225642779760573757168064489251827974225156860648420092459949602856256876987358410901647660612949019384628399218968313649934187039835898169915652086151325285591329426862653310503573563147756782090578898521350860494670209171661232359617863599414354460947699885844869077865756624729467951860502430744052687952887592804475379213187540879757085247600852472227420083190426444998076715055849467966434518164628251640503264001544174194511516028908608983323364450301964815853924810394098825967767945637348126318118897294236408672761579446706420830211976618329041538537628856183151548430669218312825542737975650155098730431487579389692091682438728342453708354694705435303277487696505871590913031210474896155902454179394887448190404463820147608336837824732243368194076599285518569828531873051991125962729799518890868813311965386956838735738681770114800075962105030375372156954724035937401293245526109118641989751618459248671876931533226634056510272179126701741799663761806329733693223437088208944733444454912283187709900705644739126365343925894056846107314165639298907174423714398598730761304371240952789296113207758815357632454254390880854955965974155324697232769716430466346641016062703078005507364911893025969932549860158690455349412069543475007411089983602953654570902309716101384195662101451669423745394571455280274535444963913409972326276298743042895870478028177505967221858231127469715430350772221920602888927334726302909202920100569813526249071008552341711618867938177440964830725357137308579695527214246443017971024390181144888794174356714751902948618545768380868578214742984828282709634767093741835902540037861841090106697941205881581439010750887740250759892043574329070614222825727966912513331724545705854204549201686370272153632605372248703727016400977262841663737738427144943037221197487029517097781711236664210825865230077815934035215372944472115348267112932730733332455925206676656067684806613553974943468010000467089813796541341191577698671214223074089199348764524714004891570892812565890828147925697229169342424816478039392191942751110489432188190973526062855534703643234625452345923102347138573751228830934558663397112659958906870564559730414933237177767202059385736129904653754316241217576981030447906466828824289295155953393046309842235564621083105976242573843206020991830735253642127835457201271855627345876195906764958436098721753052480578249422616667141201151131820349572438695128335686027265846543367998112359835176760292030817984850660079918485260770858140539049985717197352615852794627555261253689175296706058301150288029756700630290498217455037492288266137591625852114973620945153894509027707485756593054612318302493868288461338474074178002042468961234227076044023191646405315774374140476659184363931621119597249499599013336869625180518972921918868351397914153452835144006765647938602035837661612414867737028429856125664507801535750929559103395029153818412317147646056176728739697821042400506073177115682094490763821127238435693922982443175526257571541250928885629249207134455551576766320401101645287823213580334217448704404352562449402761476177561582159590026675448711353221768205971520927337500049847769001851509481883233815224854736066178714211915535343913455534403129368495524534780471972901145964446987251236956648162611755931371542599773823431510116979768158484076021808649699519683899905845129749666999862174354593978458503144258547488050622304996146562318876198546991142443629990970944702738974923949340944771799251222483157034133422693857262491666500827925338820962161159145468185129451176654472416047607117783904642475818433074422203979541380824429542019836692739341529581551132068715807588109386324444042809858562739552232228512189038026824665582494192890255006899049429165299060947667033230789032322006663353531704374992353724477379938347131486176642817328480978943416379702987125912360974544339329245682619866867459857735209485721366790162603677583152848336410959144176324230411767948248252458474791063563879176591770725943260940782624368512776070125500419886914525013858621549712076361554296243371205280034889273307338626454482918920998466413023848865149808198746507719820160115207582325266532612680292232066505198390681380470386003235167261131420496754185892914136491691437145507972879131333044622874382029018615288865490981775764081304471505006568519885472198343471639819479192033888177447453040794705199528417364298505157407343226509907269863030872812492232216898051252819131449472259090744725367582186927995468828891448280016380942578911596700172841290397905277925572087502466683350512416827117705409004223910975542441742580325244300191006438486418030962011756518265661821560822206781782803167327137930069373605888684779598050165902207928340712784191842488422675801418780158687997666146052831628190300348416041828771610132633401656619355221345440978826514544681534169832784549482526831736213506001515399957678614907246575990913260689859932506162948592964515336388277621334723471873071213317255080622149200783515512900245959839151155368506661747240697468293797161680831632645056480684148713341687611341180724101672212659531564817303521633388259311111893158241795905118190294954712190720336645062804880127502820106942719475622309823977107156108632954529065509016155136414914344856769606771063124779661005793104949745645202720044382438558113762640520126340847521161305871277091639474887085012764281908675695660137643469829033772745668011513069471102356893655013697521974743903806330849242414219568318222472622183229123145705910833774553314175748529070939444421153755594412877812028097616218268924003940113649785372455805735997781496174946781052346287111453158300643526698356277672812711632622926922794142870222401397357424261261573536054516298314997121058153638109663082792461055050455367453114185481853051089538307471093577283197996828539638191433329757806971722609087447724534337322327188905552868002372640551995606463529670070039540027513085053598524821455484955548988115505921263551558895479897954018753185598311987134982121401614537941248961989858881221610949085737543450684762043871532485657190134695381781604495115379963963841025337522776873113672626628734271966273703024409198929377646825375009206051223599933155899938268800279451903074296779717673542424350560304291361517290846263563938054994376218956818978433149235834198781060242905531434936979069630350523915211515768898391692056302460779268647678495238682007731162730034816496421722854338951294946188842280764274063731471716255244285547845546769191664199229542694999531849982068853277428535017199738784154470464199053836741565746642416171248595127261252423515958805392836081398663701867399987900799337371255458234617796875167160754885115406983908846654201849683073487723269614765106592913422968449897454168078003733182903924969537612819910446445643581386366819013790805614185411876251721317857283683306710669286337763240920068725375405439766504527372505267379510268596001745262408829844033924785467067696626765578071112536377394646421849761244097616480700375018956467324539469592826337406459559483362389708636749512051252775682287931460860169980575458918484375337587134487588760437153907561374210867916751060840127538413817001591461239718362224312862667204917507514864234831946451129863435015490564740192712581872775149724958838220000144449589379744597914789715569460894170692306764349268021575691583305438919830354468595574938805029411907411572809718247902257058335982250210339892034367152134879860139969340420489221647213610166112866215872918401742425029770165283980082404313630996316741965179674556141237317026401569285620435471708522397544401242066048300159115893411200268514773520601004080385427164413199361666493730838459516578230626594795918795537726477231658653948031917768232083633484439039236024566978147235463659836026172471033407967541145447743200549320911070712141510871373234729456364038250084087684719335121135277587817747331632580333749570445247965886772599775393497998588810040440340661060220707753279354059693773986543833936126195889337959526931515863189980840354893757049217682527883671842998188041453328825466577144931641872502850099179792526906019242250911858939377017034232035602942968819696439580672333087787365315256636127063328390814382461063635018355169488366431289895039165389443990291268701550100680694011277869147651010558420755727293222257137117665686816643555305427216109429622212468255373035357760759595795036041800662249839744699939444805453302405330677921387211693558273814205982244243354134703732885030586993567362212225845233766561666952231843256980476661592119454115223346227362910755865569282472242233412492711215829624691932463495792973431059307503470153786969385368869401571574387481792477906472312166374245921054333737003797382860600861159134105344942786054290938712823574963335788442882355099432463537020533010226527973986283920920036437475721160136147690848771648951628800646533969116915024946441713404791487619154620269381668670541070306308364906875298736837365653134093894747401317821503718590011438758432263961249985721862793688359750392255653414819595408128647828945246850908437771691376497606575894014700971980254950026298828068474585805879384572931300021785805574790548700619428637310898907650643541299791090011861122155773399951785391435522998813024170886999415464796673499978308936767658415190297296123415165423266631930634360720412522378493463323967952253314890091337363775619830600736952120035485339455201277488527973021924370352708609213088040355997027930836044732112298850980028547902862002033559713947495283257780307106605472796466413221161421696803704682896925731619805590647802281049474609540822147692575436790539809593898280883932228680367713345705459400524678598898414090555555789118960406676183146594157065479553659867649095627211045830961722093052656576445180470095356102748729149376191714028860746641896687547944545030683414009158418670881774231197926086409132480224379838052885703935121783178714301725647228485812822804693086426971003434440887440619949226790095342354198917223537531971017732155027320130502226106892400215174424211641758573367272555325155877999229342446082037066517272399593992980168503868213762881117874871731262563853735745919893617802048088483864900158930613670538793476697199616414302284137535432027781058104025128950903336519207086112234415413788917578014295545950876186679145781532742384701130004664626402681183420938305162486776246539633512439318694979142745927647377648618211667579656531026611633429498029390404458527129974951643834812228609765407805339166553976067186961280502056165766322827381059638840539853221678396647840683766543408100904625569280823331937268065535481369556874196394807652062586872140141785580260883698805512643737985884998768790055560950005423726707291395191711257957225805953229810282767669435022125252003407591073384761421825469560962759335158359429223642075027412510015255752539168222607935042243704191369064109351568732829530601789353016732303460416243627642303153324129227675910856682750262751361922071432468218229545783258892816091787128121736411281915428515320403197995849111234052317058646504545262974217191074126937082837174822992863547538073468625010608813765297658513967595111487609197822676413138779043476506142928811020447"), new Apfloat("-0.18155458676026499884711022589533728083863967778125233166674670611822891794927332367742707370085156781942564892884080232330139200060369450371393737389785030725616945683609983428704844206842579635599971963298795555228967919840702400864277779438532290092379506662785216021752108940672296274766058169808648784080707751056065613260648102605363242560469448583802459346250763677979212969702486768421381173158609558228997402453309032705199084138326032011741937875530993140777426413928316817215615317863510201443167241303143078675323989485942725785739351743273902651986232180858797871567116337533841033026726482516998292523608135453046954876148639902672248331103454782236925747466760167866319199274277718268377802174175216547239169577776514048562534143267730381545247387529453060108177110120464701437454494857269713185607475812458722401470790071905845314095251144488391971061209941396123917606367389914296275457708307900998542389697910902239091171812558468132118105359367422431515668051014563623578690457169405641916281231020691086154993074410406232285444542355703082627660254975941284883063310321084189857860967898856822214823391382386015402794924565927707735018747771665683268381084142624287436529048481571763298414640247585262877720565579925986929906320045879101860829867330760049275737253158575644644912262631406672199556650532784975626978691804168692896038336075337708628489305468660880161219227562291747147324099087805586665792774524412464710020134134517433563184115041082193309013761954708365230234019912422115275109706350563548233767977347441154699370163188922758277828933314060515588958471925100431930341835549923350217527514591663086960359904715782905593157512985198577558979642572512997260356928133370088965125113362099128259271445391578050067202377371036485703455608493415369152063370709634288736748645323540521724412098198346721356366602357213013521722546500762708156649762034612111970531186764025598255876940367905519391704986734844059191768078451284631684795662721255340875039772854087664929773722189495545368370091145023245593761738390486817827892415962000177834078546519274999905909788115196937461562536702171498558460089152139601818801456459353471097208027148272786161684471268627083657954390155678264558198397080127894250071990735993757551740290512713182793548094593044591689025797552054788949531874382654869836470539750275268558057335728651067715350416705706164491817039446961666834966109736094206081566078046757597916252621137867055466333471227675131896112391964779174525030066314244488738030756081077438051543608840182142188986477765756607817128673293982366452578959681625021152279735482672236670995068233096114729411342010273690828545362956284934820951696006550805395064264292927808285941658041749603289729002745128111413433271185513985010745502975402847606029115884112797924668677079314371116477482747609995394085636479767602808511367113126278575592628573170687670077881650664797066088421672533135432283867094513242371264191289380312773911938335901895833847418388120500990621570831100933375971413024871208706329121425188319099149544355830385086912936466568514004252133465925753016611547912056066532727047871755231649381648323753911945424382878401397829974074396492864459896980089039768193788379491320166591005051417714019790024479996150318581034981759970664945567427894977627252659386155748078834580658773216783912117618935715002879513143512950049930891387455109166624674912728387546802714786252289643295607083608028811854982786670659166378509778987685192939712099815843145980254560954681334318945391810686937051450782466875789080645337725291565715048513639805889174618471218209227537700713135558433615529986019798901291024999559582869653618083553772229075551921257568825016276888388349501490076166339056649628876502223738251794783878591754924100042057610585518283064588698013275025321799737813360410159975569267997433690934955307624113257222732891475822770318883057349875759084128331382773062916897477042204640641861223857987053125668031270152972041707801469088467363008737380179371039246934985318621168945747970721281603553785460616862710613902499867609345661855257190088540762638367902248768636317404661583332191935337766453587064085425719621888095347302150418160501438317975626698230419536974561031858030042528764814490048414335453279755241894816864478801885007080474934378840975699615418181667752237863892892396759200358318354399501075698859049881367433531284705391160098094203155518078074993592906175870010558759117424355721064634912525368648462866258063791582539743443406806142149570624855245036157173500403645444832787632707289607816051748652069882974285948501745892624408408896102655097532923496987723511193686213849017473849707400291151796846598658714909086033419858366264791638340543867669035931313078605966897591416228984655208664310269072192111121638469799304946397735517377569675934711546133533311915437252261127552256009887912788914291896563256769204356259265079828546688860973326899171025140879693289818904345982473229390596731102274767830575627175710189096339678364627003175766051883407264346203399554770010644989885059368552945752715292482587578467042897556232240818136589321921158559757831421384147559338203588562422234719261540175409817413495418492828516710160016825740494963627893331625528507980254883723996537292270505335283140475620721970155282311008668263477059747537068943377948150948557051484361247409115993509041897390194594589031324501490618104318169134543076509224248286423091711131567911898736231645275737127770750252777073449371001149858719971783711957328751727793904411723971131682415876039285774056195916696490196078346952903518150271800148675956131544475311553385386651208912595495560065322458574718006070233161014717423144798830695985019158240033677106587824019625190516450879990835686805418300641391923585732915997046153318438933038316439717010109304702598799194030100448934504759527027281356134800078397194328969993223159547165566234999307697448252811144803426789949164411533888179184033919475521274182755614254169628486699425967145566810095347617078551049857626778194466185163918615478987776148925635176080809481447211713384789231154494906197701831460713479998086677797005378099409033230853124656220011185134723323268040561316113638812902603416356832691865462964090053946946217131042227749685803262590036813259964627999264921766808030904633986353860818302865605563064410787564539102694439287566705605297191301354981371397315876303279876134477720194085387852246015946098344805729532926191229835098591880284099123712759520514714752847527027944776898862146565080687716652099443610073700839803692906615506050990062637105852457873282118634136896785021581817488077463485267323263863309945140157100052571674321626086792711334756928891587548822883612251987825915963452199195625284796904600733461339112231477316627596709072180208051183068348515623426241601707926078894861317166680096525903530663124941893893657903062182922338900072802496255249889887662926747415639995599101044467950099258683232994266718411381900637811009489633276399544965381107113836362023241916230183772692348944834075735180751329619936690818653985846257948501587881316217287890137622508181011581202733141498697416395775834663051647925968788071146731692363381897686266341787148664401870785288817055859648666354079878446338997819466140946659176132465006339159428394946850518637294818795318395631900425754632333442991061194509886494678907041084155623777410891877344117533804042747842238650701159434511383235511057323933032758685349748198963035627442066630152709271817882988154234881114143734363261945688605669770060569098486746666462966427781787413458695874187150523547237954003176688615363229730851551052197245972578548310059136427287614626859345724159848742241226329853464296837063298085191908976982171168962142346817246190083172736683079290775596090496950456163858171916390258903299595623858381752327965935302686587913621691039135955364922649168346589626453029406343205261722029655313452506040717139828959360303475410689702710138794856832300117080618641905425573580616202379277461354147482416119500603450257817330844936179190259020053401372358686849833366727657458217498481735288814290696006946329637694450838326556967572119156984856951937426619474012880886861388881826411724861705355348277098796760374926954563537488549385279303511728382414833896196562884773096762376361832564428047054558010831495618752976581359608526387300465910565912700412364373310601716616567133803380287152240742112054547857277068877443945003275972534759348012543625708768777586996885585399080887435014234656273576598513859358886503095438794049172529850975457125468354236802706846047729879683640262963613420144094954015547578215129512363173429056756037060572492918947840736349619514999784395726868016359610056965350961699354944468941057182562830359004659864976444513397287124457604539671138573674546707381329315837409873247419178577091584451985195208580282134652056284760569322933564466601034914519351445800997333090294440863711009571949614219065285752656843346630001112597611055049183561263092187434590086281831008761241700568175951293204599957027614243188465911652128880042538361346279169319200055898241657559554965535008667956460949670516525965507189134790489925599747869621090146212047684653375178065177920023126479213624054568935302741575675044240401853711446761912162642063009066067020401993727231695666763029890444064173455932429768696352388558789103889505362872464719836489916350917031553688784217623358180515963287497069395443975025604142437522664153767689904583531308584575047172315933438897677417388105281538825252091005177338729067306108021376618576887147064326046032740872702948215326879728040856871816424254588297272294886069690036565677002682916636244026391011233172558379884802619149123121017996850695338682943388417634926436847942508273652772050142431179259224584194593426188502539555507557770157087160681852944832114982807546102762113441322293252961360976809959023618725526483933831411409817784943586547725587971871958780769302445028451090664808907035479353779725152112381099071610718514884211928760123462238246090569533322249275297777459649100957970910305389489441261168087199633896649644174798084804604824163643726222500658856879202287380175985917323503557984373409608603743304121161225279470612072244747379895373781137539993588659461627018782171104667779984412733895493875733264595082559894072119201458092603889166863420230118097612283833996328535397880405015887494060434946433723417075960448624107006978691335782344831958497397038490293282822249795972430525345643079031362792517878782255802047915267939721014861404346613390876376563623519130906385937473747532731456623832584075002996955157942469230469820281185337423483398655502582212463763026889788335704314877665479249967041315305124510080089044955997654026724923206183832923986253889492551933748265594079998401138349889547486628781801829614149959412298435766463302484917993974788010815704632989655493878125458848157130697134831430867699481378124505800858700056228028831223503172547369893295769843297876602797251386658308742309615717884420727782818998233669788688793221508433753582612815973279270659861657821774645165444938883646058760995269378317866128670744176403756075733815078864349194808160894620668624767757604384378543299407206187374122868742678831261404016596428355180214943372229689809617739870973719294614393559065553718093479838657553668590873070811951339128044022667514906035596108127239686051223567954128052541976494808933711411955636978402372167729506733480309357608355149710147927064579255576657128358491742382787403689247898232482644050710970049443544390971620728343226654958078948785859996870760207722120883960745129021927735959897201545553364144438875492596973576982847635213825479011867976444235730597855387130549337644562839766806430104362958867480211501799908081258720232792800379303175848961545225350799762761287057042471303243870015424176521935404657692088863084984027306316955094019770238130086002188897091666124672634411509699205831877078699411488763669922878205725217813826991571002738113563138528066474593704660698793915647322811520292744766368126830963835936340104469904972826944945210604528080937641227073457504280956323716830912929555152570860094460558882123862908473315023661566152142242513162119724303359548935495136168457617185126821464216791518533516158775591557353609680617800346393338401069008430064269370856597249354461020017056044468800137872849897215450069039438995335371006307847672163237343460320114327290579160004366716379549649625274855341005808338706049875167699109910086701331532086116280706791102448900158816602625055298760189595008485412936641846456789104719741420043901180691003526953358252187167279842435209259315072950523111819105706456620210902199341561021847635769474070897414589524020159795815726803692626790766957900329240700925954538749565981282203086035525683522687961928744905519558585951305724421828387856969642694742220815345211626207455895060059357279031564938991581374117193726928243708306833636461608555672568125912428819034196095726999747668385194870046486568033651240825933209779573075079791838400504532288643940238598213069022573401430464686733391342914698845840175384512160244373585836525941827704918487502520148383365022476959791877341131006489779792453555878825740141975913568352706252097996439149046011670487782787446583765733362527918261257889265214002081414303192373631277946133815973223803876537820499659371794796400631786864285523978689891164916928834049958733235263911729465233643630859614084256073119033117462015990548396842459865669898663235518119568094126396620747254430943329672895765262400269391388385622316436702117384819489102155864329869030911865226895104104840118835177454233605618923090640057991079539463344112825147191342862648378016708562141269315890997111511164789993706350105660431220084178143459807738339271401676618786486063764367949694599430680225940365661401661223080391321020580820659026352411498900521113186763046494715748233155788308682957394146422957953736828345637275204873196960429351612618945490413984115141461741756322131375880742108589856329989564002076789434236884035703382325186887312842705653087391480630116393758492187978921194531605095679891691049539819691244480711246373284637938587345612104703760450278233280657700276705299020670238566781054700805762798776183389636471369007575820806538309908903904574043745018735531764821400401721014429909646842012204930637570523681420659242863619484845792857278411056212556419253365938947208048624839392132830807277706209139378378508325878915401027916905122910588941204870429825103829869647032976279065387292216377286458020933791643487905850212486435721629129799687964046021224536623669737235800327172020455363480013254151075683325893671943068489708498256375219737956656973857896540464978305803948264862787056967985607372424203722585447846701106481497563158701150146198671598338914262464330364947912555309478331959444711824177968714195696936612836626612642812905894483405746213186395921237266899614910516778085163442944936548284210982732471798163067357995882443646994479942806404825982070273814469359515621741652852627685635624224060144803995111928878287425703727400075691597962204557598234560160994118524201683814849195595764177868983960195783353270546792069432773576114458767665699811575604575900827420675684318047760308183761267793142818762068743588785177361250825483826366201778382811606559586199555803734060161170689746615035992386253232492657545003243113054116484819316689802098355713828066645716077225175559416038470928795901783834464962857700470075147817357369298043416198429958230871182986898303787486671695284738374515441449507917215264881309804812128887590110705096317781041895678320638798616089268399776038281310468427151851582810128194991285227552257188313315756525790396780237833530803247718763523603152418154246117932031747118335836074535056023928917846792648367982026977313483602416070472410708813906913781775239664946155393260104942496742491377647461649412144630038505558096597797447922494689069098679588943507497417381194005370031102515651954782067224072252458430642336549608320464894400263549893486377611918417715939877124573037647724918743200769736477200975691874119993283275353636969069383634826490397975000450163770657981065762431909845417661237893430848403923479364745999006408802368010814216047165545526964475213958955596356317046552018169059357013967284270302509833123900590320782156589423597727069718856944896903192578916548667294790211890253543980309384509118402670761763277391570067088073370958329654157023581017767831867602657376864220031414746444702900657266720428238021747266958091826914982556788794931396138617766725716464196544064431535306407792664153468400138796413062031605791348514724218282637907284622603565324842870461947570707847296831027365053887001103002536786127820999299008750911171155298260517231200461496972615386426763624479175956571665193648570853980511208729600408103393490707405651232104493888109232290671903596850616737831124356970616513512540858064025190169555819921737815612596383473094793416348558314805171037643605144272486758690016415242159302869023859120323361185947158065874141056212618317579439190777661141064751616181032917451909209355155024043353931669113708171826327139412115330141995520001762765107478566512356908277450678501626126976177671793068762985139114273664912369223714493080366661281257492549074924233410463208842022627816941274122440359935226666216536819713530388040005248625520289966186890780300564301255361137686665057038393418975841219152102211773897120009299759336592630489288499162232037226422269579186371957321603842923443589825671176775126002466213231478362074264954918263794537917414215483251089563255336782080568452721476142519773854587412168072710656758918731621190120141519485204402944180818659424100746035608124331936013483129662228927979746799975288029127499862692207597641874422071184275714392560500746637328389987286175001338526280752461259839924408766733134066866912822650525265958114091975355492581895358541411541105539569364112440025395815183193611462685313137452865802548029739094712461503689797522596643199578782094866665140095382288407048915489326207453197366458280426400173549599188293224333268294141318462793607517619687280398311162182112770063715513024863260288108169301447258418168086959979941897051454615481534159289698585929438682811217169092271420539280099424726029554955950376876088702948199019974760449327044943867140279533134841650217578481953157108468564301344708603729182597662431705260856326889527443907579449277814897150690764814182903712012464853241916442876597653595067108839403029548154180003700960830063483168085162891320130800027074931719000033163579236923852603343020847366091955877417673898101512296483452081560940427300800356826103138444354127890848031950673196841361039175347996162306289673942680202452639572868263128359082867882074327773138145759486474625477558339205527044610338043881577146310874251131265749954959064898832728493119614190313755932626020226909208379527406957688775360072756947990351856874984874329261812757006123041272172091487532709377021255702507000698368230586973099548577633151808584739493195078976625529369619957077975682211898289888048973792749665580130757554439613564613086041186677870293761288560453496110834346740806246365301119645688262480520739838983388792288736610039516999877497707798988898008730406096344341642459123825441704461404987114272842638247759929383718876950808375658932225577512075846598544706199848201668631074643924110614510571977918999509747699802101923023630091009106117480044202059633567219072766920922719013753097892891700592945780937603839126110243207456896342664740757588123567795806191288214539260098814175309133494987664540371561316737880920363961859317317658705695200653467960168168994957752146805748919072400445096163115276859878541088508672155707882379383103128780686340097562323256007860352373089234181081128683854865163330138520971829716484955458695087015710271954295469132448799480394313001693513524793695907405282958793929381382507727575392109697231714135243032706872944249346631157654129159080787316510161594932750330725597701124007521336171300270130734679624691887987833214170387804564373228832698665134835632042220343670993485025051305700047527287745576815342662926287259357747957798785713633697038398018558219234361871610533390302734509853406687230909269493913411864461729974349128837291914195512628290408426767328047757168367431537670250458138678496189987365336426588847766021774849001859674544716563558121649632400071785336689904472519722341716632794477141468915454997507321444012878603514569914911033746163595816571854569064981847647740595590409559824532479347633698217776149775759431715086188599974809327859977997203408944017196483208107604829065707284780681026078612633431141362830684756340847299990994239871832453682094560935254962373661444702351804688756524157157787789267875947094114637834977760663568334592105198268161525611653244650524478390142922841929699821790940501189330211830247968385115059249635885225384536809344226882259490204752603511676084375233460089901657008545132217735484427342242522253058199878950046936087429159188126933375881961538229816976955734704701490676523057685628326501787104024830050405212718470168645225051428921443646641678722678891618090598392362014218335264157702745616869405948585173244661650795356492083044389924417156341419004109316940960350835148655336490190676205657750158263461363985761009252291840326724346476417639018112464787934388553013870489292958908540997305782828802090006493918257481125807197004052677409524114746133619668632562310284932849203514077447563794193202978304605884234666074790738696632976897236096401665028471323030772905462821652913293279127351906215220738625344635623386545216186078061390842595279290073032468294426506303639807804883037039248152252938369913125538435348999289650775799371591309763315148569445501114490555499483735608252174752244819892659051376599173734131753090855945683256772141245140203183453316152493319679450526979969429290046420495544000367701165077533535588803364404888948733338251157504818583688746141991009555703344972100439335169781620274785876544186412222511799263268879571937780624703289660153247089039883236210012713908040782408535266245409607509847938212977408324926062248483596177341571253888028136778613957359446117698711814359168210004975291246149817256676347376531393905345322766962108663626191126491764739053163398424635671902869051004920359888065966618335093645088151223109833202336367874299770913131368085717479812796070253912440024454455327438247634352878363970920897675576217518340986657511117761734003419171950397181194264888769385456728052512664563451790399411274807365847383451981673141897516930952321780617226713150996105241421998776847877998050683911147314005167653939678455883733711276975770868208870775480210311512151592567914611632725118534674817552314277732288387047933573158260771020809396977438151531506394853233872650008152472865850590675916851197903859118730577580692731576028996302240344548517924968649810341087688795082399268813733296364735904786017787022766352929014701163100671205330517337686516725790305283090846425345135588288614722841776675259319573183568879544391873534657032143625141750202296724046395423905518871953741574423981861073051765538074599133474008761223801167534503356832718540066987701318101617861289986974778617454370058694224389667186828711842064986395074077316262157084529570221215845494396306732965192999882503902541808352307390647704611230784584059992411775538490678892869115447253800881220895136168239123645210774752313874005861534097415243270461764816781866151422466806678510160063202336712962361813873385990419986653029939618445479404593084073812518047129517221169352693980274522871349477753689596466814479916426382061010633042926634922811805158721711539136598894752602515055843589573065573263199183532612604579189049901359715243292514668092298048655433048460848760301273267247644838849192369418121367602244566250089574614633835286285400412950871121664932615252673118689802144234351653951032806407737841949426504201418577146185950649957002136958440470833266335475977089183322655892866415236199809028399244619945598296876944164960499540222239590093566135454844672609207196919830212375905486220549622445741715134601071529752649529729107064012381450659235243147525564436980899597704141913656526392818549527028866466900021366105809285073946543929669937129111899854566774660572900064367776787166779436157668622268954786780762324816943540215448952106696896109673372648198789767914153607170632305895314315762279550714908784393560017181569726287139108576297839052784796807472415865576217405282896459463723516328413518503933347783181560716468752371683387111760530277192034345671363191509563225353649242550135384117679050162279716632394132256518801797482177565739906334679102709673061445560302900140225491345362950569572725450952755773232847321666380324505820926860742948622139080275642251701525821075677690881491453808370217101199649702570137014049475286122931835248125243258049188661089733649076335590014732506921390015615919654059894872258889404058197556951209493150589130427695419612995793852570232147798110665069680040402637451795564292559084705969568891958384881973877705852334610775914872319535148286567069679729371701073345832122841914865313293734767760940233411308128076474405584083249573216049202833715699018618470790102720486101991677404272294567598673121991421432871118011553401916372016707495758478473142476598076794891558500037284094404197696734954029858945199676035821633658824060257413599284008394422259675043624956015531360132315783422394510036925331777921385555135774539284875634517503101850950587767121655074413848387525764859848359127417316282553659298863486224698512055544889749225519155850814941562482876747212752856637879729981757178672704816506487659946015581121611674571256734975952320274936136084989801511215262738531610036295395190084356572092678942594860424868723290669121663140269993836726327445833768139668075740104211891630905514222121743623461026916443816345935221797281408038053156296294073119221186683870149493342715052056065031112582283202117248480262939271299338497509966011922323682486896939816142237759307705663353664829470762022366500298247495813025455760474615749651330055773959100619996785258194299936049852795942730390456630048291587690265548556474217102030291618965693169919635097803801805064624098919241702719660607368943499595302466694714420593197482425312048155044828588254631131656008283846675003317293894151791157128346257726859677432670031076266680615085068153764530386260907335826135773173560911218811247217800474015190732715673705129765125173081216096385413000899353798679006929303340700319840383807933552693844529230655671631198501934546508102886109577602410279124675851825799718832320811940482382534333237906368503598164721119634207765910916124111027375240208918122956285295733497108999129657032035713908098761078437864132182334186073032716141166052636331916617781195359133621292129333736938023730591055821318853037264293202662298946960695664970738039556603610263431850137421000232838669153790531799519459287947646363818022895742420233207076989217907952003890742873096051709868037943623816553811379648106171977596544854361878297232689398307592971117067924673079632948017331830568878192681482230878327530290831415226644307138060590021314741664815898512170105480728697243578288951509844503613217033440039876211466627687419272240956448064934129725347016725277877835048801406042569544561165871868873351700181507273705664102523439615193026714126114570498158948167739296465860747384840667845470601893855178464969768903389583020206997330399555383773512842079965541114828079879771814450251918270225226457288675574262720179753001577579792528903907648107042052261731584674041812415745444374196766935835870111190389545846309966954490642756147563849169214145996560301440144022907175717651122952441429335648235303511661765135806315689142677414430286378745928238306695296922396845938541802174615337430147931438396489284227535070082891437315636265211816570903267871623932687576378928244113739696897568374365020953543899922143618307711479314486440697650993671637282905688732988737088416907631546358167383326617460473901453018745026369158102217552329956997188170842538877151905045614671666131095368389256545445703252313937450587338667065567105894359554354828993702963491139433436028271807501599137243505433582036960492378989762229940381092546102621444005548755559192373993141276724703153250387005598866472801037224654898494661177007144119193553859969853796516232332158068582818501304831508576952438008221059924350316211197317762274146125430625853370016594390030839392855016465670313232551928307252182834676187653318615757008513181089264633610393087406270770597614440650912411894337789972506217272292951261897728560276147982752523277978084243142492134937475990892600050639481788739224542219707265659720803877350620615086676765892480713044013935541593719135355634100501680967375749350942531463211191992187495000147512674741690378048221305902149646954961566680362963921305248587253381372649385991991349057798695205856235933526733122907409964640149183450487533248195435230134795989056146995846400916705704346010505008559114697620598831241004620385211829774270963064623340191214683815120574484681715507467070803494696140985366886113865031992351102998542921431144382711390134477439871120371088128713110498603894944875400489404281346034770562759538523928232545132208431963448958979464866970472144233982770812724995211604676988789528854738495952809352568071455791215193753199371933408939549976038411644461012525347396114040676998465890076760398412848106499670107021896647620777840776064924486406528654355768571981764856408753606606052357532348683523801352347094006403985062088264986692280577724676824084743670103397526346722162454147276337633296866357241027374215881800169232789580408899616582849496852232274455265276156328764472985910330731119159207733217214242894633234407408692066598688532069976859217954860655267249496499038732329850131332344486554922117058122038374989333583745299578039537557249658457766209180792892620718184277480509887314231972659848074854963461321966398544967533285833379703641228717610570670660993649247056602796083370451587152852012607178163624400444304155877596238051951734246758072304777775531954880281793851444047371857753409205741788663117309940577466609137830305739109997805413558988804220683797066984323599860735351047256970029208143518538505433638756102987876053481372811630371958738743344601839758460989155521961701509329595306259091713828149960401084586782265392506575068984706516344904762508493193375789796397579022169957364005763463841916003076926203273576907168821527161815768573631001653640584294444506867316732227421952512120909453805977860401096881191844698409767159485145966166037878515557367518154863759017410834359592361268266279779887529701764291186294290864844555660422742953103234776789579063872259738589461149335403497544924411630618414433746426698247292614774392931381125135991868194051849306864652768434594341169488627998232987137157884393473736681267939842151481674898934792832721719615210228456434473077312582373520776710866232826872414036539148972510081221044740675801067699774355467339153725435350723826345778454888377567253799788402663012239878417981490178044544690803937084491960036852590347109028795459394311642678427072269096547012901853632496553908107052757848442247916360114831787846856294420817326072916211974946556906973076490433582983863217759754243101499710259100110011977701427878871567238397162410732401771741075654888844205124801359994485030390614202211514038671372754342667225967797451366147210464464681290267742432257825119798413217758708656755255967342629258921029867497386366750853463895625203156012234738442491775659288119393031421013826934133972196464662883797089308595478714038577989459438836107759306141993446584406269336563696776758442156677207821791786630670291512839299635336567542754605728007870447872954677325894023324926004759802335925566477152460999206816166995231402327397303174831668685036521916647943498811720696225705182117703975038532085536064114958668384466515655943527990557947212013013858212364639795749837695988190390022289594273640901334982930287814768542224035339503280384621662115718167247118324884799698502372593918521356701002384163478856437702886166123249727870148965979736790329313945157046842877816967847312701436845885568017308736100826440278264086328562729539431906422608201462749847654067016751068081994871734940895853129016578503235776039262195565096241065256051746106768207651715953530802808551169224031968887503460174900534491292233617241618071643026189538063508995424670313158836135246379728632982112533797818021083549073693647851781100603560623454994486487304820683257673428711501117935246621080910956529218914586087996215282059759688886400452875528932346026749511685038049465115942533350589437425172742281190867492224356151247074184040001986671112867449092439851115980229407329759759804233561018301204667846851378704173136704619983501467518238094178653430489103699430558384787394920182496323491606337403686064498642025863477285897316830793089612547754783713591085864997163228007381624845755235607822766185381180891800387577563829004259387846738365776446766186468488475887900474365238689197530024259721932912876448516040071943657895715153945901115390208370895844001954870145423102408145425692542059807676819770988851057644487400035073613283095714565851367129137896189997880918091205442321614022008839601860477010838096993578348203385206115498427005297699475550644293371364822477309525614999500005986515516436565045898911496363365283122814463359623506837845799313923370897586609700386309682009254180612156234659663908349485627750364564269760039827781119634137393442147122488023224991199983464198741547823461184440983311092859063315339005856209570774217037563735716525767202146555022835634830394173335846701956090603347993027348943503358988459283180126267229524258589926148054417247211544452012680947961970739427566014231907376004777806027146356673098104051821980813609489620627536318586072902362453001822146836953398711026938314624231118172527404160023662479675591972679463080892870924033390500260912484149710728580399922596811737729933496306407864316279286759308467225586400630712058062669097702930610741950266142986951131473864541346772760763803060336404546192429256660139570339479557058048723300221682130062742082192877293317152000048405763082356842429761275660461823473207814459076898750836740848867906794151648175125343727837463852243291610840693776703896004026903245226739042203315018011723928469119245936172041590507477293955509088990008598863091037064353500783631869397593772370717898887688588656836002080816443697460008562641267127150284717822620976444113210383784337269761228551739538537661771648519656505986689078806853055487415793334545320562269007016578463759972916520814246470659069212589074457381442970249815892442236639833612914028093400563728658690099330077173462953464599938306214364544174318409982374233427001201380610944383326758306643155426759762715096895248897595850591180683986862010984106454398490864610157313808094583469141943493531284574736984840068097800568497689543962031911947367408227594763844091957565106107552153730032737968670898825671936469994812821377371148837811022242318639063512522797025452571858333170997958817988795230522004873545554550282487459644952516376749624663157914902772002822124664066921173041693470279033273487170678300885104689707366128103993312598496630483714838112259393109030130535144009845879915694634267786141263380885785849272906872244664056483955690290055278651899721051918023553310757045708515659594635266177354471360305022071898027450050585431374121948718463407693518602305177427966019074621123632314487234921417610740908947557207513826231873744514086736226235143279308125693108701314688331416463056231330867123540258163072135182575267182190427548515620883347426842739355454752310511865573766860184468595791688099260885444800167193009921183883965820563135726020816703884459589556682160115132810424824252116863915855540451425851381715508993046615859507841678722931154711559350241372172593786348680266168535145295933672774471722344578214689705563351340012396307844383667613171566836460528972255085382673246755302054150197866613699839167088618871297493257010480470455239361070724266161575145504198951866327000624965652889530195949194330816111277963387244202472753221727896667936872408277085170455804035247721214206667804118222678366309127382321473430743182505300094750183501105390685307605020319337760620854469088397002338646173153791285606939422450011302792632370182584037297765211493362061107702329576458675220311461054166606053194526254493310277022880289968536096617791651468640429001141725669363037783525088567347058387512279167476856953351923938798955727765620493388195685958796310798250237155568677912214404559537936231118212271082243617849922989075106329111106572063957896808953038509983200902269366037747622165346861497447179331602930580324150022454102342507835843349197976573287021851331096888382968690184329353251744124436065474015636142607668716392092917268632880110913217427117546679735438455672079917067162120538443971150213375204307156414558816105667545087274132381023468022066316712175812850423340012025864091639920101684176981035329660680045837014982600258305887095020132484753955366969222139647157663437576548452148124229468912960099327079014070632928251812395462385112189819892864694717015076953611613149576949836570935890074526719905624349434306779854987391953235565839847070646609005469284353923989436538555992176865963181235199956060145629695233945542149217483266953743523609953500013879273190416645921891199100962876923624231417488878753148100157917029723601497336622624806763082966696488928823603119853411128160195147064476501628997776459655867429225391299737768182355980214433207049895613039244456016803216551924999502931415811311156454661531023992298329324467877763016879839388147591280860594803347692033418184793039682106635091257031344053862345723881247581268656834354608589094822517266238480843646377599157751739437195375754651872905898013964468475328153117208208719010225797964869052848022342097211379029490253282457508340688574384761289605583949449904571053394039281660964487570054000193697712484682478313845750281326148060436634600190408421951985937574582443631064338643146777317241248138114390808904859659180355816305279541711848045459316278778708587563810575096857466668685437268723884704041210712707941260474160459880877409059661716529058563497149681274770292765153477410334797868895468975002219652166818858750507295074217077771378565682357382646692235465574069382833513216703687463284432410490799068553286410983224530674095298251647419485661227960824588952394673196872070657944880816263971733987254510989703201696536963382040691338799107791181515903205839737181299730926557256483758426657677292876629362784373024520230433768307775430481076916705402148366027762698662843157711052325479790842872339201119829825402079781919936426836667372052118560603502542516929854227081834708421392938123648811993542940105796148158924473176942603591903780457552392852833120082822407307370240446843330795479933686130866313863667356521800069708273793764401331591952924497022864877785118949301478262984007553670621453257650461932344214783208669865394966714981444117630008306894421364005096365216585031796993715399027829945440346288693325830268249652681107760671629888771276083695090436233768009046904528365363234004107251341638439728362773629077299266694331144121545792569495200273018567887609801855990205900886811752701644545124869687236607088696109495418420914460162018787786666211079535069945660089848663998164798075060795225412201924655428643451526420829761531275640771961372103315034401253091875007906339695352019514032013514814139314613175365865262433787168287029001115698997996095251867365205922683714648016009365864724204163270292439504830523673673486175495316774050519360055640404638972105730103866419485787017163430044123031076545988774922413601706324077498870741013977609100683319210182492440669559457941969656416767626506471682689046582636449935510996826450527901963670054947700276995472927964878130257225735305862840471833648520008168518268015648590130251850833942831647252289739876050983171253364514964272315381274621274048451489064721784510316838818309910560545396410048541753019451403429342419803697825873246140818271360975119904456267061847680127857145988155978222327015943577186472292875290282322620059650293921816594324599152516885476989893314314757433965893479557242976903075527231025457730702616869116558807435492663585288768172556480994463241294768316018337488908221737087775414801869344453867456080820088970623048357649619213083805282683821415276018235994272561050710639604902969414356591422883672784985021335756157623183527135610253533681008353989239304392113094778323078548686538614675649008504905250350738703575854081154937955186471606292870710048757745296580622346398728999657024584814242727803281153861543415371577286590441293275568601393566885824607739317675301479685407426171913314160026648204306152345033613536617090828600426561348821856987898401457212727377458505586293246947212627849112496400669439609513852421846880258373836399110071420751796315064294214558415739801981761017292294264076088362374828573114136409107264040790327127424006904453926965605366751038052785526038792063741147124845706254678374647656387024454317210426172517201061304334984590596002863835733568605350684736519945184956358663404400992057541788395429437839910457357818703266019379168920147880552850138176986780608428477884110334948663750105880730934236251821533163043079598314053801836800929343519635593906003871741322361026121581362888445145486293599658109210435137237871060881180313823256611694917814388325735799607697200760016864960905526619587848562550339046701894238104393832996676541742864970467566596608025539468047949345849002606223961065308891531921339931537936842792993998833414004825620659139495509836134898894586607020344204161252221739058043796100034285265299650413233602846928179827210196699532846048076182348834636113562982736523595464523139937006230497014863470909956553755337962055775211408311847504483140081683837403307420100175491533182951692065657499643402349291886677170633176517088857503926810725706765930074004312086119024852504156833505953451541703100917201212623526753251865710466706910837341519774169677105735299938290134706574586048220247411414344140652313940251347128410171542460621524057592652109919482110216578276907675533916706125303731651305900173753457512831384615059102017374373844133913607853274641406700890631174611975848649663001584593756504596652530334122687192959531272741733757148670583333588752284770039177057377888183755055767538896085138291482556099378800583189987799524878310173777242646264902694012276761795774171764579533074680594026080747192003067365210267055463903645114197220528852677758859144852092161048075737704219021730808032990443332057466649775387001822637790663118529008925114040853614512968975675439859877822568599339317170617702363610264932469752759526220732084713654477220615452277507503914967583611987751379885260417852919984857264517205286721026660457961012551308280298124393093398152691906198268301388915992269765873938812194147970919277009090684619146184976165549621598911602800409260472374801041237903334537424933862623920052763473393084867297897752871845085530010136587525521875004610333810492163987761984629555627855250387493294356817216419435832380198937928761010463536580193010540735826433760550212172484543572473756532777331627535138522214809645838502760182558217715164733351364696494231273921817297743357788885645872613102102910792138892462791007437036478573887483521125228591708478049740535996472795082928224154369833138051387863422146528573611914898429145751142659825182601036963196191538408832189418613101951262833286126012469538246543439834289311241917329814066886604301444123671018628504972080101328974176221888356043631716619632757366892133167748666308076171955825734371443919482144616816291717558137679588524161730872519255503371311380625681355046467805586801706706624500902592386391311935078875464234009463443987458645529941370928438543280471652126684390119897825161909762122326852175585939901231148874424331983020522258781070475475291869308587846683765192027821999629748631274122362107893258112586730962757960178003168636096741041545976256113595265376387142801256877272810537002398930350320912008976236818950757012497545263058896202419194556470068313511381761038131409146935613543726647240403940693861855613248095081768699277440858422497896570279386268759594974427725107226145901483215451774447467370352211786290212540792266340478897200310411813580689731742918131726358876414192921328835259430684093225506802242715588237863424275044460721615574837870327822422419299668308268921891592990490044895293126828370872949969379556331044722545336614776566670203350506254506878386897319963671004724887831257440437762231008652071616946090838875702176053791048922073334896935014405676827811664837803603304025299656436639654166962870875846259655682491259867934178703458120975188229588986918044334729397076097819066471160864259513013554259727531399959556228510083799213746121474860552235897964909084800403508742449421066765643533996077611836178897735561516465066516263214543860198903906792595837547401073492923290184069627649030588521249731321813958310077315252888778759139245273336840205090311595462247666003445576340596338401223141962188723910409610468001642950596498737387111905290386017761638576000805218030180504713015062449578789171309781229741593803177353117719231484858963152609008668267842705374415534618247716603724585543379598003324614869660455995756090644490638179111078579993404797553885117284646909087706489097436087487020820933608498602241395457626459241342231022045908637838550784983960212474056015736493807614806449604580025829629436031842855117875875925666228442986795087102635910032906162209069277227445054234233590554113313943706628945649707836323382196419627905936701052523364743111579649604933436154918936492487639494765774775811598525682228953868543866816833787946949656485413832044523653492677174044706940746178760336848898679074330274974225856926781223981432517301037141308643389933304370636855700904845251275254939070514101192332761926234619405927249168289144613863509663266087655507228610483355519839248199962008766776459435741351475312933254743837859465292171211373148145316299922738410132474224244079898866942430276922890790222293443314904996800400341480763379704588408953699881015248963496845146103959525302520873085445080020596133461783025620532591579243749566650805237814782168385169853012729739609145239330381527196187849205511528432406577895361917898312747676354684562145334623677025651716436734462028974671306527905613695497461933604992132056154233431876060400718127699483280221606117696894922006981394575490222214044588446322123729308568213155822885971284233958426897246648897215853516351926860656548745180919789690658194324809766158438588182578298578849370508735546346067691045434723402074833020655376653949731644362676799437251588563324510300040296114182860582602196009925431563838937686088969362436465637535783937456055689795027770545651568273083992418996103669371300691580081429797975769106578339060710818004340404175003065424808872363315337871698653784851602675853831943439513048234685169744318907970917747967861834225346937398560101351445731925447263614250544442859528190956426613717033010864330216076176017701793712935270633432603495197635337898957450842567483078979484040316091497001745228117059526246306650022312407196041774513525975690579105687090666962313662444316622056234639492408883171925334913291885917945755929246969156319469324182773414608879276825510953110172993314290644725530350665835685877671471102555937651402110403234435688176179290895913063657991491574402852279272948459025283230578106633124043094186073080951413677508213163369465249246512604763260087461579148037643388666017380089738893306195833841480508118369407447364266287929101003912133049702099423930169058329909094738293883411694566185578662353302909885152084965137882661582978304149464109737931668326872185361407758772354339124310074144765906903230491351225720932845047869873012072892306168781907303511605743233739434130548145213256720938469429203484882740836515140308810520626435562386474559509452991492590324538793118680865874847930727278699151299723057573501866355327722949008184669796411715515823343785717080153497645025259657503129897112676081367997662504454333581653301715599814865001793080217135581242958117299762124866680247433509910449194010973439660112698147810232027888503158371974400780881902721035200902650658998916364871389026579144825850982480823840679564702428381346068103753771623616384285892556462653252969091624519596307568743523113779778032573467961307353275839992392880608756837385890346947254758221741616802734196939351381394363694462894764379830555383716430885348513935738484117790606097887902756513683914819898331224812865200509817856587757295543357366049734174048564060125254423532422791018033095461532105131564111889052072204470345511975926573513598946603646494565677430255324240761494163947811274793464819139659979366514741912819019789491157559086984274664028947847840463813931985659154362376165194295362244117099776525121140395240925988424508920758345690383950040551083162589034042656389236128965992656531641721355746167102587201791845831197924036956118103134606013991412067650283351308067002339771748028710761313273886439021739986017265221403958762111520449336796839495169748632585499084094817904008859493864850664378327153102056436779822659166065559723327726793693628360583042233776771347245505385084465963435188397560814259955015179053988829497734560666777734469094633214128824508763685875319122842234034995420760096598284147537834357083731581346052885077689131320470207721383005387665515392628439922316403955802020713517810820775026508024207781129326108900191981732550821034652392012874572056678439073849111643494262384013904276749437597263478933998195060733123447146332547052508239853981068470638351152174198942501740466785278909480445403418015716419375"));

        for (int prec = 25000; prec <= 50000; prec += 1000)
        {
            Apcomplex tmpZ = z.precision(prec),
                      expectedInvZ = invZ.precision(prec),
                      expectedInvSqrtZ = invSqrtZ.precision(prec),
                      actualInvZ = ApcomplexMath.inverseRoot(tmpZ, 1),
                      actualInvSqrtZ = ApcomplexMath.inverseRoot(tmpZ, 2);

            assertEquals("inv prec " + prec + " precision", prec, actualInvZ.precision());
            assertEquals("inv prec " + prec + " value", expectedInvZ, actualInvZ, new Apfloat("5e-" + prec));
            assertEquals("invsqrt prec " + prec + " precision", prec, actualInvSqrtZ.precision());
            assertEquals("invsqrt prec " + prec + " value", expectedInvSqrtZ, actualInvSqrtZ, new Apfloat("5e-" + prec));
        }

        z = new Apcomplex("(3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165271201909145648566923460348610454326648213393607260249141273724587006606315588174881520920962829254091715364367892590360011330530548820466521384146951941511609433057270365759591953092186117381932611793105118548074462379962749567351885752724891227938183011949129833673362440656643086021394946395224737190702179860943702770539217176293176752384674818467669405132000568127145263560827785771342757789609173637178721468440901224953430146549585371050792279689258923542019956112129021960864034418159813629774771309960518707211349999998372978049951059731732816096318595024459455346908302642522308253344685035261931188171010003137838752886587533208381420617177669147303598253490428755468731159562863882353787593751957781857780532171226806613001927876611195909216420199,3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165271201909145648566923460348610454326648213393607260249141273724587006606315588174881520920962829254091715364367892590360011330530548820466521384146951941511609433057270365759591953092186117381932611793105118548074462379962749567351885752724891227938183011949129833673362440656643086021394946395224737190702179860943702770539217176293176752384674818467669405132000568127145263560827785771342757789609173637178721468440901224953430146549585371050792279689258923542019956112129021960864034418159813629774771309960518707211349999998372978049951059731732816096318595024459455346908302642522308253344685035261931188171010003137838752886587533208381420617177669147303598253490428755468731159562863882353787593751957781857780532171226806613001927876611195909216420199e-20)");
        invZ = new Apcomplex("(0.318309886183790671537767526745028724068887460492294518428180911365119092396046181434178376080663353821009173544276311455642405854621090970373833904108972047312915693289356046049118109403439718172703494223105859718781812969543103286206394410806134813539553699626825863476727272799898208392521732801094763613545214073217354874639667048516536717204795290926651190121934930599524109477852654399389412872736085204160015325250614659445812229055084108341819036001796031629129394796573412318684485632518425385830357671174656209774572200992380951793462857729121730710703754813506833901088477689224753400237800184028789349091246281008776667281736464380349416427846404847680388469364742310525487683211843499436782010153659045483125565037272385913106741831620126137301310655977395673814982293995334905769140292085217372834361554984204900827704600919956923458209134644251890168698857503452823024163773233193012892383081929565070416391662340642298133395889765186066818049772720421198307486712656459423699740714924,-3.18309886183790671537767526745028724068887460492294518428180911365119092396046181434178376080663353821009173544276311455642405854621090970373833904108972047312915693289356046049118109403439718172703494223105859718781812969543103286206394410806134813539553699626825863476727272799898208392521732801094763613545214073217354874639667048516536717204795290926651190121934930599524109477852654399389412872736085204160015325250614659445812229055084108341819036001796031629129394796573412318684485632518425385830357671174656209774572200992380951793462857729121730710703754813506833901088477689224753400237800184028789349091246281008776667281736464380349416427846404847680388469364742310525487683211843499436782010153659045483125565037272385913106741831620126137301310655977395673814982293995334905769140292085217372834361554984204900827704600919956923458209134644251890168698857503452823024163773233193012892383081929565070416391662340642298133395889765186066818049772720421198307486712656459423699740714924e-21)");
        invSqrtZ = new Apcomplex("(0.564189583547756286948079451560772585844029472219615815983325168731208939469524264130849715258371807352118949175282690765845344942917477845270569742604457351973786191393343714152098802730948710056242456913534549142144868681124341539089763512011185016222112993127514023862896805089668633982814102772802015847857223491343393161133327212199033121274114199242243886056093480418532190764567074258299440784132207644069312567711083044479177537911030563217695713287365789838112929530752269146700963429727905088214177563066678695686031445447823998180399322898599028069989982482713328037727115586100854156652407125940257202088435542491586162644131107913481116958281820113230632775481749843710252733830295427323497470038385845571714020385150993253030073443070150859109512539452193591487227624349141289194482012167462681566049156405755813963874741659799112317643080264922748938583144492735585571155995975646825688024585904442417990229905395179613654357994285027995462492448607663494664326023222904298961716735477,-2.820947917738781434740397257803862929220076837400135610380757333724599600774390820496406021488337914539999917013808424821174183316705600346627044878665321316704542521132711956696115985692225461524144103674892886692811597954261197086278401249934262903436722704995347238317043044430197303220059198367524344827033735111145878843692931588007247802420701476138362763155773255754035575528173694450204821839899996342068039557128799949289686121250460169211402363399566614236747718540967339344797060583864253993886722179688695573684192522504572880979317725720745670246767341402104495822263278183453247632255562470843252936393521379423529102137851920938395838025560822786482305398364309855882826387811383500700796209016865318095637641931144775510847692507725489704866162502275965441880142102620023692574058012436283742735333878669894160523634912142148192752225318741559491075014194590494884342987343226966477362759864182357605917102577568819758568478928831698791913034318162836596723514735177322541225949696045e-21)");

        for (int prec = 500; prec <= 1000; prec += 50)
        {
            Apcomplex tmpZ = z.precision(prec),
                      expectedInvZ = invZ.precision(prec),
                      expectedInvSqrtZ = invSqrtZ.precision(prec),
                      actualInvZ = ApcomplexMath.inverseRoot(tmpZ, 1),
                      actualInvSqrtZ = ApcomplexMath.inverseRoot(tmpZ, 2);

            assertEquals("inv prec " + prec + " precision", prec, actualInvZ.precision());
            assertEquals("inv prec " + prec + " value", expectedInvZ, actualInvZ, new Apfloat("5e-" + prec));
            assertEquals("invsqrt prec " + prec + " precision", prec, actualInvSqrtZ.precision());
            assertEquals("invsqrt prec " + prec + " value", expectedInvSqrtZ, actualInvSqrtZ, new Apfloat("5e-" + prec));
        }

        z = new Apcomplex("(3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165271201909145648566923460348610454326648213393607260249141273724587006606315588174881520920962829254091715364367892590360011330530548820466521384146951941511609433057270365759591953092186117381932611793105118548074462379962749567351885752724891227938183011949129833673362440656643086021394946395224737190702179860943702770539217176293176752384674818467669405132000568127145263560827785771342757789609173637178721468440901224953430146549585371050792279689258923542019956112129021960864034418159813629774771309960518707211349999998372978049951059731732816096318595024459455346908302642522308253344685035261931188171010003137838752886587533208381420617177669147303598253490428755468731159562863882353787593751957781857780532171226806613001927876611195909216420199e-20,3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165271201909145648566923460348610454326648213393607260249141273724587006606315588174881520920962829254091715364367892590360011330530548820466521384146951941511609433057270365759591953092186117381932611793105118548074462379962749567351885752724891227938183011949129833673362440656643086021394946395224737190702179860943702770539217176293176752384674818467669405132000568127145263560827785771342757789609173637178721468440901224953430146549585371050792279689258923542019956112129021960864034418159813629774771309960518707211349999998372978049951059731732816096318595024459455346908302642522308253344685035261931188171010003137838752886587533208381420617177669147303598253490428755468731159562863882353787593751957781857780532171226806613001927876611195909216420199)");
        invZ = new Apcomplex("(3.18309886183790671537767526745028724068887460492294518428180911365119092396046181434178376080663353821009173544276311455642405854621090970373833904108972047312915693289356046049118109403439718172703494223105859718781812969543103286206394410806134813539553699626825863476727272799898208392521732801094763613545214073217354874639667048516536717204795290926651190121934930599524109477852654399389412872736085204160015325250614659445812229055084108341819036001796031629129394796573412318684485632518425385830357671174656209774572200992380951793462857729121730710703754813506833901088477689224753400237800184028789349091246281008776667281736464380349416427846404847680388469364742310525487683211843499436782010153659045483125565037272385913106741831620126137301310655977395673814982293995334905769140292085217372834361554984204900827704600919956923458209134644251890168698857503452823024163773233193012892383081929565070416391662340642298133395889765186066818049772720421198307486712656459423699740714924e-21,-0.318309886183790671537767526745028724068887460492294518428180911365119092396046181434178376080663353821009173544276311455642405854621090970373833904108972047312915693289356046049118109403439718172703494223105859718781812969543103286206394410806134813539553699626825863476727272799898208392521732801094763613545214073217354874639667048516536717204795290926651190121934930599524109477852654399389412872736085204160015325250614659445812229055084108341819036001796031629129394796573412318684485632518425385830357671174656209774572200992380951793462857729121730710703754813506833901088477689224753400237800184028789349091246281008776667281736464380349416427846404847680388469364742310525487683211843499436782010153659045483125565037272385913106741831620126137301310655977395673814982293995334905769140292085217372834361554984204900827704600919956923458209134644251890168698857503452823024163773233193012892383081929565070416391662340642298133395889765186066818049772720421198307486712656459423699740714924)");
        invSqrtZ = new Apcomplex("(0.398942280401432677941940771336389031865543401129091513282882250179772434419851767808651006965474553450231659632699401869161244597728783897275069734495032085216111887271560916882166876402125564868424323893838395079833719085290052945723478416783792406475917822229067524633036157052429629054952336706428571030187185130368246152628279046184365255879570477684594489756177508286624891300014994270929571804086453067394014233254639503393473181933215576035395390550639697849997906045852679378668246627935464340165981060328216264423323994665445336102173311228974531563690928652600365857426055953950317930869595684120658335492212442508507834457013088943568704831703105447241317646019451168564794944585905138454975351406084372143958122058613660877516675727140023601583669342317551944445825346281293391559591715805382904811500194433043087334063140186294620627312015177933295795822130423296667674673191468264898033406590955879437301865909636832022723153732099412803030603571832486087242410216578677303181161040972,-0.398942280401432677937951348532374705086143940529747694598123913207048338738616221978231886695149332457228174014279359091145043032910761545024527444992360506013704395545023729865459628156657910811196135090840777534898858802933750807644091893566642264434784387109684567850959336493692171697673283014869432855022112957977024436179321320242753133483455376103275583952799802580664801293539197676129806963898925886048129072348894165986263992243095527749288022655331420135724260556412139641344190345395202430297370019028628884610807050626790668495513149123868244378285115997893768231577855442515532795450441140924271501021684964050090674386687217757800362815660179461720592095429628585826372342230264108452648535563601356604345631990883903623336860080341288484907446373257062910833105988975533864466874928921595164065602310594305200462213282003201942543469515879364472417744667720718931280198196245354196327124908226579329264507159226749362033448374883848585906758047055682395695022774374795093935940257888)");

        for (int prec = 500; prec <= 1000; prec += 50)
        {
            Apcomplex tmpZ = z.precision(prec),
                      expectedInvZ = invZ.precision(prec),
                      expectedInvSqrtZ = invSqrtZ.precision(prec),
                      actualInvZ = ApcomplexMath.inverseRoot(tmpZ, 1),
                      actualInvSqrtZ = ApcomplexMath.inverseRoot(tmpZ, 2);

            // NOTE: EXTRA ALLOWED ERROR NEEDED FOR CASE prec=1000
            assertEquals("inv prec " + prec + " precision", prec, actualInvZ.precision());
            assertEquals("inv prec " + prec + " value", expectedInvZ, actualInvZ, new Apfloat("5e-" + prec));
            assertEquals("invsqrt prec " + prec + " precision", prec, actualInvSqrtZ.precision());
            assertEquals("invsqrt prec " + prec + " value", expectedInvSqrtZ, actualInvSqrtZ, new Apfloat("7e-" + prec));
        }

        assertEquals("1 branch 0", new Apfloat(1), ApcomplexMath.inverseRoot(Apcomplex.ONE, 5, 0));
        assertEquals("1 branch 10", new Apfloat(1), ApcomplexMath.inverseRoot(Apcomplex.ONE, 5, 10));
        assertEquals("1 branch 4", new Apfloat("-1.000"), ApcomplexMath.inverseRoot(new Apfloat("1.000"), 8, 4), new Apfloat("0.005"));
        assertEquals("1 branch 6", new Apcomplex("(0,1.000)"), ApcomplexMath.inverseRoot(new Apfloat("1.000"), 8, 6), new Apfloat("0.005"));
        assertEquals("(18,26) branch 1", new Apcomplex("(-0.2366,-0.2098)"), ApcomplexMath.inverseRoot(new Apcomplex("(18.000,26.000)"), 3, 1), new Apfloat("0.005"));
        assertEquals("(18,26) branch 2", new Apcomplex("(-0.0634,0.3098)"), ApcomplexMath.inverseRoot(new Apcomplex("(18.000,26.000)"), 3, 2), new Apfloat("0.005"));

        try
        {
            ApcomplexMath.inverseRoot(z, 0);
            fail("inverse zeroth root accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: inverse zeroth root
        }

        try
        {
            ApcomplexMath.inverseRoot(new Apcomplex("0"), 2);
            fail("inverse sqrt of 0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: result would be infinite
        }

        try
        {
            ApcomplexMath.inverseRoot(new Apcomplex(new Apfloat(0), new Apfloat(3)), 2);
            fail("infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK: can't have infinite memory
        }
    }

    public static void testRoot()
    {
        assertEquals("0", new Apfloat(0), ApcomplexMath.root(Apcomplex.ZERO, 3));
        assertEquals("1", new Apfloat(1), ApcomplexMath.root(Apcomplex.ONE, 5));
        assertEquals("1st root", new Apcomplex(new Apfloat(2), new Apfloat(3)), ApcomplexMath.root(new Apcomplex(new Apfloat(2), new Apfloat(3)), 1));
        assertEquals("root(-2, 2)", new Apcomplex(Apfloat.ZERO, new Apfloat(1.4142135623730950488016887242097)), ApcomplexMath.root(new Apfloat(-2.0), 2), new Apfloat(1e-15));
        assertEquals("root(-2, -2)", new Apcomplex(Apfloat.ZERO, new Apfloat(-0.70710678118654752440084436210485)), ApcomplexMath.root(new Apfloat(-2.0), -2), new Apfloat(5e-16));
        assertEquals("root((-46,9), 3)", new Apcomplex("(2.0,3.0)"), ApcomplexMath.root(new Apcomplex("(-46.000,9.000)"), 3), new Apfloat("0.5"));
        assertEquals("root((-46,-9), 3)", new Apcomplex("(2.0,-3.0)"), ApcomplexMath.root(new Apcomplex("(-46.000,-9.000)"), 3), new Apfloat("0.5"));
        //assertEquals("0x7FFFFFFFFFFFFFFFth root of 2", new Apfloat("1.0000000000000000000751511679015"), ApcomplexMath.root(new Apfloat(2, 30), 0x7FFFFFFFFFFFFFFFL), new Apfloat(2e-29));
        //assertEquals("0x8000000000000000th root of 2", new Apfloat("0.99999999999999999992484883209847"), ApcomplexMath.root(new Apfloat(2, 30), 0x8000000000000000L), new Apfloat(2e-29));
        assertEquals("sqrt(-2)", new Apcomplex(Apfloat.ZERO, new Apfloat(1.4142135623730950488016887242097)), ApcomplexMath.sqrt(new Apfloat(-2.0)), new Apfloat(1e-15));
        assertEquals("cbrt(3/4pi theta)", new Apcomplex(new Apfloat(0.70710678118654752440084436210485), new Apfloat(0.70710678118654752440084436210485)), ApcomplexMath.cbrt(new Apcomplex(new Apfloat(-0.70710678118654752440084436210485), new Apfloat(0.70710678118654752440084436210485))), new Apfloat(1e-15));
        assertEquals("cbrt(-1)", new Apcomplex(new Apfloat(0.5), new Apfloat(0.86602540378443864676372317075294)), ApcomplexMath.cbrt(new Apcomplex(new Apfloat(-1.0))), new Apfloat(1e-15));
        assertEquals("cbrt((-46,9), 3)", new Apcomplex("(2.0,3.0)"), ApcomplexMath.cbrt(new Apcomplex("(-46.0,9.0)")), new Apfloat("0.05"));
        assertEquals("cbrt((-46,-9), 3)", new Apcomplex("(2.0,-3.0)"), ApcomplexMath.cbrt(new Apcomplex("(-46.0,-9.0)")), new Apfloat("0.05"));
        assertEquals("cbrt((18,26), 3)", new Apcomplex("(3.0,1.0)"), ApcomplexMath.cbrt(new Apcomplex("(18.0,26.0)")), new Apfloat("0.05"));

        assertEquals("0 branch 1", new Apfloat(0), ApcomplexMath.root(Apcomplex.ZERO, 3, 1));
        assertEquals("0 branch 2", new Apfloat(0), ApcomplexMath.root(Apcomplex.ZERO, 3, 2));
        assertEquals("1 branch 10", new Apfloat(1), ApcomplexMath.root(Apcomplex.ONE, 5, 10));
        assertEquals("1 branch 4", new Apfloat("-1.000"), ApcomplexMath.root(new Apfloat("1.000"), 8, 4), new Apfloat("0.005"));
        assertEquals("1st root branch 2", new Apcomplex(new Apfloat(2), new Apfloat(3)), ApcomplexMath.root(new Apcomplex(new Apfloat(2), new Apfloat(3)), 1, 2));
        assertEquals("root(-2, 2) branch 1", new Apcomplex(Apfloat.ZERO, new Apfloat(-1.4142135623730950488016887242097)), ApcomplexMath.root(new Apfloat(-2.0), 2, 1), new Apfloat(1e-15));
        assertEquals("root(-2, -2) branch -1", new Apcomplex(Apfloat.ZERO, new Apfloat(0.70710678118654752440084436210485)), ApcomplexMath.root(new Apfloat(-2.0), -2, -1), new Apfloat(5e-16));
        assertEquals("root(-2, -2) branch 3", new Apcomplex(Apfloat.ZERO, new Apfloat(0.70710678118654752440084436210485)), ApcomplexMath.root(new Apfloat(-2.0), -2, 3), new Apfloat(5e-16));
        assertEquals("roots((-46,9), 3) branch 1", new Apcomplex("(-3.598,0.232)"), ApcomplexMath.root(new Apcomplex("(-46.000,9.000)"), 3, 1), new Apfloat("0.005"));
        assertEquals("roots((-46,9), 3) branch 2", new Apcomplex("(1.598,-3.232)"), ApcomplexMath.root(new Apcomplex("(-46.000,9.000)"), 3, 2), new Apfloat("0.005"));
        assertEquals("roots((-46,-9), 3) branch 1", new Apcomplex("(-3.598,-0.232)"), ApcomplexMath.root(new Apcomplex("(-46.000,-9.000)"), 3, 1), new Apfloat("0.005"));
        assertEquals("roots((-46,-9), 3) branch 2", new Apcomplex("(1.598,3.232)"), ApcomplexMath.root(new Apcomplex("(-46.000,-9.000)"), 3, 2), new Apfloat("0.005"));
        assertEquals("cbrt((18,26), 3) branch 1", new Apcomplex("(-2.366,2.098)"), ApcomplexMath.root(new Apcomplex("(18.000,26.000)"), 3, 1), new Apfloat("0.005"));
        assertEquals("cbrt((18,26), 3) branch 2", new Apcomplex("(-0.634,-3.098)"), ApcomplexMath.root(new Apcomplex("(18.000,26.000)"), 3, 2), new Apfloat("0.005"));

        assertEquals("0 radix 12", 12, ApcomplexMath.root(new Apint(0, 12), 3).radix());
        assertEquals("-4 real part radix 12", 12, ApcomplexMath.root(new Apfloat(-4, 5, 12), 2).real().radix());

        try
        {
            ApcomplexMath.root(new Apcomplex("(2.0, 3.0)"), 0);
            fail("0th root accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: zeroth root
        }

        try
        {
            ApcomplexMath.root(new Apcomplex("0"), -5);
            fail("Inverse root of zero accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: inverse root of zero
        }

        try
        {
            ApcomplexMath.root(new Apcomplex("0"), 0);
            fail("0th root of 0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: result would be undefined
        }

        try
        {
            ApcomplexMath.root(new Apcomplex(new Apfloat(2), new Apfloat(3)), 2);
            fail("infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK: can't have infinite memory
        }
    }

    public static void testAllRoots()
    {
        Apcomplex[] allRoots = ApcomplexMath.allRoots(Apcomplex.ZERO, 3);
        assertEquals("0 length", 3, allRoots.length);
        assertEquals("0[0]", new Apfloat(0), allRoots[0]);
        assertEquals("0[1]", new Apfloat(0), allRoots[1]);
        assertEquals("0[2]", new Apfloat(0), allRoots[2]);

        allRoots = ApcomplexMath.allRoots(new Apcomplex("1"), 4);
        assertEquals("1 length", 4, allRoots.length);
        assertEquals("1[0]", new Apcomplex("1"), allRoots[0]);
        assertEquals("1[1]", new Apcomplex("(0,1)"), allRoots[1]);
        assertEquals("1[2]", new Apcomplex("-1"), allRoots[2]);
        assertEquals("1[3]", new Apcomplex("(0,-1)"), allRoots[3]);

        allRoots = ApcomplexMath.allRoots(new Apcomplex("-1.0"), 2);
        assertEquals("-1 length", 2, allRoots.length);
        assertEquals("-1[0]", new Apcomplex("(0,1.0)"), allRoots[0]);
        assertEquals("-1[1]", new Apcomplex("(0,-1.0)"), allRoots[1]);

        allRoots = ApcomplexMath.allRoots(new Apcomplex("(0,1.00)"), 2);
        assertEquals("i length", 2, allRoots.length);
        assertEquals("i[0]", new Apcomplex("(0.707,0.707)"), allRoots[0]);
        assertEquals("i[1]", new Apcomplex("(-0.707,-0.707)"), allRoots[1]);

        allRoots = ApcomplexMath.allRoots(new Apcomplex(new Apfloat(2), new Apfloat(3)), 1);
        assertEquals("1st root length", 1, allRoots.length);
        assertEquals("1st root[0]", new Apcomplex(new Apfloat(2), new Apfloat(3)), allRoots[0]);

        allRoots = ApcomplexMath.allRoots(new Apcomplex("(-46.000,9.000)"), 3);
        assertEquals("allRoots((-46,9), 3) length", 3, allRoots.length);
        assertEquals("allRoots((-46,9), 3)[0]", new Apcomplex("(2.000,3.000)"), allRoots[0], new Apfloat("0.005"));
        assertEquals("allRoots((-46,9), 3)[1]", new Apcomplex("(-3.598,0.232)"), allRoots[1], new Apfloat("0.005"));
        assertEquals("allRoots((-46,9), 3)[2]", new Apcomplex("(1.598,-3.232)"), allRoots[2], new Apfloat("0.005"));

        allRoots = ApcomplexMath.allRoots(new Apcomplex("(-46.000,-9.000)"), 3);
        assertEquals("allRoots((-46,-9), 3) length", 3, allRoots.length);
        assertEquals("allRoots((-46,-9), 3)[0]", new Apcomplex("(2.000,-3.000)"), allRoots[0], new Apfloat("0.005"));
        assertEquals("allRoots((-46,-9), 3)[1]", new Apcomplex("(-3.598,-0.232)"), allRoots[1], new Apfloat("0.005"));
        assertEquals("allRoots((-46,-9), 3)[2]", new Apcomplex("(1.598,3.232)"), allRoots[2], new Apfloat("0.005"));

        allRoots = ApcomplexMath.allRoots(new Apcomplex("-4.000"), -4);
        assertEquals("allRoots(-4,-4) length", 4, allRoots.length);
        assertEquals("allRoots(-4,-4)[0]", new Apcomplex("(0.500,-0.500)"), allRoots[0], new Apfloat("0.005"));
        assertEquals("allRoots(-4,-4)[1]", new Apcomplex("(-0.500,-0.500)"), allRoots[1], new Apfloat("0.005"));
        assertEquals("allRoots(-4,-4)[2]", new Apcomplex("(-0.500,0.500)"), allRoots[2], new Apfloat("0.005"));
        assertEquals("allRoots(-4,-4)[3]", new Apcomplex("(0.500,0.500)"), allRoots[3], new Apfloat("0.005"));

        allRoots = ApcomplexMath.allRoots(new Apint(0, 12), 2);
        assertEquals("allRoots(0, 2) length", 2, allRoots.length);
        assertEquals("allRoots(0, 2)[0] radix", 12, allRoots[0].radix());
        assertEquals("allRoots(0, 2)[1] radix", 12, allRoots[0].radix());

        try
        {
            ApcomplexMath.allRoots(new Apcomplex("(2.0, 3.0)"), 0);
            fail("0th root accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: zeroth root
        }

        try
        {
            ApcomplexMath.allRoots(Apcomplex.ZERO, -1);
            fail("inverse root of zero accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: inverse root of zero
        }

        try
        {
            ApcomplexMath.allRoots(new Apcomplex(new Apfloat(2), new Apfloat(3)), 2);
            fail("infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK: can't have infinite memory
        }

        try
        {
            ApcomplexMath.allRoots(new Apcomplex("1"), 0x80000000);
            fail("negative array size accepted");
        }
        catch (ApfloatRuntimeException are)
        {
            // OK: result would not fit in an array
        }
    }

    public static void testAgm()
    {
        Apcomplex a = ApcomplexMath.agm(new Apcomplex(new Apfloat(2, 100), new Apfloat(3, 100)),
                                        new Apcomplex(new Apfloat(4, 100), new Apfloat(5, 100)));
        assertEquals("(2,3), (4,5) precision", 100, a.precision());
        assertEquals("(2,3), (4,5) value", new Apcomplex("(2.917544260525786263696583078746606829791948283056476178719125080604001346981059514497122989501231285,3.939113046692836869408348425071199799284873423064425353396384561367021573926690156072311985300901402)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 1), new Apfloat(2, 1)),
                              new Apcomplex(new Apfloat(2, 1), new Apfloat(1, 1)));
        assertEquals("(1,2), (2,1), prec 1, precision", 1, a.precision());
        assertEquals("(1,2), (2,1), prec 1, value", new Apcomplex("(1,1)"), a, new Apfloat(1));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 20), new Apfloat(2, 20)),
                              new Apcomplex(new Apfloat(1, 10), new Apfloat(2, 10)));
        assertEquals("(1,2), (1,2), prec 10-20, precision", 10, a.precision());
        assertEquals("(1,2), (1,2), prec 10-20, value", new Apcomplex("(1,2)"), a);

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(3, 20), new Apfloat(4, 20)));
        assertEquals("(1,2), (3,4), prec 10-20, precision", 10, a.precision());
        assertEquals("(1,2), (3,4), prec 10-20, value", new Apcomplex("(1.877468123,2.921948542)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(3, 10), new Apfloat(4, 10)));
        assertEquals("(-1,2), (3,4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,2), (3,4), prec 10, value", new Apcomplex("(0.657368426,3.174083677)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(3, 10), new Apfloat(4, 10)));
        assertEquals("(1,-2), (3,4), prec 10, precision", 10, a.precision());
        assertEquals("(1,-2), (3,4), prec 10, value", new Apcomplex("(2.676463508,0.428671830)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(3, 10), new Apfloat(4, 10)));
        assertEquals("(-1,-2), (3,4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,-2), (3,4), prec 10, value", new Apcomplex("(2.051752596,-0.072458699)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(4, 10)));
        assertEquals("(1,2), (-3,4), prec 10, precision", 10, a.precision());
        assertEquals("(1,2), (-3,4), prec 10, value", new Apcomplex("(-0.657368426,3.174083677)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(4, 10)));
        assertEquals("(-1,2), (-3,4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,2), (-3,4), prec 10, value", new Apcomplex("(-1.877468123,2.921948542)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(4, 10)));
        assertEquals("(1,-2), (-3,4), prec 10, precision", 10, a.precision());
        assertEquals("(1,-2), (-3,4), prec 10, value", new Apcomplex("(-2.051752596,-0.072458699)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(4, 10)));
        assertEquals("(-1,-2), (-3,4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,-2), (-3,4), prec 10, value", new Apcomplex("(-2.676463508,0.428671830)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(3, 10), new Apfloat(-4, 10)));
        assertEquals("(1,2), (3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(1,2), (3,-4), prec 10, value", new Apcomplex("(2.676463508,-0.428671830)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(3, 10), new Apfloat(-4, 10)));
        assertEquals("(-1,2), (3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,2), (3,-4), prec 10, value", new Apcomplex("(2.051752596,0.072458699)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(3, 10), new Apfloat(-4, 10)));
        assertEquals("(1,-2), (3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(1,-2), (3,-4), prec 10, value", new Apcomplex("(1.877468123,-2.921948542)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(3, 10), new Apfloat(-4, 10)));
        assertEquals("(-1,-2), (3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,-2), (3,-4), prec 10, value", new Apcomplex("(0.657368426,-3.174083677)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(-4, 10)));
        assertEquals("(1,2), (-3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(1,2), (-3,-4), prec 10, value", new Apcomplex("(-2.051752596,0.072458699)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(-4, 10)));
        assertEquals("(-1,2), (-3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,2), (-3,-4), prec 10, value", new Apcomplex("(-2.676463508,-0.428671830)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(-4, 10)));
        assertEquals("(1,-2), (-3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(1,-2), (-3,-4), prec 10, value", new Apcomplex("(-0.657368426,-3.174083677)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 10), new Apfloat(-2, 10)),
                              new Apcomplex(new Apfloat(-3, 10), new Apfloat(-4, 10)));
        assertEquals("(-1,-2), (-3,-4), prec 10, precision", 10, a.precision());
        assertEquals("(-1,-2), (-3,-4), prec 10, value", new Apcomplex("(-1.877468123,-2.921948542)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(0)), new Apcomplex(new Apfloat(1)));
        assertEquals("0, 1", new Apfloat(0), a);

        a = ApcomplexMath.agm(Apcomplex.ONE, Apcomplex.ZERO);
        assertEquals("1, 0", new Apfloat(0), a);

        a = ApcomplexMath.agm(Apcomplex.ZERO, new Apcomplex("(0,1)"));
        assertEquals("0, i", new Apfloat(0), a);

        a = ApcomplexMath.agm(new Apcomplex("(0,1)"), Apcomplex.ZERO);
        assertEquals("i, 0", new Apfloat(0), a);

        a = ApcomplexMath.agm(new Apfloat(-1.2, 10), new Apfloat(1.2, 10));
        assertEquals("-1.2, 1.2", new Apcomplex("0"), a);

        a = ApcomplexMath.agm(new Apfloat(1.2, 10), new Apfloat(1.3, 10));
        assertEquals("1.2, 1.3 precision", 10, a.precision());
        assertEquals("1.2, 1.3 value", new Apcomplex("1.249499750"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apfloat(-1.2, 10), new Apfloat(-1.3, 10));
        assertEquals("-1.2, -1.3 precision", 10, a.precision());
        assertEquals("-1.2, -1.3 value", new Apcomplex("-1.249499750"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apfloat(-1.2, 10), new Apfloat(1.3, 10));
        assertEquals("-1.2, 1.3 precision", 10, a.precision());
        assertEquals("-1.2, 1.3 value", new Apcomplex("(0.1302427887,0.3818051176)"), a, new Apfloat("5e-10"));

        a = ApcomplexMath.agm(new Apcomplex(Apfloat.ZERO, new Apfloat(1, 10)),
                              new Apcomplex(Apfloat.ZERO, new Apfloat(2, 20)));
        assertEquals("1i, 2i, prec 10-20, precision", 10, a.precision());
        assertEquals("1i, 2i, prec 10-20, value", new Apcomplex("(0,1.456791031)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(Apfloat.ZERO, new Apfloat(-1, 20)),
                              new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("-1i, -2i, prec 10-20, precision", 10, a.precision());
        assertEquals("-1i, -2i, prec 10-20, value", new Apcomplex("(0,-1.456791031)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.agm(new Apcomplex(Apfloat.ZERO, new Apfloat(1, 10)),
                              new Apcomplex(Apfloat.ZERO, new Apfloat(1, 20)));
        assertEquals("i, i, prec 10-20, precision", 10, a.precision());
        assertEquals("i, i, prec 10-20, value", new Apcomplex("(0,1)"), a);

        a = ApcomplexMath.agm(new Apcomplex(Apfloat.ZERO, new Apfloat(1, 20)),
                              new Apcomplex(Apfloat.ZERO, new Apfloat(1, 10)));
        assertEquals("i, i, prec 20-10, precision", 10, a.precision());
        assertEquals("i, i, prec 20-10, value", new Apcomplex("(0,1)"), a);

        a = ApcomplexMath.agm(new Apcomplex(Apfloat.ZERO, new Apfloat("1.000000000e10")),
                              new Apcomplex(Apfloat.ZERO, new Apfloat("1.000000000e-10")));
        assertEquals("1e10i, 1e-10i, precision", 10, a.precision());
        assertEquals("1e10i, 1e-10i, value", new Apcomplex("(0,331126196.7)"), a, new Apfloat("0.5"));

        a = ApcomplexMath.agm(new Apcomplex(Apfloat.ZERO, new Apfloat("1.0000000000000000000e10")),
                              new Apcomplex(Apfloat.ZERO, new Apfloat("1.000000000e-10")));
        assertEquals("1e10i, 1e-10i, precision 10-20", 10, a.precision());
        assertEquals("1e10i, 1e-10i, value, precision 10-20", new Apcomplex("(0,331126196.7)"), a, new Apfloat("0.5"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(2, 3000), new Apfloat(3, 3000)),
                              new Apcomplex(new Apfloat(4, 3000), new Apfloat(5, 3000)));
        assertEquals("(2,3), (4,5) precision 3000", 3000, a.precision());
        assertEquals("(2,3), (4,5) value 3000", new Apcomplex("(2.91754426052578626369658307874660682979194828305647617871912508060400134698105951449712298950123128562803369971608115858308936341123999327021346691138617183468446010737173164686210382470722972480385653096197955212595851386829730542017554834566076203624841629838837088111415383201962147391676642929058049817171098768157845839182061367660407624587113859085064455636920311607051981588597485536439305356293985101422595947649995371848599841753952499159126778852783849867972638132163229170462055215173927004466893155827350288414651232923783719736757181961513759146922026017058314213265164777972026914569278829769779023008225176184785498679682429641281302037486486521259687322336298021286618318415793710998463135836032875126545906276937050649443180086254169974282261221943615961474499125922697942531490842409305517925243257370218203245337833554418491842015943129213675762563419679038299132768627013663409472496225092290179397048258626692981873018305739254012305825026785891069123260517749979272211407491610089628959933724343122851980961660974260652181762103017936720767079145977839033281414352440639482552619868342969287513114456467494665868602786146218358077830070649048842337372991463043085279850727992640025660675380800254896810540213312123740821486832723122822609732501981710205986586577716162755069535207170070281037295556916467016669193561963726840316454962656506333047304666768044715822566242597700196677480301214952192861971988192656671744822490672697455664407140156730909231414013792802915135066242203113030716533667093257173402035594912939322282536077875674712764039027921927395197312547828266486016427640712420409764391754400652438995087760043085410360352341590241567046208031995471079328190006083782215711890791023443288680631383270729328245112098679248986918816410547156221869838319395337641455545236996980728609516738564908590670336904562106389447988845227636370340561649189020526564471128076963110284359493508944882615621434470215215697096025821872457004381345157727508034579204708799556234634004705842562329299420551585588232322265609498279231361039397683526002176899737997607538350921282494819816685120478868231090842595792591353576389181664605419273590632452310701393757926374438862479436554656101458985479249471472107699248705141561173712162097645050207626741418394746768164124670747173806773839597678289144754574734901040652010616194325729394023497262795568290483391326776172203388816276143223100903063086097060671378537506215702097576424605396225829921915151624090646742547987876299045332164101681114965791902981323550059875554202160881410486800946958238200767776068800905156439104937878902630802229026038558376052682025048018360113971819953917492535105743883667564195635498783445254231425089663831105369011694312663969400247121569845918627337905950490146786267588981716145101263064118596121741830302772515682282427194397934820742615189129211740981191425117917799767751060222539769267812947175474295243812595876964356919305482207550832587361273920159765406099319385252761850724428775715,3.93911304669283686940834842507119979928487342306442535339638456136702157392669015607231198530090140243949188290043429681557054024497519997714555187094318119446656556235082719330646782894759630438224022549553159147839217538004290128346087094038704007635212437747396661652367449483064282116458523952847341804070956765337261810846795283243730734590753449997447035281029295162928561049306004649118029649069256105503936796313664026977955159526283914846577547135764309851062923566730093681247668624871510666978126929508258099172045717391063101370239395419225705660926248842628084124376628056009064933668417324363913211419327954479103622167143221861590472148278185932670718814750047414193099895865869943477950105875157760091959759003434005966714649800873204258385294726671809906947015516595685711589454524234633598253319737514012268529650713739946676006981825396015444889386661029474872321206464973652372992130362462446398118626962435894308122356649646871349298187623711624971863476352762452221531183277320146100515317455662186611756111268760306975797019057296312184964013993084636666054078035009036249564083908773875800800166359587268483598525125806091226592778234809046332031551598992262638508581536585324305299591750302881483402739828364598398277730258197204969256091757799115119279623658639479367023067719518871651401707913451169051716348607113140192819952450181861796320610091212802617662333291955214929303347143495395840187888757139608191824192807233083558513911809059740349333087862760616275786722727908487366481453538363428895942223277133414813782540149625894704335654023680217631160052319030202630292942814891747431353863732117821363156846713863280617323679888536660636763839907093252497571827138167886149907772607307351368861795816765379220178268412856307579450653416268935107888700416468284108022921355702851498202881502762305540563232079766276637149902654506931054207254907043868092497807310230653405754717803877751418950428686594560372055456167827239476527970744301094268551441890020407558514144268177782841988620322994163840112592854907705270880910991400937352699877335762302999419793473129331441989800560461441763701563331940266726598599293771986578531341036022144096025745155626608882782484494825814636845245413589466189311555767748030432015023292022231992210329782566184971816685421724834819675158353561452463411398141131533561617853176875463475127794571247641075930817108546879442258495905380115608124074575312893808818647331957099686532636924786401829250331323516175691344769031519765960343036596327787933330699717223235941567369627380749831452883229157945444590678814697824995710977262891494397099520222876736838798922496231008700179679987264466109787010313153842130194074782194326215333849204597293821883655534230029263086716683924996834814477422800580002681248614408636157256745534681222987070933199461219239318856812325932811189024056557789637085405421384060709459119263501223615502437218473705700310779175993531915209583296165864005423678994006648697891946757802094451363373198938559)"), a, new Apfloat("5e-2999"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-2, 30), new Apfloat("1.000000000e-40")),
                              new Apfloat(1));
        assertEquals("(-2,1e-40), (1,0) precision", 30, a.precision());
        assertEquals("(-2,1e-40), (1,0) value", new Apcomplex("(-0.422966208408801687364597406061,0.661266183461804764467239865563)"), a, new Apfloat("5e-30"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-2, 30), new Apfloat("-1.000000000e-40")),
                              new Apfloat(1));
        assertEquals("(-2,-1e-40), (1,0) precision", 30, a.precision());
        assertEquals("(-2,-1e-40), (1,0) value", new Apcomplex("(-0.422966208408801687364597406061,-0.661266183461804764467239865563)"), a, new Apfloat("5e-30"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(-1, 30), new Apfloat(1, 30)),
                              new Apcomplex(new Apfloat(-1, 30), new Apfloat(-1, 30)));
        assertEquals("(-1,1), (-1,-1) precision", 30, a.precision());
        assertEquals("(-1,1), (-1,-1) value", new Apcomplex("-1.198140234735592207439922492280"), a, new Apfloat("5e-29"));

        a = ApcomplexMath.agm(new Apcomplex(new Apfloat(1, 30), new Apfloat(1, 30)),
                              new Apcomplex(new Apfloat(1, 30), new Apfloat(-1, 30)));
        assertEquals("(1,1), (1,-1) precision", 30, a.precision());
        assertEquals("(1,1), (1,-1) value", new Apcomplex("1.198140234735592207439922492280"), a, new Apfloat("5e-29"));

        a = ApcomplexMath.agm(new Apint(1, 12), new Apint(0, 12));
        assertEquals("1, 0 radix", 12, a.radix());
        a = ApcomplexMath.agm(new Apcomplex(Apint.ZERO, new Apint(1, 12)), new Apcomplex(Apint.ZERO, new Apint(-1, 12)));
        assertEquals("i, -i radix", 12, a.radix());

        try
        {
            ApcomplexMath.agm(new Apcomplex(new Apfloat(1)), new Apcomplex(new Apfloat(2)));
            fail("AGM to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testLog()
    {
        Apcomplex a = ApcomplexMath.log(new Apcomplex(new Apfloat(2, 100)));
        assertEquals("2, 100 precision", 100, a.precision());
        assertEquals("2, 100 value", new Apcomplex("0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex(new Apfloat(-2, 100)));
        assertEquals("-2, 100 precision", 100, a.precision());
        assertEquals("-2, 100 value", new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex(Apfloat.ZERO, new Apfloat(2, 100)));
        assertEquals("2i, 100 precision", 100, a.precision());
        assertEquals("2i, 100 value", new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,1.570796326794896619231321691639751442098584699687552910487472296153908203143104499314017412671058534)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 100)));
        assertEquals("-2i, 100 precision", 100, a.precision());
        assertEquals("-2i, 100 value", new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,-1.570796326794896619231321691639751442098584699687552910487472296153908203143104499314017412671058534)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex("(1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"));
        assertEquals("(sqrt2,sqrt2), 100 precision", 99, a.precision());
        assertEquals("(sqrt2,sqrt2), 100 value", new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,0.785398163397448309615660845819875721049292349843776455243736148076954101571552249657008706335529266996)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex("(1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"));
        assertEquals("(sqrt2,-sqrt2), 100 precision", 99, a.precision());
        assertEquals("(sqrt2,-sqrt2), 100 value", new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,-0.785398163397448309615660845819875721049292349843776455243736148076954101571552249657008706335529266996)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex("(-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"));
        assertEquals("(-sqrt2,sqrt2), 100 precision", 100, a.precision());
        assertEquals("(-sqrt2,sqrt2), 100 value", new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,2.356194490192344928846982537459627163147877049531329365731208444230862304714656748971026119006587801)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex("(-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"));
        assertEquals("(-sqrt2,-sqrt2), 100 precision", 100, a.precision());
        assertEquals("(-sqrt2,-sqrt2), 100 value", new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,-2.356194490192344928846982537459627163147877049531329365731208444230862304714656748971026119006587801)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex(new Apfloat(3, 150, 16), new Apfloat(3, 150, 16)));
        assertEquals("(3,3), 150 precision", 150, a.precision());
        assertEquals("(3,3), 150 value", new Apcomplex(new Apfloat("1.71F7B3A6B918664C890B67210734F80D5E8DE46CF12CE907029E7A98171EB2A29E73BDE243BC6E9C7B3CA0B0A9179C5DD8F36C21FF43CD43E2E3768598D9A277261BF9ED2F9A4507D5456", Apfloat.DEFAULT, 16), new Apfloat("0.C90FDAA22168C234C4C6628B80DC1CD129024E088A67CC74020BBEA63B139B22514A08798E3404DDEF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245E485B576625E7EC6F44C43", Apfloat.DEFAULT, 16)), a, ApfloatMath.scale(new Apfloat(5, 1, 16), -148));

        a = ApcomplexMath.log(new Apcomplex("1"));
        assertEquals("1", new Apcomplex("0"), a);

        a = ApcomplexMath.log(new Apcomplex("-1.1"));
        assertEquals("-1.1, real precision", 1, a.real().precision());
        assertEquals("-1.1, precision", 2, a.precision());

        a = ApcomplexMath.log(new Apcomplex("-1.01"));
        assertEquals("-1.01, real precision", 1, a.real().precision());
        assertEquals("-1.01, precision", 3, a.precision());

        a = ApcomplexMath.log(new Apcomplex(new Apfloat("-1.1", 3)));
        assertEquals("-1.1, 3, real precision", 2, a.real().precision());
        assertEquals("-1.1, 3, precision", 3, a.precision());

        a = ApcomplexMath.log(new Apcomplex("(1.1,0.1)"));
        assertEquals("(1.1,0.1), precision", 1, a.precision());

        a = ApcomplexMath.log(new Apcomplex("(1.01,0.01)"));
        assertEquals("(1.01,0.01), precision", 1, a.precision());

        a = ApcomplexMath.log(new Apcomplex(Apfloat.ZERO, new Apfloat("2.567534329783818000500533029709932117494558299117260796421579784603311989535237879614227625064708170e2171472", 100)));
        assertEquals("(0,e^5000000), 100 precision", 106, a.precision());
        assertEquals("(0,e^5000000), 100 value", new Apcomplex(new Apfloat(5000000), new Apfloat("1.570796326794896619231321691639751442098584699687552910487472296153908203143104499314017412671058534")), a, new Apfloat("5e-99"));

        a = ApcomplexMath.log(new Apcomplex(Apfloat.ZERO, new Apfloat("3.894787261069254290252471079176073765185215930321023940272302235080330044710873765250549604329029361e-2171473", 100)));
        assertEquals("(0,-e^5000000), 100 precision", 100, a.precision());
        assertEquals("(0,-e^5000000), 100 value", new Apcomplex(new Apfloat(-5000000), new Apfloat("1.570796326794896619231321691639751442098584699687552910487472296153908203143104499314017412671058534")), a, new Apfloat("5e-92"));

        a = ApcomplexMath.log(new Apcomplex(new Apfloat(-2, 100, 2)));
        assertEquals("-2, 100 radix 2 precision", 101, a.precision());
        assertEquals("-2, 100 radix 2 value", new Apcomplex(new Apfloat("0.10110001011100100001011111110111110100011100111101111001101010111100100111100011101100111001100000000011111100101111011010101111010000001111001101000011001001100111001010011000101101100010110110001010000011010001011101011011100010111010101011111010001010111110011110111000011101100010000001101101111010111010110010011000010101011001", Apfloat.DEFAULT, 2), new Apfloat("11.001001000011111101101010100010001000010110100011000010001101001100010011000110011000101000101110000000110111000001110011010001001010010000001001001110000010001000101001100111110011000111010000000010000010111011111010100110001110110001001110011011001000100101000101001010000010000111100110001110001101000000010011011101111011111001", Apfloat.DEFAULT, 2)), a, new Apfloat("1e-99", 1, 2));

        try
        {
            ApcomplexMath.log(new Apcomplex("0"));
            fail("log of zero accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be -infinite
        }

        try
        {
            ApcomplexMath.log(new Apcomplex(Apfloat.ZERO, new Apfloat(2)));
            fail("log to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testLogBase()
    {
        Apcomplex a = ApcomplexMath.log(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)), new Apcomplex(new Apfloat(2, 100), new Apfloat(3, 100)));
        assertEquals("(3,4) base (2,3), 100 precision", 100, a.precision());
        assertEquals("(3,4) base (2,3), 100 value", new Apcomplex("(1.1397284224569002062964218052429388242925638371440336662842481166316121503938668267670478367736422303,-0.1503520693183908310389743432512223715076049306602668639335618106843010268220651900947797357015316935)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.log(new Apcomplex("(3.0000,4.0000)"), new Apcomplex("2.000"));
        assertEquals("(3,4) base 2, precision", 4, a.precision());
        assertEquals("(3,4) base 2, value", new Apcomplex("(2.322,1.338)"), a, new Apfloat("0.005"));

        a = ApcomplexMath.log(new Apcomplex("3.0000"), new Apcomplex("(2.00000,3.00000)"));
        assertEquals("3 base (2,3), precision", 5, a.precision());
        assertEquals("3 base (2,3), value", new Apcomplex("(0.53970,-0.41358)"), a, new Apfloat("0.00005"));

        a = ApcomplexMath.log(new Apfloat("3.00"), new Apfloat("-2.00"));
        assertEquals("3 base -2, precision", 3, a.precision());
        assertEquals("3 base -2, value", new Apcomplex("(0.0736,-0.3335)"), a, new Apfloat("0.0005"));

        a = ApcomplexMath.log(new Apfloat("-3.00"), new Apfloat("2.00"));
        assertEquals("-3 base 2, precision", 3, a.precision());
        assertEquals("-3 base 2, value", new Apcomplex("(1.58,4.53)"), a, new Apfloat("0.05"));

        a = ApcomplexMath.log(new Apfloat(16, 400), new Apfloat(2));
        assertEquals("4 base 2, precision 400", 400, a.precision());
        assertEquals("4 base 2, value", new Apfloat(4), a, new Apfloat("5e-399"));

        a = ApcomplexMath.log(new Apfloat(16), new Apfloat(2, 500));
        assertEquals("4 base 2, precision 500", 500, a.precision());
        assertEquals("4 base 2, value", new Apfloat(4), a, new Apfloat("5e-499"));

        a = ApcomplexMath.log(new Apcomplex("(2.0000, 4.0000)"), new Apfloat("1.01", 7));
        assertEquals("(2,4) base 1.01, precision", 5, a.precision());
        assertEquals("(2,4) base 1.01, value", new Apcomplex("(150.53,111.27)"), a, new Apfloat("5e-2"));

        a = ApcomplexMath.log(new Apfloat("1.01", 7), new Apcomplex("(2.0000, 4.0000)"));
        assertEquals("1.01 base (2,4), precision", 5, a.precision());
        assertEquals("1.01 base (2,4), value", new Apcomplex("(0.0042959,-0.0031753)"), a, new Apfloat("5e-7"));
    }

    public static void testExp()
    {
        Apcomplex a = ApcomplexMath.exp(new Apcomplex(new Apfloat(2, 100)));
        assertEquals("2, 100 precision", 100, a.precision());
        assertEquals("2, 100 value", new Apfloat("7.389056098930650227230427460575007813180315570551847324087127822522573796079057763384312485079121794"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat(0), new Apfloat("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068")));
        assertEquals("(0,pi), 100 precision", 100, a.precision());
        assertEquals("(0,pi), 100 value", new Apfloat("-1"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat(2, 100), new Apfloat("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068")));
        assertEquals("(2,pi), 100 precision", 100, a.precision());
        assertEquals("(2,pi), 100 value", new Apfloat("-7.389056098930650227230427460575007813180315570551847324087127822522573796079057763384312485079121794"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.exp(new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,1.570796326794896619231321691639751442098584699687552910487472296153908203143104499314017412671058534)"));
        assertEquals("(log2,pi/2), 100 precision", 100, a.precision());
        assertEquals("(log2,pi/2), 100 value", new Apcomplex(Apfloat.ZERO, new Apfloat(2)), a, new Apfloat("5e-99"));

        a = ApcomplexMath.exp(new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,-1.570796326794896619231321691639751442098584699687552910487472296153908203143104499314017412671058534)"));
        assertEquals("(log2,-pi/2), 100 precision", 100, a.precision());
        assertEquals("(log2,-pi/2), 100 value", new Apcomplex(Apfloat.ZERO, new Apfloat(-2)), a, new Apfloat("5e-99"));

        a = ApcomplexMath.exp(new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,0.785398163397448309615660845819875721049292349843776455243736148076954101571552249657008706335529266996)"));
        assertEquals("(log2,pi/4), 100 precision", 100, a.precision());
        assertEquals("(log2,pi/4), 100 value", new Apcomplex("(1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.exp(new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,-0.785398163397448309615660845819875721049292349843776455243736148076954101571552249657008706335529266996)"));
        assertEquals("(log2,-pi/4), 100 precision", 100, a.precision());
        assertEquals("(log2,-pi/4), 100 value", new Apcomplex("(1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.exp(new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,2.356194490192344928846982537459627163147877049531329365731208444230862304714656748971026119006587801)"));
        assertEquals("(log2,3pi/4), 100 precision", 100, a.precision());
        assertEquals("(log2,3pi/4), 100 value", new Apcomplex("(-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.exp(new Apcomplex("(0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687,-2.356194490192344928846982537459627163147877049531329365731208444230862304714656748971026119006587801)"));
        assertEquals("(log2,-3pi/4), 100 precision", 100, a.precision());
        assertEquals("(log2,-3pi/4), 100 value", new Apcomplex("(-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573,-1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"), a, new Apfloat("5e-99"));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat(3, 150, 16), new Apfloat("3.243F6A8885A308D313198A2E03707344A4093822299F31D0082EFA98EC4E6C89452821E638D01377BE5466CF34E90C6CC0AC29B7C97C50DD3F84D5B5B54709179216D5D98979FB1BD1311", 150, 16)));
        assertEquals("(3,pi), 150 precision", 150, a.precision());
        assertEquals("(3,pi), 150 value", new Apfloat("-14.15E5BF6FB105F2D4BDFC53744C3A390585839728AA90A12389790C837E6FF2A68ABFE2D58DAFA5273C74EB175B9D0BD5584DFC81BE96F62CE66F428A2A50FFCBDF4083E8EE811BE42F", Apfloat.DEFAULT, 16), a, ApfloatMath.scale(new Apfloat(5, 1, 16), -147));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat(0, 1, 17), new Apfloat(0, 1, 17)));
        assertEquals("1 radix", 17, a.radix());
        assertEquals("1 prec", Apfloat.INFINITE, a.precision());
        assertEquals("1 value", new Apfloat(1), a);

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat("-100000000000000000000", 20, 17), new Apfloat("0.1", 1, 17)));
        assertEquals("underflow radix", 17, a.radix());
        assertEquals("underflow prec", Apfloat.INFINITE, a.precision());
        assertEquals("underflow value", new Apfloat(0), a);

        a = ApcomplexMath.exp(new Apcomplex("(0.1,3.14159)"));
        assertEquals("(0.1,pi), precision", 2, a.precision());

        a = ApcomplexMath.exp(new Apcomplex("(0.01,3.14159)"));
        assertEquals("(0.01,pi), precision", 3, a.precision());

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat("0.1", 2), new Apfloat("3.14159")));
        assertEquals("(0.1,pi), 2, precision", 3, a.precision());

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat("1e-10", 32), new Apfloat("3.1415926535897932384626433832795")));
        assertEquals("(1e-10,pi)", new Apcomplex("-1.0000000001000000000050000000002"), a, new Apfloat("5e-31"));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat(5000000, 21), new Apfloat("1.570796326794897")));
        assertEquals("(5000000,pi/2), 15 precision", 15, a.precision());
        assertEquals("(5000000,pi/2), 15 value", new Apcomplex(Apfloat.ZERO, new Apfloat("2.56753432978382e2171472")), a, new Apfloat("5e2171458"));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat(5000000, 106), new Apfloat("4.712388980384689857693965074919254326295754099062658731462416888461724609429313497942052238013175602")));
        assertEquals("(5000000,3pi/2), 100 precision", 100, a.precision());
        assertEquals("(5000000,3pi/2), 100 value", new Apcomplex(Apfloat.ZERO, new Apfloat("-2.567534329783818000500533029709932117494558299117260796421579784603311989535237879614227625064708170e2171472")), a, new Apfloat("5e2171373"));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat(-5000000, 106), new Apfloat("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068")));
        assertEquals("(-5000000,pi), 100 precision", 100, a.precision());
        assertEquals("(-5000000,pi), 100 value", new Apfloat("-3.894787261069254290252471079176073765185215930321023940272302235080330044710873765250549604329029362e-2171473"), a, new Apfloat("5e-2171572"));

        a = ApcomplexMath.exp(new Apcomplex(new Apfloat("-1e25", 50), new Apfloat("3.1415926535897932384626433832795028841971693993751")));
        assertEquals("(-1e25,pi), 50 value", new Apfloat(0), a);

        a = ApcomplexMath.exp(new Apcomplex("(-21237598959199934509.830775042768,3.1415926535897932384626433832795)"));
        assertEquals("(-21237598959199934509.830775042768,pi) value", new Apfloat(0), a);

        a = ApcomplexMath.exp(new Apcomplex("(-2.1237598959199920000000000000000000e19,3.141592653589793)"));
        assertEquals("(-2.123759895919992e19,pi) precision", 16, a.precision());
        assertEquals("(-2.123759895919992e19,pi) value", new Apfloat("-3.462892031805012e-9223372036854769507"), a, new Apfloat("5e-9223372036854769491"));

        Apfloat r = new Apfloat("-2.718281828459045235360287471352662497757247093699959574966967627724076630353547594571382178525166427427466391932003059921817413596629043572900334295260595630738132328627943490763233829880753195251019011573834187930702154089149934884167509244761460668082264800168477411853742345442437107539077744992069551702761838606261331384583000752044933826560297606737113200709328709127443747047230696977209310141692836819025515108657463772111252389784425056953696770785449969967946864454905987931636889230098793127736178215424999229576351482208269895193668033182528869398496465105820939239829488793320362509443117301238197068416140397019837679320683282376464804295311802328782509819455815301756717361332069811250996181881593041690351598888519345807273866738589422879228499892086805825749279610484198444363463244968487560233624827041978623209002160990235304369941849146314093431738143640546253152096183690888707016768396424378140592714563549061303107208510383750510115747704171898610687396965521267154688957035035"),
                pi = new Apfloat("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165271201909145648566923460348610454326648213393607260249141273724587006606315588174881520920962829254091715364367892590360011330530548820466521384146951941511609433057270365759591953092186117381932611793105118548074462379962749567351885752724891227938183011949129833673362440656643086021394946395224737190702179860943702770539217176293176752384674818467669405132000568127145263560827785771342757789609173637178721468440901224953430146549585371050792279689258923542019956112129021960864034418159813629774771309960518707211349999998372978049951059731732816096318595024459455346908302642522308253344685035261931188171010003137838752886587533208381420617177669147303598253490428755468731159562863882353787593751957781857780532171226806613001927876611195909216420199");

        for (int prec = 500; prec <= 800; prec += 100)
        {
            Apcomplex exp = ApcomplexMath.exp(new Apcomplex(new Apfloat(1, prec), pi.precision(prec)));

            assertEquals("exp prec " + prec + " precision", prec, exp.precision());
            assertEquals("exp prec " + prec + " value", r.precision(prec), exp, new Apfloat("5e-" + (prec - 1)));
        }

        for (int prec = 900; prec <= 1000; prec++)
        {
            Apcomplex exp = ApcomplexMath.exp(new Apcomplex(new Apfloat(1, prec), pi.precision(prec)));

            assertEquals("exp prec " + prec + " precision", prec, exp.precision());
            assertEquals("exp prec " + prec + " value", r.precision(prec), exp, new Apfloat("5e-" + (prec - 1)));
        }

        try
        {
            ApcomplexMath.exp(new Apcomplex(new Apfloat("50000000000000000000", 500000000), new Apfloat(3, 500000000)));
            fail("Overflow should have occurred");
        }
        catch (OverflowException oe)
        {
            // OK; result would overflow
        }

        try
        {
            ApcomplexMath.exp(new Apcomplex(new Apfloat(2), new Apfloat(2)));
            fail("exp to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testPow()
    {
        Apcomplex a = ApcomplexMath.pow(new Apcomplex(new Apfloat(2, 100)), new Apcomplex(new Apfloat("0.5", 100)));
        assertEquals("2^0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("2^0.5, 100 value", new Apfloat("1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.pow(new Apcomplex(new Apfloat(-2, 100)), new Apcomplex(new Apfloat("0.5", 100)));
        assertEquals("(-2)^0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("(-2)^0.5, 100 value", new Apcomplex("(0,1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.pow(new Apcomplex(new Apfloat(5, 100), new Apfloat(4, 100)), new Apcomplex(new Apfloat(3, 100), new Apfloat(2, 100)));
        assertEquals("(5,4)^(3,2), 100 precision", 99, a.precision(), 1);
        assertEquals("(5,4)^(3,2), 100 value", new Apcomplex("(58.2145009705915246565265271223031301774886834072803476475245473202072182772155725270313930579883159,-35.3234369348583753301237874184698198860952187848530861615372916445617840467298560954959137658317125)"), a, new Apfloat("5e-97"));

        Apfloat r = new Apfloat("1.193EA7AAD030A976A4198D55053B7CB5BE1442D9B7E08DF03D97EEEA5149358CAA9782D20CC698505071F733039A8ED5625C15071EA7BCA1CF37D8F11024C66486D094E21E74D0A547DF6", 150, 16);
        a = ApcomplexMath.pow(new Apcomplex(r, r), new Apcomplex(r, r));
        assertEquals("3, 150 precision", 150, a.precision(), 1);
        assertEquals("3, 150 value", new Apcomplex(new Apfloat("0.26ea28ff7cb530fc94a0b61e891cb2a15e1c72d7e9421ab41a2d0c46f1ba18f0b43409b5440adf17b33a95646b30b96f44634e286d546b2f543415d6d2deaaf79a715fa30e30cb858edb14", Apfloat.DEFAULT, 16),
                                                   new Apfloat("0.aae89b585652b078f623e7f850ebc3cd7c87afeb0a32feb1003d57a82352857778a89519b99e0001e1c276d3ddb09e1c07fc6b8129240a724aea584a7283a5e98bba61f57b20926ffac626", Apfloat.DEFAULT, 16)), a, ApfloatMath.scale(new Apfloat(5, 1, 16), -149));

        a = ApcomplexMath.pow(new Apfloat(2), new Apfloat(0));
        assertEquals("2^0", new Apfloat(1), a);

        a = ApcomplexMath.pow(new Apfloat(2), new Apfloat(1));
        assertEquals("2^1", new Apfloat(2), a);

        a = ApcomplexMath.pow(new Apfloat(-2), new Apfloat(0));
        assertEquals("(-2)^0", new Apfloat(1), a);

        a = ApcomplexMath.pow(new Apfloat(-2), new Apfloat(1));
        assertEquals("(-2)^1", new Apfloat(-2), a);

        a = ApcomplexMath.pow(Apcomplex.I, new Apfloat(0));
        assertEquals("i^0", new Apfloat(1), a);

        a = ApcomplexMath.pow(Apcomplex.I, new Apfloat(1));
        assertEquals("i^1", Apcomplex.I, a);

        a = ApcomplexMath.pow(new Apfloat(0), new Apfloat(2));
        assertEquals("0^2", new Apfloat(0), a);

        a = ApcomplexMath.pow(new Apfloat(1), new Apfloat(2));
        assertEquals("1^2", new Apfloat(1), a);

        a = ApcomplexMath.pow(new Apfloat(0), Apcomplex.I);
        assertEquals("0^i", new Apfloat(0), a);

        a = ApcomplexMath.pow(new Apfloat(1), Apcomplex.I);
        assertEquals("1^i", new Apfloat(1), a);

        try
        {
            ApcomplexMath.pow(new Apfloat("1.2"), new Apfloat("57"));
        }
        catch (LossOfPrecisionException lope)
        {
            // OK
        }

        a = ApcomplexMath.pow(new Apfloat("1.02"), new Apfloat("57"));
        assertEquals("1.02^57, precision", 1, a.precision());

        a = ApcomplexMath.pow(new Apfloat("1.2"), new Apfloat("1.0"));
        assertEquals("1.2^1.0, precision", 2, a.precision());

        a = ApcomplexMath.pow(new Apfloat("1.2"), new Apfloat("1"));
        assertEquals("1.2^1, precision", 1, a.precision());

        a = ApcomplexMath.pow(new Apfloat("1.02"), new Apfloat("1.00"));
        assertEquals("1.02^1.00, precision", 3, a.precision());

        a = ApcomplexMath.pow(new Apfloat("1.02"), new Apfloat("1"));
        assertEquals("1.02^1, precision", 1, a.precision());

        a = ApcomplexMath.pow(new Apfloat("1.2"), new Apfloat("0.01"));
        assertEquals("1.2^0.01, precision", 4, a.precision());

        a = ApcomplexMath.pow(new Apfloat("1.02"), new Apfloat("0.01"));
        assertEquals("1.02^0.01, precision", 5, a.precision());

        try
        {
            ApcomplexMath.pow(new Apcomplex(new Apfloat(0)), new Apcomplex(new Apfloat(0)));
            fail("0^0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be undefined
        }

        try
        {
            ApcomplexMath.pow(new Apcomplex(new Apfloat(100, 100), new Apfloat(100, 100)),
                              new Apcomplex(new Apfloat(5000000000000000000L, 100), new Apfloat(5000000000000000000L, 100)));
            fail("Overflow should have occurred");
        }
        catch (OverflowException oe)
        {
            // OK; result would overflow
        }

        try
        {
            ApcomplexMath.pow(new Apcomplex(new Apfloat(3), new Apfloat(3)),
                              new Apcomplex(new Apfloat(-1), new Apfloat(-1)));
            fail("pow to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testArg()
    {
        Apcomplex a = ApcomplexMath.arg(new Apcomplex("(-2.0000,2.0000)"));
        assertEquals("(-2,2), precision", 5, a.precision());
        assertEquals("(-2,2), value", new Apfloat("2.3562"), a, new Apfloat("0.0005"));
    }

    public static void testAcosh()
    {
        Apcomplex a = ApcomplexMath.acosh(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(2.30550903124347694204183593813343089732908234612766434427244403789502387715767721380519816885689075,0.9368124611557199029125245765756089164871812290143448233044479241680079302681295000053794681278219233)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.acosh(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(2.30550903124347694204183593813343089732908234612766434427244403789502387715767721380519816885689075,-0.9368124611557199029125245765756089164871812290143448233044479241680079302681295000053794681278219233)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.acosh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(2.30550903124347694204183593813343089732908234612766434427244403789502387715767721380519816885689075,2.204780192434073335550118806703893967709988170360760997670496668139808476018079498622655357214295145)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.acosh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(2.30550903124347694204183593813343089732908234612766434427244403789502387715767721380519816885689075,-2.204780192434073335550118806703893967709988170360760997670496668139808476018079498622655357214295145)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.acosh(new Apfloat(2, 10));
        assertEquals("(2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(2,0), 10 value", new Apcomplex("1.316957897"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acosh(new Apfloat(-2, 10));
        assertEquals("(-2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(-2,0), 10 value", new Apcomplex("(1.316957897,3.141592654)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acosh(new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("(0,2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,2), 10 value", new Apcomplex("(1.443635475,1.570796327)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acosh(new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("(0,-2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,-2), 10 value", new Apcomplex("(1.443635475,-1.570796327)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acosh(new Apcomplex(new Apfloat(1, 100)));
        assertEquals("1, 100 value", new Apfloat(0), a);
    }

    public static void testAsinh()
    {
        Apcomplex a = ApcomplexMath.asinh(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(2.299914040879269649955789630663175555365313484764636466611830082402437121311729696004733902877606405,0.9176168533514786557598627486701745415899523820362300027773647609161124445462833451286169894870273957)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asinh(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(2.299914040879269649955789630663175555365313484764636466611830082402437121311729696004733902877606405,-0.9176168533514786557598627486701745415899523820362300027773647609161124445462833451286169894870273957)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asinh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-2.299914040879269649955789630663175555365313484764636466611830082402437121311729696004733902877606405,0.9176168533514786557598627486701745415899523820362300027773647609161124445462833451286169894870273957)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asinh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-2.299914040879269649955789630663175555365313484764636466611830082402437121311729696004733902877606405,-0.9176168533514786557598627486701745415899523820362300027773647609161124445462833451286169894870273957)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asinh(new Apfloat(2, 10));
        assertEquals("(2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(2,0), 10 value", new Apcomplex("1.443635475"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asinh(new Apfloat(-2, 10));
        assertEquals("(-2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(-2,0), 10 value", new Apcomplex("(-1.443635475)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asinh(new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("(0,2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,2), 10 value", new Apcomplex("(1.316957897,1.570796327)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asinh(new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("(0,-2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,-2), 10 value", new Apcomplex("(-1.316957897,-1.570796327)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asinh(new Apcomplex("0"));
        assertEquals("0, 100 value", new Apfloat(0), a);
    }

    public static void testAtanh()
    {
        Apcomplex a = ApcomplexMath.atanh(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(0.1175009073114338884127342577870855161752247622030620101123480342515004695503565955468640257240191129,1.409921049596575522530619384460420782588207051908724814771070766475530084440199227135813201495737847)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atanh(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(0.1175009073114338884127342577870855161752247622030620101123480342515004695503565955468640257240191129,-1.409921049596575522530619384460420782588207051908724814771070766475530084440199227135813201495737847)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atanh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-0.1175009073114338884127342577870855161752247622030620101123480342515004695503565955468640257240191129,1.409921049596575522530619384460420782588207051908724814771070766475530084440199227135813201495737847)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atanh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-0.1175009073114338884127342577870855161752247622030620101123480342515004695503565955468640257240191129,-1.409921049596575522530619384460420782588207051908724814771070766475530084440199227135813201495737847)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atanh(new Apfloat(2, 10));
        assertEquals("(2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(2,0), 10 value", new Apcomplex("(0.549306144,-1.570796327)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atanh(new Apfloat(-2, 10));
        assertEquals("(-2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(-2,0), 10 value", new Apcomplex("(-0.549306144,1.570796327)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atanh(new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("(0,2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,2), 10 value", new Apcomplex("(0,1.107148718)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atanh(new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("(0,-2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,-2), 10 value", new Apcomplex("(0,-1.107148718)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atanh(new Apcomplex("0"));
        assertEquals("0, 100 value", new Apfloat(0), a);

        try
        {
            ApcomplexMath.atanh(new Apcomplex("1"));
            fail("atanh(1) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be infinite
        }

        try
        {
            ApcomplexMath.atanh(new Apcomplex("-1"));
            fail("atanh(-1) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be infinite
        }
    }

    public static void testCosh()
    {
        Apcomplex a = ApcomplexMath.cosh(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(-6.580663040551156432560744126538803616711267345515897773220218329756121215365251384163430874396326777,-7.581552742746544353716345286538426009387527590948852812949363456244614022672964969341075109130625439)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.cosh(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(-6.580663040551156432560744126538803616711267345515897773220218329756121215365251384163430874396326777,7.581552742746544353716345286538426009387527590948852812949363456244614022672964969341075109130625439)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.cosh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-6.580663040551156432560744126538803616711267345515897773220218329756121215365251384163430874396326777,7.581552742746544353716345286538426009387527590948852812949363456244614022672964969341075109130625439)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.cosh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-6.580663040551156432560744126538803616711267345515897773220218329756121215365251384163430874396326777,-7.581552742746544353716345286538426009387527590948852812949363456244614022672964969341075109130625439)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.cosh(new Apcomplex("1.00000"));
        assertEquals("1 value", new Apfloat("1.54308"), a, new Apfloat("5e-5"));
        assertEquals("1 imag value", new Apfloat(0), a.imag());

        a = ApcomplexMath.cosh(new Apcomplex("0"));
        assertEquals("0 value", new Apfloat(1), a);
    }

    public static void testSinh()
    {
        Apcomplex a = ApcomplexMath.sinh(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(-6.548120040911001647766811018835324740820888396888583499736134313039666841835229556393917343956455199,-7.619231720321410208487135736804311796557265472675575619426852074665542955161180340917983240028178743)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.sinh(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(-6.548120040911001647766811018835324740820888396888583499736134313039666841835229556393917343956455199,7.619231720321410208487135736804311796557265472675575619426852074665542955161180340917983240028178743)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.sinh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(6.548120040911001647766811018835324740820888396888583499736134313039666841835229556393917343956455199,-7.619231720321410208487135736804311796557265472675575619426852074665542955161180340917983240028178743)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.sinh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(6.548120040911001647766811018835324740820888396888583499736134313039666841835229556393917343956455199,7.619231720321410208487135736804311796557265472675575619426852074665542955161180340917983240028178743)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.sinh(new Apcomplex("1.0000"));
        assertEquals("1 value", new Apfloat("1.1752"), a, new Apfloat("5e-4"));
        assertEquals("1 imag value", new Apfloat(0), a.imag());

        a = ApcomplexMath.sinh(new Apcomplex("0"));
        assertEquals("0, 100 value", new Apfloat(0), a);
    }

    public static void testTanh()
    {
        Apcomplex a = ApcomplexMath.tanh(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 101, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(1.0007095360672329393295854724041727462153209051467602180192607299042866403616169551650374279065226404,0.0049082580674960602590787869299327668433742155793555069748955113426747384320810439493273599689927112)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tanh(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 101, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(1.0007095360672329393295854724041727462153209051467602180192607299042866403616169551650374279065226404,-0.0049082580674960602590787869299327668433742155793555069748955113426747384320810439493273599689927112)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tanh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 101, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-1.0007095360672329393295854724041727462153209051467602180192607299042866403616169551650374279065226404,0.0049082580674960602590787869299327668433742155793555069748955113426747384320810439493273599689927112)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tanh(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 101, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-1.0007095360672329393295854724041727462153209051467602180192607299042866403616169551650374279065226404,-0.0049082580674960602590787869299327668433742155793555069748955113426747384320810439493273599689927112)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tanh(new Apcomplex("1.0000"));
        assertEquals("1 value", new Apfloat("0.76159"), a, new Apfloat("5e-5"));
        assertEquals("1 imag value", new Apfloat(0), a.imag());

        a = ApcomplexMath.tanh(new Apcomplex(new Apfloat(5000000, 100), new Apfloat("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068")));
        assertEquals("(5000000,pi), 100 precision", 4343037, a.precision(), 1);
        assertEquals("(5000000,pi), 100 value", new Apfloat(1).subtract(new Apfloat("3.03387356179746871517892756452203249280217116298171287391802587502345779082540217305405121824e-4342945")), a, new Apfloat("5e-4343037"));

        a = ApcomplexMath.tanh(new Apcomplex("0"));
        assertEquals("0, 100 value", new Apfloat(0), a);
    }

    public static void testAcos()
    {
        Apcomplex a = ApcomplexMath.acos(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(0.9368124611557199029125245765756089164871812290143448233044479241680079302681295000053794681278219233,-2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.acos(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(0.9368124611557199029125245765756089164871812290143448233044479241680079302681295000053794681278219233,2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        // NOTE: result precision is reduced
        a = ApcomplexMath.acos(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 2);
        assertEquals("(-3,4), 100 value", new Apcomplex("(2.204780192434073335550118806703893967709988170360760997670496668139808476018079498622655357214295145,-2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        // NOTE: result precision is reduced
        a = ApcomplexMath.acos(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 2);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(2.204780192434073335550118806703893967709988170360760997670496668139808476018079498622655357214295145,2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.acos(new Apfloat(2, 10));
        assertEquals("(2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(2,0), 10 value", new Apcomplex("(0,1.316957897)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acos(new Apfloat(-2, 10));
        assertEquals("(-2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(-2,0), 10 value", new Apcomplex("(3.141592654,-1.316957897)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acos(new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("(0,2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,2), 10 value", new Apcomplex("(1.570796327,-1.443635475)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acos(new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("(0,-2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,-2), 10 value", new Apcomplex("(1.570796327,1.443635475)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.acos(new Apcomplex(new Apfloat(1, 100)));
        assertEquals("1, 100 value", new Apfloat(0), a);

        a = ApcomplexMath.acos(new Apcomplex(new Apfloat("0.5", 30)));
        assertEquals("0.5, 30 value", new Apfloat("1.04719755119659774615421446109"), a, new Apfloat("5e-28"));

        try
        {
            ApcomplexMath.acos(new Apcomplex("0"));
            fail("acos(0) accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; result would have infinite precision
        }
    }

    public static void testAsin()
    {
        Apcomplex a = ApcomplexMath.asin(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(0.6339838656391767163187971150641425256114034706732080871830243719859002728749749993086379445432366107,2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asin(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(0.6339838656391767163187971150641425256114034706732080871830243719859002728749749993086379445432366107,-2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asin(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-0.6339838656391767163187971150641425256114034706732080871830243719859002728749749993086379445432366107,2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asin(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-0.6339838656391767163187971150641425256114034706732080871830243719859002728749749993086379445432366107,-2.305509031243476942041835938133430897329082346127664344272444037895023877157677213805198168856890747)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.asin(new Apfloat(2, 10));
        assertEquals("(2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(2,0), 10 value", new Apcomplex("(1.570796327,-1.316957897)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asin(new Apfloat(-2, 10));
        assertEquals("(-2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(-2,0), 10 value", new Apcomplex("(-1.570796327,1.316957897)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asin(new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("(0,2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,2), 10 value", new Apcomplex("(0,1.443635475)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asin(new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("(0,-2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,-2), 10 value", new Apcomplex("(0,-1.443635475)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.asin(new Apcomplex("0"));
        assertEquals("0, 100 value", new Apfloat(0), a);

        a = ApcomplexMath.asin(new Apcomplex(new Apfloat("0.5", 30)));
        assertEquals("0.5, 30 value", new Apfloat("0.523598775598298873077107230547"), a, new Apfloat("5e-29"));
    }

    public static void testAtan()
    {
        Apcomplex a = ApcomplexMath.atan(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(1.448306995231464542145280451034113536641512650496960876923784338820230643349283451026750333836707538,0.1589971916799991743647610360070187815733054742350614709569622676518259973409283367912158396025096925)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atan(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(1.448306995231464542145280451034113536641512650496960876923784338820230643349283451026750333836707538,-0.1589971916799991743647610360070187815733054742350614709569622676518259973409283367912158396025096925)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atan(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-1.448306995231464542145280451034113536641512650496960876923784338820230643349283451026750333836707538,0.1589971916799991743647610360070187815733054742350614709569622676518259973409283367912158396025096925)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atan(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-1.448306995231464542145280451034113536641512650496960876923784338820230643349283451026750333836707538,-0.1589971916799991743647610360070187815733054742350614709569622676518259973409283367912158396025096925)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.atan(new Apfloat(2, 10));
        assertEquals("(2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(2,0), 10 value", new Apcomplex("1.107148718"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atan(new Apfloat(-2, 10));
        assertEquals("(-2,0), 10 precision", 10, a.precision(), 1);
        assertEquals("(-2,0), 10 value", new Apcomplex("-1.107148718"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atan(new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("(0,2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,2), 10 value", new Apcomplex("(1.570796327,0.549306144)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atan(new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("(0,-2), 10 precision", 10, a.precision(), 1);
        assertEquals("(0,-2), 10 value", new Apcomplex("(-1.570796327,-0.549306144)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.atan(new Apcomplex("0"));
        assertEquals("0, 100 value", new Apfloat(0), a);

        a = ApcomplexMath.atan(new Apcomplex(new Apfloat("0.5", 30)));
        assertEquals("0.5, 30 value", new Apfloat("0.463647609000806116214256231461"), a, new Apfloat("5e-29"));
    }

    public static void testCos()
    {
        Apcomplex a = ApcomplexMath.cos(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(-27.03494560307422464769480266827091348467753695567661661019265514673434246483988229429946831870519301,-3.851153334811777536563337123053124569704160846091637003157728595256494186490481089994453362578315815)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.cos(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(-27.03494560307422464769480266827091348467753695567661661019265514673434246483988229429946831870519301,3.851153334811777536563337123053124569704160846091637003157728595256494186490481089994453362578315815)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.cos(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-27.03494560307422464769480266827091348467753695567661661019265514673434246483988229429946831870519301,3.851153334811777536563337123053124569704160846091637003157728595256494186490481089994453362578315815)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.cos(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-27.03494560307422464769480266827091348467753695567661661019265514673434246483988229429946831870519301,-3.851153334811777536563337123053124569704160846091637003157728595256494186490481089994453362578315815)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.cos(new Apcomplex("1.0000"));
        assertEquals("1 value", new Apfloat("0.5403"), a, new Apfloat("5e-5"));
        assertEquals("1 imag value", new Apfloat(0), a.imag());

        a = ApcomplexMath.cos(new Apcomplex(new Apfloat(0)));
        assertEquals("0, 100 value", new Apfloat(1), a);

        try
        {
            ApcomplexMath.cos(new Apcomplex(new Apfloat(1000, 3), new Apfloat("1.5")));
            fail("cos(1000 prec 3) accepted");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK; loss of precision
        }
    }

    public static void testSin()
    {
        Apcomplex a = ApcomplexMath.sin(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(3.853738037919377321617528940463730667068274946989034956763346803317838585207899050385464301460315524,-27.01681325800393448809754375499215226336386568976518470594798897425063415478434990691671779691472675)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.sin(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(3.853738037919377321617528940463730667068274946989034956763346803317838585207899050385464301460315524,27.01681325800393448809754375499215226336386568976518470594798897425063415478434990691671779691472675)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.sin(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(-3.853738037919377321617528940463730667068274946989034956763346803317838585207899050385464301460315524,-27.01681325800393448809754375499215226336386568976518470594798897425063415478434990691671779691472675)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.sin(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(-3.853738037919377321617528940463730667068274946989034956763346803317838585207899050385464301460315524,27.01681325800393448809754375499215226336386568976518470594798897425063415478434990691671779691472675)"), a, new Apfloat("5e-97"));

        a = ApcomplexMath.sin(new Apcomplex("1.0000"));
        assertEquals("1 value", new Apfloat("0.84147"), a, new Apfloat("5e-5"));
        assertEquals("1 imag value", new Apfloat(0), a.imag());

        a = ApcomplexMath.sin(new Apcomplex(new Apfloat(0)));
        assertEquals("0, 100 value", new Apfloat(0), a);

        try
        {
            ApcomplexMath.sin(new Apcomplex(new Apfloat(1000, 3), new Apfloat("1.5")));
            fail("sin(1000 prec 3) accepted");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK; loss of precision
        }
    }

    public static void testTan()
    {
        Apcomplex a = ApcomplexMath.tan(new Apcomplex(new Apfloat(3, 100), new Apfloat(4, 100)));
        assertEquals("(3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,4), 100 value", new Apcomplex("(-0.0001873462046294784262242556377282181042124242427296606263580802232052224832174311687842725259181727521,0.9993559873814731413916496303201330615648885028135384928319757364498179348866065958722698773248799920)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tan(new Apcomplex(new Apfloat(3, 100), new Apfloat(-4, 100)));
        assertEquals("(3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(3,-4), 100 value", new Apcomplex("(-0.0001873462046294784262242556377282181042124242427296606263580802232052224832174311687842725259181727521,-0.9993559873814731413916496303201330615648885028135384928319757364498179348866065958722698773248799920)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tan(new Apcomplex(new Apfloat(-3, 100), new Apfloat(4, 100)));
        assertEquals("(-3,4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,4), 100 value", new Apcomplex("(0.0001873462046294784262242556377282181042124242427296606263580802232052224832174311687842725259181727521,0.9993559873814731413916496303201330615648885028135384928319757364498179348866065958722698773248799920)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tan(new Apcomplex(new Apfloat(-3, 100), new Apfloat(-4, 100)));
        assertEquals("(-3,-4), 100 precision", 100, a.precision(), 1);
        assertEquals("(-3,-4), 100 value", new Apcomplex("(0.0001873462046294784262242556377282181042124242427296606263580802232052224832174311687842725259181727521,-0.9993559873814731413916496303201330615648885028135384928319757364498179348866065958722698773248799920)"), a, new Apfloat("5e-98"));

        a = ApcomplexMath.tan(new Apfloat("3.1415926535897932384626433832795", 24).divide(new Apint(180)));
        assertEquals("1 degree value", new Apfloat("0.0174550649282175857651289"), a, new Apfloat("5e-25"));
        assertEquals("1 degree imag value", new Apfloat(0), a.imag());

        a = ApcomplexMath.tan(new Apcomplex(new Apfloat(0)));
        assertEquals("0, 100 value", new Apfloat(0), a);

        try
        {
            ApcomplexMath.tan(new Apcomplex(new Apfloat(1000, 3), new Apfloat("1.5")));
            fail("tan(1000 prec 3) accepted");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK; loss of precision
        }
    }

    public static void testW()
    {
        Apcomplex a = new Apcomplex(new Apfloat(1, 45, 16), new Apfloat(2, 45, 16));
        Apcomplex w = ApcomplexMath.w(a, -1);
        assertEquals("value", new Apcomplex(new Apfloat("-0.731b614b3b95524effd3f81d9be7b8c8e1d21e69264d", 45, 16), new Apfloat("-3.7a03f382c1f5dd2df77921934d2d67941431ed093840", 45, 16)), w, new Apfloat("5", 1, 16).scale(-44));
        assertEquals("radix", 16, w.radix());
        assertEquals("precision", 45, w.precision());
    }

    public static void testProduct()
    {
        Apcomplex a = ApcomplexMath.product(new Apcomplex("(1,1)"), new Apcomplex("(1,1)"));
        assertEquals("1-1 precision", 1, a.precision());
        assertEquals("1-1 value", new Apcomplex("(0,2)"), a);

        a = ApcomplexMath.product(new Apcomplex(Apfloat.ZERO, new Apfloat(2)), new Apcomplex(new Apfloat(3, 5), new Apfloat(3, 5)), new Apcomplex(new Apfloat(4, 8), new Apfloat(4, 8)));
        assertEquals("MAX-5-8 precision", 5, a.precision());
        assertEquals("MAX-5-8 value", new Apcomplex("-48"), a);

        a = ApcomplexMath.product(Apcomplex.ZERO, new Apcomplex("12345"));
        assertEquals("0 precision", Apcomplex.INFINITE, a.precision());
        assertEquals("0 value", new Apcomplex("0"), a);

        a = ApcomplexMath.product(new Apint(0, 12));
        assertEquals("0 radix", 12, a.radix());

        a = ApcomplexMath.product(new Apcomplex("-1"),
                                  new Apcomplex("(0,2.50000000000000000000000000000000000000000000000000000000000000000000001)"),
                                  new Apcomplex("(0,2.0000000000000000000000000000000000001)"),
                                  new Apcomplex(new Apfloat(2)));
        assertEquals("Many precision", 1, a.precision());
        assertEquals("Many value", new Apcomplex("10"), a);

        Apcomplex[] x = new Apcomplex[] { new Apcomplex("1000000000000"), new Apcomplex("1") };
        ApcomplexMath.product(x);
        assertEquals("Array product 1 [0]", new Apcomplex("1000000000000"), x[0]);
        assertEquals("Array product 1 [1]", new Apcomplex("1"), x[1]);

        x = new Apcomplex[] { new Apcomplex("1"), new Apcomplex("1000000000000") };
        ApcomplexMath.product(x);
        assertEquals("Array product 2 [0]", new Apcomplex("1"), x[0]);
        assertEquals("Array product 2 [1]", new Apcomplex("1000000000000"), x[1]);

        assertEquals("Empty product", new Apcomplex("1"), ApcomplexMath.product());

        Apcomplex[] numbers = new Apcomplex[100000];    // Length should be divisible by 4 here
        for (int i = 0; i < numbers.length; i++)
        {
            numbers[i] = new Apcomplex(Apfloat.ZERO, new Apfloat(i + 1));
        }
        Apcomplex factorial = ApintMath.factorial(numbers.length);
        assertEquals("Factorial", factorial, ApcomplexMath.product(numbers));

        numbers[0] = new Apcomplex(Apfloat.ZERO, factorial.real());
        assertEquals("Factorial squared", factorial.multiply(factorial), ApcomplexMath.product(numbers));
    }

    public static void testSum()
    {
        Apcomplex a = ApcomplexMath.sum(new Apcomplex(Apfloat.ZERO, new Apfloat(12345000, 5)), new Apcomplex(Apfloat.ZERO, new Apfloat(12345, 5)));
        assertEquals("5-2 precision", 5, a.precision());
        assertEquals("5-2 value", new Apcomplex("(0,12357000)"), a);

        a = ApcomplexMath.sum(new Apcomplex(new Apfloat(12345678, 10)), new Apcomplex(new Apfloat(12345, 5)));
        assertEquals("8-5 precision", 8, a.precision());
        assertEquals("8-5 value", new Apcomplex("12358023"), a);

        a = ApcomplexMath.sum(new Apcomplex("0"), new Apcomplex("12345"));
        assertEquals("0-0 precision", 5, a.precision());
        assertEquals("0-0 value", new Apcomplex("12345"), a);

        a = ApcomplexMath.sum(new Apcomplex("2"));
        assertEquals("2 precision", 1, a.precision());
        assertEquals("2 value", new Apcomplex("2"), a);

        a = ApcomplexMath.sum(new Apcomplex("(0,1)"),
                              new Apcomplex("(0,3333333.33)"),
                              new Apcomplex("(0,2.2)"),
                              new Apcomplex("(0,444444444400000000000000000000000000)"),
                              new Apcomplex("(0,6666.6)"),
                              new Apcomplex("(0,555)"),
                              new Apcomplex(Apfloat.ZERO, new Apfloat(7)));
        assertEquals("Many precision", 36, a.precision());
        assertEquals("Many value", new Apcomplex("(0,444444444400000000000000000003340564)"), a);

        Apcomplex[] x = new Apcomplex[] { new Apcomplex("1000000000000"), new Apcomplex("1") };
        ApcomplexMath.sum(x);
        assertEquals("Array sum 1 [0]", new Apcomplex("1000000000000"), x[0]);
        assertEquals("Array sum 1 [1]", new Apcomplex("1"), x[1]);

        x = new Apcomplex[] { new Apcomplex("1"), new Apcomplex("1000000000000") };
        ApcomplexMath.sum(x);
        assertEquals("Array sum 2 [0]", new Apcomplex("1"), x[0]);
        assertEquals("Array sum 2 [1]", new Apcomplex("1000000000000"), x[1]);

        assertEquals("Empty sum", new Apcomplex("0"), ApcomplexMath.sum());
    }

    public static void testGamma()
    {
        Apcomplex a = ApcomplexMath.gamma(new Apcomplex("0.50000000"));
        assertEquals("0.5 precision", 8, a.precision());
        assertEquals("0.5 value", new Apfloat("1.7724539"), a, new Apfloat("5e-7"));
        a = ApcomplexMath.gamma(new Apcomplex("-0.50000000"));
        assertEquals("-0.5 precision", 8, a.precision());
        assertEquals("-0.5 value", new Apfloat("-3.5449077"), a, new Apfloat("5e-7"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(1, 8, 5)));
        assertEquals("1 precision", 8, a.precision());
        assertEquals("1 radix", 5, a.radix());
        assertEquals("1 value", new Apfloat("1"), a);
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(2, 100, 15)));
        assertEquals("2 precision", 100, a.precision());
        assertEquals("2 radix", 15, a.radix());
        assertEquals("2 value", new Apfloat("1"), a);
        a = ApcomplexMath.gamma(new Apcomplex("3.0000"));
        assertEquals("3 precision", 5, a.precision());
        assertEquals("3 value", new Apfloat("2"), a);
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("1.6", 50, 12)));
        assertEquals("1.5 precision", 50, a.precision());
        assertEquals("1.5 radix", 12, a.radix());
        assertEquals("1.5 value", new Apfloat("0.a77497505445a57663a7a6a27293557aa7636b52055b106267", 50, 12), a, new Apfloat("5e-50", 1, 12));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("1.6", 50, 12), new Apfloat("1.6", 50, 12)));
        assertEquals("(1.5,1.5) precision", 50, a.precision());
        assertEquals("(1.5,1.5) radix", 12, a.radix());
        assertEquals("(1.5,1.5) value", new Apcomplex(new Apfloat("0.421432a040a1238266b59319481b82250a972699b1218016a4", 50, 12), new Apfloat("0.184386517914307662b897756b0100538485b04495282b25ba", 50, 12)), a, new Apfloat("5e-50", 1, 12));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("1.6", 50, 12), new Apfloat("-1.6", 50, 12)));
        assertEquals("(1.5,-1.5) precision", 50, a.precision());
        assertEquals("(1.5,-1.5) radix", 12, a.radix());
        assertEquals("(1.5,-1.5) value", new Apcomplex(new Apfloat("0.421432a040a1238266b59319481b82250a972699b1218016a4", 50, 12), new Apfloat("-0.184386517914307662b897756b0100538485b04495282b25ba", 50, 12)), a, new Apfloat("5e-50", 1, 12));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("10.1", 50), new Apfloat(1, 50)));
        assertEquals("10.1+I precision", 49, a.precision());
        assertEquals("10.1+I radix", 10, a.radix());
        assertEquals("10.1+I value", new Apcomplex("(-275889.24637304345640288294183505994521208628420155,332093.85747733234553448206237389863106703170190430)"), a, new Apfloat("5e-44"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("100.1", 50), new Apfloat(1, 50)));
        assertEquals("100.1+I precision", 47, a.precision(), 1); //FIXME
        assertEquals("100.1+I radix", 10, a.radix());
        assertEquals("100.1+I value", new Apcomplex("(-1.6325249252175791061043773321806465941281038201e155,-1.46196463909803799277895455102943868342521844260e156)"), a, new Apfloat("5e109"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(1, 200), new Apfloat("1000000000000.1", 200)));
        assertEquals("1000000000000.1+I precision", 186, a.precision());
        assertEquals("1000000000000.1+I radix", 10, a.radix());
        assertEquals("1000000000000.1+I value", new Apcomplex("(2.2799397381057012808806414716554222166361918093242073118331397578121585980690443002813637584289628198999964037945275096726070876981676095000509004082444136731667291762706895607184590520e-682188176916,2.56143734228029034694359025317416641643312626363528304218252981145494697876711767159132470544194941716039395729054771162631751713400996765379046059018461097663341902573535496875851156846e-682188176915)"), a, new Apfloat("5e-682188177101"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(1, 40), new Apfloat("1000000000000.1", 40)));
        assertEquals("1000000000000.1+I precision", 26, a.precision());
        assertEquals("1000000000000.1+I radix", 10, a.radix());
        assertEquals("1000000000000.1+I value", new Apcomplex("(2.2799397381057012808806415e-682188176916,2.56143734228029034694359025e-682188176915)"), a, new Apfloat("5e-682188176941"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(1, 133, 2), new Apfloat("1.110100011010100101001010001000000000000000110011001100110011001100110011001100110011001100110011001100110011001100110011001100110011e39", 133, 2)));
        assertEquals("1000000000000.1+I radix 2 precision", 87, a.precision());
        assertEquals("1000000000000.1+I radix 2 radix", 2, a.radix());
        assertEquals("1000000000000.1+I radix 2 value", new Apcomplex(new Apfloat("1.111010101000100100110111110011011000101001000110001000101011000111010000100110001001e-2266180070897", Apfloat.DEFAULT, 2), new Apfloat("1.0101100001110000001011000010100111110010000010101111100000100011110010100011010001110010e-2266180070893", Apfloat.DEFAULT, 2)), a, new Apfloat("1e-2266180070981", 1, 2));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(1, 25, 36), new Apfloat("cre66i9s.3lllllllllllllllm", 25, 36)));
        assertEquals("100000000000.1+I radix 36 precision", 17, a.precision());
        assertEquals("100000000000.1+I radix 36 radix", 36, a.radix());
        assertEquals("100000000000.1+I radix 36 value", ApcomplexMath.scale(new Apcomplex(new Apfloat("0.4w3ncj1ol4pgrcbkl", Apfloat.DEFAULT, 36),new Apfloat("1.iyfv83hi4ul6s26oy", Apfloat.DEFAULT, 36)), -438339061062L), a, ApfloatMath.scale(new Apfloat("h", 1, 36), -438339061078L));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(100, 30), new Apfloat(374, 30)));
        assertEquals("(100,374) precision", 30, a.precision());
        assertEquals("(100,374) radix", 10, a.radix());
        assertEquals("(100,374) value", new Apcomplex("(47.4294943677064514689542753377,-32.7488916473624576880974867017)"), a, new Apfloat("5e-28"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat(1000000000, 30), new Apfloat(14913639641L, 30)));
        assertEquals("(1000000000,14913639641) precision", 30, a.precision());
        assertEquals("(1000000000,14913639641) radix", 10, a.radix());
        assertEquals("(1000000000,14913639641) value", new Apcomplex("(-2.99966240181596582389952917253,-0.39936803730444026987241513690)"), a, new Apfloat("5e-29"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("4825490186121.8", 30), new Apfloat(99000000000000L, 30)));
        assertEquals("(4825490186121.8,99000000000000) precision", 30, a.precision());
        assertEquals("(4825490186121.8,99000000000000) radix", 10, a.radix());
        assertEquals("(4825490186121.8,99000000000000) value", new Apcomplex("(-6.62771164327451839566254131658,5.32978019529137704595692626876)"), a, new Apfloat("5e-29"));
        try
        {
            ApcomplexMath.gamma(new Apcomplex("0"));
            fail("Gamma of zero");
        }
        catch (ArithmeticException ae)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex("-1"));
            fail("Gamma of -1");
        }
        catch (ArithmeticException ae)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex("(1e2,1)"));
            fail("Gamma loss of precision");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex("(0,1e2)"));
            fail("Gamma loss of precision");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex("1e100"));
            fail("Gamma overflow");
        }
        catch (ApfloatRuntimeException are)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex(new Apint(1), new Apfloat("4.5", Apfloat.INFINITE)));
            fail("Gamma infinite expansion");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK
        }
    }

    public static void testGammaIncomplete()
    {
        Apcomplex a = ApcomplexMath.gamma(new Apfloat(-4, 10), new Apfloat(-2, 10));
        assertEquals("-4,-2 precision", 10, a.precision());
        assertEquals("-4,-2 value", new Apcomplex("(0.2169049075,-0.1308996939)"), a, new Apfloat("5e-10"));
        a = ApcomplexMath.gamma(new Apfloat(-1, 10), new Apfloat(-2, 10));
        assertEquals("-1,-2 precision", 10, a.precision());
        assertEquals("-1,-2 value", new Apcomplex("(1.259706307,3.141592654)"), a, new Apfloat("5e-9"));
        a = ApcomplexMath.gamma(new Apfloat(0, 10), new Apfloat(-1, 10));
        assertEquals("0,-1 precision", 10, a.precision());
        assertEquals("0,-1 value", new Apcomplex("(-1.895117816,-3.141592654)"), a, new Apfloat("5e-9"));
        a = ApcomplexMath.gamma(new Apfloat(-4, 10), new Apcomplex(Apfloat.ZERO, new Apfloat(-2, 10)));
        assertEquals("-4,-2i precision", 10, a.precision());
        assertEquals("-4,-2i value", new Apcomplex("(-0.01031978444,0.00762842656)"), a, new Apfloat("5e-11"));
        a = ApcomplexMath.gamma(new Apfloat(-4, 10), new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("-4,2i precision", 10, a.precision());
        assertEquals("-4,2i value", new Apcomplex("(-0.01031978444,-0.00762842656)"), a, new Apfloat("5e-11"));
        a = ApcomplexMath.gamma(new Apfloat(-4, 10), new Apcomplex(Apfloat.ZERO, new Apfloat(-200, 10)));
        assertEquals("-4,-200i precision", 8, a.precision());
        assertEquals("-4,-200i value", new Apcomplex("(2.765031767e-12,1.453166391e-12)"), a, new Apfloat("5e-21"));
        a = ApcomplexMath.gamma(new Apcomplex(Apfloat.ZERO, new Apfloat(-4, 10)), new Apcomplex(Apfloat.ZERO, new Apfloat(2, 10)));
        assertEquals("-4i,2i precision", 10, a.precision());
        assertEquals("-4i,2i value", new Apcomplex("(89.37681331,-0.32076166)"), a, new Apfloat("5e-8"));

        a = ApcomplexMath.gamma(new Apcomplex("(1.5000000,1.0000000)"), new Apcomplex("(-1.0000000,1.0000000e-2)"));
        assertEquals("1.5+i,-1+0.01i precision", 8, a.precision());
        assertEquals("1.5+i,-1+0.01i value", new Apcomplex("(0.59778962,0.13066077)"), a, new Apfloat("5e-8"));
        a = ApcomplexMath.gamma(new Apcomplex("(1.5000000,1.0000000)"), new Apcomplex("(-1.0000000,-1.0000000e-2)"));
        assertEquals("1.5+i,-1-0.01i precision", 8, a.precision());
        assertEquals("1.5+i,-1-0.01i value", new Apcomplex("(-10.201515,-22.643499)"), a, new Apfloat("5e-6"));

        a = ApcomplexMath.gamma(new Apcomplex("(-1.0000000e-2,4.0000000e2)"), new Apcomplex("(1.0000000e-2,1.0000000e2)"));
        assertEquals("-1e-2+4e2i,1e-2+1e2i precision", 5, a.precision());
        assertEquals("-1e-2+4e2i,1e-2+1e2i value", new Apcomplex("(-1.0542844e-274,-1.2073566e-274)"), a, new Apfloat("5e-278"));

        a = ApcomplexMath.gamma(new Apcomplex("(-1.1538262523991663e-1,5.373868295802127e-5)"), new Apcomplex("(-1.773822473482807e1,1.1375203210348511e-1)"));
        assertEquals("-1.1538262523991663e-1+5.373868295802127e-5i,-1.773822473482807e1+1.1375203210348511e-1i precision", 15, a.precision());
        assertEquals("-1.1538262523991663e-1+5.373868295802127e-5i,-1.773822473482807e1+1.1375203210348511e-1i value", new Apcomplex("(-1955553.446208209,989495.924550850)"), a, new Apfloat("5e-9"));

        a = ApcomplexMath.gamma(new Apcomplex("(-8.221677442966661e-11,-8.40160119100972e-11)"), new Apcomplex("(1.2025846683194057e-3,9.851413511989293e-9)"));
        assertEquals("-8.221677442966661e-11-8.40160119100972e-11i,1.2025846683194057e-3+9.851413511989293e-9i precision", 6, a.precision());
        assertEquals("-8.221677442966661e-11-8.40160119100972e-11i,1.2025846683194057e-3+9.851413511989293e-9i value", new Apcomplex("(6.14726871,-8.18e-6)"), a, new Apfloat("5e-5"));

        a = ApcomplexMath.gamma(new Apcomplex("(5.9140452392587956e-5,8.5750776148049e8)"), new Apcomplex("(4.673198134242025,-1.3140870009190432e-7)"));
        assertEquals("5.9140452392587956e-5+8.5750776148049e8i,4.673198134242025-1.3140870009190432e-7i precision", 5, a.precision());
        assertEquals("5.9140452392587956e-5+8.5750776148049e8i,4.673198134242025-1.3140870009190432e-7i value", new Apcomplex("(-0.3208810504,0.0377036132)"), a, new Apfloat("5e-5"));

        a = ApcomplexMath.gamma(new Apcomplex("(-1.078432904072464e-1,2.2318918029265195e-9)"), new Apcomplex("(-1.318051987307835,2.2206695185439e9)"));
        assertEquals("-1.078432904072464e-1+2.2318918029265195e-9i,-1.318051987307835+2.2206695185439e9i precision", 5, a.precision());
        assertEquals("-1.078432904072464e-1+2.2318918029265195e-9i,-1.318051987307835+2.2206695185439e9i value", new Apcomplex("(1.155363210e-10,-1.180636439e-10)"), a, new Apfloat("5e-15"));

        a = ApcomplexMath.gamma(new Apcomplex("(-9.608072523703581e1,-3.7643456893059174e-1)"), new Apcomplex("(-1.3814972507571912e2,-3.4538159572099236e-1)"));
        assertEquals("-9.608072523703581e1-3.7643456893059174e-1,-1.3814972507571912e2-3.4538159572099236e-1i precision", 14, a.precision());
        assertEquals("-9.608072523703581e1-3.7643456893059174e-1,-1.3814972507571912e2-3.4538159572099236e-1i value", new Apcomplex("(-1.3024207045775e-149,1.8066865252068e-148)"), a, new Apfloat("5e-161"));

        a = ApcomplexMath.gamma(new Apcomplex("(4.929287290773249e-3,-4.681295287406053e1)"), new Apcomplex("(2.7955773531635903e-6,6.918020068517608e9)"));
        assertEquals("4.929287290773249e-3-4.681295287406053e1i,2.7955773531635903e-6+6.918020068517608e9i precision", 8, a.precision());
        assertEquals("4.929287290773249e-3-4.681295287406053e1i,2.7955773531635903e-6+6.918020068517608e9i value", new Apcomplex("(-9.62645560037940e21,-1.006027591105011e22)"), a, new Apfloat("5e14"));

//        a = ApcomplexMath.gamma(new Apcomplex("(-154,-8.66)").precision(10), new Apcomplex("(-48700,-0.0274)").precision(10));
//        assertEquals("-154-8.66i,-48700-0.0274i precision 10", 6, a.precision());
//        assertEquals("-154-8.66i,-48700-0.0274i value 10", new Apcomplex("(-3.993583e20411,-4.146069e20411)"), a, new Apfloat("5e20406"));
//        a = ApcomplexMath.gamma(new Apcomplex("(-154,-8.66)").precision(50), new Apcomplex("(-48700,-0.0274)").precision(50));
//        assertEquals("-154-8.66i,-48700-0.0274i precision 50", 46, a.precision());
//        assertEquals("-154-8.66i,-48700-0.0274i value 50", new Apcomplex("(-3.993583401859555074273878866836005256257679217e20411,-4.146069249480328790411648959625162586421100204e20411)"), a, new Apfloat("5e20366"));
//        a = ApcomplexMath.gamma(new Apcomplex("(-154,-8.66)").precision(100), new Apcomplex("(-48700,-0.0274)").precision(100));
//        assertEquals("-154-8.66i,-48700-0.0274i precision 100", 96, a.precision());
//        assertEquals("-154-8.66i,-48700-0.0274i value 100", new Apcomplex("(-3.99358340185955507427387886683600525625767921706238693274185708531557493280695373333096470482086e20411,-4.14606924948032879041164895962516258642110020393956205308840983212659862153878019935909589513889e20411)"), a, new Apfloat("5e20316"));

        a = ApcomplexMath.gamma(new Apcomplex("(-1.539926526059491e2,-8.659643233600654)").precision(10), new Apcomplex("(3.651741272548377e1,3.651741272548377e1)").precision(10));
        assertEquals("-1.539926526059491e2-8.659643233600654i,3.651741272548377e1+3.651741272548377e1i precision 10", 7, a.precision());
        assertEquals("-1.539926526059491e2-8.659643233600654i,3.651741272548377e1+3.651741272548377e1i value 10", new Apcomplex("(-1.000387473e-279,2.19340186e-280)"), a, new Apfloat("5e-288"));
        a = ApcomplexMath.gamma(new Apcomplex("(-1.539926526059491e2,-8.659643233600654)").precision(100), new Apcomplex("(3.651741272548377e1,3.651741272548377e1)").precision(100));
        assertEquals("-1.539926526059491e2-8.659643233600654i,3.651741272548377e1+3.651741272548377e1i precision 100", 97, a.precision());
//        assertEquals("-1.539926526059491e2-8.659643233600654i,3.651741272548377e1+3.651741272548377e1i value 100", new Apcomplex("(-1.000387472881261786364648259266083073101695881441237876788775418264481632743088708557828811438171176e-279,2.19340185670881734177337416745907040764484275856218117734431994846665774924244282427508784873315962e-280)"), a, new Apfloat("5e-378"));
        a = ApcomplexMath.gamma(new Apcomplex("(-1.539926526059491e2,-8.659643233600654)").precision(150), new Apcomplex("(3.651741272548377e1,3.651741272548377e1)").precision(150));
        assertEquals("-1.539926526059491e2-8.659643233600654i,3.651741272548377e1+3.651741272548377e1i precision 150", 147, a.precision());
//        assertEquals("-1.539926526059491e2-8.659643233600654i,3.651741272548377e1+3.651741272548377e1i value 150", new Apcomplex("(-1.00038747288126178636464825926608307310169588144123787678877541826448163274308870855782881143817117680690207698118346329877280924878328777673631757823e-279,2.1934018567088173417733741674590704076448427585621811773443199484666577492424428242750878487331596225808712634666511287059353332666354112564113593836e-280)"), a, new Apfloat("5e-428"));

//        a = ApcomplexMath.gamma(new Apcomplex("(-1.539926526059491e2,3.651741272548377e1)"), new Apcomplex("(-1.539926526059491e2,2.053525026457146)"));
//        assertEquals("-1.539926526059491e2+3.651741272548377e1i,-1.539926526059491e2+2.053525026457146i precision", 14, a.precision());
//        assertEquals("-1.539926526059491e2+3.651741272548377e1i,-1.539926526059491e2+2.053525026457146i value", new Apcomplex("(1.839734790895e-319,1.338102215869e-319)"), a, new Apfloat("5e-330"));

        a = ApcomplexMath.gamma(new Apcomplex("(3.651741272548377e1,6.493816315762113e2)"), new Apcomplex("(-1.539926526059491e2, 6.493816315762113e2)"));
        assertEquals("3.651741272548377e1+6.493816315762113e2i,-1.539926526059491e2+6.493816315762113e2i precision 16", 13, a.precision());
        assertEquals("3.651741272548377e1+6.493816315762113e2i,-1.539926526059491e2+6.493816315762113e2i value 16", new Apcomplex("(2.915334707954e-342,-1.655111718059e-341)"), a, new Apfloat("5e-353"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("3.651741272548377e1", 100), new Apfloat("6.493816315762113e2", 100)), new Apcomplex(new Apfloat("-1.539926526059491e2", 100), new Apfloat("6.493816315762113e2", 100)));
        assertEquals("3.651741272548377e1+6.493816315762113e2i,-1.539926526059491e2+6.493816315762113e2i precision 100", 97, a.precision());
        assertEquals("3.651741272548377e1+6.493816315762113e2i,-1.539926526059491e2+6.493816315762113e2i value 100", new Apcomplex("(2.91533470795462694774511982618601485525980988048575683897915057656824264524933126474133998657486524e-342,-1.655111718059222466900917172705448065052184678587394506208944906007626306997703966539065820047728131e-341)"), a, new Apfloat("5e-438"));

        a = ApcomplexMath.gamma(new Apcomplex("(6.493816315e2,-4.869675251e4)"), new Apcomplex("(2.053525026, -4.869675251e4)"));
        assertEquals("6.493816315762113e2-4.869675251658631e4i,2.053525026457146-4.869675251658631e4i precision 10", 5, a.precision());
        assertEquals("6.493816315762113e2-4.869675251658631e4i,2.053525026457146-4.869675251658631e4i value 10", new Apcomplex("(-3.336774750653543e-30179,-3.429254232216520e-30179)"), a, new Apfloat("5e-30183"));
        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("6.493816315762113e2", 100), new Apfloat("-4.869675251658631e4", 100)), new Apcomplex(new Apfloat("2.053525026457146", 100), new Apfloat("-4.869675251658631e4", 100)));
        assertEquals("6.493816315762113e2-4.869675251658631e4i,2.053525026457146-4.869675251658631e4i precision 100", 95, a.precision());
        assertEquals("6.493816315762113e2-4.869675251658631e4i,2.053525026457146-4.869675251658631e4i value 100", new Apcomplex("(-3.336774750653542780292101143074244789906130563661596634385946198826707896979733558295247049539498194e-30179,-3.429254232216520432738635661723033982576737397069537462221535204624693947897645031930120034934224312e-30179)"), a, new Apfloat("5e-30278"));

        a = ApcomplexMath.gamma(new Apcomplex("(6.493816315762113e2,1.154781984689458e4)"), new Apcomplex("(-8.659643233600654,1.154781984689458e4)"));
        assertEquals("6.493816315762113e2+1.154781984689458e4i,-8.659643233600654+1.154781984689458e4i precision 16", 11, a.precision());
        assertEquals("6.493816315762113e2+1.154781984689458e4i,-8.659643233600654+1.154781984689458e4i value 16", new Apcomplex("(-6.550743789275929e-5242,-1.416983672208689e-5242)"), a, new Apfloat("5e-5257"));

        a = ApcomplexMath.gamma(new Apcomplex(new Apfloat("6.493816315762113e2", 50), new Apfloat("1.154781984689458e4", 50)), new Apcomplex(new Apfloat("-8.659643233600654", 50), new Apfloat("1.154781984689458e4", 50)));
        assertEquals("6.493816315762113e2+1.154781984689458e4i,-8.659643233600654+1.154781984689458e4i precision 50", 45, a.precision());
        assertEquals("6.493816315762113e2+1.154781984689458e4i,-8.659643233600654+1.154781984689458e4i value 50", new Apcomplex("(-6.5507437892759319457546310871131583985239939226551e-5242,-1.4169836722086846119544003377316736599354795064143e-5242)"), a, new Apfloat("5e-5286")); // NOTE some 20 last digits are incorrect

//        a = ApcomplexMath.gamma(new Apcomplex("(-1.53992652e2,6.49381631e-3)"), new Apcomplex("(-1.53992652e2,6.49381631e-3)"));
//        assertEquals("-1.53992652e2+6.49381631e-3i,-1.53992652e2+6.49381631e-3i precision 9", 9, a.precision());
//        assertEquals("-1.53992652e2+6.49381631e-3i,-1.53992652e2+6.49381631e-3i value 9", new Apcomplex("(7.9792063e-273,-1.030977957e-271)"), a, new Apfloat("5e-279"));
        a = ApcomplexMath.gamma(new Apcomplex("(-1.539926526059491e2,6.493816315762112e-3)"), new Apcomplex("(-1.539926526059491e2,6.493816315762112e-3)"));
        assertEquals("-1.539926526059491e2+6.493816315762112e-3i,-1.539926526059491e2+6.493816315762112e-3i precision 16", 11, a.precision());
        assertEquals("-1.539926526059491e2+6.493816315762112e-3i,-1.539926526059491e2+6.493816315762112e-3i value 16", new Apcomplex("(7.9792063e-273,-1.030977957e-271)"), a, new Apfloat("5e-286"));

        a = ApcomplexMath.gamma(new Apcomplex("(6.493816315762113e2,-4.869675251658631e4)"), new Apcomplex("(-8.659643233600654,-4.869675251658631e4)"));
        assertEquals("6.493816315762113e2-4.869675251658631e4i,-8.659643233600654-4.869675251658631e4i precision 16", 11, a.precision());
        assertEquals("6.493816315762113e2-4.869675251658631e4i,-8.659643233600654-4.869675251658631e4i value 16", new Apcomplex("(-3.25441737499e-30179,-3.3855297992e-30179)"), a, new Apfloat("5e-30189"));

        a = ApcomplexMath.gamma(new Apcomplex("(6.493816315762113e2,1.154781984689458e4)"), new Apcomplex("(-8.659643233600654,1.154781984689458e4)"));
        assertEquals("6.493816315762113e2+1.154781984689458e4i,-8.659643233600654+1.154781984689458e4i precision 16", 11, a.precision());
        assertEquals("6.493816315762113e2+1.154781984689458e4i,-8.659643233600654+1.154781984689458e4i value 16", new Apcomplex("(-6.550743789275931e-5242,-1.416983672208684e-5242)"), a, new Apfloat("5e-5252"));

        a = ApcomplexMath.gamma(new Apcomplex("(1.154781984689458e4,-1.539926526059491e2)"), new Apcomplex("(1.154781984689458e4,3.651741272548377e1)"));
        assertEquals("1.154781984689458e4-1.539926526059491e2i,1.154781984689458e4+3.651741272548377e1i precision 16", 11, a.precision());
        assertEquals("1.154781984689458e4-1.539926526059491e2i,1.154781984689458e4+3.651741272548377e1i value 16", new Apcomplex("(-8.382175690814767e41895,-2.511119213693562e41895)"), a, new Apfloat("5e41880"));

        a = ApcomplexMath.gamma(new Apfloat("-4.0000000", Apfloat.DEFAULT, 11), new Apfloat("-6.0000000", Apfloat.DEFAULT, 11));
        assertEquals("-4,-6 precision radix 11", 7, a.precision());
        assertEquals("-4,-6 value radix 11", new Apcomplex(new Apfloat("-0.0987840", Apfloat.DEFAULT, 11), new Apfloat("-0.1492559", Apfloat.DEFAULT, 11)), a, new Apfloat("5e-7", Apfloat.DEFAULT, 11));

        a = ApcomplexMath.gamma(new Apfloat("-4.0000000", Apfloat.DEFAULT, 11), new Apfloat("-0.60000000", Apfloat.DEFAULT, 11));
        assertEquals("-4,-0.6 precision radix 11", 9, a.precision());
        assertEquals("-4,-0.6 value radix 11", new Apcomplex(new Apfloat("6.11a99456", Apfloat.DEFAULT, 11), new Apfloat("-0.14925588", Apfloat.DEFAULT, 11)), a, new Apfloat("5e-8", Apfloat.DEFAULT, 11));

        try
        {
            ApcomplexMath.gamma(Apcomplex.ZERO, Apcomplex.ZERO);
            fail("Gamma of zero");
        }
        catch (ArithmeticException ae)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex("(1e100,1)"), new Apcomplex("(1e102,1)"));
            fail("Overflow");
        }
        catch (OverflowException are)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex(new Apfloat(4)), new Apcomplex(new Apfloat(5)), Apfloat.ZERO);
            fail("Infinite expansion");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK
        }
    }

    public static void testGammaIncompleteGeneralized()
    {
        Apcomplex a = ApcomplexMath.gamma(new Apcomplex("(-1.0000000e-2,4.0000000e2)"), Apcomplex.ZERO, new Apcomplex("(1.0000000e-2,1.0000000e2)"));
        assertEquals("-1e-2+4e2i,0,1e-2+1e2i precision", 5, a.precision());
        assertEquals("-1e-2+4e2i,0,1e-2+1e2i value", new Apcomplex("(4.3680e-276,1.761e-277)"), a, new Apfloat("5e-280"));
        a = ApcomplexMath.gamma(new Apcomplex("(-1.0000000e-2,4.0000000e2)"), new Apcomplex("(1.0000000e-2,1.0000000e2)"), Apcomplex.ZERO);
        assertEquals("-1e-2+4e2i,1e-2+1e2i,0 precision", 5, a.precision());
        assertEquals("-1e-2+4e2i,1e-2+1e2i,0 value", new Apcomplex("(-4.3680e-276,-1.761e-277)"), a, new Apfloat("5e-280"));
        a = ApcomplexMath.gamma(new Apcomplex("(-1.0000000e-2,4.0000000e2)"), new Apcomplex("(1.0000000e-2,1.0000000e2)"), new Apcomplex("(1.0000000e-2,1.0000000e2)"));
        assertEquals("-1e-2+4e2i,1e-2+1e2i,1e-2+1e2i precision", Apfloat.INFINITE, a.precision());
        assertEquals("-1e-2+4e2i,1e-2+1e2i,1e-2+1e2i value", Apcomplex.ZERO, a);
        a = ApcomplexMath.gamma(new Apcomplex("(0,1.0000)"), new Apcomplex("(0,1.0000)"), new Apcomplex("(0,2.0000)"));
        assertEquals("i,i,2i precision", 5, a.precision());
        assertEquals("i,i,2i value", new Apcomplex("(0.06560,-0.12764)"), a, new Apfloat("5e-5"));
        a = ApcomplexMath.gamma(Apcomplex.ZERO, new Apcomplex("(0,1.0000)"), new Apcomplex("(0,2.0000)"));
        assertEquals("0,i,2i precision", 5, a.precision());
        assertEquals("0,i,2i value", new Apcomplex("(0.08558,-0.65933)"), a, new Apfloat("5e-5"));
        a = ApcomplexMath.gamma(Apcomplex.ZERO, new Apcomplex("(1.0000,1.0000)"), new Apcomplex("(2.0000,0)"));
        assertEquals("0,1+i,2 precision", 5, a.precision());
        assertEquals("0,1+i,2 value", new Apcomplex("(-0.04862,-0.17932)"), a, new Apfloat("5e-5"));
        a = ApcomplexMath.gamma(Apcomplex.ZERO, new Apcomplex("(1.0000,1.0000)"), new Apcomplex("(0,2.0000)"));
        assertEquals("0,1+i,2i precision", 5, a.precision());
        assertEquals("0,1+i,2i value", new Apcomplex("(0.42326,-0.21394)"), a, new Apfloat("5e-5"));
        a = ApcomplexMath.gamma(new Apcomplex("(0,1.0000)"), new Apcomplex("(0,1.0000)"), Apcomplex.ZERO);
        assertEquals("i,i,0 precision", 5, a.precision());
        assertEquals("i,i,0 value", new Apcomplex("(0.14013,0.28147)"), a, new Apfloat("5e-5"));

        try
        {
            ApcomplexMath.gamma(Apcomplex.ZERO, Apcomplex.ZERO, Apcomplex.ZERO);
            fail("Gamma of zero");
        }
        catch (ArithmeticException ae)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex("-1"), Apcomplex.ZERO, new Apcomplex("(1,2)"));
            fail("Lower gamma of non-positive integer");
        }
        catch (ArithmeticException ae)
        {
            // OK a is non-positive integer
        }
        try
        {
            ApcomplexMath.gamma(Apcomplex.ZERO, new Apcomplex("(0,-1)"), Apcomplex.ZERO);
            fail("Lower gamma of non-positive integer");
        }
        catch (ArithmeticException ae)
        {
            // OK a is non-positive integer
        }
        try
        {
            ApcomplexMath.gamma(Apcomplex.ZERO, Apcomplex.ZERO, new Apcomplex("(0,-1)"));
            fail("Lower gamma of non-positive integer");
        }
        catch (ArithmeticException ae)
        {
            // OK a is non-positive integer
        }
        try
        {
            ApcomplexMath.gamma(Apcomplex.ZERO, Apcomplex.ZERO, new Apcomplex("(1,-1)"));
            fail("Lower gamma of non-positive integer");
        }
        catch (ArithmeticException ae)
        {
            // OK a is non-positive integer
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex("(1.00e19,1.00e19)"), new Apcomplex("(1.00e18,1.00e18)"), new Apcomplex("(2.00e19,2.00e19)"));
            fail("Overflow");
        }
        catch (OverflowException are)
        {
            // OK
        }
        try
        {
            ApcomplexMath.gamma(new Apcomplex(Apfloat.ZERO, new Apfloat(4)), new Apcomplex(Apfloat.ZERO, new Apfloat(5)), new Apcomplex(Apfloat.ZERO, new Apfloat(6)));
            fail("Infinite expansion");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK
        }
    }

    public static void testUlp()
    {
        assertEquals("ulp 0", new Apcomplex("0"), ApcomplexMath.ulp(new Apcomplex("0")));
        Apfloat a = ApcomplexMath.ulp(new Apcomplex(new Apfloat(1)));
        assertEquals("ulp 1", new Apfloat("0"), a);
        assertEquals("ulp 1 precision", Apcomplex.INFINITE, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("1"));
        assertEquals("ulp 1.", new Apfloat("1"), a);
        assertEquals("ulp 1. precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("1.0"));
        assertEquals("ulp 1.0", new Apfloat("0.1"), a);
        assertEquals("ulp 1.0 precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("1.1"));
        assertEquals("ulp 1.1", new Apfloat("0.1"), a);
        assertEquals("ulp 1.1 precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("0.1"));
        assertEquals("ulp 0.1", new Apfloat("0.1"), a);
        assertEquals("ulp 0.1 precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("0.2"));
        assertEquals("ulp 0.2", new Apfloat("0.1"), a);
        assertEquals("ulp 0.2 precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("10"));
        assertEquals("ulp 10.", new Apfloat("1"), a);
        assertEquals("ulp 10. precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("1e1"));
        assertEquals("ulp 1e1", new Apfloat("1e1"), a);
        assertEquals("ulp 1e1 precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("-1e1"));
        assertEquals("ulp -1e1", new Apfloat("1e1"), a);
        assertEquals("ulp -1e1 precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex(Apfloat.ZERO, new Apfloat("1.a", 2, 11)));
        assertEquals("ulp (0, 1.a)", new Apfloat("0.1", 1, 11), a);
        assertEquals("ulp (0, 1.a) precision", 1, a.precision());
        assertEquals("ulp (0, 1.a) radix", 11, a.radix());
        a = ApcomplexMath.ulp(new Apcomplex("1e-9000000000000000000"));
        assertEquals("ulp 1e-9000000000000000000", new Apfloat("1e-9000000000000000000"), a);
        assertEquals("ulp 1e-9000000000000000000 precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex(new Apfloat("1e-9000000000000000000", 9000000000000000000L), new Apfloat("1e-9000000000000000000", 9000000000000000000L)));
        assertEquals("ulp (1e-9000000000000000000.000, 1e-9000000000000000000.000)", new Apfloat(0), a);
        assertEquals("ulp (1e-9000000000000000000.000, 1e-9000000000000000000.000) precision", Apcomplex.INFINITE, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("(1.11,-101.1)"));
        assertEquals("ulp (1.11,-101.1)", new Apfloat("0.1"), a);
        assertEquals("ulp (1.11,-101.1) precision", 1, a.precision());
        a = ApcomplexMath.ulp(new Apcomplex("(-9.9,0.0000011)"));
        assertEquals("ulp (-9.9,0.0000011)", new Apfloat("0.1"), a);
        assertEquals("ulp (-9.9,0.0000011) precision", 1, a.precision());
    }
}
