
//                                               -*- C++ -*-
/**
 * @brief HSICEstimator is a base class that  implements the HSIC sensivity indices.
 *
 *  Copyright 2005-2024 Airbus-EDF-IMACS-ONERA-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
#ifndef OPENTURNS_HSICESTIMATOR_HXX
#define OPENTURNS_HSICESTIMATOR_HXX

#include "openturns/TypedInterfaceObject.hxx"
#include "openturns/HSICEstimatorImplementation.hxx"
#include "openturns/ResourceMap.hxx"
#include "openturns/CovarianceModel.hxx"
#include "openturns/Sample.hxx"
#include "openturns/Graph.hxx"
#include "openturns/Pointer.hxx"
#include "openturns/Function.hxx"

BEGIN_NAMESPACE_OPENTURNS

/**
 * @class HSICEstimator
 *
 */
class OT_API HSICEstimator
  : public TypedInterfaceObject<HSICEstimatorImplementation>
{
  CLASSNAME

public:

  typedef Collection<CovarianceModel> CovarianceModelCollection;
  typedef Pointer<HSICEstimatorImplementation> Implementation;

  /** Default ctor */
  HSICEstimator();

  /** Constructor from implementation */
  HSICEstimator(const HSICEstimatorImplementation & implementation);

#ifndef SWIG
  /** Constructor from implementation pointer */
  HSICEstimator(HSICEstimatorImplementation * p_implementation);
#endif

  /** Get the HSIC indices.
   *  This is not const as it triggers a computation of the indices
   *  if they are not computed yet.
   * */
  Point getHSICIndices() const;

  /** Get the R2-HSIC indices.
   *  This is not const as it triggers a computation of the indices
   *  if they are not computed yet.
   * */
  Point getR2HSICIndices() const;

  /** Get the p-values by permutation.
   *  This is not const as it triggers a computation of the values
   *  if they are not computed yet.
   * */
  Point getPValuesPermutation() const;

  // Get the asymptotic p-values.
  Point getPValuesAsymptotic() const;

  /** Graphic methods */
  Graph drawHSICIndices() const;

  /** Graphic methods */
  Graph drawR2HSICIndices() const;

  /** Graphic methods */
  Graph drawPValuesPermutation() const;

  /** Graphic methods */
  Graph drawPValuesAsymptotic() const;

  // run all
  void run() const;

  /** Set the number of permutation used */
  void setPermutationSize(const UnsignedInteger B);

  /** Get the number of permutation used */
  UnsignedInteger getPermutationSize() const;

  /** Get the covariance models */
  HSICEstimatorImplementation::CovarianceModelCollection getCovarianceModelCollection() const;

  /** Set the covariance models */
  void setCovarianceModelCollection(const CovarianceModelCollection & coll);

  /** Get the input sample */
  Sample getInputSample() const;

  /** Set the input sample */
  void setInputSample(const Sample & inputSample);

  /** Get the output sample */
  Sample getOutputSample() const;

  /** Sset the output sample */
  void setOutputSample(const Sample & outputSample);

  /** Get the dimension of the indices: the number of marginals */
  UnsignedInteger getDimension() const;

  /** Get the size of the study sample */
  UnsignedInteger getSize() const;

  /** Get the underlying estimator: biased or unbiased*/
  HSICStat getEstimator() const;

};


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_HSICESTIMATOR_HXX */
