#include "aptpackage.h"

#include <sstream>

#include <apt-pkg/cachefile.h>
#include <apt-pkg/pkgcache.h>
#include <apt-pkg/pkgrecords.h>
#include <apt-pkg/fileutl.h>

using namespace std;

namespace NApt
{

AptPackage::AptPackage(AptPackage& package) :
    AptPackage(package._package, package._pPkgCache, package._recs)
{}

AptPackage::AptPackage(const std::string& name, pkgCache* pPkgCache, const pkgRecords& records) :
    AptPackage(findPkgIter(name), pPkgCache, records)
{}

AptPackage::AptPackage(pkgCache::PkgIterator pkgIter, pkgCache* pPkgCache, const pkgRecords& records) :
    _name(pkgIter.Name()),
    _pPkgCache(pPkgCache),
    _recs(records),
    _package(pkgIter),
    _availableVersion(_package.VersionList()),
    _installedVersion(_package.CurrentVer()),
    _description(versionIter().TranslatedDescription())
{
}

pkgRecords::Parser& AptPackage::parser() const {
    return const_cast<pkgRecords&>(_recs).Lookup(versionIter().FileList());
}

pkgCache::PkgIterator AptPackage::findPkgIter(const std::string& name)
{
    pkgCache::GrpIterator Grp = _pPkgCache->FindGrp(name);
    if (Grp.end() == false) {
        return Grp.FindPreferredPkg();
    }
    return _pPkgCache->PkgEnd();
}

pkgCache::VerIterator AptPackage::availableVersionIter() const
{
    return _package.VersionList();
}

bool AptPackage::isValid() const
{
    return !shortDescription().isEmpty();
}

void AptPackage::logAll() const {
    cout << name().toStdString() << " - " << shortDescription().toStdString() << endl;
    cout << "Installed Version: " << installedVersion().toStdString() << endl;
    cout << "Available Version: " << version().toStdString() << endl;
    cout << "Maintainer: " << maintainer().toStdString() << endl;
    cout << "Homepage: " << homepage().toStdString() << endl;
    cout << "Priority: " << priority().toStdString() << endl;
    if (!essential().isEmpty()) {
        cout << "Essential: " << essential().toStdString() << endl;
    }
    cout << "Section: " << section().toStdString() << endl;
    cout << "Size: " << size().toStdString() << endl;
    cout << "Filename: " << filename().toStdString() << endl;
    cout << "InstalledSize: " << installedSize().toStdString() << endl;
    string installedStateString;
    switch (installedState()) {
        case IPackage::NOT_INSTALLED:
            installedStateString = "Not installed";
            break;
        case IPackage::UPGRADABLE:
            installedStateString = "Upgradable";
            break;
        case IPackage::INSTALLED:
            installedStateString = "Installed";
            break;
    }
    cout << "InstalledState: " << installedStateString << endl;
    cout << "Description: " << description().toStdString() << endl;
    cout << "Depends: " << depends().toStdString() << endl;
    cout << "Recommends: " << recommends().toStdString() << endl;
    cout << "Suggests: " << suggests().toStdString() << endl;
    cout << "Provides: " << provides().toStdString() << endl;
    cout << "MD5sum: " << md5sum().toStdString() << endl;
    cout << "Filename: " << filename().toStdString() << endl;
}


QString AptPackage::name() const
{
    return QString::fromStdString(_name);
}

QString AptPackage::essential() const
{
    if ((_package->Flags & pkgCache::Flag::Essential) == pkgCache::Flag::Essential)
        return "yes";
    else
        return _emptyString;
}

QString AptPackage::priority() const
{
    return versionIter().PriorityType();
}

QString AptPackage::section() const
{
    return versionIter().Section();
}

QString AptPackage::installedSize() const
{
    return QString::number(getInstalledSize());
}

uint AptPackage::getInstalledSize() const
{
    return versionIter()->InstalledSize;
}

QString AptPackage::maintainer() const
{
    return QString::fromStdString(parser().Maintainer());
}

QString AptPackage::architecture() const
{
    return versionIter().Arch();
}

QString AptPackage::source() const
{
    auto iter = versionIter();
    if (!versionIter().end())
        return iter.SourcePkgName();
    else
        return _emptyString;
}

QString AptPackage::version() const
{
    if (versionIter().end()) return _emptyString;
    return versionIter().VerStr();
}

QString AptPackage::replaces() const
{
    return QString::fromStdString(parser().RecordField("Replaces"));
}

QString AptPackage::breaks() const
{
    return QString::fromStdString(parser().RecordField("Breaks"));
}


QString AptPackage::provides() const
{
    return QString::fromStdString(parser().RecordField("Provides"));
}

QString AptPackage::preDepends() const
{
    return QString::fromStdString(parser().RecordField("PreDepends"));
}

QString AptPackage::depends() const
{
    return QString::fromStdString(parser().RecordField("Depends"));
}

QString AptPackage::recommends() const
{
    return QString::fromStdString(parser().RecordField("Recommends"));
}

QString AptPackage::suggests() const
{
    return QString::fromStdString(parser().RecordField("Suggests"));
}

QString AptPackage::conflicts() const
{
    return QString::fromStdString(parser().RecordField("Conflicts"));
}

QString AptPackage::filename() const
{
    /*
    pkgCache::VerFileIterator vfi = versionIter().FileList();
    pkgCache::PkgFileIterator pfi = vfi.File();
    FileFd pkgf(pfi.FileName(), FileFd::ReadOnly);

    // Read the record and then write it out again.
    char* buffer = new char[vfi->Size+1];
    buffer[vfi->Size] = '\n';
    if (!pkgf.Seek(vfi->Offset) || !pkgf.Read(buffer, vfi->Size))
    {
        delete[] buffer;
        return QString();
    }

    std::string res(buffer, vfi->Size);
    delete[] buffer;
    return QString::fromStdString(res);
*/
    return _emptyString;
    //return QString::fromStdString(parser().RecordField("Filename"));

}

QString AptPackage::size() const
{
    return QString::number(getSize());
}

uint AptPackage::getSize() const
{
    return versionIter()->Size;
}

QString AptPackage::md5sum() const
{
    return QString::fromStdString(parser().RecordField("MD5sum"));
}

QString AptPackage::conffiles() const
{
    // TODO: can't figure out where to get this info from for now.
    return _emptyString;
}

QString AptPackage::description() const
{
    auto vi = versionIter();
    const char* v = vi.VerStr();
    if (v == 0) return _emptyString;

    pkgCache::DescIterator description = vi.TranslatedDescription();
    // Lookup should be const, but since it is not, we cast away the const here
    pkgRecords::Parser &parser = const_cast<pkgRecords&>(_recs).Lookup(description.FileList());
    return QString::fromStdString(parser.LongDesc());
}


AptPackage::InstalledState AptPackage::installedState() const
{
    if (_installedVersion.end())
        return NOT_INSTALLED;
    else if (_availableVersion.CompareVer(_installedVersion) > 0)
        return UPGRADABLE;
    else
        return INSTALLED;
}

QString AptPackage::installedVersion() const
{
    if (!_installedVersion.end()) {
        return _installedVersion.VerStr();
    } else {
        return "";
    }
}

QString AptPackage::shortDescription() const
{
    return QString::fromStdString(parser().ShortDesc());
}

QString AptPackage::homepage() const
{
    return QString::fromStdString(parser().Homepage());
}

}
