/*
 * Copyright (c) 2003-2004, Artem B. Bityuckiy
 * Copyright (c) 1999,2000, Konstantin Chuguev. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "cesbi.h"

#if defined (ICONV_TO_UCS_CES_UCS_4) \
 || defined (ICONV_FROM_UCS_CES_UCS_4)

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include "../lib/local.h"
#include "../lib/ucsconv.h"
#include "../lib/endian.h"

/*
 * BOM isn't supported. UCS-4 is Big Endian. Bad codes are rejected.
 * Bad codes: 0x0000FFFF, 0x0000FFFE, 0x0000D800-0x0000DFFF, 
 * 0x7FFFFFFF-0xFFFFFFFF.
 */

#define UCS_4_BIG_ENDIAN     0
#define UCS_4_LITTLE_ENDIAN  1

#define UCS_4   "ucs_4"
#define UCS_4BE "ucs_4be"
#define UCS_4LE "ucs_4le"

static void *
ucs_4_init (
                   const char *encoding)
{
  int *data;
  
  if ((data = (int *)malloc (sizeof(int))) == NULL)
    return (void *)NULL;
  
  if (strcmp (encoding, UCS_4LE) == 0)
    *data = UCS_4_LITTLE_ENDIAN;
  else
    *data = UCS_4_BIG_ENDIAN;
     
  return (void *)data;
}

static size_t
ucs_4_close (
                    void *data)
{
  free(data);
  return 0;
}


#if defined (ICONV_FROM_UCS_CES_UCS_4)
static size_t
ucs_4_convert_from_ucs (void *data,
                               ucs4_t in,
                               unsigned char **outbuf,
                               size_t *outbytesleft)
{
  if ((in  >= 0x0000D800 && in <= 0x0000DFFF) /* Surrogate character */
      || in > 0x7FFFFFFF || in == 0x0000FFFF || in == 0x0000FFFE)
    return (size_t)ICONV_CES_INVALID_CHARACTER; 

  if (*outbytesleft < sizeof (ucs4_t))
    return (size_t)ICONV_CES_NOSPACE;

  ucs4_t uc;

  if (*((int *)data) == UCS_4_BIG_ENDIAN)
    uc = ICONV_HTOBEL (in);
  else
    uc = ICONV_HTOLEL (in);

  memcpy(*outbuf, &uc, sizeof (ucs4_t));

  *outbuf += sizeof (ucs4_t);
  *outbytesleft -= sizeof (ucs4_t);

  return sizeof (ucs4_t);
}
#endif /* ICONV_FROM_UCS_CES_UCS_4 */

#if defined (ICONV_TO_UCS_CES_UCS_4)
static ucs4_t
ucs_4_convert_to_ucs (void *data,
                             const unsigned char **inbuf,
                             size_t *inbytesleft)
{
  ucs4_t res;
  
  if (*inbytesleft < sizeof (ucs4_t))
    return (ucs4_t)ICONV_CES_BAD_SEQUENCE;

  ucs4_t uc;

  memcpy(&uc, *inbuf, sizeof (ucs4_t));

  if (*((int *)data) == UCS_4_BIG_ENDIAN)
    res = ICONV_BETOHL (uc);
  else
    res = ICONV_LETOHL (uc);

  if ((res  >= 0x0000D800 && res <= 0x0000DFFF) /* Surrogate character */
      || res > 0x7FFFFFFF || res == 0x0000FFFF || res == 0x0000FFFE)
    return (ucs4_t)ICONV_CES_INVALID_CHARACTER;

  *inbytesleft -= sizeof (ucs4_t);
  *inbuf += sizeof(ucs4_t);
  
  return res;
}
#endif /* ICONV_TO_UCS_CES_UCS_4 */

static int
ucs_4_get_mb_cur_max (void *data)
{
  (void) data;
  return 4;
}

#if defined (ICONV_TO_UCS_CES_UCS_4)
const iconv_to_ucs_ces_handlers_t
_iconv_to_ucs_ces_handlers_ucs_4 = 
{
  ucs_4_init,
  ucs_4_close,
  ucs_4_get_mb_cur_max,
  NULL,
  NULL,
  NULL,
  ucs_4_convert_to_ucs
};
#endif

#if defined (ICONV_FROM_UCS_CES_UCS_4)
const iconv_from_ucs_ces_handlers_t
_iconv_from_ucs_ces_handlers_ucs_4 =
{
  ucs_4_init,
  ucs_4_close,
  ucs_4_get_mb_cur_max,
  NULL,
  NULL,
  NULL,
  ucs_4_convert_from_ucs
};
#endif

#endif /* ICONV_TO_UCS_CES_UCS_4 || ICONV_FROM_UCS_CES_UCS_4 */

