/*	Procedures_Table_Model

PIRL CVS ID: Procedures_Table_Model.java,v 1.5 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor;

import	javax.swing.table.DefaultTableModel;
import	java.util.Vector;


/**	A <i>Procedures_Table_Model<i> contains a table of Conductor
	procedure definition records.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.5
*/
public class Procedures_Table_Model
	extends DefaultTableModel
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor..Procedures_Table_Model (1.5 2012/04/16 06:04:10)";

/**	Flag that determines if the table cells are editable.
*/
public boolean
	Editable		= false;

/**	The row designated as the currently active procedure definitiion of
	the table.
*/
public int
	Current_Row		= -1;

private Vector
	Snapshot_Names	= new Vector (0),
	Snapshot_Data	= new Vector (0);


private static final String
	NL				= System.getProperty ("line.separator");

/*==============================================================================
	Constructors
*/
/**	Construct an empty Procedures_Table_Model.
*/
public Procedures_Table_Model ()
{}

/**	Construct a Procedures_Table_Model from a Vector of data record
	Vectors and a Vector of field names.
<p>
	@param	records	A Vector of records; each record is a Vector of
		field value Strings any of which may be null.
	@param	field_names	A Vector of field_name Strings.
	@see	#setDataVector(Vector, Vector)
*/
public Procedures_Table_Model
	(
	Vector		records,
	Vector		field_names
	)
{setDataVector (records, field_names);}

/**	Construct a Procedures_Table_Model from an array of data record
	arrays and an array of field names.
<p>
	@param	records	An array of records; each record is an array of
		field value Strings any of which may be null.
	@param	field_names	An array of field_name Strings.
	@see	#setDataVector(Vector, Vector)
*/
public Procedures_Table_Model
	(
	String[][]	records,
	String[]	field_names
	)
{setDataVector (convertToVector (records), convertToVector (field_names));}

/**	Construct a Procedures_Table_Model from a Vector of field names
	but no table data.
<p>
	A {@link #Snapshot() snapshot} of the model is taken.
<p>
	@param	field_names	A Vector of field_name Strings.
*/
public Procedures_Table_Model
	(
	Vector		field_names
	)
{
super (field_names, 0);
Snapshot ();
}

/**	Construct a Procedures_Table_Model from a Vector of field names
	but no table data.
<p>
	A {@link #Snapshot() snapshot} of the model is taken.
<p>
	@param	field_names	An array of field_name Strings.
*/
public Procedures_Table_Model
	(
	String[]	field_names
	)
{this (convertToVector (field_names));}

/*==============================================================================
	TableModel
*/
/**	Get the Class for a table column.
<p>
	@param	column	A table column index.
	@return	String.class; except if the column is not a valid value
		Object.class is returned.
*/
public Class getColumnClass
	(
	int		column
	)
{
if (column >= 0 &&
	column < getColumnCount ())
	return String.class;
return Object.class;
}

/**	Get a table cell value.
<p>
	@param	row		A table row index.
	@param	column	A table column index.
	@return	The Object at the row and column cell location of the table
		data. This will be null if either the row or column value is
		invalid.
*/
public Object getValueAt
	(
	int		row,
	int		column
	)
{
if (row < 0 ||
	row >= getRowCount () ||
	column < 0 ||
	column >= getColumnCount ())
	return null;
return ((Vector)dataVector.get (row)).get (column);
}

/**	Test if a table cell is editable.
<p>
	@param	row		A table row index.
	@param	column	A table column index.
	@return	The value of the {@link #Editable} flag.
*/
public boolean isCellEditable
	(
	int		row,
	int		column
	)
{return Editable;}

/*==============================================================================
	DefaultTableModel
*/
/**	Replace the table data content.
<p>
	Both the field names and procedure records are replaced.
<p>
	A {@link #Snapshot() snapshot} of the model is taken.
<p>
	@param	records	A Vector of records; each record is a Vector of
		field value Strings any of which may be null.
	@param	field_names	A Vector of field_name Strings.
*/
public void setDataVector
	(
	Vector	records,
	Vector	field_names
	)
{
super.setDataVector (records, field_names);
Snapshot ();
}

/**	Insert a procedure definition record in the table.
<p>
	The table snapshot data is updated with the new record.
<p>
	@param	row	A table row index. This must be in the range zero to the
		total number of records in the table, inclusive. If the row
		equals the total number of records in the table the record is
		appended to the table.
	@param	record	A procedure definition record Vector. It's size must
		be the same as the table's {@link #getColumnCount() column count}.
*/
public void insertRow
	(
	int		row,
	Vector	record
	)
{
super.insertRow (row, record);
int
	columns = getColumnCount ();
Vector
	current = (Vector)dataVector.get (row),
	original = new Vector (columns);
for (int
		column = 0;
		column < columns;
		column++)
	original.add (current.get (column));
Snapshot_Data.insertElementAt (original, row);
}
//	The addColumn and other insertRow methods defer to the one above.

/*==============================================================================
	Accessors
*/
/**	Get the procedure definition record at a table row index.
<p>
	@param	row	A table row index. This must be in the range zero to the
		total number of records in the table, exclusive.
	@return	A procedure definition record Vector.
*/
public Vector Record
	(
	int		row
	)
{
if (row >= 0 &&
	row < getRowCount ())
	return (Vector)dataVector.get (row);
return null;
}

/**	Set the procedure definition record at a table row index.
<p>
	If a new row is added to the table the table snapshot data is updated
	with the new record.
<p>
	Notification of the row record being updated or added will be
	generated.
<p>
	@param	row	A table row index. This must be in the range zero to the
		total number of records in the table, inclusive. If the row
		equals the total number of records in the table the record is
		appended to the table. Otherwise the record replaces the existing
		record at the row index.
	@param	record	A procedure definition record Vector. It's size must
		be the same as the table's {@link #getColumnCount() column count}.
	@return	This Procedures_Table_Model.
	@throws	ArrayIndexOutOfBoundsException	If the row index is invalid
		or the record size is not the same as the number of columns in
		the table.
*/
public Procedures_Table_Model Record
	(
	int		row,
	Vector	record
	)
{
if (record == null)
	return this;
if (row < 0 ||
	row > getRowCount ())
	throw new ArrayIndexOutOfBoundsException (ID + NL
		+ "Unable to set the record at row " + row + '.' + NL
		+ "The valid row range is 0 - " + getRowCount () + " inclusive.");
if (record.size () != getColumnCount ())
	throw new ArrayIndexOutOfBoundsException (ID + NL
		+ "Unable to set the record at row " + row + '.' + NL
		+ "The " + record.size () + " element record size does not match the "
			+ getColumnCount () + " column table size.");

if (row == getRowCount ())
	insertRow (row, record);
else
	{
	dataVector.set (row, record);
	fireTableRowsUpdated (row, row);
	}
return this;
}

/**	Restore the table content from the last {@link #Snapshot() snapshot}.
<p>
	The table records are restored from a copy of the last {@link
	#Snapshot_Table_Data() snapshot table data}. If the number of table
	columns has changed since the last snapshot the field names are also
	restored from a copy of the last {@link #Snapshot_Column_Names()
	snapshot column names}. <b>N.B.</b>: No check is made if the field
	names have changed; full table content can be forced by:
<p><code><pre>
    setDataVector
      (Table (Snapshot_Table_Data ()),
      Record (Snapshot_Column_Names ()));
</pre></code><p>
	Notification of the change to the table contents will be generated.
<p>
	@see	#Snapshot()
*/
public void Restore ()
{
int
	rows = dataVector.size (),
	columns = columnIdentifiers.size ();

dataVector = Table (Snapshot_Data);
if (columns == Snapshot_Names.size ())
	{
	if (rows == Snapshot_Data.size ())
		fireTableRowsUpdated (0, rows - 1);
	else
		fireTableDataChanged ();
	}
else
	{
	columnIdentifiers = Record (Snapshot_Names);
	fireTableStructureChanged ();
	}
}

/**	Save a snapshot of the table contents.
<p>
	Both the table data records and field names are saved.
<p>
	@see	#Snapshot_Table_Data()
	@see	#Snapshot_Column_Names()
*/
public void Snapshot ()
{
Snapshot_Names = Record (columnIdentifiers);
Snapshot_Data = Table (dataVector);
}

/**	The last table data records snapshot.
<p>
	@return	A Vector containing a copy of each record Vector in the table
		at the time of the last {@link #Snapshot() snapshot} was taken.
		<b>N.B.</b>: The returned Vector is the snapshot data, not a
		copy.
*/
public Vector Snapshot_Table_Data ()
{return Snapshot_Data;}

/**	The last table field names snapshot.
<p>
	@return	A Vector containing a copy of the field names for the table
		at the time of the last {@link #Snapshot() snapshot} was taken.
		<b>N.B.</b>: The returned Vector is the snapshot data, not a
		copy.
*/
public Vector Snapshot_Column_Names ()
{return Snapshot_Names;}

/**	The table field names.
<p>
	@return	A Vector containing the field names for the table.
		<b>N.B.</b>: The returned Vector is not a copy.
*/
public Vector Column_Names ()
{return columnIdentifiers;}

/**	Copy table data.
<p>
	The table is a Vector of row record Vectors. The table is expected to
	be square: the size of the first row record determines the size of
	all other rows; no attempt is made to compensenate for an irregular
	table.
<p>
	The data elements are not copied; the reference to each element
	Object is copied. For the expected case where the data elements are
	Strings, which are immutable, there is no concern that changes in the
	copied data will affect the copy, or vice versa.
<p>
	@param	records	A Vector of records; each record is a Vector of
		field value Strings any of which may be null.
	@return	A Vector containing a copy of each record Vector in the table.
		This will be null if the table is null.
*/
public static Vector Table
	(
	Vector	records
	)
{
if (records == null)
	return null;
int
	row = 0,
	rows = records.size ();
Vector
	copy = new Vector (rows);
while (row < rows)
	copy.add (Record ((Vector)records.get (row++)));
return copy;
}

/**	Copy record data.
<p>
	@param	record	A Vector of field value Strings any of which may be null.
	@return	A Vector containing a copy of the record. Must not be null.
*/
public static Vector Record
	(
	Vector	record
	)
{
if (record == null)
	return null;
int
	column = 0,
	columns = record.size ();
Vector
	copy = new Vector (columns);
while (column < columns)
	copy.add (record.get (column++));
return copy;
}


}
