/*	Memory_Chart_UI

PIRL CVS ID: Memory_Chart_UI.java,v 1.5 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import	javax.swing.JComponent;
import	javax.swing.plaf.ComponentUI;
import	java.awt.Color;
import	java.awt.Insets;
import	java.awt.Graphics;
import	java.awt.FontMetrics;
import	java.awt.geom.Rectangle2D;
import	java.awt.event.MouseEvent;
import	java.awt.event.MouseListener;
import	java.awt.event.MouseMotionListener;

/**	A <i>Memory_Chart_UI</i> provides a ComponentUI delegate for the
	Memory_Chart component.
<p>
	@see	Memory_Chart
	@version	1.5
	@see	Memory_History
*/
public class Memory_Chart_UI
	extends Memory_ChartUI
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Viewers.Memory_Chart_UI (1.5 2012/04/16 06:22:58)";


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_INSTALL		= 1 << 1,
	DEBUG_PAINT			= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG		    	= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Memory_Chart_UI object.
*/
public Memory_Chart_UI ()
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println (">-< Memory_Chart_UI.Constructor");
}

/**	Construct a Memory_Chart_UI object for the specified JComponent.
<p>
	@param	component	The JComponent for which ComponentUI delegation
		is to be provided.
	@return	A Memory_Chart_UI object.
*/
public static ComponentUI createUI
	(
	JComponent	component
	)
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println (">-< Memory_Chart_UI.createUI");
return new Memory_Chart_UI ();
}

/*==============================================================================
	Component Painting
*/
/**	Paint the component on the graphics display device.
<p>
	@param	graphics	The Graphics on which to paint the component.
	@param	component	The JComponent to be painted.
*/
public void paint
	(
	Graphics	graphics,
	JComponent	component
	)
{
if ((DEBUG & DEBUG_PAINT) != 0)
	System.out.println (">>> Memory_Chart_UI.paint");
Memory_Chart
	chart = (Memory_Chart)component;
Memory_History
	history = chart.getModel ();
if (history.Samples == 0)
	return;

//	Geometry.
Insets
	insets = component.getInsets ();
int
	canvas_width  = chart.getWidth () - insets.left - insets.right,
	canvas_height = chart.getHeight () - insets.top - insets.bottom,
	annotation_area_width  = chart.Annotation_Area_Width (),
	annotation_text_height = chart.Annotation_Text_Height (),
	seconds_scale_height   = chart.Seconds_Scale_Height (),
	chart_width  = canvas_width  - annotation_area_width,
	chart_height = canvas_height - seconds_scale_height;
if ((DEBUG & DEBUG_PAINT) != 0)
	System.out.println
		("              canvas_width = " + canvas_width + '\n'
		+"             canvas_height = " + canvas_height + '\n'
		+"     annotation_area_width = " + annotation_area_width + '\n'
		+"    annotation_text_height = " + annotation_text_height + '\n'
		+"      seconds_scale_height = " + seconds_scale_height + '\n'
		+"               chart_width = " + chart_width + '\n'
		+"              chart_height = " + chart_height);
if (chart_width < 5 ||
	chart_height < (seconds_scale_height + 10))
	{
	//	Not enough area to draw a meaningful chart.
	if ((DEBUG & DEBUG_PAINT) != 0)
		System.out.println
			("    Insufficient chart area.\n"
			+"<<< Memory_Chart_UI.paint");
	return;
	}
if (chart_width > history.Max_Samples ())
	{
	if ((DEBUG & DEBUG_PAINT) != 0)
		System.out.println
			("    Increasing Memory_History Max_Samples to " + chart_width);
	//	Add more capacity to the memory history.
	history.Max_Samples (chart_width);
	//	The model will fire a change event causing a repaint.
	if ((DEBUG & DEBUG_PAINT) != 0)
		System.out.println ("<<< Memory_Chart_UI.paint");
	return;
	}
long
	available_memory = history.Available_Memory ();

//	Set 0,0 at the upper-left corner of the chart.
graphics.translate (insets.left, insets.top);
//	Font to use for annotation and seconds scale.
graphics.setFont (chart.ANNOTATION_FONT);
FontMetrics
	font_metrics = chart.getFontMetrics (chart.ANNOTATION_FONT);
String
	annotation;
Rectangle2D
	bounds;

//	Annotation.
if (annotation_area_width > 0 &&
	(annotation_text_height << 2) <= chart_height)
	{
	//	Total available.
	annotation = Memory_Magnitude (available_memory);
	bounds = font_metrics.getStringBounds (annotation, graphics);
	graphics.setColor (Color.BLACK);
	graphics.drawString (annotation,
		canvas_width - (int)bounds.getWidth (),
		annotation_text_height - 1);

	//	Unused.
	annotation = Memory_Magnitude (available_memory - history.Allocated[0]);
	bounds = font_metrics.getStringBounds (annotation, graphics);
	graphics.setColor (Color.BLUE);
	graphics.drawString (annotation,
		canvas_width - (int)bounds.getWidth (),
		annotation_text_height << 1);

	//	Allocated but free.
	annotation = Memory_Magnitude (history.Free[0]);
	bounds = font_metrics.getStringBounds (annotation, graphics);
	graphics.setColor (Color.GREEN);
	graphics.drawString (annotation,
		canvas_width - (int)bounds.getWidth (),
		chart_height - annotation_text_height);

	//	In use.
	annotation = Memory_Magnitude (history.Allocated[0] - history.Free[0]);
	bounds = font_metrics.getStringBounds (annotation, graphics);
	graphics.setColor (Color.RED);
	graphics.drawString (annotation,
		canvas_width - (int)bounds.getWidth (),
		chart_height);
	}

int
	rate = history.Rate ();
if (rate == 0)
	rate = history.Previous_Rate ();
if (rate > 0 &&
	seconds_scale_height > 0)
	{
	//	Seconds scale annotation.
	if (rate > 1)
		annotation = "x" + String.valueOf (rate) + " s";
	else
		annotation = "secs";
	graphics.setColor (Color.BLACK);
	graphics.drawString (annotation,
		chart_width + 2, canvas_height - 1);
	}

//	Chart.
int
	sample,
	column = chart_width,
	bottom,
	top;
for (sample = 0;
	 sample < history.Samples &&
	 sample < chart_width;
   ++sample)
	{
	--column;

	if (seconds_scale_height > 0)
		{
		//	Seconds scale.
		if ((sample % 10) == 0)
			{
			//	Major tick mark.
			graphics.setColor (Color.BLACK);
			graphics.drawLine
				(column, chart_height, column, canvas_height - 1);
			if (sample != 0 &&
				rate < 20 &&
				((sample * rate) % 60) == 0)
				{
				//	Minute tick mark.
				graphics.drawLine
					(column - 1, chart_height, column - 1, canvas_height - 1);
				graphics.drawLine
					(column + 1, chart_height, column + 1, canvas_height - 1);
				}
			}
		else
		if ((sample % 5) == 0)
			{
			//	Minor tick mark.
			graphics.setColor (Color.BLACK);
			graphics.drawLine (column, chart_height, column,
				chart_height + (seconds_scale_height >> 1));
			}
		}

	bottom = chart_height - 1;
	if (history.Allocated[sample] < 0)
		{
		//	Division bar.
		graphics.setColor (Color.BLACK);
		graphics.drawLine (column, 0, column, chart_height - 1);
		continue;
		}
	if (history.Allocated[sample] > 0)
		{
		top = chart_height - 1
			-(int)((((double)(history.Allocated[sample] - history.Free[sample])
			/ available_memory) * chart_height) + 0.5);
		if (top < bottom)
			{
			graphics.setColor (Color.RED);
			graphics.drawLine (column, bottom, column, top);
			bottom = top - 1;
			}
		if (history.Free[sample] > 0)
			{
			top = chart_height - 1
				- (int)((((double)history.Allocated[sample]
				/ available_memory) * chart_height) + 0.5);
			if (top < bottom)
				{
				graphics.setColor (Color.GREEN);
				graphics.drawLine (column, bottom, column, top);
				bottom = top - 1;
				}
			}
		}
	if (bottom >= 0)
		{
		graphics.setColor (Color.BLUE);
		graphics.drawLine (column, bottom, column, 0);
		}
	}

if (seconds_scale_height > 0)
	{
	//	Max history time.
	annotation = String.valueOf (rate * sample);
	bounds = font_metrics.getStringBounds (annotation, graphics);
	int
		width = (int)bounds.getWidth () + 4;
	if (width <= sample)
		{
		graphics.setColor (Color.WHITE);
		graphics.fillRect (column, chart_height, width, seconds_scale_height);
		graphics.setColor (Color.BLACK);
		graphics.drawLine (column, chart_height, column, canvas_height - 1);
		graphics.drawString (annotation, column + 2, canvas_height);
		}
	}

//	Baseline.
graphics.setColor (Color.BLACK);
graphics.drawLine (column, chart_height, chart_width - 1, chart_height);

//	Restore the graphics origin.
graphics.translate (-insets.left, -insets.top);
if ((DEBUG & DEBUG_PAINT) != 0)
	System.out.println ("<<< Memory_Chart_UI.paint");
}


//!	Available memory amount magnitude indicator.
private static final String
	Magnitude[]			= {" ", "K", "M", "G", "T", "P"};

private String Memory_Magnitude
	(
	long	value
	)
{
int
	magnitude;
for (magnitude = 0;
	 value >= 1024;
	 magnitude++,
	 value >>>= 10);
return String.valueOf (value) + Magnitude[magnitude];
}

}	//	End of Memory_Chart_UI class.
