
header = """
#ifndef SFC_DOFT_IS_INCLUDED
#define SFC_DOFT_IS_INCLUDED


#include <map>
#include <vector>
#include <iterator>
#include <iostream>
#include <stdexcept>
#include <tr1/memory>


namespace sfc
{
  typedef std::pair< unsigned int, unsigned int > pair_ii;
  typedef std::vector< unsigned int > vector_i;
  typedef std::vector< std::pair<unsigned int, unsigned int> > vector_ii;


  /*
    A general degree of freedom (dof) builder for finite element applications,
    counting global indices based on a given dof representation object of type D.
    The type D must have an "bool D::operator<(const D&) const" implemented
    to work as key in the std::map used here.

    The structure dof2glob_map is the central structure, and is always constructed.
    The structures loc2glob_array, loc2glob_map, glob2loc_map, and glob2dof_map are 
    optional, and enabled by the bools named create_loc2glob_array etc.
    They are all initialized and updated by insert_dof(...).
  */
  template <class D>
  class DofT
  {
  protected:

    unsigned int _global_dimension;
    unsigned int _num_elements;
    unsigned int _local_dimension;

  public: // TODO: remove, temporary hack while changing sfc

    /*
       Central dynamic structure: based on dof representations
       with type D, store the corresponding global indices.
       (D Lj) -> (uint j)
    */
    std::map< D, unsigned int > dof2glob_map;

    /*
       Static and efficient local to global mapping.
       (uint e, uint i) -> (uint j)
       Index like this:
       global_index = loc2glob[element*local_dimension + local_index]
    */
    bool create_loc2glob_array;
    std::tr1::shared_ptr<unsigned int> loc2glob_array;

    /*
       Dynamic but slower local to global mapping.
       (uint e, uint i) -> (uint j)
    */
    bool create_loc2glob_map;
    std::map< pair_ii, unsigned int > loc2glob_map;
    // std::map< unsigned int e, vector_i > loc2glob_map; // TODO: this should be a bit more efficient for large local_dimension

    /*
       Dynamic global to local mapping. Given the global index of a dof,
       provides a vector with the local index of the dof for each element
       it occurs in.
       (uint j) -> vector< pair<e1, i1>, ..  pair<en, in> >
    */
    bool create_glob2loc_map;
    std::map< unsigned int, vector_ii > glob2loc_map;

    /*
       Dynamic mapping from global index to dof representation object of type D.
       (uint j) -> (D Lj)
    */
    bool create_glob2dof_map;
    std::map< unsigned int, D > glob2dof_map;


  public:

    DofT(bool create_glob2dof_map = false, bool create_glob2loc_map = false,
         bool create_loc2glob_map = false, bool create_loc2glob_array = true):
      _global_dimension(0),
      _num_elements(0),
      _local_dimension(0),
      create_loc2glob_array( create_loc2glob_array ),
      create_loc2glob_map( create_loc2glob_map ),
      create_glob2loc_map( create_glob2loc_map ),
      create_glob2dof_map( create_glob2dof_map )
    {
    }

    ~DofT()
    {
    }

    // Clear all internal structures.
    void clear();


    // Call to allocate static structures before using insert_dof(...).
    void init(unsigned int num_elements, unsigned int local_dimension);

    // Call to allocate static structures before using insert_dof(...).
    void init(unsigned int num_elements, unsigned int local_dimension, std::tr1::shared_ptr<unsigned int> loc2glob);

    // Update the internal structures with a new dof.
    unsigned int insert_dof(unsigned int e, unsigned int i, D Li);

    // Get the local to global mapping array.
    std::tr1::shared_ptr<unsigned int> get_loc2glob_array() const
    {
      return loc2glob_array;
    }


    // Return the number of elements inserted.
    unsigned int num_elements() const
    { return _num_elements; }

    // Return the number of global dofs inserted.
    unsigned int global_dimension() const
    { return _global_dimension; }

    // Return the number of dofs per elements.
    unsigned int local_dimension() const
    { return _local_dimension; }


    // Return the global index for local dof i in element e.
    unsigned int glob_index(unsigned int e, unsigned int i) const;

    // Return the global index for dof Lj represented with the templated type D.
    unsigned int glob_index(D Lj) const;

    // Return the dof (represented with the templated type D) for global index j.
    D glob_dof(unsigned int j) const;

    // Fill a vector with all the (element, index) pairs corresponding the dof with to this global index.
    void glob2loc(unsigned int j, vector_ii & loc) const;


    // Build loc2glob_array from loc2glob_map for future efficient lookup.
    void build_loc2glob();

  };


  template <class D>
  void
  DofT<D>::clear()
  {
    _global_dimension = 0;
    _num_elements     = 0;
    _local_dimension  = 0;

    dof2glob_map.clear();
    loc2glob_array.reset();
    loc2glob_map.clear();
    glob2dof_map.clear();
    glob2loc_map.clear();
  }


  template<class T>
  class array_deleter
  {
  public:
    void operator() (T *p_)
    {
      delete [] p_;
    }
  };
  
  // constructing a shared_ptr to an array:
  template<class T>
  std::tr1::shared_ptr<T> create_shared_array(unsigned int n)
  {
    T * ptr = new T[n];
    std::tr1::shared_ptr<T> sptr(ptr, array_deleter<T>());
    return sptr;
  }


  template <class D>
  void
  DofT<D>::init(unsigned int num_elements, unsigned int local_dimension)
  {
    if( !create_loc2glob_array )
    {
      std::cerr << "WARNING: Calling init with create_loc2glob_array == false makes little sense." << std::endl;
      std::cerr << "         Check your program logic." << std::endl;
    }
    create_loc2glob_array = true;

    _num_elements    = num_elements;
    _local_dimension = local_dimension;

    loc2glob_array = create_shared_array<unsigned int>(_num_elements*_local_dimension);
  }

  template <class D>
  void
  DofT<D>::init(unsigned int num_elements, unsigned int local_dimension, std::tr1::shared_ptr<unsigned int> loc2glob)
  {
    if( !create_loc2glob_array )
    {
      std::cerr << "WARNING: Calling init with create_loc2glob_array == false makes little sense." << std::endl;
      std::cerr << "         Check your program logic." << std::endl;
    }
    create_loc2glob_array = true;

    _num_elements    = num_elements;
    _local_dimension = local_dimension;

    loc2glob_array   = loc2glob;
  }

  template <class D>
  unsigned int
  DofT<D>::insert_dof(unsigned int e, unsigned int i, D Li)
  {
    if (e+1 > _num_elements)
    {
        _num_elements = e+1;
        if( create_loc2glob_array )
        {
            throw std::runtime_error("In this version of DofT we assume that the number of elements are predefined!");
        }
    }
    if (i+1 > _local_dimension)
    {
        _local_dimension = i+1;
        if( create_loc2glob_array )
        {
            throw std::runtime_error("In this version of DofT we assume that the local dimension is predefined!");
        }
    }

    // the return value
    unsigned int return_dof;

    // check if the dof is new
    typename std::map< D, unsigned int >::iterator index_iter = dof2glob_map.find(Li);

    if( index_iter != dof2glob_map.end() )
    {
      // reuse global index
      return_dof = index_iter->second;
    }
    else
    {
      // pick a new global index
      return_dof = _global_dimension;

      // count inserted global indices
      _global_dimension++;

      // the central "D -> global index" map
      dof2glob_map[Li] = return_dof;

      if ( create_glob2dof_map )
      {
        std::pair<unsigned int, D> p(return_dof, Li);
        glob2dof_map.insert(p);
        //glob2dof_map[return_dof] = Li;
      }

      if ( create_glob2loc_map )
      {
        // initialize with empty vector
        glob2loc_map[return_dof] = vector_ii();
        glob2loc_map[return_dof].reserve(_local_dimension);
      }
    }

    if ( create_glob2loc_map )
    {
      glob2loc_map[return_dof].push_back(pair_ii(e, i));
    }

    if( create_loc2glob_map )
    {
      loc2glob_map[pair_ii(e, i)] = return_dof;
    }

    if( create_loc2glob_array )
    {
      unsigned int * l2g = loc2glob_array.get();
      unsigned int k = e*_local_dimension + i;
      l2g[k] = return_dof;
    }

    return return_dof;
  }



  template <class D>
  unsigned int
  DofT<D>::glob_index(unsigned int e, unsigned int i) const
  {
    if ( create_loc2glob_array )
    {
      if( e >= _num_elements )
      {
        throw std::runtime_error("Invalid element index.");
      }
      if( i >= _local_dimension )
      {
        throw std::runtime_error("Invalid local index.");
      }
      unsigned int * l2g = loc2glob_array.get();
      return l2g[e*_local_dimension + i];
    }
    else
    {
      if ( !create_loc2glob_map )
      {
        throw std::runtime_error("This structure has not been created, you must turn on the create_loc2glob_map flag before initialization!");
      }
    
      pair_ii index(e, i);
      std::map< pair_ii, unsigned int >::const_iterator res = loc2glob_map.find(index);
      
      if ( res == loc2glob_map.end() )
      {
        throw std::runtime_error("In glob_index(e,i): Not found");
      }

      return res->second;
    }
  }


  template <class D>
  unsigned int
  DofT<D>::glob_index(D Lj) const
  {
    typename std::map< D, unsigned int >::const_iterator res = dof2glob_map.find(Lj);

    if ( res == dof2glob_map.end() )
    {
      throw std::runtime_error("In glob_index(Lj): Not found");
    }

    return res->second;
  }


  template <class D>
  D
  DofT<D>::glob_dof(unsigned int j) const
  {
    if ( !create_glob2dof_map )
    {
      throw std::runtime_error("This structure has not been created, you must turn on the create_glob2dof_map flag before initialization!");
    }

    typename std::map<unsigned int, D>::const_iterator iter = glob2dof_map.find(j);

    if ( iter == glob2dof_map.end() )
    {
      //throw std::runtime_error("In glob_dof(j): Not found");
      std::cerr << "In glob_dof(j): Not found" << std::endl;
      return D();
    }

    return iter->second;
  }


  template <class D>
  void
  DofT<D>::glob2loc(unsigned int j, vector_ii & loc) const
  {
    if ( !create_glob2loc_map )
    {
      throw std::runtime_error("This structure has not been created, you must turn on the create_glob2loc_map flag before initialization!");
    }

    typename std::map<unsigned int, vector_ii>::const_iterator it = glob2loc_map.find(j);
    vector_ii::const_iterator b = it->second.begin();
    vector_ii::const_iterator e = it->second.end();
    loc.assign( b, e );
  }


  template <class D>
  void
  DofT<D>::build_loc2glob()
  {
    if(create_loc2glob_array)
    {
      std::cerr << "WARNING: Calling build_loc2glob with create_loc2glob_array == true makes little sense." << std::endl;
      std::cerr << "         Doing nothing now. Check your program logic." << std::endl;
      return;
    }

    if(!create_loc2glob_map)
    {
      throw std::runtime_error("This structure has not been created, you must turn on the create_loc2glob_map flag before initialization!");
    }

    loc2glob_array = create_shared_array<unsigned int>(_num_elements * _local_dimension);

    typename std::map< pair_ii, unsigned int >::iterator iter;
    unsigned int * l2g = loc2glob_array.get();
    for(iter = loc2glob_map.begin(); iter != loc2glob_map.end(); iter++)
    {
      unsigned int e = iter->first.first;
      unsigned int j = iter->first.second;
      unsigned int k = e * _local_dimension + j;

      l2g[k] = iter->second;
    }
  }

} // namespace sfc

#endif

"""

implementation = ""
