/*
 * Fast index for tag data, based on integer indexes
 *
 * Copyright (C) 2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <wibble/exception.h>
#include <tagcoll/diskindex/mmap.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

// using namespace std;
using namespace wibble;

namespace tagcoll {
namespace diskindex {

MasterMMap::MasterMMap() : m_fd(-1), m_buf(0) {}

MasterMMap::MasterMMap(const std::string& filename)
	: m_filename(filename), m_fd(-1), m_buf(0)
{
	// If init throws here, the destructor isn't called (since we're in the
	// constructor), so we do the cleanup and rethrow.
	try {
		init(filename);
	} catch (...) {
		if (m_buf)
			munmap((void*)m_buf, m_size);
		if (m_fd != -1)
			close(m_fd);
		throw;
	}
}

MasterMMap::~MasterMMap()
{
	// Unmap and close the file
	if (m_buf)
		munmap((void*)m_buf, m_size);
	if (m_fd != -1)
		close(m_fd);
}

void MasterMMap::init(const std::string& filename)
{
	m_filename = filename;

	// Open the file
	if ((m_fd = open(m_filename.c_str(), O_RDONLY)) == -1)
		throw wibble::exception::System("opening index file " + m_filename);

	off_t size = lseek(m_fd, 0, SEEK_END);
	if (size == (off_t)-1)
		throw wibble::exception::System("reading the size of index file " + m_filename);
	m_size = size;

	if (m_size == 0)
		throw wibble::exception::Consistency("ensuring that there is data in the index",
									"the mmap index file " + filename + " is empty");

	// Map the file into memory
	if ((m_buf = (const char*)::mmap(0, m_size, PROT_READ, MAP_PRIVATE, m_fd, 0)) == MAP_FAILED)
		throw wibble::exception::System("mmapping file " + m_filename);
}


MMap::MMap()
	: m_master(0), m_buf(0), m_size(0) {}

MMap::MMap(const char* buf, int size)
{
	init(buf, size);
}

MMap::MMap(const MasterMMap& master, size_t idx)
{
	init(master, idx);
}

void MMap::init(const char* buf, int size)
{
	m_master = 0;
	m_buf = buf;
	m_size = size;
}

void MMap::init(const MasterMMap& master, size_t idx)
{
	m_master = &master;
	m_buf = master.m_buf;
	m_size = *(int*)m_buf;

	// Jump to the idx-th index
	for (size_t i = 0; i < idx; i++)
	{
		m_buf = m_buf + m_size + sizeof(int);
		if (m_buf > master.m_buf + master.m_size)
			throw wibble::exception::ValOutOfRange<size_t>("index ID", idx, 0, i, "looking for subindex in " + master.m_filename);
		m_size = *(int*)m_buf;
	}

	// Work with the subindex data, skipping the subindex chain size word
	m_buf += sizeof(int);
}


MasterMMapIndexer::MasterMMapIndexer(const std::string& filename)
	: finalname(filename)
{
	// Create a temporary file next to the target file
	char name[filename.size() + 8];
	memcpy(name, filename.data(), filename.size());
	memcpy(name + filename.size(), ".XXXXXX", 8);

	fd = mkstemp(name);
	tmpname = name;
	if (fd == -1)
		throw wibble::exception::System("creating temporary file " + tmpname);

}

MasterMMapIndexer::~MasterMMapIndexer()
{
	if (fd != -1)
		close(fd);

	if (!tmpname.empty())
		unlink(tmpname.c_str());
}

void MasterMMapIndexer::commit()
{
	// Give the file the right permissions according to umask

	// Read the current umask
	mode_t mask = umask(0);
	umask(mask);
	// Set the file permissions
	if (fchmod(fd, 0666 & ~mask) == -1)
		throw wibble::exception::System("setting permissions on file " + tmpname);

	// Close the file
	close(fd);
	fd = -1;

	// Rename to the final file name, performing the atomic update
	if (rename(tmpname.c_str(), finalname.c_str()) == -1)
		throw wibble::exception::System("renaming file " + tmpname + " into " + finalname);
	tmpname.clear();
}

void MasterMMapIndexer::append(const MMapIndexer& idx)
{
	// Get the encoded data
	int size = idx.encodedSize();
	char buf[size];
	idx.encode(buf);

	// Write one byte at the end to actually resize the file
	if (::write(fd, &size, sizeof(int)) != sizeof(int))
		throw wibble::exception::File(tmpname, "writing size word");

	// Write the index data
	if (::write(fd, buf, size) != size)
		throw wibble::exception::File(tmpname, "writing subindex");
}

}
}

#include <wibble/exception.tcc>
// vim:set ts=4 sw=4:
