/*
 * Simple MPEG/DVB parser to achieve network/service information without initial tuning data
 *
 * Copyright (C) 2006, 2007, 2008, 2009 Winfried Koehler 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 * Or, point your browser to http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 *
 * The author can be reached at: handygewinnspiel AT gmx DOT de
 *
 * The project's page is http://wirbel.htpc-forum.de/w_scan/index2.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "extended_frontend.h"
#include "scan.h"
#include "dump-dvbscan.h"
#include "dvbscan.h"
#include "countries.h"
#include "satellites.h"


/******************************************************************************
 * print initial tuning data for dvbscan. Nevertheless it should be also
 * reusable for 'w_scan -I <file>', therefore a part of the data might be
 * used later in w_scan.
 *****************************************************************************/

void dvbscan_dump_tuningdata (  FILE *f,
                                struct transponder *t,
                                uint16_t index,
                                struct w_scan_flags * flags) {
        struct tuning_parameters *p = &t->param;
        const char * network_name = t->network_name; 
        if (index == 0) {
                struct tm * ti;
                time_t rawtime;
                time(&rawtime);
                ti = localtime(&rawtime);
                fprintf (f, "#------------------------------------------------------------------------------\n");
                fprintf (f, "# file automatically generated by w_scan\n");
                fprintf (f, "# (http://wirbel.htpc-forum.de/w_scan/index2.html)\n");
                fprintf (f, "#! <w_scan> %d %u %u %s %s </w_scan>\n",
                                flags->version,
                                flags->tuning_timeout,
                                flags->filter_timeout,
                                scantype_to_txt(flags->scantype),
                                flags->scantype == SCAN_SATELLITE?
                                        satellite_to_short_name(flags->list_id):
                                        country_to_short_name(flags->list_id));
                fprintf (f, "#------------------------------------------------------------------------------\n");

                if (flags->scantype == SCAN_SATELLITE)
                        fprintf (f, "# satellite            : %s\n",
                                satellite_to_short_name(flags->list_id));
                else
                        fprintf (f, "# location and provider: <add description here>\n");
                fprintf (f, "# date (yyyy-mm-dd)    : %.04d-%.02d-%.02d\n",
                                        ti->tm_year + 1900, ti->tm_mon + 1, ti->tm_mday);
                fprintf (f, "# provided by (opt)    : <your name or email here>\n");
                fprintf (f, "#\n");

                switch (flags->scantype) {
                        case SCAN_TERRCABLE_ATSC:
                                fprintf (f, "# A[2] <freq> <mod> [# comment]\n");
                                break;
                        case SCAN_CABLE:
                                fprintf (f, "# C[2] [plp_id] [data_slice_id] [system_id] <freq> <sr> <fec> <mod> [# comment]\n");
                                break;
                        case SCAN_TERRESTRIAL:
                                fprintf (f, "# T[2] [plp_id] [system_id] <freq> <bw> <fec_hi> <fec_lo> <mod>");
                                fprintf (f, " <tm> <guard> <hi> [# comment]\n");
                                break;
                        case SCAN_SATELLITE:
                                fprintf (f, "# S[2] <freq> <pol> <sr> <fec> [ro] [mod] [# comment]\n");
                                break;
                        default:
                                fatal("%s (%d): UNKNOWN SCAN TYPE %d\n", __FUNCTION__, __LINE__, flags->scantype);
                        };
                fprintf (f, "#------------------------------------------------------------------------------\n");
                }       /* end if index == 0 */
        switch (flags->scantype) {
                case SCAN_TERRCABLE_ATSC:
                        fprintf (f, "A ");
                        fprintf (f, "%9i ",  p->frequency);
                        fprintf (f, "%8s", atsc_mod_to_txt(p->u.atsc.modulation));
                        break;
                case SCAN_CABLE:
                        fprintf (f, "C ");
                        if (p->u.cable.delivery_system == SYS_DVBC2)
                           fprintf (f, "2 %u %u %u", t->pids.plp_id, t->pids.data_slice_id, t->pids.system_id);
                        fprintf (f, "%9i ",  p->frequency);
                        fprintf (f, "%7i ",  p->u.cable.symbol_rate);
                        fprintf (f, "%4s ",  cable_fec_to_txt(p->u.cable.fec_inner));
                        fprintf (f, "%8s", cable_mod_to_txt(p->u.cable.modulation));
                        break;
                case SCAN_TERRESTRIAL:
                        fprintf (f, "T");
                        if (p->u.terr.delivery_system == SYS_DVBT2)
                           fprintf (f, "2 %u %u", t->pids.plp_id, t->pids.system_id);
                        fprintf (f, " %9i ",  p->frequency);
                        fprintf (f, "%4s ", terr_bw_to_txt(p->u.terr.bandwidth));
                        fprintf (f, "%4s ", terr_fec_to_txt(p->u.terr.code_rate_HP));
                        fprintf (f, "%4s ", terr_fec_to_txt(p->u.terr.code_rate_LP));
                        fprintf (f, "%8s ", terr_mod_to_txt(p->u.terr.constellation));
                        fprintf (f, "%4s ", terr_transmission_to_txt(p->u.terr.transmission_mode));
                        fprintf (f, "%4s ", terr_guard_to_txt(p->u.terr.guard_interval));
                        fprintf (f, "%4s",  terr_hierarchy_to_txt(p->u.terr.hierarchy_information));
                        break;
                case SCAN_SATELLITE:
                        fprintf (f, "%-2s ", sat_delivery_system_to_txt(p->u.sat.modulation_system));
                        fprintf (f, "%8i ",  p->frequency);
                        fprintf (f, "%1s ",  sat_pol_to_txt(p->u.sat.polarization));
                        fprintf (f, "%8i ",  p->u.sat.symbol_rate);
                        fprintf (f, "%4s",  sat_fec_to_txt(p->u.sat.fec_inner));
                        if (p->u.sat.modulation_system != SYS_DVBS) {
                                fprintf (f, " %2s ",  sat_rolloff_to_txt(p->u.sat.rolloff));
                                fprintf (f, "%6s",  sat_mod_to_txt(p->u.sat.modulation_type));
                                }
                        break;
                default:
                        ;
                };
        if (network_name != NULL)
                fprintf (f, "\t# %s", network_name);
        fprintf(f, "\n");
}
