/*
 * Helper functions to debug models
 *
 * Copyright 2009 Canonical Ltd.
 *
 * Authors:
 * - Aurélien Gâteau <aurelien.gateau@canonical.com>
 *
 * License: GPL v3
 */
#ifndef MODELDUMP_H
#define MODELDUMP_H

// Qt
#include <QHash>
#include <QModelIndex>
#include <QString>
#include <QVariant>

class QAbstractItemModel;

namespace ModelDump
{

/**
 * Abstract class defining how to dump a role
 */
class AbstractRoleDumper
{
public:
    virtual ~AbstractRoleDumper() {}

    virtual QString role() const = 0;
    virtual QString value(const QVariant& variant) const = 0;
};

class SimpleRoleDumper : public AbstractRoleDumper
{
public:
    SimpleRoleDumper(const char* name)
    : mName(name)
    {}

    virtual QString role() const { return mName; }
    virtual QString value(const QVariant& variant) const
    {
        return variant.toString();
    }

private:
    QString mName;
};

template <class T>
class PointerRoleDumper : public AbstractRoleDumper
{
public:
    PointerRoleDumper(const char* name)
    : mName(name)
    {}

    QString role() const { return mName; }
    QString value(const QVariant& variant) const
    {
        T* ptr = variant.value<T*>();
        return "0x" + QString::number((qulonglong)(ptr), 16);
    }

private:
    QString mName;
};

typedef QHash<int, AbstractRoleDumper*> RoleDumperHash;

/**
 * Dump a model to stderr, using hash to represent the different roles.
 * If root is precised, start dumping this part only.
 */
void dump(const RoleDumperHash& hash, const QAbstractItemModel* model, const QModelIndex& root = QModelIndex());

}; // namespace

#endif /* MODELDUMP_H */
