// ---------------------------------------------------------------------------
// - Lstack.cpp                                                              -
// - afnix:sps module - literal stack class implementation                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Real.hpp"
#include "Lstack.hpp"
#include "Vector.hpp"
#include "Spssid.hxx"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Exception.hpp"
#include "QuarkZone.hpp"
#include "InputStream.hpp"
#include "OutputStream.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the literal stack type serial code
  static const t_byte LSTK_BOOL_TYPE = 0x00U;
  static const t_byte LSTK_INTG_TYPE = 0x01U;
  static const t_byte LSTK_REAL_TYPE = 0x02U;
  static const t_byte LSTK_OTHR_TYPE = 0x03U;

  // map a literal stack as a serial byte
  static inline t_byte lstk_to_byte (const Lstack::t_lstk type) {
    t_byte result = 0x00U;
    switch (type) {
    case Lstack::LSTK_BOOL:
      result = LSTK_BOOL_TYPE;
      break;
    case Lstack::LSTK_INTG:
      result = LSTK_INTG_TYPE;
      break;
    case Lstack::LSTK_REAL:
      result = LSTK_REAL_TYPE;
      break;
    case Lstack::LSTK_OTHR:
      result = LSTK_OTHR_TYPE;
      break;
    }
    return result;
  }

  // map a serial byte to a literal stack type
  static inline Lstack::t_lstk lstk_to_type (const t_byte code) {
    Lstack::t_lstk result = Lstack::LSTK_OTHR;
    switch (code) {
    case LSTK_BOOL_TYPE:
      result = Lstack::LSTK_BOOL;
      break;
    case LSTK_INTG_TYPE:
      result = Lstack::LSTK_INTG;
      break;
    case LSTK_REAL_TYPE:
      result = Lstack::LSTK_REAL;
      break;
    case LSTK_OTHR_TYPE:
      result = Lstack::LSTK_BOOL;
      break;
    default:
      throw Exception ("lstack-error", "cannot convert serial code to type");
      break;
    }
    return result;
  }

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create an empty literal stack

  Lstack::Lstack (void) {
    d_size = 0L;
    d_slen = 0L;
    d_lidx = -1L;
    d_type = LSTK_OTHR;
    p_lstk = nilp;
  }

  // create a literal stack by size

  Lstack::Lstack (const long size) {
    if (size < 0L) throw Exception ("size-error","negative literal stack size");
    d_size = size;
    d_slen = 0;
    d_lidx = -1L;
    d_type = LSTK_OTHR;
    p_lstk = new Literal*[d_size];
    for (long k = 0; k < d_size; k++) p_lstk[k] = nilp;
  }

  // copy constructor for this literal stack

  Lstack::Lstack (const Lstack& that) {
    that.rdlock ();
    try {
      // copy arguments
      d_size = that.d_slen;
      d_slen = that.d_slen;
      d_lidx = that.d_lidx;
      d_type = that.d_type;
      switch (d_type) {
      case LSTK_BOOL:
	p_bstk = nilp;
	if (d_slen > 0L) {
	  p_bstk = new bool[d_size];
	  for (long k = 0L; k < d_slen; k++) p_bstk[k] = that.p_bstk[k];
	}
	break;
      case LSTK_INTG:
	p_istk = nilp;
	if (d_slen > 0L) {
	  p_istk = new long[d_size];
	  for (long k = 0L; k < d_slen; k++) p_istk[k] = that.p_istk[k];
	}
	break;
      case LSTK_REAL:
	p_rstk = nilp;
	if (d_slen > 0L) {
	  p_rstk = new t_real[d_size];
	  for (long k = 0L; k < d_slen; k++) p_rstk[k] = that.p_rstk[k];
	}
	break;
      case LSTK_OTHR:
	p_lstk = nilp;
	if (d_slen > 0L) {
	  p_lstk = new Literal*[d_size];
	  for (long k = 0L; k < d_slen; k++) {
	    Literal* lobj = that.p_lstk[k] == nilp ? nilp :
	      dynamic_cast <Literal*> (that.p_lstk[k]->clone());
	    Object::iref (p_lstk[k] = lobj);
	  }
	}
	break;	
      }
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this literal stack
  
  Lstack::~Lstack (void) {
    clear ();
  }

  // return the class name

  String Lstack::repr (void) const {
    return "Lstack";
  }

  // return a clone of this object

  Object* Lstack::clone (void) const {
    return new Lstack (*this);
  }

  // clear this literal stack

  void Lstack::clear (void) {
    wrlock ();
    try {
      // clean the literals first
      if ((d_type == LSTK_OTHR) && (d_slen > 0)) {
	for (long k = 0; k < d_slen; k++) Object::dref (p_lstk[k]);
      }
      // clean the array
      switch (d_type) {
      case LSTK_BOOL:
	delete [] p_lstk;
	p_lstk = nilp;
	break;
      case LSTK_INTG:
	delete [] p_istk;
	p_istk = nilp;
	break;
      case LSTK_REAL:
	delete [] p_rstk;
	p_rstk = nilp;
	break;
      case LSTK_OTHR:
	delete [] p_lstk;
	p_lstk = nilp;
	break;
      }
      d_size = 0L;
      d_slen = 0L;
      d_lidx = 1L;
      d_type = LSTK_OTHR;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a literal representation of this stack

  String Lstack::toliteral (void) const {
    rdlock ();
    try {
      String result =  "";
      if ((d_slen == 0) || (d_lidx == -1L)) {
	unlock ();
	return result;
      }
      if (d_type == LSTK_BOOL) {
	Boolean bval = p_bstk[d_lidx];
	result = bval.toliteral ();
      }
      if (d_type == LSTK_INTG) {
	Integer ival = p_istk[d_lidx];
	result = ival.toliteral ();
      }
      if (d_type == LSTK_REAL) {
	Real rval = p_rstk[d_lidx];
	result = rval.toliteral ();
      }
      if (d_type == LSTK_OTHR) {
	Literal* lobj = p_lstk[d_lidx];
	if (lobj != nilp) result = lobj->toliteral ();
      }	  
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a literal representation of this stack

  String Lstack::tostring (void) const {
    rdlock ();
    try {
      String result =  "";
      if ((d_slen == 0) || (d_lidx == -1L)) {
	unlock ();
	return result;
      }
      if (d_type == LSTK_BOOL) {
	Boolean bval = p_bstk[d_lidx];
	result = bval.tostring ();
      }
      if (d_type == LSTK_INTG) {
	Integer ival = p_istk[d_lidx];
	result = ival.tostring ();
      }
      if (d_type == LSTK_REAL) {
	Real rval = p_rstk[d_lidx];
	result = rval.tostring ();
      }
      if (d_type == LSTK_OTHR) {
	Literal* lobj = p_lstk[d_lidx];
	if (lobj != nilp) result = lobj->tostring ();
      }	  
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a styled representation of this stack

  String Lstack::format (const Style& lstl) const {
    rdlock ();
    try {
      String result =  "";
      if ((d_slen == 0) || (d_lidx == -1L)) {
	unlock ();
	return result;
      }
      if (d_type == LSTK_INTG) {
	long ival = p_istk[d_lidx];
	result = lstl.format (ival);
      }
      if (d_type == LSTK_REAL) {
	t_real rval = p_rstk[d_lidx];
	result = lstl.format (rval);
      } else {
	result = tostring ();
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the literal stack serial code

  t_byte Lstack::serialid (void) const {
    return SERIAL_LSTK_ID;
  }

  // serialize this literal stack

  void Lstack::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      // write the stack length
      Serial::wrlong (d_slen, os);
      // write the stack index
      Serial::wrlong (d_lidx, os);
      // write the stack code
      os.write ((char) lstk_to_byte (d_type));
      // map the stack data
      if (d_type == LSTK_BOOL) Serial::wrbool (d_slen, p_bstk, os);
      if (d_type == LSTK_INTG) Serial::wrlong (d_slen, p_istk, os);
      if (d_type == LSTK_REAL) Serial::wrreal (d_slen, p_rstk, os);
      if ((d_slen > 0) && (d_type == LSTK_OTHR)) {
	for (long k = 0; k < d_slen; k++) {
	  Literal* lobj = p_lstk[k];
	  if (lobj == nilp) {
	    Serial::wrnilid (os);
	  } else {
	    lobj->serialize (os);
	  }
	}
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this literal stack

  void Lstack::rdstream (InputStream& is) {
    wrlock ();
    try {
      // clear the stack 
      clear ();
      // get the stack length
      d_size = d_slen = Serial::rdlong (is);
      // get the stack index
      d_lidx = Serial::rdlong (is);
      // get the stack type
      d_type = lstk_to_type ((t_byte) is.read ());
      // deserialize data block
      if (d_type == LSTK_BOOL) {
	p_bstk = (d_slen == 0L) ? nilp : Serial::rdbool (is, d_size);
      }
      if (d_type == LSTK_INTG) {
	p_istk = (d_slen == 0L) ? nilp : Serial::rdlong (is, d_size);
      }
      if (d_type == LSTK_REAL) {
	p_rstk = (d_slen == 0L) ? nilp : Serial::rdreal (is, d_size);
      }
      if (d_type == LSTK_OTHR) {
	p_lstk = new Literal*[d_size];
	for (long k = 0; k < d_slen; k++) {
	  Object*  sobj = Serial::deserialize (is);
	  Literal* lobj = dynamic_cast <Literal*> (sobj);
	  if ((sobj != nilp) && (lobj == nilp)) {
	    throw Exception ("lastack-error", "invalid deserialized object");
	    Object::iref (p_lstk[k] = lobj);
	  }
	}
      }
      unlock ();
    } catch (...) {
      clear  ();
      unlock ();
      throw;
    }
  }

  // assign a literal stack  to this one
  
  Lstack& Lstack::operator = (const Lstack& that) {
    // check againt equal equal
    if (this == &that) return *this;
    // lock everything
    wrlock ();
    that.rdlock ();
    try {
      // clear the stack
      clear ();
      // copy arguments
      d_size = that.d_size;
      d_slen = that.d_slen;
      d_lidx = that.d_lidx;
      d_type = that.d_type;
      switch (d_type) {
      case LSTK_BOOL:
	p_bstk = nilp;
	if (d_slen > 0L) {
	  p_bstk = new bool[d_size];
	  for (long k = 0L; k < d_slen; k++) p_bstk[k] = that.p_bstk[k];
	}
	break;
      case LSTK_INTG:
	p_istk = nilp;
	if (d_slen > 0L) {
	  p_istk = new long[d_size];
	  for (long k = 0L; k < d_slen; k++) p_istk[k] = that.p_istk[k];
	}
	break;
      case LSTK_REAL:
	p_rstk = nilp;
	if (d_slen > 0L) {
	  p_rstk = new t_real[d_size];
	  for (long k = 0L; k < d_slen; k++) p_rstk[k] = that.p_rstk[k];
	}
	break;
      case LSTK_OTHR:
	p_lstk = nilp;
	if (d_slen > 0L) {
	  p_lstk = new Literal*[d_size];
	  for (long k = 0L; k < d_slen; k++) {
	    Literal* lobj = that.p_lstk[k] == nilp ? nilp :
	      dynamic_cast <Literal*> (that.p_lstk[k]->clone());
	    Object::iref (p_lstk[k] = lobj);
	  }
	}
	break;	
      }
      that.unlock ();
      unlock ();
      return *this;
    } catch (...) {
      that.unlock ();
      unlock ();
      throw;
    }
  }

  // return true if the literal stack is empty

  bool Lstack::empty (void) const {
    rdlock ();
    try {
      bool result = (d_slen == 0L);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the number of element in this literal stack

  long Lstack::length (void) const {
    rdlock ();
    try {
      long result = d_slen;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // resize the literal stack

  void Lstack::resize (const long size) {
    wrlock ();
    try {
      // check for valid size
      if (size < 0L) {
	throw Exception ("lstack-error", "invalid negative size in resize");
      }
      // check for null size
      if (d_size == 0L) {
	d_size = (size == 0L) ? 256L : size;
	d_slen = 0L;
	d_lidx = -1L;
	switch (d_type) {
	case LSTK_BOOL:
	  p_bstk = new bool[d_size];
	  for (long k = 0L; k < d_size; k++) p_bstk[k] = false;
	  break;
	case LSTK_INTG:
	  p_istk = new long[d_size];
	  for (long k = 0L; k < d_size; k++) p_istk[k] = 0L;
	  break;
	case LSTK_REAL:
	  p_rstk = new t_real[d_size];
	  for (long k = 0L; k < d_size; k++) p_rstk[k] = 0.0;
	  break;
	case LSTK_OTHR:
	  p_lstk = new Literal*[d_size];
	  for (long k = 0L; k < d_size; k++) p_lstk[k] = nilp;
	  break;
	}
	unlock ();
	return;
      }
      if (size <= d_size) {
	unlock ();
	return;
      }
      if (d_type == LSTK_BOOL) {
	bool* bstk = new bool[size];
	for (long k = 0;      k < d_slen; k++) bstk[k] = p_bstk[k];
	for (long k = d_slen; k < size;   k++) bstk[k] = false;
	d_size = size;
	delete [] p_bstk; p_bstk = bstk;
      }
      if (d_type == LSTK_INTG) {
	long* istk = new long[size];
	for (long k = 0;      k < d_slen; k++) istk[k] = p_istk[k];
	for (long k = d_slen; k < size;   k++) istk[k] = 0L;
	d_size = size;
	delete [] p_istk; p_istk = istk;
      }
      if (d_type == LSTK_REAL) {
	t_real* rstk = new t_real[size];
	for (long k = 0;      k < d_slen; k++) rstk[k] = p_rstk[k];
	for (long k = d_slen; k < size;   k++) rstk[k] = 0.0;
	d_size = size;
	delete [] p_rstk; p_rstk = rstk;
      }
      if (d_type == LSTK_OTHR) {
	Literal** lstk = new Literal*[size];
	for (long k = 0;      k < d_slen; k++) lstk[k] = p_lstk[k];
	for (long k = d_slen; k < size;   k++) lstk[k] = nilp;
	d_size = size;
	delete [] p_lstk; p_lstk = lstk;
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the literal index

  void Lstack::setlidx (const long lidx) {
    wrlock ();
    try {
      if ((lidx < 0L) || (lidx >= d_slen)) {
	throw Exception ("lstack-error", "invalid literal index in set");
      }
      d_lidx = lidx;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the literal index

  long Lstack::getlidx (void) const {
    rdlock ();
    try {
      long result = d_lidx;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a boolean in this stack
  
  void Lstack::add (const bool bval) {
    wrlock ();
    try {
      // retype if needed
      if (d_slen == 0L) d_type = LSTK_BOOL;
      // check for valid type
      if (d_type != LSTK_BOOL) {
	throw Exception ("lstack-error", "invalid boolean type to add");
      }
      // check if we have to resize the stack
      if (d_slen + 1L >= d_size) resize (2 * d_size);
      p_bstk[d_slen++] = bval;
      d_lidx++;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // add a boolean in this stack
  
  void Lstack::add (const long ival) {
    wrlock ();
    try {
      // retype if needed
      if (d_slen == 0L) d_type = LSTK_INTG;
      // check for valid type
      if (d_type != LSTK_INTG) {
	throw Exception ("lstack-error", "invalid integer type to add");
      }
      // check if we have to resize the stack
      if (d_slen + 1L >= d_size) resize (2 * d_size);
      p_istk[d_slen++] = ival;
      d_lidx++;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a real in this stack
  
  void Lstack::add (const t_real rval) {
    wrlock ();
    try {
      // retype if needed
      if (d_slen == 0L) d_type = LSTK_REAL;
      // check for valid type
      if (d_type != LSTK_REAL) {
	throw Exception ("lstack-error", "invalid real type to add");
      }
      // check if we have to resize the stack
      if (d_slen + 1L >= d_size) resize (2 * d_size);
      p_rstk[d_slen++] = rval;
      d_lidx++;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a literal in this stack
  
  void Lstack::add (Literal* lval) {
    wrlock ();
    try {
      // check for boolean
      Boolean* bobj = dynamic_cast <Boolean*> (lval);
      if ((bobj != nilp) && ((d_type == LSTK_BOOL) || (d_slen == 0L))) {
	add (bobj->tobool ());
	unlock ();
	return;
      }	  
      // check for integer
      Integer* iobj = dynamic_cast <Integer*> (lval);
      if ((iobj != nilp) && ((d_type == LSTK_INTG) || (d_slen == 0L))) {
	add ((long) iobj->tolong ());
	unlock ();
	return;
      }
      // check for real
      Real* robj = dynamic_cast <Real*> (lval);
      if ((robj != nilp) && ((d_type == LSTK_REAL) || (d_slen == 0L))) {
	add (robj->toreal ());
	unlock ();
	return;
      }
      // check for other
      if ((d_type == LSTK_OTHR) && 
	  ((bobj == nilp) && (iobj == nilp) && (robj == nilp))) {
	// check if we have to resize the stack
	if (d_slen + 1L >= d_size) resize (2 * d_size);
	// add the literal
	Literal* lobj = (lval == nilp) ? nilp :
	  dynamic_cast <Literal*> (lval->clone ());
	Object::iref (p_lstk[d_slen++] = lobj);
	d_lidx++;
      } else {
	// check inconsistent add
	throw Exception ("lstack-error", "invalid object to add",
			 Object::repr (lval));
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set a literal at a certain position

  void Lstack::set (const long index, Literal* lval) {
    wrlock ();
    try {
      // check that the index is valid
      if (index >= d_slen) { 
	throw Exception ("index-error","in literal stack set");
      }
      // select valid literal
      Boolean* bobj = dynamic_cast <Boolean*> (lval);
      if ((bobj != nilp) && (d_type == LSTK_BOOL)) {
	p_bstk[index] = bobj->tobool ();
	unlock ();
	return;
      }
      Integer* iobj = dynamic_cast <Integer*> (lval);
      if ((iobj != nilp) && (d_type == LSTK_INTG)) {
	p_istk[index] = iobj->tolong ();
	unlock ();
	return;
      }
      Real* robj = dynamic_cast <Real*> (lval);
      if ((robj != nilp) && (d_type == LSTK_REAL)) {
	p_rstk[index] = robj->toreal ();
	unlock ();
	return;
      }
      if ((d_type == LSTK_OTHR) && 
	  (bobj == nilp) && (iobj == nilp) && (robj == nilp)) {
	p_lstk[index] = (lval == nilp) ? nilp :
	  dynamic_cast<Literal*> (lval->clone ());
      } else {
	throw Exception ("lstack-error", "invalid object type to set",
			 Object::repr (lval));
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get a literal at a certain position

  Literal* Lstack::get (const long index) const {
    rdlock ();
    try {
      // check that we are bounded
      if ((index < 0) || (index >= d_slen)) { 
	throw Exception ("index-error","in literal stack get");
      }
      Literal* result = nilp;
      switch (d_type) {
      case LSTK_BOOL:
	result = (p_bstk == nilp) ? nilp : new Boolean (p_bstk[index]);
	break;
      case LSTK_INTG:
	result = (p_istk == nilp) ? nilp : new Integer (p_istk[index]);
	break;
      case LSTK_REAL:
	result = (p_rstk == nilp) ? nilp : new Real (p_rstk[index]);
	break;
      case LSTK_OTHR:
	result = (p_lstk[index] == nilp) ? nilp :
	dynamic_cast <Literal*> (p_lstk[index]->clone ());
	break;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the first literal in this stack

  Literal* Lstack::first (void) const {
    rdlock ();
    try {
      Literal* result = get (0);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the last literal in this stack

  Literal* Lstack::last (void) const {
    rdlock ();
    try {
      Literal* result = get (d_slen-1L);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a new stack iterator

  Iterator* Lstack::makeit (void) {
    rdlock ();
    try {
      Iterator* result = new Lstackit (this);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 9;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ADD     = zone.intern ("add");
  static const long QUARK_GET     = zone.intern ("get");
  static const long QUARK_SET     = zone.intern ("set");
  static const long QUARK_LAST    = zone.intern ("last");
  static const long QUARK_FIRST   = zone.intern ("first");
  static const long QUARK_LENGTH  = zone.intern ("length");
  static const long QUARK_EMPTYP  = zone.intern ("empty-p");
  static const long QUARK_GETLIDX = zone.intern ("get-literal-index");
  static const long QUARK_SETLIDX = zone.intern ("set-literal-index");

  // create a new object in a generic way

  Object* Lstack::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check 0 argument
    if (argc == 0) return new Lstack;
    // check 1 argument
    if (argc == 1) {
      Object* obj = argv->get (0);
      // check for an integer
      Integer* iobj = dynamic_cast <Integer*> (obj);
      if (iobj !=nilp) {
        long size = iobj->tolong ();
        return new Lstack (size);
      }
      throw Exception ("type-error", "invalid object with lstack",
                       Object::repr (obj));
    }
    throw Exception ("argument-error", "too many argument for lstack");
    
  }

  // return true if the given quark is defined

  bool Lstack::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Serial::isquark (quark, hflg) : false;
      if (result == false) {
	result = hflg ? Literal::isquark (quark, hflg) : false;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark
  
  Object* Lstack::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LAST)    return last ();
      if (quark == QUARK_FIRST)   return first ();
      if (quark == QUARK_LENGTH)  return new Integer (length  ());
      if (quark == QUARK_EMPTYP)  return new Boolean (empty   ());
      if (quark == QUARK_GETLIDX) return new Integer (getlidx ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_GET) {
	long index = argv->getlong (0);
	return get (index);
      }
      if (quark == QUARK_ADD) {
	Object*   obj = argv->get (0);
	Literal* lval = dynamic_cast <Literal*> (obj);
	if ((obj != nilp) && (lval == nilp)) {
	  throw Exception ("type-error", "invalid object to add",
			   Object::repr (obj));
	}
	add (lval);
	return nilp;
      }
      if (quark == QUARK_SETLIDX) {
	long lidx = argv->getlong (0);
	setlidx (lidx);
	return nilp;
      }
    }

    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_SET) {
	long index = argv->getlong (0);
	Object* obj = argv->get (1);
	Literal* lval = dynamic_cast <Literal*> (obj);
	if ((obj != nilp) && (lval == nilp)) {
	  throw Exception ("type-error", "invalid object wih set",
			   Object::repr (obj));
	}
	set (index, lval);
	return nilp;
      }
    }
    // check the serial method
    if (Serial::isquark (quark, true) == true) {
      return Serial::apply (robj, nset, quark, argv);
    }
    // default to literal method
    return Literal::apply (robj, nset, quark, argv);
  }

  // -------------------------------------------------------------------------
  // - iterator section                                                      -
  // -------------------------------------------------------------------------

  // create a new literal stack iterator

  Lstackit::Lstackit (Lstack* sobj) {
    Object::iref (p_sobj = sobj);
    begin ();
  }

  // destroy this literal stack iterator

  Lstackit::~Lstackit (void) {
    Object::dref (p_sobj);
  }

  // return the class name

  String Lstackit::repr (void) const {
    return "Lstackit";
  }

  // reset the iterator to the begining

  void Lstackit::begin (void) {
    wrlock ();
    try {
      d_sidx = 0;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // reset the iterator to the end

  void Lstackit::end (void) {
    wrlock ();
    if (p_sobj != nilp) p_sobj->rdlock ();
    try {
      if (p_sobj != nilp) {
	d_sidx = (p_sobj->d_slen == 0) ? 0L : p_sobj->d_slen - 1L;
      } else {
	d_sidx = 0;
      }
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
    } catch (...) {
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
      throw;
    }
  }

  // go to the next object

  void Lstackit::next (void) {
    wrlock ();
    if (p_sobj != nilp) p_sobj->rdlock ();
    try {
      if (p_sobj != nilp) {
	if (++d_sidx >= p_sobj->d_slen) d_sidx = p_sobj->d_slen;
      } else {
	d_sidx = 0;
      }
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
    } catch (...) {
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
      throw;
    }
  }

  // go to the previous object

  void Lstackit::prev (void) {
    wrlock ();
    try {
      if (--d_sidx < 0) d_sidx = 0;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the object at the current position

  Object* Lstackit::getobj (void) const {
    rdlock ();
    if (p_sobj != nilp) p_sobj->rdlock ();
    try {
      Object* result = nilp;
      if ((p_sobj != nilp) && (d_sidx < p_sobj->d_slen)) {
	result = p_sobj->get (d_sidx);
      }
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
      return result;
    } catch (...) {
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
      throw;
    }
  }

  // return true if the iterator is at the end

  bool Lstackit::isend (void) const {
    rdlock ();
    if (p_sobj != nilp) p_sobj->rdlock ();
    try {
      bool result = false;
      if (p_sobj != nilp) {
	result = (d_sidx >= p_sobj->d_slen);
      }
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
      return result;
    } catch (...) {
      if (p_sobj != nilp) p_sobj->unlock ();
      unlock ();
      throw;
    }
  }
}
