// ---------------------------------------------------------------------------
// - TlsConnect.cpp                                                           -
// - afnix:tls service - tls connect class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "TlsShake.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "TlsConnect.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a tls connect by flag

  TlsConnect::TlsConnect (const bool sflg) {
    d_sflg = sflg;
  }

  // return the class name
  
  String TlsConnect::repr (void) const {
    return "TlsConnect";
  }

  // reset the tls connection object

  void TlsConnect::reset (void) {
    wrlock ();
    try {
      d_info.reset ();
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the connection info list

  Plist TlsConnect::getinfo (void) const {
    rdlock ();
    try {
      Plist result = d_info;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // perform a handshake operation

  TlsState* TlsConnect::connect (InputStream* is, OutputStream* os) {
    // check for nil
    if ((is == nilp) || (os == nilp)) return nilp;
    // connect and lock
    wrlock ();
    try {
      // reset the object
      reset ();
      // connect the tls socket
      TlsState* result = d_sflg ? getss (is, os) : nilp;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - protected section                                                     -
  // -------------------------------------------------------------------------

  // get a server state

  TlsState* TlsConnect::getss (InputStream* is, OutputStream* os) {
    // check for nil
    if ((is == nilp) || (os == nilp)) return nilp;
    // lock and connect
    wrlock ();
    TlsState*   ssta = nilp;
    TlsProto*   tlsp = nilp;
    TlsChello*  chlo = nilp;
    TlsMessage* shlo = nilp;
    try {
      // check for consistency
      if (d_sflg == false) {
	throw Exception ("tls-error", "invalid call to client hello get");
      }
      // create a server state by flag
      ssta = new TlsState (d_sflg, d_prms);
      Object::iref (ssta);
      // get the client hello message
      chlo = dynamic_cast <TlsChello*> (TlsProto().getchlo (is));
      if (chlo == nilp) {
	throw Exception ("tls-error", "cannot get client hello message");
      }
      // update the info list
      d_info += chlo->getinfo ();
      // update the state version
      ssta->setvers (chlo->getvmaj (), chlo->getvmin ());
      // update the selected cipher
      ssta->setcifr (chlo->getcifr ());
      // create a tls protocol by state
      tlsp = TlsProto::create (ssta);
      // create a server hello message and encode it
      shlo = tlsp->getshlo (ssta);
      if (shlo == nilp) {
	throw Exception ("tls-error", "cannot get server hello message");
      }
      tlsp->encode (os, shlo);
      // clean and unlock
      delete chlo;
      delete shlo;
      delete tlsp;
      Object::tref (ssta);
      unlock ();
      return ssta;
    } catch (...) {
      delete ssta;
      delete tlsp;
      delete chlo;
      delete shlo;
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 1;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_CONNECT = zone.intern ("connect");
 
  // create a new object in a generic way

  Object* TlsConnect::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 1 argument
    if (argc == 1) {
      bool sflg = argv->getbool (0);
      return new TlsConnect (sflg);
    }
    // too many arguments
    throw Exception ("argument-error", 
                     "too many argument with tls connect constructor");
  }
  
  // return true if the given quark is defined

  bool TlsConnect::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? TlsInfos::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark
  
  Object* TlsConnect::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_CONNECT) {
	Object* obj = argv->get (0);
	InputStream* is = dynamic_cast <InputStream*> (obj);
	if (is == nilp) {
	  throw Exception ("type-error", "invalid object as input stream",
			   Object::repr (obj));
	}
	obj = argv->get (1);
	OutputStream* os = dynamic_cast <OutputStream*> (obj);
	if (os == nilp) {
	  throw Exception ("type-error", "invalid object as output stream",
			   Object::repr (obj));
	}
	return connect (is, os);
      }
    }
    // call the info method
    return TlsInfos::apply (robj, nset, quark, argv);
  }
}
