/*
 * Copyright (c) 2022-2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>

namespace arm_conv {
namespace depthwise {

void sme2_fp32_planar_5x5_s2_4rows_mla_za_impl(
  const float *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const float *weights,
  const float *bias,
  float **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  float act_min,
  float act_max
)
{
  struct Args
  {
    const float *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const float *weights;
    const float *bias;
    long unsigned int input_cols, output_cols;
    float **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
    float clamp_min, clamp_max;
  };

  Args args = { inptr, ld_in_vl, pad_top, 11u - std::min(11u, pad_top + valid_input_rows), pad_left, weights, bias, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels, act_min, act_max };

  __asm__ __volatile__(
    "ldr x5, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "mov x20, #0xb\n"
    ".inst 0xd503477f  // SMSTART ZA\n"
    "sub x20, x20, x5\n"
    "ldr x6, [%x[args], %[offsetof_Args_pad_top]]\n"
    "ptrue p2.b\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "ld1rw { z2.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_min]]\n"
    "ldr x7, [%x[args], %[offsetof_Args_n_channels]]\n"
    "whilelt p1.s, XZR, x7\n"
    "whilelt p9.s, XZR, x20\n"
    "ld1rw { z3.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_max]]\n"
    "whilelt p8.s, XZR, x6\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "ldr x17, [%x[args], %[offsetof_Args_current_channel]]\n"
    "1:"  // Channel loop
    "ldr x20, [%x[args], %[offsetof_Args_bias]]\n"
    "fmov z28.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z28.s }, p1/Z, [x20, x17, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x16, [%x[args], %[offsetof_Args_input_cols]]\n"
    "sub x20, x16, #0x1\n"
    "orr x23, x20, %x[ld_in_col], LSL #18\n"
    "mov z29.d, z28.d\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "orr x23, x7, x23, LSL #20\n"
    "mov x22, #0xb\n"
    "ldr x14, [%x[args], %[offsetof_Args_inptr]]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "add x21, x6, x5\n"
    "lsl x20, %x[ld_in_row], #0x2\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "mov z30.d, z28.d\n"
    "mov z31.d, z28.d\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "mov x8, #0x0\n"
    "ldr x13, [%x[args], %[offsetof_Args_output_cols]]\n"
    "lsl x23, x23, #0x2\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "sub x22, x22, x21\n"
    "madd x20, x20, x6, x14\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "3:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b74a9c  // rprfm pldstrm, x23, [x20]\n"
    "add x20, x20, %x[ld_in_col], LSL #2\n"
    "bgt 3b\n"
    "ldr x23, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x20, %x[ld_in_row], #0x2\n"
    "msub x14, x6, x20, x14\n"
    ".inst 0xc0040f80  // mova za.d[x8, #0], { z28.d-z31.d }\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    ".inst 0xc0040f81  // mova za.d[x8, #1], { z28.d-z31.d }\n"
    "mov x22, #0x4\n"
    "ldp x11, x10, [x23], #0x10\n"
    ".inst 0xc0040f82  // mova za.d[x8, #2], { z28.d-z31.d }\n"
    "ldp x9, x28, [x20], #0x10\n"
    ".inst 0xc0040f83  // mova za.d[x8, #3], { z28.d-z31.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "ldp x27, x26, [x23], #0x10\n"
    "ldp x25, x24, [x20], #0x10\n"
    "cbz x21, 5f\n"
    "cmp x21, x22\n"
    "csel x20, x21, x22, LT\n"
    "sub x21, x21, x20\n"
    "sub x22, x22, x20\n"
    "cbz x21, 5f\n"
    ".inst 0xc0060c10  // mova { z16.d-z19.d }, za.d[x8, #0]\n"
    "and x22, x21, #0x1\n"
    "add x21, x21, #0x1\n"
    ".inst 0xc1a3c850  // fclamp { z16.s-z19.s }, z2.s, z3.s\n"
    "lsr x21, x21, #0x1\n"
    "sub x13, x13, x21\n"
    "4:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1w { z16.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    "st1w { z17.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z18.s }, p1, [x27]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z19.s }, p1, [x26]\n"
    "add x26, x26, x24, LSL #2\n"
    "bgt 4b\n"
    "5:"  // Left padding: End
    "adds XZR, x6, x5\n"
    "bne 12f\n"
    "cbz x22, 10f\n"
    "cmp x22, #0x1\n"
    "sub x16, x16, x22\n"
    "beq 9f\n"
    "cmp x22, #0x2\n"
    "beq 8f\n"
    "cmp x22, #0x3\n"
    "beq 7f\n"
    "6:"  // Unpadded: 4 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z9.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z10.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z11.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z12.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1341920  // fmla za.s[x8, 0], { z9.s-z12.s }, z4.s\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1371a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z7.s\n"
    "ld1w { z13.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1341940  // fmla za.s[x8, 0], { z10.s-z13.s }, z4.s\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1301aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z0.s\n"
    "ld1w { z14.s }, p1/Z, [x20]\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1341960  // fmla za.s[x8, 0], { z11.s-z14.s }, z4.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "7:"  // Unpadded: 3 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z22.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z7.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z8.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z9.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1351ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z5.s\n"
    "ld1w { z10.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f18e0  // fmla za.s[x8, 0], { z7.s-z10.s }, z15.s\n"
    "ld1w { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13f1ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z15.s\n"
    "ld1w { z11.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1371900  // fmla za.s[x8, 0], { z8.s-z11.s }, z7.s\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    ".inst 0xa04049ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13b1b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z11.s\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "8:"  // Unpadded: 2 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z14.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z15.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13a1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z10.s\n"
    ".inst 0xc1341a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z4.s\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13019c0  // fmla za.s[x8, 0], { z14.s-z17.s }, z0.s\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13719c1  // fmla za.s[x8, 1], { z14.s-z17.s }, z7.s\n"
    ".inst 0xa04049e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13a1a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z10.s\n"
    ".inst 0xc1381a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z8.s\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc13619e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z6.s\n"
    ".inst 0xa04149e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13819e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z8.s\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc13e1aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z14.s\n"
    ".inst 0xa14149e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1371aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z7.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "9:"  // Unpadded: 1 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z7.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z8.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z9.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z10.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13b18e0  // fmla za.s[x8, 0], { z7.s-z10.s }, z11.s\n"
    ".inst 0xc13518e1  // fmla za.s[x8, 1], { z7.s-z10.s }, z5.s\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1311a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z1.s\n"
    "ld1w { z11.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z15.s\n"
    ".inst 0xa04049e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ec  // ld1w { z12.s-z13.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13d1900  // fmla za.s[x8, 0], { z8.s-z11.s }, z13.s\n"
    ".inst 0xc1311901  // fmla za.s[x8, 1], { z8.s-z11.s }, z1.s\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc13e1a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z14.s\n"
    ".inst 0xa14149e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13e1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z14.s\n"
    "ld1w { z12.s }, p1/Z, [x20]\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc13f1921  // fmla za.s[x8, 1], { z9.s-z12.s }, z15.s\n"
    ".inst 0xa04149ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13f1920  // fmla za.s[x8, 0], { z9.s-z12.s }, z15.s\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "10:"  // Unpadded: 0 priming loads
    "cmp x16, #0x2\n"
    "blt 20f\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z22.s }, p1/Z, [x14]\n"
    "sub x16, x16, #0x2\n"
    "ld1w { z16.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "sub x13, x13, #0x1\n"
    "ld1w { z23.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "lsr x20, x16, #0x1\n"
    "ld1w { z17.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "cmp x20, x13\n"
    "ld1w { z24.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "csel x23, x20, x13, LT\n"
    "ld1w { z18.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "and x16, x16, #0x1\n"
    "ld1w { z19.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "sub x13, x13, x23\n"
    "ld1w { z26.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z27.s }, p1/Z, [x21]\n"
    "cbz x23, 19f\n"
    "11:"  // Unpadded: Main loop
    ".inst 0xc1391ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z9.s\n"
    "ld1w { z13.s }, p2/Z, [x15, #4, MUL VL]\n"
    "add x22, x14, %x[ld_in_row], LSL #2\n"
    "subs x23, x23, #0x1\n"
    ".inst 0xc13a1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z10.s\n"
    ".inst 0xa14149e1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1341ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z4.s\n"
    ".inst 0xa04049e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1361a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z6.s\n"
    "ld1w { z11.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1301a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z0.s\n"
    ".inst 0xa04149ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1371a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z7.s\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13d1ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z13.s\n"
    "ld1w { z4.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1311ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z1.s\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1381ae2  // fmla za.s[x8, 2], { z23.s-z26.s }, z8.s\n"
    ".inst 0xa04049ec  // ld1w { z12.s-z13.s }, pn10.b/Z, [x15]\n"
    "ldr x21, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13b1a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z11.s\n"
    "ld1w { z15.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z14.s\n"
    "ld1w { z22.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z6.s\n"
    "ld1w { z16.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "ld1w { z23.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1341b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z4.s\n"
    ".inst 0xc1301b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z0.s\n"
    ".inst 0xa0414aa6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc13c1b02  // fmla za.s[x8, 2], { z24.s-z27.s }, z12.s\n"
    "ld1w { z17.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "ld1w { z24.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0060c08  // mova { z8.d-z11.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    ".inst 0xa1404aa4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x21]\n"
    "addvl x21, x21, #5\n"
    ".inst 0xc1a3c848  // fclamp { z8.s-z11.s }, z2.s, z3.s\n"
    "st1w { z8.s }, p1, [x11]\n"
    "ld1w { z18.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13719e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z7.s\n"
    "add x11, x11, x9, LSL #2\n"
    ".inst 0xc13c19e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z12.s\n"
    ".inst 0xa1404aa7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x21]\n"
    "st1w { z9.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    ".inst 0xa1414aa6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    "addvl x21, x21, #5\n"
    "st1w { z10.s }, p1, [x27]\n"
    "add x27, x27, x25, LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z14.s\n"
    "st1w { z11.s }, p1, [x26]\n"
    ".inst 0xc13f1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z15.s\n"
    "ld1w { z19.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "add x26, x26, x24, LSL #2\n"
    ".inst 0xa0404aae  // ld1w { z14.s-z15.s }, pn10.b/Z, [x21]\n"
    ".inst 0xc13f1a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z15.s\n"
    ".inst 0xa1414aa4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    "addvl x21, x21, #5\n"
    ".inst 0xc13c1a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z12.s\n"
    "ld1w { z26.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    ".inst 0xa0404aac  // ld1w { z12.s-z13.s }, pn10.b/Z, [x21]\n"
    ".inst 0xc13d1ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z13.s\n"
    ".inst 0xa1414aa4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    "addvl x21, x21, #5\n"
    ".inst 0xc13c1ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z12.s\n"
    "ld1w { z20.s }, p1/Z, [x22]\n"
    ".inst 0xa1404aa7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x21]\n"
    ".inst 0xc13f1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z15.s\n"
    ".inst 0xa0414aaa  // ld1w { z10.s-z11.s }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc13b1a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z11.s\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z22.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    "bgt 11b\n"
    "b 19f\n"
    "12:"  // Padded
    "cbz x22, 17f\n"
    "cmp x22, #0x1\n"
    "sub x16, x16, x22\n"
    "beq 16f\n"
    "cmp x22, #0x2\n"
    "beq 15f\n"
    "cmp x22, #0x3\n"
    "beq 14f\n"
    "13:"  // Padded: 4 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z9.s }, p0/Z, [x14]\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z10.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z11.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z12.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    ".inst 0xc1341920  // fmla za.s[x8, 0], { z9.s-z12.s }, z4.s\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1371ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z7.s\n"
    "ld1w { z13.s }, p0/Z, [x20]\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1361940  // fmla za.s[x8, 0], { z10.s-z13.s }, z6.s\n"
    "addvl x15, x15, #5\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1361b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z6.s\n"
    "ld1w { z14.s }, p0/Z, [x20]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1361960  // fmla za.s[x8, 0], { z11.s-z14.s }, z6.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "14:"  // Padded: 3 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z22.s }, p0/Z, [x14]\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z9.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z10.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z11.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    ".inst 0xc1351ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z5.s\n"
    "ld1w { z12.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc13f1920  // fmla za.s[x8, 0], { z9.s-z12.s }, z15.s\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    ".inst 0xa14049e0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x15]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1381ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z8.s\n"
    "addvl x15, x15, #5\n"
    "ld1w { z13.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13f1940  // fmla za.s[x8, 0], { z10.s-z13.s }, z15.s\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13f1b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z15.s\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "15:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z16.s }, p0/Z, [x14]\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    ".inst 0xc13a1a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z10.s\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    ".inst 0xc1341a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z4.s\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0xc1301ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z0.s\n"
    ".inst 0xa14049e0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x15]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1371ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z7.s\n"
    ".inst 0xa14149e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1371a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z7.s\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1301a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z0.s\n"
    ".inst 0xa14049e5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13a1b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z10.s\n"
    ".inst 0xc1351b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z5.s\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    ".inst 0xa14049e5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc1351a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z5.s\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1301a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z0.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "16:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z19.s }, p0/Z, [x14]\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z8.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z9.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z10.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    ".inst 0xc13b1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z11.s\n"
    "ld1w { z11.s }, p0/Z, [x20]\n"
    ".inst 0xc1351a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z5.s\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    ".inst 0xc1311900  // fmla za.s[x8, 0], { z8.s-z11.s }, z1.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc13f1901  // fmla za.s[x8, 1], { z8.s-z11.s }, z15.s\n"
    ".inst 0xa14149e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc13e1a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z14.s\n"
    "ld1w { z12.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1351a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z5.s\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa14149e0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1381920  // fmla za.s[x8, 0], { z9.s-z12.s }, z8.s\n"
    ".inst 0xc1371921  // fmla za.s[x8, 1], { z9.s-z12.s }, z7.s\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    ".inst 0xa14049e0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc1381aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z8.s\n"
    ".inst 0xa04149ec  // ld1w { z12.s-z13.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13d1aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z13.s\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "17:"  // Padded: 0 priming loads
    "cmp x16, #0x2\n"
    "blt 20f\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z22.s }, p0/Z, [x14]\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z16.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z23.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z17.s }, p0/Z, [x21]\n"
    "mov x12, #0x4\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z24.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z18.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z25.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1w { z19.s }, p0/Z, [x21]\n"
    "sub x16, x16, #0x2\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "sub x13, x13, #0x1\n"
    "ld1w { z26.s }, p0/Z, [x21]\n"
    "lsr x20, x16, #0x1\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "cmp x20, x13\n"
    "ld1w { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z27.s }, p0/Z, [x21]\n"
    "csel x23, x20, x13, LT\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "and x16, x16, #0x1\n"
    "sub x13, x13, x23\n"
    "cbz x23, 19f\n"
    "18:"  // Padded: Main loop
    ".inst 0xc1391ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z9.s\n"
    "ld1w { z15.s }, p2/Z, [x15, #4, MUL VL]\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc13a1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z10.s\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "add x22, x14, %x[ld_in_row], LSL #2\n"
    "subs x23, x23, #0x1\n"
    ".inst 0xc1341ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z4.s\n"
    ".inst 0xa14049e5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1361a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z6.s\n"
    "ld1w { z1.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1301a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z0.s\n"
    ".inst 0xa04149ec  // ld1w { z12.s-z13.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1371a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z7.s\n"
    ".inst 0xa14049e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13f1ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z15.s\n"
    "ld1w { z16.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc13a1ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z10.s\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1351ae2  // fmla za.s[x8, 2], { z23.s-z26.s }, z5.s\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc1311a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z1.s\n"
    "ld1w { z0.s }, p2/Z, [x15, #4, MUL VL]\n"
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13c1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z12.s\n"
    "ld1w { z12.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1361a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z6.s\n"
    "ld1w { z17.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1w { z13.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1301b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z0.s\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc13a1b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z10.s\n"
    ".inst 0xa1414a81  // ld1w { z1.s, z9.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc13e1b02  // fmla za.s[x8, 2], { z24.s-z27.s }, z14.s\n"
    "ld1w { z18.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z14.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc0060c18  // mova { z24.d-z27.d }, za.d[x8, #0]\n"
    ".inst 0xa0404a80  // ld1w { z0.s-z1.s }, pn10.b/Z, [x20]\n"
    "add x8, x8, #0x1\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc1a3c858  // fclamp { z24.s-z27.s }, z2.s, z3.s\n"
    "ld1w { z19.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "st1w { z24.s }, p1, [x11]\n"
    "mov x12, #0x8\n"
    ".inst 0xc1391a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z9.s\n"
    ".inst 0xa0404a88  // ld1w { z8.s-z9.s }, pn10.b/Z, [x20]\n"
    "add x11, x11, x9, LSL #2\n"
    ".inst 0xc1311a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z1.s\n"
    ".inst 0xa0414a80  // ld1w { z0.s-z1.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    "st1w { z25.s }, p1, [x10]\n"
    "ld1w { z15.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1311980  // fmla za.s[x8, 0], { z12.s-z15.s }, z1.s\n"
    ".inst 0xc1391981  // fmla za.s[x8, 1], { z12.s-z15.s }, z9.s\n"
    "ld1w { z20.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xa0404a8a  // ld1w { z10.s-z11.s }, pn10.b/Z, [x20]\n"
    ".inst 0xc13b1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z11.s\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z26.s }, p1, [x27]\n"
    ".inst 0xa1414a80  // ld1w { z0.s, z8.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc1381a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z8.s\n"
    "add x27, x27, x25, LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "st1w { z27.s }, p1, [x26]\n"
    ".inst 0xa0404a88  // ld1w { z8.s-z9.s }, pn10.b/Z, [x20]\n"
    "mov x12, #0x0\n"
    ".inst 0xc13919a1  // fmla za.s[x8, 1], { z13.s-z16.s }, z9.s\n"
    "add x26, x26, x24, LSL #2\n"
    ".inst 0xa1414a81  // ld1w { z1.s, z9.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc13919a0  // fmla za.s[x8, 0], { z13.s-z16.s }, z9.s\n"
    "ld1w { z21.s }, p0/Z, [x22]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "ld1w { z22.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xa0404a8e  // ld1w { z14.s-z15.s }, pn10.b/Z, [x20]\n"
    ".inst 0xc13f1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z15.s\n"
    ".inst 0xa0414a80  // ld1w { z0.s-z1.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1311a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z1.s\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z16.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z23.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1w { z17.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z24.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z18.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z25.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1w { z19.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z26.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z27.s }, p0/Z, [x21]\n"
    "bgt 18b\n"
    "19:"  // Main loop tail
    ".inst 0xc1391ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z9.s\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc13a1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z10.s\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1341ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z4.s\n"
    ".inst 0xa14049e1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1361a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z6.s\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1301a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z0.s\n"
    ".inst 0xa04149ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1371a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z7.s\n"
    ".inst 0xa04049ec  // ld1w { z12.s-z13.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1381ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z8.s\n"
    "ld1w { z16.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc13a1ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z10.s\n"
    ".inst 0xa14149e5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1311ae2  // fmla za.s[x8, 2], { z23.s-z26.s }, z1.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc1391a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z9.s\n"
    "ld1w { z1.s }, p2/Z, [x15, #4, MUL VL]\n"
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13e1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z14.s\n"
    "ld1w { z22.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc13c1a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z12.s\n"
    "ld1w { z17.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1w { z23.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1311b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z1.s\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1351b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z5.s\n"
    ".inst 0xa0414a8e  // ld1w { z14.s-z15.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1371b02  // fmla za.s[x8, 2], { z24.s-z27.s }, z7.s\n"
    "ld1w { z18.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z24.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc0060c08  // mova { z8.d-z11.d }, za.d[x8, #0]\n"
    ".inst 0xa0404a84  // ld1w { z4.s-z5.s }, pn10.b/Z, [x20]\n"
    "add x8, x8, #0x1\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc1a3c848  // fclamp { z8.s-z11.s }, z2.s, z3.s\n"
    "ld1w { z19.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "st1w { z8.s }, p1, [x11]\n"
    "mov x12, #0x8\n"
    ".inst 0xc13f1a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z15.s\n"
    ".inst 0xa0404a80  // ld1w { z0.s-z1.s }, pn10.b/Z, [x20]\n"
    "add x11, x11, x9, LSL #2\n"
    ".inst 0xc1351a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z5.s\n"
    ".inst 0xa1414a80  // ld1w { z0.s, z8.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    "st1w { z9.s }, p1, [x10]\n"
    "ld1w { z25.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1381ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z8.s\n"
    ".inst 0xc1311ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z1.s\n"
    "ld1w { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xa0404a86  // ld1w { z6.s-z7.s }, pn10.b/Z, [x20]\n"
    ".inst 0xc1371a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z7.s\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z10.s }, p1, [x27]\n"
    ".inst 0xa1414a81  // ld1w { z1.s, z9.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc1391a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z9.s\n"
    "add x27, x27, x25, LSL #2\n"
    "ld1w { z26.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "st1w { z11.s }, p1, [x26]\n"
    ".inst 0xa1404a84  // ld1w { z4.s, z12.s }, pn10.b/Z, [x20]\n"
    ".inst 0xc13c1ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z12.s\n"
    "add x26, x26, x24, LSL #2\n"
    ".inst 0xa1414a84  // ld1w { z4.s, z12.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc13c1ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z12.s\n"
    "ld1w { z21.s }, p0/Z, [x21]\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    ".inst 0xa0404a80  // ld1w { z0.s-z1.s }, pn10.b/Z, [x20]\n"
    ".inst 0xc1311a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z1.s\n"
    ".inst 0xa0414a80  // ld1w { z0.s-z1.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1311a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z1.s\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "20:"  // Main loop skip tail
    "cbz x16, 21f\n"  // Skip remainder inputs
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z16.s }, p0/Z, [x14]\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    ".inst 0xc1391a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z9.s\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1361ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z6.s\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc13a1a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z10.s\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1341a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z4.s\n"
    ".inst 0xa04049ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15]\n"
    "sub x13, x13, #0x1\n"
    ".inst 0xa04149ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1381a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z8.s\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    ".inst 0xc1301ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z0.s\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z9.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1371ae2  // fmla za.s[x8, 2], { z23.s-z26.s }, z7.s\n"
    ".inst 0xa04049e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc1391b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z9.s\n"
    ".inst 0xa14149e5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc13e1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z14.s\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    ".inst 0xc13a1a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z10.s\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1381a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z8.s\n"
    ".inst 0xc1351b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z5.s\n"
    ".inst 0xa04049e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc1301b02  // fmla za.s[x8, 2], { z24.s-z27.s }, z0.s\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc0060c18  // mova { z24.d-z27.d }, za.d[x8, #0]\n"
    ".inst 0xc1a3c858  // fclamp { z24.s-z27.s }, z2.s, z3.s\n"
    "st1w { z24.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    ".inst 0xc1301a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z0.s\n"
    "st1w { z25.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    ".inst 0xc1381a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z8.s\n"
    "add x8, x8, #0x1\n"
    "st1w { z26.s }, p1, [x27]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z27.s }, p1, [x26]\n"
    "add x26, x26, x24, LSL #2\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "21:"  // Tail input: End
    "cbz x13, 23f\n"
    "22:"  // Right padding loop
    ".inst 0xc0060c08  // mova { z8.d-z11.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "subs x13, x13, #0x1\n"
    ".inst 0xc1a3c848  // fclamp { z8.s-z11.s }, z2.s, z3.s\n"
    "st1w { z8.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "st1w { z9.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z10.s }, p1, [x27]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z11.s }, p1, [x26]\n"
    "add x26, x26, x24, LSL #2\n"
    "bgt 22b\n"
    "23:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incb x20, ALL, MUL #16\n"
    "incb x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "incw x17\n"
    "whilelt p1.s, x17, x7\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21, LSL #2\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_bias] "I" (offsetof(Args, bias)), [offsetof_Args_clamp_max] "I" (offsetof(Args, clamp_max)), [offsetof_Args_clamp_min] "I" (offsetof(Args, clamp_min)), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights))
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
