//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Descriptor/PolyItem.h
//! @brief     Defines and implements template class PolyItem.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_DESCRIPTOR_POLYITEM_H
#define BORNAGAIN_GUI_MODEL_DESCRIPTOR_POLYITEM_H

#include "GUI/Model/Util/UtilXML.h"

//! Holds a polymorphous item. Possible types of the item are specified by a Catalog.

template <typename Catalog> class PolyItem {
public:
    using BaseItem = typename Catalog::BaseItem;

    void simpleInit(const QString& label, const QString& tooltip,
                    typename Catalog::Type currentType);

    BaseItem* certainItem() const { return m_item.get(); }

    void setCertainItem(BaseItem* t) { m_item.reset(t); }

    template <typename S> S* createCertainItem();

    //! Serializes the catalog index of the currently selected type and calls
    //! main serialization method of the selected class.
    void writeTo(QXmlStreamWriter* w) const { XML::writeItemTo<Catalog>(m_item.get(), w); }

    //! Deserializes the catalog index of the currently selected type, creates a new
    //! object of this type and calls main deserialization method of the selected class.
    template <typename... Args> void readFrom(QXmlStreamReader* r, Args... args)
    {
        m_item.reset(XML::readItemFrom<Catalog>(r, args...));
    }

    QString piLabel() const { return m_label; }
    QString piTooltip() const { return m_tooltip; }
    QStringList menuEntries() const { return m_menu_entries; }

    void setCurrentIndex(int index) { m_item.reset(Catalog::create(m_types[index])); }
    int currentIndex() const { return m_types.indexOf(Catalog::type(m_item.get())); }

private:
    std::unique_ptr<BaseItem> m_item; //!< Current selection

    QString m_label;            //!< A label text (short, no trailing colon)
    QString m_tooltip;          //!< Tooltip text
    QStringList m_menu_entries; //!< List of options, usually presented as combo entries
    QVector<typename Catalog::Type> m_types = Catalog::types();
};


//! Initialize by means of a catalog class and optional creation arguments.
//!
//! The current selection will be initialized with the first type in the catalog types. The optional
//! arguments are the arguments which may be necessary for the creation method in the catalog.
template <typename C>
void PolyItem<C>::simpleInit(const QString& label, const QString& tooltip,
                             typename C::Type currentType)
{
    m_label = label;
    m_tooltip = tooltip;

    m_menu_entries.clear();
    for (const auto type : m_types)
        m_menu_entries << C::uiInfo(type).menuEntry;

    int index = C::types().indexOf(currentType);
    setCurrentIndex(index);
}

//! Directly set the new item.
template <typename C> template <typename S> S* PolyItem<C>::createCertainItem()
{
    auto* t = new S;
    m_item.reset(t);
    return t;
}

#endif // BORNAGAIN_GUI_MODEL_DESCRIPTOR_POLYITEM_H
