//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Realspace/RealspacePanel.cpp
//! @brief     Implements class RealspacePanel.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Realspace/RealspacePanel.h"
#include "GUI/View/Realspace/RealspaceWidget.h"
#include "GUI/View/Widget/StyledToolbar.h"
#include <QCheckBox>
#include <QVBoxLayout>

RealspacePanel::RealspacePanel(QWidget* parent)
    : QWidget(parent)
    , m_widget(new RealspaceWidget)
{
    setWindowTitle("Real Space");

    auto* toolbar = new StyledToolbar(this);

    const auto createAction = [&](const QString& text, const QString& tooltip) -> QAction* {
        auto* action = new QAction(text, this);
        action->setToolTip(tooltip);
        if (!toolbar->actions().empty())
            toolbar->addSeparator();
        toolbar->addAction(action);
        return action;
    };

    auto* showRoughness = new QCheckBox("Show roughness");
    m_widget->setShowRoughnessCheckbox(showRoughness);
    toolbar->addSeparator();
    toolbar->addWidget(showRoughness);

    auto* action =
        createAction("Regenerate roughness", "Regenerate random roughness on interfaces");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::regenerateRoughness);

    action = createAction("Save picture", "Save 3D real space view as .png file");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::savePicture);

    action = createAction("Default view", "Reset view and zoom level to default");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::defaultView);

    action = createAction("Side view", "View sample from the side at current zoom level");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::sideView);

    action = createAction("Top view", "View sample from the top at current zoom level");
    connect(action, &QAction::triggered, m_widget, &RealspaceWidget::topView);

    action = createAction("Enlarge", "Increase layer size");
    connect(action, &QAction::triggered, [this] { m_widget->changeLayerSize(1.25); });

    action = createAction("Reduce", "Decrease layer size");
    connect(action, &QAction::triggered, [this] { m_widget->changeLayerSize(0.8); });

    auto* mainLayout = new QVBoxLayout(this);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->setSpacing(0);
    mainLayout->addWidget(toolbar);
    mainLayout->addWidget(m_widget);
}

QSize RealspacePanel::sizeHint() const
{
    return {300, 300};
}
