/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.dbcp2;

import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;

import java.time.Duration;

import org.apache.commons.pool2.impl.GenericKeyedObjectPool;
import org.apache.commons.pool2.impl.GenericKeyedObjectPoolConfig;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class TestPoolingConnection {

    private PoolingConnection connection;

    @SuppressWarnings("resource") // Resources closed in @AfterEach method.
    @BeforeEach
    public void setUp() throws Exception {
        connection = new PoolingConnection(new TesterConnection("test", "test"));
        final GenericKeyedObjectPoolConfig<DelegatingPreparedStatement> config = new GenericKeyedObjectPoolConfig<>();
        config.setMaxTotalPerKey(-1);
        config.setBlockWhenExhausted(false);
        config.setMaxWait(Duration.ZERO);
        config.setMaxIdlePerKey(1);
        config.setMaxTotal(1);
        connection.setStatementPool(new GenericKeyedObjectPool<>(connection, config));
    }

    @AfterEach
    public void tearDown() throws Exception {
        connection.close();
        connection = null;
    }

    @Test
    public void testPrepareCall() throws Exception {
        final String sql = "select 'a' from dual";
        try (final DelegatingCallableStatement statement = (DelegatingCallableStatement) connection.prepareCall(sql)) {
            final TesterCallableStatement testStatement = (TesterCallableStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
        }
    }

    @Test
    public void testPrepareCallWithResultSetConcurrency() throws Exception {
        final String sql = "select 'a' from dual";
        final int resultSetType = 0;
        final int resultSetConcurrency = 0;
        try (final DelegatingCallableStatement statement = (DelegatingCallableStatement) connection.prepareCall(sql, resultSetType, resultSetConcurrency)) {
            final TesterCallableStatement testStatement = (TesterCallableStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
            assertEquals(resultSetType, testStatement.getResultSetType());
            assertEquals(resultSetConcurrency, testStatement.getResultSetConcurrency());
        }
    }

    @Test
    public void testPrepareCallWithResultSetHoldability() throws Exception {
        final String sql = "select 'a' from dual";
        final int resultSetType = 0;
        final int resultSetConcurrency = 0;
        final int resultSetHoldability = 0;
        try (final DelegatingCallableStatement statement = (DelegatingCallableStatement) connection.prepareCall(sql, resultSetType, resultSetConcurrency,
            resultSetHoldability)) {
            final TesterCallableStatement testStatement = (TesterCallableStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
            assertEquals(resultSetType, testStatement.getResultSetType());
            assertEquals(resultSetConcurrency, testStatement.getResultSetConcurrency());
            assertEquals(resultSetHoldability, testStatement.getResultSetHoldability());
        }
    }

    @Test
    public void testPrepareStatement() throws Exception {
        final String sql = "select 'a' from dual";
        try (final DelegatingPreparedStatement statement = (DelegatingPreparedStatement) connection.prepareStatement(sql)) {
            final TesterPreparedStatement testStatement = (TesterPreparedStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
        }
    }

    @Test
    public void testPrepareStatementWithAutoGeneratedKeys() throws Exception {
        final String sql = "select 'a' from dual";
        final int autoGeneratedKeys = 0;
        try (final DelegatingPreparedStatement statement = (DelegatingPreparedStatement) connection.prepareStatement(sql, autoGeneratedKeys)) {
            final TesterPreparedStatement testStatement = (TesterPreparedStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
            assertEquals(autoGeneratedKeys, testStatement.getAutoGeneratedKeys());
        }
    }

    @Test
    public void testPrepareStatementWithColumnIndexes() throws Exception {
        final String sql = "select 'a' from dual";
        final int[] columnIndexes = {1};
        try (final DelegatingPreparedStatement statement = (DelegatingPreparedStatement) connection.prepareStatement(sql, columnIndexes)) {
            final TesterPreparedStatement testStatement = (TesterPreparedStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
            assertArrayEquals(columnIndexes, testStatement.getColumnIndexes());
        }
    }

    @Test
    public void testPrepareStatementWithColumnNames() throws Exception {
        final String sql = "select 'a' from dual";
        final String[] columnNames = {"columnName1"};
        try (final DelegatingPreparedStatement statement = (DelegatingPreparedStatement) connection.prepareStatement(sql, columnNames)) {
            final TesterPreparedStatement testStatement = (TesterPreparedStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
            assertArrayEquals(columnNames, testStatement.getColumnNames());
        }
    }

    @Test
    public void testPrepareStatementWithResultSetConcurrency() throws Exception {
        final String sql = "select 'a' from dual";
        final int resultSetType = 0;
        final int resultSetConcurrency = 0;
        try (
            final DelegatingPreparedStatement statement = (DelegatingPreparedStatement) connection.prepareStatement(sql, resultSetType, resultSetConcurrency)) {
            final TesterPreparedStatement testStatement = (TesterPreparedStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
            assertEquals(resultSetType, testStatement.getResultSetType());
            assertEquals(resultSetConcurrency, testStatement.getResultSetConcurrency());
        }
    }

    @Test
    public void testPrepareStatementWithResultSetHoldability() throws Exception {
        final String sql = "select 'a' from dual";
        final int resultSetType = 0;
        final int resultSetConcurrency = 0;
        final int resultSetHoldability = 0;
        try (final DelegatingPreparedStatement statement = (DelegatingPreparedStatement) connection.prepareStatement(sql, resultSetType, resultSetConcurrency,
            resultSetHoldability)) {
            final TesterPreparedStatement testStatement = (TesterPreparedStatement) statement.getInnermostDelegate();
            // assert
            assertEquals(sql, testStatement.getSql());
            assertEquals(resultSetType, testStatement.getResultSetType());
            assertEquals(resultSetConcurrency, testStatement.getResultSetConcurrency());
            assertEquals(resultSetHoldability, testStatement.getResultSetHoldability());
        }
    }

    /**
     * Tests DBCP-596 PoolingConnection.toString() causes StackOverflowError.
     */
    @Test
    public void testToStringStackOverflow() {
        final PoolingConnection conn = new PoolingConnection(null);
        final GenericKeyedObjectPoolConfig<DelegatingPreparedStatement> config = new GenericKeyedObjectPoolConfig<>();
        final GenericKeyedObjectPool stmtPool = new GenericKeyedObjectPool<>(conn, config);
        conn.setStatementPool(stmtPool);
        conn.toString();
    }
}
