/* 
   xhotswap - Motif frontend for the hotswap program

   Copyright (C) 2001-2002 Tim Stadelmann

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/



/** HEADERS **/

/* autoconf header */
#if HAVE_CONFIG_H
#  include <config.h>
#endif

/* Motif headers */
#include <Xm/CascadeBG.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/LabelG.h>
#include <Xm/MainW.h>
#include <Xm/MessageB.h>
#include <Xm/MwmUtil.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>

/* X headers */
#include <X11/keysym.h>

/* ANSI C headers */
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <locale.h>

/* GNU headers */
#include <libintl.h>

/* program headers */
#include "message_box.h"
#include "gettext.h"



/** MACROS **/

#ifndef IDE_CONTROLLER
#  define IDE_CONTROLLER 1
#endif /* IDE_CONTROLLER */

#ifndef HOTSWAP_PATH
#  define HOTSWAP_PATH "hotswap"
#endif /* HOTSWAP_PATH */

#define BUFFER_LENGTH 256
#define ARGS_LENGTH 10
#define _(String) gettext(String)

/** TYPES **/

typedef struct settings_s {
  int ide_controller;		/* the number of the IDE controller */
  String hotswap_path;		/* the path of the backend */
  Boolean verbose;		/* whether to be verbose */
  Boolean print_version;	/* whether to print the version and exit */
  Boolean print_help;		/* whether to print help and exit */
} settings_t;


/** PROTOTYPES **/

static void create_widgets (Widget);
static int probe_device ();
static int check_mounted ();
static int do_remove ();
static int call_backend (String);
static void usage ();
static void quit_callback (Widget, XtPointer, XtPointer);
static void about_callback (Widget, XtPointer, XtPointer);
static void insert_callback (Widget, XtPointer, XtPointer);
static void insert_real_callback (Widget, XtPointer, XtPointer);
static void remove_callback (Widget, XtPointer, XtPointer);
static void swap_callback (Widget, XtPointer, XtPointer);



/** GLOBAL VARIABLES **/

static settings_t settings;	/* settings */
static String program_name;	/* the path of the executable */
static char device_model[40];	/* the IDE device model string,
				   limited to 40 bytes in hardware */
static char buffer[BUFFER_LENGTH];	/* a buffer for the output of the
					   backend */
static Widget app_shell;	/* the application shell */
static Widget confirm_dialog;	/* asks for confirmation */
static Widget message_dialog;	/* a dialog for displaying messages */
static Widget about_dialog;	/* the about box */
static Widget insert_button;	/* the buttons in the main window */
static Widget remove_button;
static Widget swap_button;
static Widget insert_menu_item;	/* the buttons in the device menu */
static Widget remove_menu_item;
static Widget swap_menu_item;
static Widget device_label;	/* the device label in the main window */

/* the resource list */
static XtResource resources[] = {
  {"ideController", "IdeController",
   XtRInt, sizeof (int),
   XtOffsetOf (settings_t, ide_controller),
   XtRImmediate, (XPointer) IDE_CONTROLLER},

  {"hotswapPath", "HotswapPath",
   XtRString, sizeof (String),
   XtOffsetOf (settings_t, hotswap_path),
   XtRString, HOTSWAP_PATH},

  {"verbose", "Verbose",
   XtRBoolean, sizeof (Boolean),
   XtOffsetOf (settings_t, verbose),
   XtRImmediate, (XPointer) False},

  {"printVersion", "PrintVersion",
   XtRBoolean, sizeof (Boolean),
   XtOffsetOf (settings_t, print_version),
   XtRImmediate, (XPointer) False},

  {"printHelp", "PrintHelp",
   XtRBoolean, sizeof (Boolean),
   XtOffsetOf (settings_t, print_help),
   XtRImmediate, (XPointer) False},
};

/* the option list */
static XrmOptionDescRec options[] = {
  {"--ide-controller", "ideController", XrmoptionSepArg, (XtPointer) NULL},
  {"--hotswap-path", "hotswapPath", XrmoptionSepArg, (XtPointer) NULL},
  {"--verbose", "verbose", XrmoptionNoArg, (XtPointer) "True"},
  {"--help", "printHelp", XrmoptionNoArg, (XtPointer) "True"},
  {"-h", "printHelp", XrmoptionNoArg, (XtPointer) "True"},
  {"--version", "printVersion", XrmoptionNoArg, (XtPointer) "True"},
  {"-V", "printVersion", XrmoptionNoArg, (XtPointer) "True"},
};



/** MAIN **/

int
main (int argc, char **argv)
{
  XtAppContext app_context;	/* the application context */
  Arg args[3];
  Cardinal n;

  program_name = argv[0];

  /* Get the locale from the environment.  */
  setlocale (LC_ALL, "");

  /* Initialize gettext.  */
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  /* Initialize the intrinsics and create the application shell.  */
  n = 0;
  XtSetArg (args[n], XmNtitle, "Hotswap"), n++;
  XtSetArg (args[n], XmNallowShellResize, True), n++;
  XtSetArg (args[n], XmNmwmDecorations,
	    MWM_DECOR_ALL | MWM_DECOR_RESIZEH | MWM_DECOR_MAXIMIZE), n++;
  app_shell = XtOpenApplication (&app_context, "Hotswap",
				 options, XtNumber (options),
				 &argc, argv,
				 NULL, sessionShellWidgetClass, args, n);

  XtGetApplicationResources (app_shell, &settings, resources,
			     XtNumber (resources), NULL, 0);

  /* Check for remaining arguments on the command line as well as
     requests for usage and version information.  If applicable, print
     the relevant information and exit.  */

  if (argc != 1)
    {
      printf (_("%s: bad command line option \"%s\"\n\n"),
	      program_name, argv[1]);
      usage (EXIT_FAILURE);
    }

  if (settings.print_help)
    {
      usage (EXIT_SUCCESS);
    }

  if (settings.print_version)
    {
      printf (_("\
Hotswap %s\n\
Copyright 2001-2002 Tim Stadelmann\n\
This program is free software, licensed under the conditions of the\n\
GNU General Public License version 2, or (at your option), any later\n\
version.\n\
"), VERSION);
      exit (EXIT_SUCCESS);
    }

  create_widgets (app_shell);

  /* Realize the widget hierarchy.  */
  XtRealizeWidget (app_shell);

  probe_device ();

  XtAppMainLoop (app_context);
}



/** FUNCTIONS **/

/* Create the widget tree, rooted at app_shell.  */

static void
create_widgets (Widget app_shell)
{
  XmString label;		/* for creating label strings */
  XmString ok_label;
  XmString cancel_label;
  XmString accelerator;		/* for creating accelerator strings */
  XmString title;		/* for creating title strings */
  Widget main_window;		/* the main window */
  Widget menu_bar;		/* the menu bar */
  Widget device_button;		/* the memu bar items */
  Widget help_button;
  Widget form;			/* the work area form */
  Widget device_menu;		/* the pulldown menus */
  Widget help_menu;
  Widget separator;		/* the pulldown menu items */
  Widget quit_menu_item;
  Widget about_menu_item;
  Widget device_frame;		/* the frame around the device string */
  Widget device_title;		/* its title */
  Widget action_frame;		/* the frame around the action buttons */
  Widget action_title;		/* its title  */
  Widget action_table;		/* the action menu */
  Arg args[ARGS_LENGTH];
  Cardinal n;

  /* Points to the about string without the version number insterted.  */
  String raw_about_string;
  /* Points to the finalized about string.  */
  String about_string;

  /* Create the string for the about box.  */
  raw_about_string = _("\
Hotswap Utility (xhotswap) %s\n\
\n\
Copyright  2001-2002 Tim Stadelmann\
");
  about_string
    = XtMalloc (strlen (raw_about_string) - 2 + strlen (VERSION) + 1);
  sprintf (about_string, raw_about_string, VERSION);


  /* Create the widget hierarchy.  */

  n = 0;
  main_window = XmCreateMainWindow (app_shell, "main_window", args, n);
  XtManageChild (main_window);


  /* Create the menu bar.  */

  n = 0;
  menu_bar = XmCreateMenuBar (main_window, "menu_bar", args, n);
  XtManageChild (menu_bar);


  /* Create the device menu.  */

  n = 0;
  device_menu = XmCreatePulldownMenu (menu_bar, "device_menu", args, n);

  n = 0;
  label = XmStringCreateLocalized (_("Swap"));
  accelerator = XmStringCreateLocalized (_("Ctrl+S"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  /* the mnemonic for the 'Swap' menu item */
  XtSetArg (args[n], XmNmnemonic, XStringToKeysym (_("S"))), n++;
  XtSetArg (args[n], XmNaccelerator, "Ctrl<Key>S"), n++;
  XtSetArg (args[n], XmNacceleratorText, accelerator), n++;
  swap_menu_item = XmCreatePushButtonGadget (device_menu, "swap_menu_item",
					     args, n);
  XmStringFree (label);
  XmStringFree (accelerator);
  XtManageChild (swap_menu_item);
  XtAddCallback (swap_menu_item, XmNactivateCallback, swap_callback, NULL);

  n = 0;
  label = XmStringCreateLocalized (_("Remove"));
  accelerator = XmStringCreateLocalized (_("Ctrl+R"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  /* the mnemonic for the 'Remove' menu item */
  XtSetArg (args[n], XmNmnemonic, XStringToKeysym (_("R"))), n++;
  XtSetArg (args[n], XmNaccelerator, "Ctrl<Key>R"), n++;
  XtSetArg (args[n], XmNacceleratorText, accelerator), n++;
  remove_menu_item
    = XmCreatePushButtonGadget (device_menu, "remove_menu_item", args, n);
  XmStringFree (label);
  XmStringFree (accelerator);
  XtManageChild (remove_menu_item);
  XtAddCallback (remove_menu_item, XmNactivateCallback,
		 remove_callback, NULL);

  n = 0;
  label = XmStringCreateLocalized (_("Insert"));
  accelerator = XmStringCreateLocalized (_("Ctrl+I"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  /* the mnemonic for the 'Insert' menu item */
  XtSetArg (args[n], XmNmnemonic, XStringToKeysym (_("I"))), n++;
  XtSetArg (args[n], XmNaccelerator, "Ctrl<Key>I"), n++;
  XtSetArg (args[n], XmNacceleratorText, accelerator), n++;
  insert_menu_item =
    XmCreatePushButtonGadget (device_menu, "insert_menu_item", args, n);
  XmStringFree (label);
  XmStringFree (accelerator);
  XtManageChild (insert_menu_item);
  XtAddCallback (insert_menu_item, XmNactivateCallback, insert_callback,
		 NULL);

  n = 0;
  separator = XmCreateSeparatorGadget (device_menu, "separator", args, n);
  XtManageChild (separator);

  n = 0;
  label = XmStringCreateLocalized (_("Exit"));
  accelerator = XmStringCreateLocalized (_("Ctrl+Q"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  /* the mnemonic for the 'Exit' menu item */
  XtSetArg (args[n], XmNmnemonic, XStringToKeysym (_("x"))), n++;
  XtSetArg (args[n], XmNaccelerator, "Ctrl<Key>Q"), n++;
  XtSetArg (args[n], XmNacceleratorText, accelerator), n++;
  quit_menu_item = XmCreatePushButtonGadget (device_menu, "quit_menu_item",
					     args, n);
  XmStringFree (label);
  XmStringFree (accelerator);
  XtManageChild (quit_menu_item);
  XtAddCallback (quit_menu_item, XmNactivateCallback, quit_callback, NULL);

  /* Create the help menu.  */

  n = 0;
  help_menu = XmCreatePulldownMenu (menu_bar, "help_menu", args, n);

  n = 0;
  label = XmStringCreateLocalized (_("Product Information..."));
  XtSetArg (args[n], XmNlabelString, label), n++;
  /* the mnemonic for the 'Product Information...' menu item */
  XtSetArg (args[n], XmNmnemonic, XStringToKeysym (_("P"))), n++;
  about_menu_item
    = XmCreatePushButtonGadget (help_menu, "about_menu_item", args, n);
  XmStringFree (label);
  XtManageChild (about_menu_item);
  XtAddCallback (about_menu_item, XmNactivateCallback, about_callback, NULL);


  /* Create the menu bar buttons.  */

  n = 0;
  label = XmStringCreateLocalized (_("Device"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  /* the mnemonic for the 'Device' menu item */
  XtSetArg (args[n], XmNmnemonic, XStringToKeysym (_("D"))), n++;
  XtSetArg (args[n], XmNsubMenuId, device_menu), n++;
  device_button = XmCreateCascadeButtonGadget (menu_bar, "device_button",
					       args, n), n++;
  XmStringFree (label);
  XtManageChild (device_button);

  n = 0;
  label = XmStringCreateLocalized (_("Help"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  XtSetArg (args[n], XmNmnemonic, XK_H), n++;
  XtSetArg (args[n], XmNsubMenuId, help_menu), n++;
  help_button = XmCreateCascadeButtonGadget (menu_bar, "help_button",
					     args, n), n++;
  XmStringFree (label);
  XtManageChild (help_button);

  n = 0;
  XtSetArg (args[n], XmNmenuHelpWidget, help_button), n++;
  XtSetValues (menu_bar, args, n);


  /* Create the work area widget and its children.  */

  n = 0;
  XtSetArg (args[n], XmNhorizontalSpacing, 10), n++;
  XtSetArg (args[n], XmNverticalSpacing, 10), n++;
  XtSetArg (args[n], XmNshadowThickness, 1), n++;
  XtSetArg (args[n], XmNresizePolicy, XmRESIZE_GROW), n++;
  form = XmCreateForm (main_window, "form", args, n);
  XtManageChild (form);

  n = 0;
  XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM), n++;
  XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM), n++;
  XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM), n++;
  device_frame =
    XmCreateFrame (form, "device_frame", args, n),
    XtManageChild (device_frame);

  n = 0;
  label = XmStringCreateLocalized (_("Configured Device"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  XtSetArg (args[n], XmNchildType, XmFRAME_TITLE_CHILD), n++;
  device_title = XmCreateLabelGadget (device_frame, "device_title", args, n);
  XmStringFree (label);
  XtManageChild (device_title);

  n = 0;
  XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER), n++;
  XtSetArg (args[n], XmNshadowThickness, 2), n++;
  XtSetArg (args[n], XmNmarginWidth, 3), n++;
  XtSetArg (args[n], XmNmarginHeight, 3), n++;
  device_label = XmCreateLabelGadget (device_frame, "device_label", args, n);
  /* This widget is managed in probe_device; works around a display
     bug in OpenMotif and possibly other releases.  */

  n = 0;
  XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET), n++;
  XtSetArg (args[n], XmNtopWidget, device_frame), n++;
  XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM), n++;
  XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM), n++;
  XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM), n++;
  action_frame = XmCreateFrame (form, "action_frame", args, n);
  XtManageChild (action_frame);

  n = 0;
  label = XmStringCreateLocalized (_("Action"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  XtSetArg (args[n], XmNchildType, XmFRAME_TITLE_CHILD), n++;
  action_title = XmCreateLabelGadget (action_frame, "action_title", args, n);
  XmStringFree (label);
  XtManageChild (action_title);

  n = 0;
  XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER), n++;
  XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM), n++;
  XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM), n++;
  XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM), n++;
  action_table = XmCreateRowColumn (action_frame, "action_table", args, n);
  XtManageChild (action_table);

  n = 0;
  label = XmStringCreateLocalized (_("Swap"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  swap_button = XmCreatePushButtonGadget (action_table,
					  "swap_button", args, n);
  XmStringFree (label);
  XtAddCallback (swap_button, XmNactivateCallback, swap_callback, NULL);
  XtManageChild (swap_button);

  n = 0;
  label = XmStringCreateLocalized (_("Remove"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  remove_button =
    XmCreatePushButtonGadget (action_table, "remove_button", args, n);
  XmStringFree (label);
  XtAddCallback (remove_button, XmNactivateCallback, remove_callback, NULL);
  XtManageChild (remove_button);

  n = 0;
  label = XmStringCreateLocalized (_("Insert"));
  XtSetArg (args[n], XmNlabelString, label), n++;
  insert_button =
    XmCreatePushButtonGadget (action_table, "insert_button", args, n);
  XmStringFree (label);
  XtAddCallback (insert_button, XmNactivateCallback, insert_callback, NULL);
  XtManageChild (insert_button);

  /* Create the about box.  */
  n = 0;
  label = XmStringCreateLocalized (about_string);
  ok_label = XmStringCreateLocalized (_("OK"));
  title = XmStringCreateLocalized (_("Product Information"));
  XtSetArg (args[n], XmNmwmDecorations, MWM_DECOR_ALL | MWM_DECOR_RESIZEH),
    n++;
  XtSetArg (args[n], XmNmessageString, label), n++;
  XtSetArg (args[n], XmNdialogTitle, title), n++;
  about_dialog = XmCreateMessageDialog (app_shell, "about_dialog", args, n);
  XmStringFree (label);
  XmStringFree (ok_label);
  XmStringFree (title);
  XtUnmanageChild (XtNameToWidget (about_dialog, "Cancel"));
  XtUnmanageChild (XtNameToWidget (about_dialog, "Help"));

  /* Free the memory allocated for the about string.  */
  XtFree (about_string);

  /* Create the confirmation dialog.  */
  n = 0;
  ok_label = XmStringCreateLocalized (_("OK"));
  cancel_label = XmStringCreateLocalized (_("Cancel"));
  title = XmStringCreateLocalized (_("Confirmation"));
  XtSetArg (args[n], XmNdialogTitle, title), n++;
  XtSetArg (args[n], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL), n++;
  XtSetArg (args[n], XmNmwmDecorations, MWM_DECOR_ALL | MWM_DECOR_RESIZEH),
    n++;
  confirm_dialog =
    XmCreateMessageDialog (app_shell, "confirm_dialog", args, n);
  XmStringFree (ok_label);
  XmStringFree (cancel_label);
  XmStringFree (title);
  XtUnmanageChild (XtNameToWidget (confirm_dialog, "Help"));
  XtAddCallback (confirm_dialog, XmNokCallback, insert_real_callback, NULL);

  /* Create the message dialog.  */
  n = 0;
  ok_label = XmStringCreateLocalized (_("OK"));
  cancel_label = XmStringCreateLocalized (_("Cancel"));
  title = XmStringCreateLocalized (_("Message"));
  XtSetArg (args[n], XmNdialogTitle, title), n++;
  XtSetArg (args[n], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL), n++;
  XtSetArg (args[n], XmNmwmDecorations, MWM_DECOR_ALL | MWM_DECOR_RESIZEH),
    n++;
  message_dialog =
    XmCreateMessageDialog (app_shell, "message_dialog", args, n);
  XmStringFree (ok_label);
  XmStringFree (title);
  XtUnmanageChild (XtNameToWidget (message_dialog, "Cancel"));
  XtUnmanageChild (XtNameToWidget (message_dialog, "Help"));
}


/* Probe the installed device.  Returns -1 on error, 0 otherwise.  */
static int
probe_device ()
{
  int error;			/* the error code from call_backend */
  XmString label;		/* for creating label strings */
  Arg args[ARGS_LENGTH];
  Cardinal n;

  /* Call the backend with the appropriate options. */
  error = call_backend (" -n probe-ide");
  if (error)
    strncpy (device_model, _("[error]"), 40);

  /* Only the side effect of the strtok function is used.  */
  strtok (buffer, "\n");
  strncpy (device_model, buffer, 40);

  /* Set the device label accordingly.  */
  n = 0;
  label = XmStringCreateLocalized (device_model);
  XtSetArg (args[n], XmNlabelString, label), n++;
  XtSetValues (device_label, args, n);
  XmStringFree (label);
  XtManageChild (device_label);

  /* Deactivate inappropriate widgets depending on wheter a device is
     present or not. */
  if (strcoll (_("absent"), device_model) == 0)
    {
      XtSetSensitive (remove_button, False);
      XtSetSensitive (swap_button, False);
      XtSetSensitive (remove_menu_item, False);
      XtSetSensitive (swap_menu_item, False);
      XtSetSensitive (insert_button, True);
      XtSetSensitive (insert_menu_item, True);
    }
  else if (strcoll (_("[error]"), device_model) == 0)
    {
      XtSetSensitive (remove_button, False);
      XtSetSensitive (swap_button, False);
      XtSetSensitive (remove_menu_item, False);
      XtSetSensitive (swap_menu_item, False);
      XtSetSensitive (insert_button, False);
      XtSetSensitive (insert_menu_item, False);
    }
  else
    {
      XtSetSensitive (remove_button, True);
      XtSetSensitive (swap_button, True);
      XtSetSensitive (remove_menu_item, True);
      XtSetSensitive (swap_menu_item, True);
      XtSetSensitive (insert_button, False);
      XtSetSensitive (insert_menu_item, False);
    }

  return error;
}


/* Calls the backend to find out whether there are any mounted
   filesystems left on the device.  Returns -1 if an error occured, 1
   if there are mounted filesystems, and 0 otherwise.  */
static int
check_mounted ()
{
  int mounted;			/* whether filesystems are mounted */
  int error;			/* the error code from call_backend */

  /* Check whether there are any mounted filesystems on the device.  */
  error = call_backend (" -n mounted-ide");
  if (error)
    return -1;

  if (strncmp (_("yes"), buffer, strlen (_("yes"))) == 0)
    {
      mounted = 1;
    }
  else
    {
      mounted = 0;
    }

  return mounted;
}


/* Call the backend to remove the device from the system configuration.
   Returns a non-zero value corresponding to the exit code of the backend
   if an error occured.  */
static int
do_remove ()
{
  int mounted;			/* whether filesystems are mounted */
  int error;			/* the error code from call_backend */

  mounted = check_mounted ();

  if (mounted == -1)
    {
      /* An error message is displayed from within check_mounted.  */
      return -1;
    }
  else if (mounted)
    {
      display_error (app_shell,
		     _("At least one filesystem on the current\n"
		       "device is still mounted.  Unmount the\n"
		       "mounted filesystems and try again."));
      return -1;
    }

  /* Call the backend to remove the device.  */
  error = call_backend (" -n unregister-ide");
  if (error)
    return -1;

  return 0;
}


/* Terminate the program successfully.  */

static void
quit_callback (Widget widget, XtPointer client_data, XtPointer call_data)
{
  exit (EXIT_SUCCESS);
}


/* Call the backend with arbitrary arguments and deal with errors.
   Read both stdout and stderr of the backend into buffer.  Return -1
   if an error has occured internally and the return value of the
   backend otherwise.  */

static int
call_backend (String arguments)
{
  int n_chars;			/* the return value of vsnprintf */
  char *command;		/* a pointer to the command string */
  int command_length;		/* the actual length of the string */
  FILE *backend_out;		/* the output stream from the backend */
  int error;			/* a variable for return values */
  const blocksize = 256;	/* the allocation block size */

  command = XtMalloc (blocksize);
  command_length = blocksize;

  /* Construct the command string passed to the shell.  There are two
     different conventions for the return value of snprintf and
     related functions.  Please refer to the documentation.  */

  for (;;)
    {
      n_chars = snprintf (command, command_length,
			  "%s --ide-controller %d %s 2>&1",
			  settings.hotswap_path,
			  settings.ide_controller,
			  arguments);
      if (n_chars == -1)
	{
	  command_length += blocksize;
	}
      else if (n_chars != strlen (command))
	{
	  command_length = n_chars + 1;
	}
      else
	{
	  break;
	}
      command = XtRealloc (command, command_length);
    }

  backend_out = popen (command, "r");

  /* The command string can now be discarded. */
  XtFree (command);

  if (backend_out == NULL)
    {
      display_error (app_shell,
		     _("An error occurred while calling the backend:\n"
		       "%s"),
		     strerror (errno));
      return -1;
    }

  /* Read the output of the backend.  */
  while (!feof (backend_out))
  {
    fgets (buffer, BUFFER_LENGTH, backend_out);
  }

  /* Close the connection to the backend.  pclose does return the exit
     code of the program if it has succeeded.  The fact that at least
     one version of the Linux man page fails to mention this is a bug
     in the documentation, not in the library.  */
  error = pclose (backend_out);

  if (error == -1)
    {
      display_error (app_shell,
		     _("An error occurred while calling the backend:\n"
		       "%s"),
		     strerror (errno));
    }
  else if (error)
    {
      display_error (app_shell,
		     _("An error occured while running the backend:\n"
		       "%s"), buffer);
    }

  return error;
}


/* Print usage information and exit with the error code given by
   status. */

static void
usage (int status)
{
  printf (_("Hotswap Utility (%s) - "
	    "Supports hotswapping IDE peripherals.\n"),
	  program_name);
  printf (_("Usage: %s [OPTION]...\n"), program_name);
  puts (_("\
\n\
Options:\n\
      --ide-controller N     use IDE controller N\n\
      --hotswap-path PATH    look for the `hotswap' backend at the\n\
                             location given by PATH\n\
  -h, --help                 display this help and exit\n\
  -V, --version              output version information and exit\n\
\n\
Standard toolkit options are supported as well.\n\
\n\
The Linux kernel requires hotswappable devices to be the only\n\
device on their IDE bus.\n\
\n\
Report bugs to <t.stadelmann1@physics.ox.ac.uk>.\n\
"));
  exit (status);
}


/* Post the about dialog box if the respective item is selected from
   the help menu.  */

static void
about_callback (Widget widget, XtPointer client_data, XtPointer call_data)
{
  XtManageChild (about_dialog);
}


/* Prompt for insertion of a new device.  */

static void
insert_callback (Widget widget, XtPointer client_data, XtPointer call_data)
{
  msgbox_printf (confirm_dialog,
		 _("Please insert the new device and click `OK'."));
  XtManageChild (confirm_dialog);
}


/* Call the backend to perform the actual scanning for a new device.
   Return a nonzero value on error.  */

static void
insert_real_callback (Widget widget, XtPointer client_data,
		      XtPointer call_data)
{
  int error;			/* a variable for error codes */
  
  for (;;)
    {
      
      error = call_backend (" -n rescan-ide");
      if (error)
	return;

      probe_device ();

      if (strcoll (_("absent"), device_model) != 0)
	return;
      
      if (!display_question (app_shell, _("\
No IDE device has been found.  If you are sure that the device\n\
connects to the IDE bus, verify that the module is inserted correctly.\n\
\n\
Do you want to try again?\
")))
	return;
    }
}


static void
remove_callback (Widget widget, XtPointer client_data, XtPointer call_data)
{
  /* Remove the IDE device from the system configuration.  */
  if (do_remove () != 0)
    return;

  msgbox_printf (message_dialog, _("You can now remove the device."));
  XtManageChild (message_dialog);

  probe_device ();
}


static void
swap_callback (Widget widget, XtPointer client_data, XtPointer call_data)
{
  /* Remove the IDE device from the system configuration.  */
  if (do_remove () != 0)
    return;

  msgbox_printf (confirm_dialog, _("You can now exchange the device."));

  XtManageChild (confirm_dialog);

  probe_device ();
}
