/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "administrator_filtermodel.h"
#include "connection.h"

#include "administrator_listingmodel.h"

void AdministratorListingModel::doQuery( const QString &userFilter )
{
  QString memberBase( "cn=admin,cn=internal," + Connection::self()->baseDn() );
  QString memberFilter( "(objectClass=kolabGroupOfNames)" );
  QLdapResponse memberResponse = Connection::self()->search( memberBase, QLdap::Sub, memberFilter );

  if ( !memberResponse.entries().isEmpty() ) {
    // get list of admin group members
    QStringList members = memberResponse.entries().first().values( "member" );

    QLdapEntry::List adminEntries;

    QStringList blacklist;
    blacklist.append( "manager" );

    // query all user
    QString filter = QString( "(&(cn=*)(objectClass=inetOrgPerson)%1(uid=*)(sn=*))" ).arg( userFilter );

    QLdapResponse response = Connection::self()->search( "", QLdap::Sub, filter, QStringList(), QLatin1String( "cn" ) );
    QLdapEntry::List entries = response.entries();
    for ( int i = 0; i < entries.count(); ++i ) {
      if ( members.contains( entries[ i ].dn() ) &&              // is administrator
           !blacklist.contains( entries[ i ].value( "cn" ) ) ) { // is not blacklisted
        adminEntries.append( entries[ i ] );
      }
    }

    mData.setEntries( adminEntries );
  }

  emit layoutChanged();
}

int AdministratorListingModel::columnCount( const QModelIndex &parent ) const
{
  if ( parent.isValid() )
    return 0;

  return 2;
}

int AdministratorListingModel::rowCount( const QModelIndex &parent ) const
{
  if ( parent.isValid() )
    return 0;

  return mData.entries().count();
}

QVariant AdministratorListingModel::data( const QModelIndex &index, int role ) const
{
  if ( !index.isValid() )
    return QVariant();

  const QLdapEntry entry = mData.entries()[ index.row() ];

  if ( role == Qt::DisplayRole ) {
    switch ( index.column() ) {
      case 0:
        return QString( "%1" ).arg( entry.value( "cn" ) );
        break;
      case 1:
        return entry.value( "uid" );
        break;
      default:
        return QVariant();
    }
  } else if ( role == HasModifyActionRole ) {
    return true;
  } else if ( role == HasDeleteActionRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return true;
    else
      return false;
  } else if ( role == ModifyActionLabelRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return tr( "Modify" );
    else
      return tr( "Administrator deleted, awaiting cleanup..." );
  } else if ( role == DeleteActionLabelRole ) {
    return tr( "Delete" );
  } else if ( role == ModifyActionIdRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return mData.entries()[ index.row() ].dn();
    else
      return QString();
  } else if ( role == DeleteActionIdRole ) {
    return mData.entries()[ index.row() ].dn();
  }

  return QVariant();
}

QVariant AdministratorListingModel::headerData( int section, Qt::Orientation, int role ) const
{
  if ( role != Qt::DisplayRole )
    return QVariant();

  switch ( section ) {
    case 0:
      return tr( "Name" );
      break;
    case 1:
      return tr( "UID" );
      break;
    default:
      return QVariant();
  }
}

FilterModel* AdministratorListingModel::filterModel()
{
  return new AdministratorFilterModel( this );
}
