/*
 * Qt wrapper for libindicate
 *
 * Copyright 2009 Canonical Ltd.
 *
 * Authors:
 * - Aurélien Gâteau <aurelien.gateau@canonical.com>
 *
 * License: LGPL v2.1 or LGPL v3
 */
// Self
#include "indicatortest.h"

// Qt
#include <QtTest>

// Local
#include <qindicatedecode.h>
#include <qindicateindicator.h>
#include <qindicateserver.h>

QTEST_MAIN(IndicatorTest)

static const QString SERVER_TYPE = "message";

void IndicatorTest::initTestCase()
{
    qRegisterMetaType<QIndicate::Indicator*>("QIndicate::Indicator*");
}

void IndicatorTest::init()
{
    mServer = QIndicate::Server::defaultInstance();
    mServer->setType(SERVER_TYPE);
    mServer->show();
}

void IndicatorTest::cleanup()
{
    delete mServer;
    mServer = 0;
}

void IndicatorTest::testAddRemove()
{
    QVariantList args;

    QSignalSpy addedSpy(mServer, SIGNAL(indicatorAdded(uint)));
    QSignalSpy removedSpy(mServer, SIGNAL(indicatorRemoved(uint)));

    QIndicate::Indicator indicator;
    indicator.setNameProperty("Unittest");

    indicator.show();

    QCOMPARE(addedSpy.count(), 1);
    args = addedSpy.takeFirst();
    QCOMPARE(args[0].toUInt(), indicator.id());

    indicator.hide();

    QCOMPARE(removedSpy.count(), 1);
    args = removedSpy.takeFirst();
    QCOMPARE(args[0].toUInt(), indicator.id());
}

void IndicatorTest::testProperties()
{
    QIndicate::Indicator indicator;
    QString key = "key";
    QString value = "value";
    indicator.setIndicatorProperty(key, value);

    QCOMPARE(QIndicate::Decode::stringFromValue(indicator.indicatorProperty(key)), value);

    QStringList lst = indicator.propertyList();
    QStringList expected = QStringList() << key;
    lst.sort();
    expected.sort();
    QCOMPARE(lst, expected);
}

void IndicatorTest::testCount()
{
    QIndicate::Indicator indicator;
    QCOMPARE(indicator.countProperty(), 0);

    indicator.setCountProperty(2);
    QCOMPARE(indicator.countProperty(), 2);
}

void IndicatorTest::testDrawAttention()
{
    QIndicate::Indicator indicator;
    QVERIFY(!indicator.drawAttentionProperty());

    indicator.setDrawAttentionProperty(true);
    QVERIFY(indicator.drawAttentionProperty());

    indicator.setDrawAttentionProperty(false);
    QVERIFY(!indicator.drawAttentionProperty());
}

void IndicatorTest::testTime()
{
    QIndicate::Indicator indicator;
    QCOMPARE(indicator.timeProperty(), QDateTime());

    QDateTime time = QDateTime(QDate(2010, 2, 10), QTime(17,37,5));
    indicator.setTimeProperty(time);
    QCOMPARE(indicator.timeProperty(), time);
}

void IndicatorTest::testDisplay()
{
    QIndicate::Indicator indicator;
    QSignalSpy spy(&indicator, SIGNAL(display(QIndicate::Indicator*)));

    indicator.emitDisplay();
    QCOMPARE(spy.count(), 0);

    indicator.show();
    indicator.emitDisplay();
    QCOMPARE(spy.count(), 1);
    QVariantList args = spy.takeFirst();
    QCOMPARE(args[0].value<QIndicate::Indicator*>(), &indicator);
}

#include "indicatortest.moc"
