# NAME

MooseX::Declare - Declarative syntax for Moose

# SYNOPSIS

    use MooseX::Declare;

    class BankAccount {
        has 'balance' => ( isa => 'Num', is => 'rw', default => 0 );

        method deposit (Num $amount) {
            $self->balance( $self->balance + $amount );
        }

        method withdraw (Num $amount) {
            my $current_balance = $self->balance();
            ( $current_balance >= $amount )
                || confess "Account overdrawn";
            $self->balance( $current_balance - $amount );
        }
    }

    class CheckingAccount extends BankAccount {
        has 'overdraft_account' => ( isa => 'BankAccount', is => 'rw' );

        before withdraw (Num $amount) {
            my $overdraft_amount = $amount - $self->balance();
            if ( $self->overdraft_account && $overdraft_amount > 0 ) {
                $self->overdraft_account->withdraw($overdraft_amount);
                $self->deposit($overdraft_amount);
            }
        }
    }

# DESCRIPTION

This module provides syntactic sugar for Moose, the postmodern object system
for Perl 5. When used, it sets up the `class` and `role` keywords.

__Note:__ Please see the ["WARNING"](#warning) section below!

# KEYWORDS

## class

    class Foo { ... }

    my $anon_class = class { ... };

Declares a new class. The class can be either named or anonymous, depending on
whether or not a classname is given. Within the class definition Moose and
[MooseX::Method::Signatures](https://metacpan.org/pod/MooseX::Method::Signatures) are set up automatically in addition to the other
keywords described in this document. At the end of the definition the class
will be made immutable. namespace::autoclean is injected to clean up Moose and
other imports for you.

Because of the way the options are parsed, you cannot have a class named "is",
"with" or "extends".

It's possible to specify options for classes:

- extends

        class Foo extends Bar { ... }

    Sets a superclass for the class being declared.

- with

        class Foo with Role             { ... }
        class Foo with Role1 with Role2 { ... }
        class Foo with (Role1, Role2)   { ... }

    Applies a role or roles to the class being declared.

- is mutable

        class Foo is mutable { ... }

    Causes the class not to be made immutable after its definition.

    Options can also be provided for anonymous classes using the same syntax:

        my $meta_class = class with Role;

## role

    role Foo { ... }

    my $anon_role = role { ... };

Declares a new role. The role can be either named or anonymous, depending on
whether or not a name is given. Within the role definition Moose::Role and
MooseX::Method::Signatures are set up automatically in addition to the other
keywords described in this document. Again, namespace::autoclean is injected to
clean up Moose::Role and other imports for you.

It's possible to specify options for roles:

- with

        role Foo with Bar { ... }

    Applies a role to the role being declared.

## before / after / around / override / augment

    before   foo ($x, $y, $z) { ... }
    after    bar ($x, $y, $z) { ... }
    around   baz ($x, $y, $z) { ... }
    override moo ($x, $y, $z) { ... }
    augment  kuh ($x, $y, $z) { ... }

Add a method modifier. Those work like documented in [Moose](https://metacpan.org/pod/Moose), except for
the slightly nicer syntax and the method signatures, which work like documented
in [MooseX::Method::Signatures](https://metacpan.org/pod/MooseX::Method::Signatures).

For the `around` modifier an additional argument called `$orig` is
automatically set up as the invocant for the method.

## clean

Sometimes you don't want the automatic cleaning the `class` and `role`
keywords provide using namespace::autoclean. In those cases you can specify the
`dirty` trait for your class or role:

    use MooseX::Declare;
    class Foo is dirty { ... }

This will prevent cleaning of your namespace, except for the keywords imported
from `Moose` or `Moose::Role`. Additionally, a `clean` keyword is provided,
which allows you to explicitly clean all functions that were defined prior to
calling `clean`. Here's an example:

    use MooseX::Declare;
    class Foo is dirty {
        sub helper_function { ... }
        clean;
        method foo ($stuff) { ...; return helper_function($stuff); }
    }

With that, the helper function won't be available as a method to a user of your
class, but you're still able to use it inside your class.

# NOTE ON IMPORTS

When creating a class with MooseX::Declare like:

    use MooseX::Declare;
    class Foo { ... }

What actually happens is something like this:

    {
        package Foo;
        use Moose;
        use namespace::autoclean;
        ...
        __PACKAGE__->meta->make_immutable;
    }

So if you declare imports outside the class, the symbols get imported into the
`main::` namespace, not the class' namespace. The symbols then cannot be called
from within the class:

    use MooseX::Declare;
    use Data::Dump qw/dump/;
    class Foo {
        method dump($value) { return dump($value) } # Data::Dump::dump IS NOT in Foo::
        method pp($value)   { $self->dump($value) } # an alias for our dump method
    }

To solve this, only import MooseX::Declare outside the class definition
(because you have to). Make all other imports inside the class definition.

    use MooseX::Declare;
    class Foo {
        use Data::Dump qw/dump/;
        method dump($value) { return dump($value) } # Data::Dump::dump IS in Foo::
        method pp($value)   { $self->dump($value) } # an alias for our dump method
    }

    Foo->new->dump($some_value);
    Foo->new->pp($some_value);

__NOTE__ that the import `Data::Dump::dump()` and the method `Foo::dump()`,
although having the same name, do not conflict with each other, because the
imported `dump` function will be cleaned during compile time, so only the
method remains there at run time. If you want to do more esoteric things with
imports, have a look at the `clean` keyword and the `dirty` trait.

# WARNING

__Warning:__ MooseX::Declare is based on [Devel::Declare](https://metacpan.org/pod/Devel::Declare), a giant bag of crack
originally implemented by mst with the goal of upsetting the perl core
developers so much by its very existence that they implemented proper
keyword handling in the core.

As of perl5 version 14, this goal has been achieved, and modules such
as [Devel::CallParser](https://metacpan.org/pod/Devel::CallParser), [Function::Parameters](https://metacpan.org/pod/Function::Parameters), and [Keyword::Simple](https://metacpan.org/pod/Keyword::Simple) provide
mechanisms to mangle perl syntax that don't require hallucinogenic
drugs to interpret the error messages they produce.

If you want to use declarative syntax in new code, please for the love
of kittens get yourself a recent perl and look at [Moops](https://metacpan.org/pod/Moops) instead.

# SEE ALSO

- [Moose](https://metacpan.org/pod/Moose)
- [Moose::Role](https://metacpan.org/pod/Moose::Role)
- [MooseX::Method::Signatures](https://metacpan.org/pod/MooseX::Method::Signatures)
- [namespace::autoclean](https://metacpan.org/pod/namespace::autoclean)
- vim syntax: [http://www.vim.org/scripts/script.php?script_id=2526](http://www.vim.org/scripts/script.php?script_id=2526)
- emacs syntax: [http://github.com/jrockway/cperl-mode](http://github.com/jrockway/cperl-mode)
- Geany syntax + notes: [http://www.cattlegrid.info/blog/2009/09/moosex-declare-geany-syntax.html](http://www.cattlegrid.info/blog/2009/09/moosex-declare-geany-syntax.html)
- [Devel::CallParser](https://metacpan.org/pod/Devel::CallParser)
- [Function::Parameters](https://metacpan.org/pod/Function::Parameters)
- [Keyword::Simple](https://metacpan.org/pod/Keyword::Simple)
- [Moops](https://metacpan.org/pod/Moops)

# AUTHOR

Florian Ragwitz <rafl@debian.org>

# CONTRIBUTORS

- Ash Berlin <ash\_github@firemirror.com>
- Chas. J. Owens IV <chas.owens@gmail.com>
- Chris Prather <chris@prather.org>
- Dave Rolsky <autarch@urth.org>
- Devin Austin <dhoss@cpan.org>
- Hans Dieter Pearcey <hdp@cpan.org>
- Justin Hunter <justin.d.hunter@gmail.com>
- Karen Etheridge <ether@cpan.org>
- Matt Kraai <kraai@ftbfs.org>
- Michele Beltrame <arthas@cpan.org>
- Nelo Onyiah <nelo.onyiah@gmail.com>
- Nick Perez <nperez@cpan.org>
- Piers Cawley <pdcawley@bofh.org.uk>
- Rafael Kitover <rkitover@io.com>
- Robert 'phaylon' Sedlacek <rs@474.at>
- Stevan Little <stevan.little@iinteractive.com>
- Tomas Doran <bobtfish@bobtfish.net>
- Yanick Champoux <yanick@babyl.dyndns.org>
- leedo <lee@laylward.com>

# COPYRIGHT AND LICENSE

This software is copyright (c) 2008 by Florian Ragwitz.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.
