Subring.synonym = "subring"
SAGBIBasis.synonym = "SAGBIBasis"
IntersectedSubring.synonym = "intersected subring"
-- ##################################
-- ##    sagbi-main.m2 documentation ##
-- ##################################
-- SubalgebraBases
doc ///
    Key
        SubalgebraBases
    Headline
        A package for finding canonical subalgebra bases (SAGBI bases)
    Description
        Text
            Let $R=k[f_1,\ldots,f_k]$ denote the subalgebra of
            the polynomial quotient ring $k[x_1,\ldots,x_n] / I$
            generated by $f_1, \ldots, f_k$. We say
            $f_1, \ldots, f_k$ form a {\it subalgebra basis} with
            respect to a monomial order $<$ if the {\it initial algebra}
            associated to $<$, defined as
            $$in(R) := k[in(f\%I) \mid f \in R] \subseteq k[x_1, \ldots, x_n] / in(I),$$
            is generated by the elements
            $in(f_1\%I), \ldots , in(f_k\%I)$.
            The main functions provided by this package are for computing
            these subalgebra bases: @TO sagbi @, @TO isSAGBI @, and @TO subduction@.
    References
        @UL {
        {"D. Kapur and K. Madlener.    A completion procedure
        for computing a canonical basis of a $k$-subalgebra.
        In E. Kaltofen and S. Watt, editors, ",
        EM "Computers and Mathematics",
        ",
        pages 1-11, New York, NY, 1989, Springer US."},
        {"L. Robbiano and M. Sweedler.    Subalgebra bases.
        In W. Bruns and A. Simis, editors, ",
        EM "Commutative Algebra",
        ", pages 61-87,
        Berlin, Heidelberg, 1990.
        Springer Berlin Heidelberg."},
        {"F. Ollivier.    Canonical Bases: Relations with Standard Bases,
        Finiteness conditions and application to tame automorphisms.
        In T. Mora, editor, ",
        EM "Effective Methods in Algebraic Geometry",
        ", pages 379-4000, Birkhaeuser Boston, Boston, MA, 1991."},
        {"M. Stillman and H. Tsai. Using SAGBI bases to compute invariants.",
        EM "Journal of Pure and Applied Algebra",
        ", 139(1), pages 285-302, 1999."},
        {"B. Sturmfels.    Groebner bases and Convex Polytopes.
        Volume 8 of ",
        EM "University Lecture Series",
        ". American Mathematical Society, Providence, RI, 1996."}
        }@
    Contributors
        In addition to the authors below, we thank the
        following attendees of the 2020 Macaulay2 workshops at
        Cleveland State University and
        University of Warwick for their contributions to the package.

        @UL {
        {HREF("https://aarondall.com/mathematics/","Aaron Dall")},
        {HREF("https://www.dima.unige.it/~dali/","Alessio D'Alì")},
        {HREF("https://aliciatocino.com", "Alicia Tocino")},
        {HREF("https://sites.google.com/view/ayah-almousa","Ayah Almousa")},
        {"Dharm Veer"},
        {"Dipak Kumar Pradhan"},
        {HREF("https://maths.nuigalway.ie/~emil/","Emil Sköldberg")},
        {"Eriola Hoxhaj"},
        {HREF("https://sites.google.com/view/kritigoel/home","Kriti Goel")},
        {"Liđan Edin"},
        {HREF("https://maths.nuigalway.ie/~peterphelan/","Peter Phelan")},
        {HREF("https://mathematics.ku.edu/people/ritika-nair", "Ritika Nair")},
        {HREF("https://www.utm.utoronto.ca/math-cs-stats/people/ilir-dema", "Ilir Dema")}
        }@
    SeeAlso
        "Example: Translation and rotation sub-actions of the adjoint action of SE(3)"
///
-- subalgebraBasis
doc ///
    Key
        subalgebraBasis
        (subalgebraBasis, Subring)
        (subalgebraBasis, Matrix)
        (subalgebraBasis, List)
    Headline
        Compute subalgebra basis generators
    Usage
        N = subalgebraBasis M
        N = subalgebraBasis A
        N = subalgebraBasis L
        N = subalgebraBasis B
    Inputs
        A:Subring
        M:Matrix
            of generators for @ ofClass Subring @ of @ ofClass{PolynomialRing} @
        L:List
            containing generators for @ ofClass Subring @ of @ ofClass{PolynomialRing} @
        B:SAGBIBasis
            containing a partial computation of a subalgebra basis
        AutoSubduce=>Boolean
            a flag indicating when to perform autosubduction
            on the generators before performing the subalgebra basis computation
        ReduceNewGenerators=>Boolean
            a flag indicating whether to reduce (via Gaussian elimination)
            to new subalgebra basis generators
            before adding them to the current subalgebra basis
        StorePending=>Boolean
            a flag that indicates whether the pending list should be stored in the result.
        Strategy=>String
            the name of the update strategy at the beginning of each loop:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method that combines the other two;
            starting with "DegreeByDegree" for low degrees
            and switching to "Incremental".
        SubductionMethod=>String
            the name of the method used for subduction either: "Top" or "Engine".
        Limit=>ZZ
            a degree limit for the binomial S-pairs that are computed internally.
        AutoSubduceOnPartialCompletion=>Boolean
            a flag that indicates whether autosubduction is applied
            to the subalgebra basis generators the first time no new generators are added.
            Use this only if very few new subalgebra basis generators are expected.
        PrintLevel=>ZZ
            an option to produce additional output.
            When this is greater than zero,
            information is printed about the progress of the computation
        Recompute=>Boolean
            a flag that indicates if the computation will resume on subsequent runs,
            otherwise it starts at the beginning
        RenewOptions=>Boolean
            a flag that indicates if the computation will use the options specified,
            otherwise it will use the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@)
    Outputs
        N:Matrix
            whose entries form a partial subalgebra basis
    Description
        Text
            Computes a partial subalgebra basis for its input.
            This command returns the computed generators and not a computaiton object.
            If the input is @ ofClass Subring@,
            then the computation object is stored in the cache of the input.
        Example
            R = QQ[t_(1,1)..t_(3,3),MonomialOrder=>Lex];
            M = genericMatrix(R,3,3);
            A = subring gens minors(2, M);
            subalgebraBasis A
        Text
            The command @TO "subalgebraBasis"@ @TT "A"@ is equivalent to
            the command @TO2 {(gens,SAGBIBasis),"gens"}@ @TO "sagbi"@ @TT "A"@.
    SeeAlso
        sagbi
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        "Subduction computation limit"
        PrintLevel
        Recompute
        RenewOptions
        (gens,SAGBIBasis)
///
-- sagbi
doc ///
    Key
        sagbi
        (sagbi,Subring)
        (sagbi, Matrix)
        (sagbi, List)
        (sagbi,SAGBIBasis)
    Headline
        Compute a subalgebra basis (sagbi basis)
    Usage
        N = sagbi M
        N = sagbi A
        N = sagbi L
        N = sagbi B
    Inputs
        A:Subring
        M:Matrix
            of generators for @ ofClass Subring @ of @ ofClass{PolynomialRing} @
        L:List
            containing generators for @ ofClass Subring @ of @ ofClass{PolynomialRing} @
        B:SAGBIBasis
            containing a partial computation of a subalgebra basis
        AutoSubduce=>Boolean
            a flag indicating when to perform autosubduction on the generators
            before performing the subalgebra basis computation
        ReduceNewGenerators=>Boolean
            a flag indicating whether to reduce (via Gaussian elimination) to
            new subalgebra generators before adding them to the current subalgebra basis
        StorePending=>Boolean
            a flag that indicates whether the {\it pending list} should be stored
            in the result.
        Strategy=>String
            the name of the update strategy at the beginning of each loop:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method that combines the other two;
            starting with "DegreeByDegree" for low degrees and switching to "Incremental".
        SubductionMethod=>String
            the name of the method used for subduction either: "Top" or "Engine".
        Limit=>ZZ
            a degree limit for the binomial S-pairs that are computed internally.
        AutoSubduceOnPartialCompletion=>Boolean
            a flag that indicates whether autosubduction is applied to
            the subalgebra generators the first time no new generators are added.
            Use this only if very few new subalgebra generators are expected.
        PrintLevel=>ZZ
            an option to produce additional output.
            When this is greater than zero, information is printed about
            the progress of the computation
        Recompute=>Boolean
            a flag that indicates if the computation will resume in subsequent runs,
            otherwise it starts at the beginning
        RenewOptions=>Boolean
            a flag that indicates if the computation will use the options specified,
            otherwise it will use the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@)
    Outputs
        N:SAGBIBasis
            a computation object holding the state of the subalgebra basis computation
    Description
        Text
            The output of this function is a partial
            subalgebra basis stored in a computation object.
        Example
            R = QQ[t_(1,1)..t_(3,3),MonomialOrder=>Lex];
            M = genericMatrix(R,3,3);
            A = subring gens minors(2, M);
            isSAGBI A
            S = sagbi A;
            gS = gens S
            isSAGBI gS
        Text
            Partial subalgebra bases are unavoidable since a subalgebra
            of a polynomial ring, endowed with some polynomial order,
            need not have a finite subalgebra basis.
            Here is a quintessential example of this phenomenon:
        Example
            R=QQ[x,y];
            A = subring matrix{{x+y,x*y,x*y^2}};
            gens sagbi(A,Limit=>3)
            gens sagbi(A,Limit=>10)
        Text
            Nevertheless, a finite subalgebra basis can be computed in many cases.
    SeeAlso
        subalgebraBasis
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
        Subring
        SAGBIBasis
        "Subduction strategies"
        "Subduction computation limit"
///
-- AutoSubduce
doc ///
    Key
        AutoSubduce
        [(forceSB,Subring),AutoSubduce]
        [(forceSB,SAGBIBasis),AutoSubduce]
        [sagbi,AutoSubduce]
        [sagbiBasis,AutoSubduce]
        [subalgebraBasis,AutoSubduce]
        [subduction,AutoSubduce]
    Headline
        Flag for autosubduction before the sagbi algorithm
    Description
        Text
            {\it Autosubduction} is a procedure which performs subduction
            on a collection of ring elements
            against each other, one at a time.
            This is typically performed on a set of generators of a
            @TO "Subring"@ or @TO "SAGBIBasis"@.
            The result of each subduction replaces the
            original element in the collection and zeros are removed from the collection.
            If the elements form a subalgebra basis, then the result of
            autosubduction is unique.
        Text
            By default, the function @TO "sagbi"@ performs autosubduction on
            the generators of the given @ TO Subring @
            before starting the main algorithm.
            Setting this flag to @TT "false"@ sometimes results in a
            performance improvement, but with a risk of
            including nonreduced or redundant generators.
    SeeAlso
        sagbi
        subalgebraBasis
        ReduceNewGenerators
        subduction
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        ReduceNewGenerators
        RenewOptions
///
-- ReduceNewGenerators
doc ///
    Key
        ReduceNewGenerators
        [sagbi,ReduceNewGenerators]
        [sagbiBasis,ReduceNewGenerators]
        [subalgebraBasis,ReduceNewGenerators]
        [subduction,ReduceNewGenerators]
    Headline
        Flag for reducing new generators in Sagbi algorithm
    Description
        Text
            The function @TO "sagbi"@ computes a subalgebra basis by subducting S-pairs.
            Any resulting non-zero polynomials are added to the list of subalgebra generators.
            If ReduceNewGenerators is set to @ TT "true" @,
            then the new generators are reduced (using Gaussian elimination)
            against the other new generators.
            This process is quick and ensures that duplicate generators
            are not added to the list of subalgebra generators.
    SeeAlso
        sagbi
        sagbiBasis
        subalgebraBasis
        subduction
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
///
-- StorePending
doc ///
    Key
        StorePending
        [sagbi,StorePending]
        [sagbiBasis,StorePending]
        [subalgebraBasis,StorePending]
        [subduction,StorePending]
    Headline
        Flag for storing the pending list to the result of the Sagbi algorithm
    Description
        Text
            If the @ TT "StorePending" @ flag is @ TT "true" @, then the @ TO "SAGBIBasis" @
            produced by the function @TO "sagbi"@ contains a copy of the pending list.
            The pending list contains all unprocessed {\it S-pairs} grouped by their degrees.
            The pending list may be nonempty if the @TO "sagbi"@ algorithm terminates without
            producing a complete subalgebra basis.
            The pending list is stored to speed up the computation if it is resumed later.

            If the @TT "StorePending"@ flag is @TT "false"@,
            then the pending list is not stored.
            Depending on the computation, the pending list may be very large,
            and it may not be convenient to save all the data.
    SeeAlso
        sagbi
        AutoSubduce
        ReduceNewGenerators
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
        sagbiBasis
        subduction
        subalgebraBasis
///
-- SubductionMethod
doc ///
    Key
        SubductionMethod
        [(intersect, Subring, Subring), SubductionMethod]
        [(forceSB,SAGBIBasis),SubductionMethod]
        [(forceSB,Subring),SubductionMethod]
        [(isSAGBI,Matrix),SubductionMethod]
        [(isSAGBI,List),SubductionMethod]
        [(isSAGBI,SAGBIBasis),SubductionMethod]
        [(isSAGBI,Subring),SubductionMethod]
        [sagbi,SubductionMethod]
        [sagbiBasis,SubductionMethod]
        [subalgebraBasis,SubductionMethod]
        [subduction,SubductionMethod]
    Headline
        Subduction method for the Sagbi algorithm
    Description
        Text
            There are two possible values for @TT "Strategy"@:
            @TT "Top"@ and @TT "Engine"@.

            There are two implementations to subduct the $S$-pairs
            in the function @TO "sagbi"@.
            Both implementations follow the same procedure,
            but are coded in different parts of @TT "Macaulay2"@.
            Subduction using the @TT "Strategy"@ @TT "Top"@ (default)
            is generally faster for smaller inputs.
            Subduction using the @TT "Engine"@ implementation
            can be faster for larger subalgebra bases.

            When using the @TT "Top"@ @TT "Strategy"@,
            the status of the computation using @TO "sagbi"@
            can be displayed using the @TO "PrintLevel"@ option.
    SeeAlso
        sagbi
        AutoSubduce
        ReduceNewGenerators
        StorePending
        "Subduction strategies"
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
        (intersect,Subring,Subring)
        forceSB
        isSAGBI
        sagbiBasis
        subalgebraBasis
        subduction
///
-- AutoSubduceOnPartialCompletion
doc ///
    Key
        AutoSubduceOnPartialCompletion
        [sagbi,AutoSubduceOnPartialCompletion]
        [sagbiBasis,AutoSubduceOnPartialCompletion]
        [subalgebraBasis,AutoSubduceOnPartialCompletion]
        [subduction,AutoSubduceOnPartialCompletion]
    Headline
        Subduct subalgebra generators at the end of the sagbi algorithm
    Description
        Text
            During the main loop in @TO "sagbi"@,
            if no new subalgebra generators are added and the
            @TO "AutoSubduceOnPartialCompletion"@ flag
            is @ TT "true" @ then all current subalgebra generators
            are subducted against each other.
            The autosubduction can be a significant time investment,
            but it can speed up subsequent computations by removing redundant generators.
    SeeAlso
        sagbi
        subalgebraBasis
        AutoSubduce
        ReduceNewGenerators
        subduction
        SubductionMethod
        PrintLevel
        Recompute
        ReduceNewGenerators
        RenewOptions
///
-- PrintLevel
doc ///
    Key
        PrintLevel
        [(intersect, Subring, Subring), PrintLevel]
        [(forceSB,SAGBIBasis),PrintLevel]
        [(forceSB,Subring),PrintLevel]
        [(isSAGBI,Matrix),PrintLevel]
        [(isSAGBI,List),PrintLevel]
        [(isSAGBI,Subring),PrintLevel]
        [(isSAGBI,SAGBIBasis),PrintLevel]
        [sagbi,PrintLevel]
        [sagbiBasis,PrintLevel]
        [subalgebraBasis,PrintLevel]
        [subduction,PrintLevel]
    Headline
        Levels of information displayed during Sagbi algorithm
    Description
        Text
            @TT "PrintLevel"@ is an optional input to @TO "sagbi"@
            and the methods that call it to
            control the amount of internal computational data to be displayed for debugging and
            informational purposes.
            Higher values for @TT "PrintLevel"@ result in more verbose output
            (especially Levels 5 and 6).

            @UL{
            "Level 0: No additional output",
            "Level 1+: some basic information each computation loop (but no polynomials):
            computation degree, number of S-pairs, number of new generators,
            termination conditions.",
            "Level 2+: basic polynomials: S-pairs and new subalgebra generators.",
            "Level 3+: extra polynomials: reduction ideal generators,
            zeroGens, current subalgebra generators",
            "Level 4+: the input and output of each subduction",
            "Level 5+: the processPending data",
            "Level 6+: the intermediate steps for top-level subduction"
            }@
    SeeAlso
        sagbi
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        RenewOptions
        forceSB
        isSAGBI
        (intersect,Subring,Subring)
        sagbi
        sagbiBasis
        subalgebraBasis
        subduction
        "Subduction strategies"
///
-- Recompute
doc ///
    Key
        Recompute
        [(isSAGBI,Matrix),Recompute]
        [(isSAGBI,List),Recompute]
        [(isSAGBI,SAGBIBasis),Recompute]
        [(isSAGBI,Subring),Recompute]
        [sagbi,Recompute]
        [sagbiBasis,Recompute]
        [subalgebraBasis,Recompute]
        [subduction,Recompute]
    Headline
        Flag for restarting a sagbi or isSAGBI computation
    Description
        Text
            If the flag is set to @ TT "true" @,
            then the subalgebra bases computation starts from scratch.
            This process constructs @ ofClass SAGBIBasis @ and uses the computation options
            from the previous computation.

            If the flag is set to @TT "false"@ (the typical case),
            then previous computations are used to speed up subsequent calculations.
    SeeAlso
        sagbi
        isSAGBI
        sagbiBasis
        subduction
        subalgebraBasis
        Compute
        ModifySAGBIBasis
        PrintLevel
        RenewOptions
        ReduceNewGenerators
        ModifySAGBIBasis
///
-- RenewOptions
doc ///
    Key
        RenewOptions
        [(forceSB,SAGBIBasis),RenewOptions]
        [(forceSB,Subring),RenewOptions]
        [(isSAGBI,Matrix),RenewOptions]
        [(isSAGBI,List),RenewOptions]
        [(isSAGBI,Subring),RenewOptions]
        [(isSAGBI,SAGBIBasis),RenewOptions]
        [sagbi,RenewOptions]
        [sagbiBasis,RenewOptions]
        [subalgebraBasis,RenewOptions]
        [subduction,RenewOptions]
    Headline
        Flag for reselecting the options for a subalgebra bases computation
    Description
        Text
            When a subalgebra bases computation is initialized (for the first time),
            the options of that computation
            are stored in the computation object. Whenever the computation is resumed,
            the same options will apply to the computation, with the exception of
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@.
            To change the other options in the computation, set this flag to @ TT "true" @.
            The new options will be used and stored in the computation object.
    SeeAlso
        sagbi
        forceSB
        isSAGBI
        sagbiBasis
        subalgebraBasis
        subduction
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        "Subduction computation limit"
        "Subduction strategies"
        PrintLevel
        Recompute
///
-- subduction
doc ///
    Key
        subduction
        (subduction, SAGBIBasis, Matrix)
        (subduction, SAGBIBasis, RingElement)
        (subduction, Matrix, Matrix)
        (subduction, Matrix, RingElement)
        (subduction, Subring, Matrix)
        (subduction, Subring, RingElement)
        (subduction, List, List)
        (subduction, List, RingElement)
    Headline
        Subduction against a set elements
    Usage
        N = subduction(SB, M)
        n = subduction(SB, m)
        N = subduction(S, M)
        n = subduction(S, m)
        N = subduction(F, M)
        n = subduction(F, m)
        nList = subduction(FList, mList)
        n = subduction(FList, m)
    Inputs
        SB:SAGBIBasis
            whose subalgebra generators are subducted against
        S:Subring
            whose generators are subducted against
        F:Matrix
            whose entries are subducted against
        FList:List
            whose entries are subducted against
        M:Matrix
            a 1-row matrix whose entries are to be subducted
        m:RingElement
            the element to be subducted
        mList:List
            a list of elements to be subducted
        AutoSubduce=>Boolean
            a flag indicating when to perform autosubduction on the generators
            before performing the Subalgebra basis computation
        ReduceNewGenerators=>Boolean
            a flag indicating whether to reduce (via Gaussian elimination) to
            new subalgebra generators before adding them to the current subalgebra basis
        StorePending=>Boolean
            a flag that indicates whether the {\it pending list}
            should be stored in the result.
        Strategy=>String
            the update strategy at the beginning of each loop:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method that combines the other two;
            starting with "DegreeByDegree" for low degrees and switching to "Incremental".
        SubductionMethod=>String
            the method used for subduction either: "Top" or "Engine".
        Limit=>ZZ
            a degree limit for the binomial S-pairs that are computed internally.
        AutoSubduceOnPartialCompletion=>Boolean
            apply autosubduction to the subalgebra generators the first time
            no new generators are added.
            Use this only if very few new subalgebra generators are expected.
        PrintLevel=>ZZ
            When this is greater than zero, information is printed
            about the progress of the computation
        Recompute=>Boolean
            if @ TT "true" @ then the computation will resume,
            otherwise it starts at the beginning
        RenewOptions=>Boolean
            if @ TT "true" @ then the computation uses the options specified,
            otherwise it uses the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@)
    Outputs
        N:Matrix
            entries of $M$ after subduction
        n:RingElement
            the element $m$ after subduction
        nList:List
            the elements of $mList$ after subduction
    Description
        Text
            Performs subduction on the input with respect to
            the generating set of the supplied object.
            The result of subduction of $f$ by $S$ is
            an element $g$ such that $f-g$ is in $S$.
            If the generating set is not a subalgebra basis,
            then the result of subduction may not be
            a normal form for the input.
        Example
            R = QQ[x,y];
            S = subring {x+y, x*y, x*y^2};
            f = x^4;
            subduction(S, f)
            sagbi S;
            subduction(gens sagbi S, f)
    SeeAlso
        sagbi
        Subring
        SAGBIBasis
        (symbol %, RingElement, Subring)
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
        "Subduction computation limit"
        "Subduction strategies"
        groebnerSubductionQuotient
        (symbol //, RingElement, Subring)
///
-- #######################################
-- ##    sagbi-functions.m2 documentation ##
-- #######################################
-- isSAGBI
doc ///
    Key
        isSAGBI
        (isSAGBI, Matrix)
        (isSAGBI, List)
    Headline
        Check if the generators are a subalgebra basis
    Usage
        result = isSAGBI S
        result = isSAGBI SB
        result = isSAGBI M
        result = isSAGBI L
    Inputs
        S:Subring
        SB:SAGBIBasis
        M:Matrix
        L:List
    Outputs
        result:Boolean
            if @TT "true"@, then the generators or subalgebra generators
            form a subalgebra basis.
    Description
        Text
            Checks whether the generators of @ ofClass Subring @ or
            the subalgebra generators of @ ofClass SAGBIBasis @ form a subalgebra basis.
            If @TT "isSAGBI"@ is supplied @ofClass Matrix@ or @ofClass List@ @TT "A"@,
            then the command
            @TT "isSAGBI"@ @TT "A"@ is equivalent to @TT "isSAGBI"@ @TO "subring"@ @TT "A"@.
            For further details of their respective uses see
            @ TO (isSAGBI, Subring) @ and @ TO (isSAGBI, SAGBIBasis) @.

            The result of @ TT "isSagbi" @ is stored in the cache of
            @ ofClass Subring @ or the sagbiStatus of
            @ ofClass SAGBIBasis @. If @ TT "isSagbi" @ is called again,
            then the result is looked up, unless
            the option @TO "Recompute"@ is set to @TT "true"@.
        Example
            R = QQ[x,y,z];
            S = subring {x+y+z,x*y+x*z+y*z, x*y*z, (x-y)*(x-z)*(y-z)};
            isSAGBI S
            isSAGBI sagbi(S, Limit => 5)
            S' = subring {x+y,x*y,x*y^2,x*y^4-y};
            isSAGBI S'
            isSAGBI sagbi(S', Limit => 10)
        Text
            If @TT "isSAGBI"@ is supplied @ ofClass SAGBIBasis @,
            then the generators of its subring can be checked for being
            a subalgebra basis by setting @TO "UseSubringGens"@ to @ TT "true" @.
            If @TT "isSAGBI"@ is supplied @ ofClass Subring @, then the generators of
            a partial subalgebra basis can be checked for being a
            subalgebra basis by setting @TO "UseSubringGens"@ to @ TT "false" @.

            Optional inputs for specific uses
            (see @TO (isSAGBI,Subring)@ and @TO (isSAGBI,SAGBIBasis)@)
            include the following: @TO "Compute"@, @TO "Recompute"@,
            @TO "Subduction strategies"@,
            @TO "SubductionMethod"@, @TO "PrintLevel"@,
            @TO "RenewOptions"@, @TO "UseSubringGens"@, and
            @TO "ModifySAGBIBasis"@.
    SeeAlso
        (isSAGBI, Subring)
        (isSAGBI, SAGBIBasis)
        SAGBIBasis
        (gens, SAGBIBasis)
        (ambient, SAGBIBasis)
        sagbi
///
-- (isSAGBI, Subring)
doc ///
    Key
        (isSAGBI, Subring)
    Headline
        Check if the generators are a subalgebra basis
    Usage
        result = isSAGBI S
    Inputs
        S:Subring
        Compute=>Boolean
            a flag indicating whether to perform computation.
        Recompute=>Boolean
            if @ TT "true" @ then cached results are ignored
            and a computation is performed to verify
            if the generators are a subalgebra basis.
        Strategy=>String
            the update strategy used for updating the SAGBIBasis:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method
            that combines the other two; starting with "DegreeByDegree"
            for low degrees and switching to "Incremental".
        SubductionMethod=>String
            the method used for subduction either: "Top" or "Engine".
        PrintLevel=>ZZ
            When this is greater than zero,
            information is printed about the progress of the computation.
        RenewOptions=>Boolean
            if @ TT "true" @ then the computation will use the options specified,
            otherwise it will use the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@)
        UseSubringGens=>Boolean
            a flag indicating whether to use the generators of the subring.
            If @ TT "false"@, then the method uses the subalgebra generators
            of the partial subalgebra basis.
        ModifySAGBIBasis=>Boolean
            a flag indicating whether to replace the partial subalgebra basis
            cached in the subring.
    Outputs
        result:Boolean
            if @TT "true"@, then the generators or subalgebra generators
            form a subalgebra basis.
    Description
        Text
            Checks whether the generators of @ ofClass Subring @ $S$ form a subalgebra basis.
            The result of the computation is stored in the @ TT "cache" @ of the ring.
            Subsequent calls to @TT "isSAGBI"@ on $S$ with the same options will look up the
            cached result unless the option @ TO "Recompute"@ is set to @TT "true"@.
            If the option @TO "Compute"@ is set to @TT "false"@ then @TT "isSAGBI"@
            attempts to find a previously cached result. If @TT "isSAGBI"@ cannot determine
            a result without computation and the option @TO "Compute"@ is set to @TT "false"@,
            then @TT "isSAGBI"@ returns @ TT "null" @.

            If the generators of $S$ form a subalgebra basis,
            then the subalgebra generators of the
            cached partial subalgebra basis are updated to match the ring
            generators, unless the option @ TO "ModifySAGBIBasis" @ is set to @ TT "false"@.
            If the option @TO "UseSubringGens"@ is set to @TT "false"@, then the generators
            of a partial subalgebra basis are used instead of the generators of the subring.
        Example
            R = QQ[x,y,z];
            S = subring {x+y+z,x*y+x*z+y*z, x*y*z, (x-y)*(x-z)*(y-z)};
            isSAGBI S
            S' = subring {x+y+z,x*y+x*z+y*z, x*y*z};
            isSAGBI S'
        Text
            The options @TO "Subduction strategies"@, @TO "SubductionMethod"@,
            @TO "PrintLevel"@, and @TO "RenewOptions"@ are
            only used when performing a subalgebra bases computation.
    SeeAlso
        isSAGBI
        (isSAGBI, SAGBIBasis)
        Subring
        sagbi
        Compute
        Recompute
        "Subduction strategies"
        SubductionMethod
        PrintLevel
        RenewOptions
        UseSubringGens
        ModifySAGBIBasis
///
-- (isSAGBI, SAGBIBasis)
doc ///
    Key
        (isSAGBI, SAGBIBasis)
    Headline
        Check if the generators are a subalgebra basis
    Usage
        result = isSAGBI SB
    Inputs
        SB:SAGBIBasis
        Compute=>Boolean
            a flag indicating whether to perform computation.
        Recompute=>Boolean
            if @ TT "true" @ then sagbiStatus is ignored
            and a computation is performed to verify
            if the subalgebra generators are a subalgebra basis.
        Strategy=>String
            the update strategy used for updating the SAGBIBasis:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method
            that combines the other two; starting with "DegreeByDegree"
            for low degrees and switching to "Incremental".
        SubductionMethod=>String
            the method used for subduction either: "Top" or "Engine".
        PrintLevel=>ZZ
            When this is greater than zero,
            information is printed about the progress of the computation.
        RenewOptions=>Boolean
            if @ TT "true" @ then the computation uses the options specified,
            otherwise it will use the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@).
        UseSubringGens=>Boolean
            a flag indicating whether to use the generators of the subring
            of the partial subalgebra basis.
            If @ TT "false"@ then the method uses the subalgebra generators
            of the partial subalgebra basis.
        ModifySAGBIBasis=>Boolean
            a flag indicating whether to replace the partial subalgebra basis
            cached in the subring.
    Outputs
        result:Boolean
    Description
        Text
            Checks whether the subalgebra generators of
            @ ofClass SAGBIBasis @ $SB$ form a subalgebra basis.
            The result of the computation is stored in the sagbiStatus of $SB$.
            Subsequent calls of @TT "isSAGBI"@ on $SB$ will look up the
            result of the sagbiStatus unless the option
            @ TO "Recompute"@ is set to @TT "true"@.
            If the option @TO "Compute"@ is set to @TT "false"@, then @TT "isSAGBI"@
            attempts to find a previously cached result. If @TT "isSAGBI"@ cannot determine
            a result without computation and the option @TO "Compute"@ is set to @TT "false"@,
            then @TT "isSAGBI"@ returns @ TT "null" @.

            If the subalgebra generators of $SB$ form a subalgebra basis,
            then the partial subalgebra basis is
            updated, unless @ TO "ModifySAGBIBasis" @ is set to @ TT "false"@.
            If the option @TO "UseSubringGens"@ is set to @TT "true"@, then the generators
            of the subring of $SB$ are used instead of the subalgebra generators
            of the subring.
        Example
            R = QQ[x,y,z];
            S = subring {x+y+z,x*y+x*z+y*z, x*y*z, (x-y)*(x-z)*(y-z)};
            isSAGBI S
            isSAGBI sagbi(S, Limit => 5)
            S' = subring {x+y,x*y,x*y^2,x*y^7-y};
            isSAGBI S'
            isSAGBI sagbi(S', Limit => 5)
            isSAGBI sagbi(S', Limit => 10)
        Text
            The options @TO "Subduction strategies"@, @TO "SubductionMethod"@,
            @TO "PrintLevel"@, and @TO "RenewOptions"@ are
            only used when performing a subalgebra bases computation.
    SeeAlso
        isSAGBI
        (isSAGBI, Subring)
        SAGBIBasis
        (gens, SAGBIBasis)
        (ambient, SAGBIBasis)
        (sagbiStatus, SAGBIBasis)
        sagbi
        Compute
        Recompute
        "Subduction strategies"
        SubductionMethod
        PrintLevel
        RenewOptions
        UseSubringGens
        ModifySAGBIBasis
///
-- Compute
doc ///
    Key
        Compute
        [(isSAGBI,SAGBIBasis),Compute]
        [(isSAGBI,Subring),Compute]
        [(intersect, Subring, Subring), Compute]
        [(isSAGBI,Matrix),Compute]
        [(isSAGBI,List),Compute]
    Headline
        Flag for performing computations while checking the completeness of a subalgebra basis
    Description
        Text
            A flag that determines whether additional computations
            may be performed.    When the flag is set
            to @TT "false"@, the function being called only uses the results
            of prior computations, if they have been stored.    When previous
            computations have not been performed, the function usually returns
            @TT "null"@.

            When the flag is set to @TT "true"@, additional computations will
            be performed if the prior computations are insufficient.
        Example
            R = QQ[x,y];
            S = subring{x+y,x*y};
            isSAGBI(S,Compute=>false)
            isSAGBI(S,Compute=>true)
            sagbi S;
            isSAGBI(S,Compute=>false)
    SeeAlso
        Recompute
        RenewOptions
///
-- UseSubringGens
doc ///
    Key
        UseSubringGens
        [(forceSB,SAGBIBasis),UseSubringGens]
        [(forceSB,Subring),UseSubringGens]
        [(isSAGBI,Matrix),UseSubringGens]
        [(isSAGBI,List),UseSubringGens]
        [(isSAGBI,Subring),UseSubringGens]
        [(isSAGBI,SAGBIBasis),UseSubringGens]
    Headline
        Flag for using the subring generators when checking a subalgebra basis
    Description
        Text
            When @TO "isSAGBI"@ is supplied @ ofClass SAGBIBasis @ or @ ofClass Subring @,
            there are two generating sets which can be used:
            either the generators of @ofClass Subring@ or
            the subalgebra generators of @ofClass SAGBIBasis@
            (which may be stored in the cache of
            @ofClass Subring@).

            If the flag is set to @TT "false"@,
            then the check is applied to the subalgebra generators of $SB$.
            If the flag is set to @TT "true"@,
            then the check is applied to the generators of the subring of $SB$.
    SeeAlso
        isSAGBI
        (isSAGBI,Subring)
        (isSAGBI,SAGBIBasis)
///
-- ModifySAGBIBasis
doc ///
    Key
        ModifySAGBIBasis
        [(isSAGBI,Matrix),ModifySAGBIBasis]
        [(isSAGBI,List),ModifySAGBIBasis]
        [(isSAGBI,Subring),ModifySAGBIBasis]
        [(isSAGBI,SAGBIBasis),ModifySAGBIBasis]
    Headline
        Flag for allowing the cached computation object to be modified
    Description
        Text
            When @TO "isSAGBI"@ is supplied @ ofClass Subring @ $S$,
            then this option determines
            whether the cached instance of @ ofClass SAGBIBasis @ may be replaced.
        Example
            R = QQ[x,y];
            S = subring {x^2, y^3};
            SB = sagbi(S, Limit => 2);
            S.cache#SAGBIBasis === SB
            isSAGBI(S, ModifySAGBIBasis => false)
            S.cache#SAGBIBasis === SB
            isSAGBI(S, ModifySAGBIBasis => true)
            S.cache#SAGBIBasis === SB
    SeeAlso
        isSAGBI
///
-- groebnerMembershipTest
doc ///
    Key
        groebnerMembershipTest
        (groebnerMembershipTest, RingElement, Subring)
    Headline
        Extrinsic method for subring membership
    Usage
        result = groebnerMembershipTest(f, S)
    Inputs
        f:RingElement
        S:Subring
    Outputs
        result:Boolean
            which indicates if $f$ belongs to $S$.
    Description
        Text
            Uses the extrinsic method (based on Groebner bases)
            to test membership of a polynomial in a subring.
        Example
            R = QQ[x,y];
            S = subring {x^2 - x*y, x^4};
            f1 = x^2*y^2 - 2*x^3*y;
            groebnerMembershipTest(f1, S)
            f2 = x^10;
            groebnerMembershipTest(f2, S)
        Text
            If a subalgebra basis is known for $S$,
            then it is recommended to use @TO "RingElement % Subring"@ instead.
    SeeAlso
        Subring
        (symbol %, RingElement, Subring)
        groebnerSubductionQuotient
        (symbol //, RingElement, Subring)
        sagbi
///
-- groebnerSubductionQuotient
doc ///
    Key
        groebnerSubductionQuotient
        (groebnerSubductionQuotient, RingElement, Subring)
    Headline
        Extrinsic method for subduction quotients
    Usage
        result = groebnerSubductionQuotient(f, S)
    Inputs
        f:RingElement
        S:Subring
    Outputs
        result:RingElement
            the normal form of $f$ in the @TO "presentationRing"@
    Description
        Text
            Uses the extrinsic method to compute a subduction quotient of $f$ by the
            generators of $S$.
            A subduction quotient is an element of the @TO "presentationRing"@.
            The variables of this ring correspond to the generators of $S$. A subduction
            quotient is a polynomial representation of
            $f - f\%S$ in terms of the generators of $S$.
            The value of this element in the @TO (ambient,Subring)@ ring can be recovered using
            the field @TT "presentationMap"@ of the subring.
        Example
            R = QQ[x,y];
            S = subring {x+y, x*y, x*y^2};
            f = x^3;
            f - (f%S)
            gs = groebnerSubductionQuotient(f, S)
            S#"presentationMap" gs
    SeeAlso
        Subring
        subring
        (symbol %, RingElement, Subring)
        (symbol //, RingElement, Subring)
        sagbi
        groebnerMembershipTest
///
-- symbol //
doc ///
    Key
        (symbol //, RingElement, Subring)
    Headline
        Subduction quotient with respect to a subring
    Usage
        r = f // S
    Inputs
        f:RingElement
        S:Subring
    Outputs
        r:RingElement
            a subduction quotient of f with respect to the subring
    Description
        Text
            The result is a ring element that lies in the @TO "presentationRing"@
            that has one variable for each
            generator of the subring $S$.
            A subduction quotient is an expression of $f - (f\%S) \in S$
            in terms of the generators of $S$.
            The value of this element in the @TO (ambient,Subring)@ ring can be recovered using
            the field @TT "presentationMap"@ of the subring.
        Example
            R = QQ[x,y];
            S = subring {x+y,x*y,x*y^2};
            f = x^3;
            f - (f%S)
            gs = f // S
            S#"presentationMap" gs
        Text
            This function calls @TO "groebnerSubductionQuotient"@.
    SeeAlso
        groebnerSubductionQuotient
        (symbol %, RingElement, Subring)
        (symbol %, Matrix, Subring)
        (symbol %, RingElement, SAGBIBasis)
        (symbol %, Matrix, SAGBIBasis)
        Subring
        SAGBIBasis
        groebnerMembershipTest
///
-- symbol %
doc ///
    Key
        "Reduction in subrings"
        (symbol %, RingElement, Subring)
        (symbol %, Matrix, Subring)
        (symbol %, RingElement, SAGBIBasis)
        (symbol %, Matrix, SAGBIBasis)
    Headline
        Remainder modulo a subring
    Usage
        r = f % S
        R = M % S
        r = f % SB
        R = M % SB
    Inputs
        f:RingElement
        M:Matrix
        S:Subring
        SB:SAGBIBasis
    Outputs
        r:RingElement
            the normal form of f modulo the Subring or SAGBIBasis
        R:Matrix
            the normal form of entries of M modulo the subring or the SAGBIBasis
    Description
        Text
            The result is zero if and only if the input belongs to the subring.
            If a subalgebra basis is known for the subring,
            then subduction is used to compute the normal forms.
            If no subalgebra basis is known,
            then an extrinsic method is used, similar to @TT "groebnerMembershipTest"@.
        Example
            R = QQ[x1, x2, x3];
            --usual invariants of A_3
            S = subring {x1+x2+x3, x1*x2+x1*x3+x2*x3, x1*x2*x3, (x1-x2)*(x1-x3)*(x2-x3)}
            f = x1 + x2 + 2*x3
            f % S
            g = x1^2*x2 + x2^2*x3 + x3^2*x1
            g % S
    SeeAlso
        Subring
        SAGBIBasis
        groebnerMembershipTest
///
-- SAGBILimitType
doc ///
    Key
        SAGBILimitType
        [(intersect, Subring, Subring), SAGBILimitType]
    Headline
        Flag for calculating degree limit in intersections
    Description
        Text
            There are two possible values for @TT "SAGBILimitType"@:
            @TT "Fixed"@ or @TT "Function"@.
            Determines the stopping criterion for the subalgebra basis
            computation performed by the function @TO (intersect, Subring, Subring)@.
            This function creates
            @ofClass Subring@ $S$, called the {\it composite subring}, from the input algebras    
            and attempts to compute a subalgebra basis for $S$.
            The @TO2 {"Subduction computation limit","Limit"}@ of this subalgebra basis
            computation is controlled by the two options @TT "SAGBILimitType"@ and
            @TO2 {"Subduction computation limit","Limit"}@.

            If @TT "SAGBILimitType"@ is set to @TT "Fixed"@,
            then the @TO2 {"Subduction computation limit","Limit"}@ of the
            subalgebra basis computation is set to the value of
            the option @TO2 {"Subduction computation limit","Limit"}@.
            Otherwise, if @TT "SAGBILimitType"@ is set to @TT "Function"@,
            then the @TO2 {"Subduction computation limit","Limit"}@ of the
            subalgebra basis computation is set to the product of
            the degrees of the maximum degree generators
            of the input algebras.
        Example
            R = QQ[x,y];
            I = ideal(x^3 + x*y^2 + y^3);
            Q = R/I;
            S1 = subring {x^2, x*y};
            S2 = subring {x, y^2};
            gens (S3 = intersect(S1, S2, SAGBILimitType => "Fixed", Limit => 5))
            gens (S4 = intersect(S1, S2, SAGBILimitType => "Fixed", Limit => 10))
            gens (S5 = intersect(S1, S2, SAGBILimitType => "Function"))
        Text
            By default, the option @TT "SAGBILimitType"@ is set to @TT "Fixed"@,
            and the default @TO2 {"Subduction computation limit","Limit"}@ is $20$.
            For more information see @TO (intersect, Subring, Subring)@.
    SeeAlso
        (intersect, Subring, Subring)
        IntersectedSubring
        sagbi
        SAGBILimitType
        CheckFullIntersection
        "Subduction computation limit"
///
-- CheckFullIntersection
doc ///
    Key
        CheckFullIntersection
        [(intersect, Subring, Subring), CheckFullIntersection]
    Headline
        Flag for deciding if isFullIntersection is called
    Description
        Text
            If the option is set to @TT "true"@,
            then the function @TO (intersect, Subring, Subring)@ performs
            two tests. If either of these tests fail,
            then a relevant warning message is displayed.
            Other than the warning messages,
            the output of the function @TO (intersect, Subring, Subring)@ is not affected
            by the value of this option.
            If the option @TT "CheckFullIntersection"@ is set to @TT "false"@ then
            no warnings are produced.
        Text
            The first test is performed at the beginning of the function call. It
            checks whether any of the inputs is @ofClass IntersectedSubring@ and
            applies @TO "isFullIntersection"@ to each instance.
            For more details see @TO "isFullIntersection"@.
            If the function @TO "isFullIntersection"@ returns false,
            i.e., the generators of the input are not
            guaranteed to generate the full intersection, then a warning is printed.
        Example
            R = QQ[x,y];
            S1 = subring {x^2, y^2};
            S2 = subring {x*y};
            gens (S3 = intersect(S1, S2, Limit => 6, CheckFullIntersection => false))
            isFullIntersection S3
            gens (S4 = intersect(S3, S2, Limit => 6))
            isFullIntersection S4
        Text
            In the example above,
            a warning is produced when computing $S4$ because the input algebra $S3$ is
            not guaranteed to be the full intersection.
            However, the final line shows that the generators of $S4$
            are guaranteed to generate the full intersection of $S2$ and $S3$.

            The second test is performed at the end of the function call.
            It checks if the result is
            guaranteed to generate the full intersection by calling
            @TO "isFullIntersection"@ on the result.
            If the function @TO "isFullIntersection"@ returns @TT "false"@,
            then a warning printed. For instance,
            in the example above, the computation of $S3$ would print a warning.
        Example
            intersect(S1, S2, Limit => 6)
        Text
            If this warning message appears,
            then it is recommended to set the option @TO "SAGBILimitType"@ to its
            default value of @TT "Fixed"@ or to use a
            higher value for the option @TO2 {"Subduction computation limit","Limit"}@.
    SeeAlso
        (intersect, Subring, Subring)
        isFullIntersection
        SAGBILimitType
        IntersectedSubring
        "Subduction computation limit"
///
-- intersect
doc ///
    Key
        (intersect, Subring, Subring)
    Headline
        Intersection of subrings
    Usage
        result = intersect(S1, S2)
    Inputs
        S1:Subring
        S2:Subring
        Strategy=>String
            the update strategy at the beginning of each loop:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method that combines the other two;
            starting with "DegreeByDegree" for low degrees and switching to "Incremental".
        SubductionMethod=>String
            the method used for subduction.
            @TO "SubductionMethod"@ can be either "Top" or "Engine".
        SAGBILimitType=>String
            Either "Fixed" or "Function".
            Determines the stopping criterion for the subalgebra bases computation.
            If "Fixed" then the @TO2 {"Subduction computation limit","Limit"}@ is used,
            otherwise, if "Function" is selected,
            then the maximum degree of the input generators is used as the degree limit.
        Limit=>ZZ
            a degree limit for the binomial S-pairs that are computed internally.
        PrintLevel=>ZZ
            which determines how much information is printed
            about the progress of the computation.
            Larger positive values of @TO "PrintLevel"@ result in more output.
        CheckFullIntersection=>Boolean
            which determines if @TO "isFullIntersection"@
            is run on the output @TO "IntersectedSubring"@.
        Compute=>Boolean
            which determines if the check to see if the computed generators
            form a subalgebra basis of the ring they generate is performed.
    Outputs
        result:IntersectedSubring
    Description
        Text
            Computes the intersection of input subrings of the same ambient ring.
        Example
            R = QQ[x,y];
            I = ideal(x^3 + x*y^2 + y^3);
            Q = R/I;
            S1 = subring {x^2, x*y};
            S2 = subring {x, y^2};
            gens intersect(S1, S2, Limit=>5)
            S = intersect(S1, S2);
            gens S
            isFullIntersection S
            isSAGBI S
        Text
            The generators of the resulting @TO "IntersectedSubring"@
            form a partial subalgebra basis
            for the computed intersection.
            When @TO "CheckFullIntersection"@ is true,
            the function automatically checks if the
            generators of the resulting @TO "IntersectedSubring"@ generate the
            intersection of the given subrings.
            The generators of an instance of @TO "IntersectedSubring"@
            lie in both input algebras.
            However, if the function cannot guarantee that the generators
            of the resulting @TO "IntersectedSubring"@
            generate the full intersection then a warning is given.

            The function works by creating a {\it composite subring} $S$
            from the input algebras
            and attempts to compute a subalgebra basis for $S$.
            If a subalgebra basis $G$ for $S$
            is found, then a subset of $G$ forms a subalgebra basis of the intersection of
            the input algebras. Whenever this happens, the output is guaranteed to
            be the full intersection of the input algebras and, furthermore, form a subalgebra 
            basis for the intersection. The function @TO "isFullIntersection"@
            checks whether this is the case, i.e., whether a finite subalgebra
            basis for $S$ was found.
            If the function @TO "isFullIntersection"@ returns @TT "false"@
            then there are a few different possibilities: First, the composite subring $S$ may
            have a finite subalgebra basis that could be found by
            either setting the option @TO "SAGBILimitType"@ to its
            default value of @TT "Fixed"@ or by using
            a higher @TO2 {"Subduction computation limit","Limit"}@ option.
            Second, $S$ may not have a finite subalgebra basis but the intersection does have a
            finite subalgebra basis.
            In this case, the output may be a generating set for the full
            intersection,
            but cannot be verified by the algorithm.
            Third, the intersection of the
            input algebras may not have a finite subalgebra basis,
            and so the composite subring does
            not have a finite subalgebra basis.
            In this case case,
            the computed
            generators may be a generating set (but not a subalgebra basis)
            for the intersection but the algorithm cannot
            guarantee it.
            
            This function can also be run on a sequence containing both @TO "Subring"@ and 
            @TO "IntersectedSubring"@ types. When @TO "CheckFullIntersection"@ is true, all 
            intermediate intersections are checked for completeness.
        Example
            R = QQ[x,y];
            S1 = subring{x^2,y^3};
            S2 = subring{x,y^2};
            S3 = subring{x^4,y};
            gens intersect(S1,S2,S3)
    SeeAlso
        Subring
        IntersectedSubring
        isFullIntersection
        originalSubringGens
        isSAGBI
        "Subduction computation limit"
///
-- IntersectedSubring
doc ///
    Key
        IntersectedSubring
    Headline
        The type of all subrings arising from intersection
    Description
        Text
            @TT "IntersectedSubring"@ is a type of @TO "Subring"@ produced
            by intersecting two subrings using the function @TO "intersect"@.
            The function @TO "intersect"@ is the constructor for @TT "IntersectedSubring"@.
            Applying the function @TO "isFullIntersection"@ to an instance
            $S$ of @TT "IntersectedSubring"@
            checks whether
            the generators of $S$ are guaranteed to generate the intersection
            of the original subrings.

        Example
            R = QQ[x,y];
            S1 = subring{x^3,y};
            S2 = subring{x^2,y^2};
            S = intersect(S1,S2)
            S#"originalSubrings"
        Text
            The @TT "originalSubrings"@ field lists the two subrings
            which were used to generate the intersection.
    SeeAlso
        Subring
        (intersect, Subring, Subring)
        isFullIntersection
        originalSubringGens
///
-- isFullIntersection
doc ///
    Key
        isFullIntersection
        (isFullIntersection, IntersectedSubring)
    Headline
        check correctness of intersection computation
    Usage
        result = isFullIntersection(S)
    Inputs
        S:IntersectedSubring
    Outputs
        result:Boolean
            is @TT "true"@ to indicate that the generators of $S$ generate
            the intersection of the original subrings
    Description
        Text
            Checks if the algorithm for computing the intersection of
            two subrings in the function @TO "intersect"@
            produced a subalgebra basis for the intersection.
        Example
            R = QQ[x,y];
            I = ideal(x^3 + x*y^2 + y^3);
            Q = R/I;
            S1 = subring {x^2, x*y};
            S2 = subring {x, y^2};
            S = intersect(S1, S2);
            isFullIntersection S
        Text
            This test is one-sided: if the function @TO "isFullIntersection"@
            returns @ TT "false" @, then the generators of the given subring
            may still generate the intersection as a subalgebra basis.
        Example
            R = QQ[x,y];
            S1 = subring {x^3, x^2*y};
            S2 = subring {x^4, y};
            S = intersect(S1, S2, Limit => 15, CheckFullIntersection => false);
            gens S
            isFullIntersection S
            isSAGBI S
        Text
            If the function @TT "isFullIntersection"@ returns @TT "false"@
            then, there are a few different possibilities: First, the composite subring $S$ may
            have a finite subalgebra basis that could be found by setting
            the option @TO "SAGBILimitType"@ to its
            default value of @TT "Fixed"@ or by using
            a higher @TO2 {"Subduction computation limit","Limit"}@ option in @TO "intersect"@.
            Second, $S$ may not have a finite subalgebra basis but the intersection does have a
            finite subalgebra basis.
            In this case, the output may be a generating set for the full
            intersection, but cannot be verified by the algorithm.
            Third, the intersection of the
            input algebras may not have a finite subalgebra basis,
            and so the composite subring does
            not have a finite subalgebra basis.
            In this case,
            the computed
            generators may be a generating set (but not a subalgebra basis)
            for the intersection but the algorithm cannot
            guarantee it.
    SeeAlso
        (intersect, Subring, Subring)
        Subring
        IntersectedSubring
        "Subduction computation limit"
        SAGBILimitType
///
-- originalSubringGens
doc ///
    Key
        originalSubringGens
        (originalSubringGens, IntersectedSubring)
    Headline
        generators of the original subrings
    Usage
        result = originalSubringGens(S)
    Inputs
        S:IntersectedSubring
    Outputs
        result:List
            of the generators of the original subrings used in the creation of $S$
    Description
        Text
            Returns a list of two one-row matrices of the generators
            of the original subrings whose intersection
            (via the function @TO "intersect"@) is $S$.
        Example
            R = QQ[x,y];
            I = ideal(x^3 + x*y^2 + y^3);
            Q = R/I;
            S1 = subring {x^2, x*y};
            S2 = subring {x, y^2};
            S = intersect(S1, S2);
            originalSubringGens S
    SeeAlso
        (intersect, Subring, Subring)
        Subring
        IntersectedSubring
///
-- ##############################
-- ## classes.m2 documentation ##
-- ##############################
-- Subring
doc ///
    Key
        Subring
    Headline
        The type of all subrings
    Description
        Text
            @TT "Subring"@ is a type that stores information associated
            to a subring of a polynomial ring or quotient ring,
            such as a set of subring generators and
            a reference to the ring containing these generators.
            An instance of a @TT "Subring"@ is constructed with the function @TO "subring"@.
            For many uses, it is suggested to use a @ TT "Subring"@,
            as the computation objects (@TO "SAGBIBasis"@)
            are handled behind the scenes, and
            the user experience
            is more streamlined.

            Every instance of @TT "Subring"@ has the following keys
            (some of which are strings):

            @UL {
            {TT "ambientRing", ": The polynomial or quotient ring that
            contains the subring instance's generators."},
            {TT "generators", ": A one-row matrix,
            the generators of the subring."},
            {TT "cache", ": Contains data from previous
            computations to allow for more efficient computations."},
            {TT "presentationRing", ": the polynomial ring
            with one variable for each generator of the subring."},
            {TT "presentationMap", ": a map from the presentation
            ring to the ambient ring of the subring."}
            }@
    SeeAlso
        subring
        (gens, Subring)
        (ambient, Subring)
        (numgens, Subring)
        (net, Subring)
        (presentationRing, Subring)
        sagbi
        SAGBIBasis
        subalgebraBasis
        (flattenedRing, Subring)
///
-- subring
doc ///
    Key
        subring
        (subring, List)
        (subring, Matrix)
        (subring, SAGBIBasis)
        [subring, GeneratorSymbol]
    Headline
        Constructs a subring of a polynomial ring.
    Usage
        A = subring M
        A = subring L
        A = subring S
    Inputs
        M:Matrix
            whose entries are the generators for the constructed @TO "Subring"@.
        L:List
            containing generators for the constructed @TO "Subring"@.
        S:SAGBIBasis
        GeneratorSymbol=>Symbol
            a symbol to be used for the variables of the @ TT "presentationRing" @.
    Outputs
        A:Subring
            generated by the input generators.
    Description
        Text
            This function serves as the canonical constructor for the @TO "Subring"@ type.
            For many uses, it is suggested to use @ ofClass Subring@,
            as the computation objects (@TO "SAGBIBasis"@)
            are handled behind the scenes, and
            the user experience
            is more streamlined.
        Example
            R = QQ[x];
            S = subring {x^4+x^3, x^2+x}
            SB = sagbi S;
            gens SB
            (x^3+x^2)%S
    SeeAlso
        Subring
        (gens, Subring)
        (ambient, Subring)
        (numgens, Subring)
        (net, Subring)
        sagbi
        SAGBIBasis
///
-- (gens, Subring)
doc ///
    Key
        (gens, Subring)
    Headline
        A generating set of a subring
    Usage
        result = gens S
    Inputs
        S:Subring
        CoefficientRing=>Boolean
            unused option
    Outputs
        result:Matrix
    Description
        Text
            Returns a generating set for the @TO "Subring"@ as a one-row matrix.
        Example
            R = QQ[x,y,z];
            S = subring({x^2, y^2, z^2});
            gens S
    SeeAlso
        Subring
        subring
        (gens,SAGBIBasis)
///
-- (ambient, Subring)
doc ///
    Key
        (ambient, Subring)
    Headline
        The ambient ring of a subring
    Usage
        result = ambient S
    Inputs
        S:Subring
    Outputs
        result:Ring
    Description
        Text
            Returns the ambient ring of the generating set of the @TO "Subring"@.
            This may be a @TO "PolynomialRing"@ or @TO "QuotientRing"@.
        Example
            R = QQ[x,y,z];
            S = subring({x^2, y^2, z^2});
            ambient S
            Q = R / ideal(x^3 - y);
            S' = subring({x^3, y^2});
            ambient S'
    SeeAlso
        Subring
        subring
        (gens, Subring)
        (ambient, SAGBIBasis)
///
-- (numgens, Subring)
doc ///
    Key
        (numgens, Subring)
    Headline
        The number of generators of a subring
    Usage
        result = numgens S
    Inputs
        S:Subring
    Outputs
        result:ZZ
            the number of generators of S
    Description
        Text
            Returns the number of generators of a @TO "Subring"@.
        Example
            R = QQ[x,y,z];
            S = subring({x^2, y^2, z^2, x^2+y^2});
            numgens S
    SeeAlso
        Subring
        subring
        (gens, Subring)
        (ambient, Subring)
        (numgens, SAGBIBasis)
        (net, Subring)
        (flattenedRing, Subring)
///
-- (numgens, SAGBIBasis)
doc ///
    Key
        (numgens, SAGBIBasis)
    Headline
        The number of generators of a SAGBIBasis
    Usage
        result = numgens SB
    Inputs
        SB:SAGBIBasis
    Outputs
        result:ZZ
            the number of subalgebra generators of SB
    Description
        Text
            Returns the number of subalgebra generators of @ ofClass SAGBIBasis @.
        Example
            R = QQ[x,y,z];
            S = subring({x^2, y^2, z^2, x^2+y^2});
            SB = sagbi S;
            numgens SB
    SeeAlso
        Subring
        subring
        SAGBIBasis
        sagbiBasis
        (gens, SAGBIBasis)
        (ambient, SAGBIBasis)
        (numgens, SAGBIBasis)
        (net, Subring)
        (flattenedRing, SAGBIBasis)
///
-- (net, Subring)
doc ///
    Key
        (net, Subring)
    Headline
        Short summary of a subring
    Usage
        net S
    Inputs
        S:Subring
    Outputs
        :Net
            obtained by formatting S for printing.
    Description
        Text
            Displays the presentation and ambient rings of a @TO "Subring"@.
        Example
            R = QQ[x,y];
            S = subring{x^2,x*y^2,y^4}
    SeeAlso
        Subring
        subring
        (gens, Subring)
        (ambient, Subring)
        (numgens, Subring)
        (net, SAGBIBasis)
        (flattenedRing, Subring)
///
-- GeneratorSymbol
doc ///
    Key
        GeneratorSymbol
    Headline
        variables for the presentationRing
    Description
        Text
            A symbol to be used for variables of the @ TO "presentationRing" @.
        Example
            R = QQ[x,y];
            S = subring({x^2, y^2, x*y}, GeneratorSymbol => f)
            (x+y)^4 // S
        Text
            If a symbol is not specified, then the variables of the 
            @ TO "presentationRing" @ are indexed from 0 and use the default symbol.
            If a symbol is specified, then the indices start from 1.
    SeeAlso
        Subring
        subring
        presentationRing
///
-- SAGBIBasis
doc ///
    Key
        SAGBIBasis
    Headline
        The type of all subalgebra bases
    Description
        Text
            A computation object for subalgebra bases.
            It stores a partial subalgebra bases computation for
            picking up a computation where it left off.
            For many uses, it is superior to use @ ofClass Subring@,
            as the computation @TT "SAGBIBases"@
            are handled behind the scenes, and
            the user experience
            is more streamlined.
            To create @ ofClass SAGBIBasis @, use the function @TO "sagbiBasis"@.
            The @TT "SAGBIBasis" @ type is the main output type of the function @TO "sagbi"@.
            Applying @TO "subring"@ to a @TT "SAGBIBasis"@ returns the original subring that
            generated the @TT "SAGBIBasis"@.
    SeeAlso
        sagbiBasis
        Subring
        subring
        sagbi
        (gens,SAGBIBasis)
        (status,SAGBIBasis)
        (numgens,SAGBIBasis)
///
-- sagbiBasis
doc ///
    Key
        sagbiBasis
        (sagbiBasis,Subring)
    Headline
        Constructs a computation object from a subring.
    Usage
        B = sagbiBasis S
    Inputs
        S:Subring
        AutoSubduce=>Boolean
            a flag indicating when to perform autosubduction on
            the generators before performing the subalgebra basis computation
        ReduceNewGenerators=>Boolean
            a flag indicating whether to reduce (via Gaussian elimination)
            to new subalgebra generators before adding them to the current subalgebra basis
        StorePending=>Boolean
            a flag that indicates whether the {\it pending list}
            should be stored in the result.
        Strategy=>String
            the name of the update strategy at the beginning of each loop:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method that combines the other two;
            starting with "DegreeByDegree" for low degrees and switching to "Incremental".
        SubductionMethod=>String
            the name of the method used for subduction either: "Top" or "Engine".
        Limit=>ZZ
            a degree limit for the binomial S-pairs that are computed internally.
        AutoSubduceOnPartialCompletion=>Boolean
            a flag that indicates whether autosubduction is applied to
            the subalgebra generators the first time no new generators are added.
            Use this only if very few new subalgebra generators are expected.
        PrintLevel=>ZZ
            an option to produce additional output.
            When this is greater than zero,
            information is printed about the progress of the computation
        Recompute=>Boolean
            a flag that indicates if the computation will resume in subsequent runs,
            otherwise it starts at the beginning
        RenewOptions=>Boolean
            a flag that indicates if the computation will use the options specified,
            otherwise it will use the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@)
    Outputs
        B:SAGBIBasis
            a computation object associated to S
    Description
        Text
            This function serves as the canonical constructor for the @TO "SAGBIBasis"@ type.
            The @TT "SAGBIBasis"@ type is also the output of the function @TO "sagbi"@.
            The @ TO "Subring"@ used to construct a @TT "SAGBIBasis"@ may be recovered using
            @TO "subring"@
        Example
            R = QQ[x,y];
            S = subring{x+y,x*y,x*y^2};
            T = sagbiBasis S
            sagbi(T,Limit=>20)
            subring T
    SeeAlso
        sagbi
        subalgebraBasis
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
        Subring
        SAGBIBasis
        "Subduction strategies"
        "Subduction computation limit"
        forceSB
///
-- (gens, SAGBIBasis)
doc ///
    Key
        (gens, SAGBIBasis)
    Headline
        Returns generators of a SAGBIBasis computation object
    Usage
        result = gens S
    Inputs
        S:SAGBIBasis
        CoefficientRing=>Boolean
            unused option
    Outputs
        result:Matrix
    Description
        Text
            Returns a list of subalgebra generators of the
            @TO "Subring"@ underlying the @TO "SAGBIBasis"@.
            If the subalgebra generators do not form a complete subalgebra basis
            (for example if they were obtained from a partial computation),
            then the generators obtained from this command
            may not be a generating set for the original subring.
        Example
            R = QQ[x,y];
            S = subring {x*y, x^2*y, x^3*y};
            gens S
            SB = sagbi(S, Limit => 3);
            gens SB
            isSAGBI S
            isSAGBI SB
    SeeAlso
        SAGBIBasis
        (gens,Subring)
        sagbi
///
-- (net, SAGBIBasis)
doc ///
    Key
        (net, SAGBIBasis)
    Headline
        Short summary of a subalgebra basis computation object
    Usage
        net S
    Inputs
        S:SAGBIBasis
    Outputs
        :Net
            obtained by formatting S for printing.
    Description
        Text
            Displays the number of subalgebra generators of
            @ ofClass SAGBIBasis @ along with the degree limit.
        Example
            R = QQ[x,y];
            S = subring{x^2,x*y^2,y^4};
            sagbi S
    SeeAlso
        SAGBIBasis
        (gens, SAGBIBasis)
        (ambient, SAGBIBasis)
        (ring, SAGBIBasis)
        (net, Subring)
        (status, SAGBIBasis)
        sagbiLimit
        sagbi
///
-- (ambient, SAGBIBasis)
doc ///
    Key
        (ambient, SAGBIBasis)
    Headline
        The ambient ring of a SAGBIBasis computation object
    Usage
        result = ambient S
    Inputs
        S:SAGBIBasis
    Outputs
        result:Ring
    Description
        Text
            Returns the ambient ring of @ ofClass SAGBIBasis @.
            This is the ambient ring of the generating @TO "Subring"@
            This may be a @TO "PolynomialRing"@ or @TO "QuotientRing"@.
        Example
            R = QQ[x,y,z];
            S = sagbi {x^2, y^2, z^2}
            ambient S
            Q = R / ideal(x^3 - y);
            S' = sagbi {x^3, y^2}
            ambient S'
    SeeAlso
        Subring
        subring
        SAGBIBasis
        sagbiBasis
        (gens, SAGBIBasis)
        (ambient, Subring)
///
-- (ring, SAGBIBasis)
doc ///
    Key
        (ring, SAGBIBasis)
    Headline
        The lifted ring of a SAGBIBasis computation object
    Usage
        result = ring S
    Inputs
        S:SAGBIBasis
    Outputs
        result:Ring
    Description
        Text
            Returns the lifted ring of @ ofClass SAGBIBasis @.
            This is always @ ofClass PolynomialRing @, and the
            @TO2 {(flattenedRing,SAGBIBasis),"flattened ring"}@ is a quotient of this ring.
        Example
            R = QQ[x,y,z];
            S = sagbi {x^2, y^2, z^2}
            ring S
            Q = R / ideal(x^3 - y);
            S' = sagbi {x^3, y^2}
            ring S'
    SeeAlso
        SAGBIBasis
        (ambient, SAGBIBasis)
        (ring, SAGBIBasis)
        (net, SAGBIBasis)
        sagbi
        (flattenedRing, SAGBIBasis)
///
-- sagbiDegree
doc ///
    Key
        sagbiDegree
        (sagbiDegree, SAGBIBasis)
    Headline
        The current degree of the subalgebra bases computation
    Usage
        result = sagbiDegree SB
    Inputs
        SB:SAGBIBasis
    Outputs
        result:ZZ
            the current degree of computation in SB.
    Description
        Text
            Returns the current degree of the computation stored in @ ofClass SAGBIBasis @.
            When the computation is resumed (without restarting)
            then the algorithm will pick up at this degree.
            The degree may drop during the computation.
        Example
            R = QQ[x,y];
            S = subring{x+y,x*y,x*y^2,x*y^5+y};
            T = sagbiBasis S;
            sagbiDegree T
            sagbiDegree sagbi(S,Limit=>5)
            sagbiDegree sagbi(S,Limit=>20)
            isSAGBI(S,UseSubringGens=>false)
    SeeAlso
        SAGBIBasis
        sagbiBasis
        (gens, SAGBIBasis)
        (status, SAGBIBasis)
        sagbiLimit
        sagbi
///
-- sagbiLimit
doc ///
    Key
        sagbiLimit
        (sagbiLimit, SAGBIBasis)
    Headline
        The current limit of the subalgebra bases computation
    Usage
        result = sagbiLimit SB
    Inputs
        SB:SAGBIBasis
    Outputs
        result:ZZ
            the current upper bound on degree of a subalgebra bases computation
    Description
        Text
            Returns the current limit of the subalgebra bases computation.
            When the computation is resumed then
            this limit may be set to a new specified limit.
        Example
            R = QQ[x,y];
            S = subring{x+y,x*y,x*y^2,x*y^5+y};
            T = sagbiBasis S;
            sagbiLimit T
            sagbiLimit sagbi(S,Limit=>5)
            sagbiLimit sagbi(S,Limit=>30)
    SeeAlso
        SAGBIBasis
        sagbiBasis
        (gens, SAGBIBasis)
        (status, SAGBIBasis)
        sagbiDegree
        sagbi
///
-- sagbiStatus
doc ///
    Key
        sagbiStatus
        (sagbiStatus, SAGBIBasis)
    Headline
        checks if the subalgebra bases computation is done
    Usage
        result = sagbiStatus SB
    Inputs
        SB:SAGBIBasis
    Outputs
        result:Boolean
            is @TT "true"@ if the subalgebra basis computation is complete
    Description
        Text
            Returns @ TT "true" @ if the subalgebra bases computation
            is completed otherwise returns @ TT "false" @.
            It is possible that the computed subalgebra basis generators
            form a subalgebra basis while
            @TT "sagbiStatus"@ returns @TT "false"@.
            In this case, additional computation
            is needed to confirm that the computed generators
            form a subalgebra basis.
            For more descriptive output, see @TO (status,SAGBIBasis)@.
        Example
            R = QQ[x,y];
            S = subring {x+y,x*y,x*y^2,x*y^10+y};
            sagbiStatus sagbi(S,Limit=>10)
            sagbiStatus sagbi(S,Limit=>20)
    SeeAlso
        SAGBIBasis
        sagbiBasis
        (status, SAGBIBasis)
        sagbi
///
-- (status, SAGBIBasis)
doc ///
    Key
        (status, SAGBIBasis)
    Headline
        status of the subalgebra bases computation
    Usage
        result = status SB
    Inputs
        SB:SAGBIBasis
        Monomials=>Boolean
            unused option
        PairsRemaining=>Boolean
            unused option
        TotalPairs=>Boolean
            unused option
    Outputs
        result:String
            a description of the status of a subalgebra basis computation
    Description
        Text
            Returns a string with a human readable description
            of the status of subalgebra bases computation.
        Example
            R = QQ[x,y];
            S = subring {x+y,x*y,x*y^2,x*y^10+y};
            status sagbi(S,Limit=>10)
            status sagbi(S,Limit=>20)
    SeeAlso
        SAGBIBasis
        sagbiBasis
        sagbiStatus
        sagbi
///
-- forceSB
doc ///
    Key
        forceSB
    Headline
        declare the generators to be a complete subalgebra basis
    Usage
        forceSB S
        forceSB SB
    Inputs
        S:Subring
        SB:SAGBIBasis
    Description
        Text
            If @TT "forceSB"@ is called on @ ofClass SAGBIBasis @ $SB$,
            then the function performs
            autosubduction on the subalgebra generators of $SB$. The completion flag is then
            set to complete without checking whether the generators form a subalgebra basis.

            If @TT "forceSB"@ is called on @ ofClass Subring @ $S$,
            then the function performs autosubduction
            on the generators of $S$.    The completion flag is then
            set to complete without checking whether the generators form a subalgebra basis.

            In each case above, the option UseSubringGens can be toggled between
            @TT "true"@ and @TT "false"@ to
            operate on the subring generators in the case where the input is
            @ ofClass SAGBIBasis @,
            and operate on any partial subalgebra basis created for $S$ when the the input is
            @ ofClass Subring @.

            If the generators supplied to @ TT "forceSB" @ do not form a subalgebra basis,
            then the resulting behavior may be unexpected.

        Example
            R = QQ[x,y];
            S = subring{x+y,x*y,x*y^2}
            forceSB S;
            isSAGBI S
            sagbi(S,Recompute=>true)
            isSAGBI S
        Text
            In this example, @TT "forceSB"@ causes @TO "isSAGBI"@
            to return @TT "true"@ even though
            the generators of $S$ do not form a subalgebra basis.
            Recomputing the subalgebra basis exposes that the generators do not
            form a subalgebra basis.

            Optional inputs for specific uses (see
            @TO (forceSB, SAGBIBasis)@ and @TO (forceSB, Subring)@) include the following:
            @TO "AutoSubduce"@,
            @TO2 {"Subduction computation limit","Limit"}@, @TO "PrintLevel"@,
            @TO "RenewOptions"@, @TO "Subduction strategies"@,
            @TO "UseSubringGens"@, and @TO "SubductionMethod"@.
    SeeAlso
        Subring
        SAGBIBasis
        isSAGBI
        (forceSB, SAGBIBasis)
        (forceSB, Subring)
        AutoSubduce
        "Subduction computation limit"
        PrintLevel
        RenewOptions
        "Subduction strategies"
        UseSubringGens
        SubductionMethod
///
-- (forceSB, SAGBIBasis)
doc ///
    Key
        (forceSB, SAGBIBasis)
    Headline
        declare the generators to be a complete subalgebra basis
    Usage
        forceSB SB
    Inputs
        SB:SAGBIBasis
        AutoSubduce=>Boolean
            a flag indicating whether to autosubduce the generators of $SB$.
        Limit=>ZZ
            a degree limit for the binomial S-pairs that are computed internally.
        PrintLevel=>ZZ
            indicating how much additional information should be printed.
            When this is greater than zero,
            information is printed about the progress of the computation.
        RenewOptions=>Boolean
            if @ TT "true" @ then the computation will use the options specified,
            otherwise it will use the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@)
        Strategy=>String
            the update strategy used for updating the SAGBIBasis:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method that combines the other two;
            starting with "DegreeByDegree"
            for low degrees and switching to "Incremental".
        UseSubringGens=>Boolean
            a flag indicating whether to use the generators of the subring.
            If @ TT "false"@, then the method uses the subalgebra generators
            of the partial subalgebra basis.
        SubductionMethod=>String
            the name of the method used for subduction either: "Top" or "Engine".
    Description
        Text
            When @TT "forceSB"@ is called on @ ofClass SAGBIBasis @ $SB$, the function performs
            autosubduction on the subalgebra generators of $SB$. The completion flag is then
            set to complete without checking whether the generators form a subalgebra basis.

            The option UseSubringGens can be toggled between @TT "true"@ and @TT "false"@ to
            operate on the subring generators.

            If the generators supplied to @ TT "forceSB" @ do not form a subalgebra basis,
            then the resulting behavior may be unexpected.
        Example
            R = QQ[x,y];
            S = sagbi({x+y,x*y,x*y^2},Limit=>10)
            forceSB S;
            isSAGBI S
            sagbi(S,Recompute=>true)
            isSAGBI S
        Text
            In this example, @TT "forceSB"@ causes isSAGBI to return @TT "true"@ even though
            the generators of $S$ do not form a subalgebra basis.
            Recomputing the subalgebra basis exposes that the generators do not
            form a subalgebra basis.
    SeeAlso
        Subring
        SAGBIBasis
        isSAGBI
        forceSB
        (forceSB, Subring)
        AutoSubduce
        "Subduction computation limit"
        PrintLevel
        RenewOptions
        "Subduction strategies"
        UseSubringGens
        SubductionMethod
///
-- (forceSB, Subring)
doc ///
    Key
        (forceSB, Subring)
    Headline
        declare the generators to be a complete subalgebra basis
    Usage
        forceSB S
    Inputs
        S:Subring
        AutoSubduce=>Boolean
            a flag indicating whether to autosubduce the generators of $S$.
        Limit=>ZZ
            a degree limit for the binomial S-pairs that are computed internally.
        PrintLevel=>ZZ
            indicating how much additional information should be printed.
            When this is greater than zero,
            information is printed about the progress of the computation.
        RenewOptions=>Boolean
            if @ TT "true" @ then the computation will use the options specified,
            otherwise it will use the previously selected options
            (except for the following, which may always be specified:
            @TO "PrintLevel"@, @TO2 {"Subduction computation limit","Limit"}@,
            @TO "Recompute"@, and @TO "RenewOptions"@)
        Strategy=>String
            the update strategy used for updating the SAGBIBasis:
            "DegreeByDegree", "Incremental", and "Master".
            The strategy "Master" is a hybrid method that combines the other two;
            starting with "DegreeByDegree"
            for low degrees and switching to "Incremental".
        UseSubringGens=>Boolean
            a flag indicating whether to use the generators of the subring.
            If @ TT "false"@ then the method uses the subalgebra generators
            of the partial subalgebra basis.
        SubductionMethod=>String
            the name of the method used for subduction either: "Top" or "Engine".
    Description
        Text
            When @TT "forceSB"@ is called on @ ofClass Subring @ $S$,
            the function performs autosubduction
            on the generators of $S$.    The completion flag is then
            set to complete without checking whether the generators form a subalgebra basis.

            The option UseSubringGens can be toggled between @TT "true"@ and @TT "false"@ to
            operate on any partial subalgebra basis created for $S$.

            If the generators supplied to @ TT "forceSB" @ do not form a subalgebra basis,
            then the resulting behavior may be unexpected.
        Example
            R = QQ[x,y];
            S = subring{x+y,x*y,x*y^2}
            forceSB S;
            isSAGBI S
            sagbi(S,Recompute=>true)
            isSAGBI S
        Text
            In this example, @TT "forceSB"@ causes @TO "isSAGBI"@ to return @TT "true"@
            even though
            the generators of $S$ do not form a subalgebra basis.
            Recomputing the subalgebra basis exposes that the generators do not
            form a subalgebra basis.
    SeeAlso
        Subring
        SAGBIBasis
        isSAGBI
        forceSB
        (forceSB, SAGBIBasis)
        AutoSubduce
        "Subduction computation limit"
        PrintLevel
        RenewOptions
        "Subduction strategies"
        UseSubringGens
        SubductionMethod
///
-- presentationRing
doc ///
    Key
        presentationRing
        (presentationRing, Subring)
    Headline
        returns the presentation ring of a subring
    Usage
        presRing = presentationRing S
    Inputs
        S:Subring
    Outputs
        presRing:PolynomialRing
            the presentation ring of S
    Description
        Text
            Given @ ofClass Subring @ $S$ of @ ofClass QuotientRing @ $Q$,
            the @ TT "presentationRing" @ $P$ is a
            polynomial ring with the same coefficient ring as Q
            and with one variable for each generator of S.
            There is a natural map from $P$
            to $S$ that sends each variable to its corresponding generator.
            Elements of the @ TT "presentationRing" @
            represent polynomial combinations of generators.
            Evaluating a polynomial combination of generators is equal to
            applying this map.
            Therefore, $S$ is naturally isomorphic to the quotient of $P$ by the
            kernel of the this map.

            The @ TT "presentationRing" @ naturally arises when using
            @TO (symbol //, RingElement, Subring)@, which takes
            an element of @ ofClass Subring @ and
            expresses it as a polynomial combination of its generators.

            Subrings include the field @TT "presentationMap"@, which provides
            a map from the @TT "presentationRing"@ to the @TO (ambient,Subring)@ ring.
        Example
            R = ZZ/2[x,y];
            Q = R / ideal(x + y^5);
            S = subring {x+y, x*y, x*y^2};
            f = x^2*y^3 + x^4 + y^4;
            f % S
            g = f // S
            S#"presentationMap" g == f
    SeeAlso
        Subring
        (symbol //, RingElement, Subring)
        groebnerSubductionQuotient
        (ambient, Subring)
        (flattenedRing, Subring)
///
-- #########################################
-- ## Short Explanation of Screws Example ##
-- #########################################
-- Example
doc ///
    Key
        "Example: Translation and rotation sub-actions of the adjoint action of SE(3)"
    Description
        Text
            The following example shows how to use this
            package to calculate the invariants of the translation sub-action of
            the adjoint action of $SE(3)$, as studied by Crook and Donelan.
        Example
            gndR = QQ[(t_1..t_3)|(w_1..w_3)|(v_1..v_3), MonomialOrder => Lex];
            translation := matrix {{w_1}, {w_2}, {w_3}, {t_1*w_2+t_2*w_3+v_1},
                {-t_1*w_1+t_3*w_3+v_2}, {-t_2*w_1-t_3*w_2+v_3}};
            sag := sagbi transpose translation;
            netList first entries gens sag
        Text
            The generators above are the 5 invariants Crook and Donelan
            give in their Equation (9), plus the additional 6th invariant.
            The computation below verifies Theorem 2 of Crook and Donelan,
            describing rotational invariants in the case where $m=3$.
        Example
            R = QQ[x_1..x_9, MonomialOrder => Lex];
            eqns := {x_1^2+x_2^2+x_3^2-1, x_1*x_4+x_2*x_5+x_3*x_6,
                x_1*x_7+x_2*x_8+x_3*x_9, x_1*x_4+x_2*x_5+x_3*x_6,
                x_4^2+x_5^2+x_6^2-1, x_4*x_7+x_5*x_8+x_6*x_9,
                x_1*x_7+x_2*x_8+x_3*x_9, x_4*x_7+x_5*x_8+x_6*x_9,
                x_7^2+x_8^2+x_9^2-1,
                x_1*x_5*x_9-x_1*x_6*x_8-x_2*x_4*x_9+x_2*x_6*x_7+x_3*x_4*x_8-x_3*x_5*x_7-1};
            sag1 = subring sagbi eqns;
            SB = sagbi(sag1, Limit => 100);
            isSAGBI SB
            netList first entries gens SB
    SeeAlso
        sagbi
        (gens, SAGBIBasis)
        (isSAGBI)
    References
        @HREF("https://arxiv.org/abs/2001.05417", {"D. Crook and P. Donelan.    Polynomial ",
            "invariants and SAGBI bases for multi-screws. ", EM "arXiv:2001.05417", ", 2020."})@
///
-- Strategy
doc ///
    Key
        "Subduction strategies"
        [(intersect, Subring, Subring), Strategy]
        [(forceSB,SAGBIBasis),Strategy]
        [(forceSB,Subring),Strategy]
        [(isSAGBI,Matrix),Strategy]
        [(isSAGBI,List),Strategy]
        [(isSAGBI,Subring),Strategy]
        [(isSAGBI,SAGBIBasis),Strategy]
        [sagbi,Strategy]
        [sagbiBasis,Strategy]
        [subalgebraBasis,Strategy]
        [subduction,Strategy]
    Headline
        Update procedure for the Sagbi algorithm
    Description
        Text
            There are three possible values for @TT "Strategy"@:
            @TT "DegreeByDegree"@, @TT "Incremental"@, or @TT "Master"@.

            At the beginning of each loop inside of @TO "sagbi"@,
            the main @TO "SAGBIBasis"@ computation object is updated.
            There are two types of update procedures, and the update procedure that is used
            is determined by the value of @TT "Strategy"@.

            The @TT "DegreeByDegree"@ strategy computes a partial groebner basis
            of the {\it reduction ideal} from scratch.
            The @TT "DegreeByDegree"@ strategy is better when many new generators
            are added during each loop of the @TO "sagbi"@ algorithm.
            The @TT "Incremental"@ strategy computes a full groebner basis of
            the {\it reduction ideal} by using the previous full groebner basis
            together with the data of the new subalgebra generators.
            The @TT "Incremental"@ strategy is better
            when there is little change to the number of subalgebra generators.

            For many examples, it is observed that
            many new generators are added at the beginning of subalgebra basis computations,
            but, towards the end of the computation,
            there are very few new generators that are added at each step.
            By setting Strategy to @TT "Master"@ (default),
            the @TT "DegreeByDegree"@ strategy is used at the beginning,
            and the strategy switches to @TT "Incremental"@
            part-of-the-way through the algorithm.
    SeeAlso
        sagbi
        AutoSubduce
        ReduceNewGenerators
        StorePending
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
        (intersect,Subring,Subring)
        forceSB
        isSAGBI
        sagbiBasis
        subalgebraBasis
        subduction
///
-- Limit
doc ///
    Key
        "Subduction computation limit"
        [subalgebraBasis,Limit]
        [sagbi,Limit]
        [(forceSB,Subring),Limit]
        [(forceSB,SAGBIBasis),Limit]
        [sagbiBasis,Limit]
        [subduction,Limit]
        [(intersect, Subring, Subring), Limit]
    Headline
        Bound for subalgebra basis computation
    Description
        Text
            The value of @TT "Limit"@ is used internally within the @TO "sagbi"@ function
            to stop the computation before a subalgebra basis has been found.
            Since subalgebra bases may be infinite, such a barrier is needed to prevent
            endless loops.

            The @TT "Limit"@ is an upper bound on the degrees of the $S$-pairs considered
            by the algorithm in the @TO "presentationRing"@.
            In this ring, the degrees are assigned according to the degrees
            of the leading terms of the polynomials
            that the variables represent.

            Since the @TO "sagbiDegree"@ does not monotonically increase during the
            computation, there is no guarantee that a computed partial subalgebra basis
            is complete up to @TT "Limit"@, even when the computation terminates at this bound.
        Example
            R = QQ[x,y];
            S = subring{x+y,x*y,x*y^2,x*y^5+y};
            T = sagbiBasis S;
            sagbiDegree T
            T5 = sagbi(S,Limit=>5)
            sagbiDegree T5
            gens T5
            T20 = sagbi(S,Limit=>20)
            sagbiDegree T20
            gens T20
    SeeAlso
        sagbi
        AutoSubduce
        ReduceNewGenerators
        StorePending
        "Subduction strategies"
        SubductionMethod
        AutoSubduceOnPartialCompletion
        PrintLevel
        Recompute
        RenewOptions
        (intersect,Subring,Subring)
        forceSB
        isSAGBI
        sagbiBasis
        subalgebraBasis
        subduction
///
-- flatenedRing
doc ///
    Key
        flattenedRing
        (flattenedRing, Subring)
        (flattenedRing, SAGBIBasis)
    Headline
        The flattened ring of a subring or sagbiBasis
    Usage
        result = flattened S
        result = flattened SB
    Inputs
        S:Subring
        SB:SAGBIBasis
    Outputs
        result:Ring
            the flattened ring of the ambient ring
    Description
        Text
            Returns the flattened ring of the ambient ring.
            This may be a @TO "PolynomialRing"@ or @TO "QuotientRing"@.
        Example
            R = (QQ[x])[y]/ideal(x^2-y^3);
            S = subring {x+y,x*y};
            ambient S
            flattenedRing S
    SeeAlso
        Subring
        subring
        SAGBIBasis
        sagbiBasis
        (ambient, SAGBIBasis)
///
end --
