/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QtTest>
#include <QObject>
#include <QDir>
#include <QStandardPaths>
#include <QNetworkAccessManager>

#include "cachingprovider.h"
#include "image.h"

#define BASEURL "http://localhost:8888"

class CachingProviderTests : public QObject
{
    Q_OBJECT

public:
    CachingProviderTests(QObject* parent = 0)
    {
    }

private:
    QDir cache;

private Q_SLOTS:
    void initTestCase()
    {
        cache = QDir(QStandardPaths::writableLocation(QStandardPaths::DataLocation));
    }

    void init()
    {
        QNetworkAccessManager::log().clear();
        cache.removeRecursively();
        cache.setFilter(QDir::NoDotAndDotDot | QDir::Files);
    }

    void shouldDownloadAndCache()
    {
        QString path("/image.png");

        CachingProvider provider;
        QSize size;
        QImage image = provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);

        /* Verify that the image we get back is good and that the file was cached with the
         * correct hash for the URL we requested. */
        QVERIFY(!image.isNull());
        QCOMPARE(size, image_png_size);
        QCOMPARE(cache.entryList().count(), 1);
        QCOMPARE(cache.entryList().first(), QString("2c9fc02d059d33853dff5622ee332ab4.png"));
        QCOMPARE(QNetworkAccessManager::log().count(), 1);
        QCOMPARE(QNetworkAccessManager::log().first(), path);
    }

    void shouldRetrieveFromCache()
    {
        QString path("/image.png");

        CachingProvider provider;
        QSize size;

        /* Verify that the second time we request the same image we get something valid back */
        provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);
        QImage image = provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);
        QVERIFY(!image.isNull());
        QCOMPARE(size, image_png_size);

        /* Verify that the server only received the first request (as the second one came
         * from the cache) and not two of them */
        QCOMPARE(QNetworkAccessManager::log().count(), 1);
        QCOMPARE(QNetworkAccessManager::log().first(), path);
    }

    void shouldFollowRedirectsAndCache()
    {
        QString path("/redirect/1.png");

        CachingProvider provider;
        QSize size;

        // Verify that we get a valid image back
        QImage image = provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);
        QVERIFY(!image.isNull());
        QCOMPARE(size, image_png_size);
        QCOMPARE(QNetworkAccessManager::log().count(), 4);
        QCOMPARE(QNetworkAccessManager::log().first(), path);
        QCOMPARE(QNetworkAccessManager::log().last(), QString("/image.png"));
        QCOMPARE(cache.entryList().count(), 1);

        // Verify that the second time we request the same image it is fullfilled directly from
        // the cache and thus the server log doesn't change at all
        image = provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);
        QVERIFY(!image.isNull());
        QCOMPARE(size, image_png_size);
        QCOMPARE(QNetworkAccessManager::log().count(), 4);
        QCOMPARE(QNetworkAccessManager::log().first(), path);
        QCOMPARE(QNetworkAccessManager::log().last(), QString("/image.png"));
    }

    void shouldFailOnRedirectLoop()
    {
        QString path("/redirect_loop.png");

        CachingProvider provider;
        QSize size;

        QImage image = provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);
        QVERIFY(image.isNull());
        QVERIFY(size.isNull());
        QCOMPARE(cache.entryList().count(), 0);
    }

    void shouldDownloadOnCorrupt()
    {
        QString path("/image.png");

        CachingProvider provider;
        QSize size;
        QImage image = provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);

        /* Corrupt the image in the cache so that it will fail to load at next request */
        QFile cachedFile;
        cachedFile.setFileName(cache.absolutePath() + "/" + cache.entryList().first());
        cachedFile.open(QFile::WriteOnly);
        cachedFile.write(QByteArray("corrupt"));
        cachedFile.close();
        QNetworkAccessManager::log().clear();

        /* Verify that the corrupt image will be downloaded and re-cached */
        image = provider.requestImage(QString(BASEURL "%1").arg(path), &size, size);
        QVERIFY(!image.isNull());
        QCOMPARE(size, image_png_size);
        QCOMPARE(QNetworkAccessManager::log().count(), 1);
        QCOMPARE(QNetworkAccessManager::log().first(), path);
        QCOMPARE(cache.entryList().first(), QString("2c9fc02d059d33853dff5622ee332ab4.png"));
    }
};

QTEST_MAIN(CachingProviderTests)
#include "tst_CachingProviderTests.moc"
