/*
 *  Copyright (c) 2007,2008, 2009 INRIA, UDcast
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Mohamed Amine Ismail <amine.ismail@sophia.inria.fr>
 *                              <amine.ismail@udcast.com>
 */

//
// Default network topology includes a base station (BS) and 2
// subscriber station (SS).

//      +-----+
//      | SS0 |
//      +-----+
//     10.1.1.1
//      -------
//        ((*))
//
//                  10.1.1.7
//               +------------+
//               |Base Station| ==((*))
//               +------------+
//
//        ((*))
//       -------
//      10.1.1.2
//       +-----+
//       | SS1 |
//       +-----+

#include "ns3/applications-module.h"
#include "ns3/core-module.h"
#include "ns3/global-route-manager.h"
#include "ns3/internet-module.h"
#include "ns3/ipcs-classifier-record.h"
#include "ns3/mobility-module.h"
#include "ns3/network-module.h"
#include "ns3/service-flow.h"
#include "ns3/wimax-module.h"

#include <iostream>

using namespace ns3;

NS_LOG_COMPONENT_DEFINE("WimaxSimpleExample");

int
main(int argc, char* argv[])
{
    bool verbose = false;

    int duration = 7;
    int schedType = 0;
    WimaxHelper::SchedulerType scheduler = WimaxHelper::SCHED_TYPE_SIMPLE;

    CommandLine cmd(__FILE__);
    cmd.AddValue("scheduler", "type of scheduler to use with the network devices", schedType);
    cmd.AddValue("duration", "duration of the simulation in seconds", duration);
    cmd.AddValue("verbose", "turn on all WimaxNetDevice log components", verbose);
    cmd.Parse(argc, argv);
    LogComponentEnable("UdpClient", LOG_LEVEL_INFO);
    LogComponentEnable("UdpServer", LOG_LEVEL_INFO);
    switch (schedType)
    {
    case 0:
        scheduler = WimaxHelper::SCHED_TYPE_SIMPLE;
        break;
    case 1:
        scheduler = WimaxHelper::SCHED_TYPE_MBQOS;
        break;
    case 2:
        scheduler = WimaxHelper::SCHED_TYPE_RTPS;
        break;
    default:
        scheduler = WimaxHelper::SCHED_TYPE_SIMPLE;
    }

    NodeContainer ssNodes;
    NodeContainer bsNodes;

    ssNodes.Create(2);
    bsNodes.Create(1);

    WimaxHelper wimax;

    NetDeviceContainer ssDevs;
    NetDeviceContainer bsDevs;

    ssDevs = wimax.Install(ssNodes,
                           WimaxHelper::DEVICE_TYPE_SUBSCRIBER_STATION,
                           WimaxHelper::SIMPLE_PHY_TYPE_OFDM,
                           scheduler);
    bsDevs = wimax.Install(bsNodes,
                           WimaxHelper::DEVICE_TYPE_BASE_STATION,
                           WimaxHelper::SIMPLE_PHY_TYPE_OFDM,
                           scheduler);

    wimax.EnableAscii("bs-devices", bsDevs);
    wimax.EnableAscii("ss-devices", ssDevs);

    Ptr<SubscriberStationNetDevice> ss[2];

    for (int i = 0; i < 2; i++)
    {
        ss[i] = ssDevs.Get(i)->GetObject<SubscriberStationNetDevice>();
        ss[i]->SetModulationType(WimaxPhy::MODULATION_TYPE_QAM16_12);
    }

    Ptr<BaseStationNetDevice> bs;

    bs = bsDevs.Get(0)->GetObject<BaseStationNetDevice>();

    InternetStackHelper stack;
    stack.Install(bsNodes);
    stack.Install(ssNodes);

    Ipv4AddressHelper address;
    address.SetBase("10.1.1.0", "255.255.255.0");

    Ipv4InterfaceContainer SSinterfaces = address.Assign(ssDevs);
    Ipv4InterfaceContainer BSinterface = address.Assign(bsDevs);

    if (verbose)
    {
        WimaxHelper::EnableLogComponents(); // Turn on all wimax logging
    }
    /*------------------------------*/
    UdpServerHelper udpServer;
    ApplicationContainer serverApps;
    UdpClientHelper udpClient;
    ApplicationContainer clientApps;

    udpServer = UdpServerHelper(100);

    serverApps = udpServer.Install(ssNodes.Get(0));
    serverApps.Start(Seconds(6));
    serverApps.Stop(Seconds(duration));

    udpClient = UdpClientHelper(SSinterfaces.GetAddress(0), 100);
    udpClient.SetAttribute("MaxPackets", UintegerValue(1200));
    udpClient.SetAttribute("Interval", TimeValue(Seconds(0.5)));
    udpClient.SetAttribute("PacketSize", UintegerValue(1024));

    clientApps = udpClient.Install(ssNodes.Get(1));
    clientApps.Start(Seconds(6));
    clientApps.Stop(Seconds(duration));

    Simulator::Stop(Seconds(duration + 0.1));

    wimax.EnablePcap("wimax-simple-ss0", ssNodes.Get(0)->GetId(), ss[0]->GetIfIndex());
    wimax.EnablePcap("wimax-simple-ss1", ssNodes.Get(1)->GetId(), ss[1]->GetIfIndex());
    wimax.EnablePcap("wimax-simple-bs0", bsNodes.Get(0)->GetId(), bs->GetIfIndex());

    IpcsClassifierRecord DlClassifierUgs(Ipv4Address("0.0.0.0"),
                                         Ipv4Mask("0.0.0.0"),
                                         SSinterfaces.GetAddress(0),
                                         Ipv4Mask("255.255.255.255"),
                                         0,
                                         65000,
                                         100,
                                         100,
                                         17,
                                         1);
    ServiceFlow DlServiceFlowUgs = wimax.CreateServiceFlow(ServiceFlow::SF_DIRECTION_DOWN,
                                                           ServiceFlow::SF_TYPE_RTPS,
                                                           DlClassifierUgs);

    IpcsClassifierRecord UlClassifierUgs(SSinterfaces.GetAddress(1),
                                         Ipv4Mask("255.255.255.255"),
                                         Ipv4Address("0.0.0.0"),
                                         Ipv4Mask("0.0.0.0"),
                                         0,
                                         65000,
                                         100,
                                         100,
                                         17,
                                         1);
    ServiceFlow UlServiceFlowUgs = wimax.CreateServiceFlow(ServiceFlow::SF_DIRECTION_UP,
                                                           ServiceFlow::SF_TYPE_RTPS,
                                                           UlClassifierUgs);
    ss[0]->AddServiceFlow(DlServiceFlowUgs);
    ss[1]->AddServiceFlow(UlServiceFlowUgs);

    NS_LOG_INFO("Starting simulation.....");
    Simulator::Run();

    ss[0] = nullptr;
    ss[1] = nullptr;
    bs = nullptr;

    Simulator::Destroy();
    NS_LOG_INFO("Done.");

    return 0;
}
