///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/SpinnerWidget.h>
#include <core/viewport/ViewportManager.h>

namespace Core {

/******************************************************************************
* Constructs the spinner.
******************************************************************************/
SpinnerWidget::SpinnerWidget(QWidget* parent, QLineEdit* textBox)
	: QWidget(parent), 
	_textBox(NULL), _value(0), _minValue(-FLOATTYPE_MAX), _maxValue(FLOATTYPE_MAX),  
	upperBtnPressed(false), lowerBtnPressed(false), _unit(NULL)
{
	setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum, QSizePolicy::SpinBox));
	setUnit(UNITS_MANAGER.floatIdentity());
	setTextBox(textBox);
}

/******************************************************************************
* Paint event handler.
******************************************************************************/
void SpinnerWidget::paintEvent(QPaintEvent* event)
{
	QStylePainter p(this);
    QStyleOptionSpinBox sboption;
    
    sboption.initFrom(this);
    sboption.state |= upperBtnPressed ? QStyle::State_Sunken : QStyle::State_Raised;
    sboption.rect.setHeight(sboption.rect.height() / 2);
    p.drawPrimitive(QStyle::PE_PanelButtonTool, sboption);
    p.drawPrimitive(QStyle::PE_IndicatorSpinUp, sboption);

    sboption.initFrom(this);
    sboption.state |= lowerBtnPressed ? QStyle::State_Sunken : QStyle::State_Raised;
    sboption.rect.setTop(sboption.rect.top() + sboption.rect.height() / 2);
    p.drawPrimitive(QStyle::PE_PanelButtonTool, sboption);
    p.drawPrimitive(QStyle::PE_IndicatorSpinDown, sboption);
}

/******************************************************************************
* Returns the natural size of the spinner widget.
******************************************************************************/
QSize SpinnerWidget::sizeHint() const
{
	if(textBox() == NULL) return QSize(16, 30);
	else return QSize(16, textBox()->sizeHint().height());	
}

/******************************************************************************
* Connects this spinner with the given textbox control.
******************************************************************************/
void SpinnerWidget::setTextBox(QLineEdit* box)
{
	if(box == textBox()) return;
	if(_textBox.isNull() == false) {
		disconnect(_textBox, SIGNAL(editingFinished()), this, SLOT(onTextChanged()));
	}
	_textBox = box;
	if(_textBox.isNull() == false) {
		connect(_textBox, SIGNAL(editingFinished()), this, SLOT(onTextChanged()));
		_textBox->setEnabled(isEnabled());
		updateTextBox();
	}
}

/******************************************************************************
* Will be called when the user has entered a new text into the text box.
* The text will be parsed and taken as the new value of the spinner.
******************************************************************************/
void SpinnerWidget::onTextChanged()
{
	OVITO_ASSERT(textBox());
	CHECK_OBJECT_POINTER(unit());
    
	FloatType newValue;
	try {
		if(textBox()->text() == _originalText) return;
		newValue = unit()->parseString(textBox()->text());
	}
	catch(const Exception&) {
		// Ignore invalid value and restore old text content.
		updateTextBox();
		return;
	}	
	setFloatValue(unit()->userToNative(newValue), true);
}

/******************************************************************************
* Updates the text of the connected text box after the spinner's value has changed.
******************************************************************************/
void SpinnerWidget::updateTextBox()
{
	if(textBox()) {
		CHECK_OBJECT_POINTER(unit());
		_originalText = unit()->formatValue(unit()->nativeToUser(floatValue()));
		textBox()->setText(_originalText);
	}
}

/******************************************************************************
* Sets the current value of the spinner.
******************************************************************************/
void SpinnerWidget::setFloatValue(FloatType newVal, bool emitChangeSignal)
{
	// Clamp value.
	if(newVal == _value) return;
	newVal = max(minValue(), newVal);
	newVal = min(maxValue(), newVal);
	if(_value != newVal) {
		_value = newVal;
		if(emitChangeSignal)
			spinnerValueChanged();
	}
	updateTextBox();
}

/******************************************************************************
* Sets the current integer value of the spinner.
******************************************************************************/
void SpinnerWidget::setIntValue(int newValInt, bool emitChangeSignal)
{
	FloatType newVal = (FloatType)newValInt;

	if(newVal == _value) return;
	// Clamp value.
	newVal = max(ceil(minValue()), newVal);
	newVal = min(floor(maxValue()), newVal);
	if(_value != newVal) {
		_value = newVal;
		if(emitChangeSignal)
			spinnerValueChanged();
	}
	updateTextBox();
}

/******************************************************************************
* Sets the minimum allowed value of the spinner.
* If the current value of the spinner is less than the new minimum value,
* it will be set to the new minimum value.
******************************************************************************/
void SpinnerWidget::setMinValue(FloatType minValue)
{
	OVITO_ASSERT_MSG(minValue <= maxValue(), "SpinnerWidget::setMinValue", "Minimal spinner value must not be greater than maximum spinner value.");
	_minValue = minValue;
}

/******************************************************************************
* Sets the maximum allowed value of the spinner.
* If the current value of the spinner is greater than the new maximum value,
* it will be set to the new maximum value.
******************************************************************************/
void SpinnerWidget::setMaxValue(FloatType maxValue)
{
	OVITO_ASSERT_MSG(maxValue >= minValue(), "SpinnerWidget::setMaxValue", "Maximal spinner value must not be less than minimum spinner value.");
	_maxValue = maxValue;
}

/******************************************************************************
* Sets the units of this spinner's value.
******************************************************************************/
void SpinnerWidget::setUnit(ParameterUnit* unit) 
{ 
	CHECK_OBJECT_POINTER(unit); 
	if(unit == _unit) return;

	if(_unit != NULL)
		disconnect(_unit, SIGNAL(formatChanged()), this, SLOT(updateTextBox()));
	
	_unit = unit; 
	
	if(_unit)
		connect(_unit, SIGNAL(formatChanged()), this, SLOT(updateTextBox()));

	updateTextBox(); 
}

/******************************************************************************
* Handles the change events for the spinner.
******************************************************************************/
void SpinnerWidget::changeEvent(QEvent* event)
{
	QWidget::changeEvent(event);
	if(event->type() == QEvent::EnabledChange) {
		if(textBox()) textBox()->setEnabled(isEnabled());
	}
}

/******************************************************************************
* Handles the mouse down event.
******************************************************************************/
void SpinnerWidget::mousePressEvent(QMouseEvent* event)
{
	CHECK_OBJECT_POINTER(unit());
	
	if(event->button() == Qt::LeftButton) {
		// Backup current value.
		_oldValue = floatValue();

		OVITO_ASSERT(lowerBtnPressed == false && upperBtnPressed == false);

		if(event->y() <= height()/2)
			upperBtnPressed = true;
		else
			lowerBtnPressed = true;

		_currentStepSize = unit()->stepSize(floatValue(), upperBtnPressed);
		if(textBox()) textBox()->setFocus(Qt::OtherFocusReason);
		
		repaint();
	}
	else if(event->button() == Qt::RightButton) {
		
		// restore old value
		setFloatValue(_oldValue, true);

		if(upperBtnPressed == lowerBtnPressed) {
			spinnerDragAbort();
		}

		upperBtnPressed = false;
		lowerBtnPressed = false;

		update();
	}	
}

/******************************************************************************
* Handles the mouse up event.
******************************************************************************/
void SpinnerWidget::mouseReleaseEvent(QMouseEvent* event)
{
	if(upperBtnPressed || lowerBtnPressed) {
		if(upperBtnPressed == lowerBtnPressed) {
			spinnerDragStop();
		}
		else if(upperBtnPressed)
			setFloatValue(floatValue() + unit()->stepSize(floatValue(), true), true);
		else
			setFloatValue(floatValue() - unit()->stepSize(floatValue(), false), true);

		upperBtnPressed = false;
		lowerBtnPressed = false;

		// Repaint spinner.
		update();
	}
}

/******************************************************************************
* Handles the mouse move event.
******************************************************************************/
void SpinnerWidget::mouseMoveEvent(QMouseEvent* event)
{
	if(upperBtnPressed || lowerBtnPressed) {
		if(upperBtnPressed && !lowerBtnPressed) {
			if(event->y() > height()/2 || event->y() < 0) {
				lowerBtnPressed = true;
				lastMouseY = startMouseY = mapToGlobal(event->pos()).y();
				update();
				spinnerDragStart();
			}
		}
		else if(!upperBtnPressed && lowerBtnPressed) {
			if(event->y() <= height()/2 || event->y() > height()) {
				upperBtnPressed = true;
				lastMouseY = startMouseY = mapToGlobal(event->pos()).y();
				update();
				spinnerDragStart();
			}
		}
		else { 
			QPoint cursorPos = QCursor::pos();
			int screenY = cursorPos.y();
			if(screenY != lastMouseY) {
				int screenHeight = QApplication::desktop()->screenGeometry().height();
				if(screenY <= 5 && lastMouseY == screenHeight-1) return;
				if(screenY >= screenHeight - 5 && lastMouseY == 0) return;
				
				FloatType newVal = _oldValue + _currentStepSize * (FloatType)(startMouseY - screenY) * 0.1;
	
				if(screenY < lastMouseY && screenY <= 5) {
					lastMouseY = screenHeight-1;
					startMouseY += lastMouseY - screenY;
					QCursor::setPos(cursorPos.x(), lastMouseY);					
				}
				else if(screenY > lastMouseY && screenY >= screenHeight - 5) {
					lastMouseY = 0;
					startMouseY += lastMouseY - screenY;
					QCursor::setPos(cursorPos.x(), lastMouseY);
				}
				else lastMouseY = screenY;

				if(newVal != floatValue()) {
					setFloatValue(newVal, true);
	            
					// Repaint viewports on interactive adjusting.
					VIEWPORT_MANAGER.processViewportUpdates();
				}
			}
		}
	}
}

};
