///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/geometry/MeshObject.h>
#include <core/viewport/Viewport.h>

namespace Core {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(MeshObject, SceneObject)

/******************************************************************************
* Makes the object render itself into the viewport.
******************************************************************************/
void MeshObject::renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp)
{
	vp->renderNodeMesh(mesh(), contextNode);
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void MeshObject::saveToStream(ObjectSaveStream& stream)
{
	SceneObject::saveToStream(stream);
	stream.beginChunk(0x00001);
	stream << _meshValidity;
	_mesh.saveToStream(stream);
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void MeshObject::loadFromStream(ObjectLoadStream& stream)
{
	SceneObject::loadFromStream(stream);
	stream.expectChunk(0x00001);
	stream >> _meshValidity;
	_mesh.loadFromStream(stream);
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr MeshObject::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	MeshObject::SmartPtr clone = static_object_cast<MeshObject>(SceneObject::clone(deepCopy, cloneHelper));

	clone->_meshValidity = this->_meshValidity;
	clone->_mesh = this->_mesh;

	return clone;
}

/******************************************************************************
* Renders the object in preview rendering mode using OpenGL.
******************************************************************************/
bool MeshObject::renderPreview(TimeTicks time, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
	if(mesh().vertexCount() == 0 || mesh().faceCount() == 0)
		return true;

	// Prepare mesh for shaded rendering.
	if((mesh().cacheState & TriMesh::RENDER_VERTICES_CACHED) == 0) {
		mesh().buildRenderVertices();
	}
	OVITO_ASSERT(mesh().renderVertices);
	OVITO_ASSERT(!mesh().faceGroups.empty());

	// Make sure we comply with the OpenGL GL_T2F_C4F_N3F_V3F memory layout.
	OVITO_STATIC_ASSERT(sizeof(TriMesh::RenderVertex) == sizeof(float)*(2+4+3+3));

	glPushAttrib(GL_CURRENT_BIT);

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_NORMAL_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	CHECK_OPENGL(glInterleavedArrays(GL_T2F_C4F_N3F_V3F, 0, mesh().renderVertices));
	for(QVector<TriMesh::MaterialGroup>::const_iterator group = mesh().faceGroups.begin(); group != mesh().faceGroups.end(); ++group) {
		if(group->empty()) continue;

		// Setup material
		int materialIndex = group - mesh().faceGroups.begin();
		Color ambient = Color(0.1f, 0.1f, 0.1f);
		Color diffuse = contextNode->displayColor();
		Color specular = Color(1.0, 1.0, 1.0);
		FloatType shininess = 0.1f;
		FloatType shinStrength = 0.1f;
		FloatType opacity = 1.0;
		FloatType selfIllum = 0.0;
		float c[4];
		c[0] = (float)ambient.r; c[1] = (float)ambient.g; c[2] = (float)ambient.b; c[3] = (float)opacity;
		glMaterialfv(GL_FRONT, GL_AMBIENT, c);
		c[0] = (float)diffuse.r; c[1] = (float)diffuse.g; c[2] = (float)diffuse.b; c[3] = (float)opacity;
		glMaterialfv(GL_FRONT, GL_DIFFUSE, c);
		c[0] = (float)(specular.r*shinStrength); c[1] = (float)(specular.g*shinStrength);
		c[2] = (float)(specular.b*shinStrength); c[3] = (float)opacity;
		glMaterialfv(GL_FRONT, GL_SPECULAR, c);
		c[0] = (float)(diffuse.r * selfIllum); c[1] = (float)(diffuse.g * selfIllum);
		c[2] = (float)(diffuse.b * selfIllum); c[3] = 1.0f;
		glMaterialfv(GL_FRONT, GL_EMISSION, c);
		glMaterialf(GL_FRONT, GL_SHININESS, (float)pow(2.0, (double)shininess*10.0));
		glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, 0);

		CHECK_OPENGL(glDrawElements(GL_TRIANGLES, (GLsizei)group->size(), GL_UNSIGNED_INT, group->constData()));
	}
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	glDisable(GL_COLOR_MATERIAL);
	glPopAttrib();

	return true;
}

};
