from __future__ import division, print_function, unicode_literals, with_statement

import sys
import time
import threading
import Queue
import wx

class ThreadStopException(BaseException):
    pass

class TraceThread(threading.Thread):
    def __init__(self, cpu, world, program):
        threading.Thread.__init__(self)
        self._cpu = cpu
        self._world = world
        self._globals = self._world.getGlobals(self)
        self._program = program
        self._stopped = False
        self._traceProxy = self.threadAwareProxyFunction(self._cpu.trace)
    
    def _done(self):
        if not self._stopped:
            self._cpu.done()
    
    def _failed(self, e):
        if not self._stopped:
            self._cpu.failed(e)
    
    def run(self):
        try:
            sys.settrace(self._tracefunc)
            try:
                exec self._program in self._globals
                wx.CallAfter(self._done)
            except BaseException, e:
                wx.CallAfter(self._failed, e)
        finally:
            # not really necessary - thread's done anyway
            sys.settrace(None)

    def stop(self):
        self._stopped = True
        
    def threadAwareProxyFunction(self, f):
        def guardF(rcb, *a, **kw):
            if self._stopped:
                rcb((None, ThreadStopException()))
            else:
                f(self, rcb, *a, **kw)
        def callF(*a, **kw):
            q = Queue.Queue()
            wx.CallAfter(guardF, q.put, *a, **kw)
            res, exc = q.get()
            if exc is not None:
                raise exc
            return res
        return callF

    def proxyFunction(self, f):
        def callBlocking(t, rcb, *a, **kw):
            try:
                rcb((f(*a, **kw), None))
            except Exception, e:
                rcb((None, e))
        return self.threadAwareProxyFunction(callBlocking)

    def _tracefunc(self, frame, event, arg):
        # FIXME: shame to stop only on the lines in string if stopped
        if "<string>" in frame.f_code.co_filename:
            if event == "line":
                self._traceProxy(frame)
        return self._tracefunc

STOP = 1
PAUSE = 2
RUN = 3

class CPU(object):
    def __init__(self, ui):
        self._ui = ui
        self._lineTime = 1000
        self._clear()
    
    def _clear(self):
        self._state = STOP
        self._timer = None
        self._thread = None
        self._rcb = None
        self._frame = None
    
    def _start(self):
        world = self._ui.world
        self._ui.starting()
        self._thread = TraceThread(self, world, self._ui.program)
        self._lastTime = time.time()
        self._thread.start()

    def _release(self):
        r = self._rcb
        if r is not None:
            self._rcb = None
            self._timer = None
            self._frame = None
            self._lastTime = time.time()
            r((None, None))
    
    def _waitTime(self):
        return max(1, int(
            self._lineTime - 1000*(time.time() - self._lastTime)))
    
    def _stopTimer(self):
        if self._timer is not None:
            self._timer.Stop()
            self._timer = None
    
    # ----- Thread methods ------

    def trace(self, t, rcb, frame):
        if self._state == STOP:
            return # FIXME: assert out
        self._frame = frame
        self._rcb = rcb
        if self._state == RUN:
            self._timer = wx.CallLater(self._waitTime(), self._release)
        self._ui.trace(frame)
    
    def done(self):
        self._clear()
        self._ui.done()

    def failed(self, e):
        self._clear()
        self._ui.failed(e)

    # ----- UI methods -----

    @property
    def state(self):
        return self._state

    def run(self):
        if self._state == STOP:
            self._state = RUN
            self._ui.running()
            self._start()
        elif self._state == PAUSE:
            self._state = RUN
            self._ui.running()
            self._release()
    
    def pause(self):
        if self._state == STOP:
            self._state = PAUSE
            self._ui.pausing()
            self._start()
        elif self._state == RUN:
            self._state = PAUSE
            self._ui.pausing()
            self._stopTimer()
    
    def stop(self):
        if self._state == STOP:
            return
        self._stopTimer()
        if self._thread:
            self._thread.stop()
        self._clear()
        self._ui.stopped()
        
    def step(self):
        self.pause()
        self._release()

    def setLineTime(self, t):
        self._lineTime = t
        if self._timer is not None:
            self._timer.Restart(self._waitTime())
    
__all__ = [STOP, PAUSE, RUN, CPU]

