// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! A layer on top of [Certificate] that filters out any signatures that we don't consider valid.
//!
//! FIXME: Which parts of validity checking can/should be moved to [SigStack]?

use std::ops::Add;

use chrono::{DateTime, Duration, Utc};
use pgp::crypto::aead::AeadAlgorithm;
use pgp::crypto::hash::HashAlgorithm;
use pgp::crypto::sym::SymmetricKeyAlgorithm;
use pgp::packet::{SubpacketData, SubpacketType};
use pgp::types::{PublicKeyTrait, SignedUser, SignedUserAttribute, Tag};
use pgp::{Signature, SignedKeyDetails, SignedPublicKey, SignedPublicSubKey};

use crate::key::component::{ComponentKeyPub, SignedComponentKey, SignedComponentKeyPub};
use crate::key::SignatureVerifier;
use crate::key::{primary_user_id, Certificate};
use crate::sig::stack::SigStack;
use crate::{key, policy, sig, Error};

/// A certificate that only contains Signatures that have been verified to be cryptographically
/// correct.
/// See <https://openpgp.dev/book/verification.html#when-are-signatures-valid>
///
/// This is a precondition for signatures being considered fully valid, but by itself insufficient.
///
/// Note: This function currently removes all third-party signatures
/// (because they can't be cryptographically checked in the context of looking at an individual
/// certificate).
#[derive(Debug, Clone)]
pub struct CheckedCertificate {
    cspk: SignedPublicKey,
}

impl CheckedCertificate {
    pub fn primary_key(&self) -> SignedComponentKeyPub {
        let mut sigs: Vec<_> = self.cspk.details.revocation_signatures.clone();
        sigs.append(&mut self.cspk.details.direct_signatures.clone());

        SignedComponentKeyPub::Primary((self.cspk.clone(), sigs))
    }

    pub fn subkeys(&self) -> impl Iterator<Item = SignedComponentKeyPub> + '_ {
        self.cspk
            .public_subkeys
            .iter()
            .map(|spsk| SignedComponentKeyPub::Subkey((spsk.clone(), self.dks().cloned())))
    }

    pub fn primary_user_id(&self) -> Option<&SignedUser> {
        key::primary_user_id(&self.cspk.details)
    }

    /// list of all user ids
    pub fn user_ids(&self) -> &[SignedUser] {
        &self.cspk.details.users
    }

    /// list of all user attributes
    pub fn user_attributes(&self) -> &[SignedUserAttribute] {
        &self.cspk.details.user_attributes
    }

    pub fn valid_encryption_capable_component_keys(&self) -> Vec<ComponentKeyPub> {
        // FIXME: First drop all bad signatures (bad algorithms)

        let now: DateTime<Utc> = chrono::offset::Utc::now();

        match self.primary_valid_at(&now) {
            // If the primary key is invalid now, there are no valid component keys
            Err(_) | Ok(false) => return vec![],

            Ok(true) => {}
        }

        // Filter based on component key validity
        self.encryption_capable_component_keys(&now)
            .filter(|sckp| sckp.is_component_subkey_valid_at(&now))
            .filter(|sckp| sckp.valid_encryption_algo())
            .filter(|sckp| policy::acceptable_pk_algorithm(sckp.public_params(), &now))
            .map(Into::into)
            .collect()
    }

    /// Return a Verifier object for each appropriate subkey.
    ///
    /// This includes a check that the binding signature must have an embedded back signature
    /// (we're in a CorrectCertificate, so we can assume that if a back signature exists at all, it is acceptable).
    pub fn valid_signing_capable_component_keys_at(
        &self,
        reference: &DateTime<Utc>,
    ) -> Vec<SignatureVerifier> {
        // FIXME: First drop all bad signatures (bad algorithms, missing backsig)

        match self.primary_valid_at(reference) {
            // If the primary key is invalid, there are no valid component keys
            Err(_) | Ok(false) => return vec![],

            Ok(true) => {}
        }

        // Filter based on component key validity
        self.signing_capable_component_keys(reference)
            .filter(|sckp| sckp.is_component_subkey_valid_at(reference))
            .filter(|sckp| sckp.has_valid_backsig_at(reference))
            .filter(|sckp| policy::acceptable_pk_algorithm(sckp.public_params(), reference))
            .map(ComponentKeyPub::from)
            .map(Into::into)
            .collect()
    }

    pub fn valid_authentication_capable_component_keys(
        &self,
        reference: &DateTime<Utc>,
    ) -> Vec<ComponentKeyPub> {
        // FIXME: First drop all bad signatures (bad algorithms)

        match self.primary_valid_at(reference) {
            // If the primary key is invalid, there are no valid component keys
            Err(_) | Ok(false) => return vec![],

            Ok(true) => {}
        }

        // Filter based on component key validity
        self.authentication_capable_component_keys(reference)
            .filter(|sckp| sckp.is_component_subkey_valid_at(reference))
            .filter(|sckp| sckp.valid_encryption_algo())
            .filter(|sckp| policy::acceptable_pk_algorithm(sckp.public_params(), reference))
            .map(Into::into)
            .collect()
    }

    fn component_keys(&self) -> impl Iterator<Item = SignedComponentKeyPub> + '_ {
        // Collect all signatures that are bound directly to the primary

        let pri = vec![self.primary_key()];

        pri.into_iter().chain(self.subkeys())
    }

    /// Get list of all encryption capable component keys
    fn encryption_capable_component_keys<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> impl Iterator<Item = SignedComponentKeyPub> + '_ {
        self.component_keys().filter(move |ckp| {
            SignedComponentKey::Pub(ckp.clone()).is_encryption_capable(reference)
        })
    }

    /// Get list of all signing capable component keys.
    fn signing_capable_component_keys<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> impl Iterator<Item = SignedComponentKeyPub> + '_ {
        self.component_keys()
            .filter(move |ckp| SignedComponentKey::Pub(ckp.clone()).is_signing_capable(reference))
    }

    /// Get list of all authentication capable component keys
    fn authentication_capable_component_keys<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> impl Iterator<Item = SignedComponentKeyPub> + '_ {
        self.component_keys().filter(move |ckp| {
            SignedComponentKey::Pub(ckp.clone()).is_authentication_capable(reference)
        })
    }

    /// Get latest binding signature for the primary user id.
    fn primary_user_id_binding<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> Option<&'a Signature> {
        key::primary_user_id_binding_at(&self.cspk.details, reference)
    }

    fn dks(&self) -> Option<&Signature> {
        SigStack::from_iter(self.cspk.details.direct_signatures.iter()).active()
    }

    pub fn primary_creation_time(&self) -> &DateTime<Utc> {
        self.cspk.primary_key.created_at()
    }

    // Return expiration time as a duration, in seconds
    fn primary_expiration_time(&self, reference: &DateTime<Utc>) -> Result<Option<u32>, Error> {
        // Pick the more defensive expiration value, if both are set.
        // - None: no expiration
        // - 0: no expiration
        // - any other value: expiration in "seconds after creation"
        fn shorter(s1: Option<u32>, s2: Option<u32>) -> Option<u32> {
            match (s1, s2) {
                (None, None) => None,
                (Some(s), None) | (None, Some(s)) => Some(s),

                // "0" means indefinite -> return the other value, which might be shorter
                (Some(0), Some(s)) | (Some(s), Some(0)) => Some(s),

                (Some(s1), Some(s2)) => {
                    // Both are not zero -> return the smaller number
                    Some(u32::min(s1, s2))
                }
            }
        }

        let pri_binding = self.primary_user_id_binding(reference);
        let dks = self.dks();

        let exp = match (dks, pri_binding) {
            (None, None) => {
                // found neither direct key binding, nor primary user id binding signature
                return Err(Error::NoPrimaryBinding);
            }
            (Some(s), None) | (None, Some(s)) => {
                crate::util::duration_to_seconds(s.key_expiration_time())
            }

            (Some(s1), Some(s2)) => shorter(
                crate::util::duration_to_seconds(s1.key_expiration_time()),
                crate::util::duration_to_seconds(s2.key_expiration_time()),
            ),
        };

        Ok(exp)
    }

    pub fn primary_valid_at(&self, reference: &DateTime<Utc>) -> Result<bool, Error> {
        let creation = self.primary_creation_time();
        let expiration = self.primary_expiration_time(reference)?;

        // creation time is in the future, relative to the reference time
        if creation > reference {
            return Ok(false);
        }

        // If there is an expiration time, and it is not the value 0, the key is invalid if "creation+expiration" is prior to the reference time
        if let Some(expiration) = expiration {
            if expiration != 0
                && (self.cspk.primary_key.created_at().add(
                    #[allow(clippy::expect_used)]
                    Duration::try_seconds(expiration as i64).expect("should never fail for an u32"),
                ) < *reference)
            {
                return Ok(false);
            }
        }

        // If the primary is using an insufficiently strong public key algorithm (at the reference time), we reject its use
        if !policy::acceptable_pk_algorithm(self.cspk.primary_key.public_params(), reference) {
            return Ok(false);
        }

        // Assert that a valid primary user id binding or dks exist at the reference time
        if !Self::primary_has_valid_binding_at(self, reference, creation) {
            return Ok(false);
        }

        // the primary is revoked at the reference time
        if self.primary_revoked_at(reference) {
            return Ok(false);
        }

        Ok(true)
    }

    /// Check for either a valid dks at reference, or a valid primary key binding at reference.
    fn primary_has_valid_binding_at(
        &self,
        reference: &DateTime<Utc>,
        creation: &DateTime<Utc>,
    ) -> bool {
        // Does a valid dks exist, at "reference"?
        let dks_stack: SigStack = SigStack::from_iter(self.cspk.details.direct_signatures.iter());
        if dks_stack.has_valid_binding_at(reference, creation) {
            return true;
        }

        // Is the primary user id binding valid, at "reference"?
        if let Some(user) = key::primary_user_id(&self.cspk.details) {
            let stack = SigStack::from_iter(user.signatures.iter());
            if stack.has_valid_binding_at(reference, creation) {
                return true;
            }
        }

        false
    }

    // takes into account the semantics of hard and soft revocation
    pub(crate) fn primary_revoked_at(&self, reference: &DateTime<Utc>) -> bool {
        let stack = SigStack::from_iter(
            self.cspk
                .details
                .revocation_signatures
                .iter()
                .chain(self.cspk.details.direct_signatures.iter()),
        );

        if stack.revoked_at(reference) {
            log::debug!("primary is revoked");
            return true;
        }

        // Consider if the primary user id is revoked
        //
        // (Note: this approach considers the *current* primary User ID.
        // Alternatively, the primary User ID at `reference` could be found and checked.)
        if let Some(primary_uid) = primary_user_id(&self.cspk.details) {
            let stack = SigStack::from_iter(primary_uid.signatures.iter());

            if stack.revoked_at(reference) {
                log::debug!("primary User ID is revoked");
                return true;
            }
        }

        false
    }

    pub fn preferred_symmetric_key_algo<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> Option<&'a [SymmetricKeyAlgorithm]> {
        let prim_bind = self.primary_user_id_binding(reference);
        if let Some(prim_bind) = prim_bind {
            return Some(prim_bind.preferred_symmetric_algs());
        }

        // FIXME: handle dks

        None
    }

    pub fn preferred_aead_algo<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> Option<&'a [(SymmetricKeyAlgorithm, AeadAlgorithm)]> {
        let prim_bind = self.primary_user_id_binding(reference);
        if let Some(prim_bind) = prim_bind {
            return Some(prim_bind.preferred_aead_algs());
        }

        // FIXME: handle dks

        None
    }

    pub fn preferred_hash_algorithms<'a>(
        &'a self,
        reference: &'a DateTime<Utc>,
    ) -> Option<&'a [HashAlgorithm]> {
        let prim_bind = self.primary_user_id_binding(reference);
        if let Some(prim_bind) = prim_bind {
            return Some(prim_bind.preferred_hash_algs());
        }

        // FIXME: handle dks

        None
    }

    pub fn features<'a>(&'a self, reference: &'a DateTime<Utc>) -> Option<u8> {
        let prim_bind = self.primary_user_id_binding(reference);
        if let Some(prim_bind) = prim_bind {
            let feat = prim_bind.features();
            if feat.len() == 1 {
                return Some(feat[0]);
            }
        }

        // FIXME: get dks at reference time
        if let Some(dks) = self.dks() {
            let feat = dks.features();
            if feat.len() == 1 {
                return Some(feat[0]);
            }
        }

        None
    }
}

impl From<&Certificate> for CheckedCertificate {
    fn from(value: &Certificate) -> Self {
        // Make a copy of all parts of the SignedPublicKey and drop all signatures that are not cryptographically correct

        let spk = &value.spk;

        // primary
        let primary = spk.primary_key.clone();

        // details
        let revocation_signatures: Vec<_> = spk
            .details
            .revocation_signatures
            .iter()
            .filter(|s| sig::signature_acceptable(s)) // FIXME: could this filter out any important revocations?
            .filter(|s| s.verify_key(&spk.primary_key).is_ok())
            .cloned()
            .collect();

        let direct_signatures: Vec<_> = spk
            .details
            .direct_signatures
            .iter()
            .filter(|s| sig::signature_acceptable(s))
            .filter(|s| s.verify_key(&spk.primary_key).is_ok())
            .cloned()
            .collect();

        let users: Vec<SignedUser> = spk
            .details
            .users
            .iter()
            .map(|su| {
                let id = su.id.clone();
                let signatures = su
                    .signatures
                    .iter()
                    .filter(|s| sig::signature_acceptable(s))
                    .filter(|s| s.verify_certification(&primary, Tag::UserId, &id).is_ok())
                    .cloned()
                    .collect();

                SignedUser { id, signatures }
            })
            .collect();

        let user_attributes: Vec<SignedUserAttribute> = spk
            .details
            .user_attributes
            .iter()
            .map(|sua| {
                let attr = sua.attr.clone();
                let signatures = sua
                    .signatures
                    .iter()
                    .filter(|s| sig::signature_acceptable(s))
                    .filter(|s| {
                        s.verify_certification(&primary, Tag::UserAttribute, &attr)
                            .is_ok()
                    })
                    .cloned()
                    .collect();

                SignedUserAttribute { attr, signatures }
            })
            .collect();

        let details = SignedKeyDetails::new(
            revocation_signatures,
            direct_signatures,
            users,
            user_attributes,
        );

        // subkeys
        let subkeys = spk
            .public_subkeys
            .iter()
            .map(|sk| {
                let key = sk.key.clone();
                let signatures = sk
                    .signatures
                    .iter()
                    .filter(|s| sig::signature_acceptable(s))
                    .filter(|s| {
                        if let Err(e) = s.verify_key_binding(&spk.primary_key, &key) {
                            log::warn!("Ignoring bad key binding signature {:#?}: {:?}", s, e);
                            return false;
                        }

                        // If there is an embedded back signature, we reject the entire signature if the back signature is cryptographically invalid
                        for embedded in s
                            .config
                            .hashed_subpackets
                            .iter()
                            .filter(|sp| sp.typ() == SubpacketType::EmbeddedSignature)
                        {
                            match &embedded.data {
                                SubpacketData::EmbeddedSignature(backsig) => {
                                    if !sig::signature_acceptable(backsig) {
                                        log::warn!(
                                            "Ignoring signature because of unacceptable embedded signature {:#?}",
                                            backsig
                                        );

                                        return false;
                                    }

                                    if let Err(e) =
                                        backsig.verify_backwards_key_binding(
                                            &key,
                                            &spk.primary_key,
                                        )
                                    {
                                        log::warn!(
                                            "Ignoring signature because embedded signature doesn't verify as correct {:#?}: {:?}",
                                            backsig, e
                                        );

                                        return false;
                                    }
                                }
                                _ => unreachable!("no other subpacket types should come up, here"),
                            }
                        }

                        true
                    })
                    .cloned()
                    .collect();
                SignedPublicSubKey { key, signatures }
            })
            .collect();

        // combine
        let cspk = SignedPublicKey::new(primary, details, subkeys);
        Self { cspk }
    }
}

impl From<CheckedCertificate> for Certificate {
    fn from(value: CheckedCertificate) -> Self {
        value.cspk.into()
    }
}
