// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! Handle stacks of OpenPGP signatures.
//!
//! A stack of signatures collectively refers to one component of a certificate, and can represent
//! changes over time, or a series of binding signatures and revocation signatures.

use std::ops::Add;

use chrono::{DateTime, Utc};
use pgp::Signature;

use crate::sig;
use crate::sig::{is_hard_revocation, is_revocation};

pub struct SigStack<'inner> {
    hard: Vec<&'inner Signature>,
    soft: Vec<&'inner Signature>,
    bind: Vec<&'inner Signature>,

    #[allow(dead_code)]
    invalid: Vec<&'inner Signature>,

    #[allow(dead_code)]
    third_party: Vec<&'inner Signature>,
}

impl<'a> FromIterator<&'a Signature> for SigStack<'a> {
    fn from_iter<T: IntoIterator<Item = &'a Signature>>(iter: T) -> Self {
        let mut hard = vec![];
        let mut soft = vec![];
        let mut bind = vec![];
        let mut invalid = vec![];
        let third_party = vec![];

        // FIXME: pre-process?
        // - Different Enum variants based on hard-revoked vs. not?
        // - Sort signatures by creation time?
        // - Calculate time spans for validity of the component?

        for sig in iter.into_iter() {
            if is_hard_revocation(sig) {
                hard.push(sig)
            } else if is_revocation(sig) {
                soft.push(sig)
            } else if sig::signature_acceptable(sig) {
                bind.push(sig)
            } else {
                invalid.push(sig)

                // FIXME: handle third party sigs
            }
        }

        Self {
            hard,
            soft,
            bind,
            invalid,
            third_party,
        }
    }
}

impl<'a> SigStack<'a> {
    // FIXME: this is an odd hack, and kind of expensive to make.
    // This probably should be replaced with a more reasonable access mechanism.
    pub fn all(&self) -> Vec<&Signature> {
        // TODO: always return two sets: "valid" and "invalid"?
        //
        // -> for hard revoked, all other sigs are "invalid" (because they are suspect and can't be
        // relied on, they are only of informational value)
        //
        // -> for non-hard-revoked, only "self.invalid" are considered "invalid"? (unsound
        // signatures of all kinds might be of interest to viewers, but shouldn't be relied on for
        // anything)

        let mut all: Vec<&Signature> = vec![];
        self.hard.iter().for_each(|&s| all.push(s));
        self.soft.iter().for_each(|&s| all.push(s));
        self.bind.iter().for_each(|&s| all.push(s));

        // FIXME: handle "invalid" sigs?

        // FIXME: handle third party sigs?

        // sort by creation, newest first
        // FIXME: missing creation times are not legal, do we want to handle them here?
        all.sort_by(|a, b| b.created().cmp(&a.created()));

        all
    }

    /// Get the latest active signature in this stack.
    pub fn active(&self) -> Option<&'a Signature> {
        self.active_at(None)
    }

    /// Get the currently active signature in this stack at a reference time.
    /// If the reference time is `None`, then the latest signature.
    pub(crate) fn active_at(&self, reference: Option<&DateTime<Utc>>) -> Option<&'a Signature> {
        // If there is any hard revocation, that is always active
        // (if there are multiple, picking a specific one is probably pointless)
        if let Some(&hard) = self.hard.first() {
            return Some(hard);
        }

        let mut latest_binding: Option<&Signature> = None;

        self.bind
            .iter()
            .filter(|s| {
                if let Some(reference) = reference {
                    if let Some(expired) = sig::validity_end(s) {
                        // only consider signatures that are valid longer than the reference time
                        expired > *reference
                    } else {
                        true
                    }
                } else {
                    true
                }
            })
            .filter(|&&s| {
                if let Some(reference) = reference {
                    if let Some(sig_created) = s.created() {
                        // only consider signatures that were created before the reference time
                        sig_created <= reference
                    } else {
                        false // signature has no creation time, we ignore it
                    }
                } else {
                    true
                }
            })
            .for_each(|s| {
                // replace "latest" with any newer Signatures
                if let Some(cur) = latest_binding {
                    if let Some(sig_created) = s.created() {
                        if let Some(cur_created) = cur.created() {
                            if cur_created < sig_created {
                                latest_binding = Some(s);
                            }
                        }
                    }
                } else {
                    latest_binding = Some(s);
                }
            });

        let mut latest: Option<&Signature> = latest_binding;
        self.soft
            .iter()
            .filter(|s| {
                if let Some(reference) = reference {
                    if let Some(created) = s.created() {
                        created <= reference
                    } else {
                        false // signature has no creation time, we ignore it
                    }
                } else {
                    true
                }
            })
            .filter(|s| {
                if let Some(reference) = reference {
                    if let Some(exp) = sig::validity_end(s) {
                        exp > *reference
                    } else {
                        true
                    }
                } else {
                    true
                }
            })
            .for_each(|s| {
                if let Some(cur) = latest {
                    if let (Some(cc), Some(sc)) = (cur.created(), s.created()) {
                        if cc < sc {
                            latest = Some(s)
                        }
                    }
                } else {
                    latest = Some(s)
                }
            });

        latest
    }

    /// Does this signature stack contain a (non-revocation) Signature that is temporally valid at
    /// `reference`?
    pub fn has_valid_binding_at(
        &self,
        reference: &DateTime<Utc>,
        key_creation: &DateTime<Utc>, // FIXME: factor out?
    ) -> bool {
        // FIXME: we could search more efficiently in large stacks, if the stack were sorted (maybe later)
        self.bind
            .iter()
            .any(|s| sig::is_signature_valid_at(s, key_creation, reference))
    }

    pub(crate) fn revoked_at(&self, reference: &DateTime<Utc>) -> bool {
        self.contains_hard_revocation() || self.soft_revoked_at(reference)
    }

    // CAUTION: this fn doesn't currently consider hard revocations to be valid at all times!
    fn soft_revoked_at(&self, reference: &DateTime<Utc>) -> bool {
        for sig in self.soft.iter().filter(|&s| match s.created() {
            None => false, // we just ignore signatures without creation time
            Some(created) => created <= reference,
        }) {
            // we only consider signatures with creation time
            if let Some(rev_created) = sig.created() {
                // This revocation could currently be active
                // FIXME: handle expiration "0"
                if sig.signature_expiration_time().is_none()
                    || sig
                        .signature_expiration_time()
                        .is_some_and(|rev_exp| rev_created.add(*rev_exp) > *reference)
                {
                    // we're inside the revocation's active window

                    // there could be a newer binding that overrides this revocation.
                    // -> check if there is a newer binding signature
                    //
                    // (FIXME: inefficient!!
                    // signatures should be pre-processed and ordered, not accessed in n^2 patterns).

                    if !self.bind.iter().any(|rebind| match rebind.created() {
                        None => false, // we just ignore signatures without creation time
                        Some(rebind_created) => {
                            // We are searching for a:
                            // - binding signature that was created before the reference time
                            rebind_created <= reference &&

                             // - but after the revocation was created
                             rebind_created > rev_created &&

                             // - and that is still valid at the reference time
                             if let Some(rebind_exp) = rebind.signature_expiration_time() {
                                 rebind_exp.num_seconds() == 0 ||
                                     (rebind_created.add(*rebind_exp) < *reference)
                             } else {
                                 // no expiration time, binding is valid
                                 true
                             }
                        }
                    }) {
                        // if there is none, this soft revocation is in effect
                        return true;
                    }
                }
            }
        }

        false
    }

    fn contains_hard_revocation(&self) -> bool {
        !self.hard.is_empty()
    }
}
