//  Copyright (c) 2014 Anton Bikineev
//  Copyright (c) 2022 John Maddock
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

/*
This program will generate spot tests for the Bessel I function derivative

What kind of distribution do you require for parameter x?
Choices are:
  r     Random values in a half open range
  p     Evenly spaced periodic values in a half open range
  e     Exponential power series at a particular point: a + 2^b for some range of b
[Default=r]e
Data will be in the form a + r*2^b
for random value r,
enter value for the point a [default=0]:
Data will be in the form a + r*2^b
for random value r,
enter value for the starting exponent b:-6
Data will be in the form a + r*2^b
for random value r,
enter value for the ending exponent b:6
What kind of distribution do you require for parameter v?
Choices are:
  r     Random values in a half open range
  p     Evenly spaced periodic values in a half open range
  e     Exponential power series at a particular point: a + 2^b for some range of b
[Default=r]p
Data will be in the half open range a <= x < b,
enter value for the start point fo the range [default=0]:-100
Enter value for the end point fo the range [default=1]:100
How many data points do you want?80
Any more data [y/n]?n
Enter name of test data file [default=bessel_j_derivative_data.ipp]bessel_i_prime_data.ipp
*/

#ifndef SC_
#  define SC_(x) static_cast<T>(BOOST_JOIN(x, L))
#endif
static const std::array<std::array<T, 3>, 1040
#if LDBL_MAX_EXP <= 1024
   - 3
#endif
> bessel_i_prime_data = { {
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(5.46280750189455683138351898451207439397006824699457e-340) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.39777612103065133708619206124315546634136568852863e-330) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.79642440399203002444081227788985603160023586846295e-278) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(4.18136949471207477823074340435911436814427564100662e-250) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.90467451470586286669614915953886543915332620146806e-241) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.79278588425575476107419392734822442295073811357800e-187) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(6.74900049568567569657164186255772680024611990546425e-159) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.09678874554234071823038085266998758193712450952760e-148) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(4.43006678831679704080831578793905898052962180366512e-106) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.00573337185159116494421668378037231135569844925463e-85) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(2.97378117886220461875985196516291709117536661221212e-63) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.05151120752087879149299190379873290575568854950382e-16) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+02), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.32670786997734799678294220021613111033853384876468e+10) },
#if LDBL_MAX_EXP > 1024
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.76755242253470399230847252793609449738347813539348e+334) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.50483115145949489513587409374947142674428258617165e+324) },
#endif
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(7.07697285630593420674963235774648912119957558944248e+272) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(6.56792498004268774462039610399676965187980515262410e+244) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.04786276452536831096784541243075161722021086184652e+236) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(3.17345567321601307675282358374639450063031420743454e+182) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.17520417710309661914200854515420046377064371926237e+154) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.27330675145896777104779822356827555641319991761344e+143) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.30570258976108344550275384360299726195922054419642e+101) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.84186568511080298476311658917672866106153380283169e+81) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(8.04852016104414377604590926437481666697279585183299e+58) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.90499890328876207765771734247673318547901449964199e+12) },
      { SC_(-9.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.68779416827653791286129576808108482933453322002256e+11) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.18542679888084963950414196282973818084654474952126e-321) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.17069720755816389951405545468103636214699899847522e-311) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.20335021231778929369935789279311499768503474036973e-262) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.80866787575938193457425332256265626822123352639881e-235) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(4.49351612382772294396969376919022139946024690120993e-227) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.32635319379701974578846036193905708407708768697527e-176) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(7.23151409807336697517941100106335840076150891474214e-149) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.63576869162958050220392762285506613735713047918048e-139) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.03107653136176035235760759523591142708896908751537e-98) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.23280928175358086888431510188892697271423608765143e-79) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(5.09495819112136964646021346488025703346763974022120e-58) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.55731895371937933654845195096689319438726302230835e-13) },
      { SC_(-9.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.04419656784587743182600575811873025987891903945959e+12) },
#if LDBL_MAX_EXP > 1024
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-1.27525194207158464561983681765009052289835342622273e+315) },
#endif
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-9.97451270958290059910644733258266240142724730352285e+305) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-6.67875183219230948997649502638394947448396135028601e+256) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-1.64148529318496808985981635865440654460396405164557e+230) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-7.32258889777952964551179318600167274590899421150911e+221) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-1.15472730402767664644148709471435844156168216953376e+171) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-1.18568808167108993215071409344746285496528322787973e+144) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-1.45973162417126929649867707173132197979552942722164e+134) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-1.53534655000745909938010391515909295664384509960050e+94) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-8.96750784432296225939292358310454313056395919270724e+74) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(-5.07627595095194247995364382525407334936865618760277e+53) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(-1.64629862934602109882357459657571983020557258319462e+09) },
      { SC_(-9.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.35538861553219623642198466348770334242100202263274e+13) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(9.41285686128658451060703770209881243872785094701696e-302) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.38667268778174698190814394043509213566067879298964e-293) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.81618084632350604919901063985439270020509007466291e-246) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(6.00418598361598717044275067665442789001075144084087e-221) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.33494986465037246939055415429801547003459908479049e-213) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.21448604914545876807790466476063491239282707947918e-164) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(5.94699619055216925909814350592256094554576063171103e-139) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.61183843741670967885286007598343562654959871920043e-129) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.84291063465461773653143690881457811682450558434945e-91) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.81052942435387140326347239024004029567021694826341e-73) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.72966382867938460059157387916568274850872983469897e-53) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.52257772536358422712395375562152646979636394728379e-10) },
      { SC_(-9.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.57994807599883837638606718008896814380157749111086e+14) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.20398392016698755639580833061596566806276012085978e+296) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.89005333876768933023572016227825584278771569107493e+287) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(8.24791157764993472461900299903519580572180606650177e+240) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.36841976439526175938633080014890471703325797464822e+215) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(6.69615314319490694759371821129164439636140481732043e+207) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.49820586501044413214751117131715537780030073254178e+159) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.56532401967031947921411760182333869076798675785764e+134) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.52444132052616885819761548482437311607061104140712e+124) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(9.32540136198238001847414187690479223697358800748223e+86) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.70380689612777924822547564679322817368756817544940e+68) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(4.17038568260645094717774846146070827428096781518718e+48) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.80454127178088985173735729448709725989484684671613e+06) },
      { SC_(-8.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.68431597400342478922932212120889211584833257387849e+15) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.18518243068690328208380501428337019677170243414966e-283) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.24294715814863266734407474262394233344709840653011e-274) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.87216251761416730387368528265698502751868263291315e-230) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.50721993844507064038522653724868584027904999178336e-206) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(4.78962784572914322386917563461115129075555669603445e-199) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.09406111035686595251249820408882995014841656995330e-153) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.69846140508077457766462252738563817174855908688326e-129) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.96090539303714577158641354650166737160463846384374e-120) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.49271111832464322918734679425664212778959186357223e-84) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.92699153292648426526505429839220280934405888629809e-67) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.75745564039792447131802761396005773597734385256800e-48) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.94937807561386747937654369121439044340272003140849e-07) },
      { SC_(-8.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.65088301336454052187658637194704888407704344761568e+16) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-1.51039182362239346637126775958420280482512317976055e+277) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-1.11266459951247790978860681022160763187098835415178e+269) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-1.35343320613920488181726311156110356842989297722691e+225) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-2.33291962668382308609036465242815549989016009441928e+201) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-8.13635205541668030427625085199206936193522049073036e+193) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-3.47855768566123771021962899264214146259804876331254e+148) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-2.74570319683364627277900593057487281450107479402467e+124) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-3.87445963585862054144109531000821315656303275274836e+115) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-7.52038040267563673438442394729453138985579914714060e+79) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-4.81118134893350267919377269188949696821470825980581e+62) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(-4.52793666598986959090039770024280744972643834530885e+43) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(-2.53500466027633741322621094898615167429063442163660e+03) },
      { SC_(-8.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.48276750823486474060657340933525905220305719052464e+17) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(5.29385702608361950448815925274576252614590311711346e-264) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.61943146877955126700734310595821376472501228659345e-256) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(9.25685818032656722294966561870769696533246621224521e-215) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.81408062474547955764268078970315125183440386105901e-192) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.19823855452303136776100806329827587762681916921145e-185) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.68554227475101519395094826438033576454069814128402e-142) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.71086220982726745212869232677042748853159539549568e-119) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.04593755217569112516379325811586437214702396937943e-111) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.50995537710370208667432215889880412880724222009097e-77) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.74906983165574669184974330904485358535819452559997e-61) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(5.07865541738400374336612481740406668824780447615831e-43) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.16750607052414388139926212734846634425743362561236e-04) },
      { SC_(-8.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.20352983589290883854957948796758758739875076533072e+18) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.56122536525445329795526681829151502802657626412947e+258) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(5.79044456469924453241591767742111058261290284483163e+250) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.00205581717822788873311073084903558824202822921418e+209) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.37038258894104125395860148684975986619275733360759e+187) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.33635727280191423027649660052532766188567843039225e+180) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.97479886935904189643074937615244204751728522367896e+137) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(6.50965513514942284710057962033747550939058507428260e+114) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(3.10965720080476456053551911315477466606139836341986e+106) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(8.19039519304625296558304197463347321901109692467664e+72) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.47288667688958446687170344092492783217269789134333e+56) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.60272787296093943564609724818639845399285916781526e+38) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(4.61786559378827945509343491929220442112734384135676e+00) },
      { SC_(-7.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.85175184573379626524620152616617057905152033918467e+19) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.49952278169797473456165720296675150649003637414912e-245) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.02997607271212029244008231619064931012579492345753e-238) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.34137230790743380876876098324797716830232428078237e-199) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.83564195293647507885023245979765633314608851449169e-178) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.55905649471839636740611688917453652111701808074174e-171) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.51435367155690146096244678992730623733758144129344e-131) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(5.77817105956549176122242340952130875180153322843139e-110) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.03664556449785522365131601358406407411476539330283e-102) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.84700936372874611705506871838251333454669661381793e-70) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.35095742633518295372998701144339737731040494055495e-55) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(2.80759662819294075538721467256037408188837237350578e-38) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(5.35678390586655747668969956830256857461963156009185e-02) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.47219797790328187387466999507467914949765228383357e+20) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-5.98544542698375238690459541866765136760413766576253e+239) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-4.15288641968459965186396783588594510800016829068310e+232) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-9.17679564065687187646877082976317344326512912348074e+193) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-1.10936335981821874523657616325313073287385113691556e+173) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-3.02485847798275543445528168755410651449137284022264e+166) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-3.50586857609682608042979706652357457942784491693289e+126) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-2.12672701033837631269912844464054981261581857400582e+105) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-3.43904138643969593272019360984175291077524305962326e+97) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-1.22795781612359794786505587827387701548095899404032e+66) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-8.55570830136910874218257161903315078992983216709551e+50) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(-1.31671448238794291313870689437196943375967997405859e+34) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.02330737801755495063287358232916342250245611721311e+00) },
      { SC_(-7.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.10637717271923712884854231267282410471713879970172e+21) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.43525453413554537730989598510849754042274597688378e-227) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.43153416295312211950732773302700697291801631063724e-220) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(8.62069953671594441302557036286537297164656426934448e-184) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.73677097605554530429420708651484553737597014073263e-164) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.43212940837896658457516631574782364471043421282418e-158) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.68263509419184324521097090329771388906482047724310e-120) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.39498998523654358035044288800505290846459483365495e-100) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.87911014066551582300689008745352533502265085264154e-93) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(9.72741792616118679117989957054038784372049182058765e-64) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.69564222960539878436957377831286009674489972775830e-49) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.11956137993706241084238864498886945328866698897254e-33) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.85963016633151753146474357288614003966952720164146e+01) },
      { SC_(-7.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(7.85278120357811777155984367067723692672221798856323e+21) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.97056291474797527317024567223023645623268820931924e+221) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.19597933918344870512628590100964604792750386739140e+214) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.95192162800872581133908054609791866556513941018714e+178) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.26517395314878097850933850232082682562921570540486e+159) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(9.64565248975382929756816872668023627631821694308588e+152) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.82057028425663435851508141884682982788662569510538e+115) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(9.78724660808995373789539986897115120705816876976289e+95) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(5.35705117751032116338050235543907123468992170724695e+88) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.59015857841660540213344370271149514339783185109003e+59) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.87785375490356997114259790913016177376093627593952e+45) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.66478124272370215662932625701496162674181239237814e+29) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.10791880668150778700844135426638004430401996214568e+02) },
      { SC_(-6.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.25960403690546945996193900465479285510513962344305e+22) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.98603057936174552811232241344376238324270294158177e-208) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(3.39979233084818162312947920509271097451880960306202e-202) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.55169734345263684752873234176104216345697850990618e-168) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.53981689186680809686321054686719943477771347650826e-150) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.32046322660178314676121257473929358078627161114149e-144) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(7.85626657518478268122908209294544631097506615685231e-110) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.34995235341920510317674189715250698426629599890048e-91) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.43298391632541810636189917311051998981437311256163e-84) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.57992920190790650774436780340148774196035102725641e-57) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(6.01076677196347560905048744134280479499289687655607e-44) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.14976166849346500051070584406691329301347341633046e-29) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(4.81974087956697597447185044522089246258190589499542e+03) },
      { SC_(-6.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.32134945640843712554281744063247964050692579824701e+23) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-9.37329973464498637475853751385306967863179282058407e+202) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-6.12526652721167369752129457809945478246386771383889e+196) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-2.45885958877280863328682413074235492738756146255937e+163) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-2.08465484398421388556595993061085555076651190782990e+145) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-4.44390945255779935345372434913065473308243502336806e+139) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-1.39613501621879832989769897411012396690373111371062e+105) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-6.50662447517320061919556568496157451351591371521630e+86) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-1.20537361784125871714413325430870463790603301149204e+80) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-7.88062162981288298077813424090470006402046504432517e+52) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-5.92996836263585416513135897315662069034968547570286e+39) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(-1.45689756805254503469017329296988449876418754043782e+25) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.92305867840681176538473887176002964196034872970339e+04) },
      { SC_(-6.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.97573237597237671863493132657255124908032592671867e+24) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.17231925324243504347178767346918555536850397371549e-190) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.76951359768360916047248622361115349535909562858001e-184) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.89485080002709030109360316986709634042617700615159e-153) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.17115028041843780250378785492337280685972082082491e-136) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.17764641253034064241993714661899253108416954398264e-131) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.48866140924490039924441069776736798923180467265140e-99) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.68612515138613053845540991151341580908433991605473e-82) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.84006655180869667483577713004767809830967458733536e-76) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(8.95639728217720215964675943698028091748304896954395e-51) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.45297701861840152918258347553014669698308704800326e-38) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.09634885168702298423069318138480800095430201837628e-25) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(9.19337230557646876720876583409830211961432823951551e+05) },
      { SC_(-6.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.10614422992245015475843262401611862354684868154173e+25) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(6.63302304094289572897661457944254352326324981898853e+184) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.33025050421028005196463794870293974754697403584193e+179) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.27601778963644087144281248983441618286551365495359e+148) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.11015149973671763734945498271787579147277743380015e+131) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.04588794389288589007075790914542245636879991530110e+126) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(4.98179113990008122525133129105903079585003914841183e+94) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(6.43413210422215715320464276465722114618071168871922e+77) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.03373819183896149789773080709590225752576596793829e+71) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.55968642589190968412304914291487595408492714528607e+46) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.77125102283441162277099909462097458318130504146135e+34) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(8.49402062827053148901229880644577786055907506463897e+20) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.12644346669317128367524250348809481542404145032238e+07) },
      { SC_(-5.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.82347689255271006528541323467542687915294703309466e+25) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.32720946855693803069563167596262295780957393666856e-172) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(6.04738657054198948419556675142528941474375846102943e-167) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.51934247012505263125003862197517627767001046564221e-138) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.54598195723310742577193131981188827836816462770754e-123) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.35810317401951623973375331981120591459823033907415e-118) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.17670488916081504180551358357722019066441196011773e-89) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.01652117847363759523493774716959611718228620290220e-73) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.07382049322545402791301949655447339632112529102453e-67) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.47511335111284002981847973610534993508414887913306e-44) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.32133499351554624868070815564755010085375824232389e-33) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(7.88567033382882282141643525095834570636947274359772e-21) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.27095428481594751252286767949638414570737865851989e+08) },
      { SC_(-5.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.88042387642622605392375386164326306525458865294040e+26) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-7.22693693095810802531415327564549967397878891666719e+166) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-4.44800860772356247240652905691635865740649068603850e+161) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-3.24370215001313965402936933340287801411132332153118e+133) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-1.92866029768148190813482180348373812052445468217975e+118) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-3.21428411640196071384066567556438422804610744107168e+113) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-2.73678718633212310510033773079077244342143027919025e+84) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-9.79373718771481039251691512808766718452176741180125e+68) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-2.07756894547531154268825696264501732212968621310206e+63) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-2.46909104539254198239242797526338499161900778819918e+40) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-1.98066235056800857769523289391764567592429164104159e+29) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(-7.48759792476981279127481866663527863438114493477327e+16) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.31776559601048382646253757388499797517836080903464e+09) },
      { SC_(-5.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.33736808513984300232664953648285898419977005483058e+27) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.20440305129406600589197685893213993827553250157987e-154) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.30553697937811662914259202280467804535477292247915e-149) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(7.69563398392545733570840687514410670940814922604929e-124) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(6.78218785671386258629305019411789065474118736177241e-110) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.61305274104518383652812793052030358465897100623231e-105) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(6.80271287494173415524714115954761780126964879702142e-79) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(9.56563065156063906042899081179222765537205790631481e-65) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.50565643432763488291297065266551753819136776613541e-59) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.53987483633979823202002945577825919939412284268613e-38) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.36271764017471861146959172977776925515932057653752e-28) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.59169698701673641770394588088053061523860093600185e-17) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.25288173125494071383198703351311033231527297408015e+10) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.82350796624128701998561716369486055909281207829726e+27) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.26284981995161863689181930941949642065166130170698e+149) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.38535198381201115847616386711238760482089401813715e+144) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(7.41413503951116333689744246688879419183725073642402e+118) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.16742917931932377509809765704672939366131303912376e+105) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.44009506108408089964441556298467983752808211422473e+100) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.41110854971400804797892292884754775401244907966014e+74) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.39018452790029672598454365988842830869649933103353e+60) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.71532306262405858084169437630345649734042531877666e+55) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.73725776975761070925745113024464674958899184163880e+34) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.25063799382355437748122987041696310997801434349582e+24) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.03403997605529709290741494933139417933640746300944e+13) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.08992617022702801595543345943383595522395878917979e+11) },
      { SC_(-4.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.37617783818625117944059291410264010742967660199696e+28) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.80631606092349301189948348218601032422544351739061e-137) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.69942909803089950252923881678693127650889720230837e-132) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.35031545082644481289668507737559394825453261986351e-109) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(7.82164170197302272216316373880051122934712808512557e-97) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(6.65319701181203255439724849562071696906875290628989e-93) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.39073779327289453430619994280435093196071087765504e-69) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.73707387126750800248750282862225584554769141187934e-56) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.27376326077956616414380599421380816667653343184277e-51) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(9.73620691574486916929545936678475464890983390377433e-33) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.46652225892929644699909421952940601022764621492320e-23) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.42356034378141411879935851094996564868874522538616e-13) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(8.65620941934883092212403663696354652413756408846468e+11) },
      { SC_(-4.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(9.07734996472545577064683121714728324044420384363904e+28) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-3.71831973115102851583894416657469011769986367998619e+131) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-2.15544471161285214775486185103121138244699430628672e+127) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-2.85546576542219394701148448813278070471527046076361e+104) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-1.19069478873808700460043426452800763732054641559202e+92) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-1.55139554636897092051386153015228574564959741766866e+88) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-3.57885243203929135651777240345674281162793096785920e+64) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-9.82510425634540528019918962065658839748648639799764e+51) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-2.38478576549438255367248652384729726921480837610571e+47) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-5.08954991267924063814699113129392337264400271009960e+28) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-4.25922113230784384168572612182337617471518678922180e+19) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(-2.33262656956282917086278383246647984104094486395671e+09) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.26195813898820651449202967227838550619965033557900e+12) },
      { SC_(-4.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.24381189554806917902613037734742684976210358769970e+29) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.00357836992075323562917575493624983108075772251360e-119) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.25987005311916603719701247301012415396832810976843e-115) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(4.08806288682719758194933812406854529963931785117493e-95) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.13733808992288280749829723102587266000006636514321e-84) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.56288148320602025271745778404666866910346815464731e-80) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.43324150468804075874230529746509408058163645553391e-59) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(4.46275312751529269091251061011964356047426168371894e-48) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.14245867051289473957049466907502901977778743206662e-44) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.52771787407619582039066371395752580507763287339037e-27) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.26514291094924720262772193702177491380786752680752e-19) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.05464601495421448350831280649967736929276286912799e-09) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(4.11656385405158385198843095884821918389633059140182e+13) },
      { SC_(-4.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.08344260616756017880778878294816454091754499416954e+30) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.96064539812492560390026603926611606981301585367237e+114) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(3.48801681967364677518675706805729297283105046857288e+110) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.96947210601258287195590747228136774611184297157924e+90) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.17482453078625637652529957982431652258125300579619e+79) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(7.92301863602723110424236073757134987556542188287019e+75) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(9.51182599671112543013941692038865840079985492726763e+54) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(7.22865821869198642376032285548428012842928187273709e+43) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(5.93219441293739675451370442840534938174872865701935e+39) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.68390041435881523797065469540555535429806729035628e+23) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.42047177481125984797501304742656078416079104843044e+15) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(9.03374883274453780912801186111018943442116377077584e+05) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.45345848230867249444258341968849050532155652141662e+14) },
      { SC_(-3.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.37952117805307843365122038934279935451206670770796e+30) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.41671076880927093736335322743182090339233615669403e-102) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.95170512512632870771102128417896256060575627919686e-99) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.76978422957893655497749852541820663873608556596051e-81) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.78891390817212773438932219222564140708295714577058e-71) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.94641410540924377800760599854168918166951967972569e-68) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.82395365830654795675597561165203660713190726973636e-50) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.86067274082998914010268204907559401363854324893174e-40) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.57235725878318919504092115670618571586623537804978e-36) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(6.83688617617695449583789967349689020540127744224665e-22) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.02451732690741868620971959946198122294611268713762e-14) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.82299872685207264820135919985074448718800545806536e-06) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.32254241360924630511645701759043222854789075643703e+15) },
      { SC_(-3.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(9.83690505376235837672129276579512168076746242258305e+30) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-1.99979182303951390054131864662486198581621101334075e+97) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-1.09182647403097929333047219879317560545084027442181e+94) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-2.62757329125100659272317138521573225867098750545861e+76) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-7.68380683396045962012245508312612735681639480029242e+66) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-7.82680825990777537149870291414789664450917115909809e+63) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-4.88905196458870995235656430757880679110558820325560e+45) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-1.02792502628494312209741410800401922074807627297743e+36) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-2.85065670598250014901468650727792583957372915821289e+32) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-1.06778051563574648998920603153660598280990115383442e+18) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-8.95551340277120967467294255803946171426287970874148e+10) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(-6.37725717643937950054084838753565289322577265178199e+02) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.43268230776197038701659647836897775333769914785350e+15) },
      { SC_(-3.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.66982110963209192375459225156640373531930461279595e+31) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.40879708219259741132877959881556619276515846647680e-85) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(9.40566016100782949779962217259008287522072409040894e-83) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.68005196115327446850868633785886087567386617650291e-67) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.01060239541393908959969971573128579674509122391964e-59) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.17155006818270747350400306357634513225054929707879e-56) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(6.73893660569467154871691972518151221554470861512473e-41) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.61599195955402009153188070848882601788211866776224e-32) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.94893007842383080348091767436874214437735149081358e-29) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(6.48967710291836850342104453603008662168464170781219e-17) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(9.95624636669086597669309211230607449604216652407209e-11) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.65365716167116619228751500849689378245898692250779e-03) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.81639885469924214122600027280057619589874541547799e+16) },
      { SC_(-3.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(6.75158158465640520814221608991663795444834908269221e+31) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(4.36341140241355097919072120327334063150852489343274e+80) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(7.31112559414187069979428133544692451136290247777662e+77) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(7.49917575797329029216941466648122606800982580055943e+62) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.80733315979604360403829364165312411283783116535877e+54) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.65395198605362109150161557017276008743484488643125e+52) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.37402867267788278010237451569038928490881820219217e+36) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.12311851042664772955789143069799240541199529547579e+28) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.92458366232567416818112592472290026690727597760625e+25) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.42839323605161126024891581248790112299693065324364e+13) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.16694435155108509386586942691567170178108283283119e+07) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(9.21007040348294741700470515926592623725032018157928e-01) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.10737558501919467091275204623406294392003722968420e+17) },
      { SC_(-2.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.58972688995251787389075224765476172345365195534377e+32) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.50493330340561395036360210480031855325976584716768e-69) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(7.62485243450803178974360230728414178884595983156296e-67) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.19550509674502532332280087892547971603201602869725e-54) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.16240306780296012150205073557558146683495761140908e-47) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.00962101330724016909783919336602003732705971626817e-45) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(3.32959583732301295699930882959867905049566111900929e-32) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.89233902201708071140548305788617567992969257181455e-25) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.03417007648891350991445239113888388030346050597844e-22) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.68583435974083065762395322058224410312563372019185e-12) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.34891353820013735173432712153585405046545006069715e-07) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(7.25673719544418235200900871392301230122157347795219e-01) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.90115970114754230102879627728603148340998771350345e+17) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.48292017181125330750646524596410185414369875914635e+32) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-2.33601313541825803827267913225480398218195805589289e+64) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-1.20121986221924881004169031292925772653284555162191e+62) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-5.25142046756980482667135465137005547012216342210973e+49) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-1.07689283788830888517797105980650889071919426597949e+43) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-8.57531712860174497793089936372972362799430161808820e+40) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-1.44861615915366491727807019930210396435917057215468e+28) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-2.32357557005434319810793167790597004726085844261551e+21) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-7.33807392514952746456069611958242519081909789951470e+18) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-4.58400408624895099988912024896768262262340699406576e+08) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(-3.53175569170313918173809005301283491741123789645671e+03) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.12309315318168822352103627724873068797833621991244e+01) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.22862635015407184898038463778857218516856281795059e+18) },
      { SC_(-2.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(7.09572831756088900209759697411494447367778898107573e+32) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.12096785507279605331562443216676787734870987475217e-53) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.21234070275113470496654693182056988352268822301240e-51) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.17540405120771896014398839325276811152020520901692e-41) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.55928578470387470727439810199722943754665283701547e-36) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.76740461412367373562433050871945866970201156820574e-34) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.89331531579153993561273522272037319763020883560662e-24) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.85911077386839696415817814207605844615219044298541e-18) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.97492005188783018911290854000266528926640614540837e-16) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(4.12983438657130570868297973330147836527342570044029e-08) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(7.42655521165895681800334899715215295471870395930812e-04) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.39523932231237132496358360385857135975581365280083e+02) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.45178272050670719350892315639079460693797489138485e+18) },
      { SC_(-2.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.34347979388816511038497457146077083044055027430557e+33) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.74020643326103589385865656823108616027723770258076e+48) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(5.90244223390772413523528334685178516942327619126980e+46) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.09977928632414552403487068739692894347053405009159e+37) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.97196673274830564697668330068963021586693596008169e+31) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.32958236234124723842957313366488553492369876202446e+30) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.16671973188078038581573188362186101967123325973939e+20) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(5.15164992109123866839770810266305916221129780816726e+14) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(5.47460537250619184633580522784266674049219544478803e+12) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(4.22804102720706330692809742035364788255085251225511e+04) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.92751067968070117863611938795096081641003221433353e+00) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.36896650140960079356600668729687866747915505191833e+03) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(8.63350891428525107864885709718098124591437978028005e+18) },
      { SC_(-1.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.36273852821554009433681055992689855932997671833191e+33) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(7.60282016022613997794115048546515744036579236083500e-38) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.75609835693646591807945482552146334421206141872739e-36) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(4.05160682006928171331353276106965718162384024224128e-29) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.91038967537710957910878225112769132842176841421802e-25) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(6.96264586273295343879410420898124516598646504739158e-24) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.85891983409786493668964838638954066708931247249304e-16) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.29181967685715495144315287605817268766122577715805e-12) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.04338065548365995727369734173532510187841031848746e-10) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.87051204194293884976519702726506484350520390093046e-04) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.10105191881972453256421658175848861518181741416368e-01) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.04350385489283087048235341418792150117699903156388e+04) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.91884037358696150741755416356950498761023256018246e+19) },
      { SC_(-1.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.85783088376159647023750248346369638961917361216016e+33) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-2.44534915909685338690378912810789335051890646369125e+33) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-1.18431055853990980892608473748543793165842282954885e+32) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-9.40475699321703270869329848810976627717744032246244e+24) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-1.35220314380230316340075740191843575579627685532824e+21) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-8.41659727086373570569798950998064489439511193639208e+19) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-3.82964469440551445322066736302477119425709921683343e+12) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(-4.62993495363343551516859359617259989798502717320901e+08) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(-1.64807296412412265898118025301434248990273955059995e+07) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(-1.46888858868761104082701655665480831306338109964384e+01) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.82700154356401146026037490795680735202371618787921e+00) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.07168742802799395707810725601676371799312609017045e+04) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.78322488225867641470110828738653931142305657519145e+19) },
      { SC_(-1.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.84562243198730415184101800629256899984983160839763e+33) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.18875328209491670361603361320288318111435592712453e-23) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.39997888612764781722126247912962566031095142148364e-22) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.29928435838459795832699964996280714442092172928102e-17) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(4.73688927884200739836831353747291924549657948124970e-15) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.01721067584587046402087161326151866509932183744715e-14) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.39926430523794942694139110576592385672255517898685e-09) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.02157360519288777062634320399175168962374466787261e-06) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(9.76746462907425147485493013481361541988218763188475e-06) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.74204660309411523100124573237518086104451979865955e-01) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.63791604366979186021980760040334603836430058115459e+01) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(2.64880565292853153704664930937790846476229439563697e+05) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.60709059126284634215848927189635066391009309794676e+19) },
      { SC_(-1.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(8.21714942813260018213754540812875553446477558012208e+33) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.14921207548938628551993193408837278133727972900404e+19) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.70809666503025876762307727976248150639635895999570e+18) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(5.78061378835900621869424436113698899650725116383548e+13) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.20021776700664138849527905693623524940177244147524e+11) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.82822698675947157193186605622118102261673620724958e+10) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(8.99118513572720882608277668572061557008451352422881e+05) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.93559346412670010571374992803126349895814949258121e+03) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(3.45935766242997314720974249105447190271918124372720e+02) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.50554840454607811717044733873815743549517556380612e+00) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.53354723797947987754027862141870166267869894820067e+02) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(8.51473483843048392045604743072797708759065183239064e+05) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.02078948631556887938373843929350076117701031243996e+20) },
      { SC_(-7.50000000000000000000000000000000000000000000000000e+00), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.07123352655392790536755903725128003870988901368515e+34) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.41746070122966522393246955983935371425996803947278e-10) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.06435427848444420220421050208610196045775698336279e-09) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.62313127113795582644911083470445779436690611615023e-07) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.61546142567933375547323713479924051999476525305307e-06) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(8.24296992503429982572548633663219338194001996068252e-06) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.29357190116274043182983920142690186881862337375656e-03) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.15070867139277663410679279318811429479089921975176e-02) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.48015322750358290698459590073138781803280553660306e-02) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.85613637171643983724767751021229937010050229398295e+01) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.36972114319513077631374358604601180784168088129908e+03) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.98565654287427934707018755602857967895875962070647e+06) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.39380827626743598042051734125954123737911145399929e+20) },
      { SC_(-5.00000000000000000000000000000000000000000000000000e+00), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.29484345839834474315445651422939207947567872142970e+34) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(-1.55877335355521395219196873800108038185049395392446e+06) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(-7.11023115535223615055036713748440084109215772212555e+05) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(-1.02553284187270223187834350343013191983115440678710e+04) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(-1.03345089427920750258205774274372638163736166043086e+03) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(-5.02660057252771366920917868688004595504607542925669e+02) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(-5.90966940092102862542600318304635099542395568960363e+00) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.74175503469390926311175267677911902662766798168565e-02) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(8.14758575174525306879868717335501643517247933138700e-01) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(6.55521182677398799030847421312927564053838068194102e+01) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.14704372704061937334863584431868772521173038973056e+03) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.31788939619840425225188127484025278439443933348443e+06) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.68070837571409642524890917422351435504598979311983e+20) },
      { SC_(-2.50000000000000000000000000000000000000000000000000e+00), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.45093109613169805855912068713012511894817857126233e+34) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.41789521840478238369426946538019112164007699907257e-02) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.77446216233457678304509910920487034493378173104835e-02) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(5.96616749986228886344798080730112924507064916616433e-02) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.15443959902062335180264878696666652470161160942996e-01) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.42275779909277215249788165808833419850001937481460e-01) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.54300395177297331226573309060611457386686532425843e-01) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.46661078032534146120135216400626101205678212301797e+00) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.38775484828940635005448870235941777886939476090618e+00) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.00458865357564995962068722074501358410790875946073e+02) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.16575978042828069183052498122942493415240389575251e+03) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.94075139585929041322562757096141240034386278383841e+06) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.78900055416381743475427403263526986593436075311169e+20) },
      { SC_(0.00000000000000000000000000000000000000000000000000e+00), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.50705204275601075144138377289678254826420663055414e+34) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(6.35008348804097541692686530323236025521787343900026e-04) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(8.88999868553912352029192968039914689179861759739321e-04) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(5.47663681802360819323128963368728325104898524862785e-03) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.47077110003512553658452661966995509562568018150025e-02) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.00909698247641717090112019375473684693254584036170e-02) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.46713421865807740813825685074918513443693851647136e-01) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(5.43802399620504898145105792443882032569477258458126e-01) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.00153965032775161015501113630813372500134048848368e+00) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(6.55529109101383890555013158025992878664829137595476e+01) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.14704373695072858955366967898966026064296348752094e+03) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.31788939619840976438392861462187970896714338367686e+06) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.68070837571409642524890917422351435504599017607051e+20) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+00), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.45093109613169805855912068713012511894817857126233e+34) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.41746070122966522393246955983935371425996803947278e-10) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.06435427848444420220421050208610196045775698336279e-09) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.62313127113795582644911083470445779436690611615023e-07) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.61546142567933375547323713479924051999476525305307e-06) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(8.24296992503429982572548633663219338194001996068252e-06) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.29357190116274043182983920142690186881862337375656e-03) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.15070867139277663410679279318811429479089921975176e-02) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.48015322750358290698459590073138781803280553660306e-02) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.85613637171643983724767751021229937010050229398295e+01) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.36972114319513077631374358604601180784168088129908e+03) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.98565654287427934707018755602857967895875962070647e+06) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.39380827626743598042051734125954123737911145399929e+20) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+00), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.29484345839834474315445651422939207947567872142970e+34) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.58376728743692045516042841426250119694327668199733e-16) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.11006169646554023461036717906921708596113857244156e-15) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.91126852753989478999759462630257449042681908639062e-12) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.06327043631344778731675968745212749499230563508155e-10) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(7.86154681034100663568168085349795333220543409460433e-10) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.76371899625376185490766322234863810899127697238717e-06) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.29353790937841946216703405895416555309637863703200e-04) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.21776150724437249093519678970171976232046775575283e-03) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.47554150433062000480086788453746229223137067268985e+00) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.53354617931607359507830377084391425908037745091646e+02) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(8.51473483843025256442294779698594471441424544987404e+05) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.02078948631556887938373843929350076117700967552988e+20) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+00), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.07123352655392790536755903725128003870988901368515e+34) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.18875328209491670361603361320288318111435592712453e-23) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.39997888612764781722126247912962566031095142148364e-22) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.29928435838459795832699964996280714442092172928102e-17) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(4.73688927884200739836831353747291924549657948124970e-15) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.01721067584587046402087161326151866509932183744715e-14) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.39926430523794942694139110576592385672255517898685e-09) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.02157360519288777062634320399175168962374466787261e-06) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(9.76746462907425147485493013481361541988218763188475e-06) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.74204660309411523100124573237518086104451979865955e-01) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.63791604366979186021980760040334603836430058115459e+01) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(2.64880565292853153704664930937790846476229439563697e+05) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.60709059126284634215848927189635066391009309794676e+19) },
      { SC_(1.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(8.21714942813260018213754540812875553446477558012208e+33) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.02376188278096167524443773911579005100211343113338e-30) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.66832990169031696231235299498584524976862324243847e-29) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.98209504382072888756267673523922590720256391781690e-23) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.59365313556996780081169795701129534403407399616489e-20) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.95685661289076947889181596792690128690101504393982e-19) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.07542386827737649441294949914971221522351333258584e-12) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.37489176327585048224150249111178554560067314528472e-09) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.12522965166003319394451080037387914048506624943888e-08) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(7.16709521573283034562623974867154824578347974542184e-03) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.83512316901832774251822844186895836726055108116020e+00) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.07168742806463450085743339408204400417474438731625e+04) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.78322488225867641470110828738653931142307409999681e+19) },
      { SC_(1.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.84562243198730415184101800629256899984983160839763e+33) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(7.60282016022613997794115048546515744036579236083500e-38) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.75609835693646591807945482552146334421206141872739e-36) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(4.05160682006928171331353276106965718162384024224128e-29) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.91038967537710957910878225112769132842176841421802e-25) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(6.96264586273295343879410420898124516598646504739158e-24) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.85891983409786493668964838638954066708931247249304e-16) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.29181967685715495144315287605817268766122577715805e-12) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.04338065548365995727369734173532510187841031848746e-10) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.87051204194293884976519702726506484350520390093046e-04) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.10105191881972453256421658175848861518181741416368e-01) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.04350385489283087048235341418792150117699903156388e+04) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.91884037358696150741755416356950498761023256018246e+19) },
      { SC_(1.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.85783088376159647023750248346369638961917361216016e+33) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.85238903077021393402958566815909771965730418590745e-45) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(7.49549897420502228994927131882163992188420006047751e-44) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.57011173471376358304657010597506498240538885872391e-35) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.77301323206937216283684862194706468467564269186745e-30) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.27851725302893174538205717630541589292281200140508e-29) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(4.93442792937375989316497821151823370269252748294328e-20) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.97118596896636103249738376238996800089825149260145e-15) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.72274348449591837181719959204171045783650563700067e-13) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.29882731697560557249017218814099156451566702791407e-06) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.03883811724799222244932740647529090037413696778511e-02) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.36896648271554492131691108948003807238531861398156e+03) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(8.63350891428525107864885709718098124591359003904272e+18) },
      { SC_(1.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.36273852821554009433681055992689855932997671833191e+33) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.12096785507279605331562443216676787734870987475217e-53) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.21234070275113470496654693182056988352268822301240e-51) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.17540405120771896014398839325276811152020520901692e-41) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.55928578470387470727439810199722943754665283701547e-36) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.76740461412367373562433050871945866970201156820574e-34) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.89331531579153993561273522272037319763020883560662e-24) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.85911077386839696415817814207605844615219044298541e-18) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.97492005188783018911290854000266528926640614540837e-16) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(4.12983438657130570868297973330147836527342570044029e-08) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(7.42655521165895681800334899715215295471870395930812e-04) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.39523932231237132496358360385857135975581365280083e+02) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.45178272050670719350892315639079460693797489138485e+18) },
      { SC_(2.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.34347979388816511038497457146077083044055027430557e+33) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.81326279355051341050884549734420248866059304088938e-61) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.73536985010161817145711720312847137205946467943296e-59) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(9.61282478086453518526110642234124726167583772291536e-48) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.26411482303767022311165477154649503999915219766618e-41) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.05220199720928796396425297890609022144973485756852e-39) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.10648794049066245405767516593295431357656237318415e-28) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(8.44973259642925282002469850009322594944867809322914e-22) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.64620130162528892156141520359978025442899943859154e-19) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.81610610484667585362529688845618718984312459543473e-10) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.04223964139518810974051926993567321619213360104357e-05) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.12335543318761694029435507482192457478251881545004e+01) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.22862635015407184898038463778857218517431017182079e+18) },
      { SC_(2.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(7.09572831756088900209759697411494447367778898107573e+32) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.50493330340561395036360210480031855325976584716768e-69) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(7.62485243450803178974360230728414178884595983156296e-67) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.19550509674502532332280087892547971603201602869725e-54) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.16240306780296012150205073557558146683495761140908e-47) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.00962101330724016909783919336602003732705971626817e-45) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(3.32959583732301295699930882959867905049566111900929e-32) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.89233902201708071140548305788617567992969257181455e-25) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.03417007648891350991445239113888388030346050597844e-22) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.68583435974083065762395322058224410312563372019185e-12) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.34891353820013735173432712153585405046545006069715e-07) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(7.25673719544418235200900871392301230122157347795219e-01) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.90115970114754230102879627728603148340998771350345e+17) },
      { SC_(2.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.48292017181125330750646524596410185414369875914635e+32) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.49514572878218854116023196717447599140269482172682e-77) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(9.50916014460909644402809533052970340223637811199069e-75) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(8.22739680713825330880902500608258685332502708325890e-61) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.86500221553389705041214406499482671796766445989594e-53) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(6.66753490426371633263730266331006024204077840133208e-51) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.68185358645828912686934897389372191929252079375414e-36) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(7.67438917727772829842576401275795951129596277511793e-29) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(5.03861426624718098256215510073253298891567202401119e-26) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.47742649278466804523685015007810406316163482667411e-14) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(7.32858994960378139088599290416335617778354021294280e-09) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.81510942621467279087235741211984077530089620762205e-02) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.10737558501919467091275204623406294325595295975575e+17) },
      { SC_(2.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.58972688995251787389075224765476172345365195534377e+32) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.40879708219259741132877959881556619276515846647680e-85) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(9.40566016100782949779962217259008287522072409040894e-83) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.68005196115327446850868633785886087567386617650291e-67) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.01060239541393908959969971573128579674509122391964e-59) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.17155006818270747350400306357634513225054929707879e-56) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(6.73893660569467154871691972518151221554470861512473e-41) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.61599195955402009153188070848882601788211866776224e-32) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.94893007842383080348091767436874214437735149081358e-29) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(6.48967710291836850342104453603008662168464170781219e-17) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(9.95624636669086597669309211230607449604216652407209e-11) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.65365716167116619228751500849689378245898692250779e-03) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.81639885469924214122600027280057619589874541547799e+16) },
      { SC_(3.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(6.75158158465640520814221608991663795444834908269221e+31) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(6.43410125172353698426024023843099861714536670000010e-94) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(7.52529188289009370058390210936080374110391040841931e-91) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.77504892878210637785301689691762273932621518918694e-74) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.55900793324198619036125680216967900990488722005144e-65) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.66512821685793499322428321029609555802610098781281e-62) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.18441450937891178806352589505925894846965399640692e-45) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.75374368650535357160974059667615841634239172396392e-36) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.10234493011540471262561061310343162771602446199459e-33) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.31792705747573948691376872598277770904338269895242e-19) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.10808982318468810344572138777130761261206521634547e-12) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(5.97948483409799034926140632177538921093274626702299e-05) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.43268230776197038701659647836897894883555907524602e+15) },
      { SC_(3.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.66982110963209192375459225156640373531930461279595e+31) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.41671076880927093736335322743182090339233615669403e-102) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.95170512512632870771102128417896256060575627919686e-99) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.76978422957893655497749852541820663873608556596051e-81) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.78891390817212773438932219222564140708295714577058e-71) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.94641410540924377800760599854168918166951967972569e-68) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.82395365830654795675597561165203660713190726973636e-50) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.86067274082998914010268204907559401363854324893174e-40) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.57235725878318919504092115670618571586623537804978e-36) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(6.83688617617695449583789967349689020540127744224665e-22) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.02451732690741868620971959946198122294611268713762e-14) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.82299872685207264820135919985074448718800545806536e-06) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.32254241360924630511645701759043222854789075643703e+15) },
      { SC_(3.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(9.83690505376235837672129276579512168076746242258305e+30) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(7.57218978658367824265468726814645876554374740014021e-111) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.71797990745281479996899922204781048140319201269664e-107) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(4.27191722505457115314069037613269673961827401150364e-88) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.04320355424362153665512939911841768748486687002139e-77) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.89795336064985372201406744527557076375640567426611e-74) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.29537075119309493277142956249771943429698847423832e-54) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(4.51635536607560907229541499264716544794195663185731e-44) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(3.38119380975029009100686165855118342575602133745247e-40) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.68773125643137007461612331324015703150959897057558e-24) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(7.96594571536946150289238997160586783298885221941044e-17) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(4.73152869992399144888241182767188687326865523144308e-08) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.45345848230867249444258341968816178887653113411487e+14) },
      { SC_(3.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.37952117805307843365122038934279935451206670770796e+30) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.00357836992075323562917575493624983108075772251360e-119) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.25987005311916603719701247301012415396832810976843e-115) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(4.08806288682719758194933812406854529963931785117493e-95) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(5.13733808992288280749829723102587266000006636514321e-84) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.56288148320602025271745778404666866910346815464731e-80) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.43324150468804075874230529746509408058163645553391e-59) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(4.46275312751529269091251061011964356047426168371894e-48) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.14245867051289473957049466907502901977778743206662e-44) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.52771787407619582039066371395752580507763287339037e-27) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.26514291094924720262772193702177491380786752680752e-19) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.05464601495421448350831280649967736929276286912799e-09) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(4.11656385405158385198843095884821918389633059140182e+13) },
      { SC_(4.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.08344260616756017880778878294816454091754499416954e+30) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(4.52513404351120614588350056992854103729841280989236e-128) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.98477481669179014421823649182681170095150545798452e-124) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.33928035111520718531447742469537164212247898041171e-102) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.15947752816073275231176321058967198473497588279211e-90) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.09852164274484810655088027245483232770307816246810e-86) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(3.90156362443657421977075089220969246377671165337239e-64) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.76479414225908290617139434654502482839595337062115e-52) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(9.52774909418433917568004101306309474691373875767349e-48) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(6.30778342328215967150555671199425305203910376363290e-30) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.98669919326370093442681673070512249148531183378906e-21) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(2.03494134280637173108829900558572483988385743662204e-11) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.26195813898820651449202967363144730536500957346723e+12) },
      { SC_(4.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.24381189554806917902613037734742684976210358769970e+29) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.80631606092349301189948348218601032422544351739061e-137) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.69942909803089950252923881678693127650889720230837e-132) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.35031545082644481289668507737559394825453261986351e-109) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(7.82164170197302272216316373880051122934712808512557e-97) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(6.65319701181203255439724849562071696906875290628989e-93) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.39073779327289453430619994280435093196071087765504e-69) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.73707387126750800248750282862225584554769141187934e-56) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.27376326077956616414380599421380816667653343184277e-51) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(9.73620691574486916929545936678475464890983390377433e-33) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.46652225892929644699909421952940601022764621492320e-23) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.42356034378141411879935851094996564868874522538616e-13) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(8.65620941934883092212403663696354652413756408846468e+11) },
      { SC_(4.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(9.07734996472545577064683121714728324044420384363904e+28) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.48912492600136303740139677485202221265560040243382e-145) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(5.03424802237874582705165992613550029906516842652804e-141) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.43738691452429881090018872515151435148236135490163e-116) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.46162568453219667545010613302032268537419654778027e-103) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.50128484319539443543974305051152574841269557532025e-99) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(6.47212881990707269193760058895181815650854604324284e-74) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.72927451911482529421974222756886118771472176876647e-60) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.47997958619560008642527491752279086347686564187536e-55) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.30780579368815375520638298136137483572769436568415e-35) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(6.28099728964097865591769345235718285879155265583781e-26) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(5.05535784046343258712522019888753458770151017496360e-15) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.08992617022702801595543264200407898545314843584100e+11) },
      { SC_(4.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.37617783818625117944059291410264010742967660199696e+28) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.20440305129406600589197685893213993827553250157987e-154) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.30553697937811662914259202280467804535477292247915e-149) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(7.69563398392545733570840687514410670940814922604929e-124) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(6.78218785671386258629305019411789065474118736177241e-110) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.61305274104518383652812793052030358465897100623231e-105) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(6.80271287494173415524714115954761780126964879702142e-79) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(9.56563065156063906042899081179222765537205790631481e-65) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.50565643432763488291297065266551753819136776613541e-59) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.53987483633979823202002945577825919939412284268613e-38) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.36271764017471861146959172977776925515932057653752e-28) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.59169698701673641770394588088053061523860093600185e-17) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.25288173125494071383198703351311033231527297408015e+10) },
      { SC_(5.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.82350796624128701998561716369486055909281207829726e+27) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.87603558291557597874206559120846138303501565612302e-163) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.98391971395633071568163913995931997212519509992191e-158) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.63127095552890223037940551195232934905313522560540e-131) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.64687862494866496804257450704105422383360015641588e-116) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(6.54958793010752781660334690365669140857528094222683e-112) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(6.30190214374483722416440565422546685345723182674571e-84) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(4.66389481024070744712532365038610046094042366185266e-69) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.35023060415052456894089668352444093816761836333832e-63) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.59976147358906227987749350553610573205209697061189e-41) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(7.85519474242346002735105163733430184359453509062025e-31) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(7.63169962451884210233494368108354462410292097498786e-19) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.31776559601048382646964855860354377459801973595548e+09) },
      { SC_(5.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.33736808513984300232664953648285898419977005483058e+27) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.32720946855693803069563167596262295780957393666856e-172) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(6.04738657054198948419556675142528941474375846102943e-167) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.51934247012505263125003862197517627767001046564221e-138) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.54598195723310742577193131981188827836816462770754e-123) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.35810317401951623973375331981120591459823033907415e-118) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.17670488916081504180551358357722019066441196011773e-89) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.01652117847363759523493774716959611718228620290220e-73) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.07382049322545402791301949655447339632112529102453e-67) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.47511335111284002981847973610534993508414887913306e-44) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.32133499351554624868070815564755010085375824232389e-33) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(7.88567033382882282141643525095834570636947274359772e-21) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.27095428481594751252286767949638414570737865851989e+08) },
      { SC_(5.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.88042387642622605392375386164326306525458865294040e+26) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.43198629133136049047205502598400154600077107039869e-181) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.09276768904019931128860919672395197264585470149400e-175) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(5.66803244600354594246492979166266719075532561689645e-146) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(6.80755891826569422635195692414790537381909753743691e-130) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(7.56992919929017235206361547126274109039921273603225e-125) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(3.79160426433392426862283771589250538257203658930154e-94) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(7.77441655493209306563293687899062132227958295252944e-78) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(7.61528517124703268185658404181266688309824878030493e-72) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.21377408487952530556715390555892279483346522640441e-47) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(6.12934916569906707862638505758718790723652657448284e-36) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(7.30620575362974696785441471895456314388451744810032e-23) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.12644346669317119620484423317589721740528825900730e+07) },
      { SC_(5.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.82347689255271006528541323467542687915294703309466e+25) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(3.17231925324243504347178767346918555536850397371549e-190) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.76951359768360916047248622361115349535909562858001e-184) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.89485080002709030109360316986709634042617700615159e-153) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.17115028041843780250378785492337280685972082082491e-136) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.17764641253034064241993714661899253108416954398264e-131) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.48866140924490039924441069776736798923180467265140e-99) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.68612515138613053845540991151341580908433991605473e-82) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.84006655180869667483577713004767809830967458733536e-76) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(8.95639728217720215964675943698028091748304896954395e-51) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.45297701861840152918258347553014669698308704800326e-38) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.09634885168702298423069318138480800095430201837628e-25) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(9.19337230557646876720876583409830211961432823951551e+05) },
      { SC_(6.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.10614422992245015475843262401611862354684868154173e+25) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.63983455848297220963103959189148461545205874509822e-199) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.57957670678252547164929327314495461587646685575016e-193) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(5.70277556291832045440594819124958860919583919700702e-161) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.81385541954769425623074243233191574476073703850803e-143) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.63964684894673015450897269828334681833636747164173e-138) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.47056251876456130164578049883362586851944080264388e-104) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(8.35559333374380470748318228465500766821727084742935e-87) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.76964606197474627770331957437679267601143185609914e-80) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(5.95355051131349290478358620004395539026948252096488e-54) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.10582804432775541460076973759563715201419359338650e-41) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(4.59963058373213008460599832055098319881451979687901e-27) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(6.92305867842176718348229158700918653052373226668148e+04) },
      { SC_(6.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.97573237597237671863493132657255124908032592671867e+24) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.98603057936174552811232241344376238324270294158177e-208) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(3.39979233084818162312947920509271097451880960306202e-202) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.55169734345263684752873234176104216345697850990618e-168) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.53981689186680809686321054686719943477771347650826e-150) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.32046322660178314676121257473929358078627161114149e-144) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(7.85626657518478268122908209294544631097506615685231e-110) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.34995235341920510317674189715250698426629599890048e-91) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.43298391632541810636189917311051998981437311256163e-84) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(3.57992920190790650774436780340148774196035102725641e-57) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(6.01076677196347560905048744134280479499289687655607e-44) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(3.14976166849346500051070584406691329301347341633046e-29) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(4.81974087956697597447185044522089246258190589499542e+03) },
      { SC_(6.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(3.32134945640843712554281744063247964050692579824701e+23) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.35613416954732049384755134757362969196656906675068e-217) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.06690325976013669772682998055135176487999488898190e-211) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.83208672279469131892382300177950894028000977571213e-176) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.22782456128626346904899383019492676355968130253325e-157) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.80613541823462363802216778783969579575307138592908e-151) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(3.80944267488640479453778071215699289334635386854330e-115) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(5.99883945629403477031166108577093329529226207884937e-96) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.72971488233656056481656837689568330524458478615879e-89) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.95478526302445849160799848902821258859981474360232e-60) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.05717012785166762470463268743794819256667197333339e-46) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.96448333693080338169305500727645806118867002507521e-31) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(3.10791845692568928008774766391257006324085629696154e+02) },
      { SC_(6.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(5.25960403690546945996193900465479285510513962343346e+22) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.43525453413554537730989598510849754042274597688378e-227) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.43153416295312211950732773302700697291801631063724e-220) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(8.62069953671594441302557036286537297164656426934448e-184) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.73677097605554530429420708651484553737597014073263e-164) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.43212940837896658457516631574782364471043421282418e-158) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.68263509419184324521097090329771388906482047724310e-120) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.39498998523654358035044288800505290846459483365495e-100) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.87911014066551582300689008745352533502265085264154e-93) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(9.72741792616118679117989957054038784372049182058765e-64) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.69564222960539878436957377831286009674489972775830e-49) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.11956137993706241084238864498886945328866698897254e-33) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.85963016633151753146474357288614003966952720164146e+01) },
      { SC_(7.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(7.85278120357811777155984367067723692672221798856323e+21) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(4.79546490705058207764283021407262591701992197719075e-236) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.41348195500059267930846642972164370737699855768253e-229) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.77250200640739677950844101664438393589458442694277e-191) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.95385051759407288577380510296533825352069819159554e-171) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(9.61100727277068770297995146082693755867618811908625e-165) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(6.79296583230808384126866088392010327737765345209367e-126) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.96501591647104066596283879854287371309953546396210e-105) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.12585243341092013027463568104087175438848667623723e-97) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(4.42596360303166367137782947882908595780109426984672e-67) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.48831165024691642626482285639387855554642438509546e-52) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(5.84786704868897914547938874244450368444373331189126e-36) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.03432912730012756412173304470619318114646527873107e+00) },
      { SC_(7.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.10637717271923712884854231267282410471713880439954e+21) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.49952278169797473456165720296675150649003637414912e-245) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.02997607271212029244008231619064931012579492345753e-238) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.34137230790743380876876098324797716830232428078237e-199) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.83564195293647507885023245979765633314608851449169e-178) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(1.55905649471839636740611688917453652111701808074174e-171) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.51435367155690146096244678992730623733758144129344e-131) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(5.77817105956549176122242340952130875180153322843139e-110) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.03664556449785522365131601358406407411476539330283e-102) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.84700936372874611705506871838251333454669661381793e-70) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.35095742633518295372998701144339737731040494055495e-55) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(2.80759662819294075538721467256037408188837237350578e-38) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(5.35678390586655747668969956830256857461963156009185e-02) },
      { SC_(7.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.47219797790328187387466999507467914949765228383357e+20) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(1.19796213227674637984427893012699777557367875970135e-254) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(3.38363251708332474163955512873758326606177054078103e-247) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(5.79192134978234750801588278439306631033653296483415e-207) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.42149300236098186534903616749190395255221275032461e-185) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(2.32549104281921449280290737301298793781918398536268e-178) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(8.55768211068425522551044316084335720181703393696314e-137) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.03544160222968819935717432895819943546989714319684e-114) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.33088125999525011045514590833004264626962391190763e-106) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(7.08976398107123528621758609229381835487091810539237e-74) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.15292067233845557024716529123008047971985246475709e-58) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.24228463082654571969577107163277705098876705072511e-40) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.58746867720226425254716968991447262678987229776859e-03) },
      { SC_(7.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.85175184573379626524620152616617057905149140350897e+19) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(5.29385702608361950448815925274576252614590311711346e-264) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.61943146877955126700734310595821376472501228659345e-256) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(9.25685818032656722294966561870769696533246621224521e-215) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.81408062474547955764268078970315125183440386105901e-192) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.19823855452303136776100806329827587762681916921145e-185) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.68554227475101519395094826438033576454069814128402e-142) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(1.71086220982726745212869232677042748853159539549568e-119) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(4.04593755217569112516379325811586437214702396937943e-111) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.50995537710370208667432215889880412880724222009097e-77) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.74906983165574669184974330904485358535819452559997e-61) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(5.07865541738400374336612481740406668824780447615831e-43) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.16750607052414388139926212734846634425743362561236e-04) },
      { SC_(8.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.20352983589290883854957948796758758739875076533072e+18) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.16248697882340575079990450273697510268041589752583e-273) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.87448974172876681141136019350690104899346101707112e-265) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.36759270896667150777210528656861536856524785872022e-222) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(2.13948740349753505048499929788083084011203454332936e-199) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(4.06592561632400334238955848564587224848321096887878e-192) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(7.79045234070875041696581608038172779423815914154509e-148) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(2.61316417131804022413800264205270814459450985670476e-124) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.13701869322216018156534426833123821271846527145316e-115) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(8.21616307379682608548961846840657818655961911599469e-81) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(5.02357530218167600692797020981388616676256763360065e-64) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.92285295336708990782300561205967987108766934452506e-45) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(4.92855412909209950968198174283541205114271405654841e-06) },
      { SC_(8.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.48276750823486474060657340933525905242556452160119e+17) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.18518243068690328208380501428337019677170243414966e-283) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.24294715814863266734407474262394233344709840653011e-274) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(1.87216251761416730387368528265698502751868263291315e-230) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.50721993844507064038522653724868584027904999178336e-206) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(4.78962784572914322386917563461115129075555669603445e-199) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.09406111035686595251249820408882995014841656995330e-153) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(3.69846140508077457766462252738563817174855908688326e-129) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(2.96090539303714577158641354650166737160463846384374e-120) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.49271111832464322918734679425664212778959186357223e-84) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.92699153292648426526505429839220280934405888629809e-67) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.75745564039792447131802761396005773597734385256800e-48) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(1.94937807561386747937654369121439044340272003140849e-07) },
      { SC_(8.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.65088301336454052187658637194704888407704344761568e+16) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.87724299852046791428521319520754153560876565269858e-292) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(7.65412553306091974783049905215208825181112712411563e-284) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.38014635251401466594303134650733156330018290360079e-238) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(9.86090925337253983226398171437684858931305262666233e-214) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.23983494483481662633971550678102181970534840537342e-206) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.22747012411942521193269733336905495219734499556633e-159) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(4.86135319239407332903636817978063378046405189716922e-134) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(7.16091318631641332788905462859094170849134416381681e-125) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(7.02498288677246231882636286388220700831366642696114e-88) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(4.49026229837347181644972180436663414406434840214109e-70) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(2.20890283634607540107314434573460426207329448830906e-50) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(7.23445750601868326976268835149976312891517391652967e-09) },
      { SC_(8.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.68431597400342478922932212120888999404699911219465e+15) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(9.41285686128658451060703770209881243872785094701696e-302) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(4.38667268778174698190814394043509213566067879298964e-293) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(2.81618084632350604919901063985439270020509007466291e-246) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(6.00418598361598717044275067665442789001075144084087e-221) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.33494986465037246939055415429801547003459908479049e-213) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.21448604914545876807790466476063491239282707947918e-164) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(5.94699619055216925909814350592256094554576063171103e-139) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.61183843741670967885286007598343562654959871920043e-129) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.84291063465461773653143690881457811682450558434945e-91) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.81052942435387140326347239024004029567021694826341e-73) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(6.72966382867938460059157387916568274850872983469897e-53) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.52257772536358422712395375562152646979636394728379e-10) },
      { SC_(9.00000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.57994807599883837638606718008896814380157749111086e+14) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.87167252064905485738235347153191877379241471734206e-311) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(2.34446323539101400406796530860150707782775132126067e-302) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.10732006632536686138240175402917109999138381128600e-254) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(3.40925868418232833121356917099458800587482863508881e-228) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(5.06537700480881447259605504969795478783895642293805e-220) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(2.63126088257471974509908084587034959241724565495787e-170) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(6.78443569090732282402592732015294432816039334051317e-144) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(3.38340992008189633868990248057417746395336422508207e-134) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(4.50934552276430678485301201923961936452947102334492e-95) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(3.01700473994389042330713980818514657465958811213678e-76) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.91449810825425756963089670638931287631293619618813e-55) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(8.27534601994386640897924924519538878047002821396932e-12) },
      { SC_(9.25000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.35538861553219623642198466351263112556631749824818e+13) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(8.18542679888084963950414196282973818084654474952126e-321) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(1.17069720755816389951405545468103636214699899847522e-311) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.20335021231778929369935789279311499768503474036973e-262) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(1.80866787575938193457425332256265626822123352639881e-235) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(4.49351612382772294396969376919022139946024690120993e-227) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(5.32635319379701974578846036193905708407708768697527e-176) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(7.23151409807336697517941100106335840076150891474214e-149) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(6.63576869162958050220392762285506613735713047918048e-139) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(1.03107653136176035235760759523591142708896908751537e-98) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(2.23280928175358086888431510188892697271423608765143e-79) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(5.09495819112136964646021346488025703346763974022120e-58) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(2.55731895371937933654845195096689319438726302230835e-13) },
      { SC_(9.50000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(2.04419656784587743182600575811873025987891903945959e+12) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(2.83550545573234558105468750000000000000000000000000e-02), SC_(2.18384335596940857161232177433497221916832551586631e-330) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(3.54836583137512207031250000000000000000000000000000e-02), SC_(5.47166470979730405884069162017175191656687819498888e-321) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(1.19111984968185424804687500000000000000000000000000e-01), SC_(3.09098297535156874307060417930088890849261731917614e-270) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(2.29376077651977539062500000000000000000000000000000e-01), SC_(8.98114105636141290718024688841332384221926170348717e-243) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(2.81746625900268554687500000000000000000000000000000e-01), SC_(3.73108120013170480930298717327211228704175221592385e-234) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(9.84433889389038085937500000000000000000000000000000e-01), SC_(1.00918626866997364655587505527209717575311535219007e-181) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(1.91337585449218750000000000000000000000000000000000e+00), SC_(7.21480267755741464973418660716789794205123342630387e-154) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(2.44206809997558593750000000000000000000000000000000e+00), SC_(1.21817962307069738506400361309399059373360227108107e-143) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(6.52943611145019531250000000000000000000000000000000e+00), SC_(2.20704604930042013401778037389128929119545934294888e-102) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(1.04653358459472656250000000000000000000000000000000e+01), SC_(1.54731305428608076697396048841900880420328810846151e-82) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(1.75606460571289062500000000000000000000000000000000e+01), SC_(1.27054675898984035053060114053222492212252822558823e-60) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(4.95110473632812500000000000000000000000000000000000e+01), SC_(7.45376086954227815539329617769979714923272862587806e-15) },
      { SC_(9.75000000000000000000000000000000000000000000000000e+01), SC_(8.18238830566406250000000000000000000000000000000000e+01), SC_(1.68779416827653791286129540953230368519390214824983e+11) }
   } };
//#undef SC_

