#include "test_data.hpp"

#include "libslic3r/TriangleMesh.hpp"
#include "libslic3r/GCodeReader.hpp"
#include "libslic3r/Config.hpp"
#include "libslic3r/Print.hpp"
#include "libslic3r/Format/OBJ.hpp"
#include "libslic3r/Format/STL.hpp"

#include <arrange-wrapper/ModelArrange.hpp>

#include <cstdlib>
#include <string>

#include <boost/nowide/cstdio.hpp>
#include <boost/nowide/fstream.hpp>
#include <boost/filesystem.hpp>
#include <boost/regex.hpp>

using namespace std;

namespace Slic3r { namespace Test {

// Mesh enumeration to name mapping
const std::unordered_map<TestMesh, const char*, TestMeshHash> mesh_names {
    std::pair<TestMesh, const char*>(TestMesh::A,						"A"),
    std::pair<TestMesh, const char*>(TestMesh::L,						"L"), 
    std::pair<TestMesh, const char*>(TestMesh::V,						"V"), 
    std::pair<TestMesh, const char*>(TestMesh::_40x10,					"40x10"), 
    std::pair<TestMesh, const char*>(TestMesh::cube_20x20x20,			"cube_20x20x20"), 
    std::pair<TestMesh, const char*>(TestMesh::cube_2x20x10,            "cube_2x20x10"), 
    std::pair<TestMesh, const char*>(TestMesh::sphere_50mm,				"sphere_50mm"), 
    std::pair<TestMesh, const char*>(TestMesh::bridge,					"bridge"), 
    std::pair<TestMesh, const char*>(TestMesh::bridge_with_hole,		"bridge_with_hole"), 
    std::pair<TestMesh, const char*>(TestMesh::cube_with_concave_hole,	"cube_with_concave_hole"),
    std::pair<TestMesh, const char*>(TestMesh::cube_with_hole,			"cube_with_hole"), 
    std::pair<TestMesh, const char*>(TestMesh::gt2_teeth,				"gt2_teeth"), 
    std::pair<TestMesh, const char*>(TestMesh::ipadstand,				"ipadstand"), 
    std::pair<TestMesh, const char*>(TestMesh::overhang,				"overhang"), 
    std::pair<TestMesh, const char*>(TestMesh::pyramid,					"pyramid"), 
    std::pair<TestMesh, const char*>(TestMesh::sloping_hole,			"sloping_hole"), 
    std::pair<TestMesh, const char*>(TestMesh::slopy_cube,				"slopy_cube"), 
    std::pair<TestMesh, const char*>(TestMesh::small_dorito,			"small_dorito"), 
    std::pair<TestMesh, const char*>(TestMesh::step,					"step"), 
    std::pair<TestMesh, const char*>(TestMesh::two_hollow_squares,		"two_hollow_squares")
};

TriangleMesh mesh(TestMesh m)
{
    TriangleMesh mesh;
    switch(m) {
        case TestMesh::cube_20x20x20:
            mesh = Slic3r::make_cube(20, 20, 20);
            break;
        case TestMesh::cube_2x20x10:
            mesh = Slic3r::make_cube(2, 20, 10);
            break;
        case TestMesh::sphere_50mm:
            mesh = Slic3r::make_sphere(50, PI / 243.0);
            break;
        case TestMesh::cube_with_hole:
            mesh = TriangleMesh(
                { {0.f,0.f,0.f}, {0.f,0.f,10.f}, {0.f,20.f,0.f}, {0.f,20.f,10.f}, {20.f,0.f,0.f}, {20.f,0.f,10.f}, {5.f,5.f,0.f}, {15.f,5.f,0.f}, {5.f,15.f,0.f}, {20.f,20.f,0.f}, {15.f,15.f,0.f}, {20.f,20.f,10.f}, {5.f,5.f,10.f}, {5.f,15.f,10.f}, {15.f,5.f,10.f}, {15.f,15.f,10.f} },
                { {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {6,7,4}, {8,2,9}, {0,2,8}, {10,8,9}, {0,8,6}, {0,6,4}, {4,7,9}, {7,10,9}, {2,3,9}, {9,3,11}, {12,1,5}, {13,3,12}, {14,12,5}, {3,1,12}, {11,3,13}, {11,15,5}, {11,13,15}, {15,14,5}, {5,4,9}, {11,5,9}, {8,13,12}, {6,8,12}, {10,15,13}, {8,10,13}, {15,10,14}, {14,10,7}, {14,7,12}, {12,7,6} });
            break;
        case TestMesh::cube_with_concave_hole:
            mesh = TriangleMesh(
                { {-10.f,-10.f,-5.f}, {-10.f,-10.f,5.f}, {-10.f,10.f,-5.f}, {-10.f,10.f,5.f}, {10.f,-10.f,-5.f}, {10.f,-10.f,5.f}, {-5.f,-5.f,-5.f}, {5.f,-5.f,-5.f}, {5.f,5.f,-5.f}, {5.f,10.f,-5.f}, {-5.f,5.f,-5.f}, {float(3.06161699911402e-16),5.f,-5.f},
                  {5.f,0.f,-5.f}, {0.f,0.f,-5.f}, {10.f,5.f,-5.f}, {5.f,10.f,5.f}, {-5.f,-5.f,5.f}, {5.f,0.f,5.f}, {5.f,-5.f,5.f}, {-5.f,5.f,5.f}, {10.f,5.f,5.f}, {5.f,5.f,5.f}, {float(3.06161699911402e-16),5.f,5.f}, {0.f,0.f,5.f} },
                { {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {6,7,4}, {8,2,9}, {10,2,11}, {11,12,13}, {0,2,10}, {0,10,6}, {0,6,4}, {11,2,8}, {4,7,12}, {4,12,8}, {12,11,8}, {14,4,8}, {2,3,9}, {9,3,15}, {16,1,5}, {17,18,5}, {19,3,16}, {20,21,5}, {18,16,5}, {3,1,16}, {22,3,19}, {21,3,22}, {21,17,5}, {21,22,17}, {21,15,3}, {23,17,22}, {5,4,14}, {20,5,14}, {20,14,21}, {21,14,8}, {9,15,21}, {8,9,21}, {10,19,16}, {6,10,16}, {11,22,19}, {10,11,19}, {13,23,11}, {11,23,22}, {23,13,12}, {17,23,12}, {17,12,18}, {18,12,7}, {18,7,16}, {16,7,6} });
            break;
        case TestMesh::V:
            mesh = TriangleMesh(
                { {-14.f,0.f,20.f}, {-14.f,15.f,20.f}, {0.f,0.f,0.f}, {0.f,15.f,0.f}, {-4.f,0.f,20.f}, {-4.f,15.f,20.f}, {5.f,0.f,7.14286f}, {10.f,0.f,0.f}, {24.f,0.f,20.f}, {14.f,0.f,20.f}, {10.f,15.f,0.f}, {5.f,15.f,7.14286f}, {14.f,15.f,20.f}, {24.f,15.f,20.f} },
                { {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {4,0,2}, {6,4,2}, {7,6,2}, {8,9,7}, {9,6,7}, {2,3,7}, {7,3,10}, {1,5,3}, {3,5,11}, {11,12,13}, {11,13,3}, {3,13,10}, {5,4,6}, {11,5,6}, {6,9,11}, {11,9,12}, {12,9,8}, {13,12,8}, {8,7,10}, {13,8,10} });
            break;
        case TestMesh::L:
            mesh = TriangleMesh(
                { {0.f,10.f,0.f}, {0.f,10.f,10.f}, {0.f,20.f,0.f}, {0.f,20.f,10.f}, {10.f,10.f,0.f}, {10.f,10.f,10.f}, {20.f,20.f,0.f}, {20.f,0.f,0.f}, {10.f,0.f,0.f}, {20.f,20.f,10.f}, {10.f,0.f,10.f}, {20.f,0.f,10.f} },
                { {0,1,2}, {2,1,3}, {4,5,1}, {0,4,1}, {0,2,4}, {4,2,6}, {4,6,7}, {4,7,8}, {2,3,6}, {6,3,9}, {3,1,5}, {9,3,5}, {10,11,5}, {11,9,5}, {5,4,10}, {10,4,8}, {10,8,7}, {11,10,7}, {11,7,6}, {9,11,6} });
            break;
        case TestMesh::overhang:
            mesh = TriangleMesh(
                { {1364.68505859375f,614.398010253906f,20.002498626709f}, {1389.68505859375f,614.398010253906f,20.002498626709f}, {1377.18505859375f,589.398986816406f,20.002498626709f}, {1389.68505859375f,589.398986816406f,20.002498626709f}, {1389.68505859375f,564.398986816406f,20.0014991760254f}, {1364.68505859375f,589.398986816406f,20.002498626709f}, {1364.68505859375f,564.398986816406f,20.0014991760254f}, {1360.93505859375f,589.398986816406f,17.0014991760254f}, {1360.93505859375f,585.64697265625f,17.0014991760254f}, {1357.18505859375f,564.398986816406f,17.0014991760254f}, {1364.68505859375f,589.398986816406f,17.0014991760254f}, {1364.68505859375f,571.899963378906f,17.0014991760254f}, {1364.68505859375f,564.398986816406f,17.0014991760254f}, {1348.43603515625f,564.398986816406f,17.0014991760254f}, {1352.80908203125f,589.398986816406f,17.0014991760254f}, {1357.18408203125f,589.398986816406f,17.0014991760254f}, {1357.18310546875f,614.398010253906f,17.0014991760254f}, {1364.68505859375f,606.89599609375f,17.0014991760254f}, {1364.68505859375f,614.398010253906f,17.0014991760254f}, {1352.18603515625f,564.398986816406f,20.0014991760254f}, {1363.65405273438f,589.398986816406f,23.3004989624023f}, {1359.46704101562f,589.398986816406f,23.3004989624023f}, {1358.37109375f,564.398986816406f,23.3004989624023f}, {1385.56103515625f,564.398986816406f,23.3004989624023f}, {1373.06311035156f,589.398986816406f,23.3004989624023f}, {1368.80810546875f,564.398986816406f,23.3004989624023f}, {1387.623046875f,589.398986816406f,23.3004989624023f}, {1387.623046875f,585.276000976562f,23.3004989624023f}, {1389.68505859375f,589.398986816406f,23.3004989624023f}, {1389.68505859375f,572.64599609375f,23.3004989624023f}, {1389.68505859375f,564.398986816406f,23.3004989624023f}, {1367.77709960938f,589.398986816406f,23.3004989624023f}, {1366.7470703125f,564.398986816406f,23.3004989624023f}, {1354.31201171875f,589.398986816406f,23.3004989624023f}, {1352.18603515625f,564.398986816406f,23.3004989624023f}, {1389.68505859375f,614.398010253906f,23.3004989624023f}, {1377.31701660156f,614.398010253906f,23.3004989624023f}, {1381.43908691406f,589.398986816406f,23.3004989624023f}, {1368.80700683594f,614.398010253906f,23.3004989624023f}, {1368.80810546875f,589.398986816406f,23.3004989624023f}, {1356.43908691406f,614.398010253906f,23.3004989624023f}, {1357.40502929688f,589.398986816406f,23.3004989624023f}, {1360.56201171875f,614.398010253906f,23.3004989624023f}, {1348.705078125f,614.398010253906f,23.3004989624023f}, {1350.44506835938f,589.398986816406f,23.3004989624023f}, {1389.68505859375f,606.153015136719f,23.3004989624023f}, {1347.35205078125f,589.398986816406f,23.3004989624023f}, {1346.56005859375f,589.398986816406f,23.3004989624023f}, {1346.56005859375f,594.159912109375f,17.0014991760254f}, {1346.56005859375f,589.398986816406f,17.0014991760254f}, {1346.56005859375f,605.250427246094f,23.3004989624023f}, {1346.56005859375f,614.398010253906f,23.3004989624023f}, {1346.56005859375f,614.398010253906f,20.8258285522461f}, {1346.56005859375f,614.398010253906f,17.0014991760254f}, {1346.56005859375f,564.398986816406f,19.10133934021f}, {1346.56005859375f,567.548583984375f,23.3004989624023f}, {1346.56005859375f,564.398986816406f,17.0020332336426f}, {1346.56005859375f,564.398986816406f,23.0018501281738f}, {1346.56005859375f,564.398986816406f,23.3004989624023f}, {1346.56005859375f,575.118957519531f,17.0014991760254f}, {1346.56005859375f,574.754028320312f,23.3004989624023f} },
                { {0,1,2}, {2,3,4}, {2,5,0}, {4,6,2}, {2,6,5}, {2,1,3}, {7,8,9}, {10,9,8}, {11,9,10}, {12,9,11}, {9,13,14}, {7,15,16}, {10,17,0}, {10,0,5}, {12,11,6}, {18,16,0}, {6,19,13}, {6,13,9}, {9,12,6}, {17,18,0}, {11,10,5}, {11,5,6}, {14,16,15}, {17,7,18}, {16,18,7}, {14,15,9}, {7,9,15}, {7,17,8}, {10,8,17}, {20,21,22}, {23,24,25}, {26,23,27}, {28,27,23}, {29,28,23}, {30,29,23}, {25,31,32}, {22,33,34}, {35,36,37}, {24,38,39}, {21,40,41}, {38,42,20}, {33,43,44}, {6,4,23}, {6,23,25}, {36,35,1}, {1,0,38}, {1,38,36}, {29,30,4}, {25,32,6}, {40,42,0}, {35,45,1}, {4,3,28}, {4,28,29}, {3,1,45}, {3,45,28}, {22,34,19}, {19,6,32}, {19,32,22}, {42,38,0}, {30,23,4}, {0,16,43}, {0,43,40}, {24,37,36}, {38,24,36}, {24,23,37}, {37,23,26}, {22,32,20}, {20,32,31}, {33,41,40}, {43,33,40}, {45,35,26}, {37,26,35}, {33,44,34}, {44,43,46}, {20,42,21}, {40,21,42}, {31,39,38}, {20,31,38}, {33,22,41}, {21,41,22}, {31,25,39}, {24,39,25}, {26,27,45}, {28,45,27}, {47,48,49}, {47,50,48}, {51,48,50}, {52,48,51}, {53,48,52}, {54,55,56}, {57,55,54}, {58,55,57}, {49,59,47}, {60,56,55}, {59,56,60}, {60,47,59}, {48,53,16}, {56,13,19}, {54,56,19}, {56,59,13}, {59,49,14}, {59,14,13}, {49,48,16}, {49,16,14}, {44,46,60}, {44,60,55}, {51,50,43}, {19,34,58}, {19,58,57}, {53,52,16}, {43,16,52}, {43,52,51}, {57,54,19}, {47,60,46}, {55,58,34}, {55,34,44}, {50,47,46}, {50,46,43} });
            break;
        case TestMesh::_40x10:
            mesh = TriangleMesh(
                { {12.8680295944214f,29.5799007415771f,12.f}, {11.7364797592163f,29.8480796813965f,12.f}, {11.1571502685547f,29.5300102233887f,12.f}, {10.5814504623413f,29.9830799102783f,12.f}, {10.f,29.6000003814697f,12.f}, {9.41855144500732f,29.9830799102783f,12.f}, {8.84284687042236f,29.5300102233887f,12.f}, {8.26351833343506f,29.8480796813965f,12.f}, {7.70256900787354f,29.3210391998291f,12.f}, {7.13196802139282f,29.5799007415771f,12.f}, {6.59579277038574f,28.9761600494385f,12.f}, {6.03920221328735f,29.1821594238281f,12.f}, {5.53865718841553f,28.5003795623779f,12.f}, {5.f,28.6602592468262f,12.f}, {4.54657793045044f,27.9006500244141f,12.f}, {4.02841377258301f,28.0212306976318f,12.f}, {3.63402199745178f,27.1856994628906f,12.f}, {3.13758301734924f,27.2737407684326f,12.f}, {2.81429696083069f,26.3659801483154f,12.f}, {2.33955597877502f,26.4278793334961f,12.f}, {2.0993549823761f,25.4534206390381f,12.f}, {1.64512205123901f,25.4950904846191f,12.f}, {1.49962198734283f,24.4613399505615f,12.f}, {1.0636739730835f,24.4879894256592f,12.f}, {1.02384400367737f,23.4042091369629f,12.f}, {0.603073298931122f,23.4202003479004f,12.f}, {0.678958415985107f,22.2974300384521f,12.f}, {0.269550800323486f,22.3061599731445f,12.f}, {0.469994693994522f,21.1571502685547f,12.f}, {0.067615881562233f,21.1609306335449f,12.f}, {0.399999290704727f,20.f,12.f}, {0.f,20.f,12.f}, {0.399999290704727f,5.f,12.f}, {0.f,5.f,12.f}, {0.456633001565933f,4.2804012298584f,12.f}, {0.0615576282143593f,4.21782684326172f,12.f}, {0.625140011310577f,3.5785219669342f,12.f}, {0.244717106223106f,3.45491504669189f,12.f}, {0.901369392871857f,2.91164398193359f,12.f}, {0.544967114925385f,2.73004698753357f,12.f}, {1.27852201461792f,2.29618692398071f,12.f}, {0.954914808273315f,2.06107401847839f,12.f}, {1.74730801582336f,1.74730801582336f,12.f}, {1.46446597576141f,1.46446597576141f,12.f}, {2.29618692398071f,1.27852201461792f,12.f}, {2.06107401847839f,0.954914808273315f,12.f}, {2.91164398193359f,0.901369392871857f,12.f}, {2.73004698753357f,0.544967114925385f,12.f}, {3.5785219669342f,0.625140011310577f,12.f}, {3.45491504669189f,0.244717106223106f,12.f}, {4.2804012298584f,0.456633001565933f,12.f}, {4.21782684326172f,0.0615576282143593f,12.f}, {5.f,0.399999290704727f,12.f}, {5.f,0.f,12.f}, {19.6000003814697f,0.399999290704727f,12.f}, {20.f,0.f,12.f}, {19.6000003814697f,20.f,12.f}, {20.f,20.f,12.f}, {19.5300102233887f,21.1571502685547f,12.f}, {19.9323806762695f,21.1609306335449f,12.f}, {19.3210391998291f,22.2974300384521f,12.f}, {19.7304496765137f,22.3061599731445f,12.f}, {18.9761600494385f,23.4042091369629f,12.f}, {19.3969306945801f,23.4202003479004f,12.f}, {18.5003795623779f,24.4613399505615f,12.f}, {18.9363307952881f,24.4879894256592f,12.f}, {17.9006500244141f,25.4534206390381f,12.f}, {18.3548793792725f,25.4950904846191f,12.f}, {17.1856994628906f,26.3659801483154f,12.f}, {17.6604404449463f,26.4278793334961f,12.f}, {16.3659801483154f,27.1856994628906f,12.f}, {16.862419128418f,27.2737407684326f,12.f}, {15.4534196853638f,27.9006500244141f,12.f}, {15.9715900421143f,28.0212306976318f,12.f}, {14.4613399505615f,28.5003795623779f,12.f}, {15.f,28.6602592468262f,12.f}, {13.4042100906372f,28.9761600494385f,12.f}, {13.9608001708984f,29.1821594238281f,12.f}, {12.2974300384521f,29.3210391998291f,12.f}, {7.13196802139282f,29.5799007415771f,0.f}, {8.26351833343506f,29.8480796813965f,0.f}, {8.84284687042236f,29.5300102233887f,0.f}, {9.41855144500732f,29.9830799102783f,0.f}, {10.f,29.6000003814697f,0.f}, {10.5814504623413f,29.9830799102783f,0.f}, {11.1571502685547f,29.5300102233887f,0.f}, {11.7364797592163f,29.8480796813965f,0.f}, {12.2974300384521f,29.3210391998291f,0.f}, {12.8680295944214f,29.5799007415771f,0.f}, {13.4042100906372f,28.9761600494385f,0.f}, {13.9608001708984f,29.1821594238281f,0.f}, {14.4613399505615f,28.5003795623779f,0.f}, {15.f,28.6602592468262f,0.f}, {15.4534196853638f,27.9006500244141f,0.f}, {15.9715900421143f,28.0212306976318f,0.f}, {16.3659801483154f,27.1856994628906f,0.f}, {16.862419128418f,27.2737407684326f,0.f}, {17.1856994628906f,26.3659801483154f,0.f}, {17.6604404449463f,26.4278793334961f,0.f}, {17.9006500244141f,25.4534206390381f,0.f}, {18.3548793792725f,25.4950904846191f,0.f}, {18.5003795623779f,24.4613399505615f,0.f}, {18.9363307952881f,24.4879894256592f,0.f}, {18.9761600494385f,23.4042091369629f,0.f}, {19.3969306945801f,23.4202003479004f,0.f}, {19.3210391998291f,22.2974300384521f,0.f}, {19.7304496765137f,22.3061599731445f,0.f}, {19.5300102233887f,21.1571502685547f,0.f}, {19.9323806762695f,21.1609306335449f,0.f}, {19.6000003814697f,20.f,0.f}, {20.f,20.f,0.f}, {19.6000003814697f,0.399999290704727f,0.f}, {20.f,0.f,0.f}, {5.f,0.399999290704727f,0.f}, {5.f,0.f,0.f}, {4.2804012298584f,0.456633001565933f,0.f}, {4.21782684326172f,0.0615576282143593f,0.f}, {3.5785219669342f,0.625140011310577f,0.f}, {3.45491504669189f,0.244717106223106f,0.f}, {2.91164398193359f,0.901369392871857f,0.f}, {2.73004698753357f,0.544967114925385f,0.f}, {2.29618692398071f,1.27852201461792f,0.f}, {2.06107401847839f,0.954914808273315f,0.f}, {1.74730801582336f,1.74730801582336f,0.f}, {1.46446597576141f,1.46446597576141f,0.f}, {1.27852201461792f,2.29618692398071f,0.f}, {0.954914808273315f,2.06107401847839f,0.f}, {0.901369392871857f,2.91164398193359f,0.f}, {0.544967114925385f,2.73004698753357f,0.f}, {0.625140011310577f,3.5785219669342f,0.f}, {0.244717106223106f,3.45491504669189f,0.f}, {0.456633001565933f,4.2804012298584f,0.f}, {0.0615576282143593f,4.21782684326172f,0.f}, {0.399999290704727f,5.f,0.f}, {0.f,5.f,0.f}, {0.399999290704727f,20.f,0.f}, {0.f,20.f,0.f}, {0.469994693994522f,21.1571502685547f,0.f}, {0.067615881562233f,21.1609306335449f,0.f}, {0.678958415985107f,22.2974300384521f,0.f}, {0.269550800323486f,22.3061599731445f,0.f}, {1.02384400367737f,23.4042091369629f,0.f}, {0.603073298931122f,23.4202003479004f,0.f}, {1.49962198734283f,24.4613399505615f,0.f}, {1.0636739730835f,24.4879894256592f,0.f}, {2.0993549823761f,25.4534206390381f,0.f}, {1.64512205123901f,25.4950904846191f,0.f}, {2.81429696083069f,26.3659801483154f,0.f}, {2.33955597877502f,26.4278793334961f,0.f}, {3.63402199745178f,27.1856994628906f,0.f}, {3.13758301734924f,27.2737407684326f,0.f}, {4.54657793045044f,27.9006500244141f,0.f}, {4.02841377258301f,28.0212306976318f,0.f}, {5.53865718841553f,28.5003795623779f,0.f}, {5.f,28.6602592468262f,0.f}, {6.59579277038574f,28.9761600494385f,0.f}, {6.03920221328735f,29.1821594238281f,0.f}, {7.70256900787354f,29.3210391998291f,0.f} },
                { {0,1,2}, {2,1,3}, {2,3,4}, {4,3,5}, {4,5,6}, {6,5,7}, {6,7,8}, {8,7,9}, {8,9,10}, {10,9,11}, {10,11,12}, {12,11,13}, {12,13,14}, {14,13,15}, {14,15,16}, {16,15,17}, {16,17,18}, {18,17,19}, {18,19,20}, {20,19,21}, {20,21,22}, {22,21,23}, {22,23,24}, {24,23,25}, {24,25,26}, {26,25,27}, {26,27,28}, {28,27,29}, {28,29,30}, {30,29,31}, {30,31,32}, {32,31,33}, {32,33,34}, {34,33,35}, {34,35,36}, {36,35,37}, {36,37,38}, {38,37,39}, {38,39,40}, {40,39,41}, {40,41,42}, {42,41,43}, {42,43,44}, {44,43,45}, {44,45,46}, {46,45,47}, {46,47,48}, {48,47,49}, {48,49,50}, {50,49,51}, {50,51,52}, {52,51,53}, {52,53,54}, {54,53,55}, {54,55,56}, {56,55,57}, {56,57,58}, {58,57,59}, {58,59,60}, {60,59,61}, {60,61,62}, {62,61,63}, {62,63,64}, {64,63,65}, {64,65,66}, {66,65,67}, {66,67,68}, {68,67,69}, {68,69,70}, {70,69,71}, {70,71,72}, {72,71,73}, {72,73,74}, {74,73,75}, {74,75,76}, {76,75,77}, {76,77,78}, {78,77,0}, {78,0,2}, {79,80,81}, {81,80,82}, {81,82,83}, {83,82,84}, {83,84,85}, {85,84,86}, {85,86,87}, {87,86,88}, {87,88,89}, {89,88,90}, {89,90,91}, {91,90,92}, {91,92,93}, {93,92,94}, {93,94,95}, {95,94,96}, {95,96,97}, {97,96,98}, {97,98,99}, {99,98,100}, {99,100,101}, {101,100,102}, {101,102,103}, {103,102,104}, {103,104,105}, {105,104,106}, {105,106,107}, {107,106,108}, {107,108,109}, {109,108,110}, {109,110,111}, {111,110,112}, {111,112,113}, {113,112,114}, {113,114,115}, {115,114,116}, {115,116,117}, {117,116,118}, {117,118,119}, {119,118,120}, {119,120,121}, {121,120,122}, {121,122,123}, {123,122,124}, {123,124,125}, {125,124,126}, {125,126,127}, {127,126,128}, {127,128,129}, {129,128,130}, {129,130,131}, {131,130,132}, {131,132,133}, {133,132,134}, {133,134,135}, {135,134,136}, {135,136,137}, {137,136,138}, {137,138,139}, {139,138,140}, {139,140,141}, {141,140,142}, {141,142,143}, {143,142,144}, {143,144,145}, {145,144,146}, {145,146,147}, {147,146,148}, {147,148,149}, {149,148,150}, {149,150,151}, {151,150,152}, {151,152,153}, {153,152,154}, {153,154,155}, {155,154,156}, {155,156,157}, {157,156,79}, {157,79,81}, {57,110,108}, {57,108,59}, {59,108,106}, {59,106,61}, {61,106,104}, {61,104,63}, {63,104,102}, {63,102,65}, {65,102,100}, {65,100,67}, {67,100,98}, {67,98,69}, {69,98,96}, {69,96,71}, {71,96,94}, {71,94,73}, {73,94,92}, {73,92,75}, {75,92,90}, {75,90,77}, {77,90,88}, {77,88,0}, {0,88,86}, {0,86,1}, {1,86,84}, {1,84,3}, {3,84,82}, {3,82,5}, {5,82,80}, {5,80,7}, {7,80,79}, {7,79,9}, {9,79,156}, {9,156,11}, {11,156,154}, {11,154,13}, {13,154,152}, {13,152,15}, {15,152,150}, {15,150,17}, {17,150,148}, {17,148,19}, {19,148,146}, {19,146,21}, {21,146,144}, {21,144,23}, {23,144,142}, {23,142,25}, {25,142,140}, {25,140,27}, {27,140,138}, {27,138,29}, {29,138,136}, {29,136,31}, {33,31,134}, {134,31,136}, {33,134,132}, {33,132,35}, {35,132,130}, {35,130,37}, {37,130,128}, {37,128,39}, {39,128,126}, {39,126,41}, {41,126,124}, {41,124,43}, {43,124,122}, {43,122,45}, {45,122,120}, {45,120,47}, {47,120,118}, {47,118,49}, {49,118,116}, {49,116,51}, {51,116,114}, {51,114,53}, {55,53,112}, {112,53,114}, {57,55,110}, {110,55,112}, {30,135,137}, {30,137,28}, {28,137,139}, {28,139,26}, {26,139,141}, {26,141,24}, {24,141,143}, {24,143,22}, {22,143,145}, {22,145,20}, {20,145,147}, {20,147,18}, {18,147,149}, {18,149,16}, {16,149,151}, {16,151,14}, {14,151,153}, {14,153,12}, {12,153,155}, {12,155,10}, {10,155,157}, {10,157,8}, {8,157,81}, {8,81,6}, {6,81,83}, {6,83,4}, {4,83,85}, {4,85,2}, {2,85,87}, {2,87,78}, {78,87,89}, {78,89,76}, {76,89,91}, {76,91,74}, {74,91,93}, {74,93,72}, {72,93,95}, {72,95,70}, {70,95,97}, {70,97,68}, {68,97,99}, {68,99,66}, {66,99,101}, {66,101,64}, {64,101,103}, {64,103,62}, {62,103,105}, {62,105,60}, {60,105,107}, {60,107,58}, {58,107,109}, {58,109,56}, {30,32,135}, {135,32,133}, {52,113,115}, {52,115,50}, {50,115,117}, {50,117,48}, {48,117,119}, {48,119,46}, {46,119,121}, {46,121,44}, {44,121,123}, {44,123,42}, {42,123,125}, {42,125,40}, {40,125,127}, {40,127,38}, {38,127,129}, {38,129,36}, {36,129,131}, {36,131,34}, {34,131,133}, {34,133,32}, {52,54,113}, {113,54,111}, {54,56,111}, {111,56,109} });
            break;
        case TestMesh::sloping_hole:
            mesh = TriangleMesh(
                { {-20.f,-20.f,-5.f}, {-20.f,-20.f,5.f}, {-20.f,20.f,-5.f}, {-20.f,20.f,5.f}, {20.f,-20.f,-5.f}, {20.f,-20.f,5.f}, {4.46294021606445f,7.43144989013672f,-5.f}, {20.f,20.f,-5.f}, {-19.1420993804932f,0.f,-5.f}, {-18.8330993652344f,-2.07911992073059f,-5.f}, {-17.9195003509521f,-4.06736993789673f,-5.f}, {-16.4412002563477f,-5.87785005569458f,-5.f}, {-14.4629001617432f,-7.43144989013672f,-5.f}, {-12.0711002349854f,-8.66024971008301f,-5.f}, {-9.37016010284424f,-9.51056003570557f,-5.f}, {-3.5217399597168f,-9.94521999359131f,-5.f}, {-6.4782600402832f,-9.94521999359131f,-5.f}, {-0.629840016365051f,-9.51056003570557f,-5.f}, {2.07106995582581f,-8.66024971008301f,-5.f}, {6.44122982025146f,-5.87785005569458f,-5.f}, {4.46294021606445f,-7.43144989013672f,-5.f}, {-12.0711002349854f,8.66024971008301f,-5.f}, {-9.37016010284424f,9.51056003570557f,-5.f}, {7.91947984695435f,-4.06736993789673f,-5.f}, {8.83310031890869f,-2.07911992073059f,-5.f}, {-6.4782600402832f,9.94521999359131f,-5.f}, {-0.629840016365051f,9.51056003570557f,-5.f}, {2.07106995582581f,8.66024971008301f,-5.f}, {9.14214038848877f,0.f,-5.f}, {8.83310031890869f,2.07911992073059f,-5.f}, {-3.5217399597168f,9.94521999359131f,-5.f}, {7.91947984695435f,4.06736993789673f,-5.f}, {6.44122982025146f,5.87785005569458f,-5.f}, {-14.4629001617432f,7.43144989013672f,-5.f}, {-16.4412002563477f,5.87785005569458f,-5.f}, {-17.9195003509521f,4.06736993789673f,-5.f}, {-18.8330993652344f,2.07911992073059f,-5.f}, {20.f,20.f,5.f}, {3.5217399597168f,-9.94521999359131f,5.f}, {-8.83310031890869f,-2.07911992073059f,5.f}, {-9.14214038848877f,0.f,5.f}, {-8.83310031890869f,2.07911992073059f,5.f}, {6.4782600402832f,-9.94521999359131f,5.f}, {-7.91947984695435f,4.06736993789673f,5.f}, {-6.44122982025146f,5.87785005569458f,5.f}, {-4.46294021606445f,7.43144989013672f,5.f}, {-2.07106995582581f,8.66024971008301f,5.f}, {0.629840016365051f,9.51056003570557f,5.f}, {12.0711002349854f,-8.66024971008301f,5.f}, {9.37016010284424f,-9.51056003570557f,5.f}, {3.5217399597168f,9.94521999359131f,5.f}, {6.4782600402832f,9.94521999359131f,5.f}, {9.37016010284424f,9.51056003570557f,5.f}, {12.0711002349854f,8.66024971008301f,5.f}, {14.4629001617432f,7.43144989013672f,5.f}, {16.4412002563477f,-5.87785005569458f,5.f}, {14.4629001617432f,-7.43144989013672f,5.f}, {16.4412002563477f,5.87785005569458f,5.f}, {17.9195003509521f,4.06736993789673f,5.f}, {18.8330993652344f,-2.07911992073059f,5.f}, {17.9195003509521f,-4.06736993789673f,5.f}, {18.8330993652344f,2.07911992073059f,5.f}, {19.1420993804932f,0.f,5.f}, {0.629840016365051f,-9.51056003570557f,5.f}, {-2.07106995582581f,-8.66024971008301f,5.f}, {-4.46294021606445f,-7.43144989013672f,5.f}, {-6.44122982025146f,-5.87785005569458f,5.f}, {-7.91947984695435f,-4.06736993789673f,5.f} },
                { {0,1,2}, {2,1,3}, {1,0,4}, {5,1,4}, {6,2,7}, {0,2,8}, {0,8,9}, {0,9,10}, {0,10,11}, {0,11,12}, {0,12,13}, {0,13,4}, {13,14,4}, {15,4,16}, {17,4,15}, {18,4,17}, {19,4,20}, {18,20,4}, {21,2,22}, {4,19,23}, {4,23,7}, {23,24,7}, {22,2,25}, {26,2,27}, {28,29,7}, {25,2,30}, {29,31,7}, {30,2,26}, {31,32,7}, {27,2,6}, {32,6,7}, {28,7,24}, {33,2,21}, {34,2,33}, {35,2,34}, {36,2,35}, {8,2,36}, {16,4,14}, {2,3,7}, {7,3,37}, {38,1,5}, {3,1,39}, {3,39,40}, {3,40,41}, {42,38,5}, {3,41,43}, {3,43,44}, {37,3,45}, {37,45,46}, {37,46,47}, {48,49,5}, {37,47,50}, {49,42,5}, {37,50,51}, {37,51,52}, {37,52,53}, {37,53,54}, {55,56,5}, {37,54,57}, {37,57,58}, {59,60,5}, {37,58,61}, {37,62,5}, {37,61,62}, {62,59,5}, {60,55,5}, {63,1,38}, {64,1,63}, {65,1,64}, {66,1,65}, {67,1,66}, {39,1,67}, {44,45,3}, {56,48,5}, {5,4,7}, {37,5,7}, {41,40,36}, {36,40,8}, {39,9,40}, {40,9,8}, {43,41,35}, {35,41,36}, {44,43,34}, {34,43,35}, {33,45,44}, {34,33,44}, {21,46,45}, {33,21,45}, {22,47,46}, {21,22,46}, {25,50,47}, {22,25,47}, {30,51,50}, {25,30,50}, {26,52,51}, {30,26,51}, {27,53,52}, {26,27,52}, {6,54,53}, {27,6,53}, {32,57,54}, {6,32,54}, {31,58,57}, {32,31,57}, {29,61,58}, {31,29,58}, {28,62,61}, {29,28,61}, {59,62,28}, {24,59,28}, {60,59,24}, {23,60,24}, {55,60,23}, {19,55,23}, {55,19,56}, {56,19,20}, {56,20,48}, {48,20,18}, {48,18,49}, {49,18,17}, {49,17,42}, {42,17,15}, {42,15,38}, {38,15,16}, {38,16,63}, {63,16,14}, {63,14,64}, {64,14,13}, {64,13,65}, {65,13,12}, {65,12,66}, {66,12,11}, {66,11,67}, {67,11,10}, {67,10,39}, {39,10,9} });
            break;
        case TestMesh::ipadstand:
            mesh = TriangleMesh(
                { {17.4344673156738f,float(-2.69879599481136e-16),9.5f}, {14.2814798355103f,10.f,9.5f}, {0.f,0.f,9.5f}, {31.7159481048584f,10.f,9.5f}, {62.2344741821289f,float(2.06667568800577e-16),20.f}, {31.7159481048584f,10.f,20.f}, {17.4344673156738f,float(-2.69879599481136e-16),20.f}, {62.2344741821289f,10.f,20.f}, {98.2079696655273f,10.f,0.f}, {98.2079696655273f,float(8.56525380796383e-16),10.f}, {98.2079696655273f,0.f,0.f}, {98.2079696655273f,10.f,20.f}, {98.2079696655273f,0.f,20.f}, {81.6609649658203f,float(-4.39753856997999e-16),10.f}, {90.0549850463867f,10.f,10.f}, {78.5079803466797f,10.f,10.f}, {93.2079696655273f,float(8.56525380796383e-16),10.f}, {14.2814798355103f,10.f,20.f}, {0.f,0.f,20.f}, {87.4344711303711f,float(2.81343962782118e-15),20.f}, {84.2814788818359f,10.f,20.f}, {0.f,10.f,20.f}, {0.f,0.f,0.f}, {0.f,10.f,0.f}, {62.2344741821289f,float(2.06667568800577e-16),30.f}, {66.9609756469727f,10.f,30.f}, {62.2344741821289f,10.f,30.f}, {70.1139602661133f,float(8.5525763717214e-16),30.f}, {67.7053375244141f,10.f,28.7107200622559f}, {71.6787109375f,float(1.24046736339707e-15),27.2897701263428f} },
                { {0,1,2}, {1,0,3}, {4,5,6}, {5,4,7}, {8,9,10}, {9,11,12}, {11,9,8}, {13,14,15}, {14,13,16}, {17,2,1}, {2,17,18}, {19,11,20}, {11,19,12}, {17,21,18}, {21,2,18}, {2,21,22}, {22,21,23}, {8,22,23}, {22,8,10}, {24,25,26}, {25,24,27}, {23,1,8}, {1,23,21}, {1,21,17}, {5,15,3}, {15,5,7}, {15,7,28}, {28,7,26}, {28,26,25}, {8,14,11}, {14,8,3}, {3,8,1}, {14,3,15}, {11,14,20}, {26,4,24}, {4,26,7}, {12,16,9}, {16,12,19}, {29,4,13}, {4,29,24}, {24,29,27}, {9,22,10}, {22,9,0}, {0,9,16}, {0,16,13}, {0,13,6}, {6,13,4}, {2,22,0}, {19,14,16}, {14,19,20}, {15,29,13}, {29,25,27}, {25,29,15}, {25,15,28}, {6,3,0}, {3,6,5} });
            break;
        case TestMesh::A:
            mesh = TriangleMesh(
                { {513.075988769531f,51.6074333190918f,36.0009002685547f}, {516.648803710938f,51.7324333190918f,36.0009002685547f}, {513.495178222656f,51.7324333190918f,36.0009002685547f}, {489.391204833984f,51.4824333190918f,24.0011005401611f}, {488.928588867188f,51.7324333190918f,24.0011005401611f}, {492.06201171875f,51.7324333190918f,24.0011005401611f}, {496.840393066406f,51.2324333190918f,24.0011005401611f}, {495.195404052734f,51.7324333190918f,24.0011005401611f}, {498.981994628906f,51.7324333190918f,24.0011005401611f}, {506.966613769531f,51.6074333190918f,24.0011005401611f}, {510.342010498047f,51.7324333190918f,24.0011005401611f}, {507.163818359375f,51.6074333190918f,24.0011005401611f}, {512.515380859375f,54.7190322875977f,36.0009002685547f}, {514.161987304688f,54.5058326721191f,36.0009002685547f}, {493.06201171875f,54.7190322875977f,36.0009002685547f}, {495.195404052734f,51.7324333190918f,36.0009002685547f}, {496.195404052734f,54.7190322875977f,36.0009002685547f}, {497.195404052734f,57.7058334350586f,36.0009002685547f}, {500.851989746094f,60.2658309936523f,36.0009002685547f}, {498.915405273438f,62.8258323669434f,36.0009002685547f}, {506.701995849609f,62.8258323669434f,36.0009002685547f}, {503.648590087891f,60.2658309936523f,36.0009002685547f}, {508.381805419922f,57.7058334350586f,36.0009002685547f}, {496.418792724609f,60.052433013916f,36.0009002685547f}, {506.515197753906f,72.2124328613281f,36.0009002685547f}, {502.808807373047f,74.5324325561523f,36.0009002685547f}, {503.781982421875f,71.6058349609375f,36.0009002685547f}, {515.358764648438f,55.4658317565918f,36.0009002685547f}, {499.375183105469f,76.9058380126953f,36.0009002685547f}, {501.168792724609f,78.0658340454102f,36.0009002685547f}, {504.568786621094f,78.0658340454102f,36.0009002685547f}, {506.32861328125f,81.599235534668f,36.0009002685547f}, {502.928588867188f,81.599235534668f,36.0009002685547f}, {499.528594970703f,81.599235534668f,36.0009002685547f}, {498.20361328125f,77.8658294677734f,36.0009002685547f}, {495.195404052734f,51.7324333190918f,30.0011005401611f}, {498.981994628906f,51.7324333190918f,27.0011005401611f}, {506.555206298828f,51.7324333190918f,33.0009002685547f}, {506.555206298828f,51.7324333190918f,36.0009002685547f}, {510.342010498047f,51.7324333190918f,36.0009002685547f}, {512.515380859375f,54.7190322875977f,24.0011005401611f}, {509.361999511719f,54.7190322875977f,24.0011005401611f}, {508.381805419922f,57.7058334350586f,24.0011005401611f}, {506.701995849609f,62.8258323669434f,24.0011005401611f}, {509.188812255859f,60.052433013916f,24.0011005401611f}, {493.06201171875f,54.7190322875977f,24.0011005401611f}, {503.648590087891f,60.2658309936523f,24.0011005401611f}, {500.851989746094f,60.2658309936523f,24.0011005401611f}, {498.915405273438f,62.8258323669434f,24.0011005401611f}, {502.808807373047f,62.8258323669434f,24.0011005401611f}, {491.425201416016f,54.5058326721191f,24.0011005401611f}, {506.421813964844f,76.9058380126953f,24.0011005401611f}, {502.808807373047f,74.5324325561523f,24.0011005401611f}, {504.568786621094f,78.0658340454102f,24.0011005401611f}, {506.32861328125f,81.599235534668f,24.0011005401611f}, {507.618804931641f,77.8658294677734f,24.0011005401611f}, {499.221801757812f,72.2124328613281f,24.0011005401611f}, {501.835388183594f,71.6058349609375f,24.0011005401611f}, {501.168792724609f,78.0658340454102f,24.0011005401611f}, {499.528594970703f,81.599235534668f,24.0011005401611f}, {502.048583984375f,79.8324356079102f,24.0011005401611f}, {490.253601074219f,55.4658317565918f,24.0011005401611f}, {488.928588867188f,51.7324333190918f,30.0011005401611f}, {488.928588867188f,51.7324333190918f,36.0009002685547f}, {490.253601074219f,55.4658317565918f,31.5009002685547f}, {498.20361328125f,77.8658294677734f,34.5009002685547f}, {508.381805419922f,57.7058334350586f,30.0011005401611f}, {505.585388183594f,57.7058334350586f,27.0011005401611f}, {502.788818359375f,57.7058334350586f,36.0009002685547f}, {499.992004394531f,57.7058334350586f,33.0009002685547f}, {509.851989746094f,53.2258338928223f,33.0009002685547f}, {509.361999511719f,54.7190322875977f,36.0009002685547f}, {508.871795654297f,56.2124328613281f,27.0011005401611f}, {496.695404052734f,56.2124328613281f,33.0009002685547f}, {495.695404052734f,53.2258338928223f,27.0011005401611f}, {506.32861328125f,81.599235534668f,30.0011005401611f}, {507.618804931641f,77.8658294677734f,25.5011005401611f}, {515.358764648438f,55.4658317565918f,34.5009002685547f}, {501.228607177734f,81.599235534668f,33.0009002685547f}, {504.628601074219f,81.599235534668f,27.0011005401611f}, {503.781982421875f,71.6058349609375f,33.0009002685547f}, {502.808807373047f,74.5324325561523f,30.0011005401611f}, {498.915405273438f,62.8258323669434f,30.0011005401611f}, {500.861999511719f,62.8258323669434f,27.0011005401611f}, {502.808807373047f,62.8258323669434f,36.0009002685547f}, {504.755187988281f,62.8258323669434f,33.0009002685547f}, {501.835388183594f,71.6058349609375f,33.0009002685547f}, {499.888793945312f,65.7524337768555f,33.0009002685547f}, {499.888793945312f,65.7524337768555f,36.0009002685547f}, {513.128601074219f,51.4824333190918f,36.0009002685547f}, {513.075988769531f,51.6074333190918f,24.0011005401611f}, {516.648803710938f,51.7324333190918f,24.0011005401611f}, {513.128601074219f,51.4824333190918f,24.0011005401611f}, {513.495178222656f,51.7324333190918f,24.0011005401611f}, {506.966613769531f,51.6074333190918f,36.0009002685547f}, {507.163818359375f,51.6074333190918f,36.0009002685547f}, {490.337799072266f,51.4824333190918f,24.0011005401611f}, {489.391204833984f,51.4824333190918f,36.0009002685547f}, {492.06201171875f,51.7324333190918f,36.0009002685547f}, {490.337799072266f,51.4824333190918f,36.0009002685547f}, {513.233764648438f,51.2324333190918f,24.0011005401611f}, {513.233764648438f,51.2324333190918f,36.0009002685547f}, {504.773803710938f,51.4824333190918f,36.0009002685547f}, {504.773803710938f,51.4824333190918f,24.0011005401611f}, {489.266998291016f,51.2324333190918f,24.0011005401611f}, {489.266998291016f,51.2324333190918f,36.0009002685547f}, {490.253601074219f,55.4658317565918f,25.5011005401611f}, {499.528594970703f,81.599235534668f,30.0011005401611f}, {498.20361328125f,77.8658294677734f,31.5009002685547f}, {515.358764648438f,55.4658317565918f,28.5011005401611f}, {515.358764648438f,55.4658317565918f,25.5011005401611f}, {495.246795654297f,61.0124320983887f,36.0009002685547f}, {490.253601074219f,55.4658317565918f,34.5009002685547f}, {490.253601074219f,55.4658317565918f,36.0009002685547f}, {494.228607177734f,66.6658325195312f,24.0011005401611f}, {499.068786621094f,67.5192337036133f,24.0011005401611f}, {498.20361328125f,77.8658294677734f,25.5011005401611f}, {498.20361328125f,77.8658294677734f,24.0011005401611f}, {506.608795166016f,67.5192337036133f,36.0009002685547f}, {509.09521484375f,64.7458343505859f,36.0009002685547f}, {507.618804931641f,77.8658294677734f,34.5009002685547f}, {507.618804931641f,77.8658294677734f,36.0009002685547f}, {510.385406494141f,61.0124320983887f,24.0011005401611f}, {515.358764648438f,55.4658317565918f,24.0011005401611f}, {489.32861328125f,47.7324333190918f,31.5009002685547f}, {492.95361328125f,47.7324333190918f,33.5634994506836f}, {489.32861328125f,47.7324333190918f,34.5009002685547f}, {489.32861328125f,47.7324333190918f,28.5011005401611f}, {489.32861328125f,47.7324333190918f,25.5011005401611f}, {492.95361328125f,47.7324333190918f,26.4385013580322f}, {492.95361328125f,47.7324333190918f,30.5635013580322f}, {492.95361328125f,47.7324333190918f,32.0634994506836f}, {492.95361328125f,47.7324333190918f,31.3135013580322f}, {492.95361328125f,47.7324333190918f,35.4384994506836f}, {489.32861328125f,47.7324333190918f,36.0009002685547f}, {492.95361328125f,47.7324333190918f,34.3134994506836f}, {492.95361328125f,47.7324333190918f,34.6884994506836f}, {492.95361328125f,47.7324333190918f,27.9385013580322f}, {492.95361328125f,47.7324333190918f,28.6885013580322f}, {492.95361328125f,47.7324333190918f,29.0635013580322f}, {489.32861328125f,47.7324333190918f,24.0011005401611f}, {492.95361328125f,47.7324333190918f,24.5635013580322f}, {492.95361328125f,47.7324333190918f,25.6885013580322f}, {492.95361328125f,47.7324333190918f,25.3135013580322f}, {492.95361328125f,47.7324333190918f,24.1885013580322f}, {492.95361328125f,47.7324333190918f,24.0011005401611f}, {513.443786621094f,50.7324333190918f,24.0011005401611f}, {492.95361328125f,47.7324333190918f,35.8134994506836f}, {492.95361328125f,47.7324333190918f,36.0009002685547f}, {513.443786621094f,50.7324333190918f,36.0009002685547f}, {506.350402832031f,51.4824333190918f,36.0009002685547f}, {506.350402832031f,51.4824333190918f,24.0011005401611f}, {492.743804931641f,48.2324333190918f,24.0011005401611f}, {492.638793945312f,48.4824333190918f,24.0011005401611f}, {492.743804931641f,48.2324333190918f,36.0009002685547f}, {492.638793945312f,48.4824333190918f,36.0009002685547f}, {490.089599609375f,50.9824333190918f,36.0009002685547f}, {490.089599609375f,50.9824333190918f,24.0011005401611f}, {510.342010498047f,51.7324333190918f,30.0011005401611f}, {499.068786621094f,67.5192337036133f,36.0009002685547f}, {494.228607177734f,66.6658325195312f,36.0009002685547f}, {499.375183105469f,76.9058380126953f,24.0011005401611f}, {506.421813964844f,76.9058380126953f,36.0009002685547f}, {506.608795166016f,67.5192337036133f,24.0011005401611f}, {505.728607177734f,65.7524337768555f,24.0011005401611f}, {509.09521484375f,64.7458343505859f,24.0011005401611f}, {506.701995849609f,62.8258323669434f,30.0011005401611f}, {505.728607177734f,65.7524337768555f,27.0011005401611f}, {501.835388183594f,71.6058349609375f,27.0011005401611f}, {499.888793945312f,65.7524337768555f,27.0011005401611f}, {494.228607177734f,66.6658325195312f,30.0011005401611f}, {495.553588867188f,70.3992309570312f,28.5011005401611f}, {492.903594970703f,62.9324340820312f,28.5011005401611f}, {495.553588867188f,70.3992309570312f,31.5009002685547f}, {492.903594970703f,62.9324340820312f,31.5009002685547f}, {511.488800048828f,66.6658325195312f,24.0011005401611f}, {511.488800048828f,66.6658325195312f,30.0011005401611f}, {512.778564453125f,62.9324340820312f,28.5011005401611f}, {515.358764648438f,55.4658317565918f,31.5009002685547f}, {507.618804931641f,77.8658294677734f,31.5009002685547f}, {510.198791503906f,70.3992309570312f,28.5011005401611f}, {511.488800048828f,66.6658325195312f,36.0009002685547f}, {512.778564453125f,62.9324340820312f,31.5009002685547f}, {510.198791503906f,70.3992309570312f,31.5009002685547f}, {502.788818359375f,57.7058334350586f,24.0011005401611f}, {497.195404052734f,57.7058334350586f,30.0011005401611f}, {492.903594970703f,62.9324340820312f,34.5009002685547f}, {492.903594970703f,62.9324340820312f,36.0009002685547f}, {495.553588867188f,70.3992309570312f,24.0011005401611f}, {496.725189208984f,69.4392318725586f,24.0011005401611f}, {495.553588867188f,70.3992309570312f,25.5011005401611f}, {495.246795654297f,61.0124320983887f,24.0011005401611f}, {492.903594970703f,62.9324340820312f,25.5011005401611f}, {492.903594970703f,62.9324340820312f,24.0011005401611f}, {495.553588867188f,70.3992309570312f,36.0009002685547f}, {496.725189208984f,69.4392318725586f,36.0009002685547f}, {495.553588867188f,70.3992309570312f,34.5009002685547f}, {510.198791503906f,70.3992309570312f,36.0009002685547f}, {509.002014160156f,69.4392318725586f,36.0009002685547f}, {510.198791503906f,70.3992309570312f,34.5009002685547f}, {512.778564453125f,62.9324340820312f,25.5011005401611f}, {512.778564453125f,62.9324340820312f,24.0011005401611f}, {510.198791503906f,70.3992309570312f,24.0011005401611f}, {509.002014160156f,69.4392318725586f,24.0011005401611f}, {510.198791503906f,70.3992309570312f,25.5011005401611f}, {510.385406494141f,61.0124320983887f,36.0009002685547f}, {512.778564453125f,62.9324340820312f,34.5009002685547f}, {512.778564453125f,62.9324340820312f,36.0009002685547f}, {496.840393066406f,51.2324333190918f,36.0009002685547f}, {498.981994628906f,51.7324333190918f,36.0009002685547f}, {498.981994628906f,51.7324333190918f,33.0009002685547f}, {506.555206298828f,51.7324333190918f,24.0011005401611f}, {506.555206298828f,51.7324333190918f,27.0011005401611f}, {503.82861328125f,47.7324333190918f,30.7509002685547f}, {507.45361328125f,47.7324333190918f,32.8134994506836f}, {503.82861328125f,47.7324333190918f,33.7509002685547f}, {503.82861328125f,47.7324333190918f,29.2511005401611f}, {503.82861328125f,47.7324333190918f,26.2511005401611f}, {507.45361328125f,47.7324333190918f,27.1885013580322f}, {493.921813964844f,57.2792320251465f,36.0009002685547f}, {491.425201416016f,54.5058326721191f,36.0009002685547f}, {497.195404052734f,57.7058334350586f,24.0011005401611f}, {496.418792724609f,60.052433013916f,24.0011005401611f}, {509.188812255859f,60.052433013916f,36.0009002685547f}, {511.675415039062f,57.2792320251465f,24.0011005401611f}, {514.161987304688f,54.5058326721191f,24.0011005401611f}, {507.45361328125f,47.7324333190918f,34.3134994506836f}, {503.82861328125f,47.7324333190918f,35.2509002685547f}, {507.45361328125f,47.7324333190918f,25.6885013580322f}, {503.82861328125f,47.7324333190918f,24.7511005401611f}, {500.20361328125f,47.7324333190918f,31.6885013580322f}, {500.20361328125f,47.7324333190918f,28.3135013580322f}, {500.20361328125f,47.7324333190918f,30.1885013580322f}, {507.45361328125f,47.7324333190918f,29.8135013580322f}, {507.45361328125f,47.7324333190918f,31.3135013580322f}, {507.45361328125f,47.7324333190918f,30.5635013580322f}, {503.82861328125f,47.7324333190918f,36.0009002685547f}, {507.45361328125f,47.7324333190918f,35.4384994506836f}, {507.45361328125f,47.7324333190918f,35.0634994506836f}, {507.45361328125f,47.7324333190918f,28.6885013580322f}, {507.45361328125f,47.7324333190918f,29.4385013580322f}, {503.82861328125f,47.7324333190918f,24.0011005401611f}, {507.45361328125f,47.7324333190918f,24.5635013580322f}, {507.45361328125f,47.7324333190918f,24.9385013580322f}, {500.20361328125f,47.7324333190918f,34.6884994506836f}, {500.20361328125f,47.7324333190918f,33.1884994506836f}, {500.20361328125f,47.7324333190918f,33.9384994506836f}, {500.20361328125f,47.7324333190918f,25.3135013580322f}, {500.20361328125f,47.7324333190918f,26.8135013580322f}, {500.20361328125f,47.7324333190918f,26.0635013580322f}, {500.20361328125f,47.7324333190918f,30.9385013580322f}, {500.20361328125f,47.7324333190918f,35.0634994506836f}, {500.20361328125f,47.7324333190918f,35.4384994506836f}, {500.20361328125f,47.7324333190918f,29.0635013580322f}, {500.20361328125f,47.7324333190918f,29.4385013580322f}, {500.20361328125f,47.7324333190918f,24.9385013580322f}, {500.20361328125f,47.7324333190918f,24.5635013580322f}, {507.45361328125f,47.7324333190918f,24.1885013580322f}, {507.45361328125f,47.7324333190918f,24.0011005401611f}, {513.86376953125f,49.7324333190918f,24.0011005401611f}, {507.45361328125f,47.7324333190918f,35.8134994506836f}, {507.45361328125f,47.7324333190918f,36.0009002685547f}, {513.86376953125f,49.7324333190918f,36.0009002685547f}, {500.20361328125f,47.7324333190918f,24.1885013580322f}, {500.20361328125f,47.7324333190918f,24.0011005401611f}, {502.988800048828f,49.7324333190918f,24.0011005401611f}, {500.20361328125f,47.7324333190918f,35.8134994506836f}, {500.20361328125f,47.7324333190918f,36.0009002685547f}, {502.988800048828f,49.7324333190918f,36.0009002685547f}, {504.755187988281f,62.8258323669434f,27.0011005401611f}, {499.205383300781f,51.2324333190918f,36.0009002685547f}, {498.786193847656f,51.1074333190918f,36.0009002685547f}, {502.358795166016f,51.2324333190918f,36.0009002685547f}, {499.205383300781f,51.2324333190918f,24.0011005401611f}, {502.358795166016f,51.2324333190918f,24.0011005401611f}, {498.786193847656f,51.1074333190918f,24.0011005401611f}, {502.568786621094f,50.7324333190918f,24.0011005401611f}, {505.931213378906f,51.3574333190918f,24.0011005401611f}, {509.503601074219f,51.4824333190918f,24.0011005401611f}, {502.568786621094f,50.7324333190918f,36.0009002685547f}, {505.931213378906f,51.3574333190918f,36.0009002685547f}, {509.503601074219f,51.4824333190918f,36.0009002685547f}, {499.048583984375f,50.4824333190918f,36.0009002685547f}, {492.428588867188f,48.9824333190918f,36.0009002685547f}, {499.048583984375f,50.4824333190918f,24.0011005401611f}, {492.428588867188f,48.9824333190918f,24.0011005401611f}, {506.088806152344f,50.9824333190918f,24.0011005401611f}, {506.036010742188f,51.1074333190918f,24.0011005401611f}, {506.088806152344f,50.9824333190918f,36.0009002685547f}, {506.036010742188f,51.1074333190918f,36.0009002685547f}, {498.891204833984f,50.8574333190918f,36.0009002685547f}, {498.943786621094f,50.7324333190918f,36.0009002685547f}, {498.891204833984f,50.8574333190918f,24.0011005401611f}, {498.943786621094f,50.7324333190918f,24.0011005401611f}, {499.573608398438f,49.2324333190918f,24.0011005401611f}, {499.783813476562f,48.7324333190918f,24.0011005401611f}, {499.573608398438f,49.2324333190918f,36.0009002685547f}, {499.783813476562f,48.7324333190918f,36.0009002685547f}, {506.403594970703f,50.2324333190918f,24.0011005401611f}, {506.298797607422f,50.4824333190918f,24.0011005401611f}, {506.403594970703f,50.2324333190918f,36.0009002685547f}, {506.298797607422f,50.4824333190918f,36.0009002685547f}, {501.228607177734f,81.599235534668f,27.0011005401611f}, {502.928588867188f,81.599235534668f,24.0011005401611f}, {499.2587890625f,49.9824333190918f,36.0009002685547f}, {499.363800048828f,49.7324333190918f,36.0009002685547f}, {499.2587890625f,49.9824333190918f,24.0011005401611f}, {499.363800048828f,49.7324333190918f,24.0011005401611f}, {496.695404052734f,56.2124328613281f,27.0011005401611f}, {496.195404052734f,54.7190322875977f,24.0011005401611f}, {509.851989746094f,53.2258338928223f,27.0011005401611f}, {493.464782714844f,51.1074333190918f,36.0009002685547f}, {493.464782714844f,51.1074333190918f,24.0011005401611f}, {502.768798828125f,51.7324333190918f,24.0011005401611f}, {500.215789794922f,51.3574333190918f,24.0011005401611f}, {497.628601074219f,51.2324333190918f,24.0011005401611f}, {502.768798828125f,51.7324333190918f,36.0009002685547f}, {500.215789794922f,51.3574333190918f,36.0009002685547f}, {497.628601074219f,51.2324333190918f,36.0009002685547f}, {507.033813476562f,48.7324333190918f,24.0011005401611f}, {506.823791503906f,49.2324333190918f,24.0011005401611f}, {507.033813476562f,48.7324333190918f,36.0009002685547f}, {506.823791503906f,49.2324333190918f,36.0009002685547f}, {494.4501953125f,51.1074333190918f,24.0011005401611f}, {494.4501953125f,51.1074333190918f,36.0009002685547f}, {500.807006835938f,51.3574333190918f,36.0009002685547f}, {503.591186523438f,51.4824333190918f,36.0009002685547f}, {503.591186523438f,51.4824333190918f,24.0011005401611f}, {500.807006835938f,51.3574333190918f,24.0011005401611f}, {505.728607177734f,65.7524337768555f,36.0009002685547f}, {505.728607177734f,65.7524337768555f,33.0009002685547f}, {499.221801757812f,72.2124328613281f,36.0009002685547f}, {501.835388183594f,71.6058349609375f,36.0009002685547f}, {506.515197753906f,72.2124328613281f,24.0011005401611f}, {503.781982421875f,71.6058349609375f,24.0011005401611f}, {503.781982421875f,71.6058349609375f,27.0011005401611f}, {499.888793945312f,65.7524337768555f,24.0011005401611f}, {495.695404052734f,53.2258338928223f,33.0009002685547f}, {516.648803710938f,51.7324333190918f,30.0011005401611f}, {498.20361328125f,77.8658294677734f,28.5011005401611f}, {505.585388183594f,57.7058334350586f,33.0009002685547f}, {508.871795654297f,56.2124328613281f,33.0009002685547f}, {499.992004394531f,57.7058334350586f,27.0011005401611f}, {504.628601074219f,81.599235534668f,33.0009002685547f}, {500.861999511719f,62.8258323669434f,33.0009002685547f}, {496.878601074219f,74.1324310302734f,27.0011005401611f}, {496.878601074219f,74.1324310302734f,33.0009002685547f}, {491.57861328125f,59.199031829834f,27.0011005401611f}, {490.253601074219f,55.4658317565918f,28.5011005401611f}, {491.57861328125f,59.199031829834f,33.0009002685547f}, {514.068786621094f,59.199031829834f,27.0011005401611f}, {514.068786621094f,59.199031829834f,33.0009002685547f}, {508.908813476562f,74.1324310302734f,27.0011005401611f}, {507.618804931641f,77.8658294677734f,28.5011005401611f}, {508.908813476562f,74.1324310302734f,33.0009002685547f}, {491.271789550781f,50.9824333190918f,36.0009002685547f}, {490.877807617188f,50.9824333190918f,36.0009002685547f}, {491.271789550781f,50.9824333190918f,24.0011005401611f}, {490.877807617188f,50.9824333190918f,24.0011005401611f}, {495.213806152344f,50.9824333190918f,36.0009002685547f}, {493.636993408203f,50.9824333190918f,36.0009002685547f}, {495.213806152344f,50.9824333190918f,24.0011005401611f}, {493.636993408203f,50.9824333190918f,24.0011005401611f}, {503.985412597656f,51.4824333190918f,36.0009002685547f}, {503.985412597656f,51.4824333190918f,24.0011005401611f}, {511.675415039062f,57.2792320251465f,36.0009002685547f}, {493.921813964844f,57.2792320251465f,24.0011005401611f}, {502.768798828125f,51.7324333190918f,30.0011005401611f}, {506.555206298828f,51.7324333190918f,30.0011005401611f}, {498.981994628906f,51.7324333190918f,30.0011005401611f}, {492.848815917969f,50.9824333190918f,24.0011005401611f}, {492.848815917969f,50.9824333190918f,36.0009002685547f}, {500.861999511719f,68.6792297363281f,36.0009002685547f}, {500.861999511719f,68.6792297363281f,24.0011005401611f}, {496.878601074219f,74.1324310302734f,24.0011005401611f}, {496.878601074219f,74.1324310302734f,36.0009002685547f}, {504.755187988281f,68.6792297363281f,24.0011005401611f}, {504.755187988281f,68.6792297363281f,36.0009002685547f}, {508.908813476562f,74.1324310302734f,36.0009002685547f}, {508.908813476562f,74.1324310302734f,24.0011005401611f}, {505.728607177734f,65.7524337768555f,30.0011005401611f}, {504.755187988281f,68.6792297363281f,30.0011005401611f}, {503.781982421875f,71.6058349609375f,30.0011005401611f}, {500.861999511719f,68.6792297363281f,30.0011005401611f}, {499.888793945312f,65.7524337768555f,30.0011005401611f}, {501.835388183594f,71.6058349609375f,30.0011005401611f}, {491.57861328125f,59.199031829834f,24.0011005401611f}, {491.57861328125f,59.199031829834f,36.0009002685547f}, {514.068786621094f,59.199031829834f,36.0009002685547f}, {514.068786621094f,59.199031829834f,24.0011005401611f}, {511.07861328125f,47.7324333190918f,34.8759002685547f}, {511.07861328125f,47.7324333190918f,31.8759002685547f}, {514.70361328125f,47.7324333190918f,33.9384994506836f}, {511.07861328125f,47.7324333190918f,25.1261005401611f}, {514.70361328125f,47.7324333190918f,26.0635013580322f}, {511.07861328125f,47.7324333190918f,28.1261005401611f}, {502.788818359375f,57.7058334350586f,30.0011005401611f}, {502.048583984375f,79.8324356079102f,36.0009002685547f}, {514.70361328125f,47.7324333190918f,30.9385013580322f}, {511.07861328125f,47.7324333190918f,30.3759002685547f}, {514.70361328125f,47.7324333190918f,29.0635013580322f}, {511.07861328125f,47.7324333190918f,29.6261005401611f}, {496.57861328125f,47.7324333190918f,31.1259002685547f}, {496.57861328125f,47.7324333190918f,32.6259002685547f}, {496.57861328125f,47.7324333190918f,34.1259002685547f}, {496.57861328125f,47.7324333190918f,28.8761005401611f}, {496.57861328125f,47.7324333190918f,27.3761005401611f}, {496.57861328125f,47.7324333190918f,25.8761005401611f}, {496.57861328125f,47.7324333190918f,29.6261005401611f}, {514.70361328125f,47.7324333190918f,35.4384994506836f}, {511.07861328125f,47.7324333190918f,35.6259002685547f}, {514.70361328125f,47.7324333190918f,24.5635013580322f}, {511.07861328125f,47.7324333190918f,24.3761005401611f}, {496.57861328125f,47.7324333190918f,34.8759002685547f}, {496.57861328125f,47.7324333190918f,25.1261005401611f}, {496.57861328125f,47.7324333190918f,35.6259002685547f}, {496.57861328125f,47.7324333190918f,24.3761005401611f}, {511.07861328125f,47.7324333190918f,36.0009002685547f}, {511.07861328125f,47.7324333190918f,24.0011005401611f}, {514.70361328125f,47.7324333190918f,30.1885013580322f}, {514.70361328125f,47.7324333190918f,35.8134994506836f}, {514.70361328125f,47.7324333190918f,29.8135013580322f}, {514.70361328125f,47.7324333190918f,24.1885013580322f}, {496.57861328125f,47.7324333190918f,36.0009002685547f}, {496.57861328125f,47.7324333190918f,24.0011005401611f}, {510.238800048828f,49.7324333190918f,24.0011005401611f}, {510.238800048828f,49.7324333190918f,36.0009002685547f}, {514.70361328125f,47.7324333190918f,24.0011005401611f}, {514.70361328125f,47.7324333190918f,36.0009002685547f}, {496.158813476562f,48.7324333190918f,36.0009002685547f}, {496.158813476562f,48.7324333190918f,24.0011005401611f}, {502.808807373047f,62.8258323669434f,30.0011005401611f}, {509.608795166016f,51.2324333190918f,24.0011005401611f}, {509.608795166016f,51.2324333190918f,36.0009002685547f}, {491.641204833984f,50.8574333190918f,24.0011005401611f}, {495.423797607422f,50.4824333190918f,36.0009002685547f}, {495.423797607422f,50.4824333190918f,24.0011005401611f}, {491.641204833984f,50.8574333190918f,36.0009002685547f}, {495.528594970703f,50.2324333190918f,24.0011005401611f}, {492.0087890625f,49.9824333190918f,24.0011005401611f}, {509.818786621094f,50.7324333190918f,24.0011005401611f}, {495.948608398438f,49.2324333190918f,36.0009002685547f}, {495.528594970703f,50.2324333190918f,36.0009002685547f}, {495.948608398438f,49.2324333190918f,24.0011005401611f}, {509.818786621094f,50.7324333190918f,36.0009002685547f}, {492.0087890625f,49.9824333190918f,36.0009002685547f}, {491.956207275391f,50.1074333190918f,24.0011005401611f}, {491.956207275391f,50.1074333190918f,36.0009002685547f}, {502.928588867188f,81.599235534668f,30.0011005401611f}, {491.851013183594f,50.3574333190918f,36.0009002685547f}, {491.851013183594f,50.3574333190918f,24.0011005401611f}, {496.195404052734f,54.7190322875977f,30.0011005401611f}, {509.361999511719f,54.7190322875977f,30.0011005401611f}, {488.632598876953f,51.7256317138672f,30.0011005401611f}, {488.632598876953f,51.7256317138672f,29.5091018676758f}, {488.632598876953f,51.7188339233398f,24.0011005401611f}, {488.632598876953f,51.7256317138672f,27.4929008483887f}, {488.632598876953f,51.7324333190918f,30.0011005401611f}, {488.632598876953f,51.7324333190918f,29.0175018310547f}, {488.632598876953f,51.7324333190918f,24.9847011566162f}, {488.632598876953f,51.7324333190918f,24.0011005401611f}, {488.632598876953f,51.7188339233398f,30.0011005401611f}, {488.632598876953f,51.7176322937012f,24.0011005401611f}, {488.632598876953f,51.7182312011719f,30.0011005401611f}, {488.632598876953f,51.7176322937012f,30.0011005401611f}, {488.632598876953f,51.715030670166f,24.0011005401611f}, {488.632598876953f,51.7162322998047f,30.0011005401611f}, {488.632598876953f,50.761833190918f,24.0011005401611f}, {488.632598876953f,50.7578315734863f,24.0011005401611f}, {488.632598876953f,50.7598342895508f,30.0011005401611f}, {488.632598876953f,50.7522315979004f,24.0011005401611f}, {488.632598876953f,49.7838325500488f,24.0011005401611f}, {488.632598876953f,50.2680320739746f,30.0011005401611f}, {488.632598876953f,51.7046318054199f,24.0011005401611f}, {488.632598876953f,51.709831237793f,30.0011005401611f}, {488.632598876953f,50.9120330810547f,24.0011005401611f}, {488.632598876953f,50.8882331848145f,24.0011005401611f}, {488.632598876953f,50.9002304077148f,30.0011005401611f}, {488.632598876953f,47.7324333190918f,24.0370998382568f}, {488.632598876953f,48.5612335205078f,30.0011005401611f}, {488.632598876953f,47.7324333190918f,24.0011005401611f}, {488.632598876953f,47.7324333190918f,24.1091003417969f}, {488.632598876953f,48.5612335205078f,30.0189018249512f}, {488.632598876953f,47.7324333190918f,25.3211002349854f}, {488.632598876953f,48.5612335205078f,30.0551013946533f}, {488.632598876953f,47.7324333190918f,25.4651012420654f}, {488.632598876953f,48.5612335205078f,30.6609001159668f}, {488.632598876953f,47.7324333190918f,25.5371017456055f}, {488.632598876953f,48.5612335205078f,30.7329006195068f}, {488.632598876953f,47.7324333190918f,25.6091003417969f}, {488.632598876953f,48.5612335205078f,30.7689018249512f}, {488.632598876953f,47.7324333190918f,25.8971004486084f}, {488.632598876953f,48.5612335205078f,30.8051013946533f}, {488.632598876953f,47.7324333190918f,28.321102142334f}, {488.632598876953f,48.5612335205078f,30.9491004943848f}, {488.632598876953f,47.7324333190918f,28.4651012420654f}, {488.632598876953f,48.5612335205078f,32.1609001159668f}, {488.632598876953f,47.7324333190918f,28.5371017456055f}, {488.632598876953f,48.5612335205078f,32.2329025268555f}, {488.632598876953f,47.7324333190918f,28.6811008453369f}, {488.632598876953f,48.5612335205078f,32.2689018249512f}, {488.632598876953f,47.7324333190918f,31.1049003601074f}, {488.632598876953f,48.5612335205078f,32.3411026000977f}, {488.632598876953f,47.7324333190918f,31.3929004669189f}, {488.632598876953f,49.3900299072266f,36.0009002685547f}, {488.632598876953f,47.7324333190918f,31.536901473999f}, {488.632598876953f,47.7324333190918f,31.6809005737305f}, {488.632598876953f,47.7324333190918f,34.1049003601074f}, {488.632598876953f,47.7324333190918f,34.3929023742676f}, {488.632598876953f,47.7324333190918f,34.464900970459f}, {488.632598876953f,47.7324333190918f,34.5369033813477f}, {488.632598876953f,47.7324333190918f,34.6809005737305f}, {488.632598876953f,47.7324333190918f,35.8929023742676f}, {488.632598876953f,47.7324333190918f,35.964900970459f}, {488.632598876953f,47.7324333190918f,36.0009002685547f}, {488.632598876953f,50.8816299438477f,24.0011005401611f}, {488.632598876953f,50.8850326538086f,30.0011005401611f}, {488.632598876953f,49.7480316162109f,24.0011005401611f}, {488.632598876953f,49.7426300048828f,24.0011005401611f}, {488.632598876953f,49.745231628418f,30.0011005401611f}, {488.632598876953f,49.7592315673828f,24.0011005401611f}, {488.632598876953f,49.7536315917969f,30.0011005401611f}, {488.632598876953f,49.3900299072266f,24.0011005401611f}, {488.632598876953f,49.5664329528809f,30.0011005401611f}, {488.632598876953f,50.8786315917969f,24.0011005401611f}, {488.632598876953f,50.7764320373535f,24.0011005401611f}, {488.632598876953f,50.8274307250977f,30.0011005401611f}, {488.632598876953f,50.7550315856934f,30.0011005401611f}, {488.632598876953f,50.7692337036133f,30.0011005401611f}, {488.632598876953f,50.9284324645996f,24.0011005401611f}, {488.632598876953f,50.9202308654785f,30.0011005401611f}, {488.632598876953f,51.1788330078125f,24.0011005401611f}, {488.632598876953f,51.139232635498f,24.0011005401611f}, {488.632598876953f,51.1590309143066f,30.0011005401611f}, {488.632598876953f,51.2324333190918f,24.0011005401611f}, {488.632598876953f,51.2056312561035f,30.0011005401611f}, {488.632598876953f,51.4340324401855f,24.0011005401611f}, {488.632598876953f,51.3946304321289f,24.0011005401611f}, {488.632598876953f,51.4142303466797f,30.0011005401611f}, {488.632598876953f,51.4498329162598f,24.0011005401611f}, {488.632598876953f,51.5772323608398f,30.0011005401611f}, {488.632598876953f,51.4418334960938f,30.0011005401611f}, {488.632598876953f,51.3136329650879f,30.0011005401611f}, {488.632598876953f,49.7714309692383f,30.0011005401611f}, {488.632598876953f,51.0338325500488f,30.0011005401611f}, {488.632598876953f,50.8816299438477f,30.0011005401611f}, {488.632598876953f,50.8800315856934f,30.0011005401611f}, {488.632598876953f,51.7188339233398f,36.0009002685547f}, {488.632598876953f,51.7176322937012f,36.0009002685547f}, {488.632598876953f,49.3900299072266f,30.0011005401611f}, {488.632598876953f,50.7522315979004f,30.0011005401611f}, {488.632598876953f,50.7522315979004f,36.0009002685547f}, {488.632598876953f,49.7426300048828f,30.0011005401611f}, {488.632598876953f,49.7426300048828f,36.0009002685547f}, {488.632598876953f,49.7480316162109f,30.0011005401611f}, {488.632598876953f,49.7480316162109f,36.0009002685547f}, {488.632598876953f,51.715030670166f,30.0011005401611f}, {488.632598876953f,51.715030670166f,36.0009002685547f}, {488.632598876953f,50.7578315734863f,30.0011005401611f}, {488.632598876953f,50.7578315734863f,36.0009002685547f}, {488.632598876953f,50.761833190918f,30.0011005401611f}, {488.632598876953f,50.761833190918f,36.0009002685547f}, {488.632598876953f,50.8882331848145f,30.0011005401611f}, {488.632598876953f,50.8882331848145f,36.0009002685547f}, {488.632598876953f,49.7592315673828f,30.0011005401611f}, {488.632598876953f,49.7592315673828f,36.0009002685547f}, {488.632598876953f,51.1788330078125f,30.0011005401611f}, {488.632598876953f,51.1788330078125f,36.0009002685547f}, {488.632598876953f,50.9120330810547f,30.0011005401611f}, {488.632598876953f,50.9120330810547f,36.0009002685547f}, {488.632598876953f,51.4498329162598f,30.0011005401611f}, {488.632598876953f,51.4498329162598f,36.0009002685547f}, {488.632598876953f,51.7046318054199f,30.0011005401611f}, {488.632598876953f,51.7046318054199f,36.0009002685547f}, {488.632598876953f,51.2324333190918f,30.0011005401611f}, {488.632598876953f,51.2324333190918f,36.0009002685547f}, {488.632598876953f,51.3946304321289f,30.0011005401611f}, {488.632598876953f,51.3946304321289f,36.0009002685547f}, {488.632598876953f,51.4340324401855f,30.0011005401611f}, {488.632598876953f,51.4340324401855f,36.0009002685547f}, {488.632598876953f,49.7838325500488f,30.0011005401611f}, {488.632598876953f,49.7838325500488f,36.0009002685547f}, {488.632598876953f,50.7764320373535f,30.0011005401611f}, {488.632598876953f,50.7764320373535f,36.0009002685547f}, {488.632598876953f,51.139232635498f,30.0011005401611f}, {488.632598876953f,51.139232635498f,36.0009002685547f}, {488.632598876953f,50.9284324645996f,30.0011005401611f}, {488.632598876953f,50.9284324645996f,36.0009002685547f}, {488.632598876953f,50.8816299438477f,36.0009002685547f}, {488.632598876953f,50.8786315917969f,30.0011005401611f}, {488.632598876953f,50.8786315917969f,36.0009002685547f}, {488.632598876953f,51.7324333190918f,35.0173034667969f}, {488.632598876953f,51.7324333190918f,36.0009002685547f}, {488.632598876953f,51.7324333190918f,30.9847011566162f}, {517.188415527344f,51.7140884399414f,24.0011005401611f}, {517.188415527344f,51.7140884399414f,36.0009002685547f}, {517.188415527344f,50.4475173950195f,24.0011005401611f}, {517.188415527344f,51.7324333190918f,35.3734130859375f}, {517.188415527344f,51.7324333190918f,36.0009002685547f}, {517.188415527344f,51.7324333190918f,34.1185760498047f}, {517.188415527344f,51.7324333190918f,31.88330078125f}, {517.188415527344f,51.7324333190918f,30.0011005401611f}, {517.188415527344f,51.7324333190918f,28.1187744140625f}, {517.188415527344f,51.7324333190918f,25.8834266662598f}, {517.188415527344f,51.7324333190918f,24.6285915374756f}, {517.188415527344f,51.7324333190918f,24.0011005401611f}, {517.188415527344f,47.7324333190918f,24.0600452423096f}, {517.188415527344f,47.7324333190918f,24.0011005401611f}, {517.188415527344f,50.4475173950195f,36.0009002685547f}, {517.188415527344f,47.7324333190918f,24.1779975891113f}, {517.188415527344f,47.7324333190918f,24.6498031616211f}, {517.188415527344f,47.7324333190918f,28.7625770568848f}, {517.188415527344f,47.7324333190918f,29.7061901092529f}, {517.188415527344f,47.7324333190918f,29.9420928955078f}, {517.188415527344f,47.7324333190918f,30.0600452423096f}, {517.188415527344f,47.7324333190918f,30.2959480285645f}, {517.188415527344f,47.7324333190918f,31.2395629882812f}, {517.188415527344f,47.7324333190918f,35.3521995544434f}, {517.188415527344f,47.7324333190918f,35.8240051269531f}, {517.188415527344f,47.7324333190918f,35.9419555664062f}, {517.188415527344f,47.7324333190918f,36.0009002685547f} },
                { {0,1,2}, {3,4,5}, {6,7,8}, {9,10,11}, {12,2,1}, {12,1,13}, {14,15,16}, {17,18,19}, {20,21,22}, {17,19,23}, {24,25,26}, {27,13,1}, {28,25,29}, {30,31,32}, {28,33,34}, {35,36,7}, {37,38,39}, {40,10,41}, {42,43,44}, {45,5,4}, {46,47,48}, {46,48,49}, {45,4,50}, {51,52,53}, {51,54,55}, {56,52,57}, {58,59,60}, {61,50,4}, {62,63,64}, {65,34,33}, {66,67,42}, {68,17,69}, {70,71,22}, {66,42,72}, {73,16,15}, {35,7,74}, {75,76,54}, {77,27,1}, {78,32,31}, {75,54,79}, {80,26,25}, {81,80,25}, {82,83,48}, {84,20,85}, {81,25,86}, {87,88,19}, {0,89,1}, {90,91,92}, {90,10,93}, {38,94,39}, {94,95,39}, {3,7,96}, {97,15,98}, {97,99,15}, {92,91,100}, {89,101,1}, {102,39,95}, {103,11,10}, {104,96,7}, {105,15,99}, {106,61,4}, {107,108,33}, {76,55,54}, {109,91,110}, {111,23,19}, {112,63,113}, {114,115,48}, {116,59,117}, {118,20,119}, {120,31,121}, {122,44,43}, {110,91,123}, {124,125,126}, {127,128,129}, {127,130,124}, {131,124,132}, {126,133,134}, {135,136,126}, {137,138,127}, {139,127,138}, {128,140,141}, {142,128,143}, {144,140,145}, {100,91,146}, {147,148,134}, {101,149,1}, {102,150,39}, {103,10,151}, {145,140,152}, {152,140,153}, {148,154,134}, {154,155,134}, {156,15,105}, {157,104,7}, {36,8,7}, {158,37,39}, {159,19,88}, {160,19,159}, {161,59,58}, {161,117,59}, {162,31,30}, {162,121,31}, {163,43,164}, {163,165,43}, {166,167,43}, {167,164,43}, {168,57,52}, {82,48,169}, {114,170,171}, {108,65,33}, {64,63,112}, {114,172,170}, {160,173,170}, {171,170,173}, {172,174,170}, {160,170,174}, {175,176,177}, {178,77,1}, {179,31,120}, {175,180,176}, {181,182,176}, {177,176,182}, {180,183,176}, {181,176,183}, {184,42,67}, {185,69,17}, {160,111,19}, {186,187,160}, {188,189,114}, {190,188,114}, {114,48,191}, {192,114,193}, {194,160,195}, {196,160,194}, {197,198,181}, {199,197,181}, {122,43,165}, {200,201,175}, {202,175,203}, {204,175,202}, {205,119,20}, {206,181,207}, {208,209,15}, {210,15,209}, {211,10,9}, {212,10,211}, {213,214,215}, {216,217,218}, {219,14,17}, {113,63,220}, {221,222,48}, {191,48,222}, {22,223,20}, {205,20,223}, {224,40,42}, {123,91,225}, {214,226,215}, {227,215,226}, {218,217,228}, {229,228,217}, {215,230,213}, {125,135,126}, {217,216,231}, {129,128,142}, {216,213,232}, {130,132,124}, {213,216,233}, {234,213,235}, {236,227,237}, {238,237,227}, {239,240,216}, {233,216,240}, {241,242,229}, {243,229,242}, {215,227,244}, {245,215,246}, {217,247,229}, {248,249,217}, {232,213,250}, {230,250,213}, {133,147,134}, {244,227,251}, {236,252,227}, {251,227,252}, {231,216,253}, {254,253,216}, {141,140,144}, {247,255,229}, {241,229,256}, {255,256,229}, {257,241,258}, {259,146,91}, {260,261,236}, {262,1,149}, {263,264,241}, {265,241,264}, {266,236,267}, {268,267,236}, {49,48,83}, {166,43,269}, {270,271,272}, {273,274,275}, {276,274,277}, {278,151,10}, {279,280,272}, {281,39,150}, {272,282,279}, {155,283,134}, {274,276,284}, {153,140,285}, {286,276,287}, {265,276,286}, {288,289,279}, {268,288,279}, {290,291,272}, {271,290,272}, {292,274,293}, {275,274,292}, {294,265,295}, {276,265,294}, {296,297,268}, {279,296,268}, {241,265,298}, {298,265,299}, {236,300,268}, {300,301,268}, {107,33,78}, {302,303,59}, {304,305,279}, {282,304,279}, {306,276,307}, {284,276,306}, {185,17,73}, {308,309,221}, {158,39,70}, {310,41,10}, {15,311,208}, {7,6,312}, {313,314,6}, {315,6,314}, {316,208,317}, {318,317,208}, {258,241,319}, {319,241,320}, {261,321,236}, {321,322,236}, {6,315,323}, {208,324,318}, {270,325,318}, {326,318,325}, {327,328,315}, {273,315,328}, {118,329,20}, {330,20,329}, {331,332,25}, {86,25,332}, {333,334,52}, {335,52,334}, {115,336,48}, {169,48,336}, {62,106,4}, {35,15,210}, {35,337,15}, {158,10,212}, {158,310,10}, {338,178,1}, {339,59,116}, {107,302,59}, {66,22,340}, {66,341,22}, {185,221,342}, {185,308,221}, {75,31,179}, {75,343,31}, {166,20,330}, {166,85,20}, {81,52,335}, {81,168,52}, {82,19,344}, {82,87,19}, {108,339,345}, {346,108,345}, {64,347,348}, {349,347,64}, {178,109,350}, {351,178,350}, {179,352,353}, {354,352,179}, {355,208,356}, {356,208,311}, {357,358,6}, {358,312,6}, {68,22,21}, {68,340,22}, {221,48,47}, {184,342,221}, {359,270,360}, {318,360,270}, {361,362,273}, {315,273,362}, {272,102,270}, {363,270,102}, {274,273,103}, {364,103,273}, {21,19,18}, {21,20,84}, {184,46,42}, {43,42,46}, {12,22,71}, {365,22,12}, {14,98,15}, {14,220,63}, {40,93,10}, {40,225,91}, {45,221,309}, {366,221,45}, {313,367,212}, {212,367,368}, {36,369,367}, {313,36,367}, {316,37,367}, {37,368,367}, {210,367,369}, {316,367,210}, {362,370,315}, {370,323,315}, {360,318,371}, {371,318,324}, {372,331,159}, {159,195,160}, {373,115,56}, {115,114,189}, {52,56,161}, {374,161,56}, {25,28,331}, {375,331,28}, {376,333,163}, {163,203,175}, {377,118,24}, {118,181,198}, {25,24,162}, {378,162,24}, {52,51,333}, {379,333,51}, {167,380,381}, {376,167,381}, {377,381,330}, {330,381,380}, {335,381,382}, {376,381,335}, {373,383,169}, {169,383,384}, {168,385,383}, {373,168,383}, {372,87,383}, {87,384,383}, {377,80,381}, {80,382,381}, {86,383,385}, {372,383,86}, {106,348,347}, {386,106,347}, {375,65,346}, {108,346,65}, {64,112,349}, {387,349,112}, {171,190,114}, {346,345,171}, {374,190,345}, {171,345,190}, {349,172,347}, {172,114,192}, {386,347,192}, {172,192,347}, {173,160,196}, {171,173,346}, {375,346,196}, {173,196,346}, {172,349,174}, {174,186,160}, {387,186,349}, {174,349,186}, {64,348,62}, {106,62,348}, {108,107,339}, {59,339,107}, {374,345,116}, {339,116,345}, {76,353,352}, {379,76,352}, {388,77,351}, {178,351,77}, {179,120,354}, {378,354,120}, {177,200,175}, {351,350,177}, {389,200,350}, {177,350,200}, {354,180,352}, {180,175,204}, {379,352,204}, {180,204,352}, {182,181,206}, {177,182,351}, {388,351,206}, {182,206,351}, {180,354,183}, {183,199,181}, {378,199,354}, {183,354,199}, {91,109,338}, {178,338,109}, {76,75,353}, {179,353,75}, {389,350,110}, {109,110,350}, {390,391,392}, {393,394,395}, {224,122,389}, {122,175,201}, {365,388,205}, {205,207,181}, {66,340,396}, {68,396,340}, {184,396,342}, {185,342,396}, {66,396,67}, {184,67,396}, {68,69,396}, {185,396,69}, {219,111,387}, {111,160,187}, {366,386,191}, {191,193,114}, {150,272,280}, {102,272,150}, {151,277,274}, {103,151,274}, {161,374,117}, {116,117,374}, {366,61,386}, {106,386,61}, {111,187,387}, {186,387,187}, {56,188,374}, {190,374,188}, {191,386,193}, {192,193,386}, {331,375,194}, {196,194,375}, {28,34,375}, {65,375,34}, {219,387,113}, {112,113,387}, {224,389,123}, {110,123,389}, {51,55,379}, {76,379,55}, {24,197,378}, {199,378,197}, {122,201,389}, {200,389,201}, {333,379,202}, {204,202,379}, {205,388,207}, {206,207,388}, {365,27,388}, {77,388,27}, {162,378,121}, {120,121,378}, {162,30,25}, {30,29,25}, {51,53,54}, {303,60,59}, {28,29,33}, {29,397,33}, {161,58,52}, {53,52,58}, {21,84,19}, {84,344,19}, {46,49,43}, {49,269,43}, {208,316,209}, {210,209,316}, {327,313,211}, {212,211,313}, {36,35,369}, {210,369,35}, {37,158,368}, {212,368,158}, {6,8,313}, {36,313,8}, {326,38,316}, {37,316,38}, {392,391,398}, {399,398,391}, {394,400,395}, {401,395,400}, {390,214,391}, {214,213,234}, {393,395,218}, {218,239,216}, {402,230,403}, {230,215,245}, {125,124,131}, {404,125,403}, {405,406,231}, {231,248,217}, {129,137,127}, {407,406,129}, {130,127,139}, {402,130,408}, {194,195,331}, {159,331,195}, {115,189,56}, {188,56,189}, {14,219,220}, {113,220,219}, {45,50,366}, {61,366,50}, {221,366,222}, {191,222,366}, {17,23,219}, {111,219,23}, {118,198,24}, {197,24,198}, {202,203,333}, {163,333,203}, {40,224,225}, {123,225,224}, {12,13,365}, {27,365,13}, {22,365,223}, {205,223,365}, {42,44,224}, {122,224,44}, {399,391,234}, {214,234,391}, {401,239,395}, {218,395,239}, {214,390,226}, {226,238,227}, {218,228,393}, {228,229,243}, {401,399,233}, {233,235,213}, {392,409,390}, {410,390,409}, {394,393,411}, {412,411,393}, {402,403,131}, {125,131,403}, {405,137,406}, {129,406,137}, {405,408,139}, {130,139,408}, {230,245,403}, {404,403,245}, {231,406,248}, {407,248,406}, {232,254,216}, {402,408,232}, {413,404,244}, {244,246,215}, {414,247,407}, {247,217,249}, {133,126,136}, {415,133,413}, {141,143,128}, {416,414,141}, {410,238,390}, {226,390,238}, {412,393,243}, {228,243,393}, {233,399,235}, {234,235,399}, {237,260,236}, {238,410,237}, {417,260,410}, {237,410,260}, {239,401,240}, {233,240,401}, {242,241,257}, {243,242,412}, {418,412,257}, {242,257,412}, {401,419,399}, {398,399,419}, {417,410,420}, {409,420,410}, {400,421,401}, {419,401,421}, {418,422,412}, {411,412,422}, {413,135,404}, {125,404,135}, {414,407,142}, {129,142,407}, {130,402,132}, {131,132,402}, {133,136,413}, {135,413,136}, {423,147,415}, {133,415,147}, {137,405,138}, {139,138,405}, {141,414,143}, {142,143,414}, {424,416,144}, {141,144,416}, {405,254,408}, {232,408,254}, {244,404,246}, {245,246,404}, {247,249,407}, {248,407,249}, {232,250,402}, {230,402,250}, {415,413,251}, {244,251,413}, {252,236,266}, {251,252,415}, {423,415,266}, {252,266,415}, {231,253,405}, {254,405,253}, {416,255,414}, {247,414,255}, {256,263,241}, {255,416,256}, {424,263,416}, {256,416,263}, {257,258,418}, {425,418,258}, {260,417,261}, {426,261,417}, {422,418,427}, {427,259,91}, {420,428,417}, {428,1,262}, {147,423,148}, {429,148,423}, {263,424,264}, {264,295,265}, {266,267,423}, {267,268,297}, {144,145,424}, {430,424,145}, {49,431,269}, {166,269,431}, {82,431,83}, {49,83,431}, {84,85,431}, {166,431,85}, {82,344,431}, {84,431,344}, {432,278,90}, {10,90,278}, {433,0,281}, {39,281,0}, {362,361,434}, {435,271,359}, {270,359,271}, {436,361,275}, {273,275,361}, {360,437,359}, {277,287,276}, {151,278,277}, {280,279,289}, {150,280,281}, {436,438,439}, {439,285,140}, {90,92,432}, {440,432,92}, {282,272,291}, {441,282,442}, {284,293,274}, {443,438,284}, {278,432,286}, {286,299,265}, {281,288,433}, {288,268,301}, {0,433,89}, {444,89,433}, {435,445,442}, {445,134,283}, {439,446,436}, {361,436,446}, {442,290,435}, {271,435,290}, {438,436,292}, {275,292,436}, {445,435,447}, {359,447,435}, {286,287,278}, {277,278,287}, {288,281,289}, {280,289,281}, {145,152,430}, {443,430,152}, {148,429,154}, {441,154,429}, {424,430,294}, {294,307,276}, {423,296,429}, {296,279,305}, {425,440,100}, {92,100,440}, {290,442,291}, {282,291,442}, {292,293,438}, {284,438,293}, {298,320,241}, {432,440,298}, {300,236,322}, {433,300,444}, {426,101,444}, {89,444,101}, {107,448,302}, {302,79,54}, {78,31,343}, {107,78,448}, {75,79,448}, {302,448,79}, {78,343,448}, {75,448,343}, {427,418,259}, {425,259,418}, {428,262,417}, {426,417,262}, {437,449,359}, {447,359,449}, {434,361,450}, {446,450,361}, {32,33,397}, {78,33,32}, {53,303,54}, {302,54,303}, {152,153,443}, {438,443,153}, {429,304,441}, {282,441,304}, {430,443,306}, {284,306,443}, {154,441,155}, {442,155,441}, {298,299,432}, {286,432,299}, {300,433,301}, {288,301,433}, {185,451,308}, {308,74,7}, {73,15,337}, {185,73,451}, {35,74,451}, {308,451,74}, {73,337,451}, {35,451,337}, {158,452,310}, {310,72,42}, {70,22,341}, {158,70,452}, {66,72,452}, {310,452,72}, {70,341,452}, {66,452,341}, {313,327,314}, {315,314,327}, {316,317,326}, {318,326,317}, {15,156,311}, {356,311,156}, {7,312,157}, {358,157,312}, {211,9,327}, {364,327,9}, {38,326,94}, {363,94,326}, {294,295,424}, {264,424,295}, {296,423,297}, {267,297,423}, {262,149,426}, {101,426,149}, {258,319,425}, {440,425,319}, {261,426,321}, {444,321,426}, {259,425,146}, {100,146,425}, {306,307,430}, {294,430,307}, {304,429,305}, {296,305,429}, {319,320,440}, {298,440,320}, {321,444,322}, {300,322,444}, {445,283,442}, {155,442,283}, {439,438,285}, {153,285,438}, {17,68,18}, {21,18,68}, {46,184,47}, {221,47,184}, {102,95,363}, {94,363,95}, {9,11,364}, {103,364,11}, {6,323,357}, {370,357,323}, {371,324,355}, {208,355,324}, {270,363,325}, {326,325,363}, {327,364,328}, {273,328,364}, {0,2,39}, {12,39,2}, {90,93,91}, {40,91,93}, {14,16,17}, {73,17,16}, {45,309,7}, {308,7,309}, {12,71,39}, {70,39,71}, {40,41,42}, {310,42,41}, {97,98,63}, {14,63,98}, {3,5,7}, {45,7,5}, {118,377,329}, {330,329,377}, {331,372,332}, {86,332,372}, {333,376,334}, {335,334,376}, {115,373,336}, {169,336,373}, {167,166,380}, {330,380,166}, {80,81,382}, {335,382,81}, {86,385,81}, {168,81,385}, {169,384,82}, {87,82,384}, {159,88,372}, {87,372,88}, {163,164,376}, {167,376,164}, {24,26,377}, {80,377,26}, {56,57,373}, {168,373,57}, {32,397,30}, {29,30,397}, {58,60,53}, {303,53,60}, {205,181,119}, {118,119,181}, {163,175,165}, {122,165,175}, {453,454,455}, {454,456,455}, {457,455,456}, {458,455,457}, {459,455,458}, {460,455,459}, {461,462,463}, {464,465,466}, {467,468,469}, {470,471,472}, {465,473,474}, {475,476,477}, {478,479,480}, {481,482,478}, {483,484,481}, {485,486,483}, {487,488,485}, {489,490,487}, {491,492,489}, {493,494,491}, {495,496,493}, {497,498,495}, {499,500,497}, {501,502,499}, {503,504,501}, {505,504,503}, {506,504,505}, {507,504,506}, {508,504,507}, {509,504,508}, {510,504,509}, {511,504,510}, {512,504,511}, {513,504,512}, {514,504,513}, {476,515,516}, {517,518,519}, {520,517,521}, {518,522,523}, {522,480,479}, {524,525,526}, {468,470,527}, {525,467,528}, {529,475,530}, {531,532,533}, {534,531,535}, {536,537,538}, {473,539,540}, {539,536,541}, {537,534,542}, {471,520,543}, {532,529,544}, {545,524,546}, {453,461,547}, {463,464,548}, {523,549,504}, {527,550,551}, {519,552,553}, {521,554,555}, {466,556,557}, {469,558,559}, {528,560,561}, {477,562,563}, {543,564,565}, {535,566,567}, {530,568,569}, {540,570,571}, {474,572,573}, {542,574,575}, {538,576,577}, {541,578,579}, {472,580,581}, {526,582,583}, {533,584,585}, {544,586,587}, {516,545,588}, {588,589,590}, {455,460,4}, {591,592,63}, {462,455,4}, {592,547,63}, {547,548,63}, {465,462,4}, {548,557,63}, {127,124,501}, {127,501,499}, {505,503,124}, {124,126,507}, {124,507,506}, {509,508,126}, {126,134,512}, {126,512,511}, {510,509,126}, {128,127,493}, {128,493,491}, {497,495,127}, {489,487,128}, {140,128,483}, {140,483,481}, {487,485,128}, {478,480,140}, {480,522,140}, {514,513,134}, {504,514,134}, {551,581,437}, {471,470,434}, {445,447,555}, {445,555,553}, {134,445,553}, {134,553,504}, {446,439,518}, {446,518,517}, {439,140,522}, {439,522,518}, {515,476,358}, {563,588,356}, {557,573,63}, {473,465,4}, {437,360,559}, {437,559,551}, {360,371,561}, {360,561,559}, {362,434,470}, {362,470,468}, {370,362,468}, {370,468,467}, {499,497,127}, {506,505,124}, {495,493,127}, {513,512,134}, {481,478,140}, {447,449,565}, {447,565,555}, {450,446,517}, {450,517,520}, {356,156,569}, {356,569,563}, {157,358,476}, {157,476,475}, {357,370,467}, {357,467,525}, {371,355,583}, {371,583,561}, {460,459,4}, {63,62,593}, {63,593,591}, {62,4,459}, {62,459,458}, {532,531,104}, {531,534,104}, {567,585,105}, {575,567,105}, {4,3,539}, {4,539,473}, {536,539,3}, {97,63,573}, {97,573,571}, {571,579,97}, {99,97,579}, {99,579,577}, {105,99,577}, {105,577,575}, {96,104,534}, {96,534,537}, {3,96,537}, {3,537,536}, {503,501,124}, {508,507,126}, {491,489,128}, {511,510,126}, {485,483,128}, {434,450,520}, {434,520,471}, {449,437,581}, {449,581,565}, {156,105,585}, {156,585,587}, {587,569,156}, {104,157,529}, {104,529,532}, {475,529,157}, {590,583,355}, {355,356,588}, {355,588,590}, {358,357,524}, {358,524,515}, {525,524,357}, {458,457,62}, {457,593,62}, {479,478,482}, {479,504,549}, {479,482,504}, {482,481,484}, {472,551,550}, {581,551,472}, {482,484,504}, {484,483,486}, {523,553,552}, {504,553,523}, {540,573,572}, {571,573,540}, {544,585,584}, {587,585,544}, {542,577,576}, {575,577,542}, {526,590,589}, {583,590,526}, {535,575,574}, {567,575,535}, {533,567,566}, {585,567,533}, {538,579,578}, {577,579,538}, {543,581,580}, {565,581,543}, {477,569,568}, {563,569,477}, {530,587,586}, {569,587,530}, {541,571,570}, {579,571,541}, {528,583,582}, {561,583,528}, {591,453,592}, {547,592,453}, {521,565,564}, {555,565,521}, {474,557,556}, {573,557,474}, {516,563,562}, {588,563,516}, {519,555,554}, {553,555,519}, {527,559,558}, {551,559,527}, {469,561,560}, {559,561,469}, {462,461,455}, {453,455,461}, {461,463,547}, {548,547,463}, {465,464,462}, {463,462,464}, {464,466,548}, {557,548,466}, {469,560,467}, {528,467,560}, {472,550,470}, {527,470,550}, {474,556,465}, {466,465,556}, {477,568,475}, {530,475,568}, {516,562,476}, {477,476,562}, {519,554,517}, {521,517,554}, {521,564,520}, {543,520,564}, {523,552,518}, {519,518,552}, {479,549,522}, {523,522,549}, {526,589,524}, {589,546,524}, {527,558,468}, {469,468,558}, {528,582,525}, {526,525,582}, {530,586,529}, {544,529,586}, {533,566,531}, {535,531,566}, {535,574,534}, {542,534,574}, {538,578,536}, {541,536,578}, {540,572,473}, {474,473,572}, {541,570,539}, {540,539,570}, {542,576,537}, {538,537,576}, {543,580,471}, {472,471,580}, {544,584,532}, {533,532,584}, {524,545,515}, {516,515,545}, {545,546,588}, {589,588,546}, {453,591,454}, {593,454,591}, {484,486,504}, {486,485,488}, {486,488,504}, {488,487,490}, {488,490,504}, {490,489,492}, {490,492,504}, {492,491,494}, {492,494,504}, {494,493,496}, {494,496,504}, {496,495,498}, {496,498,504}, {498,497,500}, {498,500,504}, {500,499,502}, {500,502,504}, {501,504,502}, {454,593,456}, {457,456,593}, {594,595,596}, {597,598,594}, {599,597,594}, {600,599,594}, {601,600,594}, {602,601,594}, {603,602,594}, {604,603,594}, {605,604,594}, {606,607,608}, {609,606,608}, {610,609,608}, {611,610,608}, {612,611,608}, {613,612,608}, {614,613,608}, {615,614,608}, {616,615,608}, {617,616,608}, {618,617,608}, {619,618,608}, {620,619,608}, {596,608,607}, {595,594,598}, {608,596,595}, {605,594,91}, {91,338,602}, {91,602,603}, {598,597,1}, {594,596,91}, {608,595,1}, {595,598,1}, {616,617,392}, {610,611,394}, {419,421,613}, {419,613,614}, {422,427,607}, {422,607,606}, {427,91,596}, {427,596,607}, {428,420,619}, {428,619,620}, {1,428,620}, {1,620,608}, {420,409,618}, {420,618,619}, {411,422,606}, {411,606,609}, {398,419,614}, {398,614,615}, {421,400,612}, {421,612,613}, {409,392,617}, {409,617,618}, {394,411,609}, {394,609,610}, {604,605,91}, {338,1,599}, {338,599,600}, {392,398,615}, {392,615,616}, {400,394,611}, {400,611,612}, {603,604,91}, {601,602,338}, {597,599,1}, {600,601,338} });
            break;
        case TestMesh::gt2_teeth:
            mesh = TriangleMesh(
                { {15.8899993896484f,19.444055557251f,2.67489433288574f}, {15.9129991531372f,19.1590557098389f,2.67489433288574f}, {15.9039993286133f,19.1500549316406f,2.67489433288574f}, {15.9489994049072f,19.2490558624268f,2.67489433288574f},
                  {15.9579992294312f,19.3570556640625f,2.67489433288574f}, {15.8819999694824f,18.690055847168f,2.67489433288574f}, {15.8319997787476f,17.7460556030273f,2.67489433288574f}, {15.8489999771118f,18.819055557251f,2.67489433288574f},
                  {15.8589992523193f,17.7190551757812f,2.67489433288574f}, {15.8769998550415f,19.0490550994873f,2.67489433288574f}, {15.7529993057251f,17.8080558776855f,2.67489433288574f}, {15.7869997024536f,19.5010547637939f,2.67489433288574f},
                  {14.0329990386963f,18.7170543670654f,2.67489433288574f}, {13.9599990844727f,18.7460556030273f,2.67489433288574f}, {13.9869995117188f,20.2840557098389f,2.67489433288574f}, {14.2029991149902f,20.149055480957f,2.67489433288574f},
                  {14.1939992904663f,19.9560546875f,2.67489433288574f}, {14.1939992904663f,20.1670551300049f,2.67489433288574f}, {14.2119998931885f,20.0590553283691f,2.67489433288574f}, {12.1899995803833f,19.1840553283691f,2.67489433288574f},
                  {12.096999168396f,19.1950550079346f,2.67489433288574f}, {12.1099996566772f,20.6690559387207f,2.67489433288574f}, {11.382999420166f,19.9750556945801f,2.67489433288574f}, {11.2599992752075f,19.2490558624268f,2.67489433288574f}, 
                  {11.2369995117188f,19.9320545196533f,2.67489433288574f}, {11.5349998474121f,20.0640544891357f,2.67489433288574f}, {11.6259994506836f,20.1550559997559f,2.67489433288574f}, {11.6829986572266f,20.2390556335449f,2.67489433288574f},
                  {11.7369995117188f,20.3570556640625f,2.67489433288574f}, {11.8449993133545f,20.645055770874f,2.67489433288574f}, {11.7729988098145f,20.4640560150146f,2.67489433288574f}, {11.7799987792969f,20.5370559692383f,9.41389465332031f}, 
                  {11.7639999389648f,20.4470558166504f,2.67489433288574f}, {11.9559993743896f,20.6810550689697f,2.67489433288574f}, {12.3079996109009f,20.6020545959473f,2.67489433288574f}, {12.1959991455078f,19.1860542297363f,2.67489433288574f},
                  {12.2059993743896f,20.6540546417236f,2.67489433288574f}, {12.3489990234375f,20.3740558624268f,2.67489433288574f}, {12.3579998016357f,20.2750549316406f,2.67489433288574f}, {12.3669996261597f,20.266056060791f,2.67489433288574f},
                  {12.3849992752075f,20.1670551300049f,2.67489433288574f}, {12.4269990921021f,20.0680541992188f,2.67489433288574f}, {12.5029993057251f,19.9540557861328f,2.67489433288574f}, {12.6169996261597f,19.8550548553467f,2.67489433288574f},
                  {12.7449989318848f,19.7800559997559f,2.67489433288574f}, {12.7629995346069f,19.7800559997559f,2.67489433288574f}, {12.8799991607666f,19.7350559234619f,2.67489433288574f}, {13.0369997024536f,19.7250556945801f,2.67489433288574f}, 
                  {13.0149993896484f,19.0340557098389f,2.67489433288574f}, {11.1699991226196f,19.2580547332764f,2.67489433288574f}, {11.0959987640381f,19.2580547332764f,2.67489433288574f}, {11.1209993362427f,19.9230556488037f,2.67489433288574f}, 
                  {13.0599994659424f,19.024055480957f,2.67489433288574f}, {14.9049997329712f,18.3170547485352f,2.67489433288574f}, {14.8779993057251f,18.3400554656982f,2.67489433288574f}, {14.8779993057251f,19.149055480957f,2.67489433288574f},
                  {13.3039989471436f,19.77805519104f,2.67489433288574f}, {13.1589994430542f,18.9890556335449f,2.67489433288574f}, {13.1559991836548f,19.7350559234619f,2.67489433288574f}, {13.4269990921021f,19.8600559234619f,2.67489433288574f},
                  {13.5339994430542f,19.9700546264648f,2.67389440536499f}, {13.6359996795654f,20.1220550537109f,2.67489433288574f}, {13.6359996795654f,20.1400547027588f,2.67489433288574f}, {13.6719989776611f,20.2210559844971f,2.67489433288574f},
                  {13.6899995803833f,20.2300548553467f,2.67489433288574f}, {13.7509994506836f,20.3010559082031f,2.67489433288574f}, {13.8539991378784f,20.3180541992188f,2.67489433288574f}, {14.8329992294312f,18.3580551147461f,2.67489433288574f},
                  {14.1849994659424f,19.8530559539795f,2.67489433288574f}, {14.0769996643066f,18.7000541687012f,2.67489433288574f}, {14.1099996566772f,20.2400550842285f,2.67489433288574f}, {14.2009992599487f,19.6230545043945f,2.67489433288574f},
                  {14.2729997634888f,19.4670543670654f,2.67489433288574f}, {14.3379993438721f,19.3790550231934f,2.67489433288574f}, {14.4549999237061f,19.2770557403564f,2.67489433288574f}, {14.5899991989136f,19.2040557861328f,2.67489433288574f},
                  {14.6079998016357f,19.2040557861328f,2.67489433288574f}, {14.7209997177124f,19.1600551605225f,2.67489433288574f}, {15.1379995346069f,19.210054397583f,2.67489433288574f}, {14.9949998855591f,18.2680549621582f,2.67489433288574f},
                  {15.0029993057251f,19.1580543518066f,2.67489433288574f}, {15.2369995117188f,19.2760543823242f,2.67489433288574f}, {15.3779993057251f,19.4060554504395f,2.67489433288574f}, {15.4539995193481f,19.520055770874f,2.67489433288574f},
                  {15.471999168396f,19.52805519104f,2.67489433288574f}, {15.5449991226196f,19.5830554962158f,2.67489433288574f}, {15.6529998779297f,19.573055267334f,2.67489433288574f}, {15.7059993743896f,17.8360557556152f,2.67489433288574f},
                  {15.9449996948242f,18.5560550689697f,2.67489433288574f}, {15.8589992523193f,18.9380550384521f,2.67489433288574f}, {14.9589996337891f,18.2950553894043f,2.67489433288574f}, {15.7779998779297f,19.5100555419922f,2.67489433288574f},
                  {14.0049991607666f,20.2750549316406f,2.67489433288574f}, {12.3489990234375f,20.5000553131104f,2.67489433288574f}, {13.0689992904663f,19.0150547027588f,2.67489433288574f}, {13.0999994277954f,19.0100555419922f,2.67489433288574f},
                  {15.9489994049072f,19.3670558929443f,9.41489505767822f}, {15.9489994049072f,19.2490558624268f,9.41489505767822f}, {15.75f,17.8080558776855f,9.41489505767822f}, {15.6639995574951f,19.5710544586182f,9.41489505767822f},
                  {15.5709991455078f,17.9260559082031f,9.41489505767822f}, {15.8769998550415f,18.690055847168f,9.41489505767822f}, {15.8499994277954f,18.8170547485352f,9.41489505767822f}, {15.9459991455078f,18.5520553588867f,9.41489505767822f},
                  {15.914999961853f,17.6890544891357f,9.41489505767822f}, {15.3999996185303f,19.4290542602539f,9.41489505767822f}, {15.3099994659424f,19.339054107666f,9.41489505767822f}, {15.3729991912842f,18.0440559387207f,9.41489505767822f},
                  {15.4579992294312f,19.5170555114746f,9.41489505767822f}, {15.5469999313354f,19.5820541381836f,9.41489505767822f}, {13.2309989929199f,19.7610549926758f,9.41489505767822f}, {13.168999671936f,19.7360553741455f,9.41489505767822f},
                  {13.096999168396f,19.0140552520752f,9.41489505767822f}, {13.1999988555908f,18.9870548248291f,9.41489505767822f}, {15.1399993896484f,19.2080554962158f,9.41489505767822f}, {15.0159997940063f,19.1600551605225f,9.41489505767822f},
                  {14.9859991073608f,18.2770557403564f,9.41489505767822f}, {15.1749992370605f,18.1690559387207f,9.41489505767822f}, {15.9039993286133f,19.1320552825928f,9.41489505767822f}, {15.8949995040894f,19.4460544586182f,9.41489505767822f},
                  {15.8769998550415f,19.0420551300049f,9.41489505767822f}, {12.2169990539551f,20.6500549316406f,9.41489505767822f}, {11.9379997253418f,20.6810550689697f,9.41489505767822f}, {11.8629989624023f,19.2130546569824f,9.41489505767822f}, 
                  {12.096999168396f,19.1950550079346f,9.41489505767822f}, {14.1669998168945f,18.6640548706055f,9.41489505767822f}, {14.1039991378784f,20.2460556030273f,9.41489505767822f}, {13.9849996566772f,18.7360553741455f,9.41489505767822f}, 
                  {14.7349996566772f,19.1590557098389f,9.41489505767822f}, {14.5849990844727f,19.2050552368164f,9.41489505767822f}, {14.5719995498657f,18.4850559234619f,9.41489505767822f}, {14.1939992904663f,19.6760559082031f,9.41489505767822f}, 
                  {14.1849994659424f,19.9330558776855f,9.41489505767822f}, {14.1759996414185f,18.6640548706055f,9.41489505767822f}, {14.261999130249f,19.4890556335449f,9.41489505767822f}, {14.3539991378784f,19.3610553741455f,9.41489505767822f}, 
                  {14.3559989929199f,18.5830554962158f,9.41489505767822f}, {11.6039991378784f,20.1250553131104f,9.41489505767822f}, {11.5209999084473f,20.0520553588867f,9.41489505767822f}, {11.4209995269775f,19.2480545043945f,9.41489505767822f}, 
                  {11.6989994049072f,20.2690544128418f,9.41389465332031f}, {11.7609996795654f,20.4310550689697f,9.41489505767822f}, {11.8359994888306f,19.2130546569824f,9.41489505767822f}, {14.1889991760254f,20.1710548400879f,9.41489505767822f}, 
                  {13.9689998626709f,20.2840557098389f,9.41489505767822f}, {13.8739995956421f,20.315055847168f,9.41489505767822f}, {13.7799997329712f,18.8080558776855f,9.41489505767822f}, {13.9869995117188f,20.2750549316406f,9.41489505767822f}, 
                  {12.3129997253418f,20.5980548858643f,9.41489505767822f}, {12.3399991989136f,20.5090560913086f,9.41489505767822f}, {12.3489990234375f,20.3830547332764f,9.41489505767822f}, {12.3599996566772f,20.2680549621582f,9.41489505767822f}, 
                  {12.3849992752075f,20.1850547790527f,9.41489505767822f}, {12.3849992752075f,20.1670551300049f,9.41489505767822f}, {12.4249992370605f,20.065055847168f,9.41489505767822f}, {12.4729995727539f,19.1350555419922f,9.41489505767822f}, 
                  {14.4399995803833f,19.2900543212891f,9.41489505767822f}, {14.3649997711182f,18.5740547180176f,9.41489505767822f}, {13.5729999542236f,20.0310554504395f,9.41489505767822f}, {13.4889993667603f,19.9140548706055f,9.41489505767822f}, 
                  {13.5639991760254f,18.8710556030273f,9.41489505767822f}, {13.6389999389648f,20.1310558319092f,9.41489505767822f}, {13.6719989776611f,20.2130546569824f,9.41489505767822f}, {13.75f,20.3020553588867f,9.41489505767822f}, 
                  {12.7399997711182f,19.7810554504395f,9.41489505767822f}, {12.6189994812012f,19.8520545959473f,9.41489505767822f}, {12.5799999237061f,19.1200542449951f,9.41489505767822f}, {12.8349990844727f,19.069055557251f,9.41489505767822f}, 
                  {11.2669992446899f,19.9350547790527f,9.41489505767822f}, {11.1029987335205f,19.9230556488037f,9.41489505767822f}, {11.0209999084473f,19.2600555419922f,9.41489505767822f}, {11.3819999694824f,19.9710559844971f,9.41489505767822f}, 
                  {13.418999671936f,19.8530559539795f,9.41489505767822f}, {13.4329996109009f,18.9160556793213f,9.41489505767822f}, {11.8399991989136f,20.6430549621582f,9.41489505767822f}, {13.3119993209839f,19.7800559997559f,9.41489505767822f}, 
                  {15.2189998626709f,19.2600555419922f,9.41489505767822f}, {15.1839990615845f,18.1600551605225f,9.41489505767822f}, {15.3639993667603f,18.0520553588867f,9.41489505767822f}, {13.0189990997314f,19.7250556945801f,9.41489505767822f}, 
                  {12.8949995040894f,19.7350559234619f,9.41489505767822f}, {15.9039993286133f,19.1500549316406f,9.41489505767822f}, {15.7699995040894f,19.5140552520752f,9.41489505767822f}, {15.8589992523193f,18.9340553283691f,9.41489505767822f}, 
                  {14.1939992904663f,19.9510555267334f,9.41489505767822f}, {14.2119998931885f,20.0630550384521f,9.41489505767822f}, {14.8589992523193f,19.149055480957f,9.41489505767822f}, {14.8159999847412f,18.3670558929443f,9.41489505767822f}, 
                  {14.8959999084473f,18.3220558166504f,9.41489505767822f}, {12.5189990997314f,19.9360542297363f,9.41489505767822f}, {11.0209999084473f,19.9290542602539f,9.41489505767822f}, {11.0209999084473f,19.2530555725098f,2.67489433288574f}, 
                  {11.0209999084473f,19.9300556182861f,2.67489433288574f}, {15.9799995422363f,18.505931854248f,5.58724021911621f}, {15.9799995422363f,18.5044555664062f,9.41489505767822f}, {15.9799995422363f,18.5041732788086f,2.67489433288574f}, 
                  {15.9799995422363f,18.1684837341309f,2.67489433288574f}, {15.9799995422363f,18.1288299560547f,9.41489505767822f}, {15.9799995422363f,17.9876575469971f,2.67489433288574f}, {15.9799995422363f,17.6247596740723f,3.91620373725891f}, 
                  {15.9799995422363f,17.6247596740723f,2.67489433288574f}, {15.9799995422363f,17.6254329681396f,4.32245063781738f}, {15.9799995422363f,17.8920269012451f,9.41489505767822f}, {15.9799995422363f,17.8795108795166f,2.67489433288574f}, 
                  {15.9799995422363f,17.629810333252f,4.58585262298584f}, {15.9799995422363f,17.6336059570312f,5.27938556671143f}, {15.9799995422363f,17.8311748504639f,2.67489433288574f}, {15.9799995422363f,17.638355255127f,9.41489505767822f}, 
                  {15.9799995422363f,17.6346111297607f,5.98653984069824f}, {15.9799995422363f,17.8728256225586f,2.67489433288574f}, {15.9799995422363f,18.2221603393555f,2.67489433288574f} },
                { {0,1,2}, {0,3,1}, {0,4,3}, {5,6,7}, {8,6,5}, {2,9,0}, {6,10,11}, {12,13,14}, {15,16,17}, {18,16,15}, {19,20,21}, {22,23,24}, {25,23,22}, {26,23,25}, {27,23,26}, {28,23,27}, {29,30,31}, {29,32,30}, {29,28,32}, {33,28,29}, {33,23,28}, {21,23,33}, {20,23,21}, {34,35,36}, {37,35,34}, {38,35,37}, {39,35,38}, {40,35,39}, {41,35,40}, {42,35,41}, {43,35,42}, {44,35,43}, {45,35,44}, {46,35,45}, {47,35,46}, {48,35,47}, {49,50,51}, {52,48,47}, {23,49,24}, {53,54,55}, {56,57,58}, {59,57,56}, {60,57,59}, {61,57,60}, {62,57,61}, {63,57,62}, {64,57,63}, {65,57,64}, {66,57,65}, {13,57,66}, {54,67,55}, {68,69,70}, {71,69,68}, {72,69,71}, {73,69,72}, {74,69,73}, {75,69,74}, {76,69,75}, {77,69,76}, {67,69,77}, {70,16,68}, {70,17,16}, {78,79,80}, {81,79,78}, {82,79,81}, {83,79,82}, {84,79,83}, {85,79,84}, {86,79,85}, {87,79,86}, {88,8,5}, {11,7,6}, {11,89,7}, {11,9,89}, {11,0,9}, {55,90,53}, {55,79,90}, {55,80,79}, {91,11,10}, {92,69,12}, {92,70,69}, {34,93,37}, {47,94,52}, {47,95,94}, {47,57,95}, {47,58,57}, {51,24,49}, {21,35,19}, {21,36,35}, {14,92,12}, {86,10,87}, {86,91,10}, {77,55,67}, {66,14,13}, {96,97,4}, {98,99,100}, {101,102,98}, {103,101,98}, {104,103,98}, {105,106,107}, {108,105,107}, {109,108,107}, {100,109,107}, {110,111,112}, {113,110,112}, {114,115,116}, {117,114,116}, {118,119,120}, {121,122,123}, {124,121,123}, {125,126,127}, {128,129,130}, {131,132,133}, {71,131,133}, {134,71,133}, {135,134,133}, {136,135,133}, {137,138,139}, {140,137,139}, {141,140,139}, {142,31,141}, {142,141,139}, {143,126,132}, {144,145,146}, {147,144,146}, {127,147,146}, {148,121,124}, {149,148,124}, {150,149,124}, {151,150,124}, {152,151,124}, {153,152,124}, {154,153,124}, {155,154,124}, {129,156,157}, {130,129,157}, {158,159,160}, {161,158,160}, {162,161,160}, {163,162,160}, {146,163,160}, {164,165,166}, {167,164,166}, {168,169,170}, {171,168,170}, {139,171,170}, {159,172,173}, {123,174,142}, {175,110,113}, {173,175,113}, {106,176,177}, {178,106,177}, {179,180,167}, {112,179,167}, {175,173,172}, {119,118,181}, {119,181,97}, {119,97,96}, {182,98,102}, {182,102,183}, {182,183,120}, {182,120,119}, {143,132,184}, {184,185,143}, {147,127,126}, {174,123,122}, {159,173,160}, {126,125,133}, {126,133,132}, {186,187,188}, {186,188,116}, {186,116,115}, {99,98,182}, {109,100,99}, {106,178,107}, {114,117,177}, {114,177,176}, {128,130,187}, {128,187,186}, {135,136,157}, {135,157,156}, {163,146,145}, {164,167,180}, {179,112,111}, {171,139,138}, {189,155,166}, {189,166,165}, {149,150,93}, {154,155,189}, {31,142,174}, {114,176,78}, {81,78,176}, {7,89,183}, {89,9,120}, {89,120,183}, {78,80,114}, {176,106,81}, {88,5,103}, {183,102,7}, {118,120,9}, {9,2,181}, {9,181,118}, {115,114,80}, {82,81,106}, {101,103,5}, {102,101,5}, {5,7,102}, {97,181,2}, {2,1,97}, {1,3,97}, {80,55,115}, {172,159,59}, {59,56,172}, {3,4,97}, {4,0,96}, {105,108,82}, {186,115,55}, {82,106,105}, {83,82,108}, {60,59,159}, {175,172,56}, {119,96,0}, {0,11,119}, {108,109,84}, {84,83,108}, {55,77,186}, {56,58,110}, {56,110,175}, {60,159,158}, {11,91,182}, {182,119,11}, {91,86,182}, {85,84,109}, {86,85,99}, {128,186,77}, {58,111,110}, {158,161,60}, {26,25,137}, {138,137,25}, {99,182,86}, {109,99,85}, {77,76,128}, {58,47,111}, {61,60,161}, {137,140,26}, {27,26,140}, {25,22,138}, {129,128,76}, {76,75,129}, {75,74,129}, {74,73,156}, {73,72,135}, {68,16,184}, {68,184,132}, {16,18,185}, {161,162,62}, {62,61,161}, {179,111,47}, {171,138,22}, {156,129,74}, {135,156,73}, {134,135,72}, {72,71,134}, {68,132,131}, {185,184,16}, {18,15,185}, {63,62,162}, {28,27,140}, {22,24,171}, {71,68,131}, {15,17,143}, {15,143,185}, {17,70,143}, {70,92,126}, {162,163,64}, {64,63,162}, {180,179,47}, {47,46,180}, {140,141,28}, {168,171,24}, {126,143,70}, {92,14,147}, {147,126,92}, {14,66,144}, {14,144,147}, {65,64,163}, {66,65,145}, {46,45,180}, {32,28,141}, {24,51,168}, {145,144,66}, {163,145,65}, {164,180,45}, {45,44,164}, {44,43,164}, {43,42,165}, {38,37,151}, {150,151,37}, {37,93,150}, {141,31,30}, {30,32,141}, {169,168,51}, {165,164,43}, {189,165,42}, {42,41,189}, {40,39,152}, {40,152,153}, {151,152,39}, {39,38,151}, {93,34,149}, {154,189,41}, {153,154,41}, {41,40,153}, {148,149,34}, {34,36,148}, {36,21,121}, {31,174,29}, {121,148,36}, {21,33,122}, {21,122,121}, {33,29,122}, {174,122,29}, {116,188,53}, {104,98,10}, {87,10,98}, {98,100,87}, {79,87,100}, {79,100,107}, {90,79,107}, {90,107,178}, {178,177,90}, {53,90,177}, {53,177,117}, {117,116,53}, {54,53,188}, {54,188,187}, {67,54,187}, {67,187,130}, {69,67,130}, {69,130,157}, {12,69,157}, {12,157,136}, {136,133,12}, {12,133,125}, {125,127,12}, {13,12,127}, {127,146,13}, {57,13,146}, {57,146,160}, {95,57,160}, {95,160,173}, {173,113,95}, {94,95,113}, {113,112,94}, {52,94,112}, {48,52,112}, {112,167,48}, {35,48,167}, {35,167,166}, {19,35,166}, {139,170,50}, {50,49,139}, {166,155,19}, {20,19,155}, {155,124,20}, {23,20,124}, {23,124,123}, {49,23,123}, {49,123,142}, {142,139,49}, {190,191,170}, {192,191,190}, {191,192,51}, {191,51,50}, {170,169,190}, {169,51,192}, {169,192,190}, {170,191,50}, {193,194,195}, {196,197,198}, {199,200,201}, {198,202,203}, {204,201,200}, {205,204,200}, {206,207,208}, {206,208,205}, {206,205,200}, {207,206,209}, {207,209,203}, {207,203,202}, {202,198,197}, {197,196,210}, {197,210,195}, {197,195,194}, {8,88,195}, {8,195,210}, {210,196,8}, {196,198,8}, {198,203,8}, {203,209,8}, {209,206,8}, {206,200,8}, {202,197,104}, {207,202,104}, {103,104,197}, {103,197,194}, {193,195,88}, {88,103,194}, {88,194,193}, {200,199,8}, {199,201,8}, {204,205,6}, {6,8,201}, {6,201,204}, {10,6,205}, {10,205,208}, {104,10,208}, {104,208,207} });
            break;
        case TestMesh::pyramid:
            mesh = TriangleMesh(
            	{ {10.f,10.f,40.f}, {0.f,0.f,0.f}, {20.f,0.f,0.f}, {20.f,20.f,0.f}, {0.f,20.f,0.f} },
            	{ {0,1,2}, {0,3,4}, {3,1,4}, {1,3,2}, {3,0,2}, {4,1,0} });
            break;
        case TestMesh::two_hollow_squares:
            mesh = TriangleMesh(
            	{ {66.7133483886719f,104.286666870117f,0}, {66.7133483886719f,95.7133331298828f,0}, {65.6666870117188f,94.6666717529297f,0}, {75.2866821289062f,95.7133331298828f,0}, {76.3333435058594f,105.333335876465f,0},
                  {76.3333435058594f,94.6666717529297f,0}, {65.6666870117188f,105.33332824707f,0}, {75.2866821289062f,104.286666870117f,0}, {71.1066818237305f,104.58666229248f,2.79999995231628f}, {66.4133529663086f,104.58666229248f,2.79999995231628f},
                  {75.5866851806641f,104.58666229248f,2.79999995231628f}, {66.4133529663086f,99.8933334350586f,2.79999995231628f}, {66.4133529663086f,95.4133377075195f,2.79999995231628f}, {71.1066818237305f,95.4133377075195f,2.79999995231628f}, 
                  {75.5866851806641f,95.4133377075195f,2.79999995231628f}, {75.5866851806641f,100.106666564941f,2.79999995231628f}, {74.5400161743164f,103.540000915527f,2.79999995231628f}, {70.0320129394531f,103.540000915527f,2.79999995231628f},
                  {67.4600067138672f,103.540000915527f,2.79999995231628f}, {67.4600067138672f,100.968002319336f,2.79999995231628f}, {67.4600067138672f,96.4599990844727f,2.79999995231628f}, {74.5400161743164f,99.0319976806641f,2.79999995231628f},
                  {74.5400161743164f,96.4599990844727f,2.79999995231628f}, {70.0320129394531f,96.4599990844727f,2.79999995231628f}, {123.666717529297f,94.6666717529297f,0}, {134.333312988281f,94.6666717529297f,0}, {124.413360595703f,95.4133377075195f,2.79999995231628f}, 
                  {129.106674194336f,95.4133377075195f,2.79999995231628f}, {133.586669921875f,95.4133377075195f,2.79999995231628f}, {123.666717529297f,105.33332824707f,0}, {124.413360595703f,104.58666229248f,2.79999995231628f}, 
                  {124.413360595703f,99.8933334350586f,2.79999995231628f}, {134.333312988281f,105.33332824707f,0}, {129.106674194336f,104.58666229248f,2.79999995231628f}, {133.586669921875f,104.58666229248f,2.79999995231628f},
                  {133.586669921875f,100.106666564941f,2.79999995231628f}, {124.713317871094f,104.286666870117f,0}, {124.713317871094f,95.7133331298828f,0}, {133.286712646484f,95.7133331298828f,0}, {133.286712646484f,104.286666870117f,0},
                  {132.540023803711f,103.540000915527f,2.79999995231628f}, {128.032028198242f,103.540008544922f,2.79999995231628f}, {125.460006713867f,103.540000915527f,2.79999995231628f}, {125.460006713867f,100.968002319336f,2.79999995231628f},
                  {125.460006713867f,96.4599990844727f,2.79999995231628f}, {132.540023803711f,99.0319976806641f,2.79999995231628f}, {132.540023803711f,96.4599990844727f,2.79999995231628f}, {128.032028198242f,96.4599990844727f,2.79999995231628f} },
	            { {0,1,2}, {3,4,5}, {6,4,0}, {6,0,2}, {2,1,5}, {7,4,3}, {1,3,5}, {0,4,7}, {4,6,8}, {6,9,8}, {4,8,10}, {6,2,9}, {2,11,9}, {2,12,11}, {2,5,12}, {5,13,12}, {5,14,13}, {4,10,15}, {5,4,14}, {4,15,14}, {7,16,17}, {0,7,18}, {7,17,18}, {1,19,20}, {1,0,19}, {0,18,19}, {7,3,21}, {3,22,21}, {7,21,16}, {3,23,22}, {3,1,23}, {1,20,23}, {24,25,26}, {25,27,26}, {25,28,27}, {29,24,30}, {24,31,30}, {24,26,31}, {32,29,33}, {29,30,33}, {32,33,34}, {32,34,35}, {25,32,28}, {32,35,28}, {36,37,24}, {38,32,25}, {29,32,36}, {29,36,24}, {24,37,25}, {39,32,38}, {37,38,25}, {36,32,39}, {39,40,41}, {36,39,42}, {39,41,42}, {37,43,44}, {37,36,43}, {36,42,43}, {39,38,45}, {38,46,45}, {39,45,40}, {38,47,46}, {38,37,47}, {37,44,47}, {16,8,9}, {16,10,8}, {10,16,15}, {15,16,21}, {22,15,21}, {15,22,14}, {22,23,14}, {23,20,14}, {17,16,9}, {18,17,9}, {19,18,9}, {19,9,11}, {19,11,20}, {13,14,20}, {20,11,12}, {13,20,12}, {41,40,30}, {42,41,30}, {43,42,30}, {43,30,31}, {43,31,44}, {27,28,44}, {44,31,26}, {27,44,26}, {40,33,30}, {40,34,33}, {34,40,35}, {35,40,45}, {46,35,45}, {35,46,28}, {46,47,28}, {47,44,28} });
            break;
        case TestMesh::small_dorito:
            mesh = TriangleMesh(
            	{ {6.00058937072754f,-22.9982089996338f,0}, {22.0010242462158f,-49.9998741149902f,0}, {-9.99957847595215f,-49.999870300293f,0}, {6.00071382522583f,-32.2371635437012f,28.0019245147705f},
                  {11.1670551300049f,-37.9727020263672f,18.9601669311523f}, {6.00060224533081f,-26.5392456054688f,10.7321853637695f} },
            	{ {0,1,2}, {3,4,5}, {2,1,4}, {2,4,3}, {2,3,5}, {2,5,0}, {5,4,1}, {5,1,0} });
            break;
        case TestMesh::bridge:
            mesh = TriangleMesh(
            	{ {75,84.5f,8}, {125,84.5f,8}, {75,94.5f,8}, {120,84.5f,5}, {125,94.5f,8}, {75,84.5f,0}, {80,84.5f,5}, {125,84.5f,0}, {125,94.5f,0}, {80,94.5f,5}, {75,94.5f,0}, {120,94.5f,5}, {120,84.5f,0}, {80,94.5f,0}, {80,84.5f,0}, {120,94.5f,0} },
            	{ {0,1,2}, {1,0,3}, {2,1,4}, {2,5,0}, {0,6,3}, {1,3,7}, {1,8,4}, {4,9,2}, {10,5,2}, {5,6,0}, {6,11,3}, {3,12,7}, {7,8,1}, {4,8,11}, {4,11,9}, {9,10,2}, {10,13,5}, {14,6,5}, {9,11,6}, {11,12,3}, {12,8,7}, {11,8,15}, {13,10,9}, {5,13,14}, {14,13,6}, {6,13,9}, {15,12,11}, {15,8,12} });
            break;
        case TestMesh::bridge_with_hole:
            mesh = TriangleMesh(
            	{ {75,69.5f,8}, {80,76.9091644287109f,8}, {75,94.5f,8}, {125,69.5f,8}, {120,76.9091644287109f,8}, {120,87.0908355712891f,8}, {80,87.0908355712891f,8}, {125,94.5f,8}, {80,87.0908355712891f,5}, {120,87.0908355712891f,5}, {125,94.5f,0}, {120,69.5f,0}, {120,94.5f,0}, {125,69.5f,0}, {120,94.5f,5}, {80,94.5f,5}, {80,94.5f,0}, {75,94.5f,0}, {80,69.5f,5}, {80,69.5f,0}, {80,76.9091644287109f,5}, {120,69.5,5}, {75,69.5f,0}, {120,76.9091644287109f,5} },
            	{ {0,1,2}, {1,0,3}, {1,3,4}, {4,3,5}, {2,6,7}, {6,2,1}, {7,6,5}, {7,5,3}, {5,8,9}, {8,5,6}, {10,11,12}, {11,10,13}, {14,8,15}, {8,14,9}, {2,16,17}, {16,2,15}, {15,2,14}, {14,10,12}, {10,14,7}, {7,14,2}, {16,18,19}, {18,16,20}, {20,16,1}, {1,16,8}, {8,16,15}, {6,1,8}, {3,11,13}, {11,3,21}, {21,3,18}, {18,22,19}, {22,18,0}, {0,18,3}, {16,22,17}, {22,16,19}, {2,22,0}, {22,2,17}, {5,23,4}, {23,11,21}, {11,23,12}, {12,23,9}, {9,23,5}, {12,9,14}, {23,18,20}, {18,23,21}, {10,3,13}, {3,10,7}, {1,23,20}, {23,1,4} });
            break;
        case TestMesh::step:
            mesh = TriangleMesh(
            	{ {0,20,5}, {0,20,0}, {0,0,5}, {0,0,0}, {20,0,0}, {20,0,5}, {1,19,5}, {1,1,5}, {19,1,5}, {20,20,5}, {19,19,5}, {20,20,0}, {19,19,10}, {1,19,10}, {1,1,10}, {19,1,10} },
            	{ {0,1,2}, {1,3,2}, {3,4,5}, {2,3,5}, {6,0,2}, {6,2,7}, {5,8,7}, {5,7,2}, {9,10,8}, {9,8,5}, {9,0,6}, {9,6,10}, {9,11,1}, {9,1,0}, {3,1,11}, {4,3,11}, {5,11,9}, {5,4,11}, {12,10,6}, {12,6,13}, {6,7,14}, {13,6,14}, {7,8,15}, {14,7,15}, {15,8,10}, {15,10,12}, {12,13,14}, {12,14,15} });
            break;
        case TestMesh::slopy_cube:
            mesh = TriangleMesh(
            	{ {-10,-10,0}, {-10,-10,20}, {-10,10,0}, {-10,10,20}, {0,-10,10}, {10,-10,0}, {2.92893f,-10,10}, {10,-10,2.92893f}, {0,-10,20}, {10,10,0}, {0,10,10}, {0,10,20}, {2.92893f,10,10}, {10,10,2.92893f} },
            	{ {0,1,2}, {2,1,3}, {4,0,5}, {4,1,0}, {6,4,7}, {7,4,5}, {4,8,1}, {0,2,5}, {5,2,9}, {2,10,9}, {10,3,11}, {2,3,10}, {9,10,12}, {13,9,12}, {3,1,8}, {11,3,8}, {10,11,8}, {4,10,8}, {6,12,10}, {4,6,10}, {7,13,12}, {6,7,12}, {7,5,9}, {13,7,9} });
            break;
        default:
        	throw Slic3r::InvalidArgument("Slic3r::Test::mesh(): called with invalid mesh ID");
            break;        
    }

    return mesh;
}

TriangleMesh mesh(TestMesh min, Vec3d translate, Vec3d scale)
{
    TriangleMesh m = mesh(min);
    m.translate(translate.cast<float>());
    m.scale(scale.cast<float>());
    return m;
}

TriangleMesh mesh(TestMesh m, Vec3d translate, double scale)
{
    return mesh(m, translate, Vec3d(scale, scale, scale));
}

static bool verbose_gcode() 
{
    const char *v = std::getenv("SLIC3R_TESTS_GCODE");
    if (v == nullptr)
    	return false;
    std::string s(v);
    return s == "1" || s == "on" || s == "yes";
}

void init_print(std::vector<TriangleMesh> &&meshes, Slic3r::Print &print, Slic3r::Model &model, const DynamicPrintConfig &config_in, bool comments, unsigned duplicate_count)
{
	DynamicPrintConfig config = DynamicPrintConfig::full_print_config();
    config.apply(config_in);

    if (verbose_gcode())
        config.set_key_value("gcode_comments", new ConfigOptionBool(true));

    for (const TriangleMesh &t : meshes) {
		ModelObject *object = model.add_object();
		object->name += "object.stl";
		object->add_volume(t);
		object->add_instance();
	}

    double distance = min_object_distance(config);
    arr2::ArrangeSettings arrange_settings{};
    arrange_settings.set_distance_from_objects(distance);
    arr2::ArrangeBed bed{arr2::to_arrange_bed(get_bed_shape(config), Vec2crd{0, 0})};
    if (duplicate_count > 1) {
        duplicate(model, duplicate_count, bed, arrange_settings);
    }

    arrange_objects(model, bed, arrange_settings);
    model.center_instances_around_point({100, 100});
	for (ModelObject *mo : model.objects) {
        mo->ensure_on_bed();
		print.auto_assign_extruders(mo);
    }

	print.apply(model, config);
    print.validate();
    print.set_status_silent();
}

void init_print(std::initializer_list<TestMesh> test_meshes, Slic3r::Print &print, Slic3r::Model &model, const Slic3r::DynamicPrintConfig &config_in, bool comments, unsigned duplicate_count)
{
	std::vector<TriangleMesh> triangle_meshes;
	triangle_meshes.reserve(test_meshes.size());
	for (const TestMesh test_mesh : test_meshes)
		triangle_meshes.emplace_back(mesh(test_mesh));
	init_print(std::move(triangle_meshes), print, model, config_in, comments, duplicate_count);
}

void init_print(std::initializer_list<TriangleMesh> input_meshes, Slic3r::Print &print, Slic3r::Model &model, const DynamicPrintConfig &config_in, bool comments, unsigned duplicate_count)
{
	std::vector<TriangleMesh> triangle_meshes;
	triangle_meshes.reserve(input_meshes.size());
	for (const TriangleMesh &input_mesh : input_meshes)
		triangle_meshes.emplace_back(input_mesh);
	init_print(std::move(triangle_meshes), print, model, config_in, comments, duplicate_count);
}

void init_print(std::initializer_list<TestMesh> meshes, Slic3r::Print &print, Slic3r::Model &model, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments, unsigned duplicate_count)
{
	Slic3r::DynamicPrintConfig config = Slic3r::DynamicPrintConfig::full_print_config();
	config.set_deserialize_strict(config_items);
	init_print(meshes, print, model, config, comments, duplicate_count);
}

void init_print(std::initializer_list<TriangleMesh> meshes, Slic3r::Print &print, Slic3r::Model &model, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments, unsigned duplicate_count)
{
	Slic3r::DynamicPrintConfig config = Slic3r::DynamicPrintConfig::full_print_config();
	config.set_deserialize_strict(config_items);
	init_print(meshes, print, model, config, comments, duplicate_count);
}

void init_and_process_print(std::initializer_list<TestMesh> meshes, Slic3r::Print &print, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	print.process();
}

void init_and_process_print(std::initializer_list<TriangleMesh> meshes, Slic3r::Print &print, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	print.process();
}

void init_and_process_print(std::initializer_list<TestMesh> meshes, Slic3r::Print &print, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	print.process();
}

void init_and_process_print(std::initializer_list<TriangleMesh> meshes, Slic3r::Print &print, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	print.process();
}

std::string gcode(Print & print)
{
	boost::filesystem::path temp = boost::filesystem::unique_path();
    print.set_status_silent();
    print.process();
    print.export_gcode(temp.string(), nullptr, nullptr);
    boost::nowide::ifstream t(temp.string());
	std::string str((std::istreambuf_iterator<char>(t)), std::istreambuf_iterator<char>());
	boost::nowide::remove(temp.string().c_str());
	return str;
}

Slic3r::Model model(const std::string &model_name, TriangleMesh &&_mesh)
{
    Slic3r::Model result;
	ModelObject *object = result.add_object();
	object->name += model_name + ".stl";
    object->add_volume(_mesh);
    object->add_instance();
    return result;
}

std::string slice(std::initializer_list<TestMesh> meshes, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	return gcode(print);
}

std::string slice(std::initializer_list<TriangleMesh> meshes, const DynamicPrintConfig &config, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config, comments);
	return gcode(print);
}

std::string slice(std::initializer_list<TestMesh> meshes, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	return gcode(print);
}

std::string slice(std::initializer_list<TriangleMesh> meshes, std::initializer_list<Slic3r::ConfigBase::SetDeserializeItem> config_items, bool comments)
{
	Slic3r::Print print;
	Slic3r::Model model;
	init_print(meshes, print, model, config_items, comments);
	return gcode(print);
}

bool contains(const std::string &data, const std::string &pattern)
{
    return data.find(pattern) != data.npos;    
}

bool contains_regex(const std::string &data, const std::string &pattern)
{
    boost::regex re(pattern);
    return boost::regex_match(data, re);
}

} } // namespace Slic3r::Test

#include <catch2/catch.hpp>

SCENARIO("init_print functionality", "[test_data]") {
	GIVEN("A default config") {
		Slic3r::DynamicPrintConfig config = Slic3r::DynamicPrintConfig::full_print_config();
		WHEN("init_print is called with a single mesh.") {
			Slic3r::Model model;
			Slic3r::Print print;
			Slic3r::Test::init_print({ Slic3r::Test::TestMesh::cube_20x20x20 }, print, model, config, true);
			THEN("One mesh/printobject is in the resulting Print object.") {
				REQUIRE(print.objects().size() == 1);
			}
			THEN("print.process() doesn't crash.") {
				REQUIRE_NOTHROW(print.process());
			}
			THEN("Export gcode functions outputs text.") {
				REQUIRE(! Slic3r::Test::gcode(print).empty());
			}
#if 0
			THEN("Embedded meshes exported") {
				std::string path = "C:\\data\\temp\\embedded_meshes\\";
				for (auto kvp : Slic3r::Test::mesh_names) {
					Slic3r::TriangleMesh m = mesh(kvp.first);
					std::string name = kvp.second;
					REQUIRE(Slic3r::store_stl((path + name + ".stl").c_str(), &m, true) == true);
					REQUIRE(Slic3r::store_obj((path + name + ".obj").c_str(), &m) == true);
				}
			}
#endif
		}
	}
}
