use core:io;
use util:serialize;
use core:info;
use lang:bs:macro;

/**
 * This is the base class for all responses from the server to the client.
 */
class Response : serializable {}

/**
 * Error response. Contains some kind of error message.
 */
class ErrorResponse : extends Response, serializable {
	Str message;

	init(Str message) {
		init { message = message; }
	}
}

/**
 * Authentication response. Successful case.
 */
class AuthResponse : extends Response, serializable {
	// User name.
	Str name;

	// Server version.
	Version version;

	// TODO: Other stats?

	init(Str name) {
		init {
			name = name;
			version = named{PROGVIS_VERSION}.version;
		}
	}
}

/**
 * Authentication response. The user must sign in separately.
 */
class AuthLoginResponse : extends Response, serializable {
	// URL to redirect the user to.
	Url url;

	init(Url url) {
		init { url = url; }
	}
}

/**
 * Reply for the logout request. No data.
 */
class LogoutResponse : extends Response, serializable {}

/**
 * Reply for the leaderboard request.
 */
class LeaderboardResponse : extends Response, serializable {
	Score[] scores;

	init(Score[] scores) {
		init { scores = scores; }
	}
}

/**
 * Reply containing a list of problems (depending on what was requested).
 */
class ProblemResponse : extends Response, serializable {
	// Problem list.
	Problem[] problems;

	init(Problem[] problems) {
		init {
			problems = problems;
		}
	}
}

/**
 * Stats for a problem. Solutions may be empty.
 */
class StatsResponse : extends Response, serializable {
	// Problem is a solution to this problem (if any).
	Problem? parent;

	// Solutions to this problem, if requested.
	Solution[] solutions;

	init(Problem? parent, Solution[] solutions) {
		init {
			parent = parent;
			solutions = solutions;
		}
	}
}

/**
 * Details of a problem. All information (including source) that we need to solve it.
 */
class DetailsResponse : extends Response, serializable {
	// Problem ID.
	Int id;

	// Problem title.
	Str title;

	// Source code.
	Str source;

	// Language.
	Str language;

	// Allow submitting a solution (i.e. is this my problem?)
	Bool allowSolution;

	init(Int id, Str title, Str source, Str language, Bool allowSolution) {
		init {
			id = id;
			title = title;
			source = source;
			language = language;
			allowSolution = allowSolution;
		}
	}
}

/**
 * Response to a new solution. Contains the new solution's ID so that it can be amended with a new problem later.
 */
class PostSolutionResponse : extends Response, serializable {
	// Solution ID.
	Int id;

	init(Int id) {
		init { id = id; }
	}
}

/**
 * Reply to creating an new exercise.
 */
class NewProblemResponse : extends Response, serializable {
	// New problem ID.
	Int id;

	init(Int id) {
		init { id = id; }
	}
}
