//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift open source project
//
// Copyright (c) 2023 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See http://swift.org/LICENSE.txt for license information
// See http://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//

import ArgumentParser
import Basics
import CoreCommands
import Dispatch
import PackageModel

import struct TSCBasic.AbsolutePath
import protocol TSCBasic.FileSystem
import var TSCBasic.localFileSystem
import var TSCBasic.stdoutStream

/// A protocol for functions and properties common to all destination subcommands.
protocol SwiftSDKSubcommand: ParsableCommand {
    /// Common locations options provided by ArgumentParser.
    var locations: LocationOptions { get }

    /// Run a command operating on cross-compilation destinations, passing it required configuration values.
    /// - Parameters:
    ///   - hostTriple: triple of the machine this command is running on.
    ///   - swiftSDKsDirectory: directory containing Swift SDK artifact bundles and their configuration.
    ///   - observabilityScope: observability scope used for logging.
    func run(
        hostTriple: Triple,
        _ swiftSDKsDirectory: AbsolutePath,
        _ observabilityScope: ObservabilityScope
    ) throws
}

extension SwiftSDKSubcommand {
    /// The file system used by default by this command.
    var fileSystem: FileSystem { localFileSystem }

    /// Parses destinations directory option if provided or uses the default path for cross-compilation destinations
    /// on the file system. A new directory at this path is created if one doesn't exist already.
    /// - Returns: existing or a newly created directory at the computed location.
    func getOrCreateSwiftSDKsDirectory() throws -> AbsolutePath {
        guard var destinationsDirectory = try fileSystem.getSharedSwiftSDKsDirectory(
            explicitDirectory: locations.swiftSDKsDirectory
        ) else {
            let expectedPath = try fileSystem.swiftSDKsDirectory
            throw StringError(
                "Couldn't find or create a directory where Swift SDKs are stored: `\(expectedPath)`"
            )
        }

        if !self.fileSystem.exists(destinationsDirectory) {
            destinationsDirectory = try self.fileSystem.getOrCreateSwiftPMSwiftSDKsDirectory()
        }

        return destinationsDirectory
    }

    public func run() throws {
        let observabilityHandler = SwiftToolObservabilityHandler(outputStream: stdoutStream, logLevel: .info)
        let observabilitySystem = ObservabilitySystem(observabilityHandler)
        let observabilityScope = observabilitySystem.topScope
        let swiftSDKsDirectory = try self.getOrCreateSwiftSDKsDirectory()

        let hostToolchain = try UserToolchain(destination: Destination.hostDestination())
        let triple = try Triple.getHostTriple(usingSwiftCompiler: hostToolchain.swiftCompilerPath)

        var commandError: Error? = nil
        do {
            try self.run(hostTriple: triple, swiftSDKsDirectory, observabilityScope)
            if observabilityScope.errorsReported {
                throw ExitCode.failure
            }
        } catch {
            commandError = error
        }

        // wait for all observability items to process
        observabilityHandler.wait(timeout: .now() + 5)

        if let error = commandError {
            throw error
        }
    }
}
