# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Copyright (C) 2014-2015 Canonical Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import logging

from gi.repository import Accounts, GLib, Signon

logger = logging.getLogger(__name__)


class CredentialsException(Exception):

    """Exception for credentials problems."""


class AccountsManager(object):

    """Manager for online accounts."""

    def __init__(self):
        self._manager = Accounts.Manager.new()
        self._main_loop = GLib.MainLoop()
        self.error = None

    def _start_main_loop(self):
        self._main_loop.run()
        if self.error is not None:
            raise CredentialsException(self.error)

    def add_account_with_oauth_token(self, provider_id, service_id, user_name, password, oauth_token):
        """Add an account with an oauth token.

        :param provider_id: The identifier of the account provider.
        :param servide_id: The identifier of the service that will use the
            account.
        :param user_name: The user name of the account.
        :param password: The password of the account.
        :param oauth_token: The oauth token of the account.

        """
        logger.info('Add an account for provider {} and service {}.'.format(
            provider_id, service_id))
        logger.info('user name: {}, password: {}, oauth_token: {}.'.format(
            user_name, password, oauth_token))

        account = self._create_account(provider_id)
        logger.debug('account: {}'.format(account.get_settings_dict()))

        info = self._get_identity_info(user_name, password)

        identity = Signon.Identity.new()
        identity.store_credentials_with_info(
            info, self._set_credentials_id_to_account,
            {'account': account, 'oauth_token': oauth_token})
        self._start_main_loop()
        logger.debug('identity: {}'.format(identity.list_properties()))

        self._enable_service(account, service_id)

        logger.info('Created the account with id: {}.'.format(account.id))
        self._log_accounts_info()
        return account

    def _create_account(self, provider_id):
        logger.debug('Creating the account.')
        account = self._manager.create_account(provider_id)
        account.set_enabled(True)
        try:
            account.store_blocking()
        except GLib.Error as e:
            raise CredentialsException(e)
        return account

    def _delete_account_on_error(self, account):
        # attempt to clean up after ourselves, since normal
        # addCleanups will not run in this case
        try:
            logger.debug('Cleaning up account {}'.format(account.id))
            account.delete()
            account.store_blocking()
        except:
            logger.warn('Failed to cleanup account')

    def _quit_main_loop_on_error(self, error, step):
        if error:
            logger.error('Error {}.'.format(step))
            self.error = error
            self._main_loop.quit()
            raise CredentialsException(error)

    def _get_identity_info(self, user_name, password):
        info = Signon.IdentityInfo.new()
        info.set_username(user_name)
        info.set_caption(user_name)
        info.set_secret(password, True)
        info.set_access_control_list(['*'])
        return info

    def _set_credentials_id_to_account(self, identity, id_, error, account_dict):
        self._quit_main_loop_on_error(error, 'store credentials')

        logger.debug('Setting credentials to account.')
        account = account_dict.get('account')
        oauth_token = account_dict.get('oauth_token')
        account.set_variant('CredentialsId', GLib.Variant('u', id_))
        account.store(self._on_credentials_stored, oauth_token)
        logger.debug('Account stored')

    def _on_credentials_stored(self, account, error, oauth_token):
        self._quit_main_loop_on_error(error, 'store')
        self._process_session(account, oauth_token)

    def _process_session(self, account, oauth_token):
        logger.debug('Processing session.')
        account_service = Accounts.AccountService.new(account, None)
        logger.debug('account_service: {}'.format(account_service.list_properties()))
        auth_data = account_service.get_auth_data()
        logger.debug('auth_data: {}'.format(auth_data.get_parameters()))
        identity = auth_data.get_credentials_id()
        method = auth_data.get_method()
        mechanism = auth_data.get_mechanism()
        session_data = auth_data.get_parameters()
        session_data['ProvidedTokens'] = GLib.Variant('a{sv}', {
            'TokenSecret': GLib.Variant('s', 'dummy'),
            'RefreshToken': GLib.Variant('s', 'dummy'),
            'AccessToken': GLib.Variant('s', oauth_token),
        })
        logger.debug('session_data: {}'.format(session_data))
        logger.debug('Authenticating session {0}, {1}'.format(identity, method))
        session = Signon.AuthSession.new(identity, method)
        session.process(
            session_data, mechanism, self._on_login_processed, None)

    def _on_login_processed(self, session, reply, error, userdata):
        self._quit_main_loop_on_error(error, 'process login')

        logger.debug('Session processed')
        self._main_loop.quit()

    def _enable_service(self, account, service_id):
        logger.debug('Enabling evernote service.')
        service = self._manager.get_service(service_id)
        account.select_service(service)
        account.set_enabled(True)
        try:
            account.store_blocking()
        except GLib.Error as e:
            raise CredentialsException(e)

    def _log_accounts_info(self):
        account_ids = self._manager.list()
        logger.debug('Existing accounts: {}.'.format(account_ids))
        for id_ in account_ids:
            account = self._manager.get_account(id_)
            self._log_account_info(account)

    def _log_account_info(self, account):
        logger.debug('Account info:')
        logger.debug('id: {}'.format(account.id))
        logger.debug('provider: {}'.format(account.get_provider_name()))
        logger.debug('enabled: {}'.format(account.get_enabled()))
        logger.debug('Account services:')
        for service in account.list_services():
            logger.debug('name: {}'.format(service.get_name()))
            account_service = Accounts.AccountService.new(account, service)
            logger.debug('enabled: {}'.format(account_service.get_enabled()))

    def get_accounts_list(self):
        return self._manager.list()

    def delete_account(self, account):
        """Delete an account.

        :param account: The account to delete.

        """
        logger.info('Deleting the account with id {}.'.format(account.id))
        account.delete()
        try:
            account.store_blocking()
        except GLib.Error as e:
            raise CredentialsException(e)
